/********************************************************************
 FileName:     main.c
 Dependencies: See INCLUDES section
 Processor:		PIC18F14K50 USB Microcontroller
 Hardware:		This program is adapted from the original one intended
				to be used on Microchip USB demo boards supported by
 				the MCHPFSUSB stack.  See release notes for support
 				matrix. This revision is for use on the Silicon Chip
 				USB MIDIMate, using a PIC18F14K50.
 Compiler:  	Microchip C18
 Company:		Microchip Technology, Inc.
 Revision by:	Jim Rowe. Last revision 15/10/2013 at 4:40pm
 Software License Agreement:

 The software supplied herewith by Microchip Technology Incorporated
 (the Company) for its PIC Microcontroller is intended and
 supplied to you, the Companys customer, for use solely and
 exclusively on Microchip PIC Microcontroller products. The
 software is owned by the Company and/or its supplier, and is
 protected under applicable copyright laws. All rights are reserved.
 Any use in violation of the foregoing restrictions may subject the
 user to criminal sanctions under applicable laws, as well as to
 civil liability for the breach of the terms and conditions of this
 license.

 THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.

********************************************************************
 File Description:

 Change History:
  Rev   Description
  ----  -----------------------------------------
  2.6a- No change
   2.7
********************************************************************/

#ifndef MAIN_C
#define MAIN_C

/** INCLUDES *******************************************************/
#include "C:\DATA\PIC programs\JimsCProgs\NewUSBMIDIMate\usb.h"
#include "C:\DATA\PIC programs\JimsCProgs\NewUSBMIDIMate\HardwareProfile.h"
#include "C:\DATA\PIC programs\JimsCProgs\NewUSBMIDIMate\usb_config.h"
#include "C:\DATA\PIC programs\JimsCProgs\NewUSBMIDIMate\usb_function_midi.h"

/** CONFIGURATION **************************************************/

#if defined(LOW_PIN_COUNT_USB_DEVELOPMENT_KIT)
        #pragma config CPUDIV = NOCLKDIV
        #pragma config USBDIV = OFF
        #pragma config FOSC   = HS
        #pragma config PLLEN  = ON
        #pragma config FCMEN  = OFF
        #pragma config IESO   = OFF
        #pragma config PWRTEN = OFF
        #pragma config BOREN  = OFF
        #pragma config BORV   = 30
        #pragma config WDTEN  = OFF
        #pragma config WDTPS  = 32768
        #pragma config MCLRE  = OFF
        #pragma config HFOFST = OFF
        #pragma config STVREN = ON
        #pragma config LVP    = OFF
        #pragma config XINST  = OFF
        #pragma config BBSIZ  = OFF
        #pragma config CP0    = OFF
        #pragma config CP1    = OFF
        #pragma config CPB    = OFF
        #pragma config WRT0   = OFF
        #pragma config WRT1   = OFF
        #pragma config WRTB   = OFF
        #pragma config WRTC   = OFF
        #pragma config EBTR0  = OFF
        #pragma config EBTR1  = OFF
        #pragma config EBTRB  = OFF                                                  // CONFIG7H

#else
    #error No hardware board defined, see "HardwareProfile.h" and __FILE__
#endif

/** VARIABLES ******************************************************/
#pragma udata

#if defined(__18F14K50) || defined(__18F13K50) || defined(__18LF14K50) || defined(__18LF13K50)
    #pragma udata usbram2
#else
    #pragma udata
#endif

unsigned char ReceivedDataBuffer[64];
unsigned char ToSendDataBuffer[64];
USB_AUDIO_MIDI_EVENT_PACKET midiData;
#pragma udata

USB_HANDLE USBTxHandle = 0;
USB_HANDLE USBRxHandle = 0;

USB_VOLATILE BYTE msCounter;

BOOL MIDIMsg2Send = FALSE; 	//flag used in ProcessIO()
BOOL MIDIRunStatus = FALSE;	//RS flag used in ProcessIO()
unsigned char RSSByte = 0; 	// where we save MIDI running status byte
unsigned char RSSCINcode = 0x9;	// used to save RSS CIN code
unsigned char MIDIBcount = 0;	// counter for ToSendDataBuffer
unsigned char StatByte = 0;		// where we save MIDI status byte
unsigned char StatNibble = 0;	// where we save its upper nibble
unsigned char MIDI_CIN_CODE = 9;	// default init value for CIN code

/** PRIVATE PROTOTYPES *********************************************/
void BlinkUSBStatus(void);
BOOL Switch2IsPressed(void);
BOOL Switch3IsPressed(void);
static void InitializeSystem(void);
void ProcessIO(void);
void GetNextMIDIByte(void);
void ReadMIDIINchar(void);
void UserInit(void);
void YourHighPriorityISRCode();
void YourLowPriorityISRCode();
WORD_VAL ReadPOT(void);

/** VECTOR REMAPPING ***********************************************/
#if defined(__18CXX)
	//On PIC18 devices, addresses 0x00, 0x08, and 0x18 are used for
	//the reset, high priority interrupt, and low priority interrupt
	//vectors.  However, the current Microchip USB bootloader
	//examples are intended to occupy addresses 0x00-0x7FF or
	//0x00-0xFFF depending on which bootloader is used.  Therefore,
	//the bootloader code remaps these vectors to new locations
	//as indicated below.  This remapping is only necessary if you
	//wish to program the hex file generated from this project with
	//the USB bootloader.  If no bootloader is used, edit the
	//usb_config.h file and comment out the following defines:
	//#define PROGRAMMABLE_WITH_USB_HID_BOOTLOADER
	//#define PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER

	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x1000
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x1008
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x1018
	#elif defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x800
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x808
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x818
	#else
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x00
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x08
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x18
	#endif

	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	extern void _startup (void);        // See c018i.c in your C18 compiler dir
	#pragma code REMAPPED_RESET_VECTOR = REMAPPED_RESET_VECTOR_ADDRESS
	void _reset (void)
	{
	    _asm goto _startup _endasm
	}
	#endif
	#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS
	void Remapped_High_ISR (void)
	{
	     _asm goto YourHighPriorityISRCode _endasm
	}
	#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS
	void Remapped_Low_ISR (void)
	{
	     _asm goto YourLowPriorityISRCode _endasm
	}

	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	//Note: If this project is built while one of the bootloaders has
	//been defined, but then the output hex file is not programmed with
	//the bootloader, addresses 0x08 and 0x18 would end up programmed with 0xFFFF.
	//As a result, if an actual interrupt was enabled and occured, the PC would jump
	//to 0x08 (or 0x18) and would begin executing "0xFFFF" (unprogrammed space).  This
	//executes as nop instructions, but the PC would eventually reach the REMAPPED_RESET_VECTOR_ADDRESS
	//(0x1000 or 0x800, depending upon bootloader), and would execute the "goto _startup".  This
	//would effective reset the application.

	//To fix this situation, we should always deliberately place a
	//"goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS" at address 0x08, and a
	//"goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS" at address 0x18.  When the output
	//hex file of this project is programmed with the bootloader, these sections do not
	//get bootloaded (as they overlap the bootloader space).  If the output hex file is not
	//programmed using the bootloader, then the below goto instructions do get programmed,
	//and the hex file still works like normal.  The below section is only required to fix this
	//scenario.
	#pragma code HIGH_INTERRUPT_VECTOR = 0x08
	void High_ISR (void)
	{
	     _asm goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#pragma code LOW_INTERRUPT_VECTOR = 0x18
	void Low_ISR (void)
	{
	     _asm goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#endif	//end of "#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER)"

	#pragma code


	//These are your actual interrupt handling routines.
	#pragma interrupt YourHighPriorityISRCode
	void YourHighPriorityISRCode()
	{
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.
        #if defined(USB_INTERRUPT)
	        USBDeviceTasks();
        #endif

	}	//This return will be a "retfie fast", since this is in a #pragma interrupt section
	#pragma interruptlow YourLowPriorityISRCode
	void YourLowPriorityISRCode()
	{
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.

	}	//This return will be a "retfie", since this is in a #pragma interruptlow section

#elif defined(__C30__)
    #if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
        /*
         *	ISR JUMP TABLE
         *
         *	It is necessary to define jump table as a function because C30 will
         *	not store 24-bit wide values in program memory as variables.
         *
         *	This function should be stored at an address where the goto instructions
         *	line up with the remapped vectors from the bootloader's linker script.
         *
         *  For more information about how to remap the interrupt vectors,
         *  please refer to AN1157.  An example is provided below for the T2
         *  interrupt with a bootloader ending at address 0x1400
         */
//        void __attribute__ ((address(0x1404))) ISRTable(){
//
//        	asm("reset"); //reset instruction to prevent runaway code
//        	asm("goto %0"::"i"(&_T2Interrupt));  //T2Interrupt's address
//        }
    #endif
#endif

#define MIDI_EP  1	//added here because usb_config.h seemed to be hiding it

/** DECLARATIONS ***************************************************/
#pragma code

/********************************************************************
 * Function:        void main(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Main program entry point.
 *
 * Note:            None
 *******************************************************************/
#if defined(__18CXX)
void main(void)
#else
int main(void)
#endif
{
    InitializeSystem();

    #if defined(USB_INTERRUPT)
        USBDeviceAttach();
    #endif

    while(1)
    {
        // Application-specific tasks.
		// Application related code may be added here, or in the ProcessIO() function.
        ProcessIO();
    }//end of while loop (keeps looping indefinitely, since while(1) always true)
}//end main


/********************************************************************
 * Function:        static void InitializeSystem(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        InitializeSystem is a centralize initialization
 *                  routine. All required USB initialization routines
 *                  are called from here.
 *
 *                  User application initialization routine should
 *                  also be called from here.
 *
 * Note:            None
 *******************************************************************/
static void InitializeSystem(void)
{
    #if (defined(__18CXX) & !defined(PIC18F87J50_PIM))
        ADCON1 |= 0x0F;                 // Default all pins to digital
		ADCON0bits.ADON = 0;			// make sure ADC module is disabled
		ANSEL = 0x00;					// disable all analog pins
		ANSELH = 0x00;
		TRISB = 0xA0;					// disable o/p drivers for RB(5) & RB(7)
		PORTB = 0x80;					// set RB(7) = Tx to idle
		IOCB = 0x00;					// int-on-change (PORTB) disabled
		TRISC = 0x00;					// all pins of PORTC are outputs
		PORTC = 0x00;					// clear all of them to initialise
		WPUB = 0x00;					// all weak pullups disabled
		SLRCON = 0x00;					// port slew rate control disabled
		// now we set up EUSART module for duty
		SPBRGH = 0x00;					// set baud rate for 31.250kb/s
		SPBRG = 0x17;					// with a 48MHz system clock
		TXSTAbits.BRGH = 0;
		TXSTAbits.SYNC = 0;
		PIE1bits.RCIE = 0;				// disable interrupts
		BAUDCONbits.BRG16 = 0;			// don't need 16-bit division
		BAUDCONbits.CKTXP = 0;			// transmit data not inverted
		BAUDCONbits.DTRXP = 0;			// neither is receive data
		BAUDCONbits.ABDEN = 0;			// but disable auto-baud detect
		TXSTAbits.TX9 = 0;				// transmit 8 data bits only
		RCSTAbits.RX9 = 0;				// receive 8 data bits also
		RCSTAbits.ADDEN = 0;			// disable address detection
		RCSTAbits.SPEN = 1;				// enables async port module
		TXSTAbits.TXEN = 1;				// enable async transmission
		RCSTAbits.CREN = 1;				// and async reception as well
    #endif

    UserInit();

    USBDeviceInit();	// in usb_device.c.  Initializes USB module SFRs
    					// and firmware variables to known states.
}//end InitializeSystem


/******************************************************************************
 * Function:        void UserInit(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This routine should take care of all of the demo code
 *                  initialization that is required.
 *
 * Note:
 *
 *****************************************************************************/
void UserInit(void)
{
    //Initialize all of the LED pins
    mInitAllLEDs();		// in HardwareProfile - LowPinCount USB Development Kit

    //Initialize all of the push buttons
    mInitAllSwitches();

    //initialize the variable holding the handle for the last
    // transmission
    USBTxHandle = NULL;
    USBRxHandle = NULL;
}//end UserInit

/********************************************************************
 * Function:        void ProcessIO(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is a place holder for other user
 *                  routines. It is a mixture of both USB and
 *                  non-USB tasks.
 *
 * Note:            None
 *******************************************************************/
void ProcessIO(void)
{
	int MIDIcount;

    //Blink LEDs 1 and 2 according to the USB device status
    BlinkUSBStatus();

    // User Application USB tasks
    if((USBDeviceState < CONFIGURED_STATE)||(USBSuspendControl==1)) return;
		// leave immediately if not configured or in suspend state.


	// start of USB-->MIDI transfer code section
    while(!USBHandleBusy(USBRxHandle))	// enter & remain in this loop while
    {			// USB->MIDI event packets are arriving from the USB host
 		for (MIDIcount = 1; MIDIcount <= 3; MIDIcount++)
		{
			while (PIR1bits.TXIF == 0)
				;	//loop back until EUSART's TXREG is empty (TXIF == 1)
			TXREG = ReceivedDataBuffer[MIDIcount];	// then pass byte to it
        	// for transmission out to MIDI bus
		}	// ends one loop transferring packet's data bytes to EUSART
        //Now get ready for next packet (overwrites old data)
        USBRxHandle = USBRxOnePacket(MIDI_EP,(BYTE*)&ReceivedDataBuffer,64);
    }	// loop back until no further USB OUT pkts have been received


	// start of MIDI-->USB processing and transfer code section
    while(PIR1bits.RCIF ==1)	// enter & remain in this loop while MIDI
    	{					// message bytes are being received (RCIF=1)

		MIDIBcount = 0;		// reset the MIDI byte counter (for 1st byte)
		ReadMIDIINchar(); 	// now read & transfer byte -> ToSendDataBuffer[0]
		StatByte = ToSendDataBuffer[0];	// also save as StatByte, and save
		StatNibble = (StatByte & 0xF0); // its upper nibble as StatNibble

		switch(StatNibble){		//now check StatNibble to identify MIDI msg
			case 0x80:	MIDI_CIN_CODE = 0x8;// Note-OFF msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// then fetch 2nd char, -> buffer[1]
						GetNextMIDIByte();	// & fetch 3rd char, -> buffer[2]
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host

			case 0x90:	MIDI_CIN_CODE = 0x9;// Note-ON msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						GetNextMIDIByte(); 	// & fetch 3rd char, -> buffer[2]
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host

			case 0xA0:	MIDI_CIN_CODE = 0xA;// PKP msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						GetNextMIDIByte(); 	// & fetch 3rd char, -> buffer[2]
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host
		
			case 0xB0:	MIDI_CIN_CODE = 0xB;// CC msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						GetNextMIDIByte(); 	// & fetch 3rd char, -> buffer[2]
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host
	 	
			case 0xC0: 	MIDI_CIN_CODE = 0xC;// PC msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						ToSendDataBuffer[2] = 0;	// & make 3rd byte a null
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host

			case 0xD0:	MIDI_CIN_CODE = 0xD;// CP msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						ToSendDataBuffer[2] = 0;	// & make 3rd byte a null
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host

			case 0xE0:	MIDI_CIN_CODE = 0xE;// PWC msg start, so set CIN code
						MIDIRunStatus = TRUE;	// set RS flag
						RSSByte = StatByte;		// and save StatByte -> RSSByte
						RSSCINcode = MIDI_CIN_CODE; //also save CIN code for RSS
						GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
						GetNextMIDIByte(); 	// & fetch 3rd char, -> buffer[2]
						MIDIMsg2Send = TRUE;	// also set Msg2Send flag
						break;				// then exit to send it off to host

			case 0xF0:	//must be a system common or realtime msg, so we
						// need to analyse further using full StatByte
						switch (StatByte){	
							case 0xF1: case 0xF3: 	// TC1/4frame or SongSel msg
							MIDI_CIN_CODE = 0x2;	// so set CIN code
							MIDIRunStatus = FALSE;	// clear MIDIRunStatus flag
							RSSByte = 0;			// and also clear RSSByte
							GetNextMIDIByte();	// fetch 2nd char -> buffer[1]
							ToSendDataBuffer[2] = 0;// & make 3rd byte a null
						    MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
							break;				// then exit to send it off to host

							case 0xF2:		// SPP msg start
				  			MIDI_CIN_CODE = 0x3;//so set CIN code
							MIDIRunStatus = FALSE;	// clear MIDIRunStatus flag
							RSSByte = 0;			// and also clear RSSByte
							GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
							GetNextMIDIByte(); 	// & fetch 3rd char, -> buffer[2]
						    MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
							break;				// then exit to send it off to host

							case 0xF6: case 0xF7:
							// Tune Req or Sysex END msgs
							MIDI_CIN_CODE = 0x5; // so set CIN code
							MIDIRunStatus = FALSE;	// clear MIDIRunStatus flag
							RSSByte = 0;			// and also clear RSSByte
							ToSendDataBuffer[1] = 0;// make 2nd, 3rd bytes nulls
							ToSendDataBuffer[2] = 0;
						    MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
							break;				// then exit to send it off to host

							case 0xFA: case 0xFB: case 0xFC: case 0xFF:
							// SRT start, SRT continue, SRT stop or SRT reset msgs
							MIDI_CIN_CODE = 0x5; // so set CIN code
							ToSendDataBuffer[1] = 0;// make 2nd, 3rd bytes nulls
							ToSendDataBuffer[2] = 0;
						    MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
							break;				// then exit to send it off to host

							case 0xF4: case 0xF5: case 0xF9: case 0xFD: case 0xFE:
							// all of these codes we want to ignore
						    MIDIMsg2Send = FALSE;	// so clear Msg2Send flag 
							break;			// then exit to loop back without sending

							case 0xF8:		// SRT Timing Clock msg
							MIDI_CIN_CODE = 0xF; // so set CIN code
							ToSendDataBuffer[1] = 0;// make 2nd, 3rd bytes nulls
							ToSendDataBuffer[2] = 0;
						    MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
							break;				// then exit to send it off to host

							case 0xF0:	// Sysex message starting, so
							MIDIRunStatus = FALSE;	// clear MIDIRunStatus flag
							RSSByte = 0;			// and also clear RSSByte
							GetNextMIDIByte();	// fetch 2nd char, -> buffer[1]
							if(ToSendDataBuffer[1] == 0xF7){  // if it'a an F7
								ToSendDataBuffer[2] = 0;// make 3rd byte a null
								MIDI_CIN_CODE = 0x6;// & set CIN code
						    	MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
								break;				// then exit to send it off to host	
								}				
							else{		// second byte is other than F7
								GetNextMIDIByte();	// fetch 3rd char -> buffer[2]
								if(ToSendDataBuffer[2] == 0xF7){  // if it's an F7
									MIDI_CIN_CODE = 0x7;// set CIN code
						    		MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
									break;			// then exit to send off to host
									}
								else{		// third byte was not an F7
									MIDI_CIN_CODE = 0x4;// so set CIN code
						    		MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
									break;	// then exit to send off to host
									}
								}
								break;
							}	// end of StatByte nested switch statement
						break;	// to make sure we exit StatNibble switch as well

			default:	// StatNibble must be 0 - 7 & StatByte must be 00 - 7F, so
						// first byte must be data (i.e., continuing SySex msg or
						// RS 'running status' channel voice msg)
						if(MIDIRunStatus == TRUE){	// if we are in RS mode
							MIDI_CIN_CODE = RSSCINcode; // retrieve CIN code
							ToSendDataBuffer[1] = ToSendDataBuffer[0]; // 
							ToSendDataBuffer[0] = RSSByte; // retrieve RSS StatByte
							switch(MIDI_CIN_CODE){	// separate 2-byte & 3-byte msgs
								case 0x8: case 0x9: case 0xA: case 0xB: case 0xE:
									// 3-byte Voice channel msgs
									MIDIBcount++; //increment counter so we can
									GetNextMIDIByte(); //get second data byte &
														// save -> buffer{2}
									MIDIMsg2Send = TRUE; // set Msg2Send flag
									break;	// then exit to send off to host
								case 0xC: case 0xD:
									// 2-byte Voice msgs, so all we need to do is
									ToSendDataBuffer[2] = 0; // make 3rd byte a null
									MIDIMsg2Send = TRUE;	// set Msg2Send flag
									break;	// then exit to send off to host
									}	// ends 2/3 byte switch
							break;
						}		
						else{		// must be a continuing SySex msg
							GetNextMIDIByte();	// so fetch 2nd char, -> buffer[1]
							if(ToSendDataBuffer[1] == 0xF7){	//if it's an F7
								ToSendDataBuffer[2] = 0;// make 3rd byte a null
								MIDI_CIN_CODE = 0x6;	// & set CIN code
						    	MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
								break;		// then exit to send it off to host
							}
							else {	//2nd byte not an F7, so fetch 3rd char
								GetNextMIDIByte();	// & -> buffer[2]
								if(ToSendDataBuffer[2] == 0xF7){	// if it's an F7
									MIDI_CIN_CODE = 0x7;// set CIN code
							    	MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
									break;	//then exit to send it off to host
									}
								else {	// 3rd byte not an F7, so must be all data
									MIDI_CIN_CODE = 0x4;// so set CIN code
						    		MIDIMsg2Send = TRUE;	// also set Msg2Send flag 
									break;	//then exit to send it off to host
									}
								}
						}	
			}	// ends main StatNibble switch statement

		// start of MIDI-->USB pkt transmit code, which only functions if one section
		// of the foregoing code has set MIDIMsg2Send flag to TRUE
		if((MIDIMsg2Send == TRUE) && (!USBHandleBusy(USBTxHandle)))
			// now if there's a USB pkt to send & the USB Tx handle isn't busy,
		{	// send off the 32-bit packet now
			midiData.CableNumber = 0;		// starting with the status byte
			midiData.CodeIndexNumber = MIDI_CIN_CODE;
       		midiData.DATA_0 = ToSendDataBuffer[0];	// then add the data bytes
       		midiData.DATA_1 = ToSendDataBuffer[1];
        	midiData.DATA_2 = ToSendDataBuffer[2];
       		USBTxHandle = USBTxOnePacket(MIDI_EP,(BYTE*)&midiData,4); // send it
			MIDIBcount = 0;	// now reset the MIDI byte counter
			MIDIMsg2Send = FALSE;	// also clear Msg2Send flag
		}					// ends section for sending USB IN pkt to host

	}	// keeps looping back for MIDI message reception, processing & sending
		// off as USB INPUT pkts (or not) while MIDI msgs are being received
}		//end ProcessIO

/**************************************************************/
void GetNextMIDIByte(void)
{
	while(PIR1bits.RCIF == 0)	// first wait until another MIDI byte
	;							// has been received (RCIF == 1)
	MIDIBcount++;		// when it has, bump up counter
	ReadMIDIINchar(); 	// then read & transfer into ToSendDataBuffer[MIDIBcount]
}	// then return

/**************************************************************/
void ReadMIDIINchar(void)
{
	ToSendDataBuffer[MIDIBcount] = RCREG; // first read & transfer byte into buffer
										  // (also clears RCIF flag)
	if(RCSTAbits.OERR)   // now check over-run error flag
	{
		RCSTAbits.CREN = 0;		// and clear CREN if flag was set
		RCSTAbits.CREN = 1;		// then set it again for next time
	}	
}	// then or otherwise return

/**************************************************************
 * Function:        void BlinkUSBStatus(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        BlinkUSBStatus turns on and off LEDs
 *                  corresponding to the USB device state.
 *
 * Note:            mLED macros can be found in HardwareProfile.h
 *                  USBDeviceState is declared and updated in
 *                  usb_device.c.
 *******************************************************************/
void BlinkUSBStatus(void)
{
    static WORD led_count=0;

    if(led_count == 0)led_count = 10000U;
    led_count--;

    #define mLED_Both_Off()         {mLED_1_Off();mLED_2_Off();}
    #define mLED_Both_On()          {mLED_1_On();mLED_2_On();}
    #define mLED_Only_1_On()        {mLED_1_On();mLED_2_Off();}
    #define mLED_Only_2_On()        {mLED_1_Off();mLED_2_On();}

    if(USBSuspendControl == 1)
    {
        if(led_count==0)
        {
            mLED_1_Toggle();
            if(mGetLED_1())
            {
                mLED_2_On();
            }
            else
            {
                mLED_2_Off();
            }
        }//end if
    }
    else
    {
        if(USBDeviceState == DETACHED_STATE)
        {
            mLED_Both_Off();
        }
        else if(USBDeviceState == ATTACHED_STATE)
        {
            mLED_Both_On();
        }
        else if(USBDeviceState == POWERED_STATE)
        {
            mLED_Only_1_On();
        }
        else if(USBDeviceState == DEFAULT_STATE)
        {
            mLED_Only_2_On();
        }
        else if(USBDeviceState == ADDRESS_STATE)
        {
            if(led_count == 0)
            {
                mLED_1_Toggle();
                mLED_2_Off();
            }//end if
        }
        else if(USBDeviceState == CONFIGURED_STATE)
        {
            if(led_count==0)
            {
                mLED_1_Toggle();
                if(mGetLED_1())
                {
                    mLED_2_Off();
                }
                else
                {
                    mLED_2_On();
                }
            }//end if
        }//end if(...)
    }//end if(UCONbits.SUSPND...)

}//end BlinkUSBStatus




// ******************************************************************************************************
// ************** USB Callback Functions ****************************************************************
// ******************************************************************************************************
// The USB firmware stack will call the callback functions USBCBxxx() in response to certain USB related
// events.  For example, if the host PC is powering down, it will stop sending out Start of Frame (SOF)
// packets to your device.  In response to this, all USB devices are supposed to decrease their power
// consumption from the USB Vbus to <2.5mA each.  The USB module detects this condition (which according
// to the USB specifications is 3+ms of no bus activity/SOF packets) and then calls the USBCBSuspend()
// function.  You should modify these callback functions to take appropriate actions for each of these
// conditions.  For example, in the USBCBSuspend(), you may wish to add code that will decrease power
// consumption from Vbus to <2.5mA (such as by clock switching, turning off LEDs, putting the
// microcontroller to sleep, etc.).  Then, in the USBCBWakeFromSuspend() function, you may then wish to
// add code that undoes the power saving things done in the USBCBSuspend() function.

// The USBCBSendResume() function is special, in that the USB stack will not automatically call this
// function.  This function is meant to be called from the application firmware instead.  See the
// additional comments near the function.

/******************************************************************************
 * Function:        void USBCBSuspend(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Call back that is invoked when a USB suspend is detected
 *
 * Note:            None
 *****************************************************************************/
void USBCBSuspend(void)
{
	//Example power saving code.  Insert appropriate code here for the desired
	//application behavior.  If the microcontroller will be put to sleep, a
	//process similar to that shown below may be used:

	PORTCbits.RC2 = 1;	// should turn off power to MIDI interface circuitry

	//ConfigureIOPinsForLowPower();
	//SaveStateOfAllInterruptEnableBits();
	//DisableAllInterruptEnableBits();
	//EnableOnlyTheInterruptsWhichWillBeUsedToWakeTheMicro();
	// above should enable at least USBActivityIF as a wake source
	//Sleep();
	//RestoreStateOfAllPreviouslySavedInterruptEnableBits();
	//this should be done in the USBCBWakeFromSuspend() function instead.
	//RestoreIOPinsToNormal();
	//this should be done in the USBCBWakeFromSuspend() function instead.

	//IMPORTANT NOTE: Do not clear the USBActivityIF (ACTVIF) bit here.  This bit is
	//cleared inside the usb_device.c file.  Clearing USBActivityIF here will cause
	//things to not work as intended.


    #if defined(__C30__)
    #if 0
        U1EIR = 0xFFFF;
        U1IR = 0xFFFF;
        U1OTGIR = 0xFFFF;
        IFS5bits.USB1IF = 0;
        IEC5bits.USB1IE = 1;
        U1OTGIEbits.ACTVIE = 1;
        U1OTGIRbits.ACTVIF = 1;
        Sleep();
    #endif
    #endif
}


/******************************************************************************
 * Function:        void _USB1Interrupt(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the USB interrupt bit is set
 *					In this example the interrupt is only used when the device
 *					goes to sleep when it receives a USB suspend command
 *
 * Note:            None
 *****************************************************************************/
#if 0
void __attribute__ ((interrupt)) _USB1Interrupt(void)
{
    #if !defined(self_powered)
        if(U1OTGIRbits.ACTVIF)
        {
            IEC5bits.USB1IE = 0;
            U1OTGIEbits.ACTVIE = 0;
            IFS5bits.USB1IF = 0;

            //USBClearInterruptFlag(USBActivityIFReg,USBActivityIFBitNum);
            USBClearInterruptFlag(USBIdleIFReg,USBIdleIFBitNum);
            //USBSuspendControl = 0;
        }
    #endif
}
#endif

/******************************************************************************
 * Function:        void USBCBWakeFromSuspend(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The host may put USB peripheral devices in low power
 *					suspend mode (by "sending" 3+ms of idle).  Once in suspend
 *					mode, the host may wake the device back up by sending non-
 *					idle state signalling.
 *
 *					This call back is invoked when a wakeup from USB suspend
 *					is detected.
 *
 * Note:            None
 *****************************************************************************/
void USBCBWakeFromSuspend(void)
{
	// If clock switching or other power savings measures were taken when
	// executing the USBCBSuspend() function, now would be a good time to
	// switch back to normal full power run mode conditions.  The host allows
	// a few milliseconds of wakeup time, after which the device must be
	// fully back to normal, and capable of receiving and processing USB
	// packets.  In order to do this, the USB module must receive proper
	// clocking (IE: 48MHz clock must be available to SIE for full speed USB
	// operation).

	PORTCbits.RC2 = 0;	// should restore power to the MIDI interface cktry

}

/********************************************************************
 * Function:        void USBCB_SOF_Handler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USB host sends out a SOF packet to full-speed
 *                  devices every 1 ms. This interrupt may be useful
 *                  for isochronous pipes. End designers should
 *                  implement callback routine as necessary.
 *
 * Note:            None
 *******************************************************************/
void USBCB_SOF_Handler(void)
{
    // No need to clear UIRbits.SOFIF to 0 here.
    // Callback caller is already doing that.

    if(msCounter != 0)
    {
        msCounter--;
    }
}

/*******************************************************************
 * Function:        void USBCBErrorHandler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The purpose of this callback is mainly for
 *                  debugging during development. Check UEIR to see
 *                  which error causes the interrupt.
 *
 * Note:            None
 *******************************************************************/
void USBCBErrorHandler(void)
{
    // No need to clear UEIR to 0 here.
    // Callback caller is already doing that.

	// Typically, user firmware does not need to do anything special
	// if a USB error occurs.  For example, if the host sends an OUT
	// packet to your device, but the packet gets corrupted (ex:
	// because of a bad connection, or the user unplugs the
	// USB cable during the transmission) this will typically set
	// one or more USB error interrupt flags.  Nothing specific
	// needs to be done however, since the SIE will automatically
	// send a "NAK" packet to the host.  In response to this, the
	// host will normally retry to send the packet again, and no
	// data loss occurs.  The system will typically recover
	// automatically, without the need for application firmware
	// intervention.

	// Nevertheless, this callback function is provided, such as
	// for debugging purposes.
}


/*******************************************************************
 * Function:        void USBCBCheckOtherReq(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        When SETUP packets arrive from the host, some
 * 					firmware must process the request and respond
 *					appropriately to fulfill the request.  Some of
 *					the SETUP packets will be for standard
 *					USB "chapter 9" (as in, fulfilling chapter 9 of
 *					the official USB specifications) requests, while
 *					others may be specific to the USB device class
 *					that is being implemented.  For example, a HID
 *					class device needs to be able to respond to
 *					"GET REPORT" type of requests.  This
 *					is not a standard USB chapter 9 request, and
 *					therefore not handled by usb_device.c.  Instead
 *					this request should be handled by class specific
 *					firmware, such as that contained in usb_function_hid.c.
 *
 * Note:            None
 *******************************************************************/
void USBCBCheckOtherReq(void)
{
}//end


/*******************************************************************
 * Function:        void USBCBStdSetDscHandler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USBCBStdSetDscHandler() callback function is
 *					called when a SETUP, bRequest: SET_DESCRIPTOR request
 *					arrives.  Typically SET_DESCRIPTOR requests are
 *					not used in most applications, and it is
 *					optional to support this type of request.
 *
 * Note:            None
 *******************************************************************/
void USBCBStdSetDscHandler(void)
{
    // Must claim session ownership if supporting this request
}//end


/*******************************************************************
 * Function:        void USBCBInitEP(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the device becomes
 *                  initialized, which occurs after the host sends a
 * 					SET_CONFIGURATION (wValue not = 0) request.  This
 *					callback function should initialize the endpoints
 *					for the device's usage according to the current
 *					configuration.
 *
 * Note:            None
 *******************************************************************/
void USBCBInitEP(void)
{
    //enable the HID endpoint
    USBEnableEndpoint(MIDI_EP,USB_OUT_ENABLED|USB_IN_ENABLED|USB_HANDSHAKE_ENABLED|USB_DISALLOW_SETUP);

    //Re-arm the OUT endpoint for the next packet
    USBRxHandle = USBRxOnePacket(MIDI_EP,(BYTE*)&ReceivedDataBuffer,64);
}

/********************************************************************
 * Function:        void USBCBSendResume(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USB specifications allow some types of USB
 * 					peripheral devices to wake up a host PC (such
 *					as if it is in a low power suspend to RAM state).
 *					This can be a very useful feature in some
 *					USB applications, such as an Infrared remote
 *					control	receiver.  If a user presses the "power"
 *					button on a remote control, it is nice that the
 *					IR receiver can detect this signalling, and then
 *					send a USB "command" to the PC to wake up.
 *
 *					The USBCBSendResume() "callback" function is used
 *					to send this special USB signalling which wakes
 *					up the PC.  This function may be called by
 *					application firmware to wake up the PC.  This
 *					function should only be called when:
 *
 *					1.  The USB driver used on the host PC supports
 *						the remote wakeup capability.
 *					2.  The USB configuration descriptor indicates
 *						the device is remote wakeup capable in the
 *						bmAttributes field.
 *					3.  The USB host PC is currently sleeping,
 *						and has previously sent your device a SET
 *						FEATURE setup packet which "armed" the
 *						remote wakeup capability.
 *
 *					This callback should send a RESUME signal that
 *                  has the period of 1-15ms.
 *
 * Note:            Interrupt vs. Polling
 *                  -Primary clock
 *                  -Secondary clock ***** MAKE NOTES ABOUT THIS *******
 *                   > Can switch to primary first by calling USBCBWakeFromSuspend()

 *                  The modifiable section in this routine should be changed
 *                  to meet the application needs. Current implementation
 *                  temporary blocks other functions from executing for a
 *                  period of 1-13 ms depending on the core frequency.
 *
 *                  According to USB 2.0 specification section 7.1.7.7,
 *                  "The remote wakeup device must hold the resume signaling
 *                  for at lest 1 ms but for no more than 15 ms."
 *                  The idea here is to use a delay counter loop, using a
 *                  common value that would work over a wide range of core
 *                  frequencies.
 *                  That value selected is 1800. See table below:
 *                  ==========================================================
 *                  Core Freq(MHz)      MIP         RESUME Signal Period (ms)
 *                  ==========================================================
 *                      48              12          1.05
 *                       4              1           12.6
 *                  ==========================================================
 *                  * These timing could be incorrect when using code
 *                    optimization or extended instruction mode,
 *                    or when having other interrupts enabled.
 *                    Make sure to verify using the MPLAB SIM's Stopwatch
 *                    and verify the actual signal on an oscilloscope.
 *******************************************************************/
void USBCBSendResume(void)
{
    static WORD delay_count;

    USBResumeControl = 1;                // Start RESUME signaling

    delay_count = 1800U;                // Set RESUME line for 1-13 ms
    do
    {
        delay_count--;
    }while(delay_count);
    USBResumeControl = 0;
}


/*******************************************************************
 * Function:        BOOL USER_USB_CALLBACK_EVENT_HANDLER(
 *                        USB_EVENT event, void *pdata, WORD size)
 *
 * PreCondition:    None
 *
 * Input:           USB_EVENT event - the type of event
 *                  void *pdata - pointer to the event data
 *                  WORD size - size of the event data
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called from the USB stack to
 *                  notify a user application that a USB event
 *                  occured.  This callback is in interrupt context
 *                  when the USB_INTERRUPT option is selected.
 *
 * Note:            None
 *******************************************************************/
BOOL USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, WORD size)
{
    switch(event)
    {
        case EVENT_CONFIGURED:
            USBCBInitEP();
            break;
        case EVENT_SET_DESCRIPTOR:
            USBCBStdSetDscHandler();
            break;
        case EVENT_EP0_REQUEST:
            USBCBCheckOtherReq();
            break;
        case EVENT_SOF:
            USBCB_SOF_Handler();
            break;
        case EVENT_SUSPEND:
            USBCBSuspend();
            break;
        case EVENT_RESUME:
            USBCBWakeFromSuspend();
            break;
        case EVENT_BUS_ERROR:
            USBCBErrorHandler();
            break;
        case EVENT_TRANSFER:
            Nop();
            break;
        default:
            break;
    }
    return TRUE;
}

/** EOF main.c *************************************************/
#endif
