


; remote controlled volume and input selection for low noise stereo amplifier
; added immunity to interference

; CPU configuration
	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc


;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _XT_OSC

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF
; Define variables at memory locations

; EEPROM
EEPROM0		equ	H'40'	; stored remote selection (TV, SAT1, SAT2)
EEPROM1		equ	H'41'	; input selection (input1, input2, input3)

; RAM
; bank 0 

REMOTE_M	equ	H'20'	; remote control input ms byte
REMOTE_L	equ	H'21'	; remote control input LS byte
REM_FLG		equ	H'22'	; flag to indicate current remote control sequencing 
REM_CNT		equ	H'23'	; interrupt count for polling remote control signal
REM_COD		equ	H'24'	; remote control bit counter mode.
TOGGLE		equ	H'25'	; remote control toggle bit
MUTE		equ	H'26'	; mute counter
TG_BIT		equ	H'27'	; toggle bit stored flag
MUTE_FG		equ	H'28'	; mute flag
VALUE_1		equ	H'29'	; delay value
VALUE_2		equ	H'2A'	; delay value
TEMP_R		equ	H'2B'	; remote code temporary
FLG_HL		equ	H'2C'	; high or low remote control sign
TIMER1		equ	H'2D'	; timer least significant byte
TIMER2		equ	H'2E'	; timer most significant byte
MUTE_OP		equ	H'2F'	; mute in operation
SLEEPT		equ	H'30'	; sleep timer
MOTOR_SLIP	equ	H'31'	; motor slippage flag
CODE_SOURCE	equ	H'32'	; code source (TV, SAT1, SAT2)
INPUT		equ	H'33'	; input
B_TEMP		equ	H'34'	; PORTB stored level
TEMP		equ	H'35'	; temporary
STORE1		equ	H'36'	; remote storage ms byte
STORE2		equ	H'37'	; remote storage ls byte
VALUE3		equ	H'38'	; delay loop
	
; all banks ram
W_TMP		equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 


	ORG     2140		; preprogram EEPROM locations
	DE		D'0'		; set at 00
	DE		B'10100000'	; input selection 1 (H A0)	

; define reset and interrupt vector start addresses

	org		0			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4			; interrupt vector 0004h, start interrupt routine here


; ************************************************************************************	
; start interrupt by saving w and status registers before altered by interrupt routine

INTERRUPT
; check interrupt source	

	btfsc	INTCON,INTF	; interrupt for RB0
	bcf		INTCON,INTF	
	btfss	INTCON,TMR0IF	; TMRO overflow interrupt flag
	retfie 				; return when not timer 0 overflow
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0
	bcf		STATUS,RP1

; adjust interrupt rate with counter

COUNTER
	movlw	D'207'		; freq is 4MHz/4/2/50. = 100us
	addwf	TMR0,f		; add to timer register and takes 2 cycles to start counting
	bcf		INTCON,TMR0IF	; clear TMRO interrupt flag

; increase timer

	incfsz	TIMER1,f	; increase value least significant byte
	goto	REMST		; start remote control
	incf	TIMER2,f	; increase ms byte

; check if RB0 has gone low for start of remote control sequence

REMST
RM_SQ
	btfsc	REM_FLG,0	; has remote control sequence started		
	goto	REM_SEQ
	btfsc	PORTB,0		; is RB0 low 
	goto	RECLAIM
	bsf		REM_FLG,0	; set beginning of remote control flag
	
	movlw	D'22'		; initial time period to wait till next start bit
	movwf	REM_CNT
	movlw	D'13'		; shift register counter
	movwf	REM_COD
	bsf		REMOTE_L,0	; set first bit in remote control sequence
	clrf	FLG_HL		; clear bit 1 in high low flag
	bsf		REM_FLG,7	; start bit flag set	
	bsf		PORTA,0		; decode
	goto 	RECLAIM

REM_SEQ
	btfsc	REM_FLG,1	; has it finished
	goto	RECLAIM
	decfsz	REM_CNT,f	; decrease interrupt counter for timing
	goto	CKEDGE		; check if change in level
	bcf		REM_FLG,7	; clear start bit flag
	movlw	D'18'		; 18 x 100us = 1.8ms or period between valid bits
	movwf	REM_CNT
	rlf		REMOTE_L,f	; least sig byte in remote control code sequence
	rlf		REMOTE_M,f	; most sig byte 
	movf	PORTB,w		; check bit 0 portB
	andlw	B'00000001'	; mask out bits except bit 0
	movwf	FLG_HL		; place in flag
	btfsc	PORTB,0		; check if high or low
	goto	HI			; high so clear
	bsf		REMOTE_L,0	; if low set this bit
	bsf		PORTA,0		; decode output
	goto	BY_HI		; bypass high
HI	
	bcf		REMOTE_L,0	; clear 0 bit
	bcf		PORTA,0		; decode
BY_HI		
	decfsz	REM_COD,f	; decrease shift register count
	goto	RECLAIM
	bsf		REM_FLG,1	; set end of remote control decoding flag
	goto	RECLAIM

; align change in level (edge detect) with REM_CNT counter
 
CKEDGE
	btfsc	REM_FLG,7	; start bit flag bypass edge detect as AGC level setting
	goto	RECLAIM
	movlw	B'00000001'	; select bit 0 for PORTB,0
	andwf	PORTB,w		; bit 0
	xorwf	FLG_HL,w	; compare PORTB,0 with FLAG (previous level)		
	btfsc	STATUS,Z	; Z=1 if zero then no change
	goto	RECLAIM		; no change 
	movlw	B'00000001'	; select bit 0 for PORTB,0
	andwf	PORTB,w		; bit 0 select 
	movwf	FLG_HL		; reload new level
	movlw	H'3'		; count of 3 for REM_CNT
	subwf	REM_CNT,w	; is REM_CNT larger or equal to 3
	btfss	STATUS,C
	goto	RECLAIM		; smaller than 3 
	movf	REM_CNT,w
	sublw	H'6'		; check if 6 or less
	btfss	STATUS,C
	goto	RECLAIM		; larger than 6 
	movlw	H'4'
	movwf	REM_CNT		; set REM_CNT to 4 to align with change in level
	
; end of interrupt reclaim w and status 

RECLAIM
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN
	movlw	D'10'
	movwf	SLEEPT		; time before sleep
	clrf	MUTE_OP		; mute in operation
	clrf	TG_BIT		; clear toggle bit stored flag
	clrf	REM_CNT		; remote control polling counter
	clrf	REM_FLG		; remote control flag status
	clrf	MUTE		; mute counter
	clrf	MUTE_FG		; mute flag
	clrf	CODE_SOURCE	; remote control encoding 0=TV, 1=SAT1, 2=SAT2
	clrf	STORE1		; remote storage
	clrf	STORE2

; set inputs/outputs
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	movlw	B'01000000'	; 
	movwf	OPTION_REG	; TMRO prescaler is 2, PORTB pullups disabled, RB0 interrupt on 0
	movlw   B'00111000'	; Inputs/ outputs 
	movwf   TRISA		; port A data direction register

; analog inputs, A/D
	movlw	B'00001000'	; AN3 is analog input
	movwf	ANSEL
	movlw	B'10000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01011000'	; Fosc, channel 3 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

	movlw	B'00011110'	; Q2,Q4 on Q1,Q3 off 
	movwf	PORTB		; motor off, terminals both low

	bcf		PORTA,2		; Acknowledge LED off
	bcf		PORTA,1		; Mute LED off

; Code selection is made with a switch closed at power up		
; check if S1, S2 or S3 closed on the front panel as connected to RB6, RB7 and RB5 respectively.
	call	DELAY		; 1s
	movf	PORTB,w
	andlw 	B'11100000'	; look at RB5 to RB7
	xorlw	B'11100000'	; if equal, then no switches are closed at power up
	btfsc	STATUS,Z	
	goto	READ_CODE_SELECT

; check which switch is low
	clrw				; set w low as default
;	btfss	PORTB,5		; S3. If low, set a 0 for TV	
;	movlw	D'0'	
	btfss	PORTB,6		; S1. If low, set 1 for SAT1
	movlw	D'1'
	btfss	PORTB,7		; S2. If low, set 2 for SAT2
	movlw	D'2'
	movwf	CODE_SOURCE	; sets TV, SAT1 or SAT2
	
	bsf		PORTA,2		; Acknowledge LED on

	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movf	CODE_SOURCE,w
	call	EEWRITE

 ; Wait for switches to open
WAIT_OPEN
	movf	PORTB,w
	andlw 	B'11100000'	; look at RB5 to RB7
	xorlw	B'11100000'	; if equal, then switches are open
	btfsc	STATUS,Z	
	goto	READ_CODE_SELECT

	call	DELAY		; 1s
	goto	WAIT_OPEN
	
READ_CODE_SELECT
	bcf		INTCON,RBIF	; clear port change interrupt flag

	movlw	EEPROM0
	call	EEREAD		; sets EEADR
	movwf	CODE_SOURCE	; remote control source (TV, SAT1, SAT2)

READ_INPUT_SELECT
	bcf		PORTA,2		; Acknowledge LED off
; read input selection (input1, input2 or input3)
	call	DELAY		; 1s

	movlw	EEPROM1
	call	EEREAD
	movwf	INPUT

; set outputs for initial selected input

; Check bit 6 for input 1
	btfss	INPUT,6		; if low then input1
	goto	INPUT1
; check bit 7 for input 2
	btfss	INPUT,7		; if low then input2
	goto	INPUT2
; check bit 5 for input 3	
	btfss	INPUT,5		; if low then input3
	goto	INPUT3
INPUT1
; set PORTB,6 as a low output
	bcf		PORTB,6		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'10100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,6		; low
	goto	INTERRUPT_ENABLE

INPUT2
; set PORTB,7 as a low output
	bcf		PORTB,7		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,7		; low
	goto	INTERRUPT_ENABLE

INPUT3
; set PORTB,5 as a low output
	bcf		PORTB,5		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11000001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,5		; low
;	goto	INTERRUPT_ENABLE

; interrupt enable 
INTERRUPT_ENABLE

	bsf		INTCON,TMR0IE; set interrupt enable for TMR0 
	bsf		INTCON,GIE	; set global interrupt enable for above
	
; decode remote control signal
; delay

CK_AGN
	
	movf	SLEEPT,w	; sleep timing
	btfsc	STATUS,Z
	goto	T_SET
	decf	SLEEPT,f	; decrease until zero
T_SET
	movlw	D'100'		; set delay period of about 120ms between codes
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1
	movwf	VALUE_2		; VALUE_2 = w
LP_2
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2

; check for pressed switch

	btfsc	INTCON,RBIF
	goto	SWITCH

; check during delay for remote control entered flag
	
	btfsc	REM_FLG,1	; is remote control entered flag set
	goto	CODE_D
	
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1

; check motor slippage
	call	ACQUIRE_AD	
	btfss	MOTOR_SLIP,0; if low stop motor
	goto	MOT_OFF		; stop motor

; remote control decoding
	
	btfsc	REM_FLG,1	; is remote control entered flag set
	goto	CODE_D		; check codes

	movlw	B'00011110'	; Q1,Q3 off Q2,Q4 on
	movwf	PORTB		; motor off
	
	movf	SLEEPT,w	; sleep timing
	btfss	STATUS,Z	; if zero then sleep	
	goto	CK_AGN

	bcf		PORTA,0		; decode
	bcf		PORTA,2		; ack.
	bcf		INTCON,GIE	; interrupt enable off so after sleep instructions continue on
						; rather than to interrupt vector.
	btfsc	INTCON,INTF	; interrupt flag for RB0
	goto	BY_SLEEP
	btfsc	INTCON,RBIF	; port change interrupt flag
	goto	BY_SLEEP
	bsf		INTCON,INTE	; set interrupt enable for RB0
	bsf		INTCON,RBIE	; set port change interrupt enable 

	sleep				; sleep mode
	nop
BY_SLEEP
	bcf		INTCON,INTE	; disable interrupt for RB0
	bcf		INTCON,RBIE	; bcf clear port change interrupt enable 
;	bsf		INTCON,GIE	; interrupt enable

; check wake from sleep source
	btfss	INTCON,INTF	; PORTB,0 
	goto	SW_CHANGE
	bcf		INTCON,INTF	; flag cleared
	bsf		INTCON,GIE	; interrupt enable
	clrf	REM_FLG		; clear any remote control flags
	goto	CK_AGN

SWITCH
	bcf		PORTA,2		; acknowledge LED off
	clrf	REM_FLG		; clear any remote control flags
SW_CHANGE
	btfss	INTCON,RBIF
	goto	CK_AGN
	bcf		INTCON,RBIF	; clear port change interrupt flag

; read switches
; set RB5,6,7 as inputs

; then set to the new switch by setting as inputs to release the low output

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0

; Input selection is with a switch closed		
; check if S1, S2 or S3 closed on the front panel as connected to RB6, RB7 and RB5 respectively.
	movlw	D'25'
	call	DELAYT		; 0.1s
	movf	PORTB,w
	movwf	B_TEMP

; check which switch is low
CK_SWS
	clrw
	btfss	B_TEMP,6	; S1. If low, set RB6 low for input1	
	movlw	B'10100000'	
	btfss	B_TEMP,7	; S2. If low, set RB7 low for input2
	movlw	B'01100000'
	btfss	B_TEMP,5	; S3. If low, set RB5 low for input3
	movlw	B'11000000'
	movwf	TEMP
	movf	TEMP,w		; if zero no switches closed so use original INPUT value 
	btfsc	STATUS,Z
	goto	SET_SELECT	; re-drive output
	movwf	INPUT		; sets input (1, 2 or 3)
	bcf		INTCON,GIE	; disable interrupt
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movf	INPUT,w
	call	EEWRITE
;	bsf		INTCON,GIE	; interrupt enable

 ; Wait for switches to open
WAIT_OPEN1
	movf	PORTB,w
	andlw 	B'11100000'	; look at RB5 to RB7
	xorlw	B'11100000'	; if equal, then switches are open
	btfss	STATUS,Z	
	goto	WAIT_OPEN1

SET_INPUT_SELECT
; read input selection (input1, inut2 or input3)
	movlw	EEPROM1
	call	EEREAD
	movwf	INPUT
	
; set outputs for initial selected input
SET_SELECT
; Check bit 6 for input 1
	btfss	INPUT,6		; if low then input1
	goto	INPUT1X
; check bit 7 for input 2
	btfss	INPUT,7		; if low then input1
	goto	INPUT2X
; check bit 5 for input 3	
	btfss	INPUT,5		; if low then input1
	goto	INPUT3X
INPUT1X
; set PORTB,6 as a low output
	bcf		PORTB,6		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'10100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,6		; low
	goto	INPUT_CYC

INPUT2X
; set PORTB,7 as a low output
	bcf		PORTB,7		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,7		; low
	goto	INPUT_CYC

INPUT3X
; set PORTB,5 as a low output
	bcf		PORTB,5		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11000001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,5		; low
;	goto	INPUT_CYC

INPUT_CYC
	movlw	D'25'
	call	DELAYT		; 0.1s
	bcf		INTCON,RBIF	; clear port change interrupt flag	
	goto	CK_AGN


; Philips RC5 code for infra red transmission (also used with Marantz, Grundig and Loewe)
; Comprises 2 start bits, 1 toggle bit (alternates high or low on successive same key press)
; 5-system address bits and 6 command bits. Data stream starts with the start bits. 
; biphase encoding with high to a low transition = a low and low to high = high
; each bit transmitted at 1.778ms rate. Whole code at 24.889ms and next code starts 113.778ms
; later. Can decode by detecting first start bit then waiting 1.778ms + 444.5us for next start
; bit. Successive bits are 1.778ms apart. Total of 14 bits. 

;Address
;0 TV1
;1 TV2
;2 Videotext
;3 Expansion for TV1 and TV2
;4 Laser vision player
;5 VCR1
;6 VCR2
;7 Reserved
;8 SAT1
;9 Expansion for VCR1 and VCR2
;10 SAT2
;11 Reserved
;12 CD Video
;13 Reserved
;14 CD Photo
;15 Reserved
;16 Audio preamplifier1
;17 Tuner
;18 Analogue cassetter recorder
;19 Audio Preamp2
;20 CD
;21 Audio rack
;22 Audio Satellite
;23 DCC Recorder
;24 Reserved
;25 Reserved
;26 Writable CD
;27-31 Reserved

;Keycode commands 
;0 0
;1 1
;2 2
;3 3
;4 4
;5 5
;6 6
;7 7
;8 8
;9 9
;16 Volume +
;17 Volume -
;18 Brightness +
;19 Brightness -
;20 Colour saturation +
;21 Colour Saturation -
;22 Bass +
;23 Bass -
;24 Treble +
;25 Treble -
;26 Balance right
;27 balance left
;63 System select
;71 Dim local display

; Special commands
;10 1/2/3 digits /10
;11 Freq/prog/ch11
;12 standby
;13 Mute
;14 Personal preference
;15 Display
;28 Contrast +
;29 Contrast -
;30 Search +
;31 Tint/hue
;32 CH/prog +
;33 CH/prog -
;34 Alternate CH

;37 stereo/mono
;39 Tint/hue +
;48 Pause
;50 rewind
;52 wind
;53 play
;54 stop
;55 record

; compare with address code out if invalid clear REM_FLG
; if valid check command bits and ls 2 address bits then clear REM_FLG
; but first check which code is selected 
	
CODE_D
	bcf		PORTA,0		; decode out
	movlw	D'10'
	movwf	SLEEPT		; time before sleep


; compare remote code with last received. Requires 2 matching consecutive codes before an an action occurs
;	movf	REMOTE_M,w	; most significant remote code 
;	andlw	B'00110111'	; mask out bit 7 and 6 and toggle bit
;	movwf	TEMP
;	movf	STORE1,w
;	andlw	B'00110111' ; mask bit 6 and 7
;	xorwf	TEMP,w
;	btfsc	STATUS,Z
;	goto	NEXT_BYTE	; the same codes so check next code byte

TRANSFER ; transfer values when not the same
	
;	movf	REMOTE_M,w	; transfer codes
;	movwf	STORE1	
;	movf	REMOTE_L,w
;	movwf	STORE2
;	goto	CLR_RMF

; compare next byte	
NEXT_BYTE
; compare remote code with last received. Requires 2 matching consecutive codes before an an action occurs
;	movf	REMOTE_L,w	; most significant remote code 
;	xorwf	STORE2,w
;	btfss	STATUS,Z
;	goto	TRANSFER	; not the same values

; matching consecutive codes
; check code source
	movf	CODE_SOURCE,w ; TV, SAT1, SAT2
	btfsc	STATUS,Z
	goto	TV_1
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	SAT1		
	goto 	SAT_2

; code for television (TV1)

TV_1
	movf	REMOTE_M,w	; most significant remote code 
	andlw	B'00110111'	; mask out bit 7 and 6 and toggle bit
	
	xorlw	B'00110000'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag
	
	movf	REMOTE_L,w
	xorlw	B'00010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'00010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET  
	movf	REMOTE_L,w
	xorlw	B'00001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	MUTE_ST
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'00100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP
	movf	REMOTE_L,w
	xorlw	B'00100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 
; input
	movf	REMOTE_L,w
	xorlw	B'00000001'	; bits 0-5 keycode (1), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'00000010'	; bits 0-5 keycode (2), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'00000011'	; bits 0-5 keycode (3), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	goto	CLR_RMF


; satellite
	
SAT1
	movf	REMOTE_M,w	; most significant remote code 
	andlw	B'00110111'	; mask out bit 7 and 6 and toggle bit
	
	xorlw	B'00110010'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag

	movf	REMOTE_L,w	;  least significant remote code
	xorlw	B'00010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'00010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET
	movf	REMOTE_L,w
	xorlw	B'00001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	MUTE_ST 
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'00100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP
	movf	REMOTE_L,w
	xorlw	B'00100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 	
; input
	movf	REMOTE_L,w
	xorlw	B'00000001'	; bits 0-5 keycode (1), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'00000010'	; bits 0-5 keycode (2), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'00000011'	; bits 0-5 keycode (3), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	goto	CLR_RMF

; satellite
SAT_2

	movf	REMOTE_M,w	; most significant remote code 
	andlw	B'00110111'	; mask out bit 7 and 6 and toggle bit

	xorlw	B'00110010'	; compare with start bits and ms 3-bits of address
	btfss	STATUS,Z	; if zero then matching
	goto	CLR_RMF		; clear remote flag
	
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'10010000'	; bits 0-5 keycode (16), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	UP_SET
	movf	REMOTE_L,w
	xorlw	B'10010001'	; bits 0-5 keycode (17), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	DN_SET 
	movf	REMOTE_L,w
	xorlw	B'10001101'	; bits 0-5 keycode (13), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	MUTE_ST  
	movf	REMOTE_L,w	; least sig remote code
	xorlw	B'10100000'	; bits 0-5 keycode (32), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_UP
	movf	REMOTE_L,w
	xorlw	B'10100001'	; bits 0-5 keycode (33), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	REM_DN 
; input
	movf	REMOTE_L,w
	xorlw	B'10000001'	; bits 0-5 keycode (1), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'10000010'	; bits 0-5 keycode (2), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	movf	REMOTE_L,w
	xorlw	B'10000011'	; bits 0-5 keycode (3), bit 7 and bit 6 are ls address bits
	btfsc	STATUS,Z
	goto	INPUT_CHNG
	goto	CLR_RMF

MOT_OFF
	movlw	B'00011110'	; Q1,Q3 off Q2,Q4 on
	movwf	PORTB		; motor off
	bcf		PORTA,2		; acknowledge LED off
	goto	CK_AGN
CLR_RMF	
	movlw	D'6'		; set delay period of about 1.5ms 
	call	DELAYT		; delay
	bcf		PORTA,2		; acknowledge LED off
	clrf	REM_FLG		; remote flag cleared
	goto	MOT_OFF

; up volume. 

UP_SET	
;	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	bcf		MUTE_FG,1	; mute flag off
	bcf		PORTA,1		; Mute LED off
	bsf		PORTB,4		; Q3 off
	bsf		PORTB,1		; Q4 on	
	bcf		PORTB,2		; Q2 off
 	bcf		PORTB,3		; Q1 on

; motor on (up)	
CK_DEL
	movlw	D'10'		; set delay period  
	call	DELAYT		; delay
	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	clrf	MUTE_OP		; mute operation off
	call	FLASH_ACK_LED	
	goto	CK_AGN

DN_SET	
;	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	bcf		MUTE_FG,1	; mute flag off
	bcf		PORTA,1		; Mute LED off
	bsf		PORTB,3		; Q1 off
	bsf		PORTB,2		; Q2 on
	bcf		PORTB,1		; Q4 off
 	bcf		PORTB,4		; Q3 on
;
; motor on (down)
	clrf	MUTE_OP		; mute operation off	
	goto	CK_DEL

REM_UP
;	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	bcf		MUTE_FG,1	; mute flag off
	bcf		PORTA,1		; Mute LED off
	bsf		PORTB,4		; Q3 off
	bsf		PORTB,1		; Q4 on	
	bcf		PORTB,2		; Q2 off
 	bcf		PORTB,3		; Q1 on

; motor on (up)	
CK_DLY
	movlw	D'50'		; set delay period  
	call	DELAYT		; delay
	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	clrf	MUTE_OP		; mute operation off
	movlw	B'00011110'	; Q1,Q3 off Q2,Q4 on
	movwf	PORTB		; motor on
	call	FLASH_ACK_LED	
	goto	CK_AGN

REM_DN	
;	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	bcf		MUTE_FG,1	; mute flag off
	bcf		PORTA,1		; Mute LED off
	bsf		PORTB,3		; Q1 off
	bsf		PORTB,2		; Q2 on
	bcf		PORTB,1		; Q4 off
 	bcf		PORTB,4		; Q3 on

; motor on (down)
	clrf	MUTE_OP		; mute operation off
	goto	CK_DLY

; Mute operation
	
MUTE_ST
	
	btfsc	MUTE_OP,0	; is mute already in operation
	goto	MUTE_B		; bypass mute toggle when muting process in action
	bsf		MUTE_OP,0	; mute in operation
	call	FLASH_ACK_LED
;	movlw	D'10'		; set delay period  
;	call	DELAYT		; delay

	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	clrf	TIMER1
	clrf	TIMER2		; clear timers

	btfss	MUTE_FG,1	; if set clear if clear set
	goto	ST_FG1
	bcf		MUTE_FG,1	; mute off
	bcf		PORTA,1		; Mute LED off

	bsf		PORTB,4		; Q3 off
	bsf		PORTB,1		; Q4 on	
	bcf		PORTB,2		; Q2 off
 	bcf		PORTB,3		; Q1 on

; motor on (up)

DELY
	movlw	D'200'		; set delay period when motor first switched on
DEL2
	call	DELAYT		; delay
	bcf		PORTA,2		; acknowledge LED off
	
; check motor slippage
	call	ACQUIRE_AD
	
	btfss	MOTOR_SLIP,0; if low stop motor
	goto	MOT_M		; stop motor
	btfss	PORTA,4		; if RA4 low then do not mute off
	goto	MOT_M
	movf	MUTE,f		; count down time for motor to wind up
	btfsc	STATUS,Z	; if zero then stop
	goto	MOT_M
	decfsz	MUTE,f		; decrease until zero
	goto	NOT_0
	goto	MOT_M		; end of mute
NOT_0
	btfss	MUTE,2		; flashing LED
	goto	SET_LED	
	bcf		PORTA,1		; LED off
	goto	RCE			; remote control entered
SET_LED
	bsf		PORTA,1		; Mute LED on
RCE	btfss	TIMER2,5	; .8 seconds
	goto	DELY_C
	btfsc	REM_FLG,1	; is remote control entered flag set
	goto	CODE_D
	movlw	D'47'		; delay period between mute counting
	goto	DEL2
MOT_M
	bcf		PORTA,1		; Mute LED off when MUTE finished
	clrf	MUTE_OP		; mute operation complete
	goto	CLR_RMF
	
DELY_C
	clrf	REM_FLG		; remote flag cleared
	movlw	D'47'		; delay period between mute counting
	goto	DEL2

ST_FG1
	bsf		MUTE_FG,1	; mute flag set
	bsf		PORTA,1		; Mute LED on
	clrf	MUTE		; mute counter cleared

; start motor down

	bsf		PORTB,3		; Q1 off
	bsf		PORTB,2		; Q2 on
	bcf		PORTB,1		; Q4 off
 	bcf		PORTB,4		; Q3 on

; motor on (down)		
; DELAY
	movlw	D'200'		; set delay period when motor switched on
DEL1
	call	DELAYT		; delay
	bcf		PORTA,2		; acknowledge LED off

; check motor slippage
	call	ACQUIRE_AD
		
	btfss	MOTOR_SLIP,0; if low stop motor
	goto	MOT_OM		; stop motor
	incfsz	MUTE,f		; count up time for motor to wind down
	goto	MU_LED
	bcf		PORTA,1		; timeout mute off
	goto	CLR_RMF		; time out as MUTE=0 so stop
MU_LED
	btfss	MUTE,2		; flashing LED
	goto	CLR_LED	
	bsf		PORTA,1		; LED on
	goto	RCEF		; remote control entered flag?
CLR_LED
	bcf		PORTA,1		; Mute LED off
RCEF
	btfss	TIMER2,5	; 0.8 seconds timer
	goto	DEL_C
	btfsc	REM_FLG,1	; is remote control entered flag set
	goto	CODE_D
	movlw	D'47'		; delay period between mute counting
	goto	DEL1
MOT_OM
	clrf	MUTE_OP		; mute operation complete
	bsf		PORTA,1		; LED on during MUTE
	goto	CLR_RMF

DEL_C
	clrf	REM_FLG		; remote flag cleared
	movlw	D'47'		; delay period between mute counting
	goto	DEL1

MUTE_B
	call	FLASH_ACK_LED
	bcf		MUTE_FG,1	; clear mute flag
	bcf		PORTA,1		; mute LED off
	movlw	B'00011110'	; Q1,Q3 off Q2,Q4 on
	movwf	PORTB		; motor off
	movlw	D'255'		; set delay period  
	call	DELAYT		; delay
	movlw	D'255'		; set delay period  
	call	DELAYT		; delay
	clrf	REM_FLG		; remote flag cleared
	bcf		PORTA,2		; acknowledge LED off
	clrf	MUTE_OP		; mute operation off
	goto	CK_AGN

FLASH_ACK_LED
; flash ACK LED
	movlw	D'5'
	movwf	VALUE3		; number of flashes
FLASH_CYCLE1
	movlw	D'10'		; flash period
; LED_PERIOD
	call	DELAYT
	btfss	PORTA,2		; ACK LED
	goto	SET_ACK1
	bcf		PORTA,2
	goto	FLASH1
SET_ACK1
	bsf		PORTA,2
FLASH1
	decfsz	VALUE3,f
	goto	FLASH_CYCLE1
	bcf		PORTA,2
	return

INPUT_CHNG
; change input selection
		
	movf	REMOTE_L,w	; key code 0,1 or 2
	andlw	B'00111111'	; remove the address bits 6&7
	movwf	TEMP

; check for a 1
	movf	TEMP,w
	xorlw	D'01'
	btfss	STATUS,Z	; 1. If 1, set bit6 low for input1
	goto	CK_2	
	movlw	B'10100000'	
	goto	INPUT_VAL
; Check for a 2
CK_2
	movf	TEMP,w
	xorlw	D'02'
	btfss	STATUS,Z	; 2. If 2, set bit7 low for input2
	goto	CK_3
	movlw	B'01100000'
	goto	INPUT_VAL
; Check for a 3
CK_3
	movf	TEMP,w
	xorlw	D'03'
	btfss	STATUS,Z	; 3. If 3, set bit5 low for input3
	goto	CLR_RMF		; not 1,2 or 3
	movlw	B'11000000'
;	goto	INPUT_VAL

INPUT_VAL
	movwf	INPUT		; sets input (1, 2 or 3)
	bsf		PORTA,2		; Ack LED
; store value
	bcf		INTCON,GIE	; disable interrupt
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movf	INPUT,w
	call	EEWRITE
	bsf		INTCON,GIE	; interrupt enable

 ; read input selection (input1, input2 or input3)
	movlw	EEPROM1
	call	EEREAD
	movwf	INPUT

; set outputs for initial selected input

; Check bit 6 for input 1
	btfss	INPUT,6		; if low then input1
	goto	INPUT1Y
; check bit 7 for input 2
	btfss	INPUT,7		; if low then input2
	goto	INPUT2Y
; check bit 5 for input 3	
	btfss	INPUT,5		; if low then input3
	goto	INPUT3Y
	goto	CLR_RMF		; not 1,2 or 3

INPUT1Y
; set PORTB,6 as a low output
	bcf		PORTB,6		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'10100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,6		; low
	bcf		INTCON,RBIF
	goto	ACK_LED1

INPUT2Y
; set PORTB,7 as a low output
	bcf		PORTB,7		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01100001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,7		; low
	bcf		INTCON,RBIF
	goto	ACK_LED1

INPUT3Y
; set PORTB,5 as a low output
	bcf		PORTB,5		; low
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'11000001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	bcf		STATUS,RP0	; select memory bank 0
	bcf		PORTB,5		; low
	bcf		INTCON,RBIF
;	goto	ACK_LED1

ACK_LED1	

; flash ACK LED
	movlw	D'10'
	movwf	VALUE3		; number of flashes
FLASH_CYCLE
	movlw	D'20'		; flash period
LED_PERIOD
	call	DELAYT
	btfss	PORTA,2		; ACK LED
	goto	SET_ACK
	bcf		PORTA,2
	goto	FLASH
SET_ACK
	bsf		PORTA,2
FLASH
	decfsz	VALUE3,f
	goto	FLASH_CYCLE
	goto	CLR_RMF

; **********************************************************

; subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'40'		; 200mV using ls 10 bits
	bsf		MOTOR_SLIP,0; set flag for under value
	btfss	STATUS,C	; if C = 0 then greater than 200mV then stop motor
	clrf	MOTOR_SLIP	; clear flag

; check ms bits
	movf	ADRESH,w
	btfss	STATUS,Z	; if zero keep flag setting. If not zero clear MOTOR_SLIP to stop motor
	clrf	MOTOR_SLIP
	return

; delay subroutine
DELAY	
	movlw	D'255'		; about 1s
DELAYT
	movwf	VALUE_1		; VALUE_1 = w
LP_X
	movlw	D'255'		; set delay period value 2 
	movwf	VALUE_2		; VALUE_2 = w
LP_Y
	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_Y
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_X
	return
	
; subroutine to read EEPROM memory 

EEREAD
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM

; EEPROM write 

EEWRITE
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf		STATUS,RP0	; bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE1	

	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0
	bcf		STATUS,RP1	; bank 0 
	return				; value written
	
	end


	
