
#include "targetver.h"
#include "WindowDrawing.h"
#include "Win32Utilities.h"
#include "Sequence.h"
#include "WAVBrowse.h"
#include "Playback.h"
#include "LoadSave.h"
#include "Dialogs.h"
#include "ToolbarsMenus.h"
#include <math.h>


int horiz_div_pos = 64, zoom = 8;
int playback_pos = -1, select_pos, selection_start = -1, selection_finish = -1, playback_region_start = -1, playback_region_finish = -1;
unsigned long selected_lights, current_light;
double lfGamma = 1.6;
double lfInvGamma = 1.0/1.6;
RECT rcWAV = { 64, 0, 0, 64 };
RECT rcSeq = { 64, 64, 0, 96 };
RECT rcLightBar;
HFONT hFont, hSmallFont;
playback_status last_playback_status;
int last_playback_seconds;
int last_selection_start, last_selection_finish;
DWORD EnabledLights = 0xFFFFFFFF, NumEnabledLights = 32;

sequence_state playback_ss[32];


void InvalidateLightSpan(HWND hWnd, unsigned long lights, int left, int right) {
	if( left < right ) {
		RECT rcInvalidate;
		rcInvalidate.left = left;
		rcInvalidate.right = right;
		int first = -1;
		int index = 0;
		for( int i = 0; i < 32; ++i ) {
			if( !(EnabledLights&(1<<i)) )
				continue;
			if( lights&(1<<i) ) {
				if( first == -1 )
					first = index;
			} else if( first != -1 ) {
				rcInvalidate.top    = rcSeq.top + first * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
				rcInvalidate.bottom = rcSeq.top + (index+1) * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
				InvalidateRect(hWnd, &rcInvalidate, true);
				first = -1;
			}
			++index;
		}
		if( first != -1 ) {
			rcInvalidate.top    = rcSeq.top + first * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
			rcInvalidate.bottom = rcSeq.bottom;
			InvalidateRect(hWnd, &rcInvalidate, true);
		}
	}
}

void InvalidateSelectionRange(HWND hWnd, int start, int finish, unsigned long lights) {
	if( start == -1 )
		return;
	RECT rcInvalidate = rcWAV;
	SCROLLINFO si;
	si.cbSize = sizeof (si);
	si.fMask  = SIF_ALL;
	GetScrollInfo (hWnd, SB_HORZ, &si);
	int x1 = rcWAV.left + (start >> zoom) - si.nPos;
	int x2 = rcWAV.left + (finish >> zoom) - si.nPos + 1;
	// when there is no WAV content, the text drawn in that area can extend past the right of the data
	// so extend the redraw area to cover it if necessary.
	if( !wav_content && x2 >= (wav_len>>zoom)-si.nPos - 100 )
		x2 += 100;
	if( x2 < rcInvalidate.left || x1 > rcInvalidate.right )
		return;
	rcInvalidate.left = max(rcInvalidate.left, x1);
	rcInvalidate.right = min(rcInvalidate.right, x2);
	if( rcInvalidate.right == rcInvalidate.left )
		++rcInvalidate.right;
	InvalidateRect(hWnd, &rcInvalidate, false);
	if( lights )
		InvalidateLightSpan(hWnd, lights, rcInvalidate.left, rcInvalidate.right+1);
}

void InvalidateLightRange(HWND hWnd, int start, int finish, unsigned long lights) {
	if( start == -1 )
		return;
	RECT rcInvalidate = rcWAV;
	SCROLLINFO si;
	si.cbSize = sizeof (si);
	si.fMask  = SIF_ALL;
	GetScrollInfo (hWnd, SB_HORZ, &si);
	int x1 = rcWAV.left + (start >> zoom) - si.nPos;
	int x2 = rcWAV.left + (finish >> zoom) - si.nPos + 1;
	if( x2 < rcInvalidate.left || x1 > rcInvalidate.right )
		return;
	rcInvalidate.left = max(rcInvalidate.left, x1);
	rcInvalidate.right = min(rcInvalidate.right, x2);
	if( rcInvalidate.right == rcInvalidate.left )
		++rcInvalidate.right;
	if( lights )
		InvalidateLightSpan(hWnd, lights, rcInvalidate.left, rcInvalidate.right+1);
}

void InvalidateSelectionDiff(HWND hWnd, int start, int finish, int old_start, int old_finish, unsigned long lights) {
	if( old_start == -1 ) {
		InvalidateSelectionRange(hWnd, start, finish, lights);
	} else if( start == -1 ) {
		InvalidateSelectionRange(hWnd, old_start, old_finish, lights);
	} else {
		if( old_start < start ) {
			InvalidateSelectionRange(hWnd, old_start, start-(1<<zoom), lights);
		} else if( old_start > start ) {
			InvalidateSelectionRange(hWnd, start, old_start-(1<<zoom), lights);
		}
		if( old_finish < finish ) {
			InvalidateSelectionRange(hWnd, old_finish, finish, lights);
		} else if( old_finish > finish ) {
			InvalidateSelectionRange(hWnd, finish, old_finish, lights);
		}
	}
}

void InvalidateLightBar(HWND hWnd) {
	InvalidateRect(hWnd, &rcLightBar, FALSE);
}

void InvalidateLightBarLights(HWND hWnd, int sample_pos) {
	int ms = (sample_pos * 1000LL / wav_sample_rate);

	int dim = rcLightBar.bottom - rcLightBar.top - 6;
	int x = 8;
	int y = rcLightBar.top + 2;
	for( int i = 0; i < 32; ++i ) {
		if( !(EnabledLights&(1<<i)) )
			continue;
		if( !playback_ss[i].ms_remain || !(ms >= playback_ss[i].ms_pos && ms < playback_ss[i].ms_pos + playback_ss[i].ms_remain && !playback_ss[i].ramp) ) {
			RECT rcInvalidate;
			rcInvalidate.left = x+2;
			rcInvalidate.right = x+dim-1;
			rcInvalidate.top = y+2;
			rcInvalidate.bottom = y+dim-1;
			InvalidateRect(hWnd, &rcInvalidate, FALSE);
		}

		x += dim + 8;
	}
}

void InvalidateStatus(HWND hWnd) {
	playback_status status = get_playback_status();
	int seconds = wav_sample_rate ? playback_get_current_sample() / wav_sample_rate : 0;

	if( status != last_playback_status || seconds != last_playback_seconds || selection_start != last_selection_start || selection_finish != last_selection_finish ) {
		last_playback_status = status;
		last_playback_seconds = seconds;
		last_selection_start = selection_start;
		last_selection_finish = selection_finish;

		RECT rcInvalidate;
		rcInvalidate.left = 0;
		rcInvalidate.top = 0;
		rcInvalidate.right = rcWAV.left;
		rcInvalidate.bottom = rcWAV.bottom;
		InvalidateRect(hWnd, &rcInvalidate, false);
	}
}

void MovePlaybackPos(HWND hWnd, int pos) {
	if( playback_pos != pos ) {
		RECT rcInvalidate;
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (hWnd, SB_HORZ, &si);
		int x1 = rcWAV.left + (pos >> zoom) - si.nPos;
		int x2 = rcWAV.left + (playback_pos >> zoom) - si.nPos;
		playback_pos = pos;
		if( x1 != x2 ) {
			rcInvalidate = rcWAV;
			rcInvalidate.bottom = rcSeq.bottom;
			rcInvalidate.left = max(rcInvalidate.left, x1);
			rcInvalidate.right = min(rcInvalidate.right, x1+1);
			InvalidateRect(hWnd, &rcInvalidate, false);
			rcInvalidate = rcWAV;
			rcInvalidate.bottom = rcSeq.bottom;
			rcInvalidate.left = max(rcInvalidate.left, x2);
			rcInvalidate.right = min(rcInvalidate.right, x2+1);
			InvalidateRect(hWnd, &rcInvalidate, false);
		}
	}
}

extern HWND hDialog;

void SetSelection(HWND hWnd, int start, int finish) {
	if( !sequence_loaded )
		return;
	if( hDialog && !PreChangeSelection(hDialog) ) {
		MessageBox(hWnd, L"You can not change the selection while a dialog is open. Please close it first.", L"Can't make changes with dialog open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(hDialog);
		return;
	}

	bool bHadSelection = (selection_start != -1);

	if( !(start == -1 && finish == -1) ) {
		if( start < 0 )
			start = 0;
		else if( start > wav_len )
			start = wav_len;
		if( finish < 0 )
			finish = 0;
		else if( finish > wav_len )
			finish = wav_len;

		if( finish < start )
			swap(finish, start);

		// round to the nearest millisecond since that's the sequencing precision
		int start_ms = (int)((start + wav_sample_rate / 2000) * 1000LL / wav_sample_rate);
		int finish_ms = (int)((finish + wav_sample_rate / 2000) * 1000LL / wav_sample_rate);
		start = start_ms * (long long)wav_sample_rate / 1000;
		finish = finish_ms * (long long)wav_sample_rate / 1000;

		if( start < 0 )
			start = 0;
		else if( start > wav_len )
			start = wav_len;
		if( finish < 0 )
			finish = 0;
		else if( finish > wav_len )
			finish = wav_len;
	}

	if( start != selection_start || finish != selection_finish ) {
		InvalidateSelectionDiff(hWnd, start, finish, selection_start, selection_finish, selected_lights);
		selection_start = start;
		selection_finish = finish;
	}
	if( hDialog )
		SendMessage(hDialog, WM_POSTCHANGESELECTION, 0, 0);
	DelayedUpdateDisabledState(hWnd);
	UpdatePasteSpecialMenu(hWnd);
	if( (selection_start != -1) != bHadSelection )
		UpdateSelectionStorageMenu(hWnd);
	InvalidateStatus(hWnd);
}

void ClearSelections(HWND hWnd) {
	SetSelection(hWnd, -1, -1);
	for( int i = 0; i < sizeof(selection_storage)/sizeof(*selection_storage); ++i ) {
		selection_storage[i].start = selection_storage[i].finish = -1;
		selection_storage[i].selected_lights = selection_storage[i].current_light = 0;
	}
}

void MoveSelection(HWND hWnd, int amount) {
	if( selection_start != -1 ) {
		SCROLLINFO si;

		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (hWnd, SB_HORZ, &si);

		bool bSelectionInView = selection_start >= (int)((si.nPos - 10) << zoom) && selection_finish < (int)(((si.nPos + 10) + si.nPage + 1) << zoom);

		if( selection_start + amount < 0 )
			amount = -selection_start;
		else if( selection_finish + amount > wav_len )
			amount = wav_len - selection_finish;
		if( amount ) {
			SetSelection(hWnd, selection_start + amount, selection_finish + amount);
			if( bSelectionInView ) {
				bool bSelectionStillInView = selection_start >= (int)(si.nPos << zoom) && selection_finish < (int)((si.nPos + si.nPage + 1) << zoom);
				if( !bSelectionStillInView ) {
					HorizScrollWindowBy(hWnd, amount >> zoom);
				}
			}
		}
	}
}

void MoveLightSelection(HWND hWnd, int dir, bool bOr) {
	int i, j = -1;
	for( i = 0; i < 32; ++i )
		if( EnabledLights&selected_lights&(1<<i) )
			break;
	if( i < 32 ) {
		if( dir < 0 ) {
			for( j = i-1; j >= 0; --j )
				if( EnabledLights&(1<<j) )
					break;
		} else if( dir > 0 ) {
			for( j = i+1; j < 32; ++j )
				if( EnabledLights&(1<<j) )
					break;
		}
	}
	if( j >= 0 && j < 32 ) {
		unsigned long new_selected_lights = 0, old_selected_lights = selected_lights, new_current_light = -1;
		do {
			if( i == current_light )
				new_current_light = j;
			new_selected_lights |= 1<<j;
			old_selected_lights &= ~(1<<i);
			for( ++i; i < 32; ++i )
				if( EnabledLights&selected_lights&(1<<i) )
					break;
			for( ++j; j < 32; ++j )
				if( EnabledLights&(1<<j) )
					break;
		} while( i < 32 );

		if( new_selected_lights != selected_lights )
			SetLightSelection(hWnd, new_selected_lights | (bOr ? selected_lights : 0), new_current_light);
	}
}

void SetLightSelection(HWND hWnd, unsigned long selection, unsigned long new_current_light) {
	if( hDialog && !PreChangeSelection(hDialog) ) {
		MessageBox(hWnd, L"You can not change the selection while a dialog is open. Please close it first.", L"Can't make changes with dialog open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(hDialog);
		return;
	}

	unsigned long diff = selected_lights ^ selection;
	if( new_current_light != current_light ) {
		diff |= 1<<current_light;
		diff |= 1<<new_current_light;
		current_light = new_current_light;
	}
	if( diff ) {
		SCROLLINFO si;

		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (hWnd, SB_HORZ, &si);

		int select_x1 = rcSeq.left + (selection_start >> zoom) - si.nPos;
		int select_x2 = rcSeq.left + (selection_finish >> zoom) - si.nPos + 1;
		if( select_x1 < rcSeq.left )
			select_x1 = rcSeq.left;
		else if( select_x1 > rcSeq.right )
			select_x1 = rcSeq.right;
		if( select_x2 < rcSeq.left )
			select_x2 = rcSeq.left;
		else if( select_x2 > rcSeq.right )
			select_x2 = rcSeq.right;

		selected_lights = selection;
		InvalidateLightSpan(hWnd, diff, select_x1, select_x2);
		InvalidateLightSpan(hWnd, diff, 1, rcSeq.left-1);
	}
	if( hDialog )
		SendMessage(hDialog, WM_POSTCHANGESELECTION, NULL, NULL);
}

void SetPlaybackRegion(HWND hWnd, int start, int finish) {
	if( finish < start )
		swap(finish, start);
	if( start > wav_len )
		start = wav_len;
	if( finish > wav_len )
		finish = wav_len;
	if( start != playback_region_start || finish != playback_region_finish ) {
		InvalidateSelectionDiff(hWnd, start, finish, playback_region_start, playback_region_finish, 0);
		playback_region_start = start;
		playback_region_finish = finish;

		DelayedUpdateDisabledState(hWnd);
	}
}

bool bSetScrollExtentNested;
void SetScrollExtent(HWND hWnd, int old_zoom, int mousex, bool bUpdateSelectPos) {
	SCROLLINFO si;
	int old_pos, new_pos;

	if( bSetScrollExtentNested )
		return;

	if( mousex == -1 ) {
		mousex = (rcWAV.right - rcWAV.left) / 2;
	} else {
		mousex -= rcWAV.left;
	}

	si.cbSize = sizeof (si);
	si.fMask  = SIF_ALL;
    GetScrollInfo (hWnd, SB_HORZ, &si);
	old_pos = (si.nPos + mousex) << old_zoom;
	int select_pos_samples = ((select_pos - rcWAV.left) + si.nPos) << old_zoom;
	if( old_pos >= wav_len )
		new_pos = wav_len;
	else
		new_pos = (old_pos >> zoom) - mousex;
	int extent = (wav_len>>zoom)-3;
	if( new_pos < 0 )
		new_pos = 0;
	else if( new_pos > extent - (rcWAV.right - rcWAV.left - 1) + 1 )
		new_pos = extent - (rcWAV.right - rcWAV.left - 1) + 1;

	// Set the horizontal scrolling range and page size. 
	si.cbSize = sizeof(si);
    si.fMask  = SIF_RANGE | SIF_PAGE | SIF_POS;
    si.nMin   = 0;
    si.nMax   = extent;
    si.nPage  = rcWAV.right - rcWAV.left - 1;
	si.nPos   = new_pos;
	bSetScrollExtentNested = true;
    SetScrollInfo(hWnd, SB_HORZ, &si, TRUE); 
	bSetScrollExtentNested = false;

	if( bUpdateSelectPos )
		select_pos = ((select_pos_samples + (zoom/2)) >> zoom) - si.nPos + rcWAV.left;

	SetTimer(hWnd, 4, 250, NULL);
}

void HorizScrollWindowBy(HWND hWnd, int scroll, bool bUpdateSelectPos) {
	SCROLLINFO si;
	RECT rcScroll = rcWAV;
	rcScroll.bottom = rcSeq.bottom;
	si.cbSize = sizeof (si);
	si.fMask  = SIF_ALL;
	GetScrollInfo (hWnd, SB_HORZ, &si);

	if( scroll < (int)-si.nPos )
		scroll = -(int)si.nPos;
	else if( (int)si.nPos+scroll > (int)(si.nMax - si.nPage) )
		scroll = (int)(si.nMax - si.nPage) - (int)si.nPos;
	si.nPos += scroll;
	si.fMask  = SIF_POS;
	SetScrollInfo (hWnd, SB_HORZ, &si, true);
	ScrollWindow(hWnd, -scroll, 0, &rcScroll, &rcScroll);
	if( bUpdateSelectPos )
		select_pos -= scroll;
}



static void DrawOverlappedVerticalLines(HDC hdc, int x, HPEN bgpen, HPEN l1pen, int l1y1, int l1y2, HPEN l2pen, int l2y1, int l2y2) {
	MoveToEx(hdc, x, rcWAV.top, NULL);
	SelectObject(hdc, bgpen);
	if( l1y1 < l2y1 ) {
		LineTo(hdc, x, l1y1);
		SelectObject(hdc, l1pen);
		if( l1y2 < l2y1 ) {
			LineTo(hdc, x, l1y2);
			if( l2y1 > l1y2 ) {
				SelectObject(hdc, bgpen);
				LineTo(hdc, x, l2y1);
			}
			SelectObject(hdc, l2pen);
			LineTo(hdc, x, l2y2);
		} else {
			LineTo(hdc, x, l2y1);
		}
	} else {
		LineTo(hdc, x, l2y1);
	}
	SelectObject(hdc, l2pen);
	LineTo(hdc, x, l2y2);
	if( l1y1 > l2y2 ) {
		SelectObject(hdc, bgpen);
		LineTo(hdc, x, l1y1);
	}
	if( l2y2 < l1y2 ) {
		SelectObject(hdc, l1pen);
		LineTo(hdc, x, l1y2);
	}
	SelectObject(hdc, bgpen);
	LineTo(hdc, x, rcWAV.bottom);
}
struct WAVPen {
	HPEN bg, wav1, wav2;
};
struct WAVPens {
	WAVPen normal, region, selected, regionselected, playing;
};

void DrawWAV(HWND hWnd, HDC hdc) {
	WAVPens Pens;
	HPEN hPen_white;
	HBRUSH hBrush;
	SCROLLINFO si;
	int region_x1, region_x2, select_x1, select_x2, playback_x;

	si.cbSize = sizeof (si);
    si.fMask  = SIF_ALL;
    GetScrollInfo (hWnd, SB_HORZ, &si);

	region_x1 = rcWAV.left + (playback_region_start >> zoom) - si.nPos;
	region_x2 = rcWAV.left + (playback_region_finish >> zoom) - si.nPos + 1;
	select_x1 = rcWAV.left + (selection_start >> zoom) - si.nPos;
	select_x2 = rcWAV.left + (selection_finish >> zoom) - si.nPos + 1;
	playback_x = rcWAV.left + (playback_pos >> zoom) - si.nPos;

	if( wav_len && !wav_content ) {
		HPEN hBlackPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
		HPEN hWhitePen = CreatePen(PS_SOLID, 1, RGB(255,255, 255));
		HBRUSH hBrush = CreateSolidBrush(RGB(0, 0, 0));
		SelectObject(hdc, hBlackPen);
		SelectObject(hdc, hBrush);
		Rectangle(hdc, rcWAV.left, rcWAV.top, rcWAV.right, rcWAV.bottom);

		int resolution;
		const wchar_t* suffix;
		if( zoom > 19 ) {
			resolution = 10 * 60 * 60 * 1000 * 12;
			suffix = L"0h";
		} else if( zoom > 16 ) {
			resolution = 60 * 60 * 1000 * 12;
			suffix = L"h";
		} else if( zoom > 13 ) {
			resolution = 10 * 60 * 1000 * 12;
			suffix = L"0m";
		} else if( zoom > 10 ) {
			resolution = 60 * 1000 * 12;
			suffix = L"m";
		} else if( zoom > 7 ) {
			resolution = 10 * 1000 * 12;
			suffix = L"0s";
		} else if( zoom > 3 ) {
			resolution = 1000 * 12;
			suffix = L"s";
		} else if( zoom > 0 ) {
			resolution = 100 * 12;
			suffix = L"00ms";
		} else {
			resolution = 10 * 12;
			suffix = L"0ms";
		}

		SelectObject(hdc, hWhitePen);
		SetTextColor(hdc, RGB(255, 255, 255));
		SetBkColor(hdc, RGB(0, 0, 0));
		if( !hFont )
			hFont = CreateFont(16, 0, 0, 0, FW_LIGHT, false, false, false, ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, L"Lucida Sans");
		SelectObject(hdc, hFont);

		int pos = (si.nPos<<zoom) / resolution;
		while( pos * resolution <= wav_len ) {
			wchar_t buf[32];
			int drawpos = ((pos * resolution) >> zoom) - si.nPos;
			if( drawpos > rcWAV.right - rcWAV.left )
				break;
			MoveToEx(hdc, rcWAV.left + drawpos, rcWAV.top, NULL);
			LineTo(hdc, rcWAV.left + drawpos, rcWAV.bottom);
			int len;
			if( resolution == 10 * 60 * 1000 * 12 && pos >= 6 ) {
				len = wsprintf(buf, L"%dh%d0m", pos/6, pos%6);
			} else if( resolution == 60 * 1000 * 12 && pos >= 60 ) {
				len = wsprintf(buf, L"%dh%02dm", pos/60, pos%60);
			} else if( resolution == 10 * 1000 * 12 && pos >= 6 ) {
				if( pos >= 6*60 ) {
					len = wsprintf(buf, L"%dh%02dm%d0s", pos/6/60, pos/6%60, pos%6);
				} else {
					len = wsprintf(buf, L"%02dm%d0s", pos/6, pos%6);
				}
			} else if( resolution == 1000 * 12 && pos >= 60 ) {
				if( pos >= 60*60 ) {
					len = wsprintf(buf, L"%dh%02dm%02ds", pos/60/60, pos/60%60, pos%60);
				} else {
					len = wsprintf(buf, L"%02dm%02ds", pos/60, pos%60);
				}
			} else if( resolution == 100 * 12 ) {
				if( pos >= 10*60*60 ) {
					len = wsprintf(buf, L"%dh%02dm%02d.%d00", pos/10/60/60, pos/10/60%60, pos/10%60, pos%10);
				} else if( pos >= 10*60 ) {
					len = wsprintf(buf, L"%02dm%02d.%d00", pos/10/60%60, pos/10%60, pos%10);
				} else if( pos >= 10 ) {
					len = wsprintf(buf, L"%02d.%d00", pos/10, pos%10);
				} else {
					len = wsprintf(buf, L"00.%d00", pos);
				}
			} else if( resolution == 10 * 12 ) {
				if( pos >= 100*60*60 ) {
					len = wsprintf(buf, L"%dh%02dm%02d.%02d0", pos/100/60/60, pos/100/60%60, pos/100%60, pos%100);
				} else if( pos >= 100*60 ) {
					len = wsprintf(buf, L"%02dm%02d.%02d0", pos/100/60%60, pos/100%60, pos%100);
				} else if( pos >= 100 ) {
					len = wsprintf(buf, L"%02d.%02d0", pos/100, pos%100);
				} else {
					len = wsprintf(buf, L"00.%02d0", pos);
				}
			} else {
				len = wsprintf(buf, L"%d%s", pos, suffix);
			}
			RECT rcText;
			rcText.left = rcWAV.left + drawpos + 2;
			rcText.right = rcWAV.right;
			rcText.top = rcWAV.top;
			rcText.bottom = rcWAV.bottom;
			DrawText(hdc, buf, len, &rcText, DT_LEFT|DT_VCENTER|DT_SINGLELINE);
			++pos;
		}

		DeleteObject(hBlackPen);
		DeleteObject(hWhitePen);
		DeleteObject(hBrush);
		return;
	}

	Pens.normal.bg = CreatePen(PS_SOLID, 1, RGB(220, 220, 220));
	Pens.normal.wav1 = CreatePen(PS_SOLID, 1, RGB(128, 128, 128));
	Pens.normal.wav2 = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
	Pens.region.bg = CreatePen(PS_SOLID, 1, RGB(255, 160, 160));
	Pens.region.wav1 = CreatePen(PS_SOLID, 1, RGB(220, 128, 128));
	Pens.region.wav2 = CreatePen(PS_SOLID, 1, RGB(96, 0, 0));
	Pens.selected.bg = CreatePen(PS_SOLID, 1, RGB(160, 160, 255));
	Pens.selected.wav1 = CreatePen(PS_SOLID, 1, RGB(128, 128, 220));
	Pens.selected.wav2 = CreatePen(PS_SOLID, 1, RGB(0, 0, 128));
	Pens.regionselected.bg = CreatePen(PS_SOLID, 1, RGB(255, 160, 255));
	Pens.regionselected.wav1 = CreatePen(PS_SOLID, 1, RGB(220, 128, 220));
	Pens.regionselected.wav2 = CreatePen(PS_SOLID, 1, RGB(96, 0, 128));
	Pens.playing.bg = CreatePen(PS_SOLID, 1, RGB(160, 255, 160));
	Pens.playing.wav1 = CreatePen(PS_SOLID, 1, RGB(128, 220, 128));
	Pens.playing.wav2 = CreatePen(PS_SOLID, 1, RGB(0, 128, 0));
	hPen_white = CreatePen(PS_SOLID, 1, RGB(255, 255, 255));
	hBrush = CreateSolidBrush(RGB(255, 255, 255));

	RECT rcClip;
	GetClipBox(hdc, &rcClip);
	int WAVheight = rcWAV.bottom - rcWAV.top;
	if( zoom == 0 ) {
		int max = wav_len >> zoom;
		for( int x = max(rcWAV.left, rcClip.left); x < min(rcWAV.right, rcClip.right); ++x ) {
			int i = x - rcWAV.left + si.nPos;
			if( i >= max )
				break;
			int l1y1 = rcWAV.bottom - ((wav_content[i].left+32768) * WAVheight / 65535);
			int l1y2 = i == max-1 ? l1y1 : rcWAV.bottom - ((wav_content[i+1].left+32768) * WAVheight / 65535);
			int l2y1 = rcWAV.bottom - ((wav_content[i].right+32768) * WAVheight / 65535);
			int l2y2 = i == max-1 ? l2y1 : rcWAV.bottom - ((wav_content[i+1].right+32768) * WAVheight / 65535);
			if( l1y2 == l1y1 )
				++l1y2;
			if( l2y2 == l2y1 )
				++l2y2;
			if( l1y2 < l1y1 )
				swap(l1y2, l1y1);
			if( l2y2 < l2y1 )
				swap(l2y2, l2y1);

			bool bSelected = x >= select_x1 && x < select_x2;
			bool bInRegion = x >= region_x1 && x < region_x2;
			bool bPlaying = x == playback_x;
			WAVPen& Pen = bPlaying ? Pens.playing : bSelected ? bInRegion ? Pens.regionselected : Pens.selected : bInRegion ? Pens.region : Pens.normal;
			DrawOverlappedVerticalLines(hdc, x, Pen.bg, Pen.wav1, l1y1, l1y2, Pen.wav2, l2y1, l2y2);
		}
	} else {
		int max = wav_len >> zoom;
		for( int x = rcWAV.left; x <= rcWAV.right; ++x ) {
			int i = x - rcWAV.left + si.nPos;
			if( i >= max )
				break;
			int l1y1 = rcWAV.bottom - (wav_mipmap[zoom-1][i].left.min * WAVheight / 255);
			int l1y2 = rcWAV.bottom - (wav_mipmap[zoom-1][i].left.max * WAVheight / 255);
			int l2y1 = rcWAV.bottom - (wav_mipmap[zoom-1][i].right.min * WAVheight / 255);
			int l2y2 = rcWAV.bottom - (wav_mipmap[zoom-1][i].right.max * WAVheight / 255);
			if( l1y2 == l1y1 )
				++l1y2;
			if( l2y2 == l2y1 )
				++l2y2;
			if( l1y2 < l1y1 )
				swap(l1y2, l1y1);
			if( l2y2 < l2y1 )
				swap(l2y2, l2y1);

			bool bSelected = x >= select_x1 && x < select_x2;
			bool bInRegion = x >= region_x1 && x < region_x2;
			bool bPlaying = x == playback_x;
			WAVPen& Pen = bPlaying ? Pens.playing : bSelected ? bInRegion ? Pens.regionselected : Pens.selected : bInRegion ? Pens.region : Pens.normal;
			DrawOverlappedVerticalLines(hdc, x, Pen.bg, Pen.wav1, l1y1, l1y2, Pen.wav2, l2y1, l2y2);
		}
		if( max < rcWAV.right ) {
			SelectObject(hdc, hPen_white);
			SelectObject(hdc, hBrush);
			Rectangle(hdc, rcWAV.left + max, rcWAV.top, rcWAV.right, rcWAV.bottom);
		}
	}
	DeleteObject(Pens.normal.bg);
	DeleteObject(Pens.normal.wav1);
	DeleteObject(Pens.normal.wav2);
	DeleteObject(Pens.region.bg);
	DeleteObject(Pens.region.wav1);
	DeleteObject(Pens.region.wav2);
	DeleteObject(Pens.selected.bg);
	DeleteObject(Pens.selected.wav1);
	DeleteObject(Pens.selected.wav2);
	DeleteObject(Pens.regionselected.bg);
	DeleteObject(Pens.regionselected.wav1);
	DeleteObject(Pens.regionselected.wav2);
	DeleteObject(Pens.playing.bg);
	DeleteObject(Pens.playing.wav1);
	DeleteObject(Pens.playing.wav2);
	DeleteObject(hPen_white);
	DeleteObject(hBrush);
}

static int ScaleAttribute(int attribute, unsigned char scale, int max) {
	int ret = (int)(pow(pow((double)attribute, lfGamma)*scale/255, lfInvGamma)+0.5);
	if( ret > max )
		ret = max;
	return ret;
}
static unsigned long ScaleColour(unsigned long colour, unsigned char scale) {
	int red   =  colour     &255;
	int green = (colour>> 8)&255;
	int blue  = (colour>>16)&255;

	red   = ScaleAttribute(  red, scale, 255);
	green = ScaleAttribute(green, scale, 255);
	blue  = ScaleAttribute( blue, scale, 255);

	if( red > 255 )
		red = 255;
	if( green > 255 )
		green = 255;
	if( blue > 255 )
		blue = 255;
	return red|(green<<8)|(blue<<16);
}

static int AddAttribute(int attribute_a, int attribute_b, int max) {
	int ret = (int)(pow(pow((double)attribute_a, lfGamma) + pow((double)attribute_b, lfGamma), lfInvGamma)+0.5);
	if( ret > max )
		ret = max;
	return ret;
}
static unsigned long AddColour(unsigned long colour_a, unsigned long colour_b) {
	int red   = AddAttribute(( colour_a     &255), ( colour_b     &255), 255);
	int green = AddAttribute(((colour_a>>8 )&255), ((colour_b>>8 )&255), 255);
	int blue  = AddAttribute(((colour_a>>16)&255), ((colour_b>>16)&255), 255);
	return red|(green<<8)|(blue<<16);
}
static unsigned long MakeColour(int red, int green, int blue) {
	return red|(green<<8)|(blue<<16);
}

void DrawSequences(HWND hWnd, HDC hdc, wchar_t light_names[32][256], COLORREF* light_colours, bool bUseGradients) {
	SCROLLINFO si;
	int start, finish, i;
	sequence_state ss;

	if( !sequence_loaded || !EnabledLights ) {
		HPEN hWhitePen = CreatePen(PS_SOLID, 1, RGB(255,255,255));
		HBRUSH hWhiteBrush = CreateSolidBrush(RGB(255,255,255));
		SelectObject(hdc, hWhitePen);
		SelectObject(hdc, hWhiteBrush);
		Rectangle(hdc, 0, rcSeq.top, rcSeq.right, rcSeq.bottom);
		DeleteObject(hWhitePen);
		DeleteObject(hWhiteBrush);
		return;
	}

	if( !hFont )
		hFont = CreateFont(16, 0, 0, 0, FW_LIGHT, false, false, false, ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, L"Lucida Sans");
	SelectObject(hdc, hFont);

	si.cbSize = sizeof (si);
    si.fMask  = SIF_ALL;
    GetScrollInfo (hWnd, SB_HORZ, &si);

	int select_x1, select_x2, playback_x;

	select_x1 = rcSeq.left + (selection_start >> zoom) - si.nPos;
	select_x2 = rcSeq.left + (selection_finish >> zoom) - si.nPos;
	playback_x = rcSeq.left + (playback_pos >> zoom) - si.nPos;

	COLORREF bgCol = RGB(208, 208, 208);
	HPEN hWhitePen = CreatePen(PS_SOLID, 1, RGB(255,255,255));
	HPEN hLtGreyPen = CreatePen(PS_SOLID, 1, bgCol);
	HPEN hDkGreyPen = CreatePen(PS_SOLID, 1, RGB(80, 80, 80));
	HPEN hLtBluePen = CreatePen(PS_SOLID, 1, RGB(80, 80, 255));
	HBRUSH hLtGreyBrush = CreateSolidBrush(bgCol);
	HBRUSH hLtBlueBrush = CreateSolidBrush(RGB(140, 140, 255));
	HBRUSH hBlueBrush = CreateSolidBrush(RGB(96, 96, 255));
	SetBkMode(hdc, TRANSPARENT);
	bool bLastSel = false;
	int index = 0;
	for( i = 0; i < 32; ++i ) {
		if( !(EnabledLights&(1<<i)) )
			continue;
		int top    = rcSeq.top +  index    * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
		int bottom = rcSeq.top + (index+1) * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
		if( ++index == NumEnabledLights )
			++bottom;
		MoveToEx(hdc, 0, top, NULL);
		SelectObject(hdc, hWhitePen);
		LineTo(hdc, rcWAV.left-1, top);
		SelectObject(hdc, hDkGreyPen);
		LineTo(hdc, rcWAV.left-1, bottom-1);
		LineTo(hdc, 0, bottom-1);
		SelectObject(hdc, hWhitePen);
		LineTo(hdc, 0, top);
		SelectObject(hdc, hLtGreyPen);
		SelectObject(hdc, selected_lights&(1<<i) ? i == current_light ? hBlueBrush : hLtBlueBrush : hLtGreyBrush);
		Rectangle(hdc, 1, top+1, rcWAV.left-1, bottom-1);
		MoveToEx(hdc, rcWAV.left, top, NULL);

		bool bThisSel = (selected_lights&(1<<i)) != 0;
		if( (bThisSel && !bLastSel) || (bLastSel && !bThisSel) ) {
			if( select_x1 > 0 ) {
				SelectObject(hdc, hDkGreyPen);
				LineTo(hdc, select_x1, top);
			}
			SelectObject(hdc, hLtBluePen);
			if( select_x2 < rcSeq.right ) {
				LineTo(hdc, select_x2, top);
				SelectObject(hdc, hDkGreyPen);
			}
		} else {
			SelectObject(hdc, hDkGreyPen);
		}
		bLastSel = bThisSel;
		LineTo(hdc, rcWAV.right, top);

		RECT rcText;
		rcText.left = 0;
		rcText.top = top;
		rcText.right = rcWAV.left-2;
		rcText.bottom = bottom-2;
		SetTextColor(hdc, RGB(0, 0, 0));
		DrawText(hdc, light_names[i], wcslen(light_names[i]), &rcText, DT_CENTER|DT_END_ELLIPSIS|DT_VCENTER|DT_SINGLELINE);
	}
	SelectObject(hdc, selected_lights&0x80000000 ? hLtBluePen : hDkGreyPen);
	MoveToEx(hdc, rcWAV.left, rcSeq.bottom-1, NULL);
	LineTo(hdc, rcWAV.right, rcSeq.bottom-1);
	DeleteObject(hWhitePen);
	DeleteObject(hLtGreyPen);
	DeleteObject(hDkGreyPen);
	DeleteObject(hLtBluePen);
	DeleteObject(hLtGreyBrush);
	DeleteObject(hLtBlueBrush);
	DeleteObject(hBlueBrush);

	RECT rcClip;
	GetClipBox(hdc, &rcClip);
	int left = min(max(rcClip.left, rcSeq.left), rcSeq.right);
	int right = max(min(rcClip.right, rcSeq.right), rcSeq.left);
	start = ((si.nPos + (left-rcSeq.left)) << zoom);
	finish = start + ((right - left) << zoom);
	index = 0;
	for( i = 0; i < 32; ++i ) {
		if( !(EnabledLights&(1<<i)) )
			continue;
		int top    = rcSeq.top +  index    * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights + 1;
		int bottom = rcSeq.top + (index+1) * (rcSeq.bottom - rcSeq.top - 1) / NumEnabledLights;
		++index;
		if( bottom <= rcClip.top || top > rcClip.bottom )
			continue;
		sequence_init(&sequences[i], &ss);
		sequence_advance_past(&sequences[i], &ss, (int)(start * 1000LL / wav_sample_rate));
		int samples = start;
		for( int x = left; x < right; ) {
			int red, green, blue;
			if( samples >= wav_len ) {
				red = green = blue = 0;
			} else {
				if( ss.ramp ) {
					int brightness = (ss.brightness<<8)+(ss.ramp_remainder>>4);
					red   = ((unsigned long)( light_colours[i]     &255)*255)*brightness/65279;
					green = ((unsigned long)((light_colours[i]>>8 )&255)*255)*brightness/65279;
					blue  = ((unsigned long)((light_colours[i]>>16)&255)*255)*brightness/65279;
				} else {
					red   = ((unsigned long)( light_colours[i]     &255)*65535)*ss.brightness/65025;
					green = ((unsigned long)((light_colours[i]>>8 )&255)*65535)*ss.brightness/65025;
					blue  = ((unsigned long)((light_colours[i]>>16)&255)*65535)*ss.brightness/65025;
				}
			}
//			COLORREF colour = samples >= wav_len ? 0 : ScaleColour(light_colours[i], ss.brightness);
			int sample_round = samples - samples * 1000LL / wav_sample_rate * wav_sample_rate / 1000LL;
			int width = (((ss.ramp_ms ? min(ss.ms_remain, ss.ramp_ms) : ss.ms_remain) * (long long)wav_sample_rate / 1000 - sample_round) >> zoom);
			if( !ss.ramp )
				++width;
			else if( width == 0 )
				width = 1;
			if( !bUseGradients && ss.ramp )
				width = 1;
			if( width == 0 || width > rcSeq.right - x )
				width = rcSeq.right - x;
			if( x == playback_x ) {
				  red = AddAttribute(  red,  32<<8, 65535);
				green = AddAttribute(green, 128<<8, 65535);
				 blue = AddAttribute( blue,  32<<8, 65535);
//				colour = AddColour(colour, RGB(32, 128, 32));
				width = 1;
			} else if( x < playback_x && x+width >= playback_x ) {
				width = playback_x - x;
			}
			if( selected_lights&(1<<i) ) {
				if( x == select_x1 || x == select_x2 ) {
//					colour = AddColour(colour, RGB(80, 80, 255));
					  red = AddAttribute(  red,  80<<8, 65535);
					green = AddAttribute(green,  80<<8, 65535);
					 blue = AddAttribute( blue, 255<<8, 65535);
					width = 1;
				} else {
					if( x < select_x1 && x+width > select_x1 )
						width = select_x1 - x;
					if( x < select_x2 && x+width > select_x2 )
						width = select_x2 - x;
				}
			}
			if( ss.ramp && width > 1 ) {
				sequence_state temp = ss;
				sequence_do_ramp(&temp, (int)(((samples + (width << zoom)) * 1000LL + wav_sample_rate-1) / wav_sample_rate) - (int)(samples * 1000LL / wav_sample_rate));
//				sequence_advance(&sequences[i], &temp, (samples + (width << zoom)) * 1000LL / wav_sample_rate, true);
				int final_brightness = (temp.brightness<<8)+(temp.ramp ? (temp.ramp_remainder>>4) : 0);
				int final_red   = ((unsigned long)( light_colours[i]     &255)*255)*final_brightness/65279;
				int final_green = ((unsigned long)((light_colours[i]>>8 )&255)*255)*final_brightness/65279;
				int final_blue  = ((unsigned long)((light_colours[i]>>16)&255)*255)*final_brightness/65279;

				TRIVERTEX Vertex[2];
				GRADIENT_RECT Rect;
				Vertex[0].x = x;
				Vertex[0].y = top;
				Vertex[0].Alpha = 0xFFFF;
				Vertex[0].Blue = blue;
				Vertex[0].Green = green;
				Vertex[0].Red = red;
				Vertex[1].x = x + width;
				Vertex[1].y = bottom;
				Vertex[1].Alpha = 0xFFFF;
				Vertex[1].Blue = final_blue;
				Vertex[1].Green = final_green;
				Vertex[1].Red = final_red;
				Rect.UpperLeft = 0;
				Rect.LowerRight = 1;
				GradientFill(hdc, (PTRIVERTEX)&Vertex, sizeof(Vertex)/sizeof(*Vertex), &Rect, 1, GRADIENT_FILL_RECT_H);
			} else {
				unsigned long colour = MakeColour(red*255/65535, green*255/65535, blue*255/65535);
				HPEN hSolidPen = CreatePen(PS_SOLID, 1, colour);
				HBRUSH hSolidBrush = CreateSolidBrush(colour);
				SelectObject(hdc, hSolidPen);
				SelectObject(hdc, hSolidBrush);
				Rectangle(hdc, x, top, x + width, bottom);
				DeleteObject(hSolidPen);
				DeleteObject(hSolidBrush);
			}
			samples += width << zoom;
			if( samples > wav_len )
				samples = wav_len;
			sequence_advance_past(&sequences[i], &ss, (int)(samples * 1000LL / wav_sample_rate) );
			x += width;
		}
	}
}

void InitLightbar() {
	for( int i = 0; i < 32; ++i )
		sequence_init(&sequences[i], &playback_ss[i]);
}

static void DrawLightBarLights(HWND hWnd, HDC hdc, int samples, COLORREF bgCol, COLORREF* light_colours) {
	HPEN hBgPen = CreatePen(PS_SOLID, 1, bgCol);
	HBRUSH hBgBrush = CreateSolidBrush(bgCol);

	if( samples != -1 && sequence_loaded ) {
		HPEN hWhitePen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
		int ms = (samples * 1000LL / wav_sample_rate);
		int dim = rcLightBar.bottom - rcLightBar.top - 6;
		int x = 8;
		int y = rcLightBar.top + 2;
		HRGN rgnClip = CreateEllipticalRegion(dim-3, dim-3, x+2, y+2, true);
		for( int i = 0; i < 32; ++i ) {
			if( !(EnabledLights&(1<<i)) )
				continue;

			if( PtVisible(hdc, x+dim/2, y+dim/2) ) {
				int saved_dc = SaveDC(hdc);
				CombineClipRgn(hdc, rgnClip);
				SelectObject(hdc, hBgPen);
				SelectObject(hdc, hBgBrush);
				Rectangle(hdc, x+1, y+1, x+dim, y+dim);
				RestoreDC(hdc, saved_dc);

				sequence_advance(&sequences[i], &playback_ss[i], ms);
				HBRUSH hBrush = CreateSolidBrush(ScaleColour(light_colours[i], playback_ss[i].brightness));
				SelectObject(hdc, hWhitePen);
				SelectObject(hdc, hBrush);
				Ellipse(hdc, x+2, y+2, x+dim-1, y+dim-1);
				DeleteObject(hBrush);
			}

			x += dim + 8;
			OffsetRgn(rgnClip, dim+8, 0);
		}
		DeleteObject(hBgPen);
		DeleteObject(hWhitePen);
		DeleteObject(hBgBrush);
		DeleteObject(rgnClip);
	} else {
		SelectObject(hdc, hBgPen);
		SelectObject(hdc, hBgBrush);
		int dim = rcLightBar.bottom - rcLightBar.top - 6;
		int x = 8;
		int y = rcLightBar.top + 2;
		for( int i = 0; i < 32; ++i ) {
			if( !(EnabledLights&(1<<i)) )
				continue;
			Rectangle(hdc, x+1, y+1, x+dim, y+dim);
			x += dim + 8;
		}
	}
	DeleteObject(hBgPen);
	DeleteObject(hBgBrush);
}

void DrawLightbar(HWND hWnd, HDC hdc, COLORREF* light_colours) {
	COLORREF bgCol = GetSysColor(COLOR_BTNFACE);//RGB(208, 208, 208);
	HPEN hWhitePen = CreatePen(PS_SOLID, 1, RGB(255,255,255));
	HPEN hLtGreyPen = CreatePen(PS_SOLID, 1, bgCol);
	HPEN hDkGreyPen = CreatePen(PS_SOLID, 1, RGB(80, 80, 80));
	HBRUSH hLtGreyBrush = CreateSolidBrush(bgCol);
	MoveToEx(hdc, rcLightBar.left, rcLightBar.top, NULL);
	SelectObject(hdc, hWhitePen);
	LineTo(hdc, rcLightBar.right, rcLightBar.top);
	SelectObject(hdc, hDkGreyPen);
	LineTo(hdc, rcLightBar.right, rcLightBar.bottom-1);
	LineTo(hdc, rcLightBar.left, rcLightBar.bottom-1);
	SelectObject(hdc, hWhitePen);
	LineTo(hdc, rcLightBar.left, rcLightBar.top);

	int saved_dc = SaveDC(hdc);
	int dim = rcLightBar.bottom - rcLightBar.top - 6;
	int x = 8;
	int y = rcLightBar.top + 2;
	for( int i = 0; i < 32; ++i ) {
		if( !(EnabledLights&(1<<i)) )
			continue;
		ExcludeClipRect(hdc, x, y, x+dim, y+dim);
		x += dim + 8;
	}
	SelectObject(hdc, hLtGreyPen);
	SelectObject(hdc, hLtGreyBrush);
	Rectangle(hdc, rcLightBar.left+1, rcLightBar.top+1, rcLightBar.right, rcLightBar.bottom-1);
	RestoreDC(hdc, saved_dc);

	x = 8;
	y = rcLightBar.top + 2;
	for( int i = 0; i < 32; ++i ) {
		if( !(EnabledLights&(1<<i)) )
			continue;
		MoveToEx(hdc, x, y, NULL);
		SelectObject(hdc, hDkGreyPen);
		LineTo(hdc, x + dim, y);
		SelectObject(hdc, hWhitePen);
		LineTo(hdc, x + dim, y + dim);
		LineTo(hdc, x, y + dim);
		SelectObject(hdc, hDkGreyPen);
		LineTo(hdc, x, y);

		x += dim + 8;
	}

	DeleteObject(hWhitePen);
	DeleteObject(hLtGreyPen);
	DeleteObject(hDkGreyPen);
	DeleteObject(hLtGreyBrush);

	DrawLightBarLights(hWnd, hdc, playback_get_current_sample(), bgCol, light_colours);
}

static int PrintTime(wchar_t* ToHere, int ms) {
	if( ms >= 60*60*1000 ) {
		return wsprintf(ToHere, L"d:%02d:%02d.%03d", ms/60/60/1000, ms/60/1000%60, ms/1000%60, ms%1000);
	} else if( ms >= 60*1000 ) {
		return wsprintf(ToHere, L"%02d:%02d.%03d", ms/60/1000, ms/1000%60, ms%1000);
	} else {
		return wsprintf(ToHere, L"%02d.%03d", ms/1000, ms%1000);
	}
}

void DrawStatus(HWND hWnd, HDC hdc) {
	wchar_t buf[256], bufs[4][32];
	int pos = 0;

	if( !hSmallFont )
		hSmallFont = CreateFont(14, 0, 0, 0, FW_LIGHT, false, false, false, ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, L"Lucida Sans");

	if( selection_start != -1 ) {
		int start_ms = (int)((selection_start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
		int wav_len_ms = (int)((wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
		if( selection_start == selection_finish ) {
			PrintTime(bufs[0], start_ms);
			PrintTime(bufs[1], wav_len_ms);
			pos += wsprintf(buf+pos, L"Pos: %s / %s", bufs[0], bufs[1]);
		} else {
			int finish_ms = (int)((selection_finish + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
			PrintTime(bufs[0], start_ms);
			PrintTime(bufs[1], finish_ms);
			PrintTime(bufs[2], wav_len_ms);
			PrintTime(bufs[3], finish_ms-start_ms);
			pos += wsprintf(buf+pos, L"Sel: %s - %s / %s (%s)", bufs[0], bufs[1], bufs[2], bufs[3]);
		}
	} else if( wav_len && !wav_content ) {
		int wav_len_ms = (int)((wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
		PrintTime(bufs[0], wav_len_ms);
		pos += wsprintf(buf+pos, L"Standalone sequence: %s", bufs[0]);
	} else if( wav_len ) {
		pos += wsprintf(buf+pos, L"%dHz 16bit %s ", (int)wav_sample_rate, wav_num_channels == 1 ? L"mono" : L"stereo");
		switch(get_playback_status()) {
		case playing:
			{
				int seconds = playback_get_current_sample() / wav_sample_rate;
				pos += wsprintf(buf+pos, L"Playing: %02d:%02d:%02d", seconds/3600, (seconds/60)%60, seconds%60);
				break;
			}
		case paused:
			pos += wsprintf(buf+pos, L"Playback paused");
			break;
		case stopped:
			pos += wsprintf(buf+pos, L"Playback stopped");
			break;
		}
	} else {
		pos += wsprintf(buf+pos, L"No WAV file loaded");
	}

	HBITMAP drawarea = CreateBitmap(rcWAV.left, rcWAV.bottom-rcWAV.top, 1, 32, NULL);
	HDC tempdc = CreateCompatibleDC(hdc);
	SelectObject(tempdc, drawarea);
	HPEN hPen = CreatePen(PS_SOLID, 1, RGB(64, 64, 64));
	HBRUSH hBrush = CreateSolidBrush(RGB(64, 64, 64));
	SelectObject(tempdc, hPen);
	SelectObject(tempdc, hBrush);
	Rectangle(tempdc, 0, 0, rcWAV.left, rcWAV.bottom-rcWAV.top);
	SetTextColor(tempdc, RGB(255, 255, 255));
	SetBkMode(tempdc, TRANSPARENT);
	SelectObject(tempdc, hSmallFont);
	RECT rcText;
	rcText.left = 2;
	rcText.top = 2;
	rcText.right = rcWAV.left-2;
	rcText.bottom = rcWAV.bottom-rcWAV.top-2;
	DrawText(tempdc, buf, pos, &rcText, DT_LEFT|DT_END_ELLIPSIS|DT_WORDBREAK|DT_WORD_ELLIPSIS|DT_HIDEPREFIX);
	DeleteObject(hPen);
	DeleteObject(hBrush);
	BitBlt(hdc, 0, rcWAV.top, rcWAV.left, rcWAV.bottom, tempdc, 0, 0, SRCCOPY);
	DeleteObject(drawarea);
	DeleteObject(tempdc);
}

void DelayedUpdateDisabledState(HWND hWnd) {
	SendMessage(hWnd, WM_USER, 0, 0);
}

bool CanZoomIn(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? L"you have zoomed in as far as possible" : L"no file is currently open";
	return sequence_loaded && zoom != 0;
}

bool CanZoomOut(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? L"you have zoomed out as far as possible" : L"no file is currently open";
	return sequence_loaded && !(zoom == wav_mipmap_levels-1 || (wav_len>>zoom) < (rcWAV.right-rcWAV.left));
}

bool IsSelection(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? L"there is currently no selection" : L"no file is currently open";
	return sequence_loaded && selection_start != -1;
}

bool IsSelectionAndLights(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? selected_lights ? L"there is currently no selection" : L"no lights are currently selected" : L"no file is currently open";
	return sequence_loaded && selection_start != -1;
}

bool IsSelectionAndLightsAndNotStandalone(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? wav_content ? selected_lights ? L"there is currently no selection" : L"no lights are currently selected" : L"this is a standalone sequence" : L"no file is currently open";
	return sequence_loaded && selection_start != -1 && wav_content;
}

bool SelectionHasArea(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? selection_start != -1 ? L"the current selection has no width" : L"there is currently no selection" : L"no file is currently open";
	return sequence_loaded && selection_start != -1 && selection_start != selection_finish;
}

bool SelectionHasAreaAndLights(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? selection_start != -1 ? L"the current selection has no width" : selected_lights ? L"there is currently no selection" : L"no lights are currently selected" : L"no file is currently open";
	return sequence_loaded && selection_start != -1 && selection_start != selection_finish;
}

bool SelectionHasAreaAndLightsAndNotStandalone(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? wav_content ? selection_start != -1 ? L"the current selection has no width" : selected_lights ? L"there is currently no selection" : L"no lights are currently selected" : L"this is a standalone sequence" : L"no file is currently open";
	return sequence_loaded && selection_start != -1 && selection_start != selection_finish && wav_content;
}

bool HavePlaybackRegion(const wchar_t** pReason) {
	if( pReason )
		*pReason = sequence_loaded ? L"no playback region has been defined" : L"no file is currently open";
	return sequence_loaded && playback_region_start != -1;
}


void MainSequencesChanged(HWND hWnd, int start, int finish) {
	int sample = playback_get_current_sample();
	if( sample != -1 && sample >= start - 4096 ) {
		for( int i = 0; i < 32; ++i )
			sequence_init(&sequences[i], &playback_ss[i]);
	}
}
