
#include "DSP.h"
#include <math.h>
#include <xmmintrin.h>
#ifdef _DEBUG
#include <assert.h>
#endif
#include <new>

static void prewarp(double *a0, double *a1, double *a2, double fc, double fs) {
    double pi = 4.0 * atan(1.0);
    double wp = 2.0 * fs * tan(pi * fc / fs);

    *a2 = (*a2) / (wp * wp);
    *a1 = (*a1) / wp;
}

static void bilinear(
    double a0, double a1, double a2,    /* numerator coefficients */
    double b0, double b1, double b2,    /* denominator coefficients */
    double *k,           /* overall gain factor */
    double fs,           /* sampling rate */
    double *coef         /* pointer to 4 iir coefficients */
)
{
    double ad, bd;

                 /* alpha (Numerator in s-domain) */
    ad = 4.0 * a2 * fs * fs + 2. * a1 * fs + a0;
                 /* beta (Denominator in s-domain) */
    bd = 4.0 * b2 * fs * fs + 2. * b1* fs + b0;

                 /* update gain constant for this section */
    *k *= ad/bd;

                 /* Denominator */
    *coef++ = (2.0 * b0 - 8. * b2 * fs * fs)
                           / bd;         /* beta1 */
    *coef++ = (4.0 * b2 * fs * fs - 2. * b1 * fs + b0)
                           / bd; /* beta2 */

                 /* Nominator */
    *coef++ = (2.0 * a0 - 8. * a2 * fs * fs)
                           / ad;         /* alpha1 */
    *coef = (4.0 * a2 * fs * fs - 2. * a1 * fs + a0)
                           / ad;   /* alpha2 */
}


static void szxform(
    double *a0, double *a1, double *a2, /* numerator coefficients */
    double *b0, double *b1, double *b2, /* denominator coefficients */
    double fc,         /* Filter cutoff frequency */
    double fs,         /* sampling rate */
    double *k,         /* overall gain factor */
    double *coef)      /* pointer to 4 iir coefficients */
{
                 /* Calculate a1 and a2 and overwrite the original values */
        prewarp(a0, a1, a2, fc, fs);
        prewarp(b0, b1, b2, fc, fs);
        bilinear(*a0, *a1, *a2, *b0, *b1, *b2, k, fs, coef);
}

IIRFilter::IIRFilter(int SampleRate, double CornerFrequency, double Q, double Gain, int Order, const FilterCoefficients<double>* pCoefficients) : m_Order(Order) {
	m_plfCoefficients = new double[6 * Order + 1];

	if( Q < 1 )
		Q = 1;
	else if( Q > 1000 )
		Q = 1000;

	for( int i = 0; i < Order; ++i ) {
		double a0 = pCoefficients[i].a0;
		double a1 = pCoefficients[i].a1;
		double a2 = pCoefficients[i].a2;
		double b0 = pCoefficients[i].b0;
		double b1 = pCoefficients[i].b1 / Q;
		double b2 = pCoefficients[i].b2;
		szxform(&a0, &a1, &a2, &b0, &b1, &b2, CornerFrequency, SampleRate, &Gain, m_plfCoefficients+1+i*4);
	}
	m_plfCoefficients[0] = Gain;

	Reset();

	// turn off denormals to avoid performance penalty
	_MM_SET_FLUSH_ZERO_MODE(_MM_FLUSH_ZERO_ON);
}
IIRFilter::~IIRFilter() {
	delete [] m_plfCoefficients;
}
void IIRFilter::Reset() {
	for( int i = 0; i < m_Order*2; ++i )
		m_plfCoefficients[m_Order*4+1+i] = 0;
}
double IIRFilter::Process(double Input) {
    double* pCoefficients = m_plfCoefficients;

    double* pHistory = m_plfCoefficients + 1 + 4*m_Order;
	double NewHistory;

	double Output = Input * (*pCoefficients++);
	if( m_Order == 2 ) {
		Output    -= pHistory[0] * pCoefficients[0];
		NewHistory = Output - pHistory[1] * pCoefficients[1];
		Output     = NewHistory + pHistory[0] * pCoefficients[2];
		Output    += pHistory[1] * pCoefficients[3];

		pHistory[1] = pHistory[0];
		pHistory[0] = NewHistory;

		Output    -= pHistory[2] * pCoefficients[4];
		NewHistory = Output - pHistory[3] * pCoefficients[5];
		Output     = NewHistory + pHistory[2] * pCoefficients[6];
		Output    += pHistory[3] * pCoefficients[7];

		pHistory[3] = pHistory[2];
		pHistory[2] = NewHistory;
	} else {
		for( int i = 0; i < m_Order; i++ ) {
			Output    -= pHistory[0] * pCoefficients[0];
			NewHistory = Output - pHistory[1] * pCoefficients[1];
			Output     = NewHistory + pHistory[0] * pCoefficients[2];
			Output    += pHistory[1] * pCoefficients[3];

			pHistory[1] = pHistory[0];
			pHistory[0] = NewHistory;

			pHistory += 2;
			pCoefficients += 4;
		}
	}

    return Output;
}


static const FilterCoefficients<double> SecondOrderButterworthFilterCoefficients[2] = { { 1.0, 0.0, 0.0, 1.0, 0.765367, 1.0 }, { 1.0, 0.0, 0.0, 1.0, 1.847759, 1.0 } };

SecondOrderButterworthFilter::SecondOrderButterworthFilter(int SampleRate, double CornerFrequency, double Gain, SimpleFilterType Type) :
IIRFilter(SampleRate, CornerFrequency, sqrt(2.0)/2.0, Gain, 2, SecondOrderButterworthFilterCoefficients), m_Type(Type) { }

double SecondOrderButterworthFilter::Process(double Input) {
	double Result = IIRFilter::Process(Input);
	if( m_Type == Lowpass )
		return Result;
	else
		return Input - Result;
}

static double CalcMultiOrderButterworthFrequencyRatio(double Order) {
	double val = pow(0.5, 1.0 / Order);
	return pow(((1 - val)/val), 0.25);
}

HigherOrderButterworthFilter::HigherOrderButterworthFilter(int SampleRate, double CornerFrequency, int Order, double Gain, SimpleFilterType Type) : m_Order(Order) {
#ifdef _DEBUG
	assert(Order > 2 && !(Order&1));
#endif
	if( Order < 2 )
		Order = 2;
	else if( Order&1 )
		++Order;

	double lfRatio = CalcMultiOrderButterworthFrequencyRatio(2*Order);
	switch(Type) {
	case Lowpass:
		CornerFrequency *= lfRatio;
		break;
	default:
		CornerFrequency /= lfRatio;
		break;
	}

	m_pFilters = (SecondOrderButterworthFilter*)malloc(sizeof(SecondOrderButterworthFilter)*(Order/2));
	for( int i = 0; i < Order/2; ++i )
		new (m_pFilters+i) SecondOrderButterworthFilter(SampleRate, CornerFrequency, i == 0 ? Gain : 1.0, Type);
}
HigherOrderButterworthFilter::~HigherOrderButterworthFilter() {
	for( int i = 0; i < m_Order/2; ++i )
		m_pFilters[i].~SecondOrderButterworthFilter();
	free(m_pFilters);
}

void HigherOrderButterworthFilter::Reset() {
	for( int i = 0; i < m_Order/2; ++i )
		m_pFilters[i].Reset();
}
double HigherOrderButterworthFilter::Process(double Input) {
	for( int i = 0; i < m_Order/2; ++i )
		Input = m_pFilters[i].Process(Input);
	return Input;
}


BandpassFilter::BandpassFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, ComplexFilterType Type) :
	m_LowpassFilter (SampleRate, Type == Bandpass ? UpperCornerFrequency : LowerCornerFrequency, Order, Gain, Lowpass ),
	m_HighpassFilter(SampleRate, Type == Bandpass ? LowerCornerFrequency : UpperCornerFrequency, Order,  1.0, Highpass), m_Type(Type)
{
}
void BandpassFilter::Reset() {
	m_LowpassFilter.Reset();
	m_HighpassFilter.Reset();
}
double BandpassFilter::Process(double Input) {
	switch(m_Type) {
	case Bandpass:
		return m_HighpassFilter.Process(m_LowpassFilter.Process(Input));
	default:
		return m_HighpassFilter.Process(Input) + m_LowpassFilter.Process(Input);
	}
}


MultiBandFilter::MultiBandFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, int NumBands, FrequencySpacing Spacing) : m_NumBands(NumBands) {
	double log_2 = log(2.0);
	m_pBandFilters = (BandpassFilter*)malloc( NumBands * sizeof(BandpassFilter) );
	double lfLastFreq = LowerCornerFrequency;
	for( int i = 0; i < NumBands; ++i ) {
		double lfFreq;
		switch(Spacing) {
		case Linear:
			lfFreq = LowerCornerFrequency + (UpperCornerFrequency-LowerCornerFrequency)*(i+1)/NumBands;
			break;
		case Logarithmic:
			lfFreq = LowerCornerFrequency * pow(2, log(UpperCornerFrequency/LowerCornerFrequency)/log_2*(i+1)/NumBands);
			break;
		}
		new (m_pBandFilters+i) BandpassFilter(SampleRate, lfLastFreq, lfFreq, Order, 1.0, Bandpass);
		lfLastFreq = lfFreq;
	}
	m_BandSamples = new double[NumBands];
}
MultiBandFilter::~MultiBandFilter() {
	for( int i = 0; i < m_NumBands; ++i )
		m_pBandFilters[i].~BandpassFilter();
	free(m_pBandFilters);
	delete [] m_BandSamples;
}
void MultiBandFilter::Reset() {
	for( int i = 0; i < m_NumBands; ++i )
		m_pBandFilters[i].Reset();
}
double MultiBandFilter::Process(double Input) {
	for( int i = 0; i < m_NumBands; ++i )
		m_BandSamples[i] = m_pBandFilters[i].Process(Input);
	return m_BandSamples[0];
}
double MultiBandFilter::GetBandSample(int Band) {
	if( Band >= 0 && Band < m_NumBands )
		return m_BandSamples[Band];
	else
		return 0.0;
}


template <typename T>
static inline T square(T a) { return a*a; }


ExponentialDecay::ExponentialDecay(int SampleRate, double lfAverageHalflife) {
	m_lfDecayFactor = pow( 0.5, 1/(lfAverageHalflife * SampleRate) );
	Reset();
}
void ExponentialDecay::Reset() {
	m_lfAverageEnergy = 0.0;
}
double ExponentialDecay::Process(double Input) {
	m_lfAverageEnergy = m_lfAverageEnergy * m_lfDecayFactor + square(Input);
	return m_lfAverageEnergy * (1 - m_lfDecayFactor);
}


AveragingMultiBandFilter::AveragingMultiBandFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, int NumBands, FrequencySpacing Spacing, double lfAverageHalflife) :
	MultiBandFilter(SampleRate, LowerCornerFrequency, UpperCornerFrequency, Order, Gain, NumBands, Spacing) {
	m_AverageEnergy = new double[NumBands];
	m_lfDecayFactor = pow( 0.5, 1/(lfAverageHalflife * SampleRate) );
	Reset();
}
AveragingMultiBandFilter::~AveragingMultiBandFilter() {
	delete [] m_AverageEnergy;
}
void   AveragingMultiBandFilter::Reset() {
	MultiBandFilter::Reset();
	for( int i = 0; i < m_NumBands; ++i )
		m_AverageEnergy[i] = 0.0;
}
double AveragingMultiBandFilter::Process(double Input) {
	double Output = MultiBandFilter::Process(Input);
	for( int i = 0; i < m_NumBands; ++i )
		m_AverageEnergy[i] = m_AverageEnergy[i] * m_lfDecayFactor + square(m_BandSamples[i]);
	return Output;
}
double AveragingMultiBandFilter::GetAverageEnergy(int Band) {
	if( Band >= 0 && Band < m_NumBands )
		return sqrt(m_AverageEnergy[Band]);
	else
		return 0.0;
}
