
#include "targetver.h"
#include <stdio.h>
#pragma warning(disable: 4996) // no CRT security warning
#include <stdlib.h>
#ifdef _DEBUG
#include <assert.h>
#endif
#include <string.h>
#include <Windows.h>
#include "sequence.h"
#ifdef _DEBUG
#include "WAVBrowse.h"
#endif
#include "Undo.h"


sequence sequences[32];
bool g_bLoading;

static void grow_sequence(sequence* dest) {
	dest->len += 512;
	dest->commands = (unsigned short*)realloc(dest->commands, dest->len * sizeof(*dest->commands));
}

void sequence_append_command(sequence* dest, unsigned short command) {
#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif
	if( !dest->commands || dest->used == dest->len )
		grow_sequence(dest);
	dest->commands[dest->used++] = command;
}

void sequence_insert_command(sequence* dest, int index, unsigned short command) {
#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif
	if( !dest->commands || dest->used == dest->len )
		grow_sequence(dest);
	memmove(dest->commands + index + 1, dest->commands + index, (dest->used - index) * sizeof(unsigned short));
	dest->commands[index] = command;
	++dest->used;
}

void sequence_remove_command(sequence* dest, int index, int count) {
#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif
	memmove(dest->commands + index, dest->commands + index + count, (dest->used - (index + count)) * sizeof(unsigned short));
	dest->used -= count;
}

void sequence_destroy(sequence* dest) {
#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif
	free(dest->commands);
	dest->commands = 0;
	dest->used = dest->len = 0;
}

void clear_all_sequences() {
#ifdef _DEBUG
	assert(g_bLoading || undo_nesting);
#endif
	for( int i = 0; i < 32; ++i )
		sequence_destroy(&sequences[i]);
}

void set_light_brightness(sequence* dest, unsigned char which, unsigned char level) {
	sequence_append_command(&dest[which], 0x2000|(((unsigned short)which)<<8)|level);
}

void process_command(unsigned short command, sequence* dest) {
	unsigned char i, temp[2];

	switch(command>>8) {
	case 0:
		for( i = 0; i < 32; ++i )
			set_light_brightness(dest, i, 0);
		break;
	case 1:
	case 2:
	case 3:
		for( i = 0; i < 32; ++i )
			sequence_append_command(&dest[i], command);
		break;
	default:
		switch(command>>10) {
		case 1:
			for( i = 0; i < 8; ++i )
				set_light_brightness(dest, ((command>>5)&24)+i, (command&(1<<i)) ? 255 : 0);
			break;
		case 2:
			for( i = 0; i < 8; ++i )
				if( (command&(1<<i)) )
					set_light_brightness(dest, ((command>>5)&24)+i, 0);
			break;
		case 3:
			for( i = 0; i < 8; ++i )
				if( (command&(1<<i)) )
					set_light_brightness(dest, ((command>>5)&24)+i, 255);
			break;
		default:
			temp[0] =  command    &31;
			temp[1] = (command>>5)&31;
			switch(command>>10) {
			case 4:
				set_light_brightness(dest, temp[0], 255);
				set_light_brightness(dest, temp[1], 255);
				break;
			case 5:
				set_light_brightness(dest, temp[0], 0);
				set_light_brightness(dest, temp[1], 0);
				break;
			case 6:
				set_light_brightness(dest, temp[0], 255);
				set_light_brightness(dest, temp[1], 0);
				break;
			case 7:
				sequence_append_command(&dest[temp[0]], command);
				sequence_append_command(&dest[temp[1]], command);
				break;
			default:
				temp[0] = (command>>8)&31;
				if( (command>>13) == 1 ) {
					set_light_brightness(dest, temp[0], (unsigned char)command);
				} else {
					sequence_append_command(&dest[temp[0]], command);
				}
				break;
			}
			break;
		}
		break;
	}
}

void process_commands(unsigned short* cmds, unsigned short num, sequence* dest) {
	while( num ) {
		process_command(cmds[0], dest);
		++cmds;
		--num;
	}
}

bool sequence_load(const wchar_t* filename, int max_ms) {
	unsigned short buf[256];
#ifdef DEBUG_DUMP_INFO
	wchar_t debug[256];
	int ms = 0;
#endif
	size_t read;

	g_bLoading = true;
	clear_all_sequences();
	FILE* f = _wfopen(filename, L"rb");
	if( !f ) {
		g_bLoading = false;
		return false;
	}

	do {
		read = fread(buf, sizeof(*buf), sizeof(buf)/sizeof(*buf), f);
		if( read == -1 ) {
			clear_all_sequences();
			fclose(f);
			g_bLoading = false;
			return false;
		}
#ifdef DEBUG_DUMP_INFO
		for( size_t i = 0; i < read; ++i ) {
			unsigned short cmd = ((unsigned short*)buf)[i];
			if( IS_DELAY_CMD(cmd) ) {
				wsprintf(debug, L"%06d     delay command\n", ms);
				ms += DELAY_MS(cmd);
			} else {
				if( (cmd>>13) > 1 ) {
					wsprintf(debug, L"%06d ramp (%02d) command\n", ms, (cmd>>8)&31);
				} else if( (cmd>>13) == 1 ) {
					wsprintf(debug, L"%06d setb (%02d) = %d\n", ms, (cmd>>8)&31, (cmd&255));
				} else {
					wsprintf(debug, L"%06d non-delay command\n", ms);
				}
			}
			OutputDebugString(debug);
		}
#endif
		process_commands(buf, read, sequences);
	} while( read == sizeof(buf)/sizeof(*buf) );

	for( int i = 0; i < 32; ++i ) {
		sequence_compact(&sequences[i], 0, sequences[i].used);
		if( max_ms != -1 )
			sequence_truncate(&sequences[i], max_ms, max_ms);
	}

	fclose(f);
	g_bLoading = false;
	return true;
}

void sequence_init(const sequence* sq, sequence_state* ss) {
	ss->brightness = ss->ms_pos = ss->ms_remain = ss->pos = ss->ramp = ss->ramp_ms = 0;
}

void sequence_do_ramp(sequence_state* ss, int ms) {
	if( ss->ramp ) {
		long frac_add = ss->ramp * (long)ms + ss->ramp_remainder;
//		short new_value = (short)ss->brightness + (short)(frac_add / 1024);
		long new_value = (long)ss->brightness + (long)(frac_add / 1024);
		ss->ramp_remainder = ((short)frac_add)&1023;
		if( ss->ramp < 0 && ss->ramp_remainder )
			ss->ramp_remainder |= 0xFC00;
		ss->ramp_ms -= ms;
		if( ss->ramp_ms < 0 ) {
			ss->ramp = 0;
			ss->ramp_ms = 0;
		}
		if( new_value > 255 ) {
			new_value = 255;
			ss->ramp = 0;
			ss->ramp_ms = 0;
		} else if( new_value < 0 ) {
			new_value = 0;
			ss->ramp = 0;
			ss->ramp_ms = 0;
		}
		ss->brightness = new_value;
	}
}

void sequence_execute_command(const sequence* sq, sequence_state* ss) {
	if( ss->pos < sq->used ) {
		unsigned short command = sq->commands[ss->pos++];
		unsigned char temp[2];

		switch(command>>8) {
		case 0:
			ss->brightness = 0;
			ss->ramp = 0;
			ss->ramp_ms = 0;
			break;
		case 1:
			ss->ms_remain += (command&255)+1;
			break;
		case 2:
			ss->ms_remain += ((command&255)+1)<<2;
			break;
		case 3:
			ss->ms_remain += ((command&255)+1)<<4;
			break;
		default:
			switch(command>>10) {
			case 1:
			case 2:
			case 3:
#ifdef _DEBUG
				assert(0);
#endif
			default:
				temp[0] =  command    &31;
				temp[1] = (command>>5)&31;
				switch(command>>10) {
				case 4:
				case 5:
				case 6:
				case 7:
#ifdef _DEBUG
					assert(0);
#endif
					break;
				default:
					temp[0] = (command>>8)&31;
					if( (command>>13) == 1 ) {
						ss->brightness = (unsigned char)command;
						ss->ramp = 0;
						ss->ramp_ms = 0;
					} else {
						ss->ramp_remainder = 0;
						int ramp;
						switch(command>>13) {
						case 2:
							ss->ramp_ms = (((command&255)+1)<<3);
							ramp = (255-(short)ss->brightness) * 1024L / ss->ramp_ms;
							if( ramp > 0x7FFF ) {
								ss->ramp = 0x7FFF;
								ss->ramp_ms = (255-ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						case 3:
							ss->ramp_ms = (((command&255)+1)<<3);
							ramp = -(short)ss->brightness * 1024L / ss->ramp_ms;
							if( ramp < -0x8000 ) {
								ss->ramp = -0x8000;
								ss->ramp_ms = (ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						case 4:
							ss->ramp_ms = (((command&255)+1)<<4);
							ramp = (255-(short)ss->brightness) * 1024L / ss->ramp_ms;
							if( ramp > 0x7FFF ) {
								ss->ramp = 0x7FFF;
								ss->ramp_ms = (255-ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						case 5:
							ss->ramp_ms = (((command&255)+1)<<4);
							ramp = -(short)ss->brightness * 1024L / ss->ramp_ms;
							if( ramp < -0x8000 ) {
								ss->ramp = -0x8000;
								ss->ramp_ms = (ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						case 6:
							ss->ramp_ms = (((command&255)+1)<<5);
							ramp = (255-(short)ss->brightness) * 1024L / ss->ramp_ms;
							if( ramp > 0x7FFF ) {
								ss->ramp = 0x7FFF;
								ss->ramp_ms = (255-ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						case 7:
							ss->ramp_ms = (((command&255)+1)<<5);
							ramp = -(short)ss->brightness * 1024L / ss->ramp_ms;
							if( ramp < -0x8000 ) {
								ss->ramp = -0x8000;
								ss->ramp_ms = (ss->brightness) / 32 + 1;
							} else {
								ss->ramp = ramp;
							}
							break;
						}
					}
					break;
				}
				break;
			}
			break;
		}
	}
}

void sequence_advance(const sequence* sq, sequence_state* ss, int ms, bool bUpTo, bool bConsumeMax) {
	int ms_period;

	if( ms < ss->ms_pos )
		sequence_init(sq, ss);

	ms_period = ms - ss->ms_pos;
	ss->ms_pos = ms;

	while( ms_period ) {
		if( ms_period < ss->ms_remain ) {
			sequence_do_ramp(ss, ms_period);
			ss->ms_remain -= ms_period;
			ms_period = 0;
		} else {
			sequence_do_ramp(ss, ss->ms_remain);
			ms_period -= ss->ms_remain;
			ss->ms_remain = 0;
			if( (bUpTo && ms_period == 0) || (ss->pos == sq->used) )
				break;
			while( ss->pos < sq->used && ss->ms_remain == 0 ) {
				sequence_execute_command(sq, ss);
			}
			if( ss->ms_remain == 0 && ms_period != 0 ) {
#ifdef _DEBUG
				assert(ss->pos == sq->used); // safety check to avoid endless loops
#endif
				break;
			}
		}
	}

	if( bConsumeMax ) {
		while( ss->ms_remain && ss->pos < sq->used && IS_DELAY_CMD(sq->commands[ss->pos]) )
			sequence_execute_command(sq, ss);
	}
}

void sequence_advance_past(const sequence* sq, sequence_state* ss, int ms) {
	sequence_advance(sq, ss, ms);
	while( ss->pos < sq->used && !ss->ms_remain )
		sequence_execute_command(sq, ss);
	while( ss->pos < sq->used && IS_DELAY_CMD(sq->commands[ss->pos]) )
		sequence_execute_command(sq, ss);
}

void sequence_close() {
	g_bLoading = true;
	clear_all_sequences();
	g_bLoading = false;
}

static void push_command(unsigned short*& ret, int& len, int& used, unsigned short cmd) {
	if( used >= len ) {
		len += 512;
		ret = (unsigned short*)realloc(ret, len * sizeof(unsigned short));
	}
	ret[used++] = cmd;
}

typedef enum { no_commands, light_on, light_off, light_other } command_type;

void sequence_skip_redundant_commands(sequence* dest, sequence_state* state) {
	int num_ramp_cmds = 0, num_non_ramp_cmds = 0;
	int pos = state->pos;
	while( pos < dest->used && !IS_DELAY_CMD(dest->commands[pos]) ) {
		if( IS_RAMP_CMD(dest->commands[pos]) )
			++num_ramp_cmds;
		else
			++num_non_ramp_cmds;
		++pos;
	}
	while( state->pos < dest->used && !IS_DELAY_CMD(dest->commands[state->pos]) ) {
		if( IS_RAMP_CMD(dest->commands[state->pos]) ) {
			if( num_ramp_cmds == 1 && num_non_ramp_cmds == 0 )
				break;
			--num_ramp_cmds;
		} else {
			if( num_non_ramp_cmds == 1 )
				break;
			--num_non_ramp_cmds;
		}
		++state->pos;
	}
}

static unsigned short* sequence_interleave_sequences(int& lenToHere) {
	unsigned short* ret = 0;
	int len = 0, used = 0;
	command_type types[32];
	sequence_state state[32];
	bool bCanUse;
	int first, second, num_actual;
	int ms_pos = 0;

	for( int i = 0; i < 32; ++i )
		sequence_init(&sequences[i], &state[i]);

	while(1) {
	RestartProcessing:
		int min_delay = -1;
		
		// analyze the current state
		for( int i = 0; i < 32; ++i ) {
			sequence_skip_redundant_commands(&sequences[i], &state[i]);
			if( state[i].pos < sequences[i].used ) {
				unsigned short cmd = sequences[i].commands[state[i].pos];
				if( state[i].ms_remain || IS_DELAY_CMD(cmd) ) {
					types[i] = no_commands;
					int delay = 0;
					if( IS_DELAY_CMD(cmd) )
						delay = DELAY_MS(cmd);
					delay += state[i].ms_remain;
					if( min_delay == -1 || delay < min_delay )
						min_delay = delay;
				} else if( (cmd>>13) == 1 && (cmd&255) == 255 ) {
					types[i] = light_on;
				} else if( (cmd>>13) == 1 && (cmd&255) == 0 ) {
					types[i] = light_off;
				} else {
					types[i] = light_other;
				}
			} else {
				types[i] = no_commands;
			}
		}
		
		// see if it is time for a delay
		bCanUse = true;
		for( int i = 0; i < 32; ++i ) {
			if( types[i] != no_commands ) {
				bCanUse = false;
				break;
			}
		}
		if( bCanUse ) {
			unsigned short cmd;
			if( min_delay == -1 )
				break; // end of sequence
			if( min_delay > 4096 )
				min_delay = 4096;
			if( min_delay <= 256 ) {
				cmd = (1<<8)|(min_delay-1);
			} else if( min_delay <= 1024 ) {
				cmd = (2<<8)|((min_delay>>2)-1);
				min_delay &= ~3;
			} else {
				cmd = (3<<8)|((min_delay>>4)-1);
				min_delay &= ~15;
			}
			push_command(ret, len, used, cmd);
			ms_pos += min_delay;
			for( int i = 0; i < 32; ++i )
				sequence_advance(&sequences[i], &state[i], ms_pos, true);
			goto RestartProcessing;
		}

		// see if we can use a "turn all lights off" command
		bCanUse = true;
		num_actual = 0;
		for( int i = 0; i < 32; ++i ) {
			if( types[i] == no_commands && (state[i].brightness != 0 || state[i].ramp != 0) ) {
				bCanUse = false;
				break;
			}
			if( types[i] == light_off )
				++num_actual;
		}
		if( bCanUse && num_actual > 2 ) {
			push_command(ret, len, used, 0);
			for( int i = 0; i < 32; ++i )
				if( types[i] == light_off )
					sequence_execute_command(&sequences[i], &state[i]);
			goto RestartProcessing;
		}

		// see if we can use a "set light set" command
		for( int slave = 0; slave < 4; ++slave ) {
			bCanUse = true;
			unsigned char val = 0;
			num_actual = 0;
			for( int i = slave*8; i < (slave+1)*8; ++i ) {
				if( !(types[i] == light_on || types[i] == light_off || (types[i] == no_commands && (state[i].brightness == 0 || state[i].brightness == 255) && state[i].ramp == 0)) ) {
					bCanUse = false;
					break;
				}
				if( types[i] != no_commands )
					++num_actual;
				if( types[i] == light_on || (types[i] == no_commands && state[i].brightness == 255) )
					val |= (1<<(i&7));
			}
			if( bCanUse && num_actual > 1 ) {
				push_command(ret, len, used, (1<<10)|(slave<<8)|val);
				for( int i = slave*8; i < (slave+1)*8; ++i ) {
					if( types[i] != no_commands )
						sequence_execute_command(&sequences[i], &state[i]);
				}
				goto RestartProcessing;
			}
		}

		// see if we can use a "turn on/off lights in set" command
		for( int slave = 0; slave < 4; ++slave ) {
			int num_affected = 0;
			unsigned char val = 0;
			for( int i = slave*8; i < (slave+1)*8; ++i ) {
				if( types[i] == light_off ) {
					++num_affected;
					val |= (1<<(i&7));
				}
			}
			if( num_affected >= 2 ) {
				push_command(ret, len, used, (2<<10)|(slave<<8)|val);
				for( int i = slave*8; i < (slave+1)*8; ++i ) {
					if( types[i] == light_off )
						sequence_execute_command(&sequences[i], &state[i]);
				}
				goto RestartProcessing;
			}

			num_affected = 0;
			val = 0;
			for( int i = slave*8; i < (slave+1)*8; ++i ) {
				if( types[i] == light_on ) {
					++num_affected;
					val |= (1<<(i&7));
				}
			}
			if( num_affected >= 2 ) {
				push_command(ret, len, used, (3<<10)|(slave<<8)|val);
				for( int i = slave*8; i < (slave+1)*8; ++i ) {
					if( types[i] == light_on )
						sequence_execute_command(&sequences[i], &state[i]);
				}
				goto RestartProcessing;
			}
		}

		// see if we can use a "turn on/off lights x and y" command
		first = -1;
		second = -1;
		for( int i = 0; i < 32; ++i ) {
			if( types[i] == light_on || types[i] == light_off ) {
				if( first == -1 ) {
					first = i;
				} else {
					second = i;
					break;
				}
			}
		}
		if( second != -1 ) {
			if( types[first] == light_on ) {
				if( types[second] == light_on ) {
					push_command(ret, len, used, (4<<10)|first|(second<<5));
				} else {
					push_command(ret, len, used, (6<<10)|first|(second<<5));
				}
			} else {
				if( types[second] == light_on ) {
					push_command(ret, len, used, (6<<10)|second|(first<<5));
				} else {
					push_command(ret, len, used, (5<<10)|first|(second<<5));
				}
			}
			sequence_execute_command(&sequences[first], &state[first]);
			sequence_execute_command(&sequences[second], &state[second]);
			goto RestartProcessing;
		}

		// now deal with either setting brightness, ramps, or individual light on/off commands
		for( int i = 0; i < 32; ++i ) {
			if( types[i] != no_commands ) {
				unsigned short cmd = sequences[i].commands[state[i].pos];
				if( (cmd>>13) ) {
					push_command(ret, len, used, (cmd&0xE0FF)|(i<<8));
				} else {
#ifdef _DEBUG
					assert(0);
#endif
				}
				sequence_execute_command(&sequences[i], &state[i]);
			}
		}
	}

#ifdef _DEBUG
	sequence temp;
	temp.commands = ret;
	temp.used = used;
	assert(sequence_get_length_ms(&temp) == (wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
#endif

	lenToHere = used;
	return ret;
}

void sequence_deinterleave(sequence* dest, unsigned short* commands, int num) {
	process_commands(commands, num, dest);

	for( int i = 0; i < 32; ++i )
		sequence_compact(&dest[i], 0, dest[i].used);
}

bool sequence_are_identical(sequence* a, sequence* b) {
	int ms = 0;

	sequence_state ss_a, ss_b;
	sequence_init(a, &ss_a);
	sequence_init(b, &ss_b);
	while( (ss_a.pos < a->used || ss_a.ms_remain) && (ss_b.pos < b->used || ss_b.ms_remain) ) {
		if( ss_a.brightness != ss_b.brightness || ss_a.ramp != ss_b.ramp || (ss_a.ramp ? ss_a.ramp_ms : 0) != (ss_b.ramp ? ss_b.ramp_ms : 0) )
			return false;
		int min_delay = ss_a.ms_remain;
		if( ss_b.ms_remain < min_delay )
			min_delay = ss_b.ms_remain;
		if( min_delay < 1 )
			min_delay = 1;
		ms += min_delay;
		sequence_advance(a, &ss_a, ms, true, true);
		sequence_advance(b, &ss_b, ms, true, true);
	}
	// ignore light off command at end of sequence
	if( ss_a.pos == a->used-1 && (a->commands[ss_a.pos] == 8192 || ((a->commands[ss_a.pos]>>13)==1 && (a->commands[ss_a.pos]&255) == 0)) )
		++ss_a.pos;
	// ignore light off command at end of sequence
	if( ss_b.pos == b->used-1 && (b->commands[ss_b.pos] == 8192 || ((b->commands[ss_b.pos]>>13)==1 && (b->commands[ss_b.pos]&255) == 0)) )
		++ss_b.pos;
	if( ss_a.pos < a->used || ss_a.ms_remain || ss_b.pos < b->used || ss_b.ms_remain ) {
		// lengths differ
		return false;
	} else {
		return true;
	}
}

bool sequence_save(const wchar_t* filename, bool bBackup) {
	int len, namelen;
	bool bRet;
	unsigned short* commands = sequence_interleave_sequences(len);

#ifdef _DEBUG
	// check that merging the sequence hasn't altered them
	sequence temp[32];
	memset(temp, 0, sizeof(temp));
	sequence_deinterleave(temp, commands, len);
	for( int i = 0; i < 32; ++i ) {
		bool bIdentical = sequence_are_identical(&sequences[i], &temp[i]);
		assert( bIdentical );
		sequence_destroy(&temp[i]);
	}
#endif

	if( bBackup && (namelen = wcslen(filename)) > 4 && (!wcscmp(filename+namelen-4, L".lsq") || !wcscmp(filename+namelen-4, L".lsn")) ) {
		FILE* f = _wfopen(filename, L"rb");
		if( f ) {
			int i;
			fclose(f);
			wchar_t* temp1 = (wchar_t*)malloc((wcslen(filename)+1)*sizeof(wchar_t));
			wcscpy(temp1, filename);
			temp1[namelen-2] = '~';
			wchar_t* temp2 = (wchar_t*)malloc((wcslen(filename)+1)*sizeof(wchar_t));
			wcscpy(temp2, filename);
			temp2[namelen-2] = '~';
			for( i = 0; i < 10; ++i ) {
				temp1[namelen-1] = '0' + i;
				f = _wfopen(temp1, L"rb");
				if( !f )
					break;
				fclose(f);
			}
			while( i > 0 ) {
				temp1[namelen-1] = '0' + i;
				temp2[namelen-1] = '0' + (i-1);
				MoveFileEx(temp2, temp1, MOVEFILE_REPLACE_EXISTING);
				--i;
			}
			temp1[namelen-1] = '0';
			MoveFileEx(filename, temp1, MOVEFILE_REPLACE_EXISTING);
			free(temp1);
			free(temp2);
		}
	}

	FILE* f = _wfopen(filename, L"wb");
	if( !f ) {
		free(commands);
		return false;
	}
	bRet = fwrite(commands, sizeof(unsigned short), len, f) == len;
	free(commands);
	fclose(f);
	return bRet;
}

int sequence_append_delay(sequence* sq, int milliseconds) {
	int ret = 0;
	while( milliseconds ) {
		int ms = milliseconds;
		if( ms > 4096 )
			ms = 4096;

		unsigned short cmd;
		if( ms <= 256 ) {
			cmd = (1<<8)|(ms-1);
			milliseconds -= ms;
		} else if( ms <= 1024 ) {
			cmd = (2<<8)|((ms>>2)-1);
			milliseconds -= ms&~3;
		} else {
			cmd = (3<<8)|((ms>>4)-1);
			milliseconds -= ms&~15;
		}
		sequence_append_command(sq, cmd);
		++ret;
	}
	return ret;
}

#ifdef _DEBUG
static int sequence_get_state_pos_ms(sequence* dest, sequence_state* state) {
	int ret = -state->ms_remain;
	int pos = state->pos;
	while( --pos >= 0 )
		if( IS_DELAY_CMD(dest->commands[pos]) )
			ret += DELAY_MS(dest->commands[pos]);
	return ret;
}
#endif

void sequence_break_delay(sequence* dest, sequence_state* state, sequence_state* other_state) {
	if( state->ms_remain ) {
#ifdef _DEBUG
		int old_ms = sequence_get_length_ms(dest);
		int old_pos = sequence_get_state_pos_ms(dest, state);
		int old_pos2 = other_state ? sequence_get_state_pos_ms(dest, other_state) : 0;
#endif
		unsigned short command = dest->commands[state->pos-1];
		int ms = DELAY_MS(command);
		int ms_a = ms - state->ms_remain;
		int ms_a_remainder;
		int ms_b = state->ms_remain;
		int ms_b_remainder;

		if( ms_a <= 256 ) {
			command = (1<<8)|(ms_a-1);
			ms_a_remainder = 0;
		} else if( ms_a <= 1024 ) {
			command = (2<<8)|((ms_a>>2)-1);
			ms_a_remainder = ms_a - ((ms_a>>2)<<2);
		} else {
			command = (3<<8)|((ms_a>>4)-1);
			ms_a_remainder = ms_a - ((ms_a>>4)<<4);
		}
		dest->commands[state->pos-1] = command;
		if( ms_a_remainder ) {
			sequence_insert_command(dest, state->pos-1, 0x0100|(ms_a_remainder - 1));
			if( other_state && other_state->pos >= state->pos && other_state->pos < dest->used )
				other_state->pos += 1;
			state->pos += 1;
		}
		state->ms_remain = 0;

		if( ms_b <= 256 ) {
			command = (1<<8)|(ms_b-1);
			ms_b_remainder = 0;
		} else if( ms_b <= 1024 ) {
			command = (2<<8)|((ms_b>>2)-1);
			ms_b_remainder = ms_b - ((ms_b>>2)<<2);
		} else {
			command = (3<<8)|((ms_b>>4)-1);
			ms_b_remainder = ms_b - ((ms_b>>4)<<4);
		}
		sequence_insert_command(dest, state->pos, command);
		if( other_state && other_state->pos >= state->pos ) {
			if( other_state->pos == state->pos ) {
				if( other_state->ms_remain <= ms_b_remainder && ms_b_remainder ) {
					other_state->pos += 1;
				} else {
					other_state->ms_remain -= ms_b_remainder;
				}
			} else {
				if( ms_b_remainder )
					other_state->pos += 1;
			}
			other_state->pos += 1;
		}
		if( ms_b_remainder )
			sequence_insert_command(dest, state->pos+1, 0x0100|(ms_b_remainder - 1));
#ifdef _DEBUG
		assert( old_ms == sequence_get_length_ms(dest) );
		assert( old_pos == sequence_get_state_pos_ms(dest, state) );
		if( other_state ) {
			assert(other_state->ms_remain >= 0);
			assert( old_pos2 == sequence_get_state_pos_ms(dest, other_state) );
		}
#endif
	}
}

int sequence_insert_delay(sequence* dest, int milliseconds, sequence_state* state, sequence_state* other_state) {
	bool bAdvanceOther = false;
	if( other_state && other_state->ms_pos >= state->ms_pos ) {
		other_state->ms_pos += milliseconds;
		bAdvanceOther = true;
	}

	int ret = 0;
	while( milliseconds ) {
		int ms = milliseconds;
		if( ms > 4096 )
			ms = 4096;

		unsigned short cmd;
		if( ms <= 256 ) {
			cmd = (1<<8)|(ms-1);
			milliseconds -= ms;
		} else if( ms <= 1024 ) {
			cmd = (2<<8)|((ms>>2)-1);
			milliseconds -= ms&~3;
		} else {
			cmd = (3<<8)|((ms>>4)-1);
			milliseconds -= ms&~15;
		}
		sequence_insert_command(dest, state->pos, cmd);
		++state->pos;
		++ret;
		if( bAdvanceOther )
			++other_state->pos;
	}
	return ret;
}

int sequence_replace_delay(sequence* dest, int pos, int milliseconds) {
	int start = pos;
	while( milliseconds ) {
		int ms = milliseconds;
		if( ms > 4096 )
			ms = 4096;

		unsigned short cmd;
		if( ms <= 256 ) {
			cmd = (1<<8)|(ms-1);
			milliseconds -= ms;
		} else if( ms <= 1024 ) {
			cmd = (2<<8)|((ms>>2)-1);
			milliseconds -= ms&~3;
		} else {
			cmd = (3<<8)|((ms>>4)-1);
			milliseconds -= ms&~15;
		}
		dest->commands[pos++] = cmd;
	}
	return pos - start;
}

int sequence_insert_set_brightness(sequence* dest, int which, int brightness, sequence_state* state, sequence_state* other_state) {
	sequence_insert_command(dest, state->pos, (1<<13)|(which<<8)|brightness);
	++state->pos;
	if( other_state && other_state->ms_pos >= state->ms_pos )
		++other_state->pos;
	return 1;
}

int sequence_append_ramp(sequence* dest, int which, int direction, int duration) {
	unsigned short command;
	if( duration <= 2048 ) {
		command = ((direction < 0 ? 3 : 2)<<13)|(which<<8)|(((duration+7)>>3)-1);
	} else if( duration <= 4096 ) {
		command = ((direction < 0 ? 5 : 4)<<13)|(which<<8)|(((duration+15)>>4)-1);
	} else if( duration <= 8192 ) {
		command = ((direction < 0 ? 7 : 6)<<13)|(which<<8)|(((duration+31)>>5)-1);
	} else {
		return 0;
	}
	sequence_append_command(dest, command);
	return 1;
}

void sequence_append_long_ramp(sequence* dest, int which, int start_brightness, int final_brightness, int duration) {
#ifdef _DEBUG
	assert(duration > 8192);
#endif
	int diff = abs(final_brightness - start_brightness);
	int brightness = start_brightness;
	int ms_pos = 0;
	while( brightness != final_brightness && ms_pos < duration ) {
		if( final_brightness > brightness )
			++brightness;
		else
			--brightness;

		int next_ms = abs(start_brightness - brightness) * duration / diff;
		if( next_ms > duration )
			next_ms = duration;
		if( next_ms > ms_pos ) {
			sequence_append_delay(dest, next_ms - ms_pos);
			ms_pos = next_ms;
		}
		sequence_append_command(dest, (1<<13)|(which<<8)|brightness);
	}
	if( ms_pos < duration )
		sequence_append_delay(dest, duration - ms_pos);
}

int sequence_insert_ramp(sequence* dest, int which, int direction, int duration, sequence_state* state, sequence_state* other_state) {
	unsigned short command;
	if( duration <= 2048 ) {
		command = ((direction < 0 ? 3 : 2)<<13)|(which<<8)|(((duration+7)>>3)-1);
	} else if( duration <= 4096 ) {
		command = ((direction < 0 ? 5 : 4)<<13)|(which<<8)|(((duration+15)>>4)-1);
	} else if( duration <= 8192 ) {
		command = ((direction < 0 ? 7 : 6)<<13)|(which<<8)|(((duration+31)>>5)-1);
	} else {
		return 0;
	}
	sequence_insert_command(dest, state->pos, command);
	++state->pos;
	if( other_state && other_state->ms_pos >= state->ms_pos )
		++other_state->pos;
	return 1;
}

static int trim_non_delay_cmds(sequence* dest, int start, int finish) {
	int ret = 0;
	int first_keep = finish-1;
	while( first_keep > start && IS_RAMP_CMD(dest->commands[first_keep]) )
		--first_keep;
	int last_keep = first_keep+1;
	if( last_keep < finish && IS_RAMP_CMD(dest->commands[last_keep]) )
		++last_keep;

	if( first_keep > start ) {
		ret = first_keep - start;
		sequence_remove_command(dest, start, ret);
		last_keep -= ret;
		finish -= ret;
	}
	if( last_keep < finish ) {
		ret += finish - last_keep;
		sequence_remove_command(dest, last_keep, finish - last_keep);
	}
	return ret;
}

void sequence_compact(sequence* dest, int start, int finish) {
#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif

	while( start > 0 && IS_DELAY_CMD(dest->commands[start]) )
		--start;
	while( finish < dest->used && IS_DELAY_CMD(dest->commands[finish]) )
		++finish;

	sequence_state ss, old_ss;
	sequence_init(dest, &ss);
	ss.brightness = -1;
	while( ss.pos < start )
		sequence_execute_command(dest, &ss);
	int last_useful_command = ss.pos;
	while( ss.pos < finish ) {
		old_ss = ss;
		sequence_execute_command(dest, &ss);
		if( ss.ms_remain == old_ss.ms_remain && ss.brightness == old_ss.brightness && ss.ramp == old_ss.ramp && ss.ramp_remainder == old_ss.ramp_remainder && ss.ramp_ms == old_ss.ramp_ms ) {
			// this is a redundant command
		} else {
			if( last_useful_command < ss.pos - 1 ) {
				memmove(dest->commands + last_useful_command, dest->commands + (ss.pos - 1), (dest->used - (ss.pos - 1)) * sizeof(unsigned short));
				dest->used -= ((ss.pos - 1) - last_useful_command);
				finish -= ((ss.pos - 1) - last_useful_command);
				ss.pos = last_useful_command + 1;
			}
			last_useful_command = ss.pos;
		}
	}
	if( last_useful_command < ss.pos - 1 ) {
		memmove(dest->commands + last_useful_command, dest->commands + (ss.pos - 1), (dest->used - (ss.pos - 1)) * sizeof(unsigned short));
		dest->used -= ((ss.pos - 1) - last_useful_command);
		finish -= ((ss.pos - 1) - last_useful_command);
	}

	int delay_start = -1;
	int delay_ms;
	int num_non_delay_cmds = 0;
	for( int i = start; i < finish; ++i ) {
		if( IS_DELAY_CMD(dest->commands[i]) ) {
			if( num_non_delay_cmds > 1 ) {
				int num_removed = trim_non_delay_cmds(dest, i - num_non_delay_cmds, i);
				finish -= num_removed;
				i -= num_removed;
#ifdef _DEBUG
				assert(IS_DELAY_CMD(dest->commands[i]));
#endif
			}
			num_non_delay_cmds = 0;

			if( delay_start == -1 ) {
				delay_start = i;
				delay_ms = 0;
			}
			delay_ms += DELAY_MS(dest->commands[i]);
		} else {
			if( delay_start != -1 ) {
				int num = sequence_replace_delay(dest, delay_start, delay_ms);
				if( num < (i - delay_start) ) {
					sequence_remove_command(dest, delay_start + num, (i - delay_start) - num);
					finish -= (i - delay_start) - num;
					i = delay_start + num;
				}
				delay_start = -1;
			}
			++num_non_delay_cmds;
		}
	}
	if( num_non_delay_cmds > 1 )
		trim_non_delay_cmds(dest, finish - num_non_delay_cmds, finish);
	else if( delay_start != -1 ) {
		int num = sequence_replace_delay(dest, delay_start, delay_ms);
		if( num < (finish - delay_start) )
			sequence_remove_command(dest, delay_start + num, (finish - delay_start) - num);
	}
}

int sequence_append_translated_commands(sequence* dest, sequence* source, int start, int finish, int new_light) {
	for( int i = start; i < finish; ++i ) {
		unsigned short command = source->commands[i];
		if( command&0xE000 )
			command = (command&0xE0FF)|(new_light<<8);
		sequence_append_command(dest, command);
	}
	return finish - start;
}

int sequence_replace_translated_commands(sequence* dest, sequence* source, int start, int finish, int new_light, int max_ms) {
	int i, x, delta;

	int len = source->used;
	int full_len = len;
	int add_delay_1 = 0, add_delay_2 = 0;

#ifdef _DEBUG
	assert(g_bLoading || !(dest >= sequences && dest < sequences+32) || undo_nesting);
#endif

	if( max_ms != -1 ) {
		sequence_state ss;
		sequence_init(source, &ss);
		sequence_advance(source, &ss, max_ms, true, false);
		if( ss.pos != source->used || ss.ms_remain ) {
			len = full_len = ss.pos - (ss.ms_remain ? 1 : 0);
			if( ss.ms_remain ) {
				add_delay_1 = DELAY_MS(source->commands[ss.pos-1]) - ss.ms_remain;
				if( add_delay_1 )
					++full_len;
				add_delay_2 = DELAY_MS_REMAINDER(add_delay_1);
				if( add_delay_2 )
					++full_len;
			}
		}
	}

	delta = full_len - (finish-start);
	if( delta > 0 ) {
		while( dest->used + delta > dest->len )
			grow_sequence(dest);
		memmove(dest->commands + finish + delta, dest->commands + finish, (dest->used - finish) * sizeof(unsigned short));
		dest->used += delta;
		finish += delta;
	}
	for( i = start, x = 0; i < finish && x < len; ++i ) {
		unsigned short command = source->commands[x++];
		if( command&0xE000 )
			command = (command&0xE0FF)|(new_light<<8);
		dest->commands[i] = command;
	}
	if( add_delay_1 ) {
		if( add_delay_1 <= 256 )
			dest->commands[i++] = 256|(add_delay_1-1);
		else if( add_delay_1 <= 1024 )
			dest->commands[i++] = 512|((add_delay_1>>2)-1);
		else
			dest->commands[i++] = 768|((add_delay_1>>4)-1);
	}
	if( add_delay_2 )
		dest->commands[i++] = 256|(add_delay_2-1);
	if( delta < 0 )
		sequence_remove_command(dest, i, -delta);

	return delta;
}

int sequence_get_length_ms(sequence* pSeq) {
	int pos = 0;
	int ret = 0;
	while( pos < pSeq->used ) {
		if( IS_DELAY_CMD(pSeq->commands[pos]) ) {
			ret += DELAY_MS(pSeq->commands[pos]);
		}
		++pos;
	}
	return ret;
}

void sequence_merge(sequence* pDest, sequence* pSource1, sequence_state* pState1, sequence* pSource2, sequence_state* pState2, int which, int max_length) {
	sequence_state DestState;
	sequence_init(pDest, &DestState);

#ifdef _DEBUG
	assert(g_bLoading || !(pDest >= sequences && pDest < sequences+32) || undo_nesting);
#endif

	int last_used = 1;
	int ms_pos = 0;
	DestState.brightness = -1;
	while( (pState1->pos < pSource1->used || pState1->ms_remain) && (pState2->pos < pSource2->used || pState2->ms_remain) ) {
		if( !pState1->ms_remain )
			while( pState1->pos < pSource1->used && !IS_DELAY_CMD(pSource1->commands[pState1->pos]) )
				sequence_execute_command(pSource1, pState1);
		if( !pState2->ms_remain )
			while( pState2->pos < pSource2->used && !IS_DELAY_CMD(pSource2->commands[pState2->pos]) )
				sequence_execute_command(pSource2, pState2);

		if( pState1->brightness > pState2->brightness || (pState1->brightness == pState2->brightness && pState1->ramp > pState2->ramp) ) {
			last_used = 1;
		} else if( pState2->brightness > pState1->brightness || (pState1->brightness == pState2->brightness && pState2->ramp > pState1->ramp) ) {
			last_used = 2;
		} else if( last_used == 0 ) {
			last_used = 1;
		}

		sequence_state* pUsed = last_used == 1 ? pState1 : pState2;

		if( DestState.ms_pos != ms_pos ) {
			if( ms_pos > max_length )
				ms_pos = max_length;
			sequence_append_delay(pDest, ms_pos - DestState.ms_pos);
			sequence_advance(pDest, &DestState, ms_pos, true);
			if( ms_pos == max_length )
				break;
		}
		bool bTooFast = pUsed->ramp_ms > 0 && pUsed->ramp_ms < 8 && DestState.ramp_ms > 0 && DestState.ramp_ms < 8 && (DestState.ramp_ms < 0 ? -1 : 1) == (pUsed->ramp_ms < 0 ? -1 : 1);
		if( !bTooFast ) {
			if( DestState.brightness != pUsed->brightness && (!DestState.ramp || pUsed->ramp_ms != DestState.ramp_ms || abs((int)DestState.brightness - (int)pUsed->brightness) > 1 ) ) {
				sequence_append_command(pDest, (1<<13)|(which<<8)|pUsed->brightness);
				sequence_execute_command(pDest, &DestState);
			}
			if( (DestState.ramp > 0 ? DestState.ramp_ms : DestState.ramp < 0 ? -DestState.ramp_ms : 0) != (pUsed->ramp > 0 ? pUsed->ramp_ms : pUsed->ramp < 0 ? -pUsed->ramp_ms : 0) ) {
				if( !bTooFast ) {
					sequence_append_ramp(pDest, which, pUsed->ramp < 0 ? -1 : 1, pUsed->ramp_ms);
					sequence_execute_command(pDest, &DestState);
					DestState.ramp_ms = pUsed->ramp_ms; // takes care of the fact that ramp_ms has a minimum resolution of 8ms in the command stream
				}
			}
		}

		int min_delay;
		if( pState1->ramp || pState2->ramp ) {
			min_delay = 1;
		} else {
			min_delay = pState1->ms_remain;
			if( pState2->ms_remain < min_delay )
				min_delay = pState2->ms_remain;
			if( min_delay < 1 )
				min_delay = 1;
		}
		sequence_advance(pSource1, pState1, pState1->ms_pos + min_delay, true);
		sequence_advance(pSource2, pState2, pState2->ms_pos + min_delay, true);
		ms_pos += min_delay;
	}
	if( DestState.ms_pos != ms_pos ) {
		if( ms_pos > max_length )
			ms_pos = max_length;
		sequence_append_delay(pDest, ms_pos - DestState.ms_pos);
	}
	sequence_compact(pDest, 0, pDest->used);
	wchar_t debug[256];
	wsprintf(debug, L"Merged size = %d\n", (int)pDest->used);
	OutputDebugString(debug);
}

void sequence_copy(sequence* pDest, sequence* pSource) {
#ifdef _DEBUG
	assert(g_bLoading || !(pDest >= sequences && pDest < sequences+32) || undo_nesting);
#endif

	pDest->len = pDest->used = pSource->used;
	pDest->commands = (unsigned short*)malloc(pDest->used * sizeof(unsigned short));
	memcpy(pDest->commands, pSource->commands, pDest->used * sizeof(unsigned short));
}

void sequence_stretch(sequence* pDest, sequence* pSource, int new_len_ms) {
	int old_len_ms = sequence_get_length_ms(pSource);

#ifdef _DEBUG
	assert(g_bLoading || !(pDest >= sequences && pDest < sequences+32) || undo_nesting);
#endif

	sequence_state ss;
	sequence_init(pSource, &ss);
	sequence_destroy(pDest);

	int ms_pos_dest = 0, ms_pos_source = 0;
	while( ss.pos < pSource->used ) {
		unsigned short cmd = pSource->commands[ss.pos];
		if( IS_DELAY_CMD(cmd) ) {
			ms_pos_source += DELAY_MS(cmd);
			while( ss.pos+1 < pSource->used && IS_DELAY_CMD(pSource->commands[ss.pos+1]) ) {
				sequence_execute_command(pSource, &ss);
				ms_pos_source += DELAY_MS(pSource->commands[ss.pos]);
			}
			int new_ms_pos_dest = ((unsigned long long)ms_pos_source * new_len_ms + (old_len_ms/2)) / old_len_ms;
			if( new_ms_pos_dest > ms_pos_dest ) {
				sequence_append_delay(pDest, new_ms_pos_dest - ms_pos_dest);
				ms_pos_dest = new_ms_pos_dest;
			}
		} else if( IS_RAMP_CMD(cmd) ) {
			int ramp_ms;
			switch(cmd>>13) {
			case 2:
			case 3:
				ramp_ms = (((cmd&255)+1)<<3);
				break;
			case 4:
			case 5:
				ramp_ms = (((cmd&255)+1)<<4);
				break;
			case 6:
			case 7:
				ramp_ms = (((cmd&255)+1)<<5);
				break;
			}
			int new_ms_pos_dest = ((unsigned long long)(ms_pos_source + ramp_ms) * new_len_ms + (old_len_ms/2)) / old_len_ms;
			int new_ms = new_ms_pos_dest - ms_pos_dest;
			if( new_ms ) {
				if( !sequence_append_ramp(pDest, (cmd>>8)&31, ((cmd>>13)&1) ? -1 : 1, new_ms) ) {
					int start_brightness = ss.brightness;
					int final_brightness = ((cmd>>13)&1) ? 0 : 255;
					int which = (cmd>>8)&31;

					sequence_execute_command(pSource, &ss);
					int total_delay_ms = 0;
					while( ss.pos < pSource->used && IS_DELAY_CMD(pSource->commands[ss.pos]) ) {
						total_delay_ms += DELAY_MS(pSource->commands[ss.pos]);
						sequence_execute_command(pSource, &ss);
					}

					int diff = abs(final_brightness - start_brightness);
					int brightness = start_brightness;
					int ms_start = ms_pos_source;
					while( brightness != final_brightness ) {
						if( final_brightness > brightness )
							++brightness;
						else
							--brightness;

						int next_source_ms = ms_start + abs(start_brightness - brightness) * ramp_ms / diff;
						if( next_source_ms - ms_pos_source > total_delay_ms )
							next_source_ms = ms_pos_source + total_delay_ms;
						total_delay_ms -= (next_source_ms - ms_pos_source);
						int new_ms_pos_dest = ((unsigned long long)next_source_ms * new_len_ms + (old_len_ms/2)) / old_len_ms;
						if( new_ms_pos_dest > ms_pos_dest ) {
							sequence_append_delay(pDest, new_ms_pos_dest - ms_pos_dest);
							ms_pos_dest = new_ms_pos_dest;
						}
						ms_pos_source = next_source_ms;
						sequence_append_command(pDest, (1<<13)|(which<<8)|brightness);
						if( total_delay_ms == 0 )
							break;
					}
					if( total_delay_ms ) {
						int next_source_ms = ms_start + total_delay_ms;
						int new_ms_pos_dest = ((unsigned long long)next_source_ms * new_len_ms + (old_len_ms/2)) / old_len_ms;
						if( new_ms_pos_dest > ms_pos_dest ) {
							sequence_append_delay(pDest, new_ms_pos_dest - ms_pos_dest);
							ms_pos_dest = new_ms_pos_dest;
						}
						ms_pos_source = next_source_ms;
					}
					continue;
				}
			}
		} else {
			sequence_append_command(pDest, cmd);
		}
		sequence_execute_command(pSource, &ss);
	}

	sequence_compact(pDest, 0, pDest->used);
#ifdef _DEBUG
	assert(sequence_get_length_ms(pDest) == new_len_ms);
#endif
}

void sequence_mix(sequence* pDest, sequence* pSource1, sequence_state* pState1, sequence* pSource2, sequence_state* pState2, int which, int max_length) {
	sequence_state DestState;
	sequence_init(pDest, &DestState);

#ifdef _DEBUG
	assert(g_bLoading || !(pDest >= sequences && pDest < sequences+32) || undo_nesting);
#endif

	if( !pState1->ms_remain ) {
		while( pState1->pos < pSource1->used && !IS_DELAY_CMD(pSource1->commands[pState1->pos]) )
			sequence_execute_command(pSource1, pState1);
		while( pState1->pos < pSource1->used && IS_DELAY_CMD(pSource1->commands[pState1->pos]) )
			sequence_execute_command(pSource1, pState1);
	}
	if( !pState2->ms_remain ) {
		while( pState2->pos < pSource2->used && !IS_DELAY_CMD(pSource2->commands[pState2->pos]) )
			sequence_execute_command(pSource2, pState2);
		while( pState2->pos < pSource2->used && IS_DELAY_CMD(pSource2->commands[pState2->pos]) )
			sequence_execute_command(pSource2, pState2);
	}

	int ms_pos = 0;
	DestState.brightness = -1;
	do {
		int dest_brightness = pState1->brightness * pState2->brightness / 255;
		int dest_ramp = 0;
		int dest_ramp_ms = 0;
		if( pState1->ramp && pState1->ramp_ms ) {
			if( pState2->ramp && pState2->ramp_ms ) {
				sequence_state temp1 = *pState1, temp2 = *pState2;
				dest_ramp_ms = min(pState1->ramp_ms, pState2->ramp_ms);
				sequence_do_ramp(&temp1, dest_ramp_ms);
				sequence_do_ramp(&temp2, dest_ramp_ms);
				int new_brightness = temp1.brightness * temp2.brightness / 255;
				dest_ramp = (new_brightness - dest_brightness) * 1024 / dest_ramp_ms;
				if( dest_ramp > 0 )
					dest_ramp_ms = (((255 - dest_brightness) << 10) + 512) /  dest_ramp;
				else if( dest_ramp < 0 )
					dest_ramp_ms = (((      dest_brightness) << 10) + 512) / -dest_ramp;
				else
					dest_ramp_ms = 0;
			} else if( pState1->ramp && pState1->ramp_ms && pState2->brightness ) {
				dest_ramp = pState1->ramp * pState2->brightness / 255;
				dest_ramp_ms = pState1->ramp_ms * 255 / pState2->brightness;
			}
		} else {
			if( pState2->ramp && pState2->ramp_ms && pState1->brightness ) {
				dest_ramp = pState2->ramp * pState1->brightness / 255;
				dest_ramp_ms = pState2->ramp_ms * 255 / pState1->brightness;
			}
		}

		if( DestState.brightness != dest_brightness ) {
			sequence_append_command(pDest, (1<<13)|(which<<8)|dest_brightness);
			DestState.ramp = 0;
			DestState.ramp_ms = 0;
		}
		if( DestState.ramp != dest_ramp || DestState.ramp_ms != dest_ramp_ms ) {
			if( dest_ramp == 0 || dest_ramp_ms == 0 ) {
				sequence_append_command(pDest, (1<<13)|(which<<8)|dest_brightness);
				DestState.ramp = 0;
				DestState.ramp_ms = 0;
			} else if( !sequence_append_ramp(pDest, which, dest_ramp > 0 ? 1 : -1, dest_ramp_ms) ) {
				sequence_state temp1 = *pState1;
				sequence_state temp2 = *pState2;
				int maxdelay1 = temp1.ms_remain, maxdelay2 = temp2.ms_remain;
				while( temp1.pos < pSource1->used && IS_DELAY_CMD(pSource1->commands[temp1.pos]) )
					maxdelay1 += DELAY_MS(pSource1->commands[temp1.pos++]);
				while( temp2.pos < pSource2->used && IS_DELAY_CMD(pSource2->commands[temp2.pos]) )
					maxdelay2 += DELAY_MS(pSource2->commands[temp2.pos++]);

				int total_delay_ms = min(min(maxdelay1, maxdelay2), dest_ramp_ms);
				int start_brightness = dest_brightness;
				int final_brightness = dest_ramp > 0 ? 255 : 0;

				int diff = abs(final_brightness - start_brightness);
				int brightness = start_brightness;
				int ms_start = 0, ms_now = 0;
				while( brightness != final_brightness ) {
					if( final_brightness > brightness )
						++brightness;
					else
						--brightness;

					int next_ms = ms_start + abs(start_brightness - brightness) * dest_ramp_ms / diff;
					if( next_ms > total_delay_ms )
						next_ms = total_delay_ms;
					if( next_ms > ms_now ) {
						sequence_append_delay(pDest, next_ms - ms_now);
						ms_now = next_ms;
					}
					sequence_append_command(pDest, (1<<13)|(which<<8)|brightness);
					if( ms_now - ms_start >= total_delay_ms )
						break;
				}
				if( ms_now < ms_start + total_delay_ms )
					sequence_append_delay(pDest, total_delay_ms - (ms_now - ms_start));
				ms_pos += ms_now;
				sequence_advance(pDest, &DestState, ms_pos);
			}
		}

		int min_delay = pState1->ms_remain;
		if( pState2->ms_remain < min_delay )
			min_delay = pState2->ms_remain;
		if( pState1->ramp_ms && min_delay > pState1->ramp_ms )
			min_delay = pState1->ramp_ms;
		if( pState2->ramp_ms && min_delay > pState2->ramp_ms )
			min_delay = pState2->ramp_ms;
		if( min_delay < 1 )
			min_delay = 1;

		sequence_advance(pSource1, pState1, pState1->ms_pos + min_delay);
		sequence_advance(pSource2, pState2, pState2->ms_pos + min_delay);
		ms_pos += min_delay;
		if( ms_pos > max_length )
			ms_pos = max_length;

		if( DestState.ms_pos != ms_pos ) {
			sequence_append_delay(pDest, ms_pos - DestState.ms_pos);
			sequence_advance(pDest, &DestState, ms_pos);
		}
	} while( DestState.ms_pos < max_length && (pState1->pos < pSource1->used || pState1->ms_remain) && (pState2->pos < pSource2->used || pState2->ms_remain) );

	if( ms_pos > max_length )
		ms_pos = max_length;
	if( DestState.ms_pos != ms_pos )
		sequence_append_delay(pDest, ms_pos - DestState.ms_pos);
}

void sequence_truncate(sequence* pDest, int max_len_ms, int min_len_ms) {
#ifdef _DEBUG
	assert(g_bLoading || !(pDest >= sequences && pDest < sequences+32) || undo_nesting);
#endif

	sequence_state ss;
	sequence_init(pDest, &ss);
	sequence_advance(pDest, &ss, max_len_ms, true, false);
	if( ss.pos < pDest->used || ss.ms_remain ) {
		if( ss.ms_remain )
			sequence_break_delay(pDest, &ss, NULL);
		pDest->used = ss.pos;
	} else if( ss.ms_pos < min_len_ms ) {
		sequence_append_delay(pDest, min_len_ms-ss.ms_pos);
	}
}
