
#pragma once

#define VIRTUAL inline
#define EQ0
#define INLINE inline


typedef enum { Lowpass, Highpass } SimpleFilterType;
typedef enum { Bandpass, Notch } ComplexFilterType;
typedef enum { Logarithmic, Linear } FrequencySpacing;

template <typename T>
struct FilterCoefficients {
	T a0, a1, a2;
	T b0, b1, b2;
};

class Filter {
public:
	VIRTUAL void   Reset  ()             EQ0;
	VIRTUAL double Process(double Input) EQ0;
};

class IIRFilter : public Filter {
public:
	INLINE IIRFilter(int SampleRate, double CornerFrequency, double Q, double Gain, int Order, const FilterCoefficients<double>* pCoefficients);
	INLINE ~IIRFilter();
	INLINE void Reset();
	INLINE double Process(double Input);
protected:
	int m_Order;
	double* m_plfCoefficients;
};

class SecondOrderButterworthFilter : public IIRFilter {
public:
	INLINE SecondOrderButterworthFilter(int SampleRate, double CornerFrequency, double Gain, SimpleFilterType Type);
	INLINE double Process(double Input);
protected:
	SimpleFilterType m_Type;
};

class HigherOrderButterworthFilter : public Filter {
public:
	// Order must be even and greater than 0
	INLINE HigherOrderButterworthFilter(int SampleRate, double CornerFrequency, int Order, double Gain, SimpleFilterType Type);
	INLINE ~HigherOrderButterworthFilter();
	INLINE void   Reset  ();
	INLINE double Process(double Input);
protected:
	int m_Order;
	SecondOrderButterworthFilter* m_pFilters;
};


class BandpassFilter : public Filter {
public:
	INLINE BandpassFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, ComplexFilterType Type);
	INLINE void Reset();
	INLINE double Process(double Input);
protected:
	ComplexFilterType m_Type;
	HigherOrderButterworthFilter m_LowpassFilter, m_HighpassFilter;
};


class MultiBandFilter : public Filter {
public:
	INLINE MultiBandFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, int NumBands, FrequencySpacing Spacing);
	INLINE ~MultiBandFilter();
	INLINE void   Reset  ();
	INLINE double Process(double Input);
	INLINE double GetBandSample(int Band);
protected:
	int m_NumBands;
	BandpassFilter* m_pBandFilters;
	double* m_BandSamples;
};


class ExponentialDecay : public Filter {
public:
	INLINE ExponentialDecay(int SampleRate, double lfAverageHalflife);
	INLINE void   Reset  ();
	INLINE double Process(double Input);
protected:
	double m_lfAverageEnergy;
	double m_lfDecayFactor;
};


class AveragingMultiBandFilter : public MultiBandFilter {
public:
	INLINE AveragingMultiBandFilter(int SampleRate, double LowerCornerFrequency, double UpperCornerFrequency, int Order, double Gain, int NumBands, FrequencySpacing Spacing, double lfAverageHalflife);
	INLINE ~AveragingMultiBandFilter();
	INLINE void   Reset  ();
	INLINE double Process(double Input);
	INLINE double GetAverageEnergy(int Band);
protected:
	double* m_AverageEnergy;
	double m_lfDecayFactor;
};
