
#include "targetver.h"
#include "Dialogs.h"
#include "EditSequence.h"
#include "Resource.h"
#include "Win32Utilities.h"
#include "WindowDrawing.h"
#include "LoadSave.h"
#include "Playback.h"
#include "Undo.h"
#include "ToolbarsMenus.h"
#include "WAVBrowse.h"
#include "WindowsShell.h"
#include "Progress.h"


extern HWND hDialog;

struct DialogRegistryEntry {
	int id;
	enum { Button, Edit, /*Spin, */Slider } type;
};

DialogRegistryEntry CascadeRegistryEntries[] = {
	{ IDC_ORDER_LOWTOHIGH, DialogRegistryEntry::Button },
	{ IDC_ORDER_HIGHTOLOW, DialogRegistryEntry::Button },
	{ IDC_ORDER_PINGPONG,  DialogRegistryEntry::Button },
	{ IDC_ORDER_RANDOM,    DialogRegistryEntry::Button },
	{ IDC_EFFECT_FLASH,    DialogRegistryEntry::Button },
	{ IDC_EFFECT_FADE,     DialogRegistryEntry::Button },
	{ IDC_BRIGHTNESS,      DialogRegistryEntry::Edit   },
	{ IDC_INTERVAL,        DialogRegistryEntry::Edit   },
	{ IDC_ONTIME,          DialogRegistryEntry::Edit   },
	{ IDC_RISETIME,        DialogRegistryEntry::Edit   },
	{ IDC_FALLTIME,        DialogRegistryEntry::Edit   },
	{ IDC_PREVIEW,         DialogRegistryEntry::Button },
	{ IDC_MERGE,           DialogRegistryEntry::Button },
	{ IDC_MIX,			   DialogRegistryEntry::Button }
};

DialogRegistryEntry CustomRampRegistryEntries[] = {
	{ IDC_TYPE_RAMPUP,        DialogRegistryEntry::Button },
	{ IDC_TYPE_RAMPDOWN,      DialogRegistryEntry::Button },
	{ IDC_TYPE_PEAK,          DialogRegistryEntry::Button },
	{ IDC_TYPE_TROUGH,        DialogRegistryEntry::Button },
	{ IDC_INITIAL_BRIGHTNESS, DialogRegistryEntry::Edit   },
	{ IDC_FINAL_BRIGHTNESS,   DialogRegistryEntry::Edit   },
	{ IDC_PEAK_BRIGHTNESS,    DialogRegistryEntry::Edit   },
	{ IDC_TROUGH_BRIGHTNESS,  DialogRegistryEntry::Edit   },
	{ IDC_START_DELAY,        DialogRegistryEntry::Edit   },
	{ IDC_END_DELAY,          DialogRegistryEntry::Edit   },
	{ IDC_PEAK_DELAY,         DialogRegistryEntry::Edit   },
	{ IDC_TROUGH_DELAY,       DialogRegistryEntry::Edit   },
	{ IDC_SYMMETRY,           DialogRegistryEntry::Edit   },
	{ IDC_PREVIEW,            DialogRegistryEntry::Button },
	{ IDC_MERGE,              DialogRegistryEntry::Button },
	{ IDC_MIX,				  DialogRegistryEntry::Button }
};

DialogRegistryEntry SettingsRegistryEntries[] = {
	{ IDC_MONITOR_GAMMA,					DialogRegistryEntry::Edit   },
	{ IDC_MAKE_BACKUP_FILES,				DialogRegistryEntry::Button },
	{ IDC_ASSOCIATE_FILE_EXTENSION,			DialogRegistryEntry::Button },
	{ IDC_SD_CARD_DRIVE,					DialogRegistryEntry::Edit   },
	{ IDC_EJECTAFTERPUBLISH,				DialogRegistryEntry::Button },
	{ IDC_AUDIO_DELAY_COMPENSATION,			DialogRegistryEntry::Edit   },
	{ IDC_MAXIMUM_UNDO_MEMORY,				DialogRegistryEntry::Edit   },
	{ IDC_LOOP_PLAYBACK,					DialogRegistryEntry::Button },
	{ IDC_SCROLL_TO_FOLLOW_PLAYBACK_CURSOR,	DialogRegistryEntry::Button }
};

DialogRegistryEntry BeatDetectionRegistryEntries[] = {
	{ IDC_SENSITIVITY,      DialogRegistryEntry::Slider },
	{ IDC_MINIMUM_INTERVAL, DialogRegistryEntry::Edit   },
	{ IDC_DELAY,            DialogRegistryEntry::Edit   },
	{ IDC_ORDER_LOWTOHIGH,  DialogRegistryEntry::Button },
	{ IDC_ORDER_HIGHTOLOW,  DialogRegistryEntry::Button },
	{ IDC_ORDER_PINGPONG,   DialogRegistryEntry::Button },
	{ IDC_ORDER_RANDOM,     DialogRegistryEntry::Button },
	{ IDC_EFFECT_FLASH,     DialogRegistryEntry::Button },
	{ IDC_EFFECT_FADE,      DialogRegistryEntry::Button },
	{ IDC_BRIGHTNESS,       DialogRegistryEntry::Edit   },
	{ IDC_INTERVAL,         DialogRegistryEntry::Edit   },
	{ IDC_ONTIME,           DialogRegistryEntry::Edit   },
	{ IDC_RISETIME,         DialogRegistryEntry::Edit   },
	{ IDC_FALLTIME,         DialogRegistryEntry::Edit   },
	{ IDC_PREVIEW,          DialogRegistryEntry::Button },
	{ IDC_MERGE,            DialogRegistryEntry::Button },
	{ IDC_MIX,				DialogRegistryEntry::Button }
};

DialogRegistryEntry SpectrumAnalysisRegistryEntries[] = {
	{ IDC_FREQUENCY_START,  DialogRegistryEntry::Edit   },
	{ IDC_FREQUENCY_FINISH, DialogRegistryEntry::Edit   },
	{ IDC_STEREO_MONO,      DialogRegistryEntry::Button },
	{ IDC_STEREO_LTOR,      DialogRegistryEntry::Button },
	{ IDC_STEREO_RTOL,      DialogRegistryEntry::Button },
	{ IDC_STEREO_LONLY,     DialogRegistryEntry::Button },
	{ IDC_STEREO_RONLY,     DialogRegistryEntry::Button },
	{ IDC_CHANS_LOWTOHIGH,  DialogRegistryEntry::Button },
	{ IDC_CHANS_HIGHTOLOW,  DialogRegistryEntry::Button },
	{ IDC_CHANS_RANDOM,     DialogRegistryEntry::Button },
	{ IDC_RESPONSE_TRACK,   DialogRegistryEntry::Button },
	{ IDC_RESPONSE_FLASH,   DialogRegistryEntry::Button },
	{ IDC_RESPONSE_FADE,    DialogRegistryEntry::Button },
	{ IDC_SENSITIVITY,      DialogRegistryEntry::Slider },
	{ IDC_BRIGHTNESS,       DialogRegistryEntry::Edit   },
	{ IDC_SCALE_BRIGHTNESS, DialogRegistryEntry::Button },
	{ IDC_ONTIME,           DialogRegistryEntry::Edit   },
	{ IDC_SCALE_DURATION,   DialogRegistryEntry::Button },
	{ IDC_DELAY,			DialogRegistryEntry::Edit   },
	{ IDC_MINIMUM_INTERVAL, DialogRegistryEntry::Edit   },
	{ IDC_RISETIME,         DialogRegistryEntry::Edit   },
	{ IDC_FALLTIME,         DialogRegistryEntry::Edit   },
	{ IDC_PREVIEW,          DialogRegistryEntry::Button },
	{ IDC_MERGE,            DialogRegistryEntry::Button },
	{ IDC_MIX,              DialogRegistryEntry::Button }
};

DialogRegistryEntry AutomaticSequencingRegistryEntries[] = {
//	{ IDC_NUMLIGHTS,			DialogRegistryEntry::Edit   },
	{ IDC_FLASHPERIOD,			DialogRegistryEntry::Edit   },
	{ IDC_BEATDETECTIONPCT,		DialogRegistryEntry::Edit   },
	{ IDC_PINGPONG,				DialogRegistryEntry::Button },
	{ IDC_BEAT_SENSITIVITY,		DialogRegistryEntry::Slider },
	{ IDC_FADE,					DialogRegistryEntry::Button },
	{ IDC_STEREO,				DialogRegistryEntry::Button },
	{ IDC_TRACK,				DialogRegistryEntry::Button },
	{ IDC_SCALE_DURATION,		DialogRegistryEntry::Button },
	{ IDC_SPECTRUM_SENSITIVITY,	DialogRegistryEntry::Slider }
};


int GetNum(wchar_t* buf) {
	wchar_t* end;
	bool bNeg = false;
	while( buf[0] == L' ' )
		++buf;
	if( buf[0] == '-' ) {
		bNeg = true;
		do {
			++buf;
		} while( buf[0] == L' ' );
	}
	int ret = wcstol(buf, &end, 10);
	if( bNeg ) {
		if( ret < 0 )
			return 0;
		ret = -ret;
	}
	while( *end == L',' ) {
		ret = ret * 1000 + wcstol(end+1, &end, 10);
	}
	return ret;
}

bool SaveDialogStateToRegistry(HWND hWnd, wchar_t* wcName, DialogRegistryEntry* pEntries, int NumEntries) {
	wchar_t buf[1024];
	bool bRet = false;
	DWORD Disposition;
	HKEY Dialog;
	wsprintf(buf, L"Software\\Silicon Chip\\Digital Lighting Controller\\Dialogs\\%s", wcName);
	if( RegCreateKeyEx(HKEY_CURRENT_USER, buf, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &Dialog, &Disposition) == ERROR_SUCCESS ) {
		bRet = true;
		for( int i = 0; i < NumEntries; ++i ) {
			HWND hControl = GetSubWindow(hWnd, pEntries[i].id);
			wchar_t name[32], buf[256];
			wsprintf(name, L"%08x", pEntries[i].id);
			LONG ret;
			DWORD data;

			switch(pEntries[i].type) {
			case DialogRegistryEntry::Button:
				data = SendMessage(hControl, BM_GETCHECK, 0, 0);
				ret = RegSetValueEx(Dialog, name, NULL, REG_DWORD, (LPBYTE)&data, sizeof(data));
				break;
/*
			case DialogRegistryEntry::Spin:
				data = LOWORD(SendMessage(hControl, UDM_GETPOS, 0, 0));
				ret = RegSetValueEx(Dialog, name, NULL, REG_DWORD, (LPBYTE)&data, sizeof(data));
				break;
*/
			case DialogRegistryEntry::Edit:
				buf[0] = L'\0';
				GetWindowText(hControl, buf, sizeof(buf)/sizeof(wchar_t));
				ret = RegSetValueEx(Dialog, name, NULL, REG_SZ, (LPBYTE)buf, (wcslen(buf)+1)*sizeof(wchar_t));
				break;
			case DialogRegistryEntry::Slider:
				data = SendMessage(hControl, TBM_GETPOS, 0, 0);
				ret = RegSetValueEx(Dialog, name, NULL, REG_DWORD, (LPBYTE)&data, sizeof(data));
				break;
			}

			if( ret != ERROR_SUCCESS )
				bRet = false;
		}
		RegCloseKey(Dialog);
	}
	return bRet;
}

bool LoadDialogStateFromRegistry(HWND hWnd, wchar_t* wcName, DialogRegistryEntry* pEntries, int NumEntries) {
	wchar_t buf[1024];
	bool bRet = false;
	HKEY Dialog;
	wsprintf(buf, L"Software\\Silicon Chip\\Digital Lighting Controller\\Dialogs\\%s", wcName);
	if( RegOpenKey(HKEY_CURRENT_USER, buf, &Dialog) == ERROR_SUCCESS ) {
		bRet = true;
		for( int i = 0; i < NumEntries; ++i ) {
			HWND hControl = GetSubWindow(hWnd, pEntries[i].id);
			wchar_t name[32], buf[256];
			wsprintf(name, L"%08x", pEntries[i].id);
			LONG ret;
			DWORD data, data_size, Type;

			switch(pEntries[i].type) {
			case DialogRegistryEntry::Button:
				data_size = sizeof(data);
				Type = REG_DWORD;
				ret = RegQueryValueEx(Dialog, name, NULL, &Type, (LPBYTE)&data, &data_size);
				if( ret == ERROR_SUCCESS && Type == REG_DWORD )
					SendMessage(hControl, BM_SETCHECK, (WPARAM)(data != 0), 0);
				break;
/*
			case DialogRegistryEntry::Spin:
				data_size = sizeof(data);
				Type = REG_DWORD;
				ret = RegQueryValueEx(Dialog, name, NULL, &Type, (LPBYTE)&data, &data_size);
				if( ret == ERROR_SUCCESS && Type == REG_DWORD )
					SendMessage(hControl, UDM_SETPOS, 0, (LPARAM)data);
				break;
*/
			case DialogRegistryEntry::Edit:
				data_size = sizeof(buf);
				Type = REG_SZ;
				ret = RegQueryValueEx(Dialog, name, NULL, &Type, (LPBYTE)buf, &data_size);
				if( ret == ERROR_SUCCESS && Type == REG_SZ )
					SetWindowText(hControl, buf);
				break;
			case DialogRegistryEntry::Slider:
				data_size = sizeof(data);
				Type = REG_DWORD;
				ret = RegQueryValueEx(Dialog, name, NULL, &Type, (LPBYTE)&data, &data_size);
				if( ret == ERROR_SUCCESS && Type == REG_DWORD )
					SendMessage(hControl, TBM_SETPOS, TRUE, (LPARAM)data);
				break;
			}

			if( ret != ERROR_SUCCESS )
				bRet = false;
		}
		RegCloseKey(Dialog);
	}
	return bRet;
}

DWORD LoadDwordFromRegistry(const wchar_t* name, DWORD def) {
	DWORD ret = def;
	HKEY Settings;
	if( RegOpenKey(HKEY_CURRENT_USER, L"Software\\Silicon Chip\\Digital Lighting Controller\\Settings", &Settings) == ERROR_SUCCESS ) {
		DWORD size = sizeof(ret), Type = REG_DWORD;
		if( RegQueryValueEx(Settings, name, NULL, &Type, (LPBYTE)&ret, &size) != ERROR_SUCCESS || Type != REG_DWORD )
			ret = def;
		RegCloseKey(Settings);
	}
	return ret;
}
bool SaveDwordToRegistry(const wchar_t* name, DWORD value) {
	DWORD Disposition;
	HKEY Settings;
	if( RegCreateKeyEx(HKEY_CURRENT_USER, L"Software\\Silicon Chip\\Digital Lighting Controller\\Settings", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &Settings, &Disposition) == ERROR_SUCCESS ) {
		bool bRet = RegSetValueEx(Settings, name, NULL, REG_DWORD, (LPBYTE)&value, sizeof(value)) == ERROR_SUCCESS;
		RegCloseKey(Settings);
		return bRet;
	}
	return false;
}
bool LoadStringFromRegistry(const wchar_t* name, wchar_t* to_here, DWORD size) {
	HKEY Settings;
	bool bRet = false;
	if( RegOpenKey(HKEY_CURRENT_USER, L"Software\\Silicon Chip\\Digital Lighting Controller\\Settings", &Settings) == ERROR_SUCCESS ) {
		DWORD Type = REG_SZ;
		if( RegQueryValueEx(Settings, name, NULL, &Type, (LPBYTE)to_here, &size) == ERROR_SUCCESS && Type == REG_SZ )
			bRet = true;
		RegCloseKey(Settings);
	}
	return bRet;
}
bool SaveStringToRegistry(const wchar_t* name, const wchar_t* string) {
	DWORD Disposition;
	HKEY Settings;
	if( RegCreateKeyEx(HKEY_CURRENT_USER, L"Software\\Silicon Chip\\Digital Lighting Controller\\Settings", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &Settings, &Disposition) == ERROR_SUCCESS ) {
		bool bRet = RegSetValueEx(Settings, name, NULL, REG_SZ, (LPBYTE)string, wcslen(string)*sizeof(wchar_t)) == ERROR_SUCCESS;
		RegCloseKey(Settings);
		return bRet;
	}
	return false;
}
bool SetRegistryDialogDword(const wchar_t* dialog_name, int ID, DWORD val) {
	DWORD Disposition;
	HKEY Settings;
	wchar_t buf[_MAX_PATH];
	wsprintf(buf, L"Software\\Silicon Chip\\Digital Lighting Controller\\Dialogs\\%s", dialog_name);
	if( RegCreateKeyEx(HKEY_CURRENT_USER, buf, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &Settings, &Disposition) == ERROR_SUCCESS ) {
		wsprintf(buf, L"%08x", ID);
		bool bRet = RegSetValueEx(Settings, buf, NULL, REG_DWORD, (LPBYTE)&val, sizeof(val)) == ERROR_SUCCESS;
		RegCloseKey(Settings);
		return bRet;
	}
	return false;
}


static void UpdateCascade(HWND hWnd) {
	if( ProgressDialogIsOpen() )
		return;

	wchar_t buf[32];

	CascadeOrder Order;
	CascadeEffect Effect;
	int Brightness;
	int Period = 1000, OnTime = 200, RiseTime = 100, FallTime = 100;
	bool bMerge, bMix;

	if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = HighToLow;
	else if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = PingPong;
	else if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_RANDOM), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = Random;
	else
		Order = LowToHigh;

	if( SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FADE), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Effect = Fade;
	else
		Effect = Flash;

	if( GetWindowText(GetSubWindow(hWnd, IDC_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		Brightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_INTERVAL), buf, sizeof(buf)/sizeof(wchar_t)) )
		Period = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_ONTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		OnTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_RISETIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		RiseTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FALLTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		FallTime = GetNum(buf);

	bMerge = SendMessage(GetSubWindow(hWnd, IDC_MERGE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bMix = SendMessage(GetSubWindow(hWnd, IDC_MIX), BM_GETCHECK, 0, 0) == BST_CHECKED;

	Cascade(hWnd, selection_start, selection_finish, Order, Effect, Brightness, Period, OnTime, RiseTime, FallTime, selected_lights, bMerge, bMix);
}

static void UpdateCascadeDialogState(HWND hWnd) {
	bool bFade = SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FADE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME),      bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME_SPIN), bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME),      bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME_SPIN), bFade);
}

static INT_PTR CALLBACK CascadeDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[5] = { IDC_BRIGHTNESS_SPIN, IDC_INTERVAL_SPIN, IDC_ONTIME_SPIN, IDC_RISETIME_SPIN, IDC_FALLTIME_SPIN };
			int ranges[5] = { 100, 10000, 10000, 10000, 10000 };
			int default_values[5] = { 100, 1000, 200, 100, 100 };
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FLASH), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			int num_selected_lights = 0;
			for( int i = 0; i < 32; ++i )
				if( selected_lights&(1<<i) )
					++num_selected_lights;
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_RANDOM), num_selected_lights > 1);
			LoadDialogStateFromRegistry(hWnd, L"Cascade", CascadeRegistryEntries, sizeof(CascadeRegistryEntries)/sizeof(*CascadeRegistryEntries));
			UpdateCascadeDialogState(hWnd);
			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateCascade(hWnd);
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDC_EFFECT_FLASH:
				case IDC_EFFECT_FADE:
					UpdateCascadeDialogState(hWnd);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCascade(hWnd);
					return TRUE;
				case IDC_PREVIEW:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCascade(hWnd);
					else
						CancelCascade(hWnd);
					return TRUE;
				case IDC_MERGE:
				case IDC_MIX:
					SendMessage(GetSubWindow(hWnd, wmId == IDC_MERGE ? IDC_MIX : IDC_MERGE), BM_SETCHECK, 0, 0);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCascade(hWnd);
					break;
				case IDOK:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) != BST_CHECKED )
						UpdateCascade(hWnd);
					FinaliseCascade(hWnd);
					SaveDialogStateToRegistry(hWnd, L"Cascade", CascadeRegistryEntries, sizeof(CascadeRegistryEntries)/sizeof(*CascadeRegistryEntries));
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					CancelCascade(hWnd);
					DestroyWindow(hWnd);
					return TRUE;
				default:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCascade(hWnd);
					break;
				}
			} else if( wmEvent == EN_CHANGE ) {
				if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
					UpdateCascade(hWnd);
			}
			break;
		}
	case WM_CLOSE:
		CancelCascade(hWnd);
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		CancelCascade(hWnd);
		return TRUE;
	case WM_POSTCHANGESELECTION:
		{
			int num_selected_lights = 0;
			for( int i = 0; i < 32; ++i )
				if( selected_lights&(1<<i) )
					++num_selected_lights;
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_RANDOM), num_selected_lights > 1);

			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateCascade(hWnd);
			return TRUE;
		}
	}
	return FALSE;
}

void OpenCascadeDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Cascade") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Cascade feature.", L"Cascade requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_CASCADE_DIALOG), hWnd, CascadeDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}
void DoCascade(HWND hWnd, HINSTANCE hInst) {
	if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Cascade feature.", L"Cascade requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		HWND hwndTemp = CreateDialog(hInst, MAKEINTRESOURCE(IDD_CASCADE_DIALOG), hWnd, CascadeDialogProc);
		SendMessage(hwndTemp, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
		DestroyWindow(hwndTemp);
	}
}

void UpdateCustomRamp(HWND hWnd) {
	if( ProgressDialogIsOpen() )
		return;

	wchar_t buf[32];

	CustomRampType Type;
	int InitialBrightness = 0, FinalBrightness = 100, PeakBrightness = 100, TroughBrightness = 0;
	int StartDelay = 0, EndDelay = 0, PeakDelay = 500, TroughDelay = 500;
	int Symmetry = 50;
	bool bMerge, bMix;

	if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_RAMPUP), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Type = RampUp;
	else if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_RAMPDOWN), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Type = RampDown;
	else if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_PEAK), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Type = Peak;
	else
		Type = Trough;

	if( GetWindowText(GetSubWindow(hWnd, IDC_INITIAL_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		InitialBrightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FINAL_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		FinalBrightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		PeakBrightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		TroughBrightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_START_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		StartDelay = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_END_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		EndDelay = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_PEAK_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		PeakDelay = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_TROUGH_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		TroughDelay = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_SYMMETRY), buf, sizeof(buf)/sizeof(wchar_t)) )
		Symmetry = GetNum(buf);

	bMerge = SendMessage(GetSubWindow(hWnd, IDC_MERGE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bMix = SendMessage(GetSubWindow(hWnd, IDC_MIX), BM_GETCHECK, 0, 0) == BST_CHECKED;

	CustomRamp(hWnd, selection_start, selection_finish, Type, InitialBrightness, FinalBrightness, PeakBrightness, TroughBrightness, StartDelay, EndDelay, PeakDelay, TroughDelay, Symmetry, bMerge, bMix);
}

static const int CustomRampDialogStates[4][4] = {
	{ 0, 1, 2, 3 },
	{ 1, 0, 2, 3 },
	{ 0, 3, 1, 2 },
	{ 1, 2, 0, 3 }
};

static int LastCustomRampDialogState;
static int GetCustomRampDialogState(HWND hWnd) {
	if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_RAMPDOWN), BM_GETCHECK, 0, 0) == BST_CHECKED )
		return 1;
	else if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_PEAK), BM_GETCHECK, 0, 0) == BST_CHECKED )
		return 2;
	else if( SendMessage(GetSubWindow(hWnd, IDC_TYPE_TROUGH), BM_GETCHECK, 0, 0) == BST_CHECKED )
		return 3;
	else
		return 0;
}

static void ArrangeCustomRampValues(HWND hWnd) {
	wchar_t buf[4][32];
	GetWindowText(GetSubWindow(hWnd, IDC_INITIAL_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][0]], sizeof(buf[0])/sizeof(wchar_t));
	GetWindowText(GetSubWindow(hWnd, IDC_FINAL_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][1]], sizeof(buf[1])/sizeof(wchar_t));
	GetWindowText(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][2]], sizeof(buf[1])/sizeof(wchar_t));
	GetWindowText(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][3]], sizeof(buf[1])/sizeof(wchar_t));
	LastCustomRampDialogState = GetCustomRampDialogState(hWnd);
	SetWindowText(GetSubWindow(hWnd, IDC_INITIAL_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][0]]);
	SetWindowText(GetSubWindow(hWnd, IDC_FINAL_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][1]]);
	SetWindowText(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][2]]);
	SetWindowText(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS), buf[CustomRampDialogStates[LastCustomRampDialogState][3]]);
}

static void UpdateCustomRampDialogState(HWND hWnd) {
	bool bPeak   = SendMessage(GetSubWindow(hWnd, IDC_TYPE_PEAK),   BM_GETCHECK, 0, 0) == BST_CHECKED;
	bool bTrough = SendMessage(GetSubWindow(hWnd, IDC_TYPE_TROUGH), BM_GETCHECK, 0, 0) == BST_CHECKED;
	EnableWindow(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS),        bPeak);
	EnableWindow(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS_SPIN),   bPeak);
	EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS),      bTrough);
	EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS_SPIN), bTrough);
	EnableWindow(GetSubWindow(hWnd, IDC_PEAK_DELAY),             bPeak);
	EnableWindow(GetSubWindow(hWnd, IDC_PEAK_DELAY_SPIN),        bPeak);
	EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_DELAY),           bTrough);
	EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_DELAY_SPIN),      bTrough);
	EnableWindow(GetSubWindow(hWnd, IDC_SYMMETRY),               bPeak || bTrough);
	EnableWindow(GetSubWindow(hWnd, IDC_SYMMETRY_SPIN),          bPeak || bTrough);
	LastCustomRampDialogState = GetCustomRampDialogState(hWnd);
}

static INT_PTR CALLBACK CustomRampDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[9] = { IDC_INITIAL_BRIGHTNESS_SPIN, IDC_FINAL_BRIGHTNESS_SPIN, IDC_PEAK_BRIGHTNESS_SPIN, IDC_TROUGH_BRIGHTNESS_SPIN,
				                       IDC_START_DELAY_SPIN, IDC_END_DELAY_SPIN, IDC_PEAK_DELAY_SPIN, IDC_TROUGH_DELAY_SPIN, IDC_SYMMETRY_SPIN };
			int ranges[9] = { 100, 100, 100, 100, 10000, 10000, 10000, 10000, 100 };
			int default_values[9] = { 0, 100, 100, 0, 0, 0, 500, 500, 50 };
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_TYPE_RAMPUP), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			LoadDialogStateFromRegistry(hWnd, L"Custom Ramp", CustomRampRegistryEntries, sizeof(CustomRampRegistryEntries)/sizeof(*CustomRampRegistryEntries));
			LastCustomRampDialogState = GetCustomRampDialogState(hWnd);
			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateCustomRamp(hWnd);
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDC_TYPE_RAMPUP:
				case IDC_TYPE_RAMPDOWN:
				case IDC_TYPE_PEAK:
				case IDC_TYPE_TROUGH:
					ArrangeCustomRampValues(hWnd);
					EnableWindow(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS),        wmId == IDC_TYPE_PEAK);
					EnableWindow(GetSubWindow(hWnd, IDC_PEAK_BRIGHTNESS_SPIN),   wmId == IDC_TYPE_PEAK);
					EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS),      wmId == IDC_TYPE_TROUGH);
					EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_BRIGHTNESS_SPIN), wmId == IDC_TYPE_TROUGH);
					EnableWindow(GetSubWindow(hWnd, IDC_PEAK_DELAY),             wmId == IDC_TYPE_PEAK);
					EnableWindow(GetSubWindow(hWnd, IDC_PEAK_DELAY_SPIN),        wmId == IDC_TYPE_PEAK);
					EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_DELAY),           wmId == IDC_TYPE_TROUGH);
					EnableWindow(GetSubWindow(hWnd, IDC_TROUGH_DELAY_SPIN),      wmId == IDC_TYPE_TROUGH);
					EnableWindow(GetSubWindow(hWnd, IDC_SYMMETRY),               wmId == IDC_TYPE_PEAK || wmId == IDC_TYPE_TROUGH);
					EnableWindow(GetSubWindow(hWnd, IDC_SYMMETRY_SPIN),          wmId == IDC_TYPE_PEAK || wmId == IDC_TYPE_TROUGH);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCustomRamp(hWnd);
					return TRUE;
				case IDC_PREVIEW:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCustomRamp(hWnd);
					else
						CancelCustomRamp(hWnd);
					return TRUE;
				case IDC_MERGE:
				case IDC_MIX:
					SendMessage(GetSubWindow(hWnd, wmId == IDC_MERGE ? IDC_MIX : IDC_MERGE), BM_SETCHECK, 0, 0);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCustomRamp(hWnd);
					break;
				case IDOK:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCustomRamp(hWnd);
					FinaliseCustomRamp(hWnd);
					SaveDialogStateToRegistry(hWnd, L"Custom Ramp", CustomRampRegistryEntries, sizeof(CustomRampRegistryEntries)/sizeof(*CustomRampRegistryEntries));
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					CancelCustomRamp(hWnd);
					DestroyWindow(hWnd);
					return TRUE;
				default:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateCustomRamp(hWnd);
					break;
				}
			} else if( wmEvent == EN_CHANGE ) {
				if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
					UpdateCustomRamp(hWnd);
			}
			break;
		}
	case WM_CLOSE:
		CancelCustomRamp(hWnd);
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		CancelCustomRamp(hWnd);
		return TRUE;
	case WM_POSTCHANGESELECTION:
		if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
			UpdateCustomRamp(hWnd);
		return TRUE;
	}
	return FALSE;
}

void OpenCustomRampDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Custom Ramp") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Custom Ramp feature.", L"Custom Ramp requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_CUSTOMRAMP_DIALOG), hWnd, CustomRampDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}
void DoCustomRamp(HWND hWnd, HINSTANCE hInst) {
	if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Custom Ramp feature.", L"Custom Ramp requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		HWND hwndTemp = CreateDialog(hInst, MAKEINTRESOURCE(IDD_CUSTOMRAMP_DIALOG), hWnd, CustomRampDialogProc);
		SendMessage(hwndTemp, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
		DestroyWindow(hwndTemp);
	}
}

static void UpdateSettingsSpin(HWND hWnd) {
	wchar_t buf[32];
	if( GetWindowText(GetSubWindow(hWnd, IDC_MONITOR_GAMMA), buf, sizeof(buf)/sizeof(wchar_t)) ) {
		wchar_t* end;
		double lfVal = wcstod(buf, &end);
		if( !*end && end != buf && lfVal >= 0 && lfVal <= 9.9 )
			SendMessage(GetSubWindow(hWnd, IDC_MONITOR_GAMMA_SPIN), UDM_SETPOS, 0, (int)(lfVal*10));
	}
	if( GetWindowText(GetSubWindow(hWnd, IDC_AUDIO_DELAY_COMPENSATION), buf, sizeof(buf)/sizeof(wchar_t)) ) {
		int delay = GetNum(buf);
		if( delay >= -1000 && delay <= 1000 )
			SendMessage(GetSubWindow(hWnd, IDC_AUDIO_DELAY_COMPENSATION_SPIN), UDM_SETPOS, 0, delay+1000);
	}
	if( GetWindowText(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE), buf, sizeof(buf)/sizeof(wchar_t)) ) {
		if( buf[1] >= 'a' && buf[1] <= 'z' )
			SendMessage(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE_SPIN), UDM_SETPOS, 0, buf[1]-'a');
		else if( buf[1] >= 'A' && buf[1] <= 'Z' )
			SendMessage(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE_SPIN), UDM_SETPOS, 0, buf[1]-'A');
	}
}

static void UpdateCurrentSettings(HWND hWnd) {
	double lfOldGamma = lfGamma;

	UpdateSettingsSpin(hWnd);

	lfGamma = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_MONITOR_GAMMA_SPIN), UDM_GETPOS, 0, 0))/10.0;
	lfInvGamma = 1.0 / lfGamma;

	g_bBackupSequenceFiles = SendMessage(GetSubWindow(hWnd, IDC_MAKE_BACKUP_FILES), BM_GETCHECK, 0, 0) == BST_CHECKED;
	g_bEjectAfterPublish = SendMessage(GetSubWindow(hWnd, IDC_EJECTAFTERPUBLISH), BM_GETCHECK, 0, 0) == BST_CHECKED;

	int playback_delay = (int)LOWORD(SendMessage(GetSubWindow(hWnd, IDC_AUDIO_DELAY_COMPENSATION_SPIN), UDM_GETPOS, 0, 0)) - 1000;
	set_playback_cursor_delay(playback_delay);

	max_undo_memory = (int)LOWORD(SendMessage(GetSubWindow(hWnd, IDC_MAXIMUM_UNDO_MEMORY_SPIN), UDM_GETPOS, 0, 0)) * 1048576;

	HWND hParent = GetParent(hWnd);
	if( lfGamma != lfOldGamma && FileIsOpen(NULL) ) {
		RECT rcInvalidate;
		GetClientRect(hParent, &rcInvalidate);
		rcInvalidate.top += rcWAV.top;
		InvalidateRect(hParent, &rcInvalidate, true);
	}

	SetFollowPlaybackPos( hParent, SendMessage(GetSubWindow(hWnd, IDC_SCROLL_TO_FOLLOW_PLAYBACK_CURSOR), BM_GETCHECK, 0, 0) == BST_CHECKED );
	SetLoopPlayback( hParent, SendMessage(GetSubWindow(hWnd, IDC_LOOP_PLAYBACK), BM_GETCHECK, 0, 0) == BST_CHECKED );
	PublishPath[0] = 'A' + (int)LOWORD(SendMessage(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE_SPIN), UDM_GETPOS, 0, 0));
}

HWND hLights;
COLORREF LightColours[32];
HBITMAP LightBitmaps[32];
extern unsigned long light_colours[32];
extern wchar_t light_names[32][256];
COLORREF CustomColours[16];
static INT_PTR CALLBACK LightsDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			for( int i = 0; i < 32; ++i ) {
				SetWindowText(GetSubWindow(hWnd, IDC_LIGHT1+i), light_names[i]);

				LightColours[i] = light_colours[i];
				LightBitmaps[i] = CreateSingleColourBitmap( GetWindowDC(hWnd), 17, 16, LightColours[i] );
				SendMessage(GetSubWindow(hWnd, IDC_LIGHTCOL1+i), BM_SETIMAGE, (WPARAM)IMAGE_BITMAP, (LPARAM)LightBitmaps[i]);
			}
			for( int i = 0; i < 16; ++i ) {
				wchar_t buf[32];
				wsprintf(buf, L"CustomColour%d", i);
				CustomColours[i] = LoadDwordFromRegistry(buf, 0);
			}
			int OldEnabledLights = EnabledLights;
			EnabledLights = LoadDwordFromRegistry(L"EnabledLights", 0x000000FF);
			NumEnabledLights = 0;
			for( int i = 0; i < 32; ++i ) {
				SendMessage(GetSubWindow(hWnd, IDC_LIGHT1ON+i), BM_SETCHECK, EnabledLights&(1<<i) ? BST_CHECKED : 0, 0);
				if( EnabledLights&(1<<i) )
					++NumEnabledLights;
			}
			if( OldEnabledLights != EnabledLights ) {
				InvalidateLightSpan(GetParent(GetParent(hWnd)), 0xFFFFFFFF, 0, rcWAV.right);
				InvalidateLightBar(GetParent(GetParent(hWnd)));
			}
			return TRUE;
		}
	case WM_COMMAND:
		{
			int OldEnabledLights;
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				if( wmId >= IDC_LIGHTCOL1 && wmId <= IDC_LIGHTCOL32 ) {
					CHOOSECOLOR c;
					memset(&c, 0, sizeof(c));
					c.lStructSize = sizeof(c);
					c.hwndOwner = hWnd;
					c.rgbResult = LightColours[wmId - IDC_LIGHTCOL1];
					c.lpCustColors = CustomColours;
					c.Flags = CC_ANYCOLOR|CC_FULLOPEN|CC_RGBINIT|CC_SOLIDCOLOR;
					if( ChooseColor(&c) ) {
						int i = wmId - IDC_LIGHTCOL1;
						LightColours[i] = c.rgbResult;
						DeleteObject(LightBitmaps[i]);
						LightBitmaps[i] = CreateSingleColourBitmap( GetWindowDC(hWnd), 17, 16, LightColours[i] );
						SendMessage(GetSubWindow(hWnd, IDC_LIGHTCOL1+i), BM_SETIMAGE, (WPARAM)IMAGE_BITMAP, (LPARAM)LightBitmaps[i]);
					}
				} else switch(wmId) {
				case IDOK:
					OldEnabledLights = EnabledLights;
					EnabledLights = 0;
					NumEnabledLights = 0;
					for( int i = 0; i < 32; ++i ) {
						wchar_t buf[32];
						wsprintf(buf, L"LightColour%d", i);
						SaveDwordToRegistry(buf, LightColours[i]);
						light_colours[i] = LightColours[i];
						GetWindowText(GetSubWindow(hWnd, IDC_LIGHT1+i), light_names[i], sizeof(light_names[i])/sizeof(*light_names[i]));
						wsprintf(buf, L"LightName%d", i);
						SaveStringToRegistry(buf, light_names[i]);
						if( SendMessage(GetSubWindow(hWnd, IDC_LIGHT1ON+i), BM_GETCHECK, 0, 0) == BST_CHECKED ) {
							EnabledLights |= (1<<i);
							++NumEnabledLights;
						}
					}
					if( OldEnabledLights != EnabledLights ) {
						SaveDwordToRegistry(L"EnabledLights", EnabledLights);
						InvalidateLightSpan(GetParent(GetParent(hWnd)), 0xFFFFFFFF, 0, rcWAV.right);
						InvalidateLightBar(GetParent(GetParent(hWnd)));
					}
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					DestroyWindow(hWnd);
					return TRUE;
				}
			}
			break;
		}
	case WM_DESTROY:
		for( int i = 0; i < 16; ++i ) {
			wchar_t buf[32];
			wsprintf(buf, L"CustomColour%d", i);
			SaveDwordToRegistry(buf, CustomColours[i]);
		}
		hLights = (HWND)NULL;
		break;
	}
	return FALSE;
}

HINSTANCE g_hInst;
void OpenLightsDialog(HWND hWnd, HINSTANCE hInst) {
	if( hLights ) {
		SetForegroundWindow(hLights);
	} else {
		hLights = CreateDialog(hInst, MAKEINTRESOURCE(IDD_LIGHTS), hWnd, LightsDialogProc);
		ShowWindow(hLights, TRUE);
	}
}

const wchar_t* FileExtensions[] = { L".lsq", L".lsn" };

double lfOrigGamma;
static INT_PTR CALLBACK SettingsDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[4] = { IDC_MONITOR_GAMMA_SPIN, IDC_AUDIO_DELAY_COMPENSATION_SPIN, IDC_MAXIMUM_UNDO_MEMORY_SPIN, IDC_SD_CARD_DRIVE_SPIN };
			int ranges[4] = { 100, 2000, 100|(1<<16), 25 };
			int default_values[4] = { 16, 900, 10, 0 };
			lfOrigGamma = lfGamma;
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_MAKE_BACKUP_FILES), BM_SETCHECK, BST_CHECKED, 0);
			SendMessage(GetSubWindow(hWnd, IDC_EJECTAFTERPUBLISH), BM_SETCHECK, BST_CHECKED, 0);

			wchar_t path[4] = L" A:";
			for( path[1] = L'A'; path[1] <= L'Z'; ++path[1] ) {
				if( GetDriveType(path+1) == DRIVE_REMOVABLE )
					break;
			}
			if( path[1] > L'Z' ) {
				for( path[1] = L'C'; path[1] <= L'Z'; ++path[1] ) {
					if( GetDriveType(path+1) == DRIVE_NO_ROOT_DIR )
						break;
				}
				if( path[1] > L'Z' ) {
					path[1] = L'Z';
				}
			}
			SetWindowText(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE), path);

			LoadDialogStateFromRegistry(hWnd, L"Settings", SettingsRegistryEntries, sizeof(SettingsRegistryEntries)/sizeof(*SettingsRegistryEntries));
			SendMessage(GetSubWindow(hWnd, IDC_ASSOCIATE_FILE_EXTENSION), BM_SETCHECK, AreExtensionsRegistered(FileExtensions, sizeof(FileExtensions)/sizeof(*FileExtensions)) ? BST_CHECKED : 0, 0);
			UpdateSettingsSpin(hWnd);
			SendMessage(hWnd, WM_VSCROLL, 0, (LPARAM)GetSubWindow(hWnd, IDC_MONITOR_GAMMA_SPIN));
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDC_LIGHTS:
					OpenLightsDialog(hWnd, g_hInst);
					break;
				case IDC_ASSOCIATE_FILE_EXTENSION:
					if( SendMessage(GetSubWindow(hWnd, IDC_ASSOCIATE_FILE_EXTENSION), BM_GETCHECK, 0, 0) == BST_CHECKED ) {
						if( !RegisterExtensions(FileExtensions, sizeof(FileExtensions)/sizeof(*FileExtensions)) ) {
							MessageBox(hWnd, L"There was an error registering the file extensions. It might help to run this program as Administrator.", L"File extension registration failed", MB_OK|MB_ICONEXCLAMATION);
							SendMessage(GetSubWindow(hWnd, IDC_ASSOCIATE_FILE_EXTENSION), BM_SETCHECK, 0, 0);
						}
					} else {
						if( !UnregisterExtensions(FileExtensions, sizeof(FileExtensions)/sizeof(*FileExtensions)) ) {
							MessageBox(hWnd, L"There was an error deregistering the file extensions. It might help to run this program as Administrator.", L"File extension deregistration failed", MB_OK|MB_ICONEXCLAMATION);
							SendMessage(GetSubWindow(hWnd, IDC_ASSOCIATE_FILE_EXTENSION), BM_SETCHECK, BST_CHECKED, 0);
						}
					}
					break;
				case IDOK:
					if( hLights )
						SendMessage(hLights, WM_COMMAND, IDOK, 0);
					SaveDialogStateToRegistry(hWnd, L"Settings", SettingsRegistryEntries, sizeof(SettingsRegistryEntries)/sizeof(*SettingsRegistryEntries));
					UpdateCurrentSettings(hWnd);
					lfOrigGamma = lfGamma;
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					DestroyWindow(hWnd);
					return TRUE;
				}
			}
			break;
		}
	case WM_NOTIFY:
		if( (LOWORD(wParam) == IDC_MONITOR_GAMMA || LOWORD(wParam) == IDC_AUDIO_DELAY_COMPENSATION || LOWORD(wParam) == IDC_SD_CARD_DRIVE) && HIWORD(wParam) == EN_CHANGE ) {
			wchar_t buf[64];
			wchar_t* end;
			int pos;
			bool bValid = false;
			DWORD spin;

			GetWindowText(GetSubWindow(hWnd, LOWORD(wParam)), buf, sizeof(buf)/sizeof(wchar_t));
			switch(LOWORD(wParam)) {
			case IDC_MONITOR_GAMMA:
				{
					double lfVal = wcstod(buf, &end);
					spin = IDC_MONITOR_GAMMA_SPIN;
					if( !*end && end != buf && lfVal >= 0 && lfVal <= 9.9 ) {
						pos = (int)(lfVal * 10 + 0.5);
						bValid = true;
					}
					break;
				}
			case IDC_AUDIO_DELAY_COMPENSATION:
				{
					spin = IDC_AUDIO_DELAY_COMPENSATION_SPIN;
					pos = wcstol(buf, &end, 10);
					if( !*end && end != buf && pos >= -1000 && pos <= 1000 ) {
						pos += 1000;
						bValid = true;
					}
					break;
				}
			case IDC_SD_CARD_DRIVE:
				{
					spin = IDC_SD_CARD_DRIVE_SPIN;
					if( buf[0] >= L'a' && buf[0] <= L'z' && buf[1] == L'\0' ) {
						pos = buf[0] - L'a';
						bValid = true;
					} else if( buf[0] >= L'A' && buf[0] <= L'Z' && buf[1] == L'\0' ) {
						pos = buf[0] - L'A';
						bValid = true;
					}
					break;
				}
			}
			if( bValid ) {
				SendMessage(GetSubWindow(hWnd, spin), UDM_SETPOS, 0, (LPARAM)pos);
			} else {
				SendMessage(hWnd, WM_VSCROLL, 0, (LPARAM)GetSubWindow(hWnd, IDC_MONITOR_GAMMA_SPIN));
			}
		}
		break;
	case WM_VSCROLL:
		if( lParam ) {
			DWORD ID = GetWindowLong((HWND)lParam, GWL_ID);
			if( ID == IDC_MONITOR_GAMMA_SPIN || ID == IDC_AUDIO_DELAY_COMPENSATION_SPIN || ID == IDC_SD_CARD_DRIVE_SPIN ) {
				int pos;
				wchar_t buf[8];

				pos = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_MONITOR_GAMMA_SPIN), UDM_GETPOS, 0, 0));
				wsprintf(buf, L"%d.%d", pos / 10, pos % 10);
				SetWindowText(GetSubWindow(hWnd, IDC_MONITOR_GAMMA), buf);

				pos = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_AUDIO_DELAY_COMPENSATION_SPIN), UDM_GETPOS, 0, 0));
				wsprintf(buf, L"%c%4d", pos < 1000 ? '-' : ' ', pos < 1000 ? 1000 - pos : pos - 1000);
				SetWindowText(GetSubWindow(hWnd, IDC_AUDIO_DELAY_COMPENSATION), buf);

				pos = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE_SPIN), UDM_GETPOS, 0, 0));
				wsprintf(buf, L" %c:", 'A' + pos);
				SetWindowText(GetSubWindow(hWnd, IDC_SD_CARD_DRIVE), buf);

				UpdateCurrentSettings(hWnd);
			}
		}
		break;
	case WM_CLOSE:
		if( lfGamma != lfOrigGamma ) {
			lfGamma = lfOrigGamma;
			lfInvGamma = 1.0 / lfGamma;

			if( FileIsOpen(NULL) ) {
				RECT rcInvalidate;
				HWND hParent = GetParent(hWnd);
				GetClientRect(hParent, &rcInvalidate);
				rcInvalidate.top += rcWAV.top;
				InvalidateRect(hParent, &rcInvalidate, true);
			}
		}
		if( hLights )
			CloseWindow(hLights);
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_POSTCHANGESELECTION:
		return TRUE;
	}
	return FALSE;
}

void OpenSettingsDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Settings") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else {
		g_hInst = hInst;
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_SETTINGS), hWnd, SettingsDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}

void LoadSettings(HWND hWnd, HINSTANCE hInst) {
	HWND hwndTemp = CreateDialog(hInst, MAKEINTRESOURCE(IDD_SETTINGS), hWnd, SettingsDialogProc);
	SendMessage(hwndTemp, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
	DestroyWindow(hwndTemp);

	int OldEnabledLights = EnabledLights;
	EnabledLights = LoadDwordFromRegistry(L"EnabledLights", 0x000000FF);
	NumEnabledLights = 0;
	for( int i = 0; i < 32; ++i ) {
		wchar_t buf[32];
		wsprintf(buf, L"LightColour%d", i);
		light_colours[i] = LoadDwordFromRegistry(buf, light_colours[i]);
		wsprintf(buf, L"LightName%d", i);
		LoadStringFromRegistry(buf, light_names[i], sizeof(light_names[i]));
		if( EnabledLights&(1<<i) )
			++NumEnabledLights;
	}
	if( OldEnabledLights != EnabledLights ) {
		InvalidateLightSpan(GetParent(GetParent(hWnd)), 0xFFFFFFFF, 0, rcWAV.right);
		InvalidateLightBar(GetParent(GetParent(hWnd)));
	}
}

static INT_PTR CALLBACK PublishWaitDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			wchar_t buf[256];
			wchar_t* pos;
			GetWindowText(GetSubWindow(hWnd, IDC_PUBLISH_TEXT), buf, sizeof(buf)/sizeof(wchar_t));
			pos = wcsstr(buf, L"x:");
			if( pos ) {
				pos[0] = PublishPath[0];
				SetWindowText(GetSubWindow(hWnd, IDC_PUBLISH_TEXT), buf);
			}
			SetTimer(hWnd, 1, 1000, 0);
			return TRUE;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
				case IDC_CHANGE_DRIVE:
					SendMessage(GetParent(hWnd), WM_COMMAND, (WPARAM)IDM_SETTINGS, (LPARAM)0);
					EndDialog(hWnd, 0);
					return TRUE;
				}
			}
			break;
		}
	case WM_TIMER:
		if( GetDriveType(PublishPath) != DRIVE_NO_ROOT_DIR )
			EndDialog(hWnd, 0);
		return TRUE;
	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return TRUE;
	}
	return FALSE;
}

void OpenPublishWaitDialog(HWND hWnd, HINSTANCE hInst) {
	DialogBox(hInst, MAKEINTRESOURCE(IDD_PUBLISH_WAIT), hWnd, PublishWaitDialogProc);
}


int LastSensitivity;
static void UpdateBeatDetection(HWND hWnd, bool bFinalise = false) {
	if( ProgressDialogIsOpen() )
		return;

	wchar_t buf[32];

	int Sensitivity;
	int MinInterval = 100, Delay = 0;

	CascadeEffect Effect;
	int Brightness;

	int OnTime = 200, RiseTime = 100, FallTime = 100;

	CascadeOrder Order;
	bool bMerge, bMix;

	Sensitivity = SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_GETPOS, 0, 0);
	if( GetWindowText(GetSubWindow(hWnd, IDC_MINIMUM_INTERVAL), buf, sizeof(buf)/sizeof(wchar_t)) )
		MinInterval = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		Delay = GetNum(buf);

	if( SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FADE), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Effect = Fade;
	else
		Effect = Flash;
	if( GetWindowText(GetSubWindow(hWnd, IDC_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		Brightness = GetNum(buf);

	if( GetWindowText(GetSubWindow(hWnd, IDC_ONTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		OnTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_RISETIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		RiseTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FALLTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		FallTime = GetNum(buf);

	if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = HighToLow;
	else if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = PingPong;
	else if( SendMessage(GetSubWindow(hWnd, IDC_ORDER_RANDOM), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = Random;
	else
		Order = LowToHigh;

	bMerge = SendMessage(GetSubWindow(hWnd, IDC_MERGE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bMix = SendMessage(GetSubWindow(hWnd, IDC_MIX), BM_GETCHECK, 0, 0) == BST_CHECKED;

	LastSensitivity = Sensitivity;
	if( bFinalise )
		FinaliseBeatDetection(hWnd, selection_start, selection_finish, Sensitivity, MinInterval, Delay, Effect, Brightness, OnTime, RiseTime, FallTime, Order, selected_lights, bMerge, bMix);
	else
		BeatDetection(hWnd, selection_start, selection_finish, Sensitivity, MinInterval, Delay, Effect, Brightness, OnTime, RiseTime, FallTime, Order, selected_lights, bMerge, bMix, true);
}

static void UpdateBeatDetectionDialogState(HWND hWnd) {
	wchar_t buf[32];

	bool bFade = SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FADE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME),      bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME_SPIN), bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME),      bFade);
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME_SPIN), bFade);

	if( GetWindowText(GetSubWindow(hWnd, IDC_MINIMUM_INTERVAL), buf, sizeof(buf)/sizeof(wchar_t)) ) {
		int Period = GetNum(buf);
		if( Period ) {
			wsprintf(buf, L"ms (max. %d BPM)", 60000 / Period);
			SetWindowText(GetSubWindow(hWnd, IDC_MAX_BPM), buf);
		}
	}
}

extern bool bDidBeatDetection, bDidBeatDetectionPreview;
extern int BeatDetectionPreviewStart, BeatDetectionPreviewFinish;
static INT_PTR CALLBACK BeatDetectionDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[6] = { IDC_MINIMUM_INTERVAL_SPIN, IDC_DELAY_SPIN, IDC_BRIGHTNESS_SPIN, IDC_ONTIME_SPIN, IDC_RISETIME_SPIN, IDC_FALLTIME_SPIN };
			int ranges[6] = { 1000, 2000, 100, 10000, 10000, 10000 };
			int default_values[6] = { 200, 1000, 100, 150, 100, 100 };
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_EFFECT_FLASH), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETRANGE, TRUE, MAKELPARAM(0, 40));
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETTICFREQ, 10, 0);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETPAGESIZE, 0, 10);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETPOS, true, 27);
			int num_selected_lights = 0;
			for( int i = 0; i < 32; ++i )
				if( selected_lights&(1<<i) )
					++num_selected_lights;
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_RANDOM), num_selected_lights > 1);
			LoadDialogStateFromRegistry(hWnd, L"BeatDetection", BeatDetectionRegistryEntries, sizeof(BeatDetectionRegistryEntries)/sizeof(*BeatDetectionRegistryEntries));
			wchar_t buf[32];
			if( GetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) ) {
				int delay = GetNum(buf);
				if( delay >= -1000 && delay <= 1000 )
					SendMessage(GetSubWindow(hWnd, IDC_DELAY_SPIN), UDM_SETPOS, 0, delay + 1000);
			}
			UpdateBeatDetectionDialogState(hWnd);
			SendMessage(hWnd, WM_VSCROLL, 0, (LPARAM)GetSubWindow(hWnd, IDC_DELAY_SPIN));
//			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
//				UpdateBeatDetection(hWnd);
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDC_EFFECT_FLASH:
				case IDC_EFFECT_FADE:
					UpdateBeatDetectionDialogState(hWnd);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateBeatDetection(hWnd);
					return TRUE;
				case IDC_PREVIEW:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateBeatDetection(hWnd);
					else
						CancelBeatDetection(hWnd);
					return TRUE;
				case IDC_MERGE:
				case IDC_MIX:
					SendMessage(GetSubWindow(hWnd, wmId == IDC_MERGE ? IDC_MIX : IDC_MERGE), BM_SETCHECK, 0, 0);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateBeatDetection(hWnd);
					break;
				case IDOK:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) != BST_CHECKED )
						UpdateBeatDetection(hWnd);
					UpdateBeatDetection(hWnd, true);
					SaveDialogStateToRegistry(hWnd, L"BeatDetection", BeatDetectionRegistryEntries, sizeof(BeatDetectionRegistryEntries)/sizeof(*BeatDetectionRegistryEntries));
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					CancelBeatDetection(hWnd);
					DestroyWindow(hWnd);
					return TRUE;
				default:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateBeatDetection(hWnd);
					break;
				}
			} else if( wmEvent == EN_CHANGE ) {
				if( wmId == IDC_MINIMUM_INTERVAL )
					UpdateBeatDetectionDialogState(hWnd);
				if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
					UpdateBeatDetection(hWnd);
			}
			break;
		}
	case WM_VSCROLL:
		if( lParam && GetWindowLong((HWND)lParam, GWL_ID) == IDC_DELAY_SPIN ) {
			int pos;
			wchar_t buf[8];

			pos = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_DELAY_SPIN), UDM_GETPOS, 0, 0));
			wsprintf(buf, L"%c%4d", pos < 1000 ? '-' : ' ', pos < 1000 ? 1000 - pos : pos - 1000);
			SetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf);

//			if( wParam && SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
//				UpdateBeatDetection(hWnd);
		}
		break;
	case WM_HSCROLL:
		if( lParam && GetWindowLong((HWND)lParam, GWL_ID) == IDC_SENSITIVITY ) {
			int Sensitivity = SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_GETPOS, 0, 0);
			if( Sensitivity != LastSensitivity && SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateBeatDetection(hWnd);
		}
		break;
	case WM_CLOSE:
		CancelBeatDetection(hWnd);
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		CancelBeatDetection(hWnd);
		return TRUE;
	case WM_POSTCHANGESELECTION:
		{
			int num_selected_lights = 0;
			for( int i = 0; i < 32; ++i )
				if( selected_lights&(1<<i) )
					++num_selected_lights;
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_LOWTOHIGH), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_HIGHTOLOW), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_PINGPONG), num_selected_lights > 1);
			EnableWindow(GetSubWindow(hWnd, IDC_ORDER_RANDOM), num_selected_lights > 1);

			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateBeatDetection(hWnd);
			return TRUE;
		}
	case WM_MAINSCROLL:
		if( bDidBeatDetection && bDidBeatDetectionPreview ) {
			SCROLLINFO si;
			si.cbSize = sizeof (si);
			si.fMask  = SIF_ALL;
			GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
			int view_left = si.nPos << zoom;
			int view_right = (si.nPos + (rcWAV.right-rcWAV.left)) << zoom;
			if( view_left < BeatDetectionPreviewStart || view_right > BeatDetectionPreviewFinish )
				UpdateBeatDetection(hWnd);
		}
		break;
	}
	return FALSE;
}

void OpenBeatDetectionDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Beat Detection") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Beat Detection feature.", L"Beat Detection requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_BEATDETECTION), hWnd, BeatDetectionDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}
void DoBeatDetection(HWND hWnd, HINSTANCE hInst) {
	if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the BeatDetection feature.", L"Beat Detection requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		HWND hwndTemp = CreateDialog(hInst, MAKEINTRESOURCE(IDD_BEATDETECTION), hWnd, BeatDetectionDialogProc);
		SendMessage(hwndTemp, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
		DestroyWindow(hwndTemp);
	}
}


static void UpdateSpectrumAnalysisDialogState(HWND hWnd) {
	bool bTrack = SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_TRACK), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bool bFlash = SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_FLASH), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bool bFade  = SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_FADE),  BM_GETCHECK, 0, 0) == BST_CHECKED;
	EnableWindow(GetSubWindow(hWnd, IDC_SCALE_BRIGHTNESS), !bTrack);
	EnableWindow(GetSubWindow(hWnd, IDC_SCALE_DURATION),   !bTrack);
	EnableWindow(GetSubWindow(hWnd, IDC_ONTIME),            bFlash || bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_DELAY),             bFlash || bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_DELAY_SPIN),        bFlash || bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_MINIMUM_INTERVAL),  bFlash || bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME),          bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_RISETIME_SPIN),     bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME),          bFade );
	EnableWindow(GetSubWindow(hWnd, IDC_FALLTIME_SPIN),     bFade );
}

static void UpdateSpectrumAnalysisStereoDialogState(HWND hWnd) {
	int num_selected_lights = 0;
	for( int i = 0; i < 32; ++i )
		if( selected_lights&(1<<i) )
			++num_selected_lights;
	EnableWindow(GetSubWindow(hWnd, IDC_STEREO_MONO), wav_num_channels > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_STEREO_LTOR), wav_num_channels > 1 && num_selected_lights > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_STEREO_RTOL), wav_num_channels > 1 && num_selected_lights > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_STEREO_LONLY), wav_num_channels > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_STEREO_RONLY), wav_num_channels > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_CHANS_LOWTOHIGH), num_selected_lights > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_CHANS_HIGHTOLOW), num_selected_lights > 1);
	EnableWindow(GetSubWindow(hWnd, IDC_CHANS_RANDOM), num_selected_lights > 1);
	if( num_selected_lights == 1 ) {
		if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_LTOR),  BM_GETCHECK, 0,           0) == BST_CHECKED ) {
			SendMessage(GetSubWindow(hWnd, IDC_STEREO_LTOR),  BM_SETCHECK, 0,           0);
			SendMessage(GetSubWindow(hWnd, IDC_STEREO_LONLY), BM_SETCHECK, BST_CHECKED, 0);
		} else if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_RTOL), BM_GETCHECK, 0,     0) == BST_CHECKED ) {
			SendMessage(GetSubWindow(hWnd, IDC_STEREO_RTOL),  BM_SETCHECK, 0,           0);
			SendMessage(GetSubWindow(hWnd, IDC_STEREO_RONLY), BM_SETCHECK, BST_CHECKED, 0);
		}
	}
}

static void UpdateSpectrumAnalysis(HWND hWnd, bool bFinalise = false) {
	if( ProgressDialogIsOpen() )
		return;

	wchar_t buf[32];
	int StartFrequency = 100, FinishFrequency = 6000, NumBands;
	SpectrumStereoType Stereo;
	CascadeOrder Order;
	CascadeEffect Effect;
	SpectrumScaleType ScaleType;
	int Sensitivity = 20;
	int Brightness = 100, OnTime = 100, Delay = 0, MinInterval = 100, RiseTime = 50, FallTime = 50;
	bool bScaleBrightness = true, bScaleDuration = false;
	bool bMerge = false, bMix = false;

	if( GetWindowText(GetSubWindow(hWnd, IDC_FREQUENCY_START), buf, sizeof(buf)/sizeof(wchar_t)) )
		StartFrequency = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FREQUENCY_FINISH), buf, sizeof(buf)/sizeof(wchar_t)) )
		FinishFrequency = GetNum(buf);

	if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_MONO), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Stereo = Mono;
	else if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_LTOR), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Stereo = LeftToRight;
	else if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_RTOL), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Stereo = RightToLeft;
	else if( SendMessage(GetSubWindow(hWnd, IDC_STEREO_LONLY), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Stereo = LeftOnly;
	else
		Stereo = RightOnly;

	if( SendMessage(GetSubWindow(hWnd, IDC_CHANS_LOWTOHIGH), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = LowToHigh;
	else if( SendMessage(GetSubWindow(hWnd, IDC_CHANS_HIGHTOLOW), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Order = HighToLow;
	else
		Order = Random;

	if( SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_TRACK), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Effect = Track;
	else if( SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_FLASH), BM_GETCHECK, 0, 0) == BST_CHECKED )
		Effect = Flash;
	else
		Effect = Fade;

	Sensitivity = SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_GETPOS, 0, 0);

	if( GetWindowText(GetSubWindow(hWnd, IDC_BRIGHTNESS), buf, sizeof(buf)/sizeof(wchar_t)) )
		Brightness = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_ONTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		OnTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) )
		Delay = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_MINIMUM_INTERVAL), buf, sizeof(buf)/sizeof(wchar_t)) )
		MinInterval = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_RISETIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		RiseTime = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FALLTIME), buf, sizeof(buf)/sizeof(wchar_t)) )
		FallTime = GetNum(buf);

	bScaleBrightness = SendMessage(GetSubWindow(hWnd, IDC_SCALE_BRIGHTNESS), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bScaleDuration = SendMessage(GetSubWindow(hWnd, IDC_SCALE_DURATION), BM_GETCHECK, 0, 0) == BST_CHECKED;
	if( bScaleBrightness ) {
		if( bScaleDuration )
			ScaleType = ScaleBoth;
		else
			ScaleType = ScaleBrightness;
	} else {
		if( bScaleDuration )
			ScaleType = ScaleDuration;
		else
			ScaleType = ScaleNeither;
	}
	bMerge = SendMessage(GetSubWindow(hWnd, IDC_MERGE), BM_GETCHECK, 0, 0) == BST_CHECKED;
	bMix = SendMessage(GetSubWindow(hWnd, IDC_MIX), BM_GETCHECK, 0, 0) == BST_CHECKED;

	NumBands = 0;
	for( int i = 0; i < 32; ++i )
		if( selected_lights&(1<<i) )
			++NumBands;

	bMerge = SendMessage(GetSubWindow(hWnd, IDC_MERGE), BM_GETCHECK, 0, 0) == BST_CHECKED;

	LastSensitivity = Sensitivity;
	if( bFinalise )
		FinaliseSpectrumAnalysis(hWnd, selection_start, selection_finish, StartFrequency, FinishFrequency, NumBands, Logarithmic, Stereo, Order, Effect, ScaleType, Brightness, Sensitivity, 10,
		                         OnTime, RiseTime, FallTime, Delay, MinInterval, selected_lights, bMerge, bMix);
	else
		SpectrumAnalysis(hWnd, selection_start, selection_finish, StartFrequency, FinishFrequency, NumBands, Logarithmic, Stereo, Order, Effect, ScaleType, Brightness, Sensitivity, 10,
			             OnTime, RiseTime, FallTime, Delay, MinInterval, selected_lights, bMerge, bMix, true);
}

extern bool bDidSpectrumAnalysis, bDidSpectrumAnalysisPreview;
extern int SpectrumAnalysisPreviewStart, SpectrumAnalysisPreviewFinish;
static INT_PTR CALLBACK SpectrumAnalysisDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[8] = { IDC_FREQUENCY_START_SPIN, IDC_FREQUENCY_FINISH_SPIN, IDC_BRIGHTNESS_SPIN, IDC_ONTIME_SPIN, IDC_DELAY_SPIN, IDC_MINIMUM_INTERVAL_SPIN, IDC_RISETIME_SPIN, IDC_FALLTIME_SPIN };
			int ranges[8] = { 20000|(20<<16), 20000|(20<<16), 100, 1000, 2000, 1000, 1000, 1000 };
			int default_values[8] = { 100, 10000, 100, 100, 1000, 100, 100, 100 };
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_STEREO_MONO), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_CHANS_LOWTOHIGH), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_RESPONSE_FADE), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETRANGE, TRUE, MAKELPARAM(0, 40));
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETTICFREQ, 10, 0);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETPAGESIZE, 0, 10);
			SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_SETPOS, true, 20);
			SendMessage(GetSubWindow(hWnd, IDC_SCALE_DURATION), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			UpdateSpectrumAnalysisStereoDialogState(hWnd);
			LoadDialogStateFromRegistry(hWnd, L"SpectrumAnalysis", SpectrumAnalysisRegistryEntries, sizeof(SpectrumAnalysisRegistryEntries)/sizeof(*SpectrumAnalysisRegistryEntries));
			UpdateSpectrumAnalysisDialogState(hWnd);
			wchar_t buf[32];
			if( GetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf, sizeof(buf)/sizeof(wchar_t)) ) {
				int delay = GetNum(buf);
				if( delay >= -1000 && delay <= 1000 )
					SendMessage(GetSubWindow(hWnd, IDC_DELAY_SPIN), UDM_SETPOS, 0, delay + 1000);
			}
			SendMessage(hWnd, WM_VSCROLL, 0, (LPARAM)GetSubWindow(hWnd, IDC_DELAY_SPIN));
//			if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
//				UpdateSpectrumAnalysis(hWnd);
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDC_RESPONSE_TRACK:
				case IDC_RESPONSE_FLASH:
				case IDC_RESPONSE_FADE:
					UpdateSpectrumAnalysisDialogState(hWnd);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateSpectrumAnalysis(hWnd);
					break;
				case IDC_PREVIEW:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateSpectrumAnalysis(hWnd);
					else
						CancelSpectrumAnalysis(hWnd);
					return TRUE;
				case IDC_MERGE:
				case IDC_MIX:
					SendMessage(GetSubWindow(hWnd, wmId == IDC_MERGE ? IDC_MIX : IDC_MERGE), BM_SETCHECK, 0, 0);
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateSpectrumAnalysis(hWnd);
					break;
				case IDOK:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) != BST_CHECKED )
						UpdateSpectrumAnalysis(hWnd);
					UpdateSpectrumAnalysis(hWnd, true);
					SaveDialogStateToRegistry(hWnd, L"SpectrumAnalysis", SpectrumAnalysisRegistryEntries, sizeof(SpectrumAnalysisRegistryEntries)/sizeof(*SpectrumAnalysisRegistryEntries));
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					CancelSpectrumAnalysis(hWnd);
					DestroyWindow(hWnd);
					return TRUE;
				default:
					if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
						UpdateSpectrumAnalysis(hWnd);
					break;
				}
			} else if( wmEvent == EN_CHANGE ) {
				if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
					UpdateSpectrumAnalysis(hWnd);
			}
			break;
		}
	case WM_HSCROLL:
		if( lParam && GetWindowLong((HWND)lParam, GWL_ID) == IDC_SENSITIVITY ) {
			int Sensitivity = SendMessage(GetSubWindow(hWnd, IDC_SENSITIVITY), TBM_GETPOS, 0, 0);
			if( Sensitivity != LastSensitivity && SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
				UpdateSpectrumAnalysis(hWnd);
		}
		break;
	case WM_VSCROLL:
		if( lParam && GetWindowLong((HWND)lParam, GWL_ID) == IDC_DELAY_SPIN ) {
			int pos;
			wchar_t buf[8];

			pos = LOWORD(SendMessage(GetSubWindow(hWnd, IDC_DELAY_SPIN), UDM_GETPOS, 0, 0));
			wsprintf(buf, L"%c%4d", pos < 1000 ? '-' : ' ', pos < 1000 ? 1000 - pos : pos - 1000);
			SetWindowText(GetSubWindow(hWnd, IDC_DELAY), buf);

//			if( wParam && SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
//				UpdateSpectrumAnalysis(hWnd);
		}
		break;
	case WM_CLOSE:
		CancelSpectrumAnalysis(hWnd);
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		CancelSpectrumAnalysis(hWnd);
		return TRUE;
	case WM_POSTCHANGESELECTION:
		UpdateSpectrumAnalysisStereoDialogState(hWnd);
		if( SendMessage(GetSubWindow(hWnd, IDC_PREVIEW), BM_GETCHECK, 0, 0) == BST_CHECKED )
			UpdateSpectrumAnalysis(hWnd);
		return TRUE;
	case WM_MAINSCROLL:
		if( bDidSpectrumAnalysis && bDidSpectrumAnalysisPreview ) {
			SCROLLINFO si;
			si.cbSize = sizeof (si);
			si.fMask  = SIF_ALL;
			GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
			int view_left = si.nPos << zoom;
			int view_right = (si.nPos + (rcWAV.right-rcWAV.left)) << zoom;
			if( view_left < SpectrumAnalysisPreviewStart || view_right > SpectrumAnalysisPreviewFinish )
				UpdateSpectrumAnalysis(hWnd);
		}
		break;
	}
	return FALSE;
}

void OpenSpectrumAnalysisDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Spectrum Analysis") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the Spectrum Analysis feature.", L"Spectrum Analysis requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_SPECTRUMANALYSIS), hWnd, SpectrumAnalysisDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}
void DoSpectrumAnalysis(HWND hWnd, HINSTANCE hInst) {
	if( selection_start == -1 || !selected_lights ) {
		MessageBox(hWnd, L"You must select a time range for at least one light before using the SpectrumAnalysis feature.", L"Spectrum Analysis requires light selection", MB_OK|MB_ICONEXCLAMATION);
	} else {
		HWND hwndTemp = CreateDialog(hInst, MAKEINTRESOURCE(IDD_SPECTRUMANALYSIS), hWnd, SpectrumAnalysisDialogProc);
		SendMessage(hwndTemp, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
		DestroyWindow(hwndTemp);
	}
}


void DoAutomaticSequencing(HWND hWnd) {
	if( ProgressDialogIsOpen() )
		return;

	wchar_t buf[32];
	int NumLights = 8, FlashPeriod = 100, BeatDetectionPct = 40;
	bool bPingPong, bFade, bStereo, bTrack, bScaleDuration;
	int BeatDetectionSensitivity, SpectrumAnalysisSensitivity;

	if( GetWindowText(GetSubWindow(hWnd, IDC_NUMLIGHTS), buf, sizeof(buf)/sizeof(wchar_t)) )
		NumLights = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_FLASHPERIOD), buf, sizeof(buf)/sizeof(wchar_t)) )
		FlashPeriod = GetNum(buf);
	if( GetWindowText(GetSubWindow(hWnd, IDC_BEATDETECTIONPCT), buf, sizeof(buf)/sizeof(wchar_t)) )
		BeatDetectionPct = GetNum(buf);
	bPingPong = (SendMessage(GetSubWindow(hWnd, IDC_PINGPONG), BM_GETCHECK, 0, 0) == BST_CHECKED);
	bFade     = (SendMessage(GetSubWindow(hWnd, IDC_FADE    ), BM_GETCHECK, 0, 0) == BST_CHECKED);
	bStereo   = (SendMessage(GetSubWindow(hWnd, IDC_STEREO  ), BM_GETCHECK, 0, 0) == BST_CHECKED);
	bTrack    = (SendMessage(GetSubWindow(hWnd, IDC_TRACK   ), BM_GETCHECK, 0, 0) == BST_CHECKED);
	bScaleDuration = (SendMessage(GetSubWindow(hWnd, IDC_SCALE_DURATION), BM_GETCHECK, 0, 0) == BST_CHECKED);
	BeatDetectionSensitivity    = SendMessage(GetSubWindow(hWnd, IDC_BEAT_SENSITIVITY    ), TBM_GETPOS, 0, 0);
	SpectrumAnalysisSensitivity = SendMessage(GetSubWindow(hWnd, IDC_SPECTRUM_SENSITIVITY), TBM_GETPOS, 0, 0);

	int BeatDetectionNum = (NumLights * BeatDetectionPct + 50) / 100;
	if( BeatDetectionNum == 0 && NumLights >= 2 && BeatDetectionPct > 0 )
		BeatDetectionNum = 1;
	else if( BeatDetectionNum == NumLights && NumLights >= 2 && BeatDetectionPct < 100 )
		BeatDetectionNum = NumLights-1;
	int SpectrumAnalysisNum = NumLights - BeatDetectionNum;

	unsigned long BeatDetectionLights = 0, SpectrumAnalysisLights = 0;
	int i;
	int temp = BeatDetectionNum;
	for( i = 0; i < 32; ++i ) {
		if( !temp )
			break;
		if( EnabledLights&(1<<i) ) {
			BeatDetectionLights |= (1<<i);
			--temp;
		}
	}
	    temp = SpectrumAnalysisNum;
	for( ; i < 32; ++i ) {
		if( !temp )
			break;
		if( EnabledLights&(1<<i) ) {
			SpectrumAnalysisLights |= (1<<i);
			--temp;
		}
	}
	HWND hParent = GetParent(hWnd);
	UndoBegin();
	StartProgress(hWnd, L"automatic sequencing");
	if( BeatDetectionLights ) {
		if( SpectrumAnalysisLights ) {
			if( !ProgressStep(hWnd, 2+SpectrumAnalysisNum, 0, 2) ) {
				FinishProgress(hWnd);
				UndoEnd(hParent);
				return;
			}
		}
		BeatDetection(hWnd, 0, wav_len, 20 + (BeatDetectionSensitivity-2)*2, 100, bFade ? -25 : 0, bFade ? Fade : Flash, 100, bFade ? FlashPeriod : FlashPeriod*2, 50, 50, bPingPong ? PingPong : LowToHigh, BeatDetectionLights, false, false, false);
		FinaliseBeatDetection(hWnd, 0, wav_len, 20 + (BeatDetectionSensitivity-2)*2, 100, bFade ? -25 : 0, bFade ? Fade : Flash, 100, bFade ? FlashPeriod : FlashPeriod*2, 50, 50, bPingPong ? PingPong : LowToHigh, BeatDetectionLights, false, false);
	}
	if( SpectrumAnalysisLights ) {
		if( BeatDetectionLights ) {
			if( !ProgressStep(hWnd, 2+SpectrumAnalysisNum, 2, SpectrumAnalysisNum) ) {
				FinishProgress(hWnd);
				UndoEnd(hParent);
				if( BeatDetectionLights )
					Undo(hParent);
				return;
			}
		}
		SpectrumAnalysis(hWnd, 0, wav_len, 100, 6000, SpectrumAnalysisNum, Logarithmic, bStereo ? LeftToRight : Mono, LowToHigh, bTrack ? Track : Fade, bScaleDuration ? ScaleDuration : ScaleBrightness,
		                       bTrack ? 75 : 100, 20 + (SpectrumAnalysisSensitivity-2)*2, 10, FlashPeriod, FlashPeriod/2, FlashPeriod/2, -FlashPeriod/2, 100, SpectrumAnalysisLights, false, false, false);
		FinaliseSpectrumAnalysis(hWnd, 0, wav_len, 100, 6000, SpectrumAnalysisNum, Logarithmic, bStereo ? LeftToRight : Mono, LowToHigh, bTrack ? Track : Fade, bScaleDuration ? ScaleDuration : ScaleBrightness,
		                                  bTrack ? 75 : 100, 20 + (SpectrumAnalysisSensitivity-2)*2, 10, FlashPeriod, FlashPeriod/2, FlashPeriod/2, -FlashPeriod/2, 100, SpectrumAnalysisLights, false, false);
	}
	UndoEnd(hParent);
	if( !FinishProgress(hWnd) )
		Undo(hParent);
}

static INT_PTR CALLBACK AutomaticSequencingDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			int updown_controls[3] = { IDC_NUMLIGHTS_SPIN, IDC_FLASHPERIOD_SPIN, IDC_BEATDETECTIONPCT_SPIN };
			int ranges[3] = { 32|(1<<16), 1000, 100 };
			int default_values[3] = { 0, 100, 40 };
			for( int i = 0; i < 32; ++i )
				if( EnabledLights&(1<<i) )
					++default_values[0];
			for( int i = 0; i < sizeof(updown_controls)/sizeof(*updown_controls); ++i ) {
				HWND hUpDown = GetSubWindow(hWnd, updown_controls[i]);
				SendMessage(hUpDown, UDM_SETBASE, 10, 0);
				SendMessage(hUpDown, UDM_SETRANGE, 0, ranges[i]);
				SendMessage(hUpDown, UDM_SETPOS, 0, default_values[i]);
			}
			SendMessage(GetSubWindow(hWnd, IDC_PINGPONG), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_FADE), BM_SETCHECK, BST_CHECKED, (LPARAM)NULL);
			SendMessage(GetSubWindow(hWnd, IDC_BEAT_SENSITIVITY), TBM_SETRANGE, TRUE, MAKELPARAM(0, 4));
			SendMessage(GetSubWindow(hWnd, IDC_BEAT_SENSITIVITY), TBM_SETTICFREQ, 1, 0);
			SendMessage(GetSubWindow(hWnd, IDC_BEAT_SENSITIVITY), TBM_SETPAGESIZE, 0, 1);
			SendMessage(GetSubWindow(hWnd, IDC_BEAT_SENSITIVITY), TBM_SETPOS, true, 2);
			SendMessage(GetSubWindow(hWnd, IDC_SPECTRUM_SENSITIVITY), TBM_SETRANGE, TRUE, MAKELPARAM(0, 4));
			SendMessage(GetSubWindow(hWnd, IDC_SPECTRUM_SENSITIVITY), TBM_SETTICFREQ, 1, 0);
			SendMessage(GetSubWindow(hWnd, IDC_SPECTRUM_SENSITIVITY), TBM_SETPAGESIZE, 0, 1);
			SendMessage(GetSubWindow(hWnd, IDC_SPECTRUM_SENSITIVITY), TBM_SETPOS, true, 2);
			LoadDialogStateFromRegistry(hWnd, L"AutomaticSequencing", AutomaticSequencingRegistryEntries, sizeof(AutomaticSequencingRegistryEntries)/sizeof(*AutomaticSequencingRegistryEntries));
			break;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDOK:
					DoAutomaticSequencing(hWnd);
					SaveDialogStateToRegistry(hWnd, L"AutomaticSequencing", AutomaticSequencingRegistryEntries, sizeof(AutomaticSequencingRegistryEntries)/sizeof(*AutomaticSequencingRegistryEntries));
					DestroyWindow(hWnd);
					return TRUE;
				case IDCANCEL:
					DestroyWindow(hWnd);
					return TRUE;
				default:
					break;
				}
			}
			break;
		}
	case WM_CLOSE:
		DestroyWindow(hWnd);
		return TRUE;
	case WM_DESTROY:
		hDialog = (HWND)NULL;
		break;
	case WM_CANCHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_PRECHANGESELECTION:
		if( lParam )
			((int*)lParam)[0] = 1;
		return TRUE;
	case WM_POSTCHANGESELECTION:
		return TRUE;
	}
	return FALSE;
}

void OpenAutomaticSequencingDialog(HWND hWnd, HINSTANCE hInst, HWND* phDialog) {
	wchar_t buf[256];
	if( *phDialog ) {
		GetWindowText(*phDialog, buf, sizeof(buf)/sizeof(wchar_t));
		if( wcscmp(buf, L"Automatic Sequencing") )
			MessageBox(hWnd, L"You have another dialog open. Please close it first.", L"Another dialog is already open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(*phDialog);
	} else {
		*phDialog = CreateDialog(hInst, MAKEINTRESOURCE(IDD_AUTOMATICSEQUENCING), hWnd, AutomaticSequencingDialogProc);
		ShowWindow(*phDialog, TRUE);
	}
}



int GetLength(HWND hWnd) {
	wchar_t hours[32], minutes[32], seconds[32];
	wchar_t* end, * end2;
	int h, m, s, ms;
	GetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_HOURS), hours, sizeof(hours)/sizeof(wchar_t));
	GetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_MINUTES), minutes, sizeof(minutes)/sizeof(wchar_t));
	GetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_SECONDS), seconds, sizeof(seconds)/sizeof(wchar_t));
	h = wcstol(hours, &end, 10);
	if( *end )
		return -1;
	m = wcstol(minutes, &end, 10);
	if( *end )
		return -1;
	s = wcstol(seconds, &end, 10);
	if( *end == L'.' ) {
		++end;
		ms = wcstol(end, &end2, 10);
		if( *end2 )
			return -1;
		while( end2-end < 3 ) {
			ms *= 10;
			--end;
		}
		while( end2-end > 3 ) {
			ms /= 10;
			++end;
		}
	} else {
		if( *end )
			return -1;
		ms = 0;
	}
	return ms + ((s + (m + h * 60) * 60) * 1000);
}

static INT_PTR CALLBACK NewSequenceDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			SendMessage(GetSubWindow(hWnd, IDC_SEQUENCE_WAV), BM_SETCHECK, BST_CHECKED, 0);
			return TRUE;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
				case IDOK:
					if( SendMessage(GetSubWindow(hWnd, IDC_SEQUENCE_WAV), BM_GETCHECK, 0, 0) == BST_CHECKED ) {
						EndDialog(hWnd, -1);
					} else {
						int t = GetLength(hWnd);
						if( t <= 0 ) {
							MessageBox(hWnd, L"Error: The sequence length you have entered is not a valid time. Ensure that the fields contain digits 0-9 only and that they are all integers (with the exception of the seconds field, which may contain a decimal point).", L"Invalid Sequence Length", MB_OK|MB_ICONEXCLAMATION);
						} else {
							EndDialog(hWnd, t);
						}
						return TRUE;
					}
				}
			} else if( wmEvent == EN_SETFOCUS ) {
				switch(wmId) {
				case IDC_SEQUENCE_HOURS:
				case IDC_SEQUENCE_MINUTES:
				case IDC_SEQUENCE_SECONDS:
					SendMessage(GetSubWindow(hWnd, IDC_SEQUENCE_WAV), BM_SETCHECK, 0, 0);
					SendMessage(GetSubWindow(hWnd, IDC_SEQUENCE_STANDALONE), BM_SETCHECK, BST_CHECKED, 0);
					break;
				}
			}
			break;
		}
	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return TRUE;
	}
	return FALSE;
}

bool NewSequenceDialog(HWND hWnd, HINSTANCE hInst, int* pLengthToHere) {
	*pLengthToHere = DialogBox(hInst, MAKEINTRESOURCE(IDD_NEWSEQUENCE), hWnd, NewSequenceDialogProc);
	return *pLengthToHere != 0;
}


static int g_Length;
static INT_PTR CALLBACK ChangeLengthDialogProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	switch(uMsg) {
	case WM_INITDIALOG:
		{
			wchar_t buf[32];
			int len;
			if( g_Length >= 60*60*1000 ) {
				wsprintf(buf, L"%d", g_Length/60/60/1000);
				SetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_HOURS), buf);
			}
			if( g_Length >= 60*1000 ) {
				wsprintf(buf, L"%d", g_Length/60/1000%60);
				SetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_MINUTES), buf);
			}
			len = wsprintf(buf, L"%d", g_Length/1000%60);
			if( g_Length%1000 ) {
				buf[len++] = L'.';
				len += wsprintf(buf+len, L"%03d", g_Length%1000);
				while( buf[len-1] == L'0' )
					buf[--len] = L'\0';
			}
			SetWindowText(GetSubWindow(hWnd, IDC_SEQUENCE_SECONDS), buf);
			return TRUE;
		}
	case WM_COMMAND:
		{
			int wmId    = LOWORD(wParam);
			int wmEvent = HIWORD(wParam);
			if( wmEvent == 0 ) {
				switch(wmId) {
				case IDCANCEL:
					EndDialog(hWnd, 0);
					return TRUE;
				case IDOK:
					{
						int t = GetLength(hWnd);
						if( t <= 0 ) {
							MessageBox(hWnd, L"Error: The sequence length you have entered is not a valid time. Ensure that the fields contain digits 0-9 only and that they are all integers (with the exception of the seconds field, which may contain a decimal point).", L"Invalid Sequence Length", MB_OK|MB_ICONEXCLAMATION);
						} else {
							EndDialog(hWnd, t);
						}
						return TRUE;
					}
				}
			}
			break;
		}
	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return TRUE;
	}
	return FALSE;
}

bool ChangeLengthDialog(HWND hWnd, HINSTANCE hInst, int* pLengthHere) {
	g_Length = *pLengthHere;
	*pLengthHere = DialogBox(hInst, MAKEINTRESOURCE(IDD_CHANGELENGTH), hWnd, ChangeLengthDialogProc);
	return *pLengthHere > 0;
}
