
//#include "targetver.h"
#include <Windows.h>
#include <mmsystem.h>
#include <Dsound.h>
#include "playback.h"

struct PlaybackInfo {
	signed short* buffer, * init_buffer;
	int length, init_length, sample_rate, channels;
	int init_sample_pos, sample_pos, write_pos, buffer_size;
	int finish_write;
	volatile int play, pause, playing, error, loop, looped;
	int paused, initialised;

	CRITICAL_SECTION cs;

	LPDIRECTSOUND8 lpds;
	LPDIRECTSOUNDBUFFER lpdsbuffer;
};

int playback_cursor_delay_ms;
PlaybackInfo g_pi;

static void FillAudioBuffer() {
	DWORD dwPlay, dwWrite;
	g_pi.lpdsbuffer->GetCurrentPosition(&dwPlay, &dwWrite);
	int write = (int)dwPlay - g_pi.write_pos;
	if( write < 0 )
		write += g_pi.buffer_size;
	if( write < 256 )
		return;
	if( g_pi.length > 0 && write > g_pi.length * g_pi.channels * 2 )
		write = g_pi.length * g_pi.channels * 2;

	LPVOID lpvWrite1, lpvWrite2;
	DWORD  dwLength1, dwLength2;
	if( g_pi.lpdsbuffer->Lock(g_pi.write_pos, write, &lpvWrite1, &dwLength1, &lpvWrite2, &dwLength2, 0) == DS_OK ) {
		if( g_pi.length == 0 || !g_pi.buffer )
			memset(lpvWrite1, 0, dwLength1);
		else
			memcpy(lpvWrite1, g_pi.buffer, dwLength1);
		if( g_pi.buffer )
			g_pi.buffer += dwLength1>>1;
		if( g_pi.length )
			g_pi.length -= dwLength1 / (g_pi.channels * 2);
		if( lpvWrite2 ) {
			if( g_pi.length == 0 || !g_pi.buffer )
				memset(lpvWrite2, 0, dwLength2);
			else
				memcpy(lpvWrite2, g_pi.buffer, dwLength2);
			if( g_pi.buffer )
				g_pi.buffer += dwLength2>>1;
			if( g_pi.length )
				g_pi.length -= dwLength2 / (g_pi.channels * 2);
		}
		g_pi.lpdsbuffer->Unlock(lpvWrite1, dwLength1, lpvWrite2, dwLength2);

		EnterCriticalSection(&g_pi.cs);
		g_pi.write_pos += dwLength1 + dwLength2;
		if( g_pi.write_pos >= g_pi.buffer_size )
			g_pi.write_pos -= g_pi.buffer_size;

		g_pi.sample_pos += (dwLength1 + dwLength2) / (g_pi.channels * 2);
		LeaveCriticalSection(&g_pi.cs);
	}

	if( g_pi.length == 0 ) {
		if( g_pi.loop ) {
			g_pi.buffer = g_pi.init_buffer;
			g_pi.length = g_pi.init_length;
			g_pi.sample_pos = g_pi.init_sample_pos;
			g_pi.looped = 1;
		} else if( g_pi.finish_write == -1 ) {
			g_pi.finish_write = g_pi.buffer_size;
		} else {
			g_pi.finish_write -= write;
			if( g_pi.finish_write <= 0 ) {
				EnterCriticalSection(&g_pi.cs);
				g_pi.lpdsbuffer->Release();
				g_pi.play = 0;
				g_pi.playing = 0;
				LeaveCriticalSection(&g_pi.cs);
			}
		}
	}
}

DWORD WINAPI playback_thread(void* param) {
	while(1) {
		if( g_pi.play ) {
			if( g_pi.playing ) {
				int pause = g_pi.pause;
				if( pause != g_pi.paused ) {
					if( pause )
						g_pi.lpdsbuffer->Stop();
					else
						g_pi.lpdsbuffer->Play(0, 0, DSBPLAY_LOOPING);
					g_pi.paused = pause;
				}
				FillAudioBuffer();
			} else if( !g_pi.playing && !g_pi.error ) {
				DSBUFFERDESC dsbdesc; 
				WAVEFORMATEX wfm;
				wfm.cbSize = sizeof(wfm);
				wfm.wFormatTag = WAVE_FORMAT_PCM;
				wfm.nChannels = g_pi.channels;
				wfm.nSamplesPerSec = g_pi.sample_rate;
				wfm.nAvgBytesPerSec = g_pi.sample_rate*g_pi.channels*2;
				wfm.nBlockAlign = g_pi.channels*2;
				wfm.wBitsPerSample = 16;
				memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); 
				dsbdesc.dwSize = sizeof(DSBUFFERDESC); 
				dsbdesc.dwFlags = DSBCAPS_GLOBALFOCUS|DSBCAPS_TRUEPLAYPOSITION|DSBCAPS_GETCURRENTPOSITION2;
				g_pi.buffer_size = 32768 * g_pi.channels;
				dsbdesc.dwBufferBytes = g_pi.buffer_size;
				dsbdesc.lpwfxFormat = &wfm;
				dsbdesc.guid3DAlgorithm = DS3DALG_DEFAULT;

				if( g_pi.lpds->CreateSoundBuffer(&dsbdesc, &g_pi.lpdsbuffer, NULL) == DS_OK ||
				   (dsbdesc.dwFlags = DSBCAPS_GLOBALFOCUS|DSBCAPS_GETCURRENTPOSITION2, g_pi.lpds->CreateSoundBuffer(&dsbdesc, &g_pi.lpdsbuffer, NULL) == DS_OK) ) {
					g_pi.finish_write = -1;
					g_pi.write_pos = 0;
					g_pi.looped = 0;
					FillAudioBuffer();
					g_pi.lpdsbuffer->Play(0, 0, DSBPLAY_LOOPING);
					g_pi.playing = 1;
					g_pi.paused = 0;
				} else {
					g_pi.error = 1;
				}
			}
		} else if( g_pi.playing ) {
			EnterCriticalSection(&g_pi.cs);
			g_pi.lpdsbuffer->Stop();
			g_pi.lpdsbuffer->Release();
			g_pi.playing = 0;
			LeaveCriticalSection(&g_pi.cs);
		}
		Sleep(25);
	}
	return 0;
}

bool init_playback(HWND hWnd) {
	if( DirectSoundCreate8(NULL, &g_pi.lpds, NULL) != DS_OK ||
		g_pi.lpds->SetCooperativeLevel(hWnd, DSSCL_PRIORITY) != DS_OK )
		return false;
	InitializeCriticalSection(&g_pi.cs);
	g_pi.initialised = 1;
	CreateThread(NULL, 0, &playback_thread, 0, 0, NULL);
	return true;
}

bool start_playback(signed short* buffer, int length, int sample_rate, int channels, int sample_offset) {
	stop_playback();
	g_pi.buffer = g_pi.init_buffer = buffer;
	g_pi.length = g_pi.init_length = length;
	g_pi.sample_rate = sample_rate;
	g_pi.channels = channels;
	g_pi.init_sample_pos = g_pi.sample_pos = sample_offset;
	g_pi.play = 1;
	g_pi.pause = 0;
	while( !g_pi.playing && !g_pi.error )
		Sleep(10);

	if( g_pi.error ) {
		g_pi.play = 0;
		g_pi.error = 0;
		return false;
	} else {
		return true;
	}
}

int playback_get_current_sample() {
	int ret;
	if( !g_pi.initialised )
		return 0;
	EnterCriticalSection(&g_pi.cs);
	if( g_pi.playing ) {

		DWORD dwPlay, dwWrite;
		g_pi.lpdsbuffer->GetCurrentPosition(&dwPlay, &dwWrite);
		int buffered = g_pi.write_pos - (int)dwPlay;
		if( buffered <= 0 )
			buffered += g_pi.buffer_size;
		ret = g_pi.sample_pos - buffered / (g_pi.channels * 2) - playback_cursor_delay_ms * g_pi.sample_rate / 1000;
		if( ret < g_pi.init_sample_pos ) {
			if( g_pi.looped ) {
				ret += g_pi.init_length;
				if( ret < g_pi.init_sample_pos )
					ret = g_pi.init_sample_pos;
			} else {
				ret = g_pi.init_sample_pos;
			}
		}
	} else {
		ret = -1;
	}
	LeaveCriticalSection(&g_pi.cs);

	return ret;
}

void stop_playback() {
	if( g_pi.play ) {
		g_pi.play = 0;
		while( g_pi.playing )
			Sleep(10);
	}
}

void pause_resume_playback() {
	if( g_pi.playing )
		g_pi.pause = !g_pi.pause;
	while( g_pi.paused != g_pi.pause )
		Sleep(10);
}

playback_status get_playback_status() {
	if( g_pi.playing )
		if( g_pi.paused )
			return paused;
		else
			return playing;
	else
		return stopped;
}

void set_loop_playback(bool bLoop) {
	g_pi.loop = bLoop;
}

void set_playback_cursor_delay(int ms) {
	playback_cursor_delay_ms = ms;
}

bool AmPlaying(const wchar_t** pReason) {
	if( pReason )
		*pReason = L"playback has not started";
	return g_pi.playing != 0;
}
