
#include "targetver.h"
#include "EditSequence.h"
#include "Sequence.h"
#include "WindowDrawing.h"
#include "WAVBrowse.h"
#include "LoadSave.h"
#include "Undo.h"
#include "DSP.cpp"
#include "Progress.h"
#ifdef _DEBUG
#include <assert.h>
#endif


extern HWND hDialog;
bool bDidCascade;


bool SetLightBrightness(HWND hWnd, int i, int start, int finish, int brightness, sequence* arg, bool bForce) {
	sequence* pSeq = &sequences[i];

	if( hDialog && !bForce ) {
		MessageBox(hWnd, L"You can not make direct changes to a sequence while a dialog is open. Please close it first.", L"Can't make changes with dialog open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(hDialog);
		return false;
	}

	// determine the state of the light at the beginning and end of the period
	int start_ms = (int)((start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
	int finish_ms = (int)((finish + 1) * 1000LL / wav_sample_rate);
	int finish_ramp_later = 0;

#ifdef _DEBUG
	if( brightness == SETBRIGHTNESS_PASTE || brightness == SETBRIGHTNESS_INSERT )
		assert(sequence_get_length_ms(arg) == (finish_ms - start_ms));
#endif

	if( finish_ms == start_ms ) {
		sequence_state ss;
		sequence_init(pSeq, &ss);
		sequence_advance(pSeq, &ss, start_ms, true, false);
		UndoPrepare(i);
		while( ss.pos < pSeq->used && !IS_DELAY_CMD(pSeq->commands[ss.pos]) )
			sequence_remove_command(pSeq, ss.pos);
		if( brightness != SETBRIGHTNESS_CANCELACTIONS ) {
			if( ss.ms_remain )
				sequence_break_delay(pSeq, &ss, NULL);
			sequence_insert_set_brightness(pSeq, i, brightness, &ss, NULL);
		}
		UndoFinish(i, start, wav_len);
		MainSequencesChanged(hWnd, start, wav_len);
		return true;
	} else {
		sequence_state start_ss, finish_ss;
		sequence_init(pSeq, &start_ss);
		sequence_advance(pSeq, &start_ss, start_ms, true, false);
		finish_ss = start_ss;
		sequence_advance(pSeq, &finish_ss, finish_ms, true, false);
		int orig_start_pos = start_ss.pos;
		UndoPrepare(i);
		if( start_ss.ramp || brightness != SETBRIGHTNESS_CANCELACTIONS ) {
			if( start_ss.ms_remain )
				sequence_break_delay(pSeq, &start_ss, &finish_ss);
			while( start_ss.pos < pSeq->used && !IS_DELAY_CMD(pSeq->commands[start_ss.pos]) )
				++start_ss.pos;
			if( brightness == SETBRIGHTNESS_INTERPOLATEDRAMP && start_ss.brightness != finish_ss.brightness ) {
				sequence_insert_set_brightness(pSeq, i, start_ss.brightness, &start_ss, &finish_ss);
				if( !sequence_insert_ramp(pSeq, i, finish_ss.brightness > start_ss.brightness ? 1 : -1, (finish_ms - start_ms) * 255 / (finish_ss.brightness > start_ss.brightness ? finish_ss.brightness : 255-finish_ss.brightness), &start_ss, &finish_ss) )
					finish_ramp_later = brightness;
			} else if( brightness == SETBRIGHTNESS_RAMPUP && start_ss.brightness != 255 ) {
				sequence_insert_set_brightness(pSeq, i, start_ss.brightness, &start_ss, &finish_ss);
				if( !sequence_insert_ramp(pSeq, i, 1, (finish_ms - start_ms), &start_ss, &finish_ss) )
					finish_ramp_later = brightness;
			} else if( brightness == SETBRIGHTNESS_RAMPDOWN && start_ss.brightness != 0 ) {
				sequence_insert_set_brightness(pSeq, i, start_ss.brightness, &start_ss, &finish_ss);
				if( !sequence_insert_ramp(pSeq, i, -1, (finish_ms - start_ms), &start_ss, &finish_ss) )
					finish_ramp_later = brightness;
			} else if( brightness != SETBRIGHTNESS_PASTE && brightness != SETBRIGHTNESS_INSERT ) {
				sequence_insert_set_brightness(pSeq, i, brightness < 0 ? start_ss.brightness : brightness, &start_ss, &finish_ss);
			}
		}
		if( start_ss.pos != finish_ss.pos && brightness != SETBRIGHTNESS_PASTE && brightness != SETBRIGHTNESS_INSERT ) {
			// remove intervening commands
			int dest = start_ss.pos;
			int source = start_ss.pos;
			int end = finish_ss.pos;

			while( source < end ) {
				if( IS_DELAY_CMD(pSeq->commands[source]) )
					pSeq->commands[dest++] = pSeq->commands[source];
				++source;
			}
			if( source != dest ) {
				sequence_remove_command(pSeq, dest, source-dest);
				finish_ss.pos -= source-dest;
			}
		}
		if( finish_ss.ramp || brightness >= 0 || finish_ss.brightness != brightness || brightness == SETBRIGHTNESS_PASTE || brightness == SETBRIGHTNESS_INSERT ) {
			if( finish_ss.ms_remain )
				sequence_break_delay(pSeq, &finish_ss, NULL);
			if( brightness == SETBRIGHTNESS_PASTE || brightness == SETBRIGHTNESS_INSERT ) {
				int start_pos = start_ss.pos;
				if( brightness == SETBRIGHTNESS_INSERT ) {
					while( start_pos < pSeq->used && !IS_DELAY_CMD(pSeq->commands[start_pos]) )
						++start_pos;
				}
				int delta = sequence_replace_translated_commands(pSeq, arg, start_pos, finish_ss.pos, i, finish_ms - start_ms);
				finish_ss.pos += delta;
				sequence_advance(pSeq, &start_ss, finish_ss.ms_pos, true, false);
				if( !start_ss.ramp )
					brightness = start_ss.brightness;
			}
			if( finish_ss.brightness != brightness )
				sequence_insert_set_brightness(pSeq, i, finish_ss.brightness, &finish_ss, NULL);
			if( finish_ss.ramp )
				sequence_insert_ramp(pSeq, i, finish_ss.ramp < 0 ? -1 : 1, finish_ss.ramp_ms, &finish_ss, NULL);
		} else if( brightness == SETBRIGHTNESS_RAMPUP || brightness == SETBRIGHTNESS_RAMPDOWN ) {
			if( finish_ss.ms_remain )
				sequence_break_delay(pSeq, &finish_ss, NULL);
			if( finish_ss.pos == pSeq->used || IS_DELAY_CMD(pSeq->commands[finish_ss.pos]) ) {
				sequence_insert_set_brightness(pSeq, i, finish_ss.brightness, &finish_ss, NULL);
			}
		}

		if( finish_ramp_later ) {
			// this happens when the ramp we want to create has a duration that's too long to encode in a single command
			int duration = finish_ms - start_ms;
			int start_brightness = start_ss.brightness;
			int final_brightness = finish_ramp_later == -3 ? 255 : finish_ramp_later == -4 ? 0 : finish_ss.brightness;
			int diff = abs(final_brightness - start_brightness);
			brightness = start_brightness;
			while( brightness != final_brightness ) {
				if( final_brightness > brightness )
					++brightness;
				else
					--brightness;

				int next_ms = start_ms + abs(start_brightness - brightness) * duration / diff;
				sequence_advance(pSeq, &start_ss, next_ms, true, false);
				if( start_ss.ms_remain )
					sequence_break_delay(pSeq, &start_ss, &finish_ss);
				sequence_insert_set_brightness(pSeq, i, brightness, &start_ss, NULL);
			}
		}

#ifdef _DEBUG
		assert(sequence_get_length_ms(pSeq) == (wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
#endif
		sequence_compact(pSeq, orig_start_pos, finish_ss.pos);
#ifdef _DEBUG
		assert(sequence_get_length_ms(pSeq) == (wav_len + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
#endif
		UndoFinish(i, start, finish);
		MainSequencesChanged(hWnd, start, finish);
		return false;
	}
}

void SetSelectedLightBrightness(HWND hWnd, int brightness) {
	if( !selected_lights ) {
		MessageBox(hWnd, L"You must select the lights to modify.", L"No lights selected", MB_OK|MB_ICONINFORMATION);
	} else if( selection_start == -1 ) {
		MessageBox(hWnd, L"You must select the time or period during which to modify the light actions.", L"No selection", MB_OK|MB_ICONINFORMATION);
	} else if( hDialog ) {
		MessageBox(hWnd, L"You can not make direct changes to a sequence while a dialog is open. Please close it first.", L"Can't make changes with dialog open", MB_OK|MB_ICONEXCLAMATION);
		SetForegroundWindow(hDialog);
	} else {
		bool bRedrawRest = false;
		UndoBegin();
		for( int i = 0; i < 32; ++i ) {
			if( selected_lights&(1<<i) ) {
				if( SetLightBrightness(hWnd, i, selection_start, selection_finish, brightness) )
					bRedrawRest = true;
			}
		}
		UndoEnd(hWnd);
		int finish = selection_finish;
		if( bRedrawRest ) {
			SCROLLINFO si;
			si.cbSize = sizeof (si);
			si.fMask  = SIF_ALL;
			GetScrollInfo (hWnd, SB_HORZ, &si);
			finish = (rcWAV.right - rcWAV.left + si.nPos) << zoom;
		}
		InvalidateLightRange(hWnd, selection_start, finish, selected_lights);
	}
}

bool Cascade(HWND hWnd, int start, int finish, CascadeOrder Order, CascadeEffect Effect, int Brightness, int Period, int OnTime, int RiseTime, int FallTime, unsigned long Lights, bool bMerge, bool bMix, int* TriggerPoints, SpectrumScaleFactors* TriggerPointScale) {
	if( (Period < 10 && !TriggerPoints) || Brightness == 0 )
		return false;

	if( bDidCascade && !TriggerPoints )
		Undo(GetParent(hWnd), false);

	StartProgress(hWnd, L"cascade");

	Brightness = Brightness * 255 / 100;

	sequence sq[32];
	int sq_times[32];
	memset(&sq, 0, sizeof(sq));

	int start_ms = (int)((start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
	int finish_ms = (int)((finish + 1) * 1000LL / wav_sample_rate);
	int cur_light_index, dir = 1, last = -1, num = 0;
	for( int i = 0; i < 32; ++i ) {
		sq_times[i] = start_ms;
		if( Lights&(1<<i) ) {
			sequence_append_command(&sq[i], (1<<13)|(i<<8)|0);
			++num;
		}
	}

	switch(Order) {
	case LowToHigh:
		cur_light_index = 31;
		break;
	case HighToLow:
		cur_light_index = 0;
		break;
	case PingPong:
		cur_light_index = -1;
		break;
	case Random:
		cur_light_index = -1;
		srand(start_ms);
		break;
	}
	int ms_now = start_ms;
	while( ms_now + OnTime + (Effect == Fade ? RiseTime + FallTime : 0) <= finish_ms ) {

		if( !UpdateProgress(hWnd, ms_now-start_ms, finish_ms-start_ms) ) {
			for( int i = 0; i < 32; ++i )
				sequence_destroy(&sq[i]);
			FinishProgress(hWnd);
			if( bDidCascade )
				InvalidateLastUndo(GetParent(hWnd));
			bDidCascade = false;
			return true;
		}

		if( TriggerPoints ) {
			ms_now = *TriggerPoints++;
			if( ms_now == -1 )
				break;
		}

		int loop = 0;
		switch(Order) {
		case LowToHigh:
			do {
				cur_light_index = (cur_light_index+1)&31;
				++loop;
			} while( !(Lights&(1<<cur_light_index)) || (sq_times[cur_light_index] > ms_now && loop < 32) );
			break;
		case HighToLow:
			do {
				cur_light_index = (cur_light_index+31)&31;
				++loop;
			} while( !(Lights&(1<<cur_light_index)) || (sq_times[cur_light_index] > ms_now && loop < 32) );
			break;
		case PingPong:
			if( num > 1 )
				last = cur_light_index;
			if( dir == 1 ) {
			GoUp:
				do {
					++cur_light_index;
					++loop;
				} while( cur_light_index < 32 && (cur_light_index == last || !(Lights&(1<<cur_light_index)) || (sq_times[cur_light_index] > ms_now && loop < 63)) );

				if( cur_light_index == 32 )
					dir = -1;
			}
			if( dir == -1 ) {
				do {
					--cur_light_index;
				} while( cur_light_index >= 0 && (cur_light_index == last || !(Lights&(1<<cur_light_index)) || (sq_times[cur_light_index] > ms_now && loop < 63)) );

				if( cur_light_index == -1 ) {
					dir = 1;
					goto GoUp;
				}
			}
			break;
		case Random:
			if( num > 1 )
				last = cur_light_index;

			do {
				cur_light_index = rand()&31;
				++loop;
			} while( cur_light_index == last || !(Lights&(1<<cur_light_index)) || (sq_times[cur_light_index] > ms_now && loop < 320) );
			break;
		}

		if( sq_times[cur_light_index] < ms_now ) {
			sequence_append_delay(&sq[cur_light_index], ms_now - sq_times[cur_light_index]);
			sq_times[cur_light_index] = ms_now;
		}

		int ActualOnTime = OnTime;
		int ActualBrightness = Brightness;
		if( TriggerPointScale ) {
			ActualOnTime = ActualOnTime * TriggerPointScale[0].Duration / 255;
			ActualBrightness = ActualBrightness * TriggerPointScale[0].Brightness / 255;
			++TriggerPointScale;
		}
		if( sq_times[cur_light_index] == ms_now ) {
			int left = finish_ms - ms_now;
			switch(Effect) {
			case Flash:
				sequence_append_command(&sq[cur_light_index], (1<<13)|(cur_light_index<<8)|ActualBrightness);
				sequence_append_delay(&sq[cur_light_index], min(ActualOnTime, left));
				sq_times[cur_light_index] += min(ActualOnTime, left);
				sequence_append_command(&sq[cur_light_index], (1<<13)|(cur_light_index<<8));
				break;
			case Fade:
				{
					int ActualRiseTime = RiseTime, ActualFallTime = FallTime;
					if( TriggerPointScale ) {
						ActualRiseTime = ActualRiseTime * TriggerPointScale[-1].Duration / 255;
						ActualFallTime = ActualFallTime * TriggerPointScale[-1].Duration / 255;
					}

					if( ActualRiseTime * 255 / ActualBrightness ) {
						if( !sequence_append_ramp(&sq[cur_light_index], cur_light_index, 1, ActualRiseTime * 255 / ActualBrightness) )
							sequence_append_ramp(&sq[cur_light_index], cur_light_index, 1, 8191);
						sequence_append_delay(&sq[cur_light_index], min(ActualRiseTime, left));
						if( ActualBrightness != 255 && left >= ActualRiseTime )
							sequence_append_command(&sq[cur_light_index], (1<<13)|(cur_light_index<<8)|ActualBrightness);
					} else {
						sequence_append_command(&sq[cur_light_index], (1<<13)|(cur_light_index<<8)|ActualBrightness);
					}
					sq_times[cur_light_index] += min(ActualRiseTime, left);
					left -= min(ActualRiseTime, left);
					if( left > 0 ) {
						sequence_append_delay(&sq[cur_light_index], min(ActualOnTime, left));
						sq_times[cur_light_index] += min(ActualOnTime, left);
						left -= min(ActualOnTime, left);
						if( left > 0 ) {
							if( ActualFallTime * 255 / ActualBrightness ) {
								if( !sequence_append_ramp(&sq[cur_light_index], cur_light_index, -1, ActualFallTime * 255 / ActualBrightness) )
									sequence_append_ramp(&sq[cur_light_index], cur_light_index, -1, 8191);
								sequence_append_delay(&sq[cur_light_index], min(ActualFallTime, left));
								sq_times[cur_light_index] += min(ActualFallTime, left);
							} else {
								sequence_append_command(&sq[cur_light_index], (1<<13)|(cur_light_index<<8)|0);
							}
						}
					}
					break;
				}
			}
		}

		if( !TriggerPoints )
			ms_now += Period;
	}

	bool bRedrawRest = false;

	if( !TriggerPoints )
		UndoBegin();
	for( int i = 0; i < 32; ++i ) {
		if( Lights&(1<<i) ) {
			if( sq_times[i] < finish_ms )
				sequence_append_delay(&sq[i], finish_ms - sq_times[i]);
			if( bMerge || bMix ) {
				sequence temp;
				memset(&temp, 0, sizeof(temp));
				sequence_state ss, orig_ss;
				sequence_init(&sequences[i], &orig_ss);
				sequence_init(&sq[i], &ss);
				sequence_advance(&sequences[i], &orig_ss, start_ms, true);
				if( bMerge )
					sequence_merge(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				else
					sequence_mix(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &temp, true) )
					bRedrawRest = true;
				sequence_destroy(&temp);
			} else {
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &sq[i], true) )
					bRedrawRest = true;
			}
		}
		sequence_destroy(&sq[i]);
	}
	if( !TriggerPoints )
		UndoEnd(hWnd);

	if( bRedrawRest ) {
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
		finish = (rcWAV.right - rcWAV.left + si.nPos) << zoom;
	}
	InvalidateLightRange(GetParent(hWnd), start, finish, Lights);

	if( !TriggerPoints )
		bDidCascade = true;

	FinishProgress(hWnd);
	MarkUndo();

	return true;
}
void FinaliseCascade(HWND hWnd) {
	UnmarkUndo();
	bDidCascade = false;
}
void CancelCascade(HWND hWnd) {
	if( bDidCascade )
		UndoToMark(GetParent(hWnd));
	bDidCascade = false;
}

bool bDidCustomRamp;
bool CustomRamp(HWND hWnd, int start, int finish, CustomRampType Type, int InitialBrightness, int FinalBrightness, int PeakBrightness, int TroughBrightness, int StartDelay, int EndDelay, int PeakDelay, int TroughDelay, int Symmetry, bool bMerge, bool bMix) {
	int start_ms = (int)((start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
	int finish_ms = (int)((finish + 1) * 1000LL / wav_sample_rate);

	if( StartDelay + EndDelay + (Type == Peak ? PeakDelay : Type == Trough ? TroughDelay : 0) >= (finish_ms - start_ms) )
		return false;
	if( (Type == RampUp && FinalBrightness <= InitialBrightness) ||
		(Type == RampDown && FinalBrightness >= InitialBrightness) ||
		(Type == Peak && (PeakBrightness < InitialBrightness || PeakBrightness < FinalBrightness)) ||
		(Type == Trough && (TroughBrightness > InitialBrightness || TroughBrightness > FinalBrightness)) )
		return false;

	if( bDidCustomRamp )
		Undo(GetParent(hWnd), false);

	sequence sq[32];
	memset(&sq, 0, sizeof(sq));

	for( int i = 0; i < 32; ++i ) {
		if( selected_lights&(1<<i) ) {
			int time = (finish_ms-start_ms) - StartDelay - EndDelay;
			if( StartDelay )
				sequence_append_delay(&sq[i], StartDelay);

			switch(Type) {
			case RampUp:
				sequence_append_command(&sq[i], (1<<13)|(i<<8)|(InitialBrightness*255/100));
				if( FinalBrightness == 100 ) {
					if( sequence_append_ramp(&sq[i], i, 1, time) )
						sequence_append_delay(&sq[i], time);
					else
						sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, FinalBrightness*255/100, time);
				} else {
					if( FinalBrightness != 0 )
						sequence_append_ramp(&sq[i], i, 1, time * 100 / FinalBrightness);
					sequence_append_delay(&sq[i], time);
					sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
				}
				break;
			case RampDown:
				sequence_append_command(&sq[i], (1<<13)|(i<<8)|(InitialBrightness*255/100));
				if( FinalBrightness == 0 ) {
					if( sequence_append_ramp(&sq[i], i, -1, time) )
						sequence_append_delay(&sq[i], time);
					else
						sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, FinalBrightness*255/100, time);
				} else {
					if( FinalBrightness != 100 )
						sequence_append_ramp(&sq[i], i, -1, time * 100 / (100 - FinalBrightness));
					sequence_append_delay(&sq[i], time);
					sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
				}
				break;
			case Peak:
				{
					time -= PeakDelay;
					int risetime = time * Symmetry / 100;
					int falltime = time - risetime;
					sequence_append_command(&sq[i], (1<<13)|(i<<8)|(InitialBrightness*255/100));
					if( PeakBrightness && risetime * 100 / PeakBrightness ) {
						if( PeakBrightness == 100 ) {
							if( sequence_append_ramp(&sq[i], i, 1, risetime) )
								sequence_append_delay(&sq[i], risetime);
							else
								sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, PeakBrightness*255/100, risetime);
						} else {
							if( sequence_append_ramp(&sq[i], i, 1, risetime * 100 / PeakBrightness) )
								sequence_append_delay(&sq[i], risetime);
							else
								sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, PeakBrightness*255/100, risetime);
							sequence_append_command(&sq[i], (1<<13)|(i<<8)|(PeakBrightness*255/100));
						}
					} else {
						sequence_append_command(&sq[i], (1<<13)|(i<<8)|(PeakBrightness*255/100));
						sequence_append_delay(&sq[i], risetime);
					}
					if( PeakDelay )
						sequence_append_delay(&sq[i], PeakDelay);
					if( FinalBrightness != 100 && falltime * 100 / (100 - FinalBrightness) ) {
						if( FinalBrightness == 0 ) {
							if( sequence_append_ramp(&sq[i], i, -1, falltime) )
								sequence_append_delay(&sq[i], falltime);
							else
								sequence_append_long_ramp(&sq[i], i, PeakBrightness*255/100, FinalBrightness*255/100, falltime);
						} else {
							if( sequence_append_ramp(&sq[i], i, -1, falltime * 100 / (100 - FinalBrightness)) )
								sequence_append_delay(&sq[i], falltime);
							else
								sequence_append_long_ramp(&sq[i], i, PeakBrightness*255/100, FinalBrightness*255/100, falltime);
							sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
						}
					} else {
						sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
						sequence_append_delay(&sq[i], falltime);
					}
					break;
				}
			case Trough:
				{
					time -= TroughDelay;
					int risetime = time * Symmetry / 100;
					int falltime = time - risetime;
					sequence_append_command(&sq[i], (1<<13)|(i<<8)|(InitialBrightness*255/100));
					if( TroughBrightness != 100 && falltime * 100 / (100 - TroughBrightness) ) {
						if( TroughBrightness == 0 ) {
							if( sequence_append_ramp(&sq[i], i, -1, falltime) )
								sequence_append_delay(&sq[i], falltime);
							else
								sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, TroughBrightness*255/100, falltime);
						} else {
							if( sequence_append_ramp(&sq[i], i, -1, falltime * 100 / (100 - TroughBrightness)) )
								sequence_append_delay(&sq[i], falltime);
							else
								sequence_append_long_ramp(&sq[i], i, InitialBrightness*255/100, TroughBrightness*255/100, falltime);
							sequence_append_command(&sq[i], (1<<13)|(i<<8)|(TroughBrightness*255/100));
						}
					} else {
						sequence_append_command(&sq[i], (1<<13)|(i<<8)|(TroughBrightness*255/100));
						sequence_append_delay(&sq[i], falltime);
					}
					if( TroughDelay )
						sequence_append_delay(&sq[i], TroughDelay);
					if( FinalBrightness && risetime * 100 / FinalBrightness ) {
						if( FinalBrightness == 100 ) {
							if( sequence_append_ramp(&sq[i], i, 1, risetime) )
								sequence_append_delay(&sq[i], risetime);
							else
								sequence_append_long_ramp(&sq[i], i, TroughBrightness*255/100, FinalBrightness*255/100, risetime);
						} else {
							if( sequence_append_ramp(&sq[i], i, 1, risetime * 100 / FinalBrightness) )
								sequence_append_delay(&sq[i], risetime);
							else
								sequence_append_long_ramp(&sq[i], i, TroughBrightness*255/100, FinalBrightness*255/100, risetime);
							sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
						}
					} else {
						sequence_append_command(&sq[i], (1<<13)|(i<<8)|(FinalBrightness*255/100));
						sequence_append_delay(&sq[i], risetime);
					}
					break;
				}
			}
			if( EndDelay )
				sequence_append_delay(&sq[i], EndDelay);
		}
	}

	bool bRedrawRest = false;

	UndoBegin();
	for( int i = 0; i < 32; ++i ) {
		if( selected_lights&(1<<i) ) {
			if( bMerge || bMix ) {
				sequence temp;
				memset(&temp, 0, sizeof(temp));
				sequence_state ss, orig_ss;
				sequence_init(&sequences[i], &orig_ss);
				sequence_init(&sq[i], &ss);
				sequence_advance(&sequences[i], &orig_ss, start_ms, true);
				if( bMerge )
					sequence_merge(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				else
					sequence_mix(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &temp, true) )
					bRedrawRest = true;
				sequence_destroy(&temp);
			} else {
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &sq[i], true) )
					bRedrawRest = true;
			}
		}
		sequence_destroy(&sq[i]);
	}
	UndoEnd(hWnd);

	if( bRedrawRest ) {
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
		finish = (rcWAV.right - rcWAV.left + si.nPos) << zoom;
	}
	InvalidateLightRange(GetParent(hWnd), start, finish, selected_lights);
	MarkUndo();

	bDidCustomRamp = true;
	return true;
}
void FinaliseCustomRamp(HWND hWnd) {
	UnmarkUndo();
	bDidCustomRamp = false;
}
void CancelCustomRamp(HWND hWnd) {
	if( bDidCustomRamp )
		UndoToMark(GetParent(hWnd));
	bDidCustomRamp = false;
}


bool bDidBeatDetection, bDidBeatDetectionPreview;
int BeatDetectionPreviewStart, BeatDetectionPreviewFinish;
bool BeatDetection(HWND hWnd, int start, int finish, int Sensitivity, int MinInterval, int Delay, CascadeEffect Effect, int Brightness,
	                          int OnTime, int RiseTime, int FallTime, CascadeOrder Order, unsigned long Lights, bool bMerge, bool bMix, bool bPreview) {
	if( !wav_content ) {
		MessageBox(hWnd, L"There is no audio data to analyse, this is a standalone sequence.", L"No Data to Analyse", MB_ICONEXCLAMATION|MB_OK);
		return false;
	}
	if( finish <= start )
		return false;

	if( bPreview ) {
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
		int view_left = si.nPos << zoom;
		int view_right = (si.nPos + (rcWAV.right-rcWAV.left)) << zoom;
		if( start < view_right && finish > view_left && view_right - view_left < (finish - start) / 2 ) {
			if( start < view_left )
				start = view_left;
			if( finish > view_right )
				finish = view_right;
			BeatDetectionPreviewStart = start;
			BeatDetectionPreviewFinish = finish;
		} else {
			bPreview = false;
		}
	}

	StartProgress(hWnd, L"beat detection");
	ProgressSubStep(hWnd, 10, 0, 9);

	int start_ms = (int)((start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
	int finish_ms = (int)((finish + 1) * 1000LL / wav_sample_rate);

	if( bDidBeatDetection )
		Undo(GetParent(hWnd), false);

	BandpassFilter NotchFilterLeft(wav_sample_rate, 400, 4000, 4, 1.0, Notch), NotchFilterRight(wav_sample_rate, 200, 4000, 4, 1.0, Notch);
	ExponentialDecay LongDecayLeft(wav_sample_rate, 0.05), LongDecayRight(wav_sample_rate, 0.05);
	ExponentialDecay ShortDecayLeft(wav_sample_rate, 0.03), ShortDecayRight(wav_sample_rate, 0.03);

	wav_data* pData = wav_content + start;
	int BeatTimer = 0;
	bool bAbove, bLastAbove = true;
	int last_event_ms = start_ms;
	int* TriggerPoints = (int*)NULL;
	int NumTriggerPoints = 0;
	bool bIgnoreFirst = true;
	int UpdateTimer = 1;
	for( int pos = start; pos < finish; ++pos ) {
		double lfLeft = NotchFilterLeft.Process(pData[0].left/32768.0), lfRight = NotchFilterRight.Process(pData[0].left/32768.0);
		double lfLongLeft  =  LongDecayLeft.Process(lfLeft),  lfLongRight =  LongDecayRight.Process(lfRight);
		double lfShortLeft = ShortDecayLeft.Process(lfLeft), lfShortRight = ShortDecayRight.Process(lfRight);
		++pData;

		bAbove = lfShortLeft * (Sensitivity + (bLastAbove ? 1 : 0)) / 20.0 > lfLongLeft || lfShortRight * (Sensitivity + (bLastAbove ? 1 : 0)) / 20.0 > lfLongRight ||
			     lfShortLeft + (Sensitivity - 60) / 10000.0 > lfLongLeft || lfShortRight + (Sensitivity - 60) / 10000.0 > lfLongRight;
		if( bAbove && !bLastAbove && !BeatTimer ) {
			if( !bIgnoreFirst ) {
				int pos_ms = (int)((pos + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate) + Delay;
				if( pos_ms >= 0 ) {
					BeatTimer = MinInterval * (long long)wav_sample_rate / 1000;
					if( !(NumTriggerPoints&255) )
						TriggerPoints = (int*)realloc(TriggerPoints, (NumTriggerPoints+256)*sizeof(int));
					TriggerPoints[NumTriggerPoints++] = pos_ms;
				}
			}
		} else {
			bIgnoreFirst = false;
		}
		bLastAbove = bAbove;
		if( BeatTimer )
			--BeatTimer;
		if( --UpdateTimer == 0 ) {
			if( !UpdateProgress(hWnd, pos-start, finish-start) ) {
				free(TriggerPoints);
				FinishProgress(hWnd);
				if( bDidBeatDetection )
					InvalidateLastUndo(GetParent(hWnd));
				bDidBeatDetection = false;
				return true;
			}
			UpdateTimer = 10000;
		}
	}

	if( !(NumTriggerPoints&255) )
		TriggerPoints = (int*)realloc(TriggerPoints, (NumTriggerPoints+1)*sizeof(int));
	TriggerPoints[NumTriggerPoints] = -1;
	UndoBegin();
	ProgressSubStep(hWnd, 10, 9, 1);
	Cascade(hWnd, start, finish, Order, Effect, Brightness, 0, OnTime, RiseTime, FallTime, Lights, bMerge, bMix, TriggerPoints);
	UndoEnd(hWnd);
	free(TriggerPoints);
	if( !FinishProgress(hWnd) ) {
		Undo(GetParent(hWnd));
	} else {
		MarkUndo();
		bDidBeatDetection = true;
	}

	bDidBeatDetectionPreview = bPreview;
	return true;
}
void FinaliseBeatDetection(HWND hWnd, int start, int finish, int Sensitivity, int MinInterval, int Delay, CascadeEffect Effect, int Brightness, int OnTime, int RiseTime, int FallTime, CascadeOrder Order, unsigned long Lights, bool bMerge, bool bMix) {
	if( bDidBeatDetectionPreview ) {
		UndoToMark(GetParent(hWnd), false);
		if( !BeatDetection(hWnd, start, finish, Sensitivity, MinInterval, Delay, Effect, Brightness, OnTime, RiseTime, FallTime, Order, Lights, bMerge, bMix, false) )
			InvalidateLastUndo(GetParent(hWnd));
	}
	UnmarkUndo();
	bDidBeatDetection = false;
}
void CancelBeatDetection(HWND hWnd) {
	if( bDidBeatDetection )
		UndoToMark(GetParent(hWnd));
	bDidBeatDetection = false;
}


static void MergeSequences(HWND hWnd, sequence* sq, int start, int finish, int start_ms, int finish_ms, unsigned long Lights, bool bMerge, bool bMix) {
	bool bRedrawRest = false;

	UndoBegin();
	for( int i = 0; i < 32; ++i ) {
		if( Lights&(1<<i) ) {
			if( bMerge || bMix ) {
				sequence temp;
				memset(&temp, 0, sizeof(temp));
				sequence_state ss, orig_ss;
				sequence_init(&sequences[i], &orig_ss);
				sequence_init(&sq[i], &ss);
				sequence_advance(&sequences[i], &orig_ss, start_ms, true);
				if( bMerge )
					sequence_merge(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				else
					sequence_mix(&temp, &sequences[i], &orig_ss, &sq[i], &ss, i, finish_ms - start_ms);
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &temp, true) )
					bRedrawRest = true;
				sequence_destroy(&temp);
			} else {
				if( SetLightBrightness(GetParent(hWnd), i, start, finish, SETBRIGHTNESS_INSERT, &sq[i], true) )
					bRedrawRest = true;
			}
		}
		sequence_destroy(&sq[i]);
	}
	UndoEnd(hWnd);

	if( bRedrawRest ) {
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
		finish = (rcWAV.right - rcWAV.left + si.nPos) << zoom;
	}
	InvalidateLightRange(GetParent(hWnd), start, finish, Lights);
}

struct FilterPair {
	FilterPair(int sample_rate) : LongFilter(sample_rate, 0.05), ShortFilter(sample_rate, 0.03), TriggerPoints(NULL), TriggerPointScale(NULL), NumTriggerPoints(0), BeatTimer(0), ScaleTimer(0), bLastAbove(true), bIgnoreFirst(true) { }
	~FilterPair() { free(TriggerPoints); free(TriggerPointScale); }
	ExponentialDecay LongFilter, ShortFilter;
	int* TriggerPoints;
	SpectrumScaleFactors* TriggerPointScale;
	int NumTriggerPoints, BeatTimer, ScaleTimer;
	bool bLastAbove, bIgnoreFirst;
	double lfScaleMax;
};

bool bDidSpectrumAnalysis, bDidSpectrumAnalysisPreview;
int SpectrumAnalysisPreviewStart, SpectrumAnalysisPreviewFinish;
bool SpectrumAnalysis(HWND hWnd, int start, int finish, double lfLowerCornerFrequency, double lfUpperCornerFrequency, int NumBands, FrequencySpacing Spacing,
	                             SpectrumStereoType Stereo, CascadeOrder Order, CascadeEffect Effect, SpectrumScaleType ScaleType,
								 int Brightness, int Sensitivity, int UpdateIntervalMilliseconds,
								 int OnTime, int RiseTime, int FallTime, int Delay, int MinInterval, unsigned long Lights, bool bMerge, bool bMix, bool bPreview) {
	if( !wav_content ) {
		MessageBox(hWnd, L"There is no audio data to analyse, this is a standalone sequence.", L"No Data to Analyse", MB_ICONEXCLAMATION|MB_OK);
		return false;
	}

	if( lfLowerCornerFrequency >= lfUpperCornerFrequency || NumBands < 1 )
		return false;

	if( wav_num_channels == 1 )
		Stereo = OneChannelOnly;

	if( bPreview ) {
		SCROLLINFO si;
		si.cbSize = sizeof (si);
		si.fMask  = SIF_ALL;
		GetScrollInfo (GetParent(hWnd), SB_HORZ, &si);
		int view_left = si.nPos << zoom;
		int view_right = (si.nPos + (rcWAV.right-rcWAV.left)) << zoom;
		if( start < view_right && finish > view_left && view_right - view_left < (finish - start) / 2 ) {
			if( start < view_left )
				start = view_left;
			if( finish > view_right )
				finish = view_right;
			SpectrumAnalysisPreviewStart = start;
			SpectrumAnalysisPreviewFinish = finish;
		} else {
			bPreview = false;
		}
	}

	StartProgress(hWnd, L"spectrum analysis");
	if( Effect == Flash || Effect == Fade )
		ProgressSubStep(hWnd, 50+NumBands, 0, 50);

	int start_ms = (int)((start + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate);
	int finish_ms = (int)((finish + 1) * 1000LL / wav_sample_rate);

	if( bDidSpectrumAnalysis )
		Undo(GetParent(hWnd), false);

	sequence sq[32];
	memset(&sq, 0, sizeof(sq));

	double lfAveragePeriod = 0.001;
	AveragingMultiBandFilter* pFilt1, * pFilt2;
	FilterPair* pFilters;
	if( Stereo == LeftToRight || Stereo == RightToLeft ) {
		pFilt1 = new AveragingMultiBandFilter(wav_sample_rate, lfLowerCornerFrequency, lfUpperCornerFrequency, 4, 1.0, NumBands / 2, Spacing, lfAveragePeriod*5);
		pFilt2 = new AveragingMultiBandFilter(wav_sample_rate, lfLowerCornerFrequency, lfUpperCornerFrequency, 4, 1.0, NumBands - NumBands / 2, Spacing, lfAveragePeriod*5);
	} else {
		pFilt1 = new AveragingMultiBandFilter(wav_sample_rate, lfLowerCornerFrequency, lfUpperCornerFrequency, 4, 1.0, NumBands, Spacing, lfAveragePeriod*5);
		pFilt2 = (AveragingMultiBandFilter*)NULL;
	}
	if( Effect == Flash || Effect == Fade ) {
		pFilters = (FilterPair*)malloc(NumBands * sizeof(FilterPair));
		for( int i = 0; i < NumBands; ++i )
			new (pFilters+i) FilterPair(wav_sample_rate);
	} else {
		Brightness = Brightness * 255 / 100;
	}

	int pos = start - (int)(lfAveragePeriod * wav_sample_rate + 1.5);
	if( pos < 0 )
		pos = 0;
	wav_data* pData = wav_content + pos;
	switch(Stereo) {
	case LeftToRight:
		while( pos < start ) {
			pFilt1->Process( pData[0].left / 32768.0 );
			pFilt2->Process( pData[0].right / 32768.0 );
			++pos;
			++pData;
		}
		break;
	case RightToLeft:
		while( pos < start ) {
			pFilt1->Process( pData[0].right / 32768.0 );
			pFilt2->Process( pData[0].left / 32768.0 );
			++pos;
			++pData;
		}
		break;
	case LeftOnly:
		while( pos < start ) {
			pFilt1->Process( pData[0].left / 32768.0 );
			++pos;
			++pData;
		}
		break;
	case RightOnly:
		while( pos < start ) {
			pFilt1->Process( pData[0].right / 32768.0 );
			++pos;
			++pData;
		}
		break;
	case Mono:
		while( pos < start ) {
			pFilt1->Process( (pData[0].left + pData[0].right) / 65536.0 );
			++pos;
			++pData;
		}
		break;
	default:
		while( pos < start ) {
			pFilt1->Process( pData[0].left / 32768.0 );
			++pos;
			pData = (wav_data*)( (short*)pData + 1 );
		}
		break;
	}

	int LastBright[32];
	for( int i = 0; i < 32; ++i )
		LastBright[i] = -1;

	int BandOrder[32];
	if( Order == LowToHigh ) {
		int pos = 0;
		for( int i = 0; i < 32; ++i )
			if( Lights&(1<<i) )
				BandOrder[pos++] = i;
	} else if( Order == HighToLow ) {
		if( Stereo == LeftToRight || Stereo == RightToLeft ) {
			int pos = NumBands/2;
			for( int i = 0; i < 32; ++i ) {
				if( Lights&(1<<i) ) {
					BandOrder[--pos] = i;
					if( pos == 0 )
						pos = NumBands;
				}
			}
		} else {
			int pos = 0;
			for( int i = 31; i >= 0; --i )
				if( Lights&(1<<i) )
					BandOrder[pos++] = i;
		}
	} else {
		int Random[32];
		int pos = 0, pos2 = 0;
		srand(start_ms);
		for( int i = 0; i < 32; ++i )
			if( Lights&(1<<i) )
				Random[pos++] = i;
		pos = 0;
		int left = NumBands;
		if( Stereo == LeftToRight || Stereo == RightToLeft )
			left /= 2;
		for( int i = 0; i < NumBands; ++i ) {
			int num = rand()%left;
			BandOrder[pos++] = Random[num];
			memmove(Random + num, Random + num + 1, (left - num - 1) * sizeof(*Random));
			if( --left == 0 ) {
				memcpy(Random, Random + (NumBands/2), (NumBands - NumBands/2) * sizeof(*Random));
				left = NumBands - NumBands/2;
			}
		}
	}

	if( Effect == Flash || Effect == Fade ) {
		int UpdateTimer = 1;
		while( pos < finish ) {
			switch(Stereo) {
			case LeftToRight:
				pFilt1->Process( pData[0].left / 32768.0 );
				pFilt2->Process( pData[0].right / 32768.0 );
				++pos;
				++pData;
				break;
			case RightToLeft:
				pFilt1->Process( pData[0].right / 32768.0 );
				pFilt2->Process( pData[0].left / 32768.0 );
				++pos;
				++pData;
				break;
			case LeftOnly:
				pFilt1->Process( pData[0].left / 32768.0 );
				++pos;
				++pData;
				break;
			case RightOnly:
				pFilt1->Process( pData[0].right / 32768.0 );
				++pos;
				++pData;
				break;
			case Mono:
				pFilt1->Process( (pData[0].left + pData[0].right) / 65536.0 );
				++pos;
				++pData;
				break;
			default:
				pFilt1->Process( pData[0].left / 32768.0 );
				++pos;
				pData = (wav_data*)( (short*)pData + 1 );
				break;
			}
			if( --UpdateTimer == 0 ) {
				if( !UpdateProgress(hWnd, pos-start, finish-start) ) {
					delete pFilt1;
					delete pFilt2;
					for( int i = 0; i < NumBands; ++i )
						pFilters[i].~FilterPair();
					free(pFilters);
					FinishProgress(hWnd);
					if( bDidSpectrumAnalysis )
						InvalidateLastUndo(GetParent(hWnd));
					bDidSpectrumAnalysis = false;
					return true;
				}
				UpdateTimer = 10000;
			}

			for( int i = 0; i < NumBands; ++i ) {
				double lfSample;
				if( (Stereo == LeftToRight || Stereo == RightToLeft) && i >= NumBands/2 )
					lfSample = pFilt2->GetBandSample(i - NumBands/2);
				else
					lfSample = pFilt1->GetBandSample(i);
				FilterPair* pFilter = &pFilters[i];
				double lfShort = pFilter->ShortFilter.Process(lfSample);
				double lfLong = pFilter->LongFilter.Process(lfSample);

				bool bAbove = lfShort * (Sensitivity + (pFilter->bLastAbove ? 1 : 0)) / 20.0 > lfLong || lfShort + (Sensitivity - 60) / 10000.0 > lfLong;
				if( bAbove && !pFilter->bLastAbove && !pFilter->BeatTimer ) {
					if( !pFilter->bIgnoreFirst ) {
						pFilter->BeatTimer = MinInterval * (long long)wav_sample_rate / 1000;
						if( !(pFilter->NumTriggerPoints&255) ) {
							pFilter->TriggerPoints = (int*)realloc(pFilter->TriggerPoints, (pFilter->NumTriggerPoints+256)*sizeof(int));
							if( ScaleType != ScaleNeither )
								pFilter->TriggerPointScale = (SpectrumScaleFactors*)realloc(pFilter->TriggerPointScale, (pFilter->NumTriggerPoints+256)*sizeof(SpectrumScaleFactors));
						}
						int ms_pos = (int)((pos + ((wav_sample_rate+999) / 1000 - 1)) * 1000LL / wav_sample_rate) + Delay;
						if( ms_pos >= 0 ) {
							pFilter->TriggerPoints[pFilter->NumTriggerPoints] = ms_pos;
							if( ScaleType != ScaleNeither ) {
								pFilter->ScaleTimer = wav_sample_rate * Sensitivity / 5000;
								pFilter->lfScaleMax = 0;
								pFilter->TriggerPointScale[pFilter->NumTriggerPoints].Brightness = 255;
								pFilter->TriggerPointScale[pFilter->NumTriggerPoints].Duration = 255;
							}
							++pFilter->NumTriggerPoints;
						}
						if( !UpdateProgress(hWnd, pos-start, finish-start) ) {
							delete pFilt1;
							delete pFilt2;
							for( int i = 0; i < NumBands; ++i )
								pFilters[i].~FilterPair();
							free(pFilters);
							FinishProgress(hWnd);
							if( bDidSpectrumAnalysis )
								InvalidateLastUndo(GetParent(hWnd));
							bDidSpectrumAnalysis = false;
							return true;
						}
					}
				} else {
					pFilter->bIgnoreFirst = false;
				}
				if( pFilter->BeatTimer )
					--pFilter->BeatTimer;
				if( pFilter->ScaleTimer ) {
					double lfDiff = lfShort * (Sensitivity + (pFilter->bLastAbove ? 1 : 0)) / 20.0 - lfLong;
					if( lfDiff > pFilter->lfScaleMax )
						pFilter->lfScaleMax = lfDiff;
					if( --pFilter->ScaleTimer == 0 ) {
						int Scale = (int)(sqrt(pFilter->lfScaleMax) * 20000);

						if( Scale > 255 )
							Scale = 255;
						else if( Scale < 1 )
							Scale = 1;
						switch(ScaleType) {
						case ScaleBrightness:
							pFilter->TriggerPointScale[pFilter->NumTriggerPoints-1].Brightness = Scale;
							break;
						case ScaleDuration:
							pFilter->TriggerPointScale[pFilter->NumTriggerPoints-1].Duration = Scale;
							break;
						case ScaleBoth:
							pFilter->TriggerPointScale[pFilter->NumTriggerPoints-1].Brightness = Scale;
							pFilter->TriggerPointScale[pFilter->NumTriggerPoints-1].Duration = Scale;
							break;
						case ScaleNeither:
							break;
						}
					}
				}
				pFilter->bLastAbove = bAbove;
			}
		}
	} else {
		int ms_pos = start_ms;
		while( pos < finish && ms_pos < finish_ms ) {
			int next_ms = ms_pos + UpdateIntervalMilliseconds;
			if( next_ms > finish_ms )
				next_ms = finish_ms;

			for( int i = 0; i < NumBands; ++i ) {
				int Index = BandOrder[i];
				int Bright;
				switch(Stereo) {
				case Mono:
				case LeftOnly:
				case RightOnly:
					Bright = (int)(pFilt1->GetAverageEnergy(i) * Brightness * 2 + 0.5);
					break;
				default:
					if( i < NumBands / 2 )
						Bright = (int)(pFilt1->GetAverageEnergy(i) * Brightness * 2 + 0.5);
					else
						Bright = (int)(pFilt2->GetAverageEnergy(i - NumBands / 2) * Brightness * 2 + 0.5);
					break;
				}
				if( Bright > 255 )
					Bright = 255;
				if( abs(Bright - LastBright[Index]) > (40 - Sensitivity) ) {
					sequence_append_command(&sq[Index], (1<<13)|(i<<8)|Bright);
					LastBright[Index] = Bright;
				}
				sequence_append_delay(&sq[Index], next_ms - ms_pos);
			}
			int next_pos = ((long long)next_ms * (long long)wav_sample_rate) / 1000;
			switch(Stereo) {
			case LeftToRight:
				while( pos < next_pos ) {
					pFilt1->Process( pData[0].left / 32768.0 );
					pFilt2->Process( pData[0].right / 32768.0 );
					++pos;
					++pData;
				}
				break;
			case RightToLeft:
				while( pos < next_pos ) {
					pFilt1->Process( pData[0].right / 32768.0 );
					pFilt2->Process( pData[0].left / 32768.0 );
					++pos;
					++pData;
				}
				break;
			case LeftOnly:
				while( pos < next_pos ) {
					pFilt1->Process( pData[0].left / 32768.0 );
					++pos;
					++pData;
				}
				break;
			case RightOnly:
				while( pos < next_pos ) {
					pFilt1->Process( pData[0].right / 32768.0 );
					++pos;
					++pData;
				}
				break;
			case Mono:
				while( pos < next_pos ) {
					pFilt1->Process( (pData[0].left + pData[0].right) / 65536.0 );
					++pos;
					++pData;
				}
				break;
			default:
				while( pos < next_pos ) {
					pFilt1->Process( pData[0].left / 32768.0 );
					++pos;
					pData = (wav_data*)( (short*)pData + 1 );
				}
				break;
			}
			ms_pos = next_ms;

			if( !UpdateProgress(hWnd, pos-start, finish-start) ) {
				delete pFilt1;
				delete pFilt2;
				for( int i = 0; i < 32; ++i )
					sequence_destroy(&sq[i]);
				FinishProgress(hWnd);
				if( bDidSpectrumAnalysis )
					InvalidateLastUndo(GetParent(hWnd));
				bDidSpectrumAnalysis = false;
				return true;
			}
		}
	}

	if( Effect == Flash || Effect == Fade ) {
		UndoBegin();
		for( int i = 0; i < NumBands; ++i ) {
			FilterPair* pFilter = &pFilters[i];
			if( !(pFilter->NumTriggerPoints&255) )
				pFilter->TriggerPoints = (int*)realloc(pFilter->TriggerPoints, (pFilter->NumTriggerPoints+1)*sizeof(int));
			pFilter->TriggerPoints[pFilter->NumTriggerPoints] = -1;
			ProgressSubStep(hWnd, 50+NumBands, 50+i, 1);
			Cascade(hWnd, start, finish, LowToHigh, Effect, Brightness, 0, OnTime, RiseTime, FallTime, 1<<BandOrder[i], bMerge, bMix, pFilter->TriggerPoints, pFilter->TriggerPointScale);
		}
		UndoEnd(hWnd);
	} else {
		MergeSequences(hWnd, sq, start, finish, start_ms, finish_ms, Lights, bMerge, bMix);
	}

	delete pFilt1;
	delete pFilt2;
	if( Effect == Flash || Effect == Fade ) {
		for( int i = 0; i < NumBands; ++i )
			pFilters[i].~FilterPair();
		free(pFilters);
	}

	if( !FinishProgress(hWnd) ) {
		Undo(GetParent(hWnd));
	} else {
		MarkUndo();
		bDidSpectrumAnalysis = true;
	}

	bDidSpectrumAnalysisPreview = bPreview;
	return true;
}
void FinaliseSpectrumAnalysis(HWND hWnd, int start, int finish, double lfLowerCornerFrequency, double lfUpperCornerFrequency, int NumBands, FrequencySpacing Spacing, SpectrumStereoType Stereo, CascadeOrder Order, CascadeEffect Effect, SpectrumScaleType ScaleType, int Brightness, int Sensitivity, int UpdateIntervalMilliseconds, int OnTime, int RiseTime, int FallTime, int Delay, int MinInterval, unsigned long Lights, bool bMerge, bool bMix) {
	if( bDidSpectrumAnalysisPreview ) {
		UndoToMark(GetParent(hWnd), false);
		if( !SpectrumAnalysis(hWnd, start, finish, lfLowerCornerFrequency, lfUpperCornerFrequency, NumBands, Spacing, Stereo, Order, Effect, ScaleType, Brightness, Sensitivity, UpdateIntervalMilliseconds, OnTime, RiseTime, FallTime, Delay, MinInterval, Lights, bMerge, bMix, false) )
			InvalidateLastUndo(GetParent(hWnd));
	}
	UnmarkUndo();
	bDidSpectrumAnalysis = false;
}
void CancelSpectrumAnalysis(HWND hWnd) {
	if( bDidSpectrumAnalysis )
		UndoToMark(GetParent(hWnd));
	bDidSpectrumAnalysis = false;
}

void SetSequenceLength(HWND hWnd, int new_length_ms) {
	int old_length = wav_len;
	int new_length = new_length_ms * 12;
	UndoBegin();
	for( int i = 0; i < 32; ++i ) {
		UndoPrepare(i);
		sequence_truncate(&sequences[i], new_length_ms, new_length_ms);
		int start = min(old_length, new_length);
		int finish = max(old_length, new_length);
		UndoFinish(i, start, finish);
		MainSequencesChanged(hWnd, start, finish);
	}
	UndoChangingLength(old_length);
	UndoEnd(hWnd);
	wav_len = new_length;
	InvalidateSelectionRange(hWnd, min(old_length, new_length), max(old_length, new_length), 0xFFFFFFFF);
	if( selection_start > new_length || selection_finish > new_length )
		SetSelection(hWnd, min(selection_start, new_length), min(selection_finish, new_length));
}
