
//// change these to suit the application ////

#define CRYSTAL_HZ     24576000
#define SEQ_SPI        1
#define SEQ_DMA		   3
#define SEQ_CLK_RP     4
#define SEQ_CLK_PORT   B
#define SEQ_CLK_PIN    4
#define SEQ_DOUT_RP    5
#define SEQ_DOUT_PORT  B
#define SEQ_DOUT_PIN   5
#define SEQ_SS_RP      7
#define SEQ_SS_PORT    B
#define SEQ_SS_PIN     7

//////////////////////////////////////////////

#include "p33Fxxxx.h"
#include <string.h>

#define _SPISTATname(spi) SPI##spi##STAT
#define SPISTATname(spi)  _SPISTATname(spi)
#define _SPISTATbits(spi) SPI##spi##STATbits
#define SPISTATbits(spi)  _SPISTATbits(spi)
#define _SPICON1name(spi) SPI##spi##CON1
#define SPICON1name(spi)  _SPICON1name(spi)
#define _SPICON1bits(spi) SPI##spi##CON1bits
#define SPICON1bits(spi)  _SPICON1bits(spi)
#define _SPICON2name(spi) SPI##spi##CON2
#define SPICON2name(spi)  _SPICON2name(spi)
#define _SPIBUFname(spi)  SPI##spi##BUF
#define SPIBUFname(spi)   _SPIBUFname(spi)
#define _SPIIEname(spi)   _SPI##spi##IE
#define SPIIEname(spi)    _SPIIEname(spi)
#define _SPIIFname(spi)   _SPI##spi##IF
#define SPIIFname(spi)    _SPIIFname(spi)
#define _SPIIntname(spi)  _SPI##spi##Interrupt
#define SPIIntname(spi)   _SPIIntname(spi)
#define _RPORbits(a,b)    RPOR##a##bits.RP##b##R
#define RPORbits(a,b)     _RPORbits(a,b)
#define _TRISbits(a,b)    TRIS##a##bits.TRIS##a##b
#define TRISbits(a,b)     _TRISbits(a,b)
#define _LATbits(a,b)     LAT##a##bits.LAT##a##b
#define LATbits(a,b)      _LATbits(a,b)


volatile unsigned short mains_cycles;
unsigned char mains_frequency;
unsigned short zero_crossing_offset;

unsigned char phase_CLKDIV_cache, phase_PLLFBD_cache;
unsigned short timer_phase_length;

extern unsigned char lights[32];
extern unsigned char controller_phases[4];
unsigned char dc_slaves[4];
unsigned short filament_preheat_off[2], triac_turnoff_delayed[2];
unsigned char filament_preheat_amount;
unsigned char light_on_time[32], light_off_time[32];
signed char light_dither[32];
unsigned char cur_timing;

#define NUM_SLICES 21
static const unsigned char sine_area[19] = { 255, 251, 244, 234, 222, 208, 192, 175, 156, 137, 118, 99, 80, 63, 47, 33, 21, 11, 4 };
static const unsigned char triangle_area[19] = { 255, 242, 228, 215, 201, 188, 174, 161, 148, 134, 121, 107, 94, 81, 67, 54, 40, 27, 13 };
static const unsigned char sine_table[256] = { 255, 255, 19, 19, 19, 19, 19, 19, 18, 18, 18, 18, 18, 18, 18, 18, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 1, 1, 1 };
static const unsigned char triangle_table[256] = { 255, 255, 255, 255, 255, 255, 255, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 12, 12,12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1 };

void calc_mains_timing() {
	unsigned long timer_hz = (unsigned long)CRYSTAL_HZ / (CLKDIVbits.PLLPRE+2) * (PLLFBD+2) / ((CLKDIVbits.PLLPOST+1)*4*8); // 8 = timer divided by eight
	timer_phase_length = timer_hz / (mains_frequency*2);

	phase_CLKDIV_cache = CLKDIV;
	phase_PLLFBD_cache = PLLFBD;
}

static void setup_light_spi() {
	SPISTATname(SEQ_SPI)=0;
	SPICON1name(SEQ_SPI)=0;
	SPICON2name(SEQ_SPI)=0;
	// set up SPI pins
#if SEQ_SPI == 1
	#define SD_CARD_SPI_DOUT_PIN 7
	#define SD_CARD_SPI_CLK_PIN 8
	#define SD_CARD_SPI_SS_PIN 9
#else
	#define SD_CARD_SPI_DOUT_PIN 10
	#define SD_CARD_SPI_CLK_PIN 11
	#define SD_CARD_SPI_SS_PIN 12
#endif
#if SEQ_CLK_RP < 2
	RPORbits(0,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 4
	RPORbits(1,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 6
	RPORbits(2,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 8
	RPORbits(3,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 10
	RPORbits(4,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 12
	RPORbits(5,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 14
	RPORbits(6,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 16
	RPORbits(7,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 18
	RPORbits(8,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 20
	RPORbits(9,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 22
	RPORbits(10,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#elif SEQ_CLK_RP < 24
	RPORbits(11,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#else
	RPORbits(12,SEQ_CLK_RP) = SD_CARD_SPI_CLK_PIN;
#endif
#if SEQ_DOUT_RP < 2
	RPORbits(0,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 4
	RPORbits(1,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 6
	RPORbits(2,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 8
	RPORbits(3,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 10
	RPORbits(4,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 12
	RPORbits(5,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 14
	RPORbits(6,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 16
	RPORbits(7,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 18
	RPORbits(8,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 20
	RPORbits(9,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 22
	RPORbits(10,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#elif SEQ_DOUT_RP < 24
	RPORbits(11,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#else
	RPORbits(12,SEQ_DOUT_RP) = SD_CARD_SPI_DOUT_PIN;
#endif
	TRISbits(SEQ_DOUT_PORT, SEQ_DOUT_PIN) = 0;
	TRISbits(SEQ_CLK_PORT, SEQ_CLK_PIN) = 0;
	LATbits(SEQ_SS_PORT, SEQ_SS_PIN) = 1;
	TRISbits(SEQ_SS_PORT, SEQ_SS_PIN) = 0;

	SPICON1bits(SEQ_SPI).SPRE = 2;		// about 100kHz
	SPICON1bits(SEQ_SPI).MODE16=1;		/* 16 bit mode */
	SPICON1bits(SEQ_SPI).CKP=0;			/* clock Polarity 0: idle low, active high 1: idle high, active low */
	SPICON1bits(SEQ_SPI).CKE=1;			/* clock edge selection */
	SPICON1bits(SEQ_SPI).MSTEN=1;		/* enable Master Mode */
	SPISTATbits(SEQ_SPI).SPIROV=0;
	SPIIFname(SEQ_SPI)=0;
	SPIIEname(SEQ_SPI)=1;
	SPISTATbits(SEQ_SPI).SPIEN=1;
}

unsigned char num_control_spi_words;
void set_control_spi_words(unsigned char num_words) {
	SPICON1bits(SEQ_SPI).SPRE = (num_words > 1 ? 2 : 0);
	num_control_spi_words = num_words;
}

unsigned char spi_send_byte;
unsigned short next_spi_word;

void __attribute__((__interrupt__,no_auto_psv)) SPIIntname(SEQ_SPI)( void ) {
	SPIIFname(SEQ_SPI)=0;
	if( spi_send_byte == 0 ) {
		spi_send_byte = 1;
		SPISTATbits(SEQ_SPI).SPIROV=0;
		SPIBUFname(SEQ_SPI) = next_spi_word;
	} else {
		unsigned char i;
		for( i = 0; i < 8; ++i )
			Nop();
		LATbits(SEQ_SS_PORT, SEQ_SS_PIN) = 1;
	}
}

static inline void send_light_command(unsigned short* command) {
	SPISTATbits(SEQ_SPI).SPIROV=0;
	LATbits(SEQ_SS_PORT, SEQ_SS_PIN) = 0;
	if( num_control_spi_words == 1 ) {
		spi_send_byte = 1;
		SPIBUFname(SEQ_SPI) = command[0];
	} else {
		spi_send_byte = 0;
		next_spi_word = command[0];
		SPIBUFname(SEQ_SPI) = command[1];
	}
}

unsigned short light_commands[2];

static void update_lights() {
	unsigned char i, num = num_control_spi_words * 16, next_earliest = 255;
	for( i = 0; i < num; ++i ) {
		if( light_on_time[i] == cur_timing )
			light_commands[i>>4] |=   1<<(i&15);
		else if( light_on_time[i] > cur_timing && light_on_time[i] < next_earliest )
			next_earliest = light_on_time[i];
		if( light_off_time[i] == cur_timing )
			light_commands[i>>4] &= ~(1<<(i&15));
		else if( light_off_time[i] > cur_timing && light_off_time[i] < next_earliest )
			next_earliest = light_off_time[i];
	}

	send_light_command(light_commands);

	cur_timing = next_earliest;
	IFS0bits.OC1IF = 0;
	if( cur_timing == 0xFF )
		OC1R = 65535;
	else
		OC1R = (unsigned long)timer_phase_length * cur_timing / NUM_SLICES;

	if( TMR2 >= OC1R && cur_timing == next_earliest )
		update_lights();
}

void __attribute__((__interrupt__,no_auto_psv)) _OC1Interrupt( void ) {
	IFS0bits.OC1IF = 0;
	update_lights();
}

static unsigned char get_light_timing(const unsigned char* table, unsigned char desired_brightness, signed char dither) {
	short brightness = (short)desired_brightness + (short)dither;
	if( brightness < 0 )
		brightness = 0;
	else if( brightness > 255 )
		brightness = 255;
	return table[brightness];
}

static void recalc_light_timing() {
	unsigned char i, num = num_control_spi_words * 16, preheat, brightness, on_time, off_time, earliest = 0xFF;
	unsigned short when;
	for( i = 0; i < num; ) {
		const unsigned char* area_table = dc_slaves[i>>3] ? triangle_area : sine_area;
		const unsigned char* table = dc_slaves[i>>3] ? triangle_table : sine_table;
		unsigned char next = i + 8;
		for( ; i < next; ++i ) {
			preheat = filament_preheat_off[i>>4]&(1<<(i&15)) ? 0 : filament_preheat_amount;
			brightness = lights[i];
			if( brightness < preheat )
				brightness = preheat;
			on_time = get_light_timing(table, brightness, light_dither[i]);
			if( on_time != 0xFF ) {
				light_dither[i] += brightness - area_table[on_time-1];
				on_time += controller_phases[i>>3];
				if( on_time >= NUM_SLICES )
					on_time -= NUM_SLICES;

				if( triac_turnoff_delayed[i>>4]&(1<<(i&15)) ) {
					off_time = controller_phases[i>>3] + NUM_SLICES-2;
				} else {
					off_time = on_time + 1;
				}
				if( off_time >= NUM_SLICES )
					off_time -= NUM_SLICES;
			} else {
				light_dither[i] += brightness;
				off_time = 0xFF;
			}

			light_on_time[i] = on_time;
			light_off_time[i] = off_time;

			if( on_time == NUM_SLICES-1 ) // deal with "wrap-around" due to lights on different phases
				earliest = 0;
			else if( on_time < earliest )
				earliest = on_time;
			if( off_time == NUM_SLICES-1 ) // deal with "wrap-around" due to lights on different phases
				earliest = 0;
			else if( off_time < earliest )
				earliest = off_time;
		}
	}

	cur_timing = earliest;
	IFS0bits.OC1IF = 0;

	if( cur_timing == 0 ) {
		when = 0;
	} else if( cur_timing == 0xFF ) {
		when = 65535;
	} else {
		when = (unsigned long)timer_phase_length * earliest / NUM_SLICES;
	}

	OC1R = when;
	if( TMR2 >= OC1R && cur_timing == earliest )
		update_lights();
}

void DelayMs(unsigned short ms, unsigned char* interrupt, unsigned char interrupt_cmp) {
	unsigned short start_tmr2 = TMR2;
	unsigned short start_mains_cycles = mains_cycles;
	unsigned short threems = 3 * ms;
	unsigned char cycle_period = (mains_frequency == 50 ? 30 : 25);
	while( threems >= cycle_period ) {
		while( mains_cycles == start_mains_cycles && (!interrupt || *interrupt != interrupt_cmp) )
			Nop();
		++start_mains_cycles;
		threems -= cycle_period;
	}
	if( threems ) {
		unsigned short end_tmr2 = start_tmr2 + ((unsigned long)threems * (unsigned long)timer_phase_length / cycle_period);
		if( end_tmr2 >= timer_phase_length )
			end_tmr2 -= timer_phase_length;
		if( end_tmr2 < start_tmr2 ) {
			while( mains_cycles == start_mains_cycles && (!interrupt || *interrupt != interrupt_cmp) )
				Nop();
		}
		while( TMR2 < end_tmr2 && (!interrupt || *interrupt != interrupt_cmp) )
			Nop();
	}
}

unsigned short GetRandomSeed() {
	return TMR2;
}

void __attribute__((__interrupt__,no_auto_psv)) _CMPInterrupt(void) {
	IFS1bits.CMIF = 0;
	if( TMR2 >= 10000 ) {
		++mains_cycles;
		TMR2 = 0;
		CMCONbits.C2EVT = 0;

		if( phase_CLKDIV_cache != CLKDIV || phase_PLLFBD_cache != PLLFBD )
			calc_mains_timing();

		recalc_light_timing();
	}
}

unsigned char init_mains_phase() {
	// set up timer 2 for AC phase monitoring
	T2CONbits.TCKPS = 1; // divide master clock by 8
	TMR2 = 0;
    PR2 = 65535;
	T2CONbits.TON = 1;

	// set up timer 3 for mains frequency detection
	T3CONbits.TCKPS = 3; // divide master clock by 256
	TMR3 = 0;
    PR3 = 65535;

	// set up the comparator for AC phase monitoring
	CVRCONbits.CVREN = 0;
	CVRCONbits.CVRSS = 0;
	CMCONbits.C2POS = 1;
	CMCONbits.C2NEG = 0;
	CMCONbits.C2EN = 1;
	IEC1bits.CMIE = 1;

	// clear remote latches
	AD1PCFGLbits.PCFG5 = 1;
	TRISBbits.TRISB3 = 0;
	LATBbits.LATB3 = 0;

	while(mains_cycles == 0)
		;
	T3CONbits.TON = 1;
	while(mains_cycles != 11)
		;
	T3CONbits.TON = 0;
	if( TMR3 >= 13000-400 && TMR3 <= 13000+400 )
		mains_frequency = 60;
	else if( TMR3 >= 15600-400 && TMR3 <= 15600+400 )
		mains_frequency = 50;
	else
		return 0;

	// finished clearing remote latches
	LATBbits.LATB3 = 1;

	setup_light_spi();
	calc_mains_timing();

	// set up output compare 1 for AC phase triggering
	OC1CONbits.OCM = 3;
	OC1R = 65535;
    IEC0bits.OC1IE = 1;

	{
		unsigned short start_mains_cycles = mains_cycles;
		while( mains_cycles == start_mains_cycles )
			;
	}

	return 1;
}

