
#include "p33Fxxxx.h"
#include "infrared.h"
#include "dac.h"
#include "mains_phase.h"
#include "led.h"
#include "playback.h"
#include "ir_codes.h"
#include "chainsense.h"
#include "sdcard/ff.h"
#include "Bootloader/hexfile.h"
#include <string.h>

/*
  Crystal = 24.576MHz
  PLLPRE = 6  (N1 = 8)
  PLLPOST = 0 (N2 = 2)
  PLLDIV = 48 (M = 46)
  Fosc = 36.864MHz
*/

/*
  To do:
    * Add support for DC outputs
    * Improve dithering
    * Update article to have improved IR info
    * Fix initial latch clear for both bootloader and main program.
	* Re-test 32kHz, 12/24/32/48kHz
    * Finish moving changes across from latest version
	* Enable pullup for IR receiver?

  Pin map (* = unwired):
    1  MCLR
    2  RA0/AN0  Output      Green LED cathode
    3  RA1/AN1  Input       Infrared receiver
    4  RB0/AN2  Comparator  AC phase detection
    5  RB1/AN3  Comparator  AC phase detection
    6  RB4/AN4  Analog      Sequencer Chain Length Sense
    7  RB3/AN5  Output      Sequencer Master Clear
    8  VSS
    9  XTI                  24.576MHz Crystal
    10 XTO                  24.576MHz Crystal
    11 RB4      Output      Sequencer Serial Clock
    12 RA4      Input       SD Card Detect
    13 VDD
    14 RB5      Output      Sequencer Serial Data
    15 RB6      Output      SD Card Chip Select
    16 RB7      Output      Sequencer Latch/CS
    17 RB8      Output      SD Card Data Out
    18 RB9      -           Tied to VDD
    19 VSS
    20 VCAP
    21 RB10     Output      SD Card Clock
    22 RB11     Input       SD Card Data In
    23 DAC1RP   DAC
    24 DAC1RN   DAC
    25 DAC1LP   DAC
    26 DAC1LN   DAC
    27 AVSS
    28 AVDD
*/

// Device configuration registers
_FGS(GWRP_OFF & GCP_OFF); // we don't want to write protect the flash
_FOSCSEL(FNOSC_FRC); // start up initially without the PLL but instead using the internal fast RC oscillator
_FOSC(FCKSM_CSECMD & OSCIOFNC_OFF & POSCMD_HS);
_FWDT(FWDTEN_OFF); // not using the watchdog timer

#define HEX_FILE_NAME "Christmas Light Controller.hex"

extern unsigned char memoryCardSystemUp;
unsigned char dont_start_playback_automatically, infrared_logging, infrared_log_open;
unsigned short ir_log_update_mains_cycles;
FIL ir_log;

static void printushortdec(char* dest, unsigned short num) {
	unsigned char i;
	dest += 4;
	for( i = 5; i > 0; --i ) {
		dest[0] = num ? '0'+(num%10) : ' ';
		--dest;
		num /= 10;
	}
}

static void printushorthex(char* dest, unsigned short num) {
	unsigned char i;
	dest += 3;
	for( i = 4; i > 0; --i ) {
		unsigned char dig = num&15;
		dest[0] = dig < 10 ? '0'+dig : 'A'+dig-10;
		--dest;
		num >>= 4;
	}
}

static void rapid_flash(unsigned short delay) {
    led_on();
	DelayMs(delay, 0, 0);
    led_off();
	DelayMs(delay, 0, 0);
}
static void check_mains_presence() {
	unsigned short mains_cycles_start = mains_cycles;
	while( mains_cycles < mains_cycles_start + 10 )
		Nop();
}
extern unsigned long* flash_row_buffer;
void check_reflash_bootloader() {
	FIL hex_file;
	hex_result result;
	unsigned long stack_flash_row_buffer[64];

	if( f_open(&hex_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK )
		return;
	result = check_hex_file(&hex_file);
	if( result == difference ) {
		unsigned char i;
		for( i = 0; i < 10; ++i )
			rapid_flash(100);
		led_on();
		while(1) {
			if( f_lseek(&hex_file, 0) != FR_OK ) {
				error_flash(2, 8, &memoryCardSystemUp, 0);
				break;
			}
			check_mains_presence();
			flash_row_buffer = stack_flash_row_buffer;
			reflash_from_hex_file(&hex_file);
			led_off();
			if( f_lseek(&hex_file, 0) != FR_OK ) {
				error_flash(2, 8, &memoryCardSystemUp, 0);
				continue;
			}	
			result = check_hex_file(&hex_file);
			if( result == no_difference )
				break;
			error_flash(2, 3+result, &memoryCardSystemUp, 0);
		}
	}
}

int main (void) {
	unsigned short ir_ack_start = 0, ir_ack_delay = 0;

    // set PLL to get Fcy which is an integer multiple of 48kHz * 128 (24.576MHz / 8 * 52 / 2 / 2 = 39.936MHz)
	CLKDIVbits.PLLPRE = 6;
	CLKDIVbits.PLLPOST = 0;
    PLLFBD = 50;

    __builtin_write_OSCCONH(0x03); // switch to PLL
    __builtin_write_OSCCONL(0x01);

	/* Now to set up interrupt priority.
       The following interrupts are used:

		Interrupt	Purpose			Notes	Priority
		OC2			LED						1
		T3			LED						1
		CN			Infrared				2
		T1			Infrared				2
		SPI1		Slave data		Quick	3
		OC1			Slave timing			4
		CMP			Mains phase				5
		DMA0, DMA1	DAC						6
    */

	IPC1bits.OC2IP  = 1;
	IPC2bits.T3IP   = 1;
	IPC4bits.CNIP   = 2;
	IPC0bits.T1IP   = 2;
	IPC2bits.SPI1IP = 3;
    IPC0bits.OC1IP  = 4;
	IPC4bits.CMIP   = 5;
	IPC1bits.DMA0IP = 6;
	IPC3bits.DMA1IP = 6;

	if( !init_mains_phase() ) {
		error_flash(3, 7, 0, 0);
	}
	init_led();
	init_ir();
	init_dac();
	restore_default_remote_codes();
	set_playback_order(sorted);

	if( memoryCardSystemUp && !init_playback() ) {
		check_reflash_bootloader();
		if( dont_start_playback_automatically || !playback_start() ) {
			led_on();
		} else {
			goto ShowPlaybackError;
		}
	}

	while(1) {
		if( ir_ack_delay ) {
			if( ir_ack_start != mains_cycles ) {
				ir_ack_start = mains_cycles;
				if( --ir_ack_delay == 0 )
					led_on();
			}
		}

		if( ir_final_code != IR_NOCODE ) {
			const char* fn;
			unsigned long code = ir_final_code;
			ir_final_code = IR_NOCODE;

			fn = get_remote_control_function(code);
			if( fn && (!(code&IR_REPEAT) || !strcmp(fn, "volup") || !strcmp(fn, "voldn") || !strcmp(fn, "back") || !strcmp(fn, "forward")) ) {
				if( !strcmp(fn, "volup") ) {
					playback_volume_up();
				} else if( !strcmp(fn, "voldn") ) {
					playback_volume_down();
				} else if( fn[0] >= '0' && fn[0] <= '9' ) {
					playback_play_single_file(fn[0] == '0' ? 9 : fn[0]-'1');
				} else if( !strcmp(fn, "play") ) {
					playback_start();
				} else if( !strcmp(fn, "stop") ) {
					playback_stop();
				} else if( !strcmp(fn, "pause") ) {
					playback_pause();
				} else if( !strcmp(fn, "order") ) {
					switch(get_playback_order()) {
					case sorted:
						set_playback_order(shuffle);
						break;
					case shuffle:
						set_playback_order(directory);
						break;
					case directory:
						set_playback_order(sorted);
						break;
					}
				} else if( !strcmp(fn, "reset") ) {
					playback_stop();
					playback_stop();
				} else if( !strcmp(fn, "next") ) {
					playback_next_track(1);
				} else if( !strcmp(fn, "prev") ) {
					playback_prev_track(1);
				} else if( !strcmp(fn, "back") ) {
					playback_back();
				} else if( !strcmp(fn, "forward") ) {
					playback_forward();
				} else if( !strcmp(fn, "mute") ) {
					playback_toggle_mute();
				}
				ir_ack_delay = 8;
			} else {
				ir_ack_delay = 2;
			}

			if( infrared_logging ) {
				unsigned int written;
				if( !infrared_log_open ) {
					if( f_open(&ir_log, "0:\\IR_log.txt", FA_OPEN_ALWAYS|FA_WRITE) == FR_OK ) {
						if( f_lseek(&ir_log, ir_log.fsize) == FR_OK ) {
							infrared_log_open = 1;
						} else {
							f_close(&ir_log);
						}
					}
				}
				if( infrared_log_open ) {
					char buf[20];
//					sprintf(buf, "%5d %s(0x%04x)\r\n", mains_cycles, code&IR_RC5 ? "RC5" : "NEC", (unsigned int)code);
					printushortdec(buf, mains_cycles);
					strcpy(buf+5, code&IR_RC5 ? " RC5(0x" : " NEC(0x");
					printushorthex(buf+12, code);
					strcpy(buf+16, ")\r\n");
					if( f_write(&ir_log, buf, 19, &written) != FR_OK || written != 19 ) {
						f_close(&ir_log);
						infrared_log_open = 0;
					} else {
						ir_log_update_mains_cycles = mains_cycles;
					}
				}
			}
			ir_ack_start = mains_cycles;
			led_off();
		} else if( infrared_log_open && mains_cycles - ir_log_update_mains_cycles > 100 ) {
			f_close(&ir_log);
			infrared_log_open = 0;
		}

		if( !memoryCardSystemUp ) {
			if( !ir_ack_delay )
				set_led_brightness((mains_cycles&255) > 127 ? ((255-(mains_cycles&255))*2) : ((mains_cycles&255)*2));
			playback_reset();
		} else {
			if( get_playback_status() == reset ) {
				led_off();
				if( !init_playback() && (dont_start_playback_automatically || !playback_start()) ) {
					led_on();
				} else {
					playback_error_t err;
					int num_flashes;
				ShowPlaybackError:
					err = get_playback_error();

					switch(err) {
					case sd_card_invalid_response:
						num_flashes = 1;
						break;
					case sd_card_wrong_voltage:
						num_flashes = 9;
						break;
					case sd_card_timeout:
						num_flashes = 10;
						break;
					case fat_mount_failed:
						num_flashes = 2;
						break;
					case open_root_dir_failed:
						num_flashes = 3;
						break;
					case config_file_invalid:
						num_flashes = 11;
						break;
					case no_wav_files:
						num_flashes = 12;
						break;
					case invalid_wav_file:
						num_flashes = 13;
						break;
					case invalid_wav_format:
						num_flashes = 14;
						break;
					case file_read_error:
						num_flashes = 15;
						break;
					default:
						num_flashes = 0;
						break;
					}
					if( num_flashes )
						error_flash(3, num_flashes, &memoryCardSystemUp, 0);
				}
			} else {
				do_playback();
			}
		}
	}

    return 0;
}
