
#include "p33Fxxxx.h"
#include "../sdcard/diskio.h"
#include "../sdcard/ff.h"
#include "../sdcard/spi.h"
#include "hexfile.h"
#include "flash_operations.h"
#include <string.h>

/*
  Pin map (* = unwired):
    1  MCLR
    2  RA0/AN0  Output      Green LED cathode
    3  RA1/AN1  Input       Infrared receiver
    4  RB0/AN2  Comparator  AC phase detection
    5  RB1/AN3  Comparator  AC phase detection
    6  RB4/AN4  Analog      Sequencer Chain Detection
    7  RB3/AN5  Output      Sequencer Master Clear
    8  VSS
    9  XTI                  24.576MHz Crystal
    10 XTO                  24.576MHz Crystal
    11 RB4      Output      Sequencer Serial Clock
    12 RA4      Input       SD Card Detect
    13 VDD
    14 RB5      Output      Sequencer Serial Data
    15 RB6      Output      SD Card Chip Select
    16 RB7      Output      Sequencer Latch/CS
    17 RB8      Output      SD Card Data Out
    18 RB9      -           Tied to VDD
    19 VSS
    20 VCAP
    21 RB10     Output      SD Card Clock
    22 RB11     Input       SD Card Data In
    23 DAC1RP   DAC
    24 DAC1RN   DAC
    25 DAC1LP   DAC
    26 DAC1LN   DAC
    27 AVSS
    28 AVDD
*/

// Device configuration registers
_FGS(GWRP_OFF & GCP_OFF); // we don't want to write protect the flash
_FOSCSEL(FNOSC_FRC); // start up initially without the crystal or PLL
_FOSC(FCKSM_CSECMD & OSCIOFNC_OFF & POSCMD_HS);
_FWDT(FWDTEN_OFF); // not using the watchdog timer

unsigned char memoryCardSystemUp;

void DelayMs(unsigned short ms, unsigned char* interrupt, unsigned char interrupt_cmp) {
	do {
		unsigned short delay = ms;
		if( delay > 400 )
			delay = 400;

		TMR1 = 0;
		PR1 = 77*delay;
		IFS0bits.T1IF = 0;
		T1CONbits.TON = 1;
		while( !IFS0bits.T1IF ) {
			memoryCardSystemUp = !PORTAbits.RA4;
			if( interrupt && *interrupt == interrupt_cmp )
				return;
		}
		T1CONbits.TON = 0;
		IFS0bits.T1IF = 0;

		ms -= delay;
	} while( ms );
}

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200

FATFS fs;
DIR cur_pos;
FILINFO cur_file_info;
FIL cur_file;

#define HEX_FILE_NAME "Christmas Light Controller.hex"

static void delay_comparator_dead_time() {
	unsigned short until = TMR3 + (unsigned short)(7370000.0/2*42/4/256/100/20/2);
	while( (signed short)(until - TMR3) >= 0 )
		;
}

unsigned char check_ac_presence() {
	unsigned char mains_cycles, mains_frequency;

	// set up timer 3 for mains frequency detection
	T3CONbits.TCKPS = 3; // divide master clock by 256
	TMR3 = 0;

	// set up the comparator for AC phase monitoring
	CMCONbits.C2POS = 1;
	CMCONbits.C2EN = 1;
	IFS1bits.CMIF = 0;
	IFS0bits.T3IF = 0;

	while( !IFS1bits.CMIF )
		;
	T3CONbits.TON = 1;
	delay_comparator_dead_time();
	IFS1bits.CMIF = 0;

	for( mains_cycles = 0; mains_cycles < 10; ++mains_cycles ) {
		while( !IFS1bits.CMIF )
			;
		delay_comparator_dead_time();
		IFS1bits.CMIF = 0;
	}

	T3CON = 0;
	CVRCON = 0;
	CMCON = 0;
	IFS1bits.CMIF = 0;

	// oscillator is 7.37MHz +/- 5%
	if( !IFS0bits.T3IF && TMR3 >= (unsigned short)(7370000.0/2*42/4/256/12/2 * .95) && TMR3 <= (unsigned short)(7370000.0/2*42/4/256/12/2 * 1.05) )
		mains_frequency = 60;
	else if( !IFS0bits.T3IF && TMR3 >= (unsigned short)(7370000.0/2*42/4/256/10/2 * .95) && TMR3 <= (unsigned short)(7370000.0/2*42/4/256/10/2 * 1.05) )
		mains_frequency = 50;
	else
		mains_frequency = 0;

	TMR3 = 0;
	IFS0bits.T3IF = 0;

	return mains_frequency;
}

static void boot_main_program() {
	// restore state to default

	// undo pull-up for card sense pin
	CNPU1bits.CN0PUE = 0;

	// reset SPI for SD card
	SPI2STAT = 0;
	SPI2CON1 = 0;
	SPI2CON2 = 0;

	RPINR22 = 0x1F1F;
	RPOR5 = 0;
	TRISA = 0x079F;
	LATA = 0x0000;
	TRISB = 0xFFFF;
	LATB = 0x0000;
	_SPI2IF = 0;
	AD1PCFGL = 0;

	// reset DMA for SD card
	DMA2CON = 0;
	DMA2STA = 0;
	DMA2PAD = 0;
	DMA2CNT = 0;
	DMA2REQ = 0;

	// reset LED stuff
	T3CON = 0;
    PR3 = 0xFFFF;
	OC2CON = 0;
	IEC0bits.T3IE = 0;
   	IEC0bits.OC2IE = 0;

	// reset timer used for DelayMs
	T1CON = 0;
	PR1 = 0xFFFF;
	IFS0bits.T1IF = 0;
	__asm__ volatile("goto 0x200");
}

inline static void init_led() {
    TRISAbits.TRISA0 = 0;
	LATAbits.LATA0 = 1;
}
inline static void led_on() {
	LATAbits.LATA0 = 0;
}
inline static void led_off() {
	LATAbits.LATA0 = 1;
}
static void error_flash(unsigned char long_flashes, unsigned char short_flashes, unsigned char* interrupt, unsigned char interrupt_cmp) {
	unsigned char i;

	LATAbits.LATA0 = 1;

	while(!interrupt || *interrupt != interrupt_cmp) {
		for( i = 0; i < long_flashes; ++i ) {
			LATAbits.LATA0 = 0;
			DelayMs(1000, interrupt, interrupt_cmp);
			LATAbits.LATA0 = 1;
			DelayMs(1000, interrupt, interrupt_cmp);
		}
		for( i = 0; i < short_flashes; ++i ) {
			LATAbits.LATA0 = 0;
			DelayMs(300, interrupt, interrupt_cmp);
			LATAbits.LATA0 = 1;
			DelayMs(200, interrupt, interrupt_cmp);
		}
		DelayMs(2000, interrupt, interrupt_cmp);
	}
}

static unsigned char init_sdcard_and_open_hex_file() {
	unsigned char i;

	for( i = 0; i < INIT_ATTEMPTS; ++i ) {
		unsigned char result;
		DelayMs(INIT_DELAY, &memoryCardSystemUp, 0);
		result = disk_initialize(0);
		if( result != STA_NOINIT )
			break;
	}
	if( i == INIT_ATTEMPTS ) {
		return 1;
	}
	if( f_mount(0, &fs) != FR_OK ) {
		return 2;
	}
	if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK )
		return 3;
	return 0;
}

void pre_erase_flash_block(unsigned short addr) {
	if( addr == 0 ) {
		// Since erasing & rewriting this block is critical, let's make sure there is enough
		// power in the filter capacitor to finish it! Otherwise just reboot.
		if( !check_ac_presence() && !check_ac_presence() && !check_ac_presence() ) {
			error_flash(1, 7, &memoryCardSystemUp, !memoryCardSystemUp);
			asm volatile("reset");
		}
	}
}

unsigned long boot_address_cache[2];
void flash_block_erased(unsigned short addr) {
	if( addr == 0 ) {
		write_flash_word(0, boot_address_cache[0]);
		write_flash_word(2, boot_address_cache[1]);
	}
}

static void rapid_flash(unsigned short delay) {
    led_on();
	DelayMs(delay, 0, 0);
    led_off();
	DelayMs(delay, 0, 0);
}

int main(void) {
	unsigned char error, ret, i;

	// ensure that remote latches stay cleared during bootloader
	AD1PCFGLbits.PCFG5 = 1;
	TRISBbits.TRISB3 = 0;
	LATBbits.LATB3 = 0;

	// set up and switch to PLL
	PLLFBD = 41;
	CLKDIVbits.PLLPRE = 0;
	CLKDIVbits.PLLPOST = 1;
    __builtin_write_OSCCONH(0x01); // switch to FRC PLL
    __builtin_write_OSCCONL(0x01);

	CNPU1bits.CN0PUE = 1;

	// set up timer 0 for DelayMs
	T1CONbits.TCKPS = 3; // divided by 256;

	init_led();
	rapid_flash(200);
	rapid_flash(200);

	// check for mains presence, complain if it is missing
	pre_erase_flash_block(0);

	boot_address_cache[0] = read_program_word(0);
	boot_address_cache[1] = read_program_word(2);

	while(1) {
		memoryCardSystemUp = !PORTAbits.RA4;
		if( !memoryCardSystemUp )
			break;
		error = init_sdcard_and_open_hex_file();
		if( error ) {
			if( error == 3 ) // no hex file detected
				break;
		Failure:
			error_flash(1, error, &memoryCardSystemUp, 0);
			while( PORTAbits.RA4 )
				;
		} else {
			ret = check_hex_file(&cur_file);
			if( ret == difference ) {
				for( i = 0; i < 10; ++i )
					rapid_flash(100);
				led_on();
				if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK ) {
					error = 3;
					goto Failure;
				}	
				reflash_from_hex_file(&cur_file);
				led_off();
				if( f_open(&cur_file, "0:\\" HEX_FILE_NAME, FA_READ|FA_OPEN_EXISTING) != FR_OK ) {
					error = 3;
					goto Failure;
				}	
				ret = check_hex_file(&cur_file);
			}
			if( ret == no_difference ) {
				break;
			} else {
				error_flash(1, 3 + ret, &memoryCardSystemUp, 0);
				while( PORTAbits.RA4 )
					;
			}
		}
	}
	boot_main_program();

	return 0;
}
