
#include "sdcard/ff.h"
#include "sequencer.h"
#include "playback.h" // for is_wav_ext(), is_lsn_ext()
#include <string.h>

unsigned short ms_delay;
unsigned char lights[32];
unsigned char controller_phases[4];
signed short light_ramps[32], ramp_remainders[32];
unsigned short ramps[2];
FIL sequencer_file;
unsigned char valid_sequence, had_valid_sequence;
unsigned short sequence_buf[256];
unsigned short sequence_buf_len, sequence_buf_pos;
unsigned long last_ms;

int strncasecmp(const char* s1, const char* s2, size_t n) {
	while(n) {
		if( *s1 != *s2 )
			return (int)*s1 - (int)*s2;
		if( !*s1 )
			return 0;
		++s1;
		++s2;
		--n;
	}
	return 0;
}

unsigned char init_sequencer(char* wav_file_path) {
	char* p;
	unsigned char wav_ext;

	reset_sequencer(0);

	p = strrchr(wav_file_path, '.');
	if( p && ((wav_ext = is_wav_ext(p)) || is_lsn_ext(p)) ) {
		if( wav_ext )
			strcpy(p+1, "lsq");
		if( f_open(&sequencer_file, wav_file_path, FA_READ|FA_OPEN_EXISTING) == FR_OK ) {
			sequence_buf_len = sequence_buf_pos = 0;
			valid_sequence = 1;
			last_ms = 0;
			return 1;
		}
	}
	return 0;
}

void reset_sequencer(unsigned char allow_resume) {
	had_valid_sequence = valid_sequence && allow_resume;
	memset(lights, 0, 32);
	ramps[0] = ramps[1] = 0;
	ms_delay = 0;
	last_ms = 0;
	valid_sequence = 0;
	sequence_buf_len = sequence_buf_pos = 0;
}

unsigned char resume_sequencer() {
	if( had_valid_sequence && f_lseek(&sequencer_file, 0) == FR_OK )
		valid_sequence = 1;
	had_valid_sequence = 0;
	return valid_sequence;
}

static const unsigned short reciprocals[256] =
                                  { 65535, 32767, 21844, 16383, 13106, 10922, 9361, 8191, 7281, 6553, 5957, 5460, 5040, 4680, 4368, 4095, 3854, 3640, 3448,
                                    3276, 3120, 2978, 2848, 2730, 2620, 2520, 2426, 2340, 2259, 2184, 2113, 2047, 1985, 1927, 1871, 1819, 1770, 1724, 1679,
                                    1637, 1597, 1559, 1523, 1488, 1455, 1424, 1393, 1364, 1336, 1310, 1284, 1259, 1236, 1213, 1191, 1169, 1149, 1129, 1110,
                                    1091, 1073, 1056, 1039, 1023, 1007, 992, 977, 963, 949, 935, 922, 909, 897, 885, 873, 861, 850, 839, 829, 818, 808, 798,
                                    789, 779, 770, 761, 752, 744, 735, 727, 719, 711, 704, 696, 689, 682, 675, 668, 661, 654, 648, 642, 635, 629, 623, 617,
                                    611, 606, 600, 595, 589, 584, 579, 574, 569, 564, 559, 554, 550, 545, 541, 536, 532, 528, 523, 519, 515, 511, 507, 503,
                                    499, 495, 492, 488, 484, 481, 477, 474, 470, 467, 464, 461, 457, 454, 451, 448, 445, 442, 439, 436, 433, 430, 427, 425,
                                    422, 419, 416, 414, 411, 409, 406, 404, 401, 399, 396, 394, 391, 389, 387, 385, 382, 380, 378, 376, 373, 371, 369, 367,
                                    365, 363, 361, 359, 357, 355, 353, 351, 349, 348, 346, 344, 342, 340, 339, 337, 335, 333, 332, 330, 328, 327, 325, 323,
                                    322, 320, 319, 317, 316, 314, 313, 311, 310, 308, 307, 305, 304, 302, 301, 300, 298, 297, 296, 294, 293, 292, 290, 289,
                                    288, 286, 285, 284, 283, 281, 280, 279, 278, 277, 276, 274, 273, 272, 271, 270, 269, 268, 266, 265, 264, 263, 262, 261,
                                    260, 259, 258, 257, 256, 255 };

static void execute_command(unsigned short command) {
	unsigned char i, temp[2];

	switch(command>>8) {
	case 0:
		memset(lights, 0, sizeof(lights));
		ramps[0] = ramps[1] = 0;
		break;
	case 1:
		ms_delay += (command&255)+1;
		break;
	case 2:
		ms_delay += ((command&255)+1)<<2;
		break;
	case 3:
		ms_delay += ((command&255)+1)<<4;
		break;
	default:
		switch(command>>10) {
		case 1:
			for( i = 0; i < 8; ++i )
				lights[((command>>5)&24)+i] = (command&(1<<i)) ? 255 : 0;
			((unsigned char*)ramps)[(command>>8)&3] = 0;
			break;
		case 2:
			for( i = 0; i < 8; ++i )
				if( (command&(1<<i)) )
					lights[((command>>5)&24)+i] = 0;
			((unsigned char*)ramps)[(command>>8)&3] = 0;
			break;
		case 3:
			for( i = 0; i < 8; ++i )
				if( (command&(1<<i)) )
					lights[((command>>5)&24)+i] = 255;
			((unsigned char*)ramps)[(command>>8)&3] = 0;
			break;
		default:
			temp[0] =  command    &31;
			temp[1] = (command>>5)&31;
			if( (command>>10) < 8 ) {
				ramps[temp[0]>>4] &= ~(1<<(temp[0]&15));
				ramps[temp[1]>>4] &= ~(1<<(temp[1]&15));
			}
			switch(command>>10) {
			case 4:
				lights[temp[0]] = 255;
				lights[temp[1]] = 255;
				break;
			case 5:
				lights[temp[0]] = 0;
				lights[temp[1]] = 0;
				break;
			case 6:
				lights[temp[0]] = 255;
				lights[temp[1]] = 0;
				break;
			case 7:
				lights[temp[0]] ^= 255;
				lights[temp[1]] ^= 255;
				break;
			default:
				temp[0] = (command>>8)&31;
				if( (command>>13) == 1 ) {
					lights[temp[0]] = command;
					ramps[temp[0]>>4] &= ~(1<<(temp[0]&15));
				} else {
					long ramp;
					ramps[temp[0]>>4] |=   1<<(temp[0]&15);
					ramp_remainders[temp[0]] = 0;
					switch(command>>13) {
					case 2:
						ramp =   ((255-lights[temp[0]]) * (((long)reciprocals[command&255])+1))>> 9 ;
//						ramp = (255-lights[temp[0]]) * 1024L / (((command&255)+1)<<3);
						if( ramp > 32767 )
							light_ramps[temp[0]] = 32767;
						else
							light_ramps[temp[0]] = ramp;
						break;
					case 3:
						ramp = -(((    lights[temp[0]]) * (((long)reciprocals[command&255])+1))>> 9);
//						ramp = lights[temp[0]] * -1024L / (((command&255)+1)<<3);
						if( ramp < -32768L )
							light_ramps[temp[0]] = -32768;
						else
							light_ramps[temp[0]] = ramp;
						break;
					case 4:
						ramp =   ((255-lights[temp[0]]) * (((long)reciprocals[command&255])+1))>>10 ;
//						ramp = (255-lights[temp[0]]) * 1024L / (((command&255)+1)<<4);
						if( ramp > 32767 )
							light_ramps[temp[0]] = 32767;
						else
							light_ramps[temp[0]] = ramp;
						break;
					case 5:
						ramp = -(((    lights[temp[0]]) * (((long)reciprocals[command&255])+1))>>10);
//						ramp = lights[temp[0]] * -1024L / (((command&255)+1)<<4);
						if( ramp < -32768L )
							light_ramps[temp[0]] = -32768;
						else
							light_ramps[temp[0]] = ramp;
						break;
					case 6:
						ramp =   ((255-lights[temp[0]]) * (((long)reciprocals[command&255])+1))>>11 ;
//						ramp = (255-lights[temp[0]]) * 1024L / (((command&255)+1)<<5);
						if( ramp > 32767L )
							light_ramps[temp[0]] = 32767;
						else
							light_ramps[temp[0]] = ramp;
						break;
					case 7:
						ramp = -(((    lights[temp[0]]) * (((long)reciprocals[command&255])+1))>>11);
//						ramp = lights[temp[0]] * -1024L / (((command&255)+1)<<5);
						if( ramp < -32768L )
							light_ramps[temp[0]] = -32768;
						else
							light_ramps[temp[0]] = ramp;
						break;
					}
				}
				break;
			}
			break;
		}
		break;
	}
}

static void do_ramp_range(unsigned short ms, unsigned short start, unsigned short end) {
	unsigned short i;
	unsigned short present = ramps[start>>4];
	for( i = start; i < end; ++i ) {
		if( present&1 ) {
			long frac_add = (long)light_ramps[i] * (long)ms + ramp_remainders[i];
			short new_value = (short)lights[i] + (short)(frac_add / 1024);
			ramp_remainders[i] = ((short)frac_add)&1023;
			if( light_ramps[i] < 0 && ramp_remainders[i] )
				ramp_remainders[i] -= 1024;//|= 0xFC00;
			if( new_value > 255 ) {
				new_value = 255;
				ramps[start>>4] &= ~(1<<(i-start));
			} else if( new_value < 0 ) {
				new_value = 0;
				ramps[start>>4] &= ~(1<<(i-start));
			}
			lights[i] = new_value;
		}
		present >>= 1;
	}
}

static void do_ramps(unsigned short ms) {
	if( ramps[0] )
		do_ramp_range(ms, 0, 16);
	if( ramps[1] )
		do_ramp_range(ms, 16, 32);
}

unsigned short get_next_command() {
	if( sequence_buf_pos >= sequence_buf_len ) {
		unsigned int read;
		if( f_read(&sequencer_file, (BYTE*)sequence_buf, sizeof(sequence_buf), &read) != FR_OK || read < sizeof(unsigned short) ) {
			valid_sequence = 0;
			return 0;
		}
		sequence_buf_len = read>>1;
		sequence_buf_pos = 0;
	}
	return sequence_buf[sequence_buf_pos++];
}

extern volatile unsigned char DACBuffer_full;

unsigned char advance_sequence(unsigned long milliseconds) {
	unsigned short ms_period;

	if( milliseconds < last_ms ) {
		reset_sequencer(0);
		if( f_lseek(&sequencer_file, 0) == FR_OK ) {
			valid_sequence = 1;
		} else {
			return 0;
		} 
	}

	while( milliseconds > last_ms + 0xFFFF ) {
		advance_sequence(0xFFFF);
		last_ms += 0xFFFF;
	}

	ms_period = milliseconds - last_ms;
//	last_ms = milliseconds;

	do {
		if( ms_period < ms_delay ) {
			do_ramps(ms_period);
			ms_delay -= ms_period;
			last_ms += ms_period;
			ms_period = 0;
		} else {
			do_ramps(ms_delay);
			ms_period -= ms_delay;
			last_ms += ms_delay;
			ms_delay = 0;
			do {
				execute_command(get_next_command());
			} while( valid_sequence && ms_delay == 0 && DACBuffer_full );
		}
	} while( valid_sequence && ms_period );

	return valid_sequence;
}
