
#include "playback.h"
#include "sdcard/sd.h"
#include "sdcard/ff.h"
#include "wav.h"
#include "dac.h"
#include "mains_phase.h"
#include "sequencer.h"
#include "config.h"
#include "chainsense.h"
#include <string.h>
#include <stdlib.h>

unsigned char/*playback_state_t*/ playback_state;
unsigned char/*playback_error_t*/ last_playback_error;
unsigned char/*playback_order_t*/ desired_playback_order, current_playback_order;

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200
#define SORT_ENTRIES     102
#define SORT_ENTRY_SIZE (DAC_BUFFER_PAGES*512/SORT_ENTRIES)

extern CARD_INFO cardInfo;
FATFS fs;
DIR cur_pos;
FILINFO cur_file_info;
FIL cur_file;
unsigned long wav_data_start;
unsigned long wav_bytes_remaining;
unsigned long wav_data_size;
unsigned short wav_data_align, wav_sample_rate;
unsigned char wav_num_channels, wav_bytes_per_sample, wav_dummy, muted;
unsigned char transitioning_to_next_file, order_pos, play_just_one, dont_repeat_all;
unsigned char order[SORT_ENTRIES];
unsigned short order_num_files;
unsigned short volume;

static unsigned char do_open_wav() {
	char wavbuf[256];
	unsigned int nRead;
	WAVHeader header;

	if( f_read(&cur_file, (BYTE*)wavbuf, sizeof(wavbuf), &nRead) != FR_OK ) {
		playback_state = error;
		last_playback_error = file_read_error;
		return last_playback_error;
	}
	WAVinitHeader(&header);
	if( !WAVreadWAVHeader((BYTE*)wavbuf, &header, nRead) ) {
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	if( header.fmtHeader.audioFormat != 1 || (header.fmtHeader.numChannels != 1 && header.fmtHeader.numChannels != 2) ||
        header.fmtHeader.bitsPerSample != 16 || header.fmtHeader.sampleRate >= 65536 || !set_sample_rate(header.fmtHeader.sampleRate) ) {
		playback_state = error;
		last_playback_error = invalid_wav_format;
		return last_playback_error;
	}
	wav_data_start = header.dataHeader.dataPtr - (BYTE*)wavbuf;
	if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
		playback_state = error;
		last_playback_error = invalid_wav_file;
		return last_playback_error;
	}
	wav_data_size = wav_bytes_remaining = header.dataHeader.data.chunkSize;
	wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8 - 1));
	if( header.fmtHeader.numChannels == 1 && wav_data_align > 256 )
		wav_data_align -= 256;
	wav_sample_rate = header.fmtHeader.sampleRate;
	wav_num_channels = header.fmtHeader.numChannels;
	wav_bytes_per_sample = header.fmtHeader.numChannels * header.fmtHeader.bitsPerSample / 8;
	wav_dummy = 0;
	return 0;
}

static unsigned char do_open_lsn() {
	unsigned short buf[128];
	unsigned int nRead;
	unsigned short i;

	wav_data_size = 0;
	while( f_read(&cur_file, (BYTE*)buf, sizeof(buf), &nRead) == FR_OK && nRead > 0 ) {
		nRead >>= 1;
		for( i = 0; i < nRead; ++i ) {
			unsigned short cmd = buf[i];
			switch( cmd>>8 ) {
			case 1:
				wav_data_size += 3* ((cmd&255)+1);
				break;
			case 2:
				wav_data_size += 3*(((cmd&255)+1)<<2);
				break;
			case 3:
				wav_data_size += 3*(((cmd&255)+1)<<4);
				break;
			}
		}
	}
	wav_data_size <<= 4;
	if( wav_data_size == 0 ) {
		playback_state = error;
		last_playback_error = file_read_error;
		return last_playback_error;
	}
	wav_bytes_remaining = wav_data_size;

	wav_data_start = 0;
	wav_data_align = 0;
	wav_sample_rate = 12000;
	wav_num_channels = 2;
	wav_bytes_per_sample = 4;
	wav_dummy = 1;
	set_sample_rate(wav_sample_rate);
	return 0;
}

static unsigned char open_wav_file() {
	char path[_MAX_LFN+5] = "0:\\";
	const char* p;

	strcpy(path+3, cur_file_info.lfname);
	if( f_open(&cur_file, path, FA_READ|FA_OPEN_EXISTING) != FR_OK ) {
		playback_state = error;
		last_playback_error = file_read_error;
		return last_playback_error;
	}
	p = strrchr(cur_file_info.fname, '.');
	if( p && is_wav_ext(p) ) {
		if( do_open_wav() )
			return last_playback_error;
	} else {
		if( do_open_lsn() )
			return last_playback_error;
	}

	playback_state = stopped;
	last_playback_error = ok;
	reset_sample_timer();
	if( init_sequencer(path) ) {
		set_control_spi_words( (get_chain_length()>>1)+1 );
	}
	return last_playback_error;
}

static void insert_shuffle_index(unsigned short pos, unsigned short num, unsigned char which) {
	while( pos <= num ) {
		unsigned char old = order[pos];
		order[pos] = which;
		which = old;
		++pos;
	}
}

unsigned long gerhard_random;
unsigned short get_gerhard_random() {
	gerhard_random = (gerhard_random * 32719 + 3) % 32749;
	return gerhard_random;
}
void seed_gerhard_random(unsigned short seed) {
	gerhard_random += seed;
	get_gerhard_random();
	gerhard_random += seed;
}

static unsigned char goto_file_index(unsigned short index) {
	if( f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		const char* p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			if( index-- == 0 )
				return last_playback_error;
		}
	} while( cur_pos.sect != 0 );

	// this should not happen as we should only be passing in the index of files that exist
	playback_state = error;
	last_playback_error = no_wav_files;
	return last_playback_error;
}

static unsigned char find_file_index(unsigned char file_index) {
	unsigned short i;
	for( i = 0; i < order_num_files; ++i )
		if( order[i] == file_index )
			return i;
	return 0;
}

static unsigned char init_shuffle(unsigned char retain_old_pos) {
	unsigned short i;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		const char* p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	seed_gerhard_random(GetRandomSeed());
	for( i = 0; i < order_num_files; ++i ) {
		insert_shuffle_index(get_gerhard_random()%(i+1), i, i);
	}

	if( order_num_files == 0 ) {
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

typedef int (*compfunc_t) (const void*, const void*);
/*
int my_strcmp(const void* a, const void* b) {
//	return strcmp((const char*)a, (const char*)b);
	return ((char*)a)[0] - ((char*)b)[0];
}
*/

static unsigned char init_sort(unsigned char retain_old_pos) {
	unsigned short i;
	unsigned char* sort_buffer;
	unsigned char* dest;
	FILINFO old_file_info;
	unsigned short old_file_index = 0;

	order_num_files = 0;
	memcpy(&old_file_info, &cur_file_info, sizeof(old_file_info));

	if( f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	dest = sort_buffer = get_dac_buffer(); // DAC isn't running while we're doing this so we can use its buffer to sort the file names
	do {
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		const char* p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			if( !strcmp(cur_file_info.fname, old_file_info.fname) )
				old_file_index = order_num_files;
			strncpy((char*)dest, cur_file_info.lfname, SORT_ENTRY_SIZE-2);
			dest[SORT_ENTRY_SIZE-2] = '\0';
			dest[SORT_ENTRY_SIZE-1] = order_num_files;
			dest += SORT_ENTRY_SIZE;
			if( ++order_num_files == SORT_ENTRIES )
				break;
		}
	} while( cur_pos.sect != 0 );

	qsort(sort_buffer, order_num_files, SORT_ENTRY_SIZE, (compfunc_t)&strcmp);

	dest = sort_buffer+SORT_ENTRY_SIZE-1;
	for( i = 0; i < order_num_files; ++i ) {
		order[i] = dest[0];
		dest += SORT_ENTRY_SIZE;
	}

	memset(sort_buffer, 0, SORT_ENTRY_SIZE*SORT_ENTRIES);

	if( order_num_files == 0 ) {
		playback_state = error;
		last_playback_error = no_wav_files;
	} else {
		order_pos = retain_old_pos ? find_file_index(old_file_index) : 0;
	}

	if( playback_state == stopped )
		return goto_file_index(order[order_pos]);
	else
		return last_playback_error;
}

static unsigned char update_playback_order(unsigned char retain_old_pos) {
	if( desired_playback_order != current_playback_order ) {
		current_playback_order = desired_playback_order;
		if( desired_playback_order == shuffle )
			return init_shuffle(retain_old_pos);
		else if( desired_playback_order == sorted )
			return init_sort(retain_old_pos);
	}
	return last_playback_error;
}

static unsigned char goto_next_wav_file(unsigned char retain_old_pos) {
	unsigned char found = 0;

	if( update_playback_order(retain_old_pos) )
		return last_playback_error;
	if( !retain_old_pos )
		order_pos = SORT_ENTRIES-1;

	if( current_playback_order != directory ) {
		if( ++order_pos >= order_num_files )
			order_pos = 0;
		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		else
			return open_wav_file();
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			found = 1;
			break;
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		playback_state = error;
		last_playback_error = no_wav_files;
		return last_playback_error;
	} else {
		return open_wav_file();
	}
}

extern int strncasecmp(const char* s1, const char* s2, size_t n);
unsigned char goto_specific_wav_file(char* name, unsigned short name_len) {
	unsigned char found = 0, pos = 0;

	if( current_playback_order == directory && f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}

	if( update_playback_order(0) )
		return last_playback_error;

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			if( strlen(cur_file_info.lfname) == name_len && !strncasecmp(cur_file_info.lfname, name, name_len) ) {
				found = 1;
				break;
			}
			++pos;
		}
	} while( cur_pos.sect != 0 );

	if( current_playback_order != directory ) {
		for( order_pos = 0; order_pos < order_num_files; ++order_pos ) {
			if( order[order_pos] == pos )
				break;
		}
		if( order_pos == order_num_files )
			found = 0;
	}

	if( !found ) {
		playback_state = error;
		last_playback_error = no_wav_files;
		return last_playback_error;
	} else {
		return open_wav_file();
	}
}

static unsigned char goto_first_wav_file() {
	if( current_playback_order == directory && f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}
	return goto_next_wav_file(0);
}

static unsigned char goto_prev_wav_file() {
	FILINFO last_file_info;
	DIR last_pos;
	char old_fname[13];
	char old_lfname[_MAX_LFN+1];
	unsigned char first_file = 1;
	unsigned char found = 0;

	if( update_playback_order(1) )
		return last_playback_error;

	if( current_playback_order != directory ) {
		if( --order_pos >= order_num_files )
			order_pos = order_num_files-1;

		if( goto_file_index(order[order_pos]) )
			return last_playback_error;
		else
			return open_wav_file();
	}

	strcpy(old_fname, cur_file_info.fname);
	strcpy(old_lfname, cur_file_info.lfname);
	if( f_opendir(&cur_pos, "0:\\") != FR_OK ) {
		playback_state = error;
		last_playback_error = open_root_dir_failed;
		return last_playback_error;
	}

	do {
		const char* p;
		if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		p = strrchr(cur_file_info.fname, '.');
		if( p && is_playable_ext(p) ) {
			found = 1;
			if( !strcmp(cur_file_info.fname, old_fname) && !first_file ) {
				break;
			} else {
				memcpy(&last_file_info, &cur_file_info, sizeof(last_file_info));
				memcpy(&last_pos, &cur_pos, sizeof(last_pos));
				first_file = 0;
			}
		}
	} while( cur_pos.sect != 0 );

	if( !found ) {
		playback_state = error;
		last_playback_error = fat_mount_failed;
		return last_playback_error;
	} else {
		memcpy(&cur_file_info, &last_file_info, sizeof(cur_file_info));
		memcpy(&cur_pos, &last_pos, sizeof(cur_pos));
		strcpy(cur_file_info.lfname, old_lfname);
		return open_wav_file();
	}
}

extern unsigned char memoryCardSystemUp;

unsigned char init_playback() {
	unsigned char i;

	volume = 32768;

	for( i = 0; i < INIT_ATTEMPTS; ++i ) {
		unsigned char result;
		DelayMs(INIT_DELAY, &memoryCardSystemUp, 0);
		result = disk_initialize(0);
		if( result != STA_NOINIT )
			break;
	}
	if( i == INIT_ATTEMPTS ) {
		playback_state = error;
		switch(cardInfo.ERROR) {
//		case ERROR_NOT_SDMMC_CARD:
		default:
			last_playback_error = sd_card_invalid_response;
			break;
		case ERROR_BAD_VOLTAGE_RANGE:
			last_playback_error = sd_card_wrong_voltage;
			break;
		case ERROR_SDMMC_CARD_TIMEOUT:
			last_playback_error = sd_card_timeout;
			break;
		}
		return last_playback_error;
	}
	if( f_mount(0, &fs) != FR_OK ) {
		playback_state = error;
		last_playback_error = fat_mount_failed;
		return last_playback_error;
	}
	current_playback_order = directory;
	play_just_one = 0;
	reset_config_to_default();
	i = find_and_read_config();
	if( !i ) {
		playback_state = error;
		last_playback_error = config_file_invalid;
		return last_playback_error;
	}
	if( i == 1 )
		return goto_first_wav_file();
	else
		return last_playback_error;
}

unsigned char playback_start() {
	if( playback_state == stopped || playback_state == paused ) {
		if( playback_state == stopped ) {
			dac_reset_buffer();
			reset_sample_timer();
			resume_sequencer();
		}
		pause_dac(0);
		playback_state = playing;
	}
	play_just_one = 0;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_reset() {
	playback_state = reset;
	last_playback_error = ok;
	pause_dac(1);
	reset_sequencer(0);
	play_just_one = 0;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_stop() {
	reset_sequencer(playback_state == playing || playback_state == paused);
	if( playback_state == playing || playback_state == paused ) {
		if( playback_state == playing )
			pause_dac(1);
		if( f_lseek(&cur_file, wav_data_start) != FR_OK ) {
			playback_state = error;
			last_playback_error = invalid_wav_file;
		} else {
			wav_data_align = 512 - ((512 - (wav_data_start&511)) & ~(wav_bytes_per_sample - 1));
			wav_bytes_remaining = wav_data_size;
			playback_state = stopped;
		}
	} else if( playback_state == stopped ) {
		return goto_first_wav_file();
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char playback_pause() {
	if( playback_state == playing || playback_state == paused ) {
		pause_dac(is_dac_running());
		playback_state = is_dac_running() ? playing : paused;
	}
	return last_playback_error;
}

static unsigned char playback_next_track_int(unsigned char clear_buffer, unsigned char loop) {
	if( playback_state == playing && clear_buffer )
		dac_clear_buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_next_wav_file(1) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				if( !loop )
					return playback_reset();
				last_playback_error = ok;
				if( goto_first_wav_file() ) {
					pause_dac(1);
					reset_sequencer(0);
					return last_playback_error;
				}
			} else {
				pause_dac(1);
				reset_sequencer(0);
				return last_playback_error;
			}
		} else if( current_playback_order != directory && order_pos == 0 && !loop ) {
			return playback_reset();
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char playback_next_track(unsigned char clear_buffer) {
	return playback_next_track_int(clear_buffer, 1);
}

unsigned char playback_prev_track(unsigned char clear_buffer) {
	if( playback_state == playing && clear_buffer )
		dac_clear_buffer();
	if( playback_state != error ) {
		playback_state_t old_playback_state = playback_state;
		if( goto_prev_wav_file() ) {
			pause_dac(1);
			reset_sequencer(0);
			return last_playback_error;
		}
		playback_state = old_playback_state;
	}
	play_just_one = 0;
	return last_playback_error;
}

unsigned char playback_forward() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining > seek_distance ) {
			wav_bytes_remaining -= seek_distance;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				pause_dac(1);
				reset_sequencer(0);
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			offset_sample_timer(seek_distance / wav_bytes_per_sample);
		}
	}
	return last_playback_error;
}

unsigned char playback_back() {
	if( playback_state == playing ) { // calculate number of samples in 10 seconds
		unsigned long seek_distance = (unsigned long)wav_sample_rate * (unsigned long)wav_bytes_per_sample * 10;
		seek_distance = (seek_distance + 256) & ~511; // round to nearest sector
		if( wav_bytes_remaining < wav_data_size ) {
			wav_bytes_remaining += seek_distance;
			if( wav_bytes_remaining > wav_data_size )
				wav_bytes_remaining = wav_data_size;
			if( !wav_dummy && f_lseek(&cur_file, (wav_data_start) + wav_data_size - wav_bytes_remaining) != FR_OK ) {
				pause_dac(1);
				reset_sequencer(0);
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			offset_sample_timer(-(long)(seek_distance / wav_bytes_per_sample));
		}
	}
	return last_playback_error;
}

unsigned char playback_play_single_file(unsigned char index) {
	if( update_playback_order(1) )
		return last_playback_error;

	if( playback_state == playing )
		pause_dac(1);

	if( current_playback_order == directory ) {
		unsigned char found = 0;

		if( f_opendir(&cur_pos, "0:\\") != FR_OK ) {
			playback_state = error;
			last_playback_error = open_root_dir_failed;
			return last_playback_error;
		}

		do {
			const char* p;
			if( f_readdir(&cur_pos, &cur_file_info) != FR_OK ) {
				playback_state = error;
				last_playback_error = file_read_error;
				return last_playback_error;
			}
			p = strrchr(cur_file_info.fname, '.');
			if( p && is_playable_ext(p) ) {
				if( index-- == 0 ) {
					found = 1;
					break;
				}
			}
		} while( cur_pos.sect != 0 );

		if( !found ) {
			playback_state = error;
			last_playback_error = no_wav_files;
			return last_playback_error;
		} else {
			if( open_wav_file() )
				return last_playback_error;
		}
	} else {
		if( index >= order_num_files ) {
			if( playback_state != stopped ) {
				playback_stop();
			}
			return last_playback_error;
		}
		order_pos = index-1;
		if( goto_next_wav_file(1) ) {
			if( last_playback_error == no_wav_files && current_playback_order == directory ) {
				last_playback_error = ok;
				if( goto_first_wav_file() ) {
					reset_sequencer(0);
					return last_playback_error;
				}
			} else {
				return last_playback_error;
			}
		}
	}

	if( playback_state == playing ) {
		pause_dac(1);
		return last_playback_error;
	} else {
		return playback_start();
	}
}

unsigned char playback_volume_up() {
	if( volume < 16 )
		volume = 16;
	else
		volume = (volume * 0x11111L)>>16;

	if( volume > 32768 )
		volume = 32768;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_volume_down() {
	volume = (volume * 0xF000L)>>16;
	muted = 0;
	return last_playback_error;
}

unsigned char playback_volume_set(unsigned char percent) {
	volume = 32768;
	while( percent < 100 ) {
		volume = (volume * 0xE940L)>>16;
		++percent;
	}
	return last_playback_error;
}

unsigned char playback_toggle_mute() {
	muted = !muted;
	return last_playback_error;
}

playback_state_t get_playback_status() {
	return playback_state;
}

playback_error_t get_playback_error() {
	return last_playback_error;
}

playback_order_t get_playback_order() {
	return desired_playback_order;
}

unsigned char set_playback_order(playback_order_t order) {
	desired_playback_order = order;
	return last_playback_error;
}

static unsigned char play_file() {
	UINT read = 0;
	signed short wavbuf[256];
	unsigned short read_size = sizeof(wavbuf);
	unsigned short vol;
	unsigned short num_channels;

	if( transitioning_to_next_file ) {
		num_channels = 1;
		vol = 0;
	} else {
		num_channels = wav_num_channels;
		vol = muted ? 0 : volume;
		if( num_channels == 1 )
			read_size >>= 1;
		if( wav_data_align ) {
			if( wav_data_align + read_size >= 512 )
				read_size = 512 - wav_data_align;
			memset(wavbuf, 0, wav_data_align);
		}
		if( wav_dummy ) {
			memset(wavbuf, 0, read_size);
			read = read_size;
		} else if( f_read(&cur_file, ((BYTE*)wavbuf) + wav_data_align, read_size, &read) != FR_OK ) {
			pause_dac(1);
			reset_sequencer(0);
			playback_state = error;
			last_playback_error = file_read_error;
			return last_playback_error;
		}
		if( read > wav_bytes_remaining )
			read = wav_bytes_remaining;
		wav_bytes_remaining -= read;
		if( wav_data_align ) {
			wav_data_align += read;
			if( wav_data_align >= 512 )
				wav_data_align = 0;
		}
		if( read != read_size || wav_bytes_remaining == 0 ) {
			memset(((BYTE*)wavbuf)+read, 0, read_size-read);
			transitioning_to_next_file = DAC_BUFFER_PAGES;
		}
	}
	write_dac(wavbuf, num_channels, vol);
	return 0;
}

unsigned char do_playback() {
	if( playback_state == playing ) {
		unsigned char num_channels;

		do {
			if( play_file(&num_channels) )
				return last_playback_error;
	
			if( DACBuffer_full && valid_sequence )
				advance_sequence(read_sample_timer_ms());
	
			if( transitioning_to_next_file && --transitioning_to_next_file == 0 ) {
				if( play_just_one ) {
					play_just_one = 0;
					if( playback_stop() )
						return last_playback_error;
					return goto_first_wav_file();
				} else {
					return playback_next_track_int(0, !dont_repeat_all);
				}
			}
		} while( !DACBuffer_full );
	}
	return last_playback_error;
}
