; -------------------------------------------------------------------------------------------
; Square Wave Generator
; Copy Right (C) A. Sum 30-Dec-2010
; PIC 12F675
; Use Internal 4Mhz OSC
; Demonstrates the use of interrupts
; Push button connected to GP2 (pin 5) set the output frequency 
; GP1 (pin 6) outputs Square wave
; Timer 1 set to interrupt at twice the desired output frequency
; Timer 1 interrupt service toggle the output so achieveing 50% duty cycle
; Push button is debounced with Timer 0 interrupt to provide a debounce time of approx 130 ms
; Timer 1 Registers values corresponding to various frequencies stored in EEPROM
; References:
; PIC12F629/675 Data Sheet
; PICmicro Mid-Range MCU Family Reference Manual
; -------------------------------------------------------------------------------------------

	list      p=12f675           ; list directive to define processor
	#include <p12f675.inc>        ; processor specific variable definitions

	errorlevel  -302              ; suppress message 302 from list file

	__CONFIG   _CP_OFF & _CPD_OFF & _BODEN_OFF & _MCLRE_ON & _WDT_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT 

; '__CONFIG' directive is used to embed configuration word within .asm file.
; The lables following the directive are located in the respective .inc file.
; See data sheet for additional information on configuration word settings.

;---- VARIABLE DEFINITIONS ----
w_temp        EQU     0x20        ; variable used for context saving 
status_temp   EQU     0x21        ; variable used for context saving
freq		EQU		0x22		; 0=200Hz, 1=1KHz, 2=2kHz, 3=10kHz
TM1L		EQU		0x24		; TMR1L value storage location
TM1H		EQU		0x25		; TMR1H value atorage location
dbcnt		EQU		0x26		; use by debounce routine

#define	CLKOUT	GPIO,1		    ; Clk output GPIO 1

; --- MACROS--- 
; bank0 selects RAM bank 0, bank1 selects bank 1
; push_macro and pop_macro used in interrupt service routine to save W register and STATUS register

bank0	macro
		bcf	STATUS,RP0
		endm
bank1	macro
		bsf	STATUS,RP0
		endm

push_macro	macro
		movwf   w_temp            ; save off current W register contents
		swapf	STATUS,W          ; move status register into W register
		movwf	status_temp       ; save off contents of STATUS register
			endm

pop_macro	macro
		movf    status_temp,W     ; retrieve copy of STATUS register
		movwf	STATUS            ; restore pre-isr STATUS register contents
		swapf   w_temp,F
		swapf   w_temp,W          ; restore pre-isr W register contents
			endm

;-------------------------------------------------------------------------------------
; code starts here 
		ORG     0x000             ; processor reset vector
		goto    main              ; go to beginning of program
	
		ORG     0x004             ; interrupt vector location
		goto	service			  ; Interrupt service
main
		call    0x3FF             ; retrieve factory calibration value
		bank1				      ; set file register bank to 1 
		movwf   OSCCAL            ; update register with factory cal value 
		clrf	ANSEL		      ; GP 0-3 as digital I/O
		clrf	VRCON		      ; turn off VREF
		movlw	b'00111100'	      ; GP 0-1 as outputs, GP 2-5 input
		movwf	TRISIO
		movlw	b'00000111'	      ; Weak Pull up enabled, Falling edge
		movwf	OPTION_REG	      ; assign PS to TMR0 1:256
		movlw	b'00000100'	      ; Weak Pull up GP2 input
		movwf	WPU
		bank0					  ; Back to Bank 0
		clrf	GPIO
		movlw	0x07		 	  ; turn off comparator
		movwf	CMCON
		clrf	freq			  ; set to 200Hz as default
								  ; set TM1L,TM1H,TMR1L,TMR1H values to default 200Hz
		bank1
		movlw	0x00
		movwf	EEADR
		bsf		EECON1,RD
		movf	EEDATA,W	 	  ; read first locaction of EEPROM
		bank0
		movwf	TMR1L			  ; set TRR1L
		movwf	TM1L			  ; also save a copy in TM1L
		bank1
		movlw	0x01
		movwf	EEADR
		bsf		EECON1,RD
		movf	EEDATA,W		  ; read second location of EEPROM
		bank0
		movwf	TMR1H			  ; set TMR1H
		movwf	TM1H			  ; also save a copy in TM1H
							; enable timer 1 and GP2 interrupts
		clrf	STATUS
		clrf	INTCON
		clrf	PIR1
		movlw	0x02
		movwf	dbcnt			; set count to 2
		bank1
		movlw	b'00000001'	    ; TMR1IE set
		movwf	PIE1
		bank0
		movwf	T1CON		    ; set TMR1ON (enable Timer 1)
		movlw	b'11010000'	    ; GIE,PEIE,INTE set
		movwf	INTCON	

CLKLOOP					
		nop						; do nothing and wait for interrupts			
		nop						; can add your non-interrupt driven codes here		
		nop
		nop
		goto	CLKLOOP	

service							; Interrupt Service Routines
		push_macro				; Save W and STATUS

tm1check
		btfss	PIR1,0			; Timer1 interrupt ?
		goto	pbcheck			; no
		clrf	T1CON			; Yes disable Timer 1
		BCF		PIR1,0			; also clear Timer 1 int flag
		bank0
		movlw	b'00000010'		; GP1 bit
		xorwf	CLKOUT			; toggle GP1
reloadtm1						; reload TMR1L,TMR1H registers
		movf	TM1L,W
		movwf	TMR1L
		movf	TM1H,W
		movwf	TMR1H
		movlw	b'00000001'		; re-enable Timer 1
		movwf	T1CON
		bank1
		movwf	PIE1			;re-enable Timer 1 interrupt
		bank0			
isrdone
		pop_macro				;Restore saved W, STATUS
		retfie                  ; return from interrupt

pbcheck							;check for Push Button interrupt
		btfss	INTCON,1			;INTF flag set?
		goto	tm0check			; 
		bcf		INTCON,1			;yes clear INTF flag
		bcf		INTCON,4			;also disable further GP2 interrupt
		call	debounce			;debounce GP2 (enable TMR0 int, TMR0 oveflow will re-enable GP2 int)
		goto	isrdone
debounce
		clrf	TMR0
		BSF		INTCON,5			;enable timer 0 interrupt
		return
upfreq								;update new TMR1L,TMR1H values and also store them in TM1L,TM1H
		incf	freq,1				;increment	freq
		movlw	0x03				;modulo 3
		andwf	freq,1
		movf	freq,W
		addwf	freq,W				;calculate eeprom address offset
		bank1
		movwf	EEADR
		bsf		EECON1,RD			;read eeprom
		movf	EEDATA,W
		bank0
		movwf	TMR1L
		movwf	TM1L
		movf	freq,W
		addwf	freq,W				
		addlw	0x01				;increment addr
		bank1
		movwf	EEADR
		bsf		EECON1,RD
		movf	EEDATA,W
		movwf	TMR1H
		movwf	TM1H		
		return
tm0check
		btfss	INTCON,2			; Timer 0 interrupt ?
		goto	isrdone				; no
		BCF		INTCON,5			; disable timer 0 int
		BCF		INTCON,2			; clear timer 0 int flag
		decfsz	dbcnt,1				; second time timer 0 interrupt ?
		goto	notyet
tm0chk2	movlw	0x02				; reload dbcnt value
		movwf	dbcnt
		btfsc	GPIO,2				; is GP2 pin still low after timeout
		goto	tm0chk1				; no, don't update freq value
		call	upfreq				; yes PB press confirmed, update new freq values
tm0chk1
		BCF		INTCON,1			; clear INTF flag
		BSF		INTCON,4			; re-enable GP2 int
		goto	isrdone
notyet	clrf	TMR0
		BSF		INTCON,5			; enable timer 0 int
		goto	isrdone

; ---- EEPROM Data ----
; initialize eeprom locations
; values adjusted forinterrupt latency and timmer 1 interrupt service routine
		ORG		0x2100
		DE		0x6F,0xF6,0x2A,0xFE,0x22,0xFF,0xE5,0xFF

		END


	
