//
//***** ******* ******* ******* ******* ******* ******* *******
//Noughts and Crosses machine
//Software for Arduino
//
//Response to competition proposed by Dick Smith
//and announced in Silicon Chip October 2021.
//***** ******* ******* ******* ******* ******* ******* *******

//I think I need just one UNO
//and one Adafruit ILI9341 shield with tft and ts.

#include <SPI.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_ILI9341.h>
#include <Adafruit_STMPE610.h>
//#include "Adafruit_I2CDevice.h"

//The STMPE610 uses hardware SPI on the shield,
//and pin #8
#define STMPE_CS 8
Adafruit_STMPE610 ts = Adafruit_STMPE610(STMPE_CS);

//For the Adafruit shield, the default pins are:
#define TFT_DC 9
#define TFT_CS 10

//It is necessary to read quite a lot of examples
//to get the tft and the ts to work.
//From those examples, I concluded that I'll have
//least grief if I use the SPI interface for the tft
//and the I2C interface for the ts.
//I hope that this isn't a hope that bites me on the bum.
//Use hardware SPI (on Uno, #13, #12, #11)
//and the above for CS/DC.
Adafruit_ILI9341 tft = Adafruit_ILI9341(TFT_CS, TFT_DC);

//It seems safe to leave this commented;
// Option #1 - uses I2C, connect to hardware I2C port only!
// SCL to I2C clock (#A5 on Uno)
// and SDA to I2C data (#A4 on Uno)
// tie MODE to GND and POWER CYCLE (there is no reset pin)
//Adafruit_STMPE610 ts = Adafruit_STMPE610();

//***** ******* ******* ******* ******* ******* ******* *******
//A discussion of clones
//***** ******* ******* ******* ******* ******* ******* *******

//This program was developed using the genuine
//V2 TFT/TS from Adafruit;
//That is moderately expensive and many Arduino
//enthusiasts use clones enthusiastically;
//Although it was neither unusually easy nor
//unusually difficult to debug my program,
//once I had succeeded, I felt suitably brave and
//enthusiastic, so I bought a clone;

//Uh, oh;
//The clone was a clone of Adafruit's V1 TFT/TS;
//The differences are substantial;
//In particular, the V2 gadget uses the small
//six pin socket at one end of the UNO;
//The V1 gadget has no connector in this position;
//A big empty space makes this conspicuous;

//Further, the current Adafruit software library
//supports the V2 gadget, not the V1 gadget;
//After much searching, I discovered a library:
//MCUFRIEND_kbv, that does support the V1 Adafruit TFT/TS;
//It was easy to install and the examples all worked well,
//and a calibration procedure provides a text file
//of Arduino c statements that can be pasted into my program;
//The MCUFRIEND_kbv library seems to be highly compatible
//with the Adafruit GFX library, so gremlins should be rare;

//Nevertheless the number of chores dampened my enthusiasm
//and the task of adapting this program to work with
//a V1 Adafruit TFT/TS is an opportunity for improvement,
//or for someone more brave than me;

//***** ******* ******* ******* ******* ******* ******* *******
//Significant pixels on the screen are all constants,
//and I want to define mnemonic names for most of them.
//***** ******* ******* ******* ******* ******* ******* *******

//Warning!
//Initially, many values will be Select On Test (SOT) guesses.

//I want a title.
unsigned int const titleXcursor = 320 / 4;
unsigned int const titleYcursor = 12;
unsigned int const titleTextSize = 2;

//At the left, I anticipate three mini windows.
//The upper window will be touch sensitive,
//the lower two will be display only.

//It is useful to draw a border around each box.
//Done thoughtlessly, I could need X and Y values
//for the box and its border, and that'd be a pain.
//A little thought provides a little simplification.
//I draw each box using tft.fillRect() and the fill colour,
//and then draw it again using tft.drawRect()
//and the border colour.
//Provided I always do these in this order,
//the border will overwrite the fill.

//All three boxes will be the same size.
unsigned int const boxSizeX = 72;
unsigned int const boxSizeY = 36;

//The touch sensitive window displays:
//+---------------+
//|    Human is _ |
//| Computer is _ |
//+---------------+
//Where _ is replaced by X or O appropriately,
//and toggles if pressed.
//I'll use the standard Adafruit monospaced font.

//From the GFX library tutorial I discovered that
//for this font, the cursor should be positioned
//at the top left corner of the first character to display.

//I'll name the box "human",
//even though "player" might (or might not) be a better name.
//Box to display "Human is X".
//I need X, Y limits for the touch screen too.
unsigned int const humanXmin = 6;
unsigned int const humanYmin = 36 + 15;
unsigned int const humanXmax = humanXmin + boxSizeX;
unsigned int const humanYmax = humanYmin + boxSizeY;
unsigned int const humanCursorX = humanXmin + 6;
unsigned int const humanCursorY1 = humanYmin + 6;
unsigned int const humanCursorY2 = humanCursorY1 + 12;
unsigned int const humanTextSize = 1;

//Box to display "Your turn",
//or "Computer wins",
//or "Human wins"
//or "Draw".
//I need X, Y limits even though this box isn't touch sensitive.
unsigned int const yourTurnXmin = humanXmin;
unsigned int const yourTurnYmin = humanYmax + 30;
unsigned int const yourTurnXmax = yourTurnXmin + boxSizeX;
unsigned int const yourTurnYmax = yourTurnYmin + boxSizeY;
unsigned int const yourTurnCursorX = yourTurnXmin + 6;
unsigned int const yourTurnCursorY = yourTurnYmax - 24;
unsigned int const yourTurnTextSize = 1;

//Box to display "Be firmer" or "Be gentle" pressure message.
//I need X, Y limits even though this box isn't touch sensitive.
unsigned int const pressureXmin = humanXmin;
unsigned int const pressureYmin = yourTurnYmax + 30;
unsigned int const pressureXmax = pressureXmin + boxSizeX;
unsigned int const pressureYmax = pressureYmin + boxSizeY;
unsigned int const pressureCursorX = pressureXmin + 6;
unsigned int const pressureCursorY = pressureYmax - 24;
unsigned int const pressureTextSize = 1;

//Define the upper and lower limits of normal pressure.
unsigned int const pressureTooGentle = 40;
unsigned int const pressureTooFirm = 180;

//To define the size of the board,
//I'll require it to be square, not rectangular.
//I'll also define the size of each cell
//and make the board from three of them.
unsigned int const boardCellSize = 66;
unsigned int const boardSize = boardCellSize * 3;
unsigned int const boardTextSize = 3;

//I need two margins for the cells of the board:
//When reading a ts touch,
//I want to reject ambiguous touches.
//The Y margin can be the same as the X margin.
//Although boardCellMargin must be << boardCellSize,
//I'll define it manually, not cleverly.
unsigned int const boardCellMargin = 6;

//When using drawChar to write an X or O into a cell,
//I want to specify a margin from the
//top left corner of each cell.
//In this case, I might want the X margin
//to be slightly different from the Y margin.
unsigned int const drawCharXoffset = 25;
unsigned int const drawCharYoffset = 24;

//I also need to define the corners of the board.
unsigned int const boardXmin = 96;
unsigned int const boardYmin = 36;
unsigned int const boardXmax = boardXmin + boardSize;
unsigned int const boardYmax = boardYmin + boardSize;

//To translate X and Y from the ts to significant
//locations on the screen, it is useful to have a list
//of significant values.

//Initially, I put these tables in PROGMEM,
//but after I had read some fine print,
//I thought that I'd lose more on the swings than
//I'd gain on the roundabouts, so I put them into RAM.
//Hang on!  Tables of constants in RAM?  That can't be right.
//I read the documentation a bit more and put them
//back into PROGMEM.
//Part of the confusion arose because I forgot that
//myVar = myTable[myIndex] doesn't work.
//To read a byte or word from PROGMEM I need either:
//myVar = pgm_read_byte_near(myTable[myIndex]);
//or:
//myVar = pgm_read_word_near(myTable[myIndex]);

//Warning!
//the ..._near in the name is an opportunity to worry.
//There are two other readers:
//  pgm_read_byte_far(), and:
//  pgm_read_word_far().
//It is natural to wonder if I need to use these.
//The answer is elusive and it took me ages to stumble
//over an explanation that persuaded me that for devices
//with less than 64kB of memory, _near is certain to be
//appropriate, and that even for devices with more memory,
//the linker processes PROGMEM cleverly so that _near
//is almost certain to be appropriate.
//Phew!

unsigned int const PROGMEM significantXmin[3] =
{
  //Left column of cells.
  boardXmin + boardCellMargin,
  //Middle column of cells.
  boardXmin + boardCellSize + boardCellMargin,
  //Right column of cells.
  boardXmin + 2 * boardCellSize + boardCellMargin
};

unsigned int const PROGMEM significantXmax[3] =
{
  //Left column of cells.
  boardXmin + boardCellSize - boardCellMargin,
  //Middle column of cells.
  boardXmin + 2 * boardCellSize - boardCellMargin,
  //Right column of cells.
  boardXmin + 3 * boardCellSize - boardCellMargin
};

unsigned int const PROGMEM significantYmin[3] =
{
  //Top column of cells.
  boardYmin + boardCellMargin,
  //Middle column of cells.
  boardYmin + boardCellSize + boardCellMargin,
  //Bottom column of cells.
  boardYmin + 2 * boardCellSize + boardCellMargin
};

unsigned int const PROGMEM significantYmax[3] =
{
  //Top column of cells.
  boardYmin + boardCellSize - boardCellMargin,
  //Middle column of cells.
  boardYmin + 2 * boardCellSize - boardCellMargin,
  //Bottom column of cells.
  boardYmin + 3 * boardCellSize - boardCellMargin
};

//When a cell has been identified,
//it is useful to know where to place the cursor
//to write the X or 0.
unsigned int const PROGMEM cellCursorX[3] =
{
  //Left column of cells.
  boardXmin + drawCharXoffset,
  //Middle column of cells.
  boardXmin + boardCellSize + drawCharXoffset,
  //Right column of cells.
  boardXmin + 2 * boardCellSize + drawCharXoffset
};

unsigned int const PROGMEM cellCursorY[3] =
{
  //Top column of cells.
  boardYmin + drawCharYoffset,
  //Middle column of cells.
  boardYmin +  boardCellSize + drawCharYoffset,
  //Bottom column of cells.
  boardYmin + 2 * boardCellSize + drawCharYoffset
};

//It isn't convenient to use a similar strategy for
//detecting when the single humanIs_ window has been pressed,
//so I'll do that in the relevant procedure.

//***** ******* ******* ******* ******* ******* ******* *******
//Tables for implementing each part of
//the Newell, Simon strategy.
//***** ******* ******* ******* ******* ******* ******* *******

//If each of the eight steps of the Newell, Simon strategy
//is translated naively into procedural
//for(eachCell) {if(thisTest){ ... }} statements,
//the program becomes a very large and very inefficient ROM.

//It is important to find a better way.
//The first insight is to loop through the comparisons,
//not the cells.
//For example, for the tryWin() test,
//there are four comparisons for the one center cell,
//three comparisons for each of the 4 corner cells,
//and two comparisons for each of the 4 edge cells,
//for a total of 24 comparisons.
//Each of these comparisons can be identified by the
//indices of the relevant cells.
//For example:
//The cell to play, the first adjacent cell,
//the second adjacent cell.
//Although a relation exists between these cells,
//it is not simple, and clever implementations using
//arithmetic become messy.
//Instead, it is easier to work out the indices manually
//and record them in a table.
//Although it is possible to pack these indices,
//that also risks losing more on the swings than
//it gains on the roundabouts.

//***** ******* ******* ******* ******* ******* ******* *******
//Table for tryWin.
//tryBlock is so similar that it can use this table too.
//***** ******* ******* ******* ******* ******* ******* *******

//The table has a simple implied structure.
//1:
//The center cell identifies itself and is followed by
//a byte advising that four vectors need to be tested.
//The following eight bytes identify
//the four vectors that include the center cell.
//2:
//Each corner cell identifies itself and is followed by
//a byte advising that three vectors need to be tested.
//The following six bytes identify
//the three vectors that include that cell.
//3:
//Each edge cell identifies itself and is followed by
//a byte advising that two vectors need to be tested.
//The following four bytes identify
//the two vectors that include that cell.

//I tried to define those values mnemonically,
//but it all became very messy and my mnemonics became
//more confusing than mnemonic.
//Consequently, the relevant values are defined explicitly.

unsigned int const tryWinIndexMax = 19 + 60 + 44;
unsigned char const PROGMEM
tryWinIndices[tryWinIndexMax] =
{
  //One center cell with four comparisons.
  1, 1, 16,  //Center is [1,1], 4 vectors need 16 bytes
  0, 0, 2, 2, //Top left to bottom right
  0, 2, 2, 0, //Bottom left to top right
  0, 1, 2, 1, //Middle horizontal
  1, 0, 1, 2, //Middle vertical

  //Four corner cells, each with three comparisons.
  0, 0, 12,  //Top left corner
  1, 0, 2, 0,
  1, 1, 2, 2,
  0, 1, 0, 2,

  2, 0, 12,  //Top right corner
  0, 0, 1, 0,
  0, 2, 1, 1,
  2, 1, 2, 2,

  2, 2, 12,  //Bottom right corner
  0, 0, 1, 1,
  2, 0, 2, 1,
  0, 2, 1, 2,

  0, 2, 12,  //Bottom left corner
  1, 1, 2, 0,
  0, 0, 0, 1,
  1, 2, 2, 2,

  //Four edge cells, each with two comparisons.
  0, 1, 8,  //Left middle
  0, 0, 0, 2,
  1, 1, 2, 1,

  1, 0, 8,  //Top middle
  0, 0, 2, 0,
  1, 1, 1, 2,

  2, 1, 8,  //Right middle
  2, 0, 2, 2,
  1, 1, 0, 1,

  1, 2, 8,    //Bottom middle
  0, 2, 2, 2,
  1, 1, 1, 0
};

//***** ******* ******* ******* ******* ******* ******* *******
//Tables for corners and edges.
//***** ******* ******* ******* ******* ******* ******* *******

//For corners and edges,
//it is useful to define two more tables.

//A table of corner indices.
//It is useful to identify two corners:
//the trial corner and its opposite corner.

//As I constructed the table,
//I noticed that each cell is identified twice,
//once as itself and again as its opposite corner.
//I was tempted to eliminate this redundancy,
//but as I wrote the procedure, I discovered that
//I lost more on the swings than
//I gained on the roundabouts.

//I do not need to record the number of vectors
//within this table.  It is always 4.

unsigned int const tryCornerIndexMax = 16;
unsigned char const PROGMEM
tryCornerIndices[tryCornerIndexMax] =
{
  0, 0, 2, 2,
  2, 2, 0, 0,
  0, 2, 2, 0,
  2, 0, 0, 2
};

//A table of edge indices.

//As for the corner indices,
//it is tempting to minimise redundancy.
//In this case, the details are different.
//I do not need to know about "opposite edge",
//and I just need two bytes for each cell.

unsigned int const tryEdgeIndexMax = 8;
unsigned char const PROGMEM
tryEdgeIndices[tryEdgeIndexMax] =
{
  0, 1,
  1, 0,
  2, 1,
  1, 2
};

//***** ******* ******* ******* ******* ******* ******* *******
//Text messages
//***** ******* ******* ******* ******* ******* ******* *******

//Although the clever F("Quoted text") macro is convenient,
//I think it might not be clever enough to recognise common
//text and might create some text redundantly.
//I also found that it is less convenient when I have a
//combination of some constant text and some variable text
//in one .print() statement.

//Consequently, I prefer to define constant text explicitly
//in suitable constant arrays, together with a suitable name.

//I also tried the commented statements below,
//together with .print(<constant name>) statements.
//That didn't work, either.
//It seems that the Arduino implementation of .print
//is somewhat rudimentary and needs text strings to be
//in the .print() statement.

//unsigned char const titleText[] = "Noughts and Crosses ";
//unsigned char const humanIsText[] = "Human is ";
//unsigned char const computerIsText[] = "Computer is ";
//unsigned char const yourTurnText[] = "Your turn ";
//unsigned char const beGentleText[] = "Be gentle ";
//unsigned char const beFirmerText[] = "Be firmer ";

//***** ******* ******* ******* ******* ******* ******* *******
// Color definitions
//***** ******* ******* ******* ******* ******* ******* *******

//It is useful to know that the Adafruit library file:
//#include "Adafruit_ILI9341.h" included above,
//defines mnemonic names for some of the colours
//that are available.
//Oodles of other colours are possible,
//but I might as well use a suitable subset
//of those defined by Adafruit.

//I tried to give them names relevant to this program, too,
//but the translation from the ILI9341_ names
//to something more mnemonic seemed to confuse
//the Adafruit functions and White became Blue,
//and Green became an extraordinarily pale
//imitation of green.

//The Adafruit colour definitions, copied for convenience.
// Color definitions
//#define ILI9341_BLACK 0x0000       ///<   0,   0,   0
//#define ILI9341_NAVY 0x000F        ///<   0,   0, 123
//#define ILI9341_DARKGREEN 0x03E0   ///<   0, 125,   0
//#define ILI9341_DARKCYAN 0x03EF    ///<   0, 125, 123
//#define ILI9341_MAROON 0x7800      ///< 123,   0,   0
//#define ILI9341_PURPLE 0x780F      ///< 123,   0, 123
//#define ILI9341_OLIVE 0x7BE0       ///< 123, 125,   0
//#define ILI9341_LIGHTGREY 0xC618   ///< 198, 195, 198
//#define ILI9341_DARKGREY 0x7BEF    ///< 123, 125, 123
//#define ILI9341_BLUE 0x001F        ///<   0,   0, 255
//#define ILI9341_GREEN 0x07E0       ///<   0, 255,   0
//#define ILI9341_CYAN 0x07FF        ///<   0, 255, 255
//#define ILI9341_RED 0xF800         ///< 255,   0,   0
//#define ILI9341_MAGENTA 0xF81F     ///< 255,   0, 255
//#define ILI9341_YELLOW 0xFFE0      ///< 255, 255,   0
//#define ILI9341_WHITE 0xFFFF       ///< 255, 255, 255
//#define ILI9341_ORANGE 0xFD20      ///< 255, 165,   0
//#define ILI9341_GREENYELLOW 0xAFE5 ///< 173, 255,  41
//#define ILI9341_PINK 0xFC18        ///< 255, 130, 198

//***** ******* ******* ******* ******* ******* ******* *******
//Global variables.
//***** ******* ******* ******* ******* ******* ******* *******

//Although it is a bit inelegant to define most variables
//globally, it is easy to do that to get started,
//and I'll leave more elegant definitions until appropriate.

//The most obvious variable is the computer's view of the board.
//And the simplest structure is a two dimensional array of chars.
//That is a bit extravagant,
//but allows simple representations of:
//<space> for empty cells;
//X for cells containing X;
//O for cells containing O.
unsigned char thisBoard[3][3];

//Although the name of the game is "Noughts and Crosses",
//I'll use 'O', not '0' for "nought".

//I need two indices to address each cell within the board.
unsigned char thisXindex;
unsigned char thisYindex;

//And the corresponding cursor positions on the tft.
unsigned int thisXcursor;
unsigned int thisYcursor;

//I need to distinguish between several states of the game:
unsigned char const gameInProgress = 0;
unsigned char const gameWonByHuman = 1;
unsigned char const gameWonByComputer = 2;
unsigned char const gameDrawn = 3;
unsigned char gameState;
//To catch gameDrawn when the human is O,
//it is necessary to count the number of blanks on the board;
int blanksAvailable;

//I need to know who is X.
//Although it is a bit redundant to use two variables,
//this simplifies writing the appropriate character
//to the board and/or to the tft.
unsigned char humanSymbol;
unsigned char computerSymbol;

//The tft and ts need some esoteric magic numbers.
//In particular, the ts is slightly bigger than the tft,
//so values from the margin should be ignored or constrained
//or handled in an appropriate way.

#define TS_MINX 150
#define TS_MINY 130
#define TS_MAXX 3800
#define TS_MAXY 4000

//I discovered that debounce is essential.
//The FIFO buffer within the SMTPE610 is more of a
//liability than an asset and ts.bufferEmpty()
//is not a reliable test that the ts has finished bouncing.
//Although ts.getPoint() flushes the FIFO buffer
//and returns only the latest point,
//it can't flush bounce points yet to arrive into the buffer.
//I prefer to use millis() with two variables.
unsigned long const debounceMillis = 600;
unsigned long previousMillis;
unsigned long currentMillis;

//I feel happier with thisX, thisY, and thisZ as
//explicit variables, rather than using the
//clever p.x, p.y, p.z syntax, so I copy the values
//read from the ts into normal variables.
unsigned int thisX;
unsigned int thisY;
unsigned char thisZ;

//A flag to record when the optimum strategy has been found.
unsigned char foundMove;

//I need six bytes to help mark the winning row when
//someone wins;
//I'd like to draw a line, but to do that,
//I'd need to know which cells are at an end of a row,
//and which cell is in the middle,
//and to draw a line really well, I'd need that line
//to extend beyond the middle of the cell;
//For now, I'll just remember the X and Y cursor values
//of the three winning cells,
//and not fuss over which is which.
//I'll display their winning status by changing
//their colour;
unsigned int winXa;
unsigned int winYa;
unsigned int winXb;
unsigned int winYb;
unsigned int winXc;
unsigned int winYc;

//***** ******* ******* ******* ******* ******* ******* *******
//A note on debugging.
//***** ******* ******* ******* ******* ******* ******* *******

//Circa 2021, the usual IDE for developing Arduino sketches
//is still at V1, and V2 is still a beta release,
//for brave developers.

//In particular, V1 has none of:
//A simulator;
//Breakpoints;
//Single step.

//Without these, about the only way to debug any sketch
//is to sprinkle ad hoc Serial.println() commands liberally.
//This is the technique that was used often in the
//early days of computing, of punched cards and line
//printers that could print only uppercase letters.
//Those were the days.  Sigh.

//Some developers have made clever use of less than well
//documented prepass capabilities to introduce macros
//and conditional compilation.
//I consideed these, but I prefer to persevere with the old,
//punched card way until V2 becomes safe to use.

//In the days of punched cards, the debug statements became
//at least unnecessary and usually problematic after the
//bug had been discovered and corrected.
//Programmers used one or both of two strategies to
//manage old debug statements.
//Often, they were deleted, by removing the debug cards.
//This was easy, but often discovered to be premature when
//the next bug announced its presence.

//A better practice, practiced less often than appropriate,
//was to replace the active debug cards with cards rewritten
//to turn the debug statements into comments.
//In the days of punched cards, this was a cumbersome
//chore, generating extra work for the people punching
//the cards, and introducing questions like,
//"When can I bin the old cards?"
//It is not surprising that debug cards were discarded
//more often than they were turned into comments.

//With online editing, is is less difficult to turn debug
//statements into comments, and there is less excuse
//to delete them.

//To provide some assistance, I bracket debug statements
//with the comment lines:
//begin opportunity to comment for debug.
//end opportunity to comment for debug.

//These lines help debug statements to be identified before
//and after they have become comments.

//***** ******* ******* ******* ******* ******* ******* *******
//Setup
//***** ******* ******* ******* ******* ******* ******* *******

void setup()
{
  //Code to run once:

  //begin opportunity to comment for debug.
  Serial.begin(9600);
  Serial.println("ILI9341 Test.  ");
  //end opportunity to comment for debug.

  tft.begin();
  ts.begin();

  /*
    //begin opportunity to comment for debug.
    Serial.println("tft, ts begun. ");
    Serial.print("boardYmax = "); Serial.println(boardYmax);
    //end opportunity to comment for debug.
  */

  //Clear the display.
  tft.fillScreen(ILI9341_BLACK);
  //tft.fillRect(0, 0, 320, 240, ILI9341_BLACK);

  //Set the rotation.
  //Legal values are:
  //  0 for Portrait with the USB connector top right;
  //  1 for Landscape with the USB connector top left;
  //  2 for ??;
  //  3 for ??
  //and no others.
  tft.setRotation(1);

  //Display a title.
  tft.setCursor(titleXcursor, titleYcursor);
  tft.setTextColor(ILI9341_BLUE);
  tft.setTextSize(titleTextSize);
  tft.println(F("Noughts and Crosses "));

  //begin opportunity to comment for debug.
  Serial.println(F("Title done. "));
  //end opportunity to comment for debug.

  //Draw the board.
  drawBoard();

  //Draw each of the I/O boxes.

  //The humanIs _ box.
  tft.drawRect(humanXmin, humanYmin,
               boxSizeX, boxSizeY, ILI9341_BLUE);
  tft.setCursor(humanCursorX, humanCursorY1);
  tft.setTextColor(ILI9341_BLUE);
  tft.setTextSize(humanTextSize);
  tft.setCursor(humanCursorX, humanCursorY1);
  tft.print(F("Human is X "));
  tft.setCursor(humanCursorX, humanCursorY2);
  tft.print(F("Duino is O "));

  //Record that the human is X,
  //and that the computer is O.
  humanSymbol = 'X';
  computerSymbol = 'O';

  //Indicate that the game is in progress now.
  gameState = gameInProgress;

  //Use a procedure to draw the Your turn box
  //and write "Your turn " into it.
  writeYourTurnBox();

  //The box to display "Be firmer" or "Be gentle"
  //pressure message.
  //I don't draw this box unless it is needed.
  /*
    tft.drawRect(pressureXmin, pressureYmin,
               boxXSize, boxYSize, ILI9341_BLUE);
    tft.setCursor(pressureCursorX, pressureCursorY);
    tft.setTextColor(ILI9341_BLUE);
    tft.setTextSize(pressureTextSize);
    tft.print(F("Be gentle "));
  */

  //Start with a clear board.
  clearBoard();

  //begin opportunity to comment for debug.
  Serial.println(F("clearBoard done. "));
  //end opportunity to comment for debug.

  //begin opportunity to comment for debug.
  //Print the tables:
  //tryWinIndices, tryCornerIndices, tryEdgeIndices.
  //debugPrintTables();
  //end opportunity to comment for debug.

  //Prepare for debounce.
  previousMillis = millis();
}

//***** ******* ******* ******* ******* ******* ******* *******
//Loop
//***** ******* ******* ******* ******* ******* ******* *******

void loop()
{
  //To run repeatedly:

  //While waiting until the ts has stopped bouncing,
  //it is important to read the buffer to flush old points.
  currentMillis = millis();
  if (currentMillis - previousMillis < debounceMillis)
  {
    //Serial.print("PreviousMillis = ");
    //Serial.println(previousMillis);
    //Serial.print("CurrentMillis = ");
    //Serial.println(currentMillis);
    //Serial.print("DebounceMillis = ");
    //Serial.println(debounceMillis);
    //delay(100);
    TS_Point p = ts.getPoint();
    return;
  }

  //Keep waiting while the buffer remains empty.
  if (ts.bufferEmpty())
  {
    //Serial.print("currentMillis = ");
    //Serial.println(currentMillis);
    //delay(100);
    return;
  }

  //When the buffer ceases to be empty,
  //it should be safe to read the point touched.
  //Hopefully.

  TS_Point p = ts.getPoint();
  //The Adafruit documentation suggests using map()
  //to deal with marginal points.
  //I'll consider that first and consider
  //constrain() if map() is temperamental.

  //The instructions immediately below were suggested
  //by Adafruit but have been reduced to comments:
  //thisX = map(p.x, TS_MINX, TS_MAXX, 0, tft.width());
  //thisY = map(p.y, TS_MINY, TS_MAXY, 0, tft.height());
  //thisZ = p.z;
  //The explanation follows:

  //After quite a lot of testing,
  //and a little help from Excel,
  //I was able to construct the table below:
  //Item:                   ts p.x:   tft Y:
  //Top horizontal row      2753      69
  //Middle horizontal row   1758      135
  //Bottom horizontal row   802       801
  //
  //Item:                   ts p.y:   tft X:
  //Leftmost column         681       42
  //Left cells column       1630      129
  //Middle cells column     2373      195
  //Right cells column      3110      261

  //It is conspicuous that because the tft can
  //do rotation in its head but the ts can't,
  //I need to swap p.x and p.y to get X and Y right.
  //The less obvious detail is:
  //Small p.x produces large Y; large p.x produces small Y.
  //Small p.y produces small X; large p.y produces large X.

  //Although this seems complicated,
  //it can be achieved by changing the instructions
  //recommended by Adafruit to those below:
  thisX = map(p.y, TS_MINY, TS_MAXY, 0, tft.width());
  thisY = map(p.x, TS_MAXX, TS_MINX, 0, tft.height());
  thisZ = p.z;

  //Each touch produces a stream of points
  //and it is important to wait for the end of that stream,
  //just like waiting for a mechanical switch
  //to cease bouncing.
  previousMillis = millis();

  //begin opportunity to comment for debug.
  //Discover numbers to calibrate ts to match tft.
  //Serial.print("tftWidth = "); Serial.println(tft.width());
  //Serial.print("tftHeight = "); Serial.println(tft.height());
  //Serial.print("ts.bufferSize ");
  //Serial.println(ts.bufferSize());
  //Serial.print("previousMillis = ");
  //Serial.println(previousMillis);
  //Serial.print("p.x = "); Serial.println(p.x);
  //Serial.print("p.y = "); Serial.println(p.y);
  //Serial.print("thisX = "); Serial.println(thisX);
  //Serial.print("thisY = "); Serial.println(thisY);
  //Serial.print("thisZ = "); Serial.println(thisZ);
  //end opportunity to comment for debug.

  //I want to display appropriate pressure messages.
  displayPressureMessage();

  //Because I want the togglePlayer button to be active
  //for all values of gameState,
  //it is not appropriate to test gameState.
  if (thisX > humanXmin + boardCellMargin &&
      thisX < humanXmax - boardCellMargin &&
      thisY > humanYmin + boardCellMargin &&
      thisY < humanYmax - boardCellMargin)
  {
    togglePlayer();
  }

  //mapXYtoCell() does several chores:
  //If the ts has been touched unambiguously
  //within a cell within the Nought and Crosses grid,
  //and gameState == gameInProgress,
  //mapXYtoCell returns with sensible numbers in
  //thisXcursor and thisYcursor, ready to display
  //the X or O within that cell.
  //Otherwise, it returns with both values = 0.

  mapXYtoCell();

  if (thisXcursor != 0 && thisYcursor != 0)
  {
    if (thisBoard[thisXindex][thisYindex] == ' ')
      makeMove();
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//displayPressureMessage
//***** ******* ******* ******* ******* ******* ******* *******

//Since ts.getPoint returns Z as well as X and Y,
//it is fun to do something with that Z,
//which is a measure of pressure.
//I'll just do one of three simple actions.
//If pressure < pMin, display "Be firmer".
//If pressure > pMax, display "Be gentle".
//If neither of those, clear the area reserved for this message.

void displayPressureMessage()
{

  //Assume that the neither case applies,
  //and clear the pressure message area.
  tft.fillRect(pressureXmin,
               pressureYmin,
               boxSizeX,
               boxSizeY,
               ILI9341_BLACK);

  tft.setCursor(pressureCursorX, pressureCursorY);
  tft.setTextColor(ILI9341_BLUE);
  tft.setTextSize(pressureTextSize);

  //begin opportunity to comment for debug.
  //Serial.println("Pressure box erased ");
  //end opportunity to comment for debug.

  //If that assumption was wrong, display the appropriate message.
  if (thisZ < pressureTooGentle)
  {
    tft.drawRect(pressureXmin, pressureYmin,
                 boxSizeX, boxSizeY, ILI9341_BLUE);
    tft.print(F("Be firmer "));
  }
  if (thisZ > pressureTooFirm)
  {
    tft.drawRect(pressureXmin, pressureYmin,
                 boxSizeX, boxSizeY, ILI9341_BLUE);
    tft.print(F("Be gentle "));
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//togglePlayer
//***** ******* ******* ******* ******* ******* ******* *******

//If togglePlayer() is called, I know that thisX and thisY
//are integers unambiguously within the Human is _ box.
//I toggle the human and computer indicators
//and start a new game.
void togglePlayer()
{
  //Clear the message area.
  tft.fillRect(humanXmin, humanYmin,
               boxSizeX, boxSizeY, ILI9341_BLACK);
  tft.drawRect(humanXmin, humanYmin,
               boxSizeX, boxSizeY, ILI9341_BLUE);

  tft.setCursor(humanCursorX, humanCursorY1);
  tft.setTextColor(ILI9341_BLUE);
  tft.setTextSize(humanTextSize);
  //Warning! Because the clever F() trick works only
  //when everything to display is a constant,
  //I need explicit tft.print calls for each if case.

  if (humanSymbol == 'X')
  {
    humanSymbol = 'O';
    computerSymbol = 'X';
    tft.print(F("Human is O "));
    tft.setCursor(humanCursorX, humanCursorY2);
    tft.print(F("Duino is X "));
    clearBoard();
    drawBoard();
    makeComputerMove();
  }
  else
  {
    humanSymbol = 'X';
    computerSymbol = 'O';
    tft.print(F("Human is X "));
    tft.setCursor(humanCursorX, humanCursorY2);
    tft.print(F("Duino is O "));
    clearBoard();
    drawBoard();
  }

  //The game is always InProgress after the players
  //have been toggled.
  gameState = gameInProgress;

  writeYourTurnBox();

}

//***** ******* ******* ******* ******* ******* ******* *******
//writeYourTurnBox
//***** ******* ******* ******* ******* ******* ******* *******

void writeYourTurnBox()
{
  tft.fillRect(yourTurnXmin, yourTurnYmin,
               boxSizeX, boxSizeY, ILI9341_BLACK);
  tft.drawRect(yourTurnXmin, yourTurnYmin,
               boxSizeX, boxSizeY, ILI9341_BLUE);
  tft.setCursor(yourTurnCursorX, yourTurnCursorY);
  tft.setTextSize(yourTurnTextSize);

  if (gameState == gameInProgress)
  {
    tft.setTextColor(ILI9341_BLUE);
    tft.print(F("Your turn "));
  }

  if (gameState == gameWonByHuman)
  {
    tft.setTextColor(ILI9341_RED);
    tft.print(F("Human wins "));
  }

  if (gameState == gameWonByComputer)
  {
    tft.setTextColor(ILI9341_WHITE);
    tft.print(F("Duino wins "));
  }

  if (gameState == gameDrawn)
  {
    tft.setTextColor(ILI9341_WHITE);
    tft.print(F("Game drawn "));
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//clearBoard
//***** ******* ******* ******* ******* ******* ******* *******

//To clear the board at the start of each game,
//I just write a space character into each cell.

void clearBoard()
{
  int i;
  int j;
  for (i = 0; i < 3; i++)
  {
    for (j = 0; j < 3; j++)
    {
      thisBoard[i][j] = ' ';
    }
  }
  blanksAvailable = 9;
}

//***** ******* ******* ******* ******* ******* ******* *******
//drawBoard
//***** ******* ******* ******* ******* ******* ******* *******

//I'll use a procedure to draw the board so I don't need
//to duplicate code inefficiently.

void drawBoard()
{
  //Draw the board.
  tft.fillRect(boardXmin, boardYmin,
               boardSize, boardSize, ILI9341_BLACK);
  tft.drawRect(boardXmin, boardYmin,
               boardSize, boardSize, ILI9341_BLUE);
  tft.drawFastVLine(boardXmin + boardCellSize, boardYmin,
                    boardSize, ILI9341_BLUE);
  tft.drawFastVLine(boardXmin + 2 * boardCellSize, boardYmin,
                    boardSize, ILI9341_BLUE);
  tft.drawFastHLine(boardXmin, boardYmin + boardCellSize,
                    boardSize, ILI9341_BLUE);
  tft.drawFastHLine(boardXmin, boardYmin + 2 * boardCellSize,
                    boardSize, ILI9341_BLUE);
}

//***** ******* ******* ******* ******* ******* ******* *******
//drawWinLine
//***** ******* ******* ******* ******* ******* ******* *******

//When a game has been won,
//it is useful to draw a line through the winning row;
//However, that is surprisingly difficult;
//For a line, it is important to know which cell
//is at the start of the line and which is at
//the other end;
//That isn't extremely difficult,
//but it is sufficiently difficult that I prefer
//to leave it to be an opportunity for improvement;

//Instead, at the two places in the program where
//a winner is discovered, a few instructions remember
//the Xcursor and Ycursor for the three cells
//forming the winning row;

//I'll indicate that they are part of the winning row
//by changing their colour;

//Nevertheless, I'll use the name drawWinLine;

void drawWinLine()
{

  unsigned char thisSymbol;

  //begin opportunity to comment for debug.
  Serial.println("At drawWinLine ");
  //end opportunity to comment for debug.

  if (gameState == gameWonByHuman)
  {
    thisSymbol = humanSymbol;
  }

  if (gameState == gameWonByComputer)
  {
    thisSymbol = computerSymbol;
  }

  if (gameState == gameWonByHuman
      || gameState == gameWonByComputer)
  {
    //begin opportunity to comment for debug.
    //Serial.print("winXa = "); Serial.print(winXa);
    //Serial.print(", winYa = "); Serial.println(winYa);
    //Serial.print("winXb = "); Serial.print(winXb);
    //Serial.print(", winYb = "); Serial.println(winYb);
    //Serial.print("winXc = "); Serial.print(winXc);
    //Serial.print(", winYc = "); Serial.println(winYc);
    //end opportunity to comment for debug.

    tft.drawChar(winXa, winYa,
                 thisSymbol,
                 ILI9341_WHITE, ILI9341_BLACK,
                 boardTextSize);

    tft.drawChar(winXb, winYb,
                 thisSymbol,
                 ILI9341_WHITE, ILI9341_BLACK,
                 boardTextSize);

    tft.drawChar(winXc, winYc,
                 thisSymbol,
                 ILI9341_WHITE, ILI9341_BLACK,
                 boardTextSize);
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//mapXYtoCell
//***** ******* ******* ******* ******* ******* ******* *******

//I use the tables of significantXmin, significantXmax,
//and significantYmin, significantYmax to identify which,
//if any, cell has been pressed.
void mapXYtoCell()
{
  int i;
  int tempMin;
  int tempMax;

  thisXindex = 0;
  thisYindex = 0;
  thisXcursor = 0;
  thisYcursor = 0;

  //If the gameState isn't gameInProgress,
  //return with Xcursor and Ycursor both = 0.
  if (!(gameState == gameInProgress)) return;

  //begin opportunity to comment for debug.
  //Serial.print("thisX = "); Serial.println(thisX);
  //Serial.print("thisY = "); Serial.println(thisY);
  //end opportunity to comment for debug.

  for (i = 0; i < 3; i++)
  {
    tempMin = pgm_read_word_near(significantXmin + i);
    tempMax = pgm_read_word_near(significantXmax + i);

    //begin opportunity to comment for debug.
    //Serial.print("Xmin(i) = "); Serial.println(tempMin);
    //Serial.print("Xmax(i) = "); Serial.println(tempMax);
    //end opportunity to comment for debug.

    if (thisX > tempMin && thisX < tempMax)
    {
      thisXindex = i;
      thisXcursor = pgm_read_word_near(cellCursorX + i);
    }
    tempMin = pgm_read_word_near(significantYmin + i);
    tempMax = pgm_read_word_near(significantYmax + i);

    //begin opportunity to comment for debug.
    //Serial.print("Ymin(i) = "); Serial.println(tempMin);
    //Serial.print("Ymax(i) = "); Serial.println(tempMax);
    //end opportunity to comment for debug.

    if (thisY > tempMin && thisY < tempMax)
    {
      thisYindex = i;
      thisYcursor = pgm_read_word_near(cellCursorY + i);
    }
  }
  //begin opportunity to comment for debug.
  //Serial.print("Xcursor = "); Serial.println(thisXcursor);
  //Serial.print("Ycursor = "); Serial.println(thisYcursor);
  //end opportunity to comment for debug;

}

//***** ******* ******* ******* ******* ******* ******* *******
//makeMove
//***** ******* ******* ******* ******* ******* ******* *******

//If makeMove() is called, I know that thisX and thisY
//are integers unambiguously within a specific cell
//within the game board.
//I mark that cell with the human's symbol before proceeding
//to give the computer its turn.
//The Newell, Simon strategy has eight rules.
//I'll implement this strategy in an obvious way.
void makeMove()
{
  //begin opportunity to comment for debug.
  Serial.println(" ");
  Serial.println("At makeMove ");
  //end opportunity to comment for debug.

  thisBoard[thisXindex][thisYindex] = humanSymbol;
  tft.drawChar(thisXcursor, thisYcursor,
               humanSymbol,
               ILI9341_BLUE, ILI9341_BLACK,
               boardTextSize);

  blanksAvailable--;
  if (blanksAvailable <= 0)
  {
    gameState = gameDrawn;
  }
  //Although the human never wins,
  //to ensure that the program is complete,
  //it is useful to confirm that the human hasn't won.
  //If they have, don't call makeComputerMove.
  confirmGameState();

  //begin opportunity to comment for debug.
  //Serial.print("gameState = "); Serial.println(gameState);
  //Serial.print("blanksAvailable = ");
  //Serial.println(blanksAvailable);
  //end opportunity to comment for debug.

  if (gameState == gameInProgress)
  {
    makeComputerMove();
  }

  //begin opportunity to comment for debug.
  //Serial.print("foundMove = "); Serial.println(foundMove);
  //Serial.print("humanSymbol = "); Serial.println(humanSymbol);
  //Serial.print("computerSymbol = ");
  //Serial.println(computerSymbol);
  //end opportunity to comment for debug.

  //begin opportunity to comment for debug.
  //Serial.print("gameState = "); Serial.println(gameState);
  //end opportunity to comment for debug.

  //If after makeComputerMove
  //gameState isn't gameInProgress,
  //I need to do two chores:
  //1: Write to the YourTurn box;
  //2: If a game has been won,
  //Draw a line through the winning game.

  if (gameState != gameInProgress)
  {
    writeYourTurnBox();
    drawWinLine();
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//makeComputerMove
//***** ******* ******* ******* ******* ******* ******* *******

//I use the strategy often credited to Newell and Simon.
//It has eight steps.
//1:
//Win: If the player has two in a row,
//they can place a third to get three in a row.
//2:
//Block: If the opponent has two in a row,
//the player must play the third themselves to block the opponent.
//3:
//Fork: If it is possible to create two non-blocked
//lines of 2 with one move, then make that move.
//This creates two ways to win and the opponent
//can block only one of them.
//4:
//Blocking an opponent's fork: There are several cases:
//4a:
//If the opponent has two opportunities to create a fork,
//it is important to block both of those opportunities
//simultaneously, and to do that in a way that makes
//two in a row.
//4b:
//If the opponent has only one opportunity to create a fork,
//the player should block it.
//4c:
//If the opponent has no opportunity to create a fork,
//the player should make two in a row to force the
//opponent into defending, but only if this can be done
//without providing the opponent with an opportunity
//to create a fork.
//For example, if "X" has two opposite corners
//and "O" has the center, "O" might be tempted to play
//a corner move to win, but that would force "X" to play
//the opposite corner, creating a fork leading to a win.
//See rules 6 and 8, below.
//5:
//Center: A player marks the center.
//If it is the first move of the game,
//a subtlety might be relevant.
//Playing a corner move gives the second player more
//opportunities to make a mistake and may be a better choice.
//however, it makes no difference between perfect players.
//6:
//Opposite corner: If the opponent is in the corner,
//the player plays the opposite corner.
//But not if 4c, above applies.
//7:
//Empty corner: The player plays in a corner square.
//8:
//Empty side: The player plays in a middle square
//on any of the four sides.

//In more detail:
//The first player, who shall be designated "X",
//has three possible strategically distinct positions
//to mark during the first turn.
//Superficially, it might seem that there are
//nine possible positions, corresponding to
//the nine squares in the grid.
//However, by rotating the board, we will find that,
//in the first turn, every corner mark is strategically
//equivalent to every other corner mark.
//The same is true of every edge (side middle) mark.
//From a strategical point of view, there are only
//three possible first marks: corner, edge, or center.
//Player X can win or force a draw from any of
//these starting marks.

//There are only eight strategies,
//and they need to be investigated in the optimum order.

//The variable: foundMove is initially clear,
//and is set by the first successful strategy.
//An obvious if() test allows all subsequent
//strategies to be bypassed.
//It is tempting to nest those tests to create
//a clever appearance, but in practice,
//simple unnested tests work just as well,
//despite appearing to be inefficient.

void makeComputerMove()
{
  //begin opportunity to comment for debug.
  Serial.println("At makeComputerMove ");
  //end opportunity to comment for debug.

  foundMove = 0;
  thisXcursor = 0;
  thisYcursor = 0;

  tryWin();
  if (!foundMove) tryBlock();
  if (!foundMove) tryFork();
  if (!foundMove) tryTwoInaRow();
  if (!foundMove) tryCenter();
  if (!foundMove) tryOppositeCorner();
  if (!foundMove) tryCorner();
  if (!foundMove) tryEdge();

  //Control always arrives here.
  //Each try...() procedure writes
  //the following global variables:
  //foundMove,
  //thisXindex, thisYindex,
  //thisXcursor, thisYcursor.
  //It is safe for both the human and the computer
  //to share these variables to indicate their move.

  //If foundMove is true,
  //thisXindex, thisYindex,
  //thisXcursor, thisYcursor are defined
  //and have sensible values.
  if (foundMove)
  {
    thisBoard[thisXindex][thisYindex] = computerSymbol;
    tft.drawChar(thisXcursor, thisYcursor,
                 computerSymbol,
                 ILI9341_BLUE, ILI9341_BLACK,
                 boardTextSize);
    blanksAvailable--;
    if (blanksAvailable <= 0 && gameState == gameInProgress)
    {
      gameState = gameDrawn;
    }
  }
  //It is unusual but possible for
  //foundMove to remain false.
  //It indicates an attempt to play
  //on a board that is already full.
  if (!foundMove)
  {
    gameState = gameDrawn;
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//tryWin
//***** ******* ******* ******* ******* ******* ******* *******

//TryWin and tryBlock are almost identical and use the
//same table of indices: tryWinIndices.
//They share a common worker procedure: tryWinBlock().
//TryWin passes it computerSymbol.
//TryBlock passes it humanSymbol.
void tryWin()
{
  tryWinBlock(computerSymbol);

  //If foundMove, then someone has won;
  //I need to adjust gameState appropriately;
  //Although I have allocated a value to gameWonByHuman,
  //the game will never be won by the human;
  if (foundMove)
  {
    gameState = gameWonByComputer;
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryBlock
//***** ******* ******* ******* ******* ******* ******* *******

void tryBlock()
{
  tryWinBlock(humanSymbol);

  //See also tryWin;
  //foundMove there indicates a win;
  //foundMove here doesn't,
  //so it is appropriate to do something there,
  //and nothing here;
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryWinBlock
//***** ******* ******* ******* ******* ******* ******* *******

//Most of the cleverness of the strategy is in the
//table of comparisons: tryWinIndices.
//The program just loops through those comparisons.

void tryWinBlock(unsigned char thisSymbol)
{
  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryWinBlock ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (!foundMove && i < tryWinIndexMax)
  {
    //The first bytes from the table identify thisX, thisY.
    thisXindex = pgm_read_byte_near(tryWinIndices + i);
    thisYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering jMax
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    //Step over the indices and the vector count.
    j = i + 3;

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //The loop limit is nVectorBytes past j.
    jMax = j + nVectorBytes;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;

    //Continue testing this cell only if it is blank.
    if (thisBoard[thisXindex][thisYindex] == ' ')
    {

      //The subsequent bytes from the table
      //identify the aX, aY, bX, bY of the vectors
      //that might provide an opportunity to Win or Block.

      while (!foundMove && j < jMax)
      {
        aXindex = pgm_read_byte_near(tryWinIndices + j);
        aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
        bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
        bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

        if (thisBoard[aXindex][aYindex] == thisSymbol
            && thisBoard[bXindex][bYindex] == thisSymbol)
        {
          foundMove = true;
          thisXcursor =
            pgm_read_word_near(cellCursorX + thisXindex);
          thisYcursor =
            pgm_read_word_near(cellCursorY + thisYindex);

          //It is unlikely that this will be a winning move,
          //but I record all six relevant indices in
          //anticipation that it might be;
          //The bookkeeping to reduce the implied
          //redundancy loses more on the swings than
          //it gains on the roundabouts;
          winXa = pgm_read_word_near(cellCursorX + aXindex);
          winYa = pgm_read_word_near(cellCursorY + aYindex);
          winXb = pgm_read_word_near(cellCursorX + bXindex);
          winYb = pgm_read_word_near(cellCursorY + bYindex);
          winXc = thisXcursor;
          winYc = thisYcursor;
        }
        j = j + 4;
      }
    }
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryFork
//***** ******* ******* ******* ******* ******* ******* *******

//Most of the cleverness of the strategy is in the
//table of comparisons:
//I'm reasonably confident that I can use the table
//used for tryWin, but using different comparisons.
//The program just loops through those comparisons.

void tryFork()
{
  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int nOpportunities;

  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryFork ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  nOpportunities = 0; //Belt and bracers; see later.
  while (!foundMove && i < tryWinIndexMax)
  {
    //The first bytes from the table identify thisX, thisY.
    thisXindex = pgm_read_byte_near(tryWinIndices + i);
    thisYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering jMax
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    //Step over the indices and the vector count.
    j = i + 3;

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //The loop limit is nVectorBytes past j.
    jMax = j + nVectorBytes;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;

    //begin opportunity to comment for debug.
    //Serial.print("thisXindex = "); Serial.println(thisXindex);
    //Serial.print("thisYindex = "); Serial.println(thisYindex);
    //Serial.print("nVectorBytes = "); Serial.println(nVectorBytes);
    //Serial.print("jMax = "); Serial.println(jMax);
    //end opportunity to comment for debug.

    //Continue testing this cell only if it is blank.
    if (thisBoard[thisXindex][thisYindex] == ' ')
    {

      nOpportunities = 0;
      while (j < jMax)
      {
        aXindex = pgm_read_byte_near(tryWinIndices + j);
        aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
        bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
        bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

        //For a Fork, I'm not looking for both cells
        //to be computerSymbol, but for one computerSymbol
        //and one blank.
        if ((thisBoard[aXindex][aYindex] == computerSymbol
             && thisBoard[bXindex][bYindex] == ' ') ||
            (thisBoard[aXindex][aYindex] == ' '
             && thisBoard[bXindex][bYindex] == computerSymbol))
        {
          nOpportunities++;
        }
        j = j + 4;
      }

      if (nOpportunities >= 2)
      {
        foundMove = true;
        thisXcursor =
          pgm_read_word_near(cellCursorX + thisXindex);
        thisYcursor =
          pgm_read_word_near(cellCursorY + thisYindex);
      }
    }
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryTwoInaRow
//Newell and Simon, and Crowley and Seigler
//call this "blockFork", see below.
//***** ******* ******* ******* ******* ******* ******* *******

//Newell and Simon, and Crowley and Seigler
//call this "blockFork", but that leads them to a
//complicated description that is difficult to understand
//and even more difficult to implement.
//As I analysed their descriptions,
//I realized that the real goal is to make two in a row,
//but the danger of creating two opportunities for the opponent
//to fork is so omnipresent that it needs special attention.

//After untangling their convoluted descriptions,
//I concluded that they were both describing the need
//for "Look Ahead", and warning that any strategy that
//didn't include lookahead would be imperfect.

//I use an explicit procedure: countForks
//to implement this lookahead.

void tryTwoInaRow()
{
  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int nOpportunities;

  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  unsigned int cXindex;
  unsigned int cYindex;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryTwoInaRow ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  nOpportunities = 0; //Belt and bracers; see later.
  while (!foundMove && i < tryWinIndexMax)
  {
    //The first bytes from the table identify thisX, thisY.
    thisXindex = pgm_read_byte_near(tryWinIndices + i);
    thisYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering jMax
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    //Step over the indices and the vector count.
    j = i + 3;

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //The loop limit is nVectorBytes past j.
    jMax = j + nVectorBytes;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;

    //begin opportunity to comment for debug.
    //Serial.print("thisXindex = "); Serial.println(thisXindex);
    //Serial.print("thisYindex = "); Serial.println(thisYindex);
    //Serial.print("nVectorBytes = "); Serial.println(nVectorBytes);
    //Serial.print("jMax = "); Serial.println(jMax);
    //end opportunity to comment for debug.

    //Continue testing this cell only if it is blank.
    if (thisBoard[thisXindex][thisYindex] == ' ')
    {
      //The subsequent bytes from the table
      //identify the aX, aY, bX, bY of the vectors that
      //might provide an opportunity to make two in a row.

      nOpportunities = 0;
      while (j < jMax)
      {
        aXindex = pgm_read_byte_near(tryWinIndices + j);
        aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
        bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
        bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

        //For two in a row, I'm not looking for both cells
        //to be computerSymbol, but for one computerSymbol
        //and one blank.
        if ((thisBoard[aXindex][aYindex] == computerSymbol
             && thisBoard[bXindex][bYindex] == ' ') ||
            (thisBoard[aXindex][aYindex] == ' '
             && thisBoard[bXindex][bYindex] == computerSymbol))
        {
          nOpportunities++;

          if (thisBoard[aXindex][aYindex] == ' ')
          {
            cXindex = aXindex;
            cYindex = aYindex;
          }
          if (thisBoard[bXindex][bYindex] == ' ')
          {
            cXindex = bXindex;
            cYindex = bYindex;
          }
        }
        j = j + 4;
      }

      //If nOpportunities == 1, this might be the cell to play,
      //but it might not be, and the subtlety is complicated.
      //In more detail:
      //If nOpportunities = 1, I've found a two in a row cell.
      //There might be others, and this might not be the
      //best cell for me to play.
      //In particular, it might allow, and even force,
      //my opponent to create a fork that they can exploit.

      //begin opportunity to comment for debug.
      //Serial.print("nOpportunities = ");
      //Serial.println(nOpportunities);
      //end opportunity to comment for debug.

      if (nOpportunities == 1)
      {
        //Despite the name,
        //countForks doesn't count forks;
        //it counts threats of forks.
        //A value of countForks = 2 implies a fork;
        //A value < 2 implies a cell that is safe to play.
        if (countForks(cXindex, cYindex) < 2)
        {
          foundMove = true;
          thisXcursor =
            pgm_read_word_near(cellCursorX + thisXindex);
          thisYcursor =
            pgm_read_word_near(cellCursorY + thisYindex);
        }
      }

      //If nOpportunities > 2, the procedure tryFork missed
      //an opportunity to fork, and I need to report this error.
      //I indicate foundMove = true,
      //even though this is not the least little bit true.

      if (nOpportunities >= 2)
      {
        foundMove = true;
        thisXcursor =
          pgm_read_word_near(cellCursorX + thisXindex);
        thisYcursor =
          pgm_read_word_near(cellCursorY + thisYindex);
        humanSymbol = '?';
        computerSymbol = '?';
      }
    }
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryCenter
//***** ******* ******* ******* ******* ******* ******* *******

//It is ridiculously easy to test if it possible
//to occupy the center cell.

void tryCenter()
{

  //begin opportunity to comment for debug.
  Serial.println("At tryCenter ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  thisXindex = 1;
  thisYindex = 1;

  if (thisBoard[thisXindex][thisYindex] == ' ')
  {
    foundMove = true;
    thisXcursor =
      pgm_read_word_near(cellCursorX + thisXindex);
    thisYcursor =
      pgm_read_word_near(cellCursorY + thisYindex);
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryOppositeCorner
//***** ******* ******* ******* ******* ******* ******* *******

//The table appropriate for these tests is:
//tryCornerIndices, not: tryWinIndices.

void tryOppositeCorner()
{
  unsigned int i;
  unsigned int aXindex;
  unsigned int aYindex;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryOppositeCorner ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (!foundMove && i < tryCornerIndexMax)
  {
    //For each i, there are four bytes to read:
    //thisX, thisY, aX, aY.
    thisXindex = pgm_read_byte_near(tryCornerIndices + i);
    thisYindex = pgm_read_byte_near(tryCornerIndices + i + 1);
    aXindex = pgm_read_byte_near(tryCornerIndices + i + 2);
    aYindex = pgm_read_byte_near(tryCornerIndices + i + 3);

    //I need a blank in my corner and a humanSymbol
    //in the opposite corner.
    if (thisBoard[thisXindex][thisYindex] == ' '
        && thisBoard[aXindex][aYindex] == humanSymbol)
    {
      foundMove = true;
      thisXcursor =
        pgm_read_word_near(cellCursorX + thisXindex);
      thisYcursor =
        pgm_read_word_near(cellCursorY + thisYindex);
    }
    i = i + 4;
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryCorner
//***** ******* ******* ******* ******* ******* ******* *******

//I reuse the table: cornerIndices[] using only the
//lower byte and ignoring the opposite corner
//identified by the upper byte.

void tryCorner()
{
  unsigned int i;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryCorner ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (!foundMove && i < tryCornerIndexMax)
  {
    //For each i, there are two bytes to read:
    //thisX, thisY,
    //and two bytes to ignore: aX, aY.
    thisXindex = pgm_read_byte_near(tryCornerIndices + i);
    thisYindex = pgm_read_byte_near(tryCornerIndices + i + 1);

    //I need a blank in my corner.
    if (thisBoard[thisXindex][thisYindex] == ' ')
    {
      foundMove = true;
      thisXcursor =
        pgm_read_word_near(cellCursorX + thisXindex);
      thisYcursor =
        pgm_read_word_near(cellCursorY + thisYindex);
    }
    i = i + 4;
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//tryEdge
//***** ******* ******* ******* ******* ******* ******* *******

//tryEdge is similar to tryCorner, but uses its own
//table: edgeIndices[].

void tryEdge()
{
  unsigned int i;

  i = 0;
  //begin opportunity to comment for debug.
  Serial.println("At tryEdge ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (!foundMove && i < tryEdgeIndexMax)
  {
    thisXindex = pgm_read_byte_near(tryEdgeIndices + i);
    thisYindex = pgm_read_byte_near(tryEdgeIndices + i + 1);

    //I need a blank in my edge cell.
    if (thisBoard[thisXindex][thisYindex] == ' ')
    {
      foundMove = true;
      thisXcursor =
        pgm_read_word_near(cellCursorX + thisXindex);
      thisYcursor =
        pgm_read_word_near(cellCursorY + thisYindex);
    }
    i = i + 2;
  }

}

//***** ******* ******* ******* ******* ******* ******* *******
//countForks
//***** ******* ******* ******* ******* ******* ******* *******

//Most of the cleverness of the strategy is in the
//table of comparisons:
//I'm reasonably confident that I can use the table
//used for tryWin, but using different comparisons.
//The program just loops through those comparisons.

unsigned int countForks(unsigned int trialXindex,
                        unsigned int trialYindex)
{
  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int nOpportunities;

  unsigned int tempXindex;
  unsigned int tempYindex;
  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  //For tryFork, the loop that finds thisXindex
  //and thisYindex is an outer loop within which
  //the loop that finds the a... and b... indices is nested.

  //It is a bit of a nuisance that countForks is called
  //with trialXindex and trialYindex rather than
  //with an index into the tryWinIndices table.
  //It is only a bit of a nuisance, not a huge nuisance.

  //The loop that finds trialXindex and trialYindex
  //becomes a simple loop within which nothing is nested.
  //It uses nVectorBytes to skip over cells not of interest,
  //unitl it finds the appropriate index i into
  //the tryWinIndices table when
  //trialXindex = tempXindex and trialYindex = tempYindex.

  //The loop that finds the a... and b... indices
  //becomes a second simple loop not nested inside anything.

  i = 0;
  //Use ...index = 4 to indicate trial not found.
  tempXindex = 4;
  tempYindex = 4;
  //begin opportunity to comment for debug.
  Serial.println("At countForks ");
  //Serial.print("cX, cY = ");
  //Serial.println(trialXindex, trialYindex);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (i < tryWinIndexMax
         && !(tempXindex == trialXindex
              && tempYindex == trialYindex))
  {
    tempXindex = pgm_read_byte_near(tryWinIndices + i);
    tempYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //begin opportunity to comment for debug.
    //Serial.print("tempXindex = ");
    //Serial.println(tempXindex);
    //Serial.print("tempYindex = ");
    //Serial.println(tempYindex);
    //end opportunity to comment for debug.

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering jMax
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    //Step over the indices and the vector count.
    j = i + 3;

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //The loop limit is nVectorBytes past j.
    jMax = j + nVectorBytes;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;
  }

  //When tempXindex = trialXindex
  //and tempYindex = trialYindex,
  //it is certain that they point to
  //a blank cell on the board.
  if (thisBoard[tempXindex][tempYindex] == ' ')
  {

    //The subsequent bytes from the table
    //identify the aX, aY, bX, bY of the vectors
    //that might provide an opportunity to Fork.

    nOpportunities = 0;
    while (j < jMax)
    {
      aXindex = pgm_read_byte_near(tryWinIndices + j);
      aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
      bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
      bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

      //begin opportunity to comment for debug.
      //Serial.print("tempXindex = ");
      //Serial.print(tempXindex);
      //Serial.print(", tempYindex = ");
      //Serial.println(tempYindex);

      //Serial.print("aXindex = ");
      //Serial.print(aXindex);
      //Serial.print(", aYindex = ");
      //Serial.println(aYindex);

      //Serial.print("bXindex = ");
      //Serial.print(bXindex);
      //Serial.print(", bYindex = ");
      //Serial.println(bYindex);
      //end opportunity to comment for debug.

      //For a Fork, I'm not looking for both cells
      //to be humanSymbol, but for one humanSymbol
      //and one blank.
      if ((thisBoard[aXindex][aYindex] == humanSymbol
           && thisBoard[bXindex][bYindex] == ' ') ||
          (thisBoard[aXindex][aYindex] == ' '
           && thisBoard[bXindex][bYindex] == humanSymbol))
      {
        nOpportunities++;
      }
      j = j + 4;
    }
  }
  //begin opportunity to comment for debug.
  //Serial.print("nOpportunities = ");
  //Serial.println(nOpportunities);
  //end opportunity to comment for debug.

  return nOpportunities;
}

//***** ******* ******* ******* ******* ******* ******* *******
//confirmGameState
//***** ******* ******* ******* ******* ******* ******* *******

//confirmGameState is similar to, but different from,
//tryFork and countForks.

//Most of the cleverness of the strategy is in the
//table of comparisons:
//I'm reasonably confident that I can use the table
//used for tryWin, but using different comparisons.
//The program just loops through those comparisons.

void confirmGameState()
{
  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int nOpportunities;

  unsigned int trialXindex;
  unsigned int trialYindex;

  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  //For tryFork, the loop that finds thisXindex
  //and thisYindex is an outer loop within which
  //the loop that finds the a... and b... indices is nested.

  //It is a bit of a nuisance that confirmGameState uses
  //thisXindex and thisYindex and needs to find
  //the corresponding index into the tryWinIndices table.
  //It is only a bit of a nuisance, not a huge nuisance.

  //The loop that finds trialXindex and trialYindex
  //becomes a simple loop within which nothing is nested.
  //It uses nVectorBytes to skip over cells not of interest,
  //unitl it finds the appropriate index i into
  //the tryWinIndices table when
  //trialXindex = thisXindex and trialYindex = thisYindex.

  //The loop that finds the a... and b... indices
  //becomes a second simple loop not nested inside anything.

  i = 0;
  //Use ...index = 4 to indicate trial not found.
  trialXindex = 4;
  trialYindex = 4;

  //begin opportunity to comment for debug.
  Serial.println("At confirmGameState ");
  //Serial.print("ComputerSymbol = ");
  //Serial.println(computerSymbol);
  //Serial.print("HumanSymbol = ");
  //Serial.println(humanSymbol);
  //end opportunity to comment for debug.

  while (i < tryWinIndexMax
         && !(thisXindex == trialXindex
              && thisYindex == trialYindex))
  {
    trialXindex = pgm_read_byte_near(tryWinIndices + i);
    trialYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //begin opportunity to comment for debug.
    //Serial.print("trialXindex = ");
    //Serial.println(trialXindex);
    //Serial.print("trialYindex = ");
    //Serial.println(trialYindex);
    //end opportunity to comment for debug.

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering jMax
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    //Step over the indices and the vector count.
    j = i + 3;

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //The loop limit is nVectorBytes past j.
    jMax = j + nVectorBytes;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;
  }

  //When thisXindex = trialXindex
  //and thisYindex = trialYindex,
  //it is certain that they don't point to
  //a blank cell on the board,
  //but to the cell just filled with humanSymbol.
  //The if test below should be true always, and redundant.
  if (thisBoard[thisXindex][thisYindex] == humanSymbol)
  {

    //The subsequent bytes from the table
    //identify the aX, aY, bX, bY of the vectors
    //that might provide an opportunity to Fork.

    while (j < jMax)
    {
      aXindex = pgm_read_byte_near(tryWinIndices + j);
      aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
      bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
      bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

      //For a win by the human,
      //I'm looking for both cells to be humanSymbol.
      if ((thisBoard[aXindex][aYindex] == humanSymbol
           && thisBoard[bXindex][bYindex] == humanSymbol))
      {
        gameState = gameWonByHuman;

        //It is unlikely that this will be a winning move,
        //but I record all six relevant indices in
        //anticipation that it might be;
        //The bookkeeping to reduce the implied
        //redundancy loses more on the swings than
        //it gains on the roundabouts;
        winXa = pgm_read_word_near(cellCursorX + aXindex);
        winYa = pgm_read_word_near(cellCursorY + aYindex);
        winXb = pgm_read_word_near(cellCursorX + bXindex);
        winYb = pgm_read_word_near(cellCursorY + bYindex);
        winXc = pgm_read_word_near(cellCursorX + thisXindex);
        winYc = pgm_read_word_near(cellCursorY + thisYindex);
      }
      j = j + 4;
    }
  }
}

//***** ******* ******* ******* ******* ******* ******* *******
//debugPrintTables
//***** ******* ******* ******* ******* ******* ******* *******

//It isn't easy to print the try... tables for debugging.

/*
  void debugPrintTables()
  {

  unsigned int i;
  unsigned int j;
  unsigned int jMax;
  unsigned int nVectorBytes;
  unsigned int aXindex;
  unsigned int aYindex;
  unsigned int bXindex;
  unsigned int bYindex;

  i = 0;
  while (i < tryWinIndexMax)
  {
    //The first bytes from the table identify thisX, thisY.
    thisXindex = pgm_read_byte_near(tryWinIndices + i);
    thisYindex = pgm_read_byte_near(tryWinIndices + i + 1);

    //The number of vectors might be 4, 3, 2,
    //corresponding to middle, corner, edge cells.
    //The byte to read follows the indices.

    //It seems clever to defer discovering innerLoopLimit
    //until after knowing that this cell is = ' ',
    //but that interferes with the bookkeeping for i.

    nVectorBytes =
      pgm_read_byte_near(tryWinIndices + i + 2);
    //I need to add i to get the valid Loop Limit.
    jMax = i + 3 + nVectorBytes;

    //begin opportunity to comment for debug.
    Serial.print("At i = ");
    Serial.print(i); Serial.print(", ");
    Serial.print(thisXindex); Serial.print(", ");
    Serial.print(thisYindex); Serial.print(", ");
    Serial.println(nVectorBytes);
    //end opportunity to comment for debug.

    //Step over the indices and the vector count.
    j = i + 3;

    //Because of the table structure,
    //I don't do i++ at the end of the while,
    //but set the next i here, so the if test below
    //doesn't confuse the bookkeeping.
    i = jMax;

    //The subsequent bytes from the table
    //identify the aX, aY, bX, bY of the vectors
    //that might provide an opportunity to Win or Block.

    while (j < jMax)
    {
      aXindex = pgm_read_byte_near(tryWinIndices + j);
      aYindex = pgm_read_byte_near(tryWinIndices + j + 1);
      bXindex = pgm_read_byte_near(tryWinIndices + j + 2);
      bYindex = pgm_read_byte_near(tryWinIndices + j + 3);

      //begin opportunity to comment for debug.
      Serial.print("At j = ");
      Serial.print(j); Serial.print(", ");
      Serial.print(aXindex); Serial.print(", ");
      Serial.print(aYindex); Serial.print(", ");
      Serial.print(bXindex); Serial.print(", ");
      Serial.println(bYindex);
      //end opportunity to comment for debug.

      j = j + 4;
    }
  }
  }

*/
//***** ******* ******* ******* ******* ******* ******* *******
//End
//***** ******* ******* ******* ******* ******* ******* *******

//
//
//
//
//
