/*

Pseudo Random Linear Congruence Generator Functions and Helpers, by Mauro Grassi, September 2010.

A LCG generator with rule X_(n+1)= g * X_n + d will have maximum period modulo m if the following conditions are met:

(i)   (d, m)=1 (coprime)
(ii)  all the prime divisors of m divide (g-1); and
(iii) if m is divisible by 4 so is (g-1);

*/

#include "random.h"
#include <math.h>

#ifdef __cplusplus
extern "C" {
#endif   

	unsigned long 	randomNumber;
	unsigned int 	randomDelta;
	unsigned int 	randomGradient;

	unsigned long gcd(unsigned long u, unsigned long v)
	{
		int shift;

		/* GCD(0,x) := x */
		if (u == 0 || v == 0)
			return u | v;

		/* Let shift := lg K, where K is the greatest power of 2
		dividing both u and v. */
		for (shift = 0; ((u | v) & 1) == 0; ++shift) {
			u >>= 1;
			v >>= 1;
		}

		while ((u & 1) == 0)
			u >>= 1;

		/* From here on, u is always odd. */
		do {
			while ((v & 1) == 0)  /* Loop X */
				v >>= 1;

			/* Now u and v are both odd, so diff(u, v) is even.
			Let u = min(u, v), v = diff(u, v)/2. */
			if (u < v) {
				v -= u;
			} else {
				unsigned long diff = u - v;
				u = v;
				v = diff;
			}
			v >>= 1;
		} while (v != 0);

		return u << shift;
	}

	unsigned long initRandom(unsigned long modulo, unsigned long seed, unsigned long gradient, unsigned long delta)
	{
		randomNumber=(seed % modulo);
		randomDelta=delta;
		randomGradient=gradient;
		return randomNumber;
	}

	unsigned long numDivisors(unsigned long candidate)
	{
		unsigned long numdivisors;
		unsigned long i;

		numdivisors=0;
		for(i=1; i<=sqrt(candidate); i++)
		{
			if((candidate % i)==0)
			{
				numdivisors++;
			}
		}
		return numdivisors;
	}

	unsigned char isPrime(unsigned long primeCandidate)
	{
		if((primeCandidate>1)&&(numDivisors(primeCandidate)==1))return 1;
		return 0;
	}

	unsigned long getPrimeProduct(unsigned long candidate)
	{
		/* returns the product of all prime divisors of candidate */
		unsigned long product;
		unsigned long i;

		if(isPrime(candidate))
		{
			product=candidate;
		}
		else
		{
			product=1;
			for(i=2; i<=(candidate/2); i++)
			{
				if((candidate % i)==0)
				{
					if(isPrime(i))product*=i;
				}
			}
		}	
		return product;
	}

	unsigned long initRandomAutomatic(unsigned long modulo, unsigned long seed)
	{
		randomNumber=(seed % modulo);
		randomGradient=2*getPrimeProduct(modulo)+1;
		randomDelta=1;
		return randomNumber;
	}

	unsigned long getRandom(unsigned long modulo)
	{
		randomNumber=((randomNumber*randomGradient+randomDelta)%(modulo));
		return randomNumber;
	}

#ifdef __cplusplus
}
#endif 

