#ifndef __OPCODES_H__
#define __OPCODES_H__
                 
#include "vm.h"
           
#ifdef __cplusplus
extern "C" {
#endif   
  
/* Latest Version */

#define NUM_GLOBAL_VARS				18

#if(LONG_ADDRESSES)

#define MAX_ADDRESS					(0x0003FFFF+(4096+64))
#define SIZE_OF_FLOAT_MASK			0x0FFFFFFF		/* align to byte boundary */
#define SIZE_OF_BYTE_MASK			0x0FFFFFFF		/* align to byte boundary */

#else 

#define MAX_ADDRESS					(0x000FFFF+(4096+64))
#define SIZE_OF_FLOAT_MASK			0x0FFF		/* align to byte boundary */
#define SIZE_OF_BYTE_MASK			0x0FFF		/* align to byte boundary */

#endif

#define INDIRECT_OFFSET				0
#define MAX_LOCAL_ALLOCATION_SIZE	1024
#define SIZE_OF_FLOAT				4
#define SIZE_OF_STRING_VAR			33			/* This is the default for string allocation (32 characters + '\0') */
#define MAX_SIZE_OF_STRING			256
 
/* Argument Types */

#define NO_ARG_TYPE					0x00		/* none */
#define BYTE_ARG_TYPE				0x01		/* single byte */
#define SHORT_ARG_TYPE				0x02		/* 2 byte short */
#define USHORT_ARG_TYPE   		 	0x03		/* 2 byte unsigned short */
#define LIT_ARG_TYPE				0x04		/* float */
#define LITI_ARG_TYPE				0x05		/* integer constant, for efficiency */
#define REF_ARG_TYPE				0x10
#define REF_ARG_LOCAL				0x11
#define REF_ARG_GLOBAL				0x12
#define REF_ARG_STACK				0x13
#define ERR_ARG_TYPE				0xFF		/* error! */

/* Opcode Modes */

#define MODE_DS						0x00
#define MODE_SS						0x01
#define MODE_DS_POST_INC			0x02
#define MODE_SS_POST_INC			0x03
   
#define MODE_NONE_OR_STACK_TOP		0x00
#define MODE_LITERAL				0x01
#define MODE_REFERENCE				0x02
#define MODE_RESERVED				0x03
  
/* Opcodes */
    
#define OPCODE_NOP_HALT				0x00 
#define OPCODE_NEG					0x01
#define OPCODE_NOT					0x02
#define OPCODE_RCP					0x03
#define OPCODE_CLR					0x04
#define OPCODE_MOD					0x10
#define OPCODE_MUL					0x14
#define OPCODE_OR					0x18
#define OPCODE_POW					0x1C
#define OPCODE_SUB					0x20
#define OPCODE_XOR					0x24
#define OPCODE_ADD					0x28
#define OPCODE_AND					0x2C
#define OPCODE_DIV					0x30
#define OPCODE_MOVW					0x40 
#define OPCODE_MOVWR				0x50 
#define OPCODE_MOVWB		 		0x44
#define OPCODE_MOVWRB	 			0x54
#define OPCODE_PRINT_DECIMAL		0x60
#define OPCODE_PRINT_BASE			0x64
#define OPCODE_PRINT_CHAR			0x68
#define OPCODE_OPENFILE				0x70
#define OPCODE_CLEARFILE			0x71
#define OPCODE_SERIAL				0x72
#define OPCODE_MATCH_NMEA			0x73
#define OPCODE_NMEA					0x74
#define OPCODE_ISEQU				0x80
#define OPCODE_ISNEQ				0x84
#define OPCODE_ISGT					0x88
#define OPCODE_ISGTE				0x8C
#define OPCODE_ISLT					0x90
#define OPCODE_ISLTE				0x94
#define OPCODE_OPEN_PIPE			0xA0
#define OPCODE_CLOSE_PIPE			0xA1
#define OPCODE_BRA		 			0xC0
#define OPCODE_BRZ					0xC1 
#define OPCODE_BRNZ					0xC2
#define OPCODE_CALL					0xC3  
#define OPCODE_RETURN				0xC4 
#define OPCODE_GLOBAL_CALL			0xC5 
#define OPCODE_HEADER_OFFSET_WRITE	0xC6 
#define OPCODE_SET_DS_LIMIT			0xC8
#define OPCODE_MOVSLW				0xD0
#define OPCODE_MOVSGW				0xD1
#define OPCODE_MOVDLW				0xD2 
#define OPCODE_MOVDGW				0xD3   

#define OPCODE_MOVSDW				0xD4

#define OPCODE_MOVS					0xD8
#define OPCODE_CLEAR_DS_SS			0xD9
#define OPCODE_PRINT_W				0xE0
#define OPCODE_PRINT_STRING			0xE1
#define OPCODE_SLEEP				0xE2
#define OPCODE_PREC					0xE3
#define OPCODE_PRINTF_W				0xE4
#define OPCODE_SLEEPUNTIL			0xE5 
#define OPCODE_CLEARFTIMEFLAGS		0xE6
#define OPCODE_MOVTIMEW				0xE7
#define OPCODE_TIMEUNTIL			0xE8
#define OPCODE_PUSH					0xF0 
#define OPCODE_PUSH_N	 			0xF1
#define OPCODE_POP					0xF2
#define OPCODE_POP_IND				0xF3
#define OPCODE_POP_IND_B			0xF4
#define OPCODE_SIZE_SHORT			0xF5
#define OPCODE_SIZE_LONG			0xF6
#define OPCODE_NOP					0xFD
#define OPCODE_RESET				0xFE
#define OPCODE_HALT					0xFF
     	 
/* Global Variable IDs */

/* Maths related globals */

#define GL_FUNCTION_INT								0x0040
#define GL_FUNCTION_SQRT							0x0041
#define GL_FUNCTION_SIN								0x0042
#define GL_FUNCTION_COS								0x0043
#define GL_FUNCTION_TAN								0x0044
#define GL_FUNCTION_ACOS							0x0045
#define GL_FUNCTION_ASIN							0x0046
#define GL_FUNCTION_ATAN							0x0047
#define GL_FUNCTION_EXP								0x0048
#define GL_FUNCTION_LN								0x0049
#define GL_FUNCTION_LOG10							0x004A
#define GL_FUNCTION_FPART							0x004B
#define GL_FUNCTION_ABS 							0x004C
#define GL_FUNCTION_SIGMA							0x004D
#define GL_FUNCTION_SIGMA_SQUARES					0x004E
#define GL_FUNCTION_GCD								0x004F
#define GL_FUNCTION_INIT_RANDOM						0x0050
#define GL_FUNCTION_RANDOM							0x0051
#define GL_FUNCTION_NUM_DIVISORS					0x0052
#define GL_FUNCTION_IS_PRIME						0x0053
#define GL_FUNCTION_PRIME_PRODUCT					0x0054
#define GL_FUNCTION_CONVERT_BCD_TO_DECIMAL			0x0055
#define GL_FUNCTION_CONVERT_DECIMAL_TO_BCD			0x0056

/* I2C related globals */
 
#define GL_FUNCTION_INIT_I2C						0x0020
#define GL_FUNCTION_CLOSE_I2C						0x0021
#define GL_FUNCTION_PUT_I2C_ONE_BYTE				0x0022
#define GL_FUNCTION_PUT_I2C_TWO_BYTE				0x0023
#define GL_FUNCTION_GET_I2C_TWO_BYTE				0x0024
#define GL_FUNCTION_GET_I2C_BYTE_ARRAY				0x0025
#define GL_FUNCTION_PUT_I2C_BYTE_ARRAY				0x0026

/* UART related globals */

#define GL_FUNCTION_INIT_UART						0x0030
#define GL_FUNCTION_CLOSE_UART						0x0031
#define GL_FUNCTION_GET_UART						0x0032
#define GL_FUNCTION_PUT_UART						0x0033
#define GL_FUNCTION_NOT_EMPTY_UART					0x0034
#define GL_FUNCTION_GET_SIZE_UART					0x0035
#define GL_FUNCTION_LAST_RX_UART					0x0036
#define GL_FUNCTION_CLEAR_RX_UART					0x0037
#define GL_FUNCTION_GET_UART_ERROR					0x0038
#define GL_FUNCTION_RECEIVED_NMEA_UART				0x0039
#define GL_FUNCTION_DO_NMEA_MATCH					0x003A
#define GL_FUNCTION_NEW_RX_UART						0x003B

/* SPI related globals */

#define GL_FUNCTION_INIT_SPI						0x00E0
#define GL_FUNCTION_CLOSE_SPI						0x00E1
#define GL_FUNCTION_WRITE_SPI						0x00E2
#define GL_FUNCTION_WRITE_SPI_WITHOUT_CS			0x00E3
#define GL_FUNCTION_SET_SPI_CS						0x00E4

/* ADC related globals */
 
#define GL_FUNCTION_GET_ADC_REFERENCE				0x0010
#define GL_FUNCTION_OPEN_ADC						0x0011
#define GL_FUNCTION_CLOSE_ADC						0x0012
#define GL_FUNCTION_READ_ADC						0x0013
#define GL_FUNCTION_READ_ADCP						0x0014
#define GL_FUNCTION_SET_ADC_SUPPLY_VOLTAGE			0x0015
#define GL_FUNCTION_GET_ADC_SUPPLY_VOLTAGE			0x0016
#define GL_FUNCTION_GET_ADC_REFERENCE_OPTIONS		0x0017
#define GL_FUNCTION_SELF_CALIBRATE_ADC				0x0018
#define GL_FUNCTION_SET_VBG_IO						0x0019
#define GL_FUNCTION_READ_COMPARATOR					0x001A
#define GL_FUNCTION_READ_V							0x001B

/* General IO related globals */

#define GL_FUNCTION_OPEN_IO							0x0060
#define GL_FUNCTION_SET_IO							0x0061
#define GL_FUNCTION_GET_IO							0x0062
#define GL_FUNCTION_CLOSE_IO						0x0063
#define GL_FUNCTION_SET_LED							0x0064
#define GL_FUNCTION_TOGGLE_IO						0x0065	
#define	GL_FUNCTION_IF_IO_HIGH 						0x0066
#define	GL_FUNCTION_IF_IO_LOW  						0x0067
#define GL_FUNCTION_FLASH_LED_TIMES					0x0068
#define GL_FUNCTION_INTERNAL_SET_LED				0x0069
 
/* Process related globals */
 
#define GL_FUNCTION_START_PROCESS					0x0080
#define GL_FUNCTION_STOP_PROCESS					0x0081
#define GL_FUNCTION_PAUSE_PROCESS					0x0082

/* Pipe Related globals */

#define GL_FUNCTION_GET_PIPE						0x0088
  
/* Capture related globals */

#define GL_FUNCTION_OPEN_CAPTURE_LF					0x0070
#define GL_FUNCTION_OPEN_CAPTURE_MF					0x0071
#define GL_FUNCTION_OPEN_CAPTURE_HF					0x0072
#define GL_FUNCTION_READ_CAPTURE					0x0073
#define GL_FUNCTION_CLOSE_CAPTURE					0x0074
#define GL_FUNCTION_OPEN_CAPTURE_AUTO_FREQUENCY		0x0075
#define GL_FUNCTION_OPEN_CAPTURE_OPTIONS			0x0076
#define GL_FUNCTION_OPEN_CAPTURE_COUNTER_RISING		0x0077	
#define GL_FUNCTION_OPEN_CAPTURE_COUNTER_FALLING	0x0078	

/* System globals */

#define GL_FUNCTION_GLOBAL_READ						0x0090
#define GL_FUNCTION_GLOBAL_WRITE					0x0091
#define GL_FUNCTION_READ_EE							0x0092
#define GL_FUNCTION_WRITE_EE						0x0093
#define GL_FUNCTION_GLOBAL_READ_F					0x0094
#define GL_FUNCTION_GLOBAL_WRITE_F					0x0095
#define GL_FUNCTION_READ_EE_F						0x0096
#define GL_FUNCTION_WRITE_EE_F						0x0097
#define GL_FUNCTION_SIZE_EE							0x0098

/* Time Functions */

#define GL_FUNCTION_GET_LOCAL_TIME					0x00A0
#define GL_FUNCTION_SET_LOCAL_TIME					0x00A1

#define GL_FUNCTION_GET_YEAR						0x00A2
#define GL_FUNCTION_GET_MONTH						0x00A3
#define GL_FUNCTION_GET_DAY		 					0x00A4
#define GL_FUNCTION_GET_HOUR						0x00A5
#define GL_FUNCTION_GET_MINUTES						0x00A6
#define GL_FUNCTION_GET_SECONDS						0x00A7
#define GL_FUNCTION_SET_YEAR						0x00A8
#define GL_FUNCTION_SET_MONTH						0x00A9
#define GL_FUNCTION_SET_DAY		 					0x00AA
#define GL_FUNCTION_SET_HOUR						0x00AB
#define GL_FUNCTION_SET_MINUTES						0x00AC
#define GL_FUNCTION_SET_SECONDS						0x00AD
#define GL_FUNCTION_ADD_MONTHS						0x00AE
#define GL_FUNCTION_ADD_DAYS						0x00AF
#define GL_FUNCTION_ADD_HOURS						0x00B0
#define GL_FUNCTION_ADD_MINUTES						0x00B1
#define GL_FUNCTION_ADD_SECONDS						0x00B2
#define GL_FUNCTION_GET_TOTAL_SECONDS				0x00B3
#define GL_FUNCTION_GET_WEEKDAY						0x00B4
#define GL_FUNCTION_IS_LEAP_YEAR					0x00B5
#define GL_FUNCTION_GET_DAYS_IN_MONTH_YEAR			0x00B6
#define GL_FUNCTION_SET_TIME_W						0x00B7
#define GL_FUNCTION_SET_SHOW_TIME					0x00B8
#define GL_FUNCTION_SET_TOTAL_SECONDS				0x00B9
#define GL_FUNCTION_GET_TOTAL_SECONDS_DIV			0x00BA
#define GL_FUNCTION_GET_TOTAL_SECONDS_MOD			0x00BB
#define GL_FUNCTION_SET_TOTAL_SECONDS_DIV_MOD		0x00BC

/* One Wire Globals */

#define GL_FUNCTION_RESET_ONE_WIRE								0x00C0
#define GL_FUNCTION_SEND_ONE_WIRE_COMMAND						0x00C1
#define GL_FUNCTION_INIT_ONE_WIRE								0x00C2
#define GL_FUNCTION_CLOSE_ONE_WIRE								0x00C3
#define GL_FUNCTION_ONE_WIRE_CRC								0x00C4
#define GL_FUNCTION_COPY_ONE_WIRE_BUFFER_TO_ROM_CODE_BUFFER		0x00C5
#define GL_FUNCTION_COPY_ROM_CODE_BUFFER_TO_ONE_WIRE_BUFFER		0x00C6
#define GL_FUNCTION_SEND_ONE_WIRE_COMMAND_USING_ROM_CODE_BUFFER	0x00C7

typedef VM_RETURN_TYPE (*instructionFunction)(VIRTUAL_MACHINE*, VM_ARGUMENT*);

#if(IS_PC_HOST)
extern const instructionFunction opcodes[256];
extern const unsigned char opcodeArgTypeTable[256];
extern const int globalFunctionArgTable[];
extern const char globalVariablesSizes[NUM_GLOBAL_VARS];
extern unsigned char* globalVariablesAddresses[NUM_GLOBAL_VARS];
#else  
extern rom const instructionFunction opcodes[256];
extern rom const unsigned char opcodeArgTypeTable[256];
extern rom const char globalFunctionArgTable[];
extern const rom char globalVariablesSizes[NUM_GLOBAL_VARS];
extern rom const unsigned int globalVariablesAddresses[NUM_GLOBAL_VARS];
#endif

extern float returnGlobal;

/* Helper functions */
VM_RETURN_TYPE getNextROMByte(VIRTUAL_MACHINE* vm, unsigned char *c);
unsigned char  convertWToB(float w);
VM_RETURN_TYPE unlinkStack(VIRTUAL_MACHINE* vm, unsigned char times);
VM_RETURN_TYPE preludeBinaryArg(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE getLocalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size);
VM_RETURN_TYPE getGlobalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size);
VM_RETURN_TYPE getStackMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value);
VM_RETURN_TYPE putLocalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size);
VM_RETURN_TYPE putGlobalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size);
VM_RETURN_TYPE putStackMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value);
VM_RETURN_TYPE writePointerMemory(VIRTUAL_MACHINE* vm, unsigned char mode, unsigned char* c);
VM_RETURN_TYPE readPointerMemory(VIRTUAL_MACHINE* vm, unsigned char mode, unsigned char* c);
unsigned char fileNamePipeFunctionVM(VIRTUAL_MACHINE* vm, unsigned short mode);
VM_RETURN_TYPE pushStack(VIRTUAL_MACHINE* vm, unsigned char* value);
VM_RETURN_TYPE popStack(VIRTUAL_MACHINE* vm, unsigned char* value);
VM_RETURN_TYPE popStackN(VIRTUAL_MACHINE* vm, unsigned char* values, unsigned short n);

/* Instruction Functions */

VM_RETURN_TYPE nopFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE negFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE notFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE rcpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE clrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE modFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE mulFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE orFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE powFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE subFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE xorFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE addFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE andFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE divFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movwrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movwbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movwrbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE isequFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE isneqFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE isgtFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE isgteFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE isltFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE islteFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE braFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE brzFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE brnzFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE callFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg); 
VM_RETURN_TYPE returnFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE pushFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE pushnFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE popFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE popindFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE popindbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE resetFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE haltFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE glCallFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printWFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printStFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE sleepFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE precFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printfFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movslwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movsgwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movdlwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movdgwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movsdwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE openFFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE clearFFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE timeUFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE sleepUFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE clearTFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movTWFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE closePpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE openPpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE hdrWrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printdFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE printcFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE movsFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE clrDSFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE setDLFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE serialFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE nmeaFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE mnmeaFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE szShortFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
VM_RETURN_TYPE szLongFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg);
unsigned char forcePipeFunctionVM(VIRTUAL_MACHINE* vm, unsigned short mode);

extern void convertEpoch2TimeInternal(unsigned long, TIME_T*);

#ifdef __cplusplus
}
#endif

#endif
