/* $Id: parser.yy 48 2009-09-05 08:07:10Z tb $ -*- mode: c++ -*- */
/** \file parser.yy Contains the example Bison parser source

  This file was modified from the original,
  by Mauro Grassi, October 2010.
  For the USB Data Logger Project Compiler...

*/

%{ /*** C/C++ Declarations ***/

#include <stdio.h>
#include <string>
#include <vector>

#include "expression.h"

%}

/*** yacc/bison Declarations ***/

/* Require bison 2.3 or later */
%require "2.3"

/* add debug output code to generated parser. disable this for release
 * versions. */
%debug

/* start symbol is named "start" */
%start start

/* write out a header file containing the token defines */
%defines

/* use newer C++ skeleton file */
%skeleton "lalr1.cc"

/* namespace to enclose parser in */
%name-prefix="example"

/* set the parser's class identifier */
%define "parser_class_name" "Parser"

/* keep track of the current position within the input */
%locations
%initial-action
{
    // initialize the initial location object
    @$.begin.filename = @$.end.filename = &driver.streamname;
};

/* The driver is passed by reference to the parser and to the scanner. This
 * provides a simple but effective pure interface, not relying on global
 * variables. */
%parse-param { class Driver& driver }

/* verbose error messages */
%error-verbose

 /*** BEGIN EXAMPLE - Change the example grammar's tokens below ***/

%union {
    int                 integerVal;
    double              doubleVal;
    std::string*        stringVal;
    class BaseNode*     BaseNode;
    class CNStringVariableReference* CNStringVariableReference;
}

%token          END      0              "end of file"
%token          EOL                     "end of line"
%token <integerVal> INTEGER             "integer"
%token <integerVal> HEXINTEGER          "hexadecimal integer"
%token <integerVal> STACK_VARIABLE      "stack variable"
%token <doubleVal>  DOUBLE              "float"
%token <integerVal> BINARYINTEGER       "binary integer"
%token <stringVal>  KEYWORDPRINT        "keyword print"
%token <stringVal>  KEYWORDIF           "keyword if"
%token <stringVal>  KEYWORDELSE         "keyword else"
%token <stringVal>  KEYWORDWHILE        "keyword while"
%token <stringVal>  KEYWORDEQ           "keyword eq"
%token <stringVal>  KEYWORDNEQ          "keyword neq"
%token <stringVal>  KEYWORDLT           "keyword lt"
%token <stringVal>  KEYWORDLTE          "keyword lte"
%token <stringVal>  KEYWORDGT           "keyword gt"
%token <stringVal>  KEYWORDGTE          "keyword gte"
%token <stringVal>  KEYWORDAND          "keyword and"
%token <stringVal>  KEYWORDOR           "keyword or"
%token <stringVal>  KEYWORDXOR          "keyword xor"
%token <stringVal>  KEYWORDFUNCTION     "keyword function"
%token <stringVal>  KEYWORDSCRIPT       "keyword script"
%token <stringVal>  KEYWORDSLEEP        "keyword sleep"
%token <stringVal>  LOCAL_VARIABLE      "local variable"
%token <stringVal>  GLOBAL_VARIABLE     "global variable"
%token <stringVal>  LOCAL_STR_VARIABLE  "local string variable"
%token <stringVal>  GLOBAL_STR_VARIABLE "global string variable"
%token <stringVal>  DEFINE_CONSTANT     "define constant"
%token <stringVal>  LOCAL_FUNCTIONNAME  "local function name"
%token <stringVal>  GLOBAL_FUNCTIONNAME "global function name"
%token <stringVal>  OBJECT_NAME         "object name"
%token <stringVal>  COMMENT             "comment"
%token <stringVal>  STRING              "literal string"
%token <stringVal>  NEWLINE             "keyword newline"
%token <stringVal>  KEYWORDPRECISION    "keyword precision"
%token <stringVal>  KEYWORDHEADER       "keyword header"
%token <stringVal>  KEYWORDPFUNCTION    "keyword print function"
%token <stringVal>  KEYWORDOPENFILE     "keyword open file"
%token <stringVal>  KEYWORDCLEARFILE    "keyword clear file"
%token <stringVal>  KEYWORDSERIAL       "keyword serial"
%token <stringVal>  KEYWORDMATCHNMEA    "keyword match nmea"
%token <stringVal>  KEYWORDNMEA         "keyword nmea"
%token <stringVal>  KEYWORDRESET        "keyword reset"
%token <stringVal>  KEYWORDSLEEPUNTIL   "keyword sleep until"
%token <stringVal>  KEYWORDTIMEUNTIL    "keyword time until"
%token <stringVal>  KEYWORDTIME         "keyword time"
%token <stringVal>  KEYWORDOPENPIPE     "keyword open pipe"
%token <stringVal>  KEYWORDCLOSEPIPE    "keyword close pipe"
%token <stringVal>  KEYWORDBASE         "keyword base"
%token <stringVal>  KEYWORDCHAR         "keyword char"
%token <stringVal>  KEYWORDDECIMAL      "keyword decimal"

%type <BaseNode>    constant argument argumentlist argumentexp variable functioncall functionref binaryexpr
%type <BaseNode>    keywordif keywordprint keywordelse keywordwhile keywordeq keywordneq keywordlt keywordlte keywordgt keywordgte keywordand keywordor keywordxor keywordfunction comment
%type <BaseNode>    atomexpr powexpr unaryexpr mulexpr addexpr expr assignment stringassignment functiondef whileloop statement statementlist statementblock
%type <BaseNode>    conditional start script keywordscript string printargument printargumentlist printcommand keywordnewline
%type <BaseNode>    keywordsleep sleepcommand keywordprecision precisioncommand keywordpfunction
%type <BaseNode>    keywordheader header headerstatement headerstatementlist headerstatementblock
%type <BaseNode>    body total keywordopenfile keywordclearfile openfilecommand clearfilecommand nmeacommand serialcommand keywordmatchnmea keywordnmea keywordserial matchnmeacommand
%type <BaseNode>    resetcommand keywordreset keywordsleepuntil keywordtimeuntil timeargument sleepuntilcommand keywordtime
%type <BaseNode>    keywordopenpipe keywordclosepipe openpipecommand closepipecommand
%type <BaseNode>    keywordbase keyworddecimal keywordchar stringvariablevalue
%type <CNStringVariableReference> stringvariable

%destructor { delete $$; } constant argument argumentlist argumentexp variable stringvariable functioncall functionref binaryexpr
%destructor { delete $$; } keywordif keywordprint keywordelse keywordwhile keywordeq keywordneq keywordlt keywordlte keywordgt keywordgte keywordand keywordor keywordxor keywordfunction comment
%destructor { delete $$; } atomexpr powexpr unaryexpr mulexpr addexpr expr assignment stringassignment functiondef whileloop statement statementlist statementblock
%destructor { delete $$; } conditional script keywordscript string printargument printargumentlist printcommand keywordnewline keywordpfunction
%destructor { delete $$; } keywordsleep keywordsleepuntil sleepuntilcommand timeargument sleepcommand keywordprecision precisioncommand
%destructor { delete $$; } keywordheader header headerstatement headerstatementlist headerstatementblock
%destructor { delete $$; } body total keywordopenfile keywordclearfile openfilecommand clearfilecommand
%destructor { delete $$; } resetcommand keywordreset keywordtimeuntil keywordtime keywordserial keywordmatchnmea matchnmeacommand keywordnmea serialcommand nmeacommand
%destructor { delete $$; } keywordopenpipe keywordclosepipe openpipecommand closepipecommand
%destructor { delete $$; } keywordbase keyworddecimal keywordchar stringvariablevalue

 /*** END EXAMPLE - Change the example grammar's tokens above ***/

%{

#include "driver.h"
#include "scanner.h"

/* this "connects" the bison parser in the driver to the flex scanner class
 * object. it defines the yylex() function call to pull the next token from the
 * current lexer object of the driver context. */
#undef yylex
#define yylex driver.lexer->lex

%}

%% /*** Grammar Rules ***/

 /*** BEGIN EXAMPLE - Change the example grammar rules below ***/

constant : INTEGER
           {
             $$ = new CNConstant($1);
           }
           |
           HEXINTEGER
           {
             $$ = new CNConstant($1);
           }
           |
           BINARYINTEGER
           {
             $$ = new CNConstant($1);
           }
           |
           DOUBLE
           {
             $$ = new CNConstant($1);
           }


stringvariablevalue :

           keywordchar '(' LOCAL_STR_VARIABLE ')'
           {
            if (!driver.calc.existsLocalStringVariable(*$3))
            {
             error(yyloc, std::string("Error  : Unknown string variable \"") + *$3 + "\".");
             delete $3;
             delete $1;
             YYERROR;
            }
            else
            {
              $$ = new CNLocalVariableByteReference(*$3, driver.calc.localStringVariables[driver.calc.scriptNumber][*$3][POSITION_OF_ADDRESS], new CNConstant(0));
              delete $3;
              delete $1;
             }
           }
           |
           keywordchar '(' LOCAL_STR_VARIABLE '(' expr ')' ')'
           {
            if (!driver.calc.existsLocalStringVariable(*$3))
            {
                error(yyloc, std::string("Error  : Unknown string variable \"") + *$3 + "\".");
                delete $3;
                delete $1;
                YYERROR;
            }
            else
            {
              $$ = new CNLocalVariableByteReference(*$3, driver.calc.localStringVariables[driver.calc.scriptNumber][*$3][POSITION_OF_ADDRESS], $5);
              delete $3;
              delete $1;
            }
           }
           |
           keywordchar '(' GLOBAL_STR_VARIABLE ')'
           {
            if (!driver.calc.existsGlobalStringVariable(*$3))
            {
             error(yyloc, std::string("Error  : No global string variable \"") + *$3 + "\".");
             delete $3;
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalVariableByteReference(*$3, driver.calc.globalVariables[*$3][POSITION_OF_ADDRESS], new CNConstant(0));
            delete $3;
            delete $1;
           }
           }
           |
           keywordchar '(' GLOBAL_STR_VARIABLE '(' expr ')' ')'
           {
            if (!driver.calc.existsGlobalStringVariable(*$3))
            {
             error(yyloc, std::string("Error  : No global string variable \"") + *$3 + "\".");
             delete $3;
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalVariableByteReference(*$3, driver.calc.globalVariables[*$3][POSITION_OF_ADDRESS], $5);
            delete $1;
            delete $3;
           }
           }

stringvariable :
            LOCAL_STR_VARIABLE
           {
            if (!driver.calc.existsLocalStringVariable(*$1))
            {
             std::vector<ADDRESS_TYPE> myVector;
             myVector.clear();
             myVector.insert(myVector.end(), driver.calc.currentLocalVars);
             myVector.insert(myVector.end(), (ADDRESS_TYPE)driver.calc.sizeOfStringVariable);
             driver.calc.localStringVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
             error(yyloc, std::string("Warning: Unknown string variable \"") + *$1 + "\".");
             $$ = new CNLocalStringVariableReference(*$1, driver.calc.currentLocalVars, 0, driver.calc.sizeOfStringVariable);
             driver.calc.currentLocalVars+=driver.calc.sizeOfStringVariable;
             delete $1;
            }
            else
            {
             $$ = new CNLocalStringVariableReference(*$1, driver.calc.localStringVariables[driver.calc.scriptNumber][*$1][POSITION_OF_ADDRESS], 0, driver.calc.getSizeOfLocalStringVariable(*$1));
             delete $1;
            }
           }
           |
           LOCAL_STR_VARIABLE '(' expr ')'
           {
            if (!driver.calc.existsLocalStringVariable(*$1))
            {
             error(yyloc, std::string("Warning: Unknown local string variable \"") + *$1 + "\".");
             if(($3->isConstant())&&($3->evaluate()>0))
             {
                double localRequestedAllocationSize;
                localRequestedAllocationSize=(($3->evaluate())+1);
                /* if constant expression, allocate that many bytes to this variable */
                if(localRequestedAllocationSize<=(double)driver.calc.maxLocalAllocationSize)
                {
                        /* allocate the space */
                        std::vector<ADDRESS_TYPE> myVector;
                        myVector.clear();
                        myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                        myVector.insert(myVector.end(), (ADDRESS_TYPE)localRequestedAllocationSize);
                        driver.calc.localStringVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                        $$ = new CNLocalStringVariableReference(*$1, driver.calc.currentLocalVars, new CNConstant(0), (ADDRESS_TYPE)localRequestedAllocationSize);
                        driver.calc.currentLocalVars+=(ADDRESS_TYPE)localRequestedAllocationSize;
                        error(yyloc, std::string("Warning: Non standard allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes."));
                }
                else
                {
                        /* allocation size requested is too large */
                        error(yyloc, std::string("Error  : Allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes is too large. The maximum is "+getStdStringFromString(formatStringFloat((double)driver.calc.maxLocalAllocationSize, 0))+" bytes."));
                        delete $1;
                        delete $3;
                        YYERROR;
                }
             }
             else
             {
                std::vector<ADDRESS_TYPE> myVector;
                myVector.clear();
                myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                myVector.insert(myVector.end(), (ADDRESS_TYPE)driver.calc.sizeOfStringVariable);
                driver.calc.localStringVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                $$ = new CNLocalStringVariableReference(*$1, driver.calc.currentLocalVars, new CNConstant(0), (ADDRESS_TYPE)driver.calc.sizeOfStringVariable);
                driver.calc.currentLocalVars+=driver.calc.sizeOfStringVariable;
             }
             delete $1;
            }
           else
           {
            $$ = new CNLocalStringVariableReference(*$1, driver.calc.localStringVariables[driver.calc.scriptNumber][*$1][POSITION_OF_ADDRESS], $3, (ADDRESS_TYPE)driver.calc.getSizeOfLocalStringVariable(*$1));
            delete $1;
           }
           }
           |
           GLOBAL_STR_VARIABLE
           {
            if (!driver.calc.existsGlobalStringVariable(*$1))
            {
             error(yyloc, std::string("Error  : Unknown global string variable \"") + *$1 + "\".");
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalStringVariableReference(*$1, driver.calc.globalStringVariables[*$1][POSITION_OF_ADDRESS], 0, (ADDRESS_TYPE)driver.calc.getSizeOfGlobalStringVariable(*$1));
            delete $1;
           }
           }
           |
           GLOBAL_STR_VARIABLE '(' expr ')'
           {
            if (!driver.calc.existsGlobalStringVariable(*$1))
            {
             error(yyloc, std::string("Error  : Unknown global string variable \"") + *$1 + "\".");
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalStringVariableReference(*$1, driver.calc.globalStringVariables[*$1][POSITION_OF_ADDRESS], $3, (ADDRESS_TYPE)driver.calc.getSizeOfGlobalStringVariable(*$1));
            delete $1;
           }
           }

variable : DEFINE_CONSTANT
           {
            if (!driver.calc.existsDefineConstant(*$1))
            {
             driver.calc.defineConstants.insert(std::pair<std::string, double>(*$1, 0));
             error(yyloc, std::string("Error  : Unknown define constant \"") + *$1 + "\".");
             delete $1;
             YYERROR;
             //$$ = new CNDefineConstantReference(*$1, 0);
             delete $1;
            }
            else
           {
            error(yyloc, std::string("Warning: Define constant \""+ *$1 +"\"=" + getStdStringFromString(formatStringFloat(driver.calc.defineConstants[*$1], 3))+"."));
            $$ = new CNDefineConstantReference(*$1, driver.calc.defineConstants[*$1]);
            delete $1;
           }
           }
           |
           STACK_VARIABLE
           {
            error(yyloc, std::string("Warning: Stack variable referenced."));
            $$ = new CNStackVariableReference(std::string("STACK"), $1, &driver.calc);
           }
           |
           LOCAL_VARIABLE
           {
            if (!driver.calc.existsLocalVariable(*$1))
            {
            std::vector<ADDRESS_TYPE> myVector;
            myVector.clear();
            myVector.insert(myVector.end(), driver.calc.currentLocalVars);
            myVector.insert(myVector.end(), (ADDRESS_TYPE)driver.calc.sizeOfFloat);
            driver.calc.localVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
            error(yyloc, std::string("Warning: Unknown variable \"") + *$1 + "\" @ 0x"+getStdStringFromString(HexToString(driver.calc.currentLocalVars, 4))+".");
            $$ = new CNLocalVariableReference(*$1, driver.calc.currentLocalVars, 0);
            driver.calc.currentLocalVars+=driver.calc.sizeOfFloat;
            delete $1;
            }
            else
            {
             $$ = new CNLocalVariableReference(*$1, driver.calc.localVariables[driver.calc.scriptNumber][*$1][POSITION_OF_ADDRESS], 0);
             delete $1;
            }
           }
           |
           LOCAL_VARIABLE '[' expr ']'
           {
            if (!driver.calc.existsLocalVariable(*$1))
            {
             error(yyloc, std::string("Warning: Unknown variable \"") + *$1 + "\".");
             $$ = new CNLocalVariableReference(*$1, driver.calc.currentLocalVars, new CNMultiply(new CNConstant(driver.calc.sizeOfFloat), $3));
             if(($3->isConstant())&&($3->evaluate()>0))
             {
                double localRequestedAllocationSize;
                localRequestedAllocationSize=(($3->evaluate())+1)*((double)driver.calc.sizeOfFloat);
                /* if constant expression, allocate that many bytes to this variable */
                if(localRequestedAllocationSize<=(double)driver.calc.maxLocalAllocationSize)
                {
                        /* allocate the space */
                        std::vector<ADDRESS_TYPE> myVector;
                        myVector.clear();
                        myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                        myVector.insert(myVector.end(), (ADDRESS_TYPE)localRequestedAllocationSize);
                        driver.calc.localVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                        driver.calc.currentLocalVars+=(ADDRESS_TYPE)localRequestedAllocationSize;
                        error(yyloc, std::string("Warning: Non standard allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes."));
                }
                else
                {
                        /* allocation size requested is too large */
                        error(yyloc, std::string("Error  : Allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes is too large. The maximum is "+getStdStringFromString(formatStringFloat((double)driver.calc.maxLocalAllocationSize, 0))+" bytes."));
                        delete $1;
                        delete $3;
                        YYERROR;
                }
             }
             else
             {
                std::vector<ADDRESS_TYPE> myVector;
                myVector.clear();
                myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                myVector.insert(myVector.end(), (ADDRESS_TYPE)driver.calc.sizeOfFloat);
                driver.calc.localVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                driver.calc.currentLocalVars+=driver.calc.sizeOfFloat;
             }
             delete $1;
            }
           else
           {
            $$ = new CNLocalVariableReference(*$1, driver.calc.localVariables[driver.calc.scriptNumber][*$1][POSITION_OF_ADDRESS], new CNMultiply($3, new CNConstant(driver.calc.sizeOfFloat)));
            delete $1;
           }
           }
           |
           LOCAL_VARIABLE '(' expr ')'
           {
            if (!driver.calc.existsLocalVariable(*$1))
            {
             error(yyloc, std::string("Warning: Unknown variable \"") + *$1 + "\".");
             $$ = new CNLocalVariableByteReference(*$1, driver.calc.currentLocalVars, $3);
             if(($3->isConstant())&&($3->evaluate()>0))
             {
                double localRequestedAllocationSize;
                localRequestedAllocationSize=(($3->evaluate())+1);
                /* if constant expression, allocate that many bytes to this variable */
                if(localRequestedAllocationSize<=(double)driver.calc.maxLocalAllocationSize)
                {
                        /* allocate the space */
                        std::vector<ADDRESS_TYPE> myVector;
                        myVector.clear();
                        myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                        myVector.insert(myVector.end(), (ADDRESS_TYPE)localRequestedAllocationSize);
                        driver.calc.localVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                        driver.calc.currentLocalVars+=(ADDRESS_TYPE)localRequestedAllocationSize;
                        error(yyloc, std::string("Warning: Non standard allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes."));
                }
                else
                {
                        /* allocation size requested is too large */
                        error(yyloc, std::string("Error  : Allocation size requested, "+getStdStringFromString(formatStringFloat((double)localRequestedAllocationSize, 0))+" bytes is too large. The Maximum is "+getStdStringFromString(formatStringFloat((double)driver.calc.maxLocalAllocationSize, 0))+" bytes."));
                        delete $1;
                        delete $3;
                        YYERROR;
                }
             }
             else
             {
                std::vector<ADDRESS_TYPE> myVector;
                myVector.clear();
                myVector.insert(myVector.end(), driver.calc.currentLocalVars);
                myVector.insert(myVector.end(), (ADDRESS_TYPE)driver.calc.sizeOfFloat);
                driver.calc.localVariables[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$1, myVector));
                driver.calc.currentLocalVars+=driver.calc.sizeOfFloat;
             }
             delete $1;
            }
           else
           {
            $$ = new CNLocalVariableByteReference(*$1, driver.calc.localVariables[driver.calc.scriptNumber][*$1][POSITION_OF_ADDRESS], $3);
            delete $1;
           }
           }
           |
           GLOBAL_VARIABLE
           {
            if (!driver.calc.existsGlobalVariable(*$1))
            {
             error(yyloc, std::string("Error  : Unknown global variable \"") + *$1 + "\".");
             delete $1;
             YYERROR;
            }
           else
           {

            if(driver.calc.isFloatGlobalVariable(*$1))
            {
               $$ = new CNGlobalVariableReference(*$1, driver.calc.globalVariables[*$1][POSITION_OF_ADDRESS], 0);
               delete $1;
            }
            else
            {
                ADDRESS_TYPE size;
                size=driver.calc.globalVariables[*$1][POSITION_OF_SIZE];
                if((size==(ADDRESS_TYPE)sizeof(char))||(size==(ADDRESS_TYPE)sizeof(unsigned short))||(size==(ADDRESS_TYPE)sizeof(unsigned long)))
                {
                    $$ = new CNGlobalVariableReferenceAutomatic(*$1, driver.calc.globalVariables[*$1][POSITION_OF_ADDRESS], size);
                    delete $1;
                }
                else
                {
                    $$ = new CNGlobalVariableByteReference(*$1, driver.calc.globalVariables[*$1][POSITION_OF_ADDRESS], 0);
                    delete $1;
                }
            }
           }
           }
           |
           GLOBAL_VARIABLE '[' expr ']'
           {
            if (!driver.calc.existsGlobalVariable(*$1))
            {
             error(yyloc, std::string("Error  : Unknown global variable \"") + *$1 + "\".");
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalVariableReference(*$1, driver.calc.globalVariables[*$1][POSITION_OF_ADDRESS], $3);
            delete $1;
           }
           }
           |
           GLOBAL_VARIABLE '(' expr ')'
           {
            if (!driver.calc.existsGlobalVariable(*$1))
            {
             error(yyloc, std::string("Error  : Unknown global variable \"") + *$1 + "\".");
             delete $1;
             YYERROR;
            }
           else
           {
            $$ = new CNGlobalVariableByteReference(*$1, driver.calc.globalVariables[*$1][POSITION_OF_ADDRESS], $3);
            delete $1;
           }
           }

keywordscript : KEYWORDSCRIPT
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

keywordprint :  KEYWORDPRINT
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

keywordsleep : KEYWORDSLEEP
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

keywordopenpipe : KEYWORDOPENPIPE
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

keywordclosepipe : KEYWORDCLOSEPIPE
                {
                $$ = new CNString(*$1);
                delete $1;
                }

keywordif : KEYWORDIF
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordelse : KEYWORDELSE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordwhile : KEYWORDWHILE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordfunction : KEYWORDFUNCTION
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordeq: KEYWORDEQ
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordneq: KEYWORDNEQ
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordlt: KEYWORDLT
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordlte: KEYWORDLTE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordgt: KEYWORDGT
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordgte: KEYWORDGTE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordand: KEYWORDAND
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordor: KEYWORDOR
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordxor: KEYWORDXOR
            {
             $$ = new CNString(*$1);
             delete $1;
            }

atomexpr : constant
           {
            $$ = $1;
           }
           |
           functionref
           {
            $$ = $1;
           }
           |
           variable
           {
            $$ = $1;
           }
           |
           stringvariablevalue
           {
            $$ = $1;
           }
           |
           '(' expr ')'
           {
            $$ = $2;
           }
           |
           '&' variable
           {
                 if (driver.calc.existsDefineConstant(*$2->getReferenceName()))
                 {
                     error(yyloc, std::string("Error  : Cannot access address of define constant."));
                     delete $2;
                     YYERROR;
                 }
                 else
                 if (driver.calc.existsLocalVariable(*$2->getReferenceName()))
                 {
                     if(!($2->offsetExpression))
                     {
                     $$ = new CNConstant(0+driver.calc.localVariables[driver.calc.scriptNumber][*$2->getReferenceName()][POSITION_OF_ADDRESS]);
                     delete $2;
                     }
                     else
                     if($2->offsetExpression->isConstant())
                     {
                     /* accessing the address of the variable, as a numeric constant */
                     $$ = new CNConstant($2->offsetExpression->evaluate()+driver.calc.localVariables[driver.calc.scriptNumber][*$2->getReferenceName()][POSITION_OF_ADDRESS]);
                     delete $2;
                     }
                     else
                     {
                     error(yyloc, std::string("Error  : Non constant address requested!"));
                     delete $2;
                     YYERROR;
                     }
                 }
                 else
                 if (driver.calc.existsGlobalVariable(*$2->getReferenceName()))
                 {
                     error(yyloc, std::string("Error  : Cannot access address of global variable \""+ *$2->getReferenceName() + "\"."));
                     delete $2;
                     YYERROR;
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad address of argument \"") + *$2->getReferenceName() + "\".");
                     delete $2;
                     YYERROR;
                 }
             }
             |
             '&' stringvariable
             {
                 if (driver.calc.existsLocalStringVariable(*$2->getReferenceName()))
                 {
                     if(!($2->offsetExpression))
                     {
                     /* accessing the address of the string variable, as a numeric constant */
                     $$ = new CNConstant(0+driver.calc.localStringVariables[driver.calc.scriptNumber][*$2->getReferenceName()][POSITION_OF_ADDRESS]);
                     delete $2;
                     }
                     else
                     if($2->offsetExpression->isConstant())
                     {
                     /* accessing the address of the string variable, as a numeric constant */
                     $$ = new CNConstant($2->offsetExpression->evaluate()+(double)driver.calc.localStringVariables[driver.calc.scriptNumber][*$2->getReferenceName()][POSITION_OF_ADDRESS]);
                     delete $2;
                     }
                     else
                     {
                     error(yyloc, std::string("Error  : Non constant address requested!"));
                     delete $2;
                     YYERROR;
                     }
                 }
                 else
                 if (driver.calc.existsGlobalStringVariable(*$2->getReferenceName()))
                 {
                     error(yyloc, std::string("Error  : Cannot access address of global string variable \""+ *$2->getReferenceName() + "\"."));
                     delete $2;
                     YYERROR;
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad address of argument \"") + *$2->getReferenceName() + "\".");
                     delete $2;
                     YYERROR;
                 }
             }
             |
             '#' variable
             {
                 /* size of operator */
                 if (driver.calc.existsDefineConstant(*$2->getReferenceName()))
                 {
                     error(yyloc, std::string("Error  : Cannot access size of define constant."));
                     delete $2;
                     YYERROR;
                 }
                 else
                 if (driver.calc.existsLocalVariable(*$2->getReferenceName()))
                 {
                     ADDRESS_TYPE sizeofvar;

                     sizeofvar=driver.calc.getSizeOfLocalVariable(*$2->getReferenceName());
                     $$ = new CNConstant((double)sizeofvar);
                     delete $2;
                 }
                 else
                 if (driver.calc.existsGlobalVariable(*$2->getReferenceName()))
                 {
                     ADDRESS_TYPE sizeofvar;

                     sizeofvar=driver.calc.getSizeOfGlobalVariable(*$2->getReferenceName());
                     $$ = new CNConstant((double)sizeofvar);
                     delete $2;
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad size of argument \"") + *$2->getReferenceName() + "\".");
                     delete $2;
                     YYERROR;
                 }
             }
             |
             '#' stringvariable
             {
                 /* size of operator */
                 if (driver.calc.existsLocalStringVariable(*$2->getReferenceName()))
                 {
                     ADDRESS_TYPE sizeofvar;

                     sizeofvar=driver.calc.getSizeOfLocalStringVariable(*$2->getReferenceName());
                     $$ = new CNConstant((double)sizeofvar);
                     delete $2;
                 }
                 else
                 if (driver.calc.existsGlobalStringVariable(*$2->getReferenceName()))
                 {
                     ADDRESS_TYPE sizeofvar;

                     sizeofvar=driver.calc.getSizeOfGlobalStringVariable(*$2->getReferenceName());
                     $$ = new CNConstant((double)sizeofvar);
                     delete $2;
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad size of argument \"") + *$2->getReferenceName() + "\".");
                     delete $2;
                     YYERROR;
                 }
             }


powexpr : atomexpr
          {
            $$ = $1;
          }
          |
          atomexpr '^' powexpr
          {
           $$ = new CNPower($1, $3);
          }

unaryexpr :
        powexpr
        {
         $$ = $1;
        }
        |
        '+' powexpr
        {
         $$ = $2;
        }
        |
        '-' powexpr
        {
         $$ = new CNNegate($2);
        }
        |
        '!' powexpr
        {
         $$ = new CNLogicalNot($2);
        }

binaryexpr :
          unaryexpr
          {
           $$ = $1;
          }
          |
          unaryexpr keywordeq unaryexpr
          {
           $$ = new CNBooleanEqu($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordneq unaryexpr
          {
           $$ = new CNBooleanNeq($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordgte unaryexpr
          {
           $$ = new CNBooleanGte($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordlte unaryexpr
          {
           $$ = new CNBooleanLte($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordgt unaryexpr
          {
           $$ = new CNBooleanGt($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordlt unaryexpr
          {
           $$ = new CNBooleanLt($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordand unaryexpr
          {
           $$ = new CNBooleanAnd($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordor unaryexpr
          {
           $$ = new CNBooleanOr($1, $3);
           delete $2;
          }
          |
          unaryexpr keywordxor unaryexpr
          {
           $$ = new CNBooleanXor($1, $3);
           delete $2;
          }

mulexpr : binaryexpr
          {
           $$ = $1;
          }
          |
          mulexpr '*' binaryexpr
          {
           $$ = new CNMultiply($1, $3);
          }
          |
          mulexpr '/' binaryexpr
          {
           $$ = new CNDivide($1, $3);
          }
          |
          mulexpr '%' binaryexpr
          {
           $$ = new CNModulo($1, $3);
          }
          |
          mulexpr '&' binaryexpr
          {
           $$ = new CNLogicalAnd($1, $3);
          }
          |
          mulexpr '|' binaryexpr
          {
           $$ = new CNLogicalOr($1, $3);
          }
          |
          mulexpr '?' binaryexpr
          {
           $$ = new CNLogicalXor($1, $3);
          }

addexpr : mulexpr
          {
           $$ = $1;
          }
          |
          addexpr '+' mulexpr
          {
           $$ = new CNAdd($1, $3);
          }
          |
          addexpr '-' mulexpr
          {
            $$ = new CNSubtract($1, $3);
          }

expr :    addexpr
          {
           if($1->isConstant())
           {
            $$ = new CNConstant($1->evaluate());
            delete $1;
           }
           else
           {
            $$ = $1;
           }
          }
          |
            keywordtimeuntil '(' timeargument ')'
            {
                $$ = new CNTimeUntilCommand($3);
                delete $1;
            }
            |
            keywordtime '(' timeargument ')'
            {
                $$ = new CNTimeCommand($3);
                delete $1;
            }

assignment : variable '=' expr ';'
             {
                 if (driver.calc.existsDefineConstant(*$1->getReferenceName()))
                 {
                     if($3->isConstant())
                     {
                     driver.calc.defineConstants[*$1->getReferenceName()]=(double)$3->evaluate();
                     error(yyloc, std::string("Warning: Define constant \""+ *$1->getReferenceName() +"\" redefined."));
                     $$ = 0;
                     delete $1;
                     delete $3;
                     }
                     else
                     {
                     error(yyloc, std::string("Error  : Define constant assignment is not a constant."));
                     delete $1;
                     delete $3;
                     YYERROR;
                     }
                 }
                 else
                 if (driver.calc.existsLocalVariable(*$1->getReferenceName()))
                 {
                     // error(yyloc, std::string("Warning: Access local variable \""+ *$1->getReferenceName() +"\"."));
                     $$ = new CNLocalVariableAssignment($1, $3);
                 }
                 else
                 if (driver.calc.existsGlobalVariable(*$1->getReferenceName()))
                 {
                     // error(yyloc, std::string("Warning: Access global variable \""+ *$1->getReferenceName() + "\"."));
                     $$ = new CNGlobalVariableAssignment($1, $3);
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad assignment target \"") + *$1->getReferenceName() + "\".");
                     delete $1;
                     delete $3;
                     YYERROR;
                 }
             }

stringassignment : stringvariable '=' printargumentlist ';'
             {
                 if (driver.calc.existsLocalStringVariable(*$1->getReferenceName()))
                 {
                     error(yyloc, std::string("Warning: Access local string variable \""+ *$1->getReferenceName() +"\"."));
                     $$ = new CNLocalStringVariableAssignment($1, $3);
                 }
                 else
                 if (driver.calc.existsGlobalStringVariable(*$1->getReferenceName()))
                 {
                     error(yyloc, std::string("Warning: Access global string variable \""+ *$1->getReferenceName() + "\"."));
                     $$ = new CNGlobalStringVariableAssignment($1, $3);
                 }
                 else
                 {
                     error(yyloc, std::string("Error  : Bad assignment target \"") + *$1->getReferenceName() + "\".");
                     delete $1;
                     delete $3;
                     YYERROR;
                 }
             }

keywordtimeuntil : KEYWORDTIMEUNTIL
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

keywordtime : KEYWORDTIME
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

argument :  expr
            {
             $$ = $1;
            }

argumentlist :
            argument
            {
             $$ = new CNArgument($1, 0);
            }
            | argument ',' argumentlist
            {
             $$ = new CNArgument($1, $3);
            }

argumentexp :
            /* empty */
            {
             $$ = 0;
            }
            |
            argumentlist
            {
             $$ = $1;
            }

functionref :  GLOBAL_FUNCTIONNAME '(' argumentexp ')'
               {
                 if(!driver.calc.existsGlobalFunction(*$1))
                 {
                    error(yyloc, std::string("Error  : Global function \""+ *$1 + "\" does not exist."));
                    delete $1;
                    delete $3;
                    YYERROR;
                 }
                 else
                 {
                    $$ = new CNGlobalFunctionCall(*$1, $3, &driver.calc);
                    delete $1;
                 }
               }
               |
               LOCAL_FUNCTIONNAME '(' argumentexp ')'
               {
                 if(!driver.calc.existsLocalFunction(*$1))
                 {
                    error(yyloc, std::string("Error  : Local function \""+ *$1 + "\" does not exist."));
                    delete $1;
                    delete $3;
                    YYERROR;
                 }
                 else
                 {
                    $$ = new CNLocalFunctionCall(*$1, $3, &driver.calc);
                    delete $1;
                 }
               }

functioncall :
               functionref ';'
               {
                $$ = $1;
               }

functiondef :
               keywordfunction LOCAL_FUNCTIONNAME '(' ')' statementblock
               {
                 if(driver.calc.existsLocalFunction(*$2))
                 {
                    error(yyloc, std::string("Error  : Local function \""+ *$2 + "\" redefined."));
                    delete $1;
                    delete $2;
                    delete $5;
                    YYERROR;
                 }
                 else
                 {
                    std::vector<ADDRESS_TYPE> myVector;
                    myVector.clear();
                    myVector.insert(myVector.end(), driver.calc.currentFunctions);
                    myVector.insert(myVector.end(), (ADDRESS_TYPE)0);
                    driver.calc.localFunctions[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$2, myVector));
                    error(yyloc, std::string("Warning: Defined local function \"" + *$2 + "\"."));
                    $$ = new CNFunctionDefinition(*$2, $5, (ADDRESS_TYPE)0, &driver.calc);
                    driver.calc.currentFunctions++;
                    delete $1;
                    delete $2;
                 }
               }
               |
               keywordfunction LOCAL_FUNCTIONNAME '(' expr ')' statementblock
               {
                 if(driver.calc.existsLocalFunction(*$2))
                 {
                    error(yyloc, std::string("Error  : Local function \""+ *$2 + "\" redefined."));
                    delete $1;
                    delete $2;
                    delete $6;
                    YYERROR;
                 }
                 else
                 {
                  if($4->isConstant())
                  {
                   unsigned int args;
                   args=(unsigned int)$4->evaluate();
                   if((args>=0)&&(args<256))
                   {
                   error(yyloc, std::string("Warning: Defined local function \"" + *$2 + "\"."));
                   std::vector<ADDRESS_TYPE> myVector;
                   myVector.clear();
                   myVector.insert(myVector.end(), driver.calc.currentFunctions);
                   myVector.insert(myVector.end(), (ADDRESS_TYPE)$4->evaluate());
                   driver.calc.localFunctions[driver.calc.scriptNumber].insert(std::pair<std::string, std::vector<ADDRESS_TYPE>>(*$2, myVector));
                   $$ = new CNFunctionDefinition(*$2, $6, (unsigned short)$4->evaluate(), &driver.calc);
                   driver.calc.currentFunctions++;
                   delete $1;
                   delete $2;
                   delete $4;
                   }
                   else
                   {
                    error(yyloc, std::string("Error  : Function \""+ *$2 +"\" has a bad number of arguments, valid range is between 0 and 255 (inclusive)."));
                    delete $1;
                    delete $2;
                    delete $4;
                    delete $6;
                    YYERROR;
                   }
                  }
                  else
                  {
                   error(yyloc, std::string("Error  : Function \"" + *$2 +"\" has no constant number of arguments."));
                   delete $1;
                   delete $2;
                   delete $4;
                   delete $6;
                   YYERROR;
                  }
                 }
                }

whileloop : keywordwhile '(' expr ')' statementblock
            {
                if($3->isConstant())
                {
                    if((unsigned int)$3->evaluate())
                    {
                            error(yyloc, std::string("Warning: Loop conditional always true."));
                            $$ = new CNWhileLoop($3, $5);
                            delete $1;
                    }
                    else
                    {
                            error(yyloc, std::string("Warning: Loop conditional always false."));
                            $$ = new CNWhileLoop($3, $5);
                            delete $1;
                    }
                }
                else
                {
                    $$ = new CNWhileLoop($3, $5);
                    delete $1;
                }
            }

conditional :   keywordif '(' expr ')' statementblock keywordelse statementblock
                {
                    if($3->isConstant())
                    {
                        if((unsigned int)$3->evaluate())
                        {
                            error(yyloc, std::string("Warning: Conditional always true."));
                            $$ = $5;
                            delete $1;
                            delete $3;
                            delete $6;
                            delete $7;
                        }
                        else
                        {
                            error(yyloc, std::string("Warning: Conditional always false."));
                            $$ = $7;
                            delete $1;
                            delete $3;
                            delete $5;
                            delete $6;
                        }
                    }
                    else
                    {
                        $$ = new CNConditional($3, $5, $7);
                        delete $1;
                        delete $6;
                    }
                }
                |
                keywordif '(' expr ')' statementblock
                {
                    if($3->isConstant())
                    {
                        if((unsigned int)$3->evaluate())
                        {
                            error(yyloc, std::string("Warning: Conditional always true."));
                            $$ = $5;
                            delete $1;
                            delete $3;
                        }
                        else
                        {
                            error(yyloc, std::string("Warning: Conditional always false."));
                            $$ = 0;
                            delete $1;
                            delete $3;
                            delete $5;
                        }
                    }
                    else
                    {
                        $$ = new CNConditional($3, $5, 0);
                        delete $1;
                    }
                }

comment :  COMMENT
            {
             $$ = 0;
             delete $1;
            }


string :    STRING
            {
              $$ = new CNString(*$1);
              delete $1;
            }

keywordnewline : NEWLINE
                {
                  $$ = new CNNewline(std::string("\r\n"));
                  delete $1;
                }

keywordpfunction :  KEYWORDPFUNCTION '(' expr ')'
                {
                  $$ = $3;
                }

keywordbase : KEYWORDBASE
                {
                  $$ = new CNString(*$1);
                }

keyworddecimal : KEYWORDDECIMAL
                {
                  $$ = new CNString(*$1);
                }

keywordchar :  KEYWORDCHAR
                {
                  $$ = new CNString(*$1);
                }

printargument : string
                {
                    $$ = new CNPrintArgumentString($1);
                }
                |
                keywordnewline
                {
                    $$ = new CNPrintArgumentString($1);
                }
                |
                keywordpfunction
                {
                    $$ = new CNPrintBuiltInFunction($1);
                }
                |
                keywordbase '(' expr ',' expr ',' expr ')'
                {
                    $$ = new CNPrintBaseFunction($3, new CNAdd(new CNSubtract($5, new CNConstant(1)), new CNMultiply(new CNConstant(16), $7)));
                    delete $1;
                }
                |
                keywordbase '(' expr ',' expr ')'
                {
                    $$ = new CNPrintBaseFunction($3, new CNSubtract($5, new CNConstant(1)));
                    delete $1;
                }
                |
                keyworddecimal '(' expr ',' expr ')'
                {
                    $$ = new CNPrintDecimal($3, $5);
                    delete $1;
                }
                |
                keywordchar '(' expr ')'
                {
                    $$ = new CNPrintChar($3);
                    delete $1;
                }
                |
                stringvariable
                {
                    $$ = $1;
                }
                |
                expr
                {
                    $$ = new CNPrintArgumentExpression($1);
                }

printargumentlist :
                    /* empty */
                    {
                    $$ = 0;
                    }
                    |
                    printargument
                    {
                    $$ = new CNPrintList($1, 0);
                    }
                    |
                    printargumentlist ',' printargument
                    {
                    $$ = new CNPrintList($1, $3);
                    }

keywordprecision : KEYWORDPRECISION
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

precisioncommand : keywordprecision '(' expr ')' ';'
                {
                 $$ = new CNPrecisionCommand($3);
                 delete $1;
                }

keywordreset : KEYWORDRESET
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

resetcommand : keywordreset ';'
                {
                 $$ = new CNResetCommand();
                 delete $1;
                }

timeargument :  atomexpr
                {
                    $$ = new CNTimeArgument(0, 0, 0, 0, $1);
                }
                |
                atomexpr ':' atomexpr
                {
                    $$ = new CNTimeArgument(0, 0, 0, $1, $3);
                }
                |
                atomexpr ':' atomexpr ':' atomexpr
                {
                    $$ = new CNTimeArgument(0, 0, $1, $3, $5);
                }
                |
                atomexpr ':' atomexpr ':' atomexpr ':' atomexpr
                {
                    $$ = new CNTimeArgument(0, $1, $3, $5, $7);
                }
                |
                atomexpr ':' atomexpr ':' atomexpr ':' atomexpr ':' atomexpr
                {
                    $$ = new CNTimeArgument($1, $3, $5, $7, $9);
                }

keywordsleepuntil : KEYWORDSLEEPUNTIL
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }

sleepuntilcommand : keywordsleepuntil '(' timeargument ')' ';'
                {
                    $$ = new CNSleepUntilCommand($3);
                    delete $1;
                }

closepipecommand : keywordclosepipe '(' expr ')' ';'
                {
                    $$ = new CNClosePipeCommand($3);
                    delete $1;
                }

openpipecommand : keywordopenpipe '(' expr ')' ';'
                {
                    $$ = new CNOpenPipeCommand($3);
                    delete $1;
                }

sleepcommand :  keywordsleep '(' expr ')' ';'
                {
                    $$ = new CNSleepCommand($3);
                    delete $1;
                }

printcommand :  keywordprint printargumentlist ';'
                {
                    $$ = new CNPrintList(0, $2);
                    delete $1;
                }

keywordopenfile: KEYWORDOPENFILE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordclearfile: KEYWORDCLEARFILE
            {
             $$ = new CNString(*$1);
             delete $1;
            }

keywordserial: KEYWORDSERIAL
            {
             $$ = new CNString(*$1);
            }

keywordnmea: KEYWORDNMEA
            {
             $$ = new CNString(*$1);
            }

keywordmatchnmea: KEYWORDMATCHNMEA
            {
             $$ = new CNString(*$1);
            }

clearfilecommand :
             keywordclearfile printargumentlist ';'
             {
                $$ = new CNClearFileCommand($2);
                delete $1;
             }

openfilecommand :
            keywordopenfile printargumentlist ';'
            {
                $$ = new CNOpenFileCommand($2);
                delete $1;
            }

serialcommand :
            keywordserial printargumentlist ';'
            {
                $$ = new CNSerialCommand($2);
                delete $1;
            }

nmeacommand :
            keywordnmea printargumentlist ';'
            {
                $$ = new CNNMEACommand($2);
                delete $1;
            }

matchnmeacommand :
            keywordmatchnmea printargumentlist ';'
            {
                $$ = new CNMatchNMEACommand($2);
                delete $1;
            }

statement :
                assignment
                {
                 $$ = new CNStatement($1);
                }
                | conditional
                {
                 $$ = new CNStatement($1);
                }
                | whileloop
                {
                 $$ = new CNStatement($1);
                }
                | functioncall
                {
                 $$ = new CNStatement($1);
                }
                | functiondef
                {
                 $$ = new CNStatement($1);
                }
                | comment
                {
                 $$ = new CNStatement($1);
                }
                |
                printcommand
                {
                 $$ = new CNStatement($1);
                }
                |
                sleepcommand
                {
                 $$ = new CNStatement($1);
                }
                |
                sleepuntilcommand
                {
                 $$ = new CNStatement($1);
                }
                |
                precisioncommand
                {
                 $$ = new CNStatement($1);
                }
                |
                stringassignment
                {
                 $$ = new CNStatement($1);
                }
                |
                openfilecommand
                {
                 $$ = new CNStatement($1);
                }
                |
                clearfilecommand
                {
                 $$ = new CNStatement($1);
                }
                |
                serialcommand
                {
                 $$ = new CNStatement($1);
                }
                |
                nmeacommand
                {
                 $$ = new CNStatement($1);
                }
                |
                matchnmeacommand
                {
                 $$ = new CNStatement($1);
                }
                |
                resetcommand
                {
                 $$ = new CNStatement($1);
                }
                |
                openpipecommand
                {
                $$ = new CNStatement($1);
                }
                |
                closepipecommand
                {
                $$ = new CNStatement($1);
                }

statementlist :
               statement
               {
                $$ = $1;
               }
               |
               statementlist statement
               {
                $$ = new CNStatementList($1, $2);
               }


statementblock :
                statement
                {
                 $$ = $1;
                }
                | '{' statementlist '}'
                {
                 $$ = $2;
                }
                | '{' '}'
                {
                 $$ = 0;
                }

script :    keywordscript OBJECT_NAME statementblock
            {
                $$ = new CNScript(*$2, $3);
                driver.calc.scriptNames.push_back(*$2);
                delete $1;
                delete $2;
            }

keywordheader : KEYWORDHEADER
                {
                 $$ = new CNString(*$1);
                 delete $1;
                }


headerstatement : OBJECT_NAME '=' expr ';'
                {
                    if($3->isConstant())
                    {
                        if(!driver.calc.existsHeaderObject(*$1))
                        {
                        error(yyloc, std::string("Error  : Header object name \""+*$1+"\" does not exist."));
                        delete $1;
                        delete $3;
                        YYERROR;
                        }
                        else
                        {
                        $$ = new CNHeaderStatement(*$1, $3, &driver.calc);
                        delete $1;
                        }
                    }
                    else
                    {
                        error(yyloc, std::string("Error  : Non constant expression in header statement."));
                        delete $1;
                        delete $3;
                        YYERROR;
                    }
                }
                |
                 DEFINE_CONSTANT '=' expr ';'
                 {
                    if($3->isConstant())
                    {
                    if (!driver.calc.existsDefineConstant(*$1))
                    {
                    driver.calc.defineConstants.insert(std::pair<std::string, double>(*$1, $3->evaluate()));
                    $$ = new CNDefineConstantReference(*$1, 0);
                    delete $1;
                    }
                    else
                    {
                    error(yyloc, std::string("Warning: Define constant \""+ *$1 +"\" redefined."));
                    driver.calc.defineConstants[*$1]=($3->evaluate());
                    $$ = new CNDefineConstantReference(*$1, driver.calc.defineConstants[*$1]);
                    delete $1;
                    }
                    }
                    else
                    {
                        error(yyloc, std::string("Error  : Non constant expression in header statement."));
                        delete $1;
                        delete $3;
                        YYERROR;
                    }
                  }
                  |
                  comment
                 {
                  $$ = 0;
                  delete $1;
                 }

headerstatementlist :
               headerstatement
               {
                $$ = $1;
               }
               |
               headerstatementlist headerstatement
               {
                $$ = new CNHeaderStatementList($1, $2);
               }


headerstatementblock :
                headerstatement
                {
                 $$ = $1;
                }
                | '{' headerstatementlist '}'
                {
                 $$ = $2;
                }
                | '{' '}'
                {
                 $$ = 0;
                }

header :    keywordheader OBJECT_NAME headerstatementblock
            {
                $$ = new CNHeader(*$2, $3);
                /* Register The Starting Location Of The Script */
                driver.calc.scriptStarts.insert(driver.calc.scriptStarts.end(), std::pair<int, int>(yyloc.begin.line, yyloc.begin.column));
                delete $1;
                delete $2;
            }

body :      script
            {
             $$ = $1;
            }
            |
            body script
            {
              $$ = new CNScriptList($1, $2);
            }

total :     header body
            {
                $$ = new CNTotal($1, $2);
                /* Register The End Location Of The Script */
                driver.calc.scriptEnds.insert(driver.calc.scriptEnds.end(), std::pair<int, int>(yyloc.end.line, yyloc.end.column));
            }
start :
            /* empty */
            {
                $$ = 0;
            }
            |
            start total
            {
              driver.calc.expressions.push_back($2);
              /* produce any map outputs before clearing local contexts! */
              driver.calc.expressionsMapOutput.push_back(driver.calc.getLocalVariablesMap(0, 0)+"\r\n"+driver.calc.getLocalStringVariablesMap(0, 0)+"\r\n"+driver.calc.getGlobalVariablesMap(0, 0)+"\r\n"+driver.calc.getGlobalStringVariablesMap(0, 0)+"\r\n");
              driver.calc.scriptNumber++;
              driver.calc.restartContext();
            }

 /*** END EXAMPLE - Change the example grammar rules above ***/

%% /*** Additional Code ***/

void example::Parser::error(const Parser::location_type& l,
                const std::string& m)
{
    driver.error(l, m);
}
