/*   
        
	Virtual Machine Engine, multitasking, 
	(with virtual memory caches) code, 
	
	by Mauro Grassi, October 2010...
	  
	Latest Version...
	 
*/

#include "vm.h"

#if(IS_PC_HOST)

#include <math.h>
#include <windows.h>
#include <errno.h>
#include <stdio.h>
#include "usb.h"
#include "driver.h"
#include "expression.h"
#include "common.h"
#include <streambuf>
#include <iostream>
#include "global.h"
  
unsigned short 				vmTimer; 
extern "C" unsigned char 	opcodeNames[256][16];
extern unsigned long     	alarmDelta;
extern LOCAL_SETTINGS	 	localSettings;

#else
  
#include "opcodes.h"
#include <math.h>
#include "main.h"
#include "uart.h"
#include "ff.h"
#include "mystring.h"
#include "delay.h"
#include "keys.h"
#include "serialUSB.h" 
#include "capture.h"

#pragma udata vmdata

#endif

FIL							fvm;
VM_ARGUMENT 				argument;
unsigned char				fileName[VM_FILENAME_MAX];

#if(IS_PC_HOST)

#else

#pragma code usercode

#endif

VM_RETURN_TYPE clearCache(CACHE_OBJECT* cObj, unsigned char cID, ADDRESS_TYPE size, ADDRESS_TYPE maxAddress, unsigned char* start)
{
	if(cObj)
	{
		cObj->cacheID=cID;
		cObj->cacheSize=size;
		cObj->cacheMaxAddress=maxAddress;
		cObj->bottomAddress=0;
		cObj->topAddress=0;
		cObj->cacheStart=start;
		#if(IS_PC_HOST)		
		#else
		cObj->filler=0;
		#endif
	}
	return VM_OK;
}

VM_RETURN_TYPE initCache(CACHE_OBJECT* cObj, unsigned char mode, unsigned char cID, ADDRESS_TYPE size, ADDRESS_TYPE maxAddress, unsigned char* start, unsigned char* filename)
{
	/* 
		Initialize the virtual memory cache
	
		Note: if start==0 then it is a global cache...
	
	*/
	#if(IS_PC_HOST)
	#else
	FRESULT 		fresult;
	#endif
	unsigned char* 	ptr;
	
	if(cObj)
	{
		cObj->cacheID=cID;
		cObj->cacheSize=size;
		cObj->cacheMaxAddress=maxAddress;
		cObj->cacheMode=mode;
		cObj->bottomAddress=0;
		cObj->topAddress=0;
		cObj->cacheStart=start;
		#if(IS_PC_HOST)		
		#else
		cObj->filler=0;
		#endif
		
		if(cObj->cacheMode &  CACHE_MODE_STATS)
		{
			cObj->cacheArgs.cachePerformance=0;
			cObj->cacheHits=0;
			cObj->cacheTotal=0;
			cObj->cacheEndPointer=0;
		} 
		else
		if(cObj->cacheMode & CACHE_MODE_FILENAME)
		{
			cObj->cacheArgs.cacheFileName=filename;
		}
		
		if(cObj->cacheMode & CACHE_APPEND_ALWAYS)
		{
			#if(IS_PC_HOST)
				ptr=getFileNameCache(cObj, fileName);
				/* this returns 0 if the file doesn't exist */
				cObj->cacheEndPointer=(ADDRESS_TYPE)getFileSize(getStringFromUnsignedCharArrayAuto(ptr));
			#else
				ptr=getFileNameCache(cObj, fileName);
				fresult=f_stat(ptr, &finfo);
				if(fresult==FR_OK)
				{
					cObj->cacheEndPointer=(ADDRESS_TYPE)finfo.fsize;
				}
				else
				{
					f_unlink(ptr);
					cObj->cacheEndPointer=0;
				}	
			#endif					
		}
		else
		if(cObj->cacheMode & CACHE_CREATE_ALWAYS)
		{
			#if(IS_PC_HOST)
				ptr=getFileNameCache(cObj, fileName);
				if(System::IO::File::Exists(getStringFromUnsignedCharArrayAuto(ptr)))
				{
					try
					{
						System::IO::File::Delete(getStringFromUnsignedCharArrayAuto(ptr));
					} 
					catch(...)
					{
					
					}	
				}
			#else
				ptr=getFileNameCache(cObj, fileName);
				f_unlink(ptr);
				cObj->cacheEndPointer=0;
			#endif					
		}
	}
	return VM_OK;
}

VM_RETURN_TYPE addHitMiss(CACHE_OBJECT* cObj, unsigned short hit, unsigned short miss)
{
	unsigned short total;

	if(cObj)
	{
	total=hit+miss;
	
	if(((unsigned long)cObj->cacheTotal+(unsigned long)total)<65535)
	{
		cObj->cacheTotal+=total;
		cObj->cacheHits+=hit;
		if(cObj->cacheTotal>0)
		{
			cObj->cacheArgs.cachePerformance=(float)100.0*((float)cObj->cacheHits/(float)cObj->cacheTotal);
		} 
		else
		{
			cObj->cacheArgs.cachePerformance=(float)0.0;
		}
	}
	else
	{
		while(((unsigned long)cObj->cacheTotal+(unsigned long)total)>=65535)
		{
			cObj->cacheTotal/=2;
			cObj->cacheHits/=2;
		}
		cObj->cacheTotal+=total;
		cObj->cacheHits+=hit;
		if(cObj->cacheTotal>0)
		{
			cObj->cacheArgs.cachePerformance=(float)100.0*((float)cObj->cacheHits/(float)cObj->cacheTotal);
		} 
		else
		{
			cObj->cacheArgs.cachePerformance=(float)0.0;
		}
	}
	}
	else
	{
		/* global cache */


	}
	return VM_OK;
}


unsigned char disFixVM(unsigned char c)
{
	c&=0x0F;
	if((c>=0)&&(c<=9))return c+'0';
	else
	return c+'A'-10;
}

unsigned short fileNameGetExtension(unsigned char* fName, unsigned char* outext, unsigned short maxChars)
{
	/* 
		Gets the extension of the fileName, up to maxChars chars 
					
	*/
	unsigned short i;
	unsigned short j;
	unsigned char  c;
	
	i=0;
	c=' ';
	j=0;
	while((i<maxChars)&&(c!='\0'))
	{
		c=fName[i];
		if(c=='.')
		{
			j=0;
		}
		else
		{
			outext[j++]=c;
		}
		i++;
	}
	outext[j]='\0';
	return j;
}

#if(IS_PC_HOST)
unsigned char isExtension(unsigned char* fName, unsigned short maxChars, const unsigned char* ext)
#else
unsigned char isExtension(unsigned char* fName, unsigned short maxChars, const rom unsigned char* ext)
#endif
{	
	/* Returns 1 if the extension matches! */
	unsigned char  result;
	unsigned short j;
	
	result=1;
	j=fileNameGetExtension(fName, (unsigned char*)tempString, maxChars);
	#if(IS_PC_HOST)
		result=1;
	#else
		result=cmpStringRom(tempString, (const rom unsigned char*)ext);
	#endif
	return result;
}

unsigned char* getFileNameCache(CACHE_OBJECT* cObj, unsigned char* fileN)
{
	if(cObj)
	{
		if((cObj->cacheMode & CACHE_MODE_FILENAME)&&(cObj->cacheArgs.cacheFileName[0]!='\0'))
		{
			return cObj->cacheArgs.cacheFileName;		
		}
		else
		{
		switch(cObj->cacheID & 0xC0)
		{
			default:
			case FILE_CACHE_ID:
				/* this is the default */
				fileN[0]='l';
				fileN[1]='o';
				fileN[2]='g';
				fileN[3]=disFixVM((cObj->cacheID & VM_ID_MASK)>>4);
			    fileN[4]=disFixVM((cObj->cacheID & VM_ID_MASK));
				fileN[5]='.';
				fileN[6]='t';
				fileN[7]='x';
				fileN[8]='t';
				fileN[9]='\0';
				break;
				
			case ROM_CACHE_ID:
			    fileN[0]='c';
			    fileN[1]='o';
			    fileN[2]=disFixVM((cObj->cacheID & VM_ID_MASK)>>4);
			    fileN[3]=disFixVM((cObj->cacheID & VM_ID_MASK));
			    fileN[4]='.';
			    fileN[5]='b';
			    fileN[6]='i';
			    fileN[7]='n';
			    fileN[8]='\0';
			    break;
				
			case RAM_CACHE_ID:
			    fileN[0]='d';
			    fileN[1]='o';
			    fileN[2]=disFixVM((cObj->cacheID & VM_ID_MASK)>>4);
			    fileN[3]=disFixVM((cObj->cacheID & VM_ID_MASK));
			    fileN[4]='.';
			    fileN[5]='b';
			    fileN[6]='i';
			    fileN[7]='n';
			    fileN[8]='\0';
				break;
				
			case STACK_CACHE_ID:
			    fileN[0]='s';
			    fileN[1]='o';
			    fileN[2]=disFixVM((cObj->cacheID & VM_ID_MASK)>>4);
			    fileN[3]=disFixVM((cObj->cacheID & VM_ID_MASK));
			    fileN[4]='.';
			    fileN[5]='b';
			    fileN[6]='i';
			    fileN[7]='n';
			    fileN[8]='\0';
			    break;		
			}
		}
	}	
	else
	{
		fileN[0]='\0';
	}
	return fileN;
}

unsigned char* igetFileNameVM(unsigned char exID, unsigned char* fileN)
{
	    fileN[0]='v';
	    fileN[1]='m';
	    fileN[2]=disFixVM((exID & VM_ID_MASK)>>4);
	    fileN[3]=disFixVM((exID & VM_ID_MASK));
	    fileN[4]='.';
	    fileN[5]='b';
	    fileN[6]='i';
	    fileN[7]='n';
	    fileN[8]='\0';
	    return fileN;
}

#if(IS_PC_HOST)

#else

unsigned char* igetFileNameNM(unsigned char exID, unsigned char* fileN)
{
	    fileN[0]='n';
	    fileN[1]='m';
	    fileN[2]=disFixVM((exID & VM_ID_MASK)>>4);
	    fileN[3]=disFixVM((exID & VM_ID_MASK));
	    fileN[4]='.';
	    fileN[5]='t';
	    fileN[6]='x';
	    fileN[7]='t';
	    fileN[8]='\0';
	    return fileN;
}

unsigned int getScriptName(unsigned char* outstr, unsigned char exID, unsigned char maxChars)
{
	/* returns the number of chars in the name returned */
	FRESULT 	 	fresult;
	unsigned int    uz;
	unsigned char*  ptr;
	FIL				ftemp;
	
	uz=0;
	ptr=igetFileNameNM(exID, fileName);
	if((memoryCardSystemUpMacro())&&(f_open(&ftemp, (char*)ptr, FA_READ | FA_OPEN_EXISTING)==FR_OK))
	{
		fresult=f_read(&ftemp, outstr, maxChars-1, &uz);
		if(fresult==FR_OK)
		{
			outstr+=uz;
		}
		else
		{
			uz=0;
		}
		f_close(&ftemp);
	}
	
	if(uz>0)
	{

	}
	else
	{
		*outstr++='[';
		*outstr++='0';
		*outstr++='x';
		*outstr++=fileName[2];
		*outstr++=fileName[3];
		*outstr++=']';
		uz=6;
	}	
	*outstr++='\0';
	return uz;
}
#endif

unsigned char* getFileNameVM(VIRTUAL_MACHINE* vm, unsigned char* fileN)
{
	if(vm)
	{
		return igetFileNameVM(vm->execID, fileN);
	}	
	else
	{
		fileN[0]='\0';
	}
	return fileN;
}

#if(IS_PC_HOST)

VM_RETURN_TYPE clearFileCache(VIRTUAL_MACHINE* vm)
{
 	return	initCache(&vm->FILEImage, FILECACHE_MODE, FILE_CACHE_ID + (VM_ID_MASK & vm->execID), FILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_FILE_OFFSET], &vm->fName[0]);
}

VM_RETURN_TYPE readCacheFromDisk(CACHE_OBJECT* cObj,  unsigned char* cStart, ADDRESS_TYPE offset, ADDRESS_TYPE endOffset)
{
	VM_RETURN_TYPE 	result;
	ADDRESS_TYPE 	i;
	unsigned int 	sz;
	unsigned int	uz;
	unsigned char	fileName[256];
	unsigned char*  ptr;
	
	if(cObj)
	{
			result=CACHE_ERROR;
			if(endOffset>offset)sz=(endOffset-offset); else sz=0;	
			if(sz)
			{
				ptr=getFileNameCache(cObj, fileName);
				result=CACHE_ERROR_SEEK_READ;
				uz=0;	
				if(sz>=0)
				{
					ReadFile(getStringFromUnsignedCharArrayAuto(ptr), offset, cStart, sz, &uz);
					result=VM_OK;
				}
				i=uz;
				sz-=uz;
				while(sz--)
				{
					cStart[i++]=0;
				}

			}
			else
			{
				result=VM_OK;		
			}
	}
	else
	{
		/* global cache */
		result=VM_OK;
	}
	return result;
}

VM_RETURN_TYPE writeCacheToDisk(CACHE_OBJECT* cObj, unsigned char* cStart, ADDRESS_TYPE offset, ADDRESS_TYPE endOffset)
{
	VM_RETURN_TYPE result;
	unsigned int   sz;
	unsigned int   uz;
	unsigned char  fileName[256];
	unsigned char* ptr;

	if(cObj)
	{
		result=CACHE_ERROR;
		if(cObj->cacheMode & CACHE_DIRTY)
		{
			/* actually write it to disk */
			if(cObj->cacheMode & CACHE_READ_ONLY)
			{
				result=VM_OK;
			}
			else
			{
			if(cObj->cacheMode & CACHE_MODE_LIMIT_UPPER)
			{
				if((unsigned long)endOffset<cObj->cacheEndPointer)
				{
				
				}
				else
				{
					endOffset=cObj->cacheEndPointer;
				}
			}
			if(endOffset>offset)sz=(endOffset-offset); else sz=0;	
			if(sz)
			{
				ptr=getFileNameCache(cObj, fileName);
				result=CACHE_ERROR_SEEK_WRITE;
				uz=0;	
				if(sz>=0)
				{
					result=CACHE_ERROR_WRITE;
					WriteFile(getStringFromUnsignedCharArrayAuto(ptr), offset, cStart, sz, &uz);
					if(cObj->cacheMode & CACHE_MODE_LIMIT_UPPER)
					{
						/* truncate the file here */					
						TruncateFileAt(getStringFromUnsignedCharArrayAuto(ptr), endOffset);
					}
					result=VM_OK;	
				}
			}
			else
			{
				result=VM_OK;
			}
			}
			cObj->cacheMode&=(~CACHE_DIRTY);
		}
		else
		{
			result=VM_OK;
		}
	}
	else 
	{
		/* global cache */
		result=VM_OK;
	}
	return result;
}

#else 

VM_RETURN_TYPE readCacheFromDisk(CACHE_OBJECT* cObj,  unsigned char* cStart, ADDRESS_TYPE offset, ADDRESS_TYPE endOffset)
{
	VM_RETURN_TYPE  result;
	FRESULT 	 	fresult;
	ADDRESS_TYPE    i;
	unsigned int    sz;
	unsigned int    uz;
	unsigned char*  ptr;
	
	if(cObj)
	{
			result=CACHE_ERROR;
			if(endOffset>offset)sz=(endOffset-offset); else sz=0;	
			if(sz)
			{			
			ptr=getFileNameCache(cObj, fileName);		
			f_close(&fvm);
			if((memoryCardSystemUp)&&(f_open(&fvm, (char*)ptr, FA_READ | FA_OPEN_ALWAYS)==FR_OK))
			{
				result=CACHE_ERROR_SEEK_READ;
				uz=0;	
				if(sz>=0)
				{
					fresult= f_lseek(&fvm, offset);
					if(fresult==FR_OK)
					{
						result=CACHE_ERROR_READ;			
						fresult= f_read(&fvm, cStart, sz, &uz);
						if(fresult==FR_OK)
						{	
							result=VM_OK;		
						}
						else
						{
							
						}
					}
					else
					{
					
					}
				}
				i=uz;
				sz-=uz;
				while(sz--)
				{
					cStart[i++]=0;
				}
			}
			else
			{
				i=0;
				while(sz--)
				{
					cStart[i++]=0;
				}
			}
			f_close(&fvm);	
			}
			else
			{
				result=VM_OK;		
			}
	}
	else
	{
		result=VM_OK;
	}
	return result;
}

VM_RETURN_TYPE writeCacheToDisk(CACHE_OBJECT* cObj, unsigned char* cStart, ADDRESS_TYPE offset, ADDRESS_TYPE endOffset)
{
	VM_RETURN_TYPE 	result;
	FRESULT 		fresult;
	ADDRESS_TYPE 	i;
	unsigned int 	sz;
	unsigned int 	uz;
	unsigned char*  ptr;
	
	if(cObj)
	{
		result=CACHE_ERROR;
		if(cObj->cacheMode & CACHE_DIRTY)
		{
			/* Actually write it to disk */
			if(cObj->cacheMode & CACHE_READ_ONLY)
			{
				result=VM_OK;
			}
			else
			{	
			
			if(cObj->cacheMode & CACHE_MODE_LIMIT_UPPER)
			{
				if((unsigned long)endOffset<cObj->cacheEndPointer)
				{
				
				}
				else
				{
					endOffset=cObj->cacheEndPointer;
				}
			}
			if(endOffset>offset)sz=(endOffset-offset); else sz=0;	
			
			if(sz)
			{			
			ptr=getFileNameCache(cObj, fileName);
			f_close(&fvm);
			if((memoryCardSystemUpMacro())&&(f_open(&fvm, (char*)ptr, FA_WRITE | FA_OPEN_ALWAYS)==FR_OK))
			{
				result=CACHE_ERROR_SEEK_WRITE;
				uz=0;	
				if(sz>=0)
				{
					fresult= f_lseek(&fvm, offset);
					if(fresult==FR_OK)
					{
						result=CACHE_ERROR_WRITE;
						
						fresult= f_write(&fvm, cStart, sz, &uz);
						if((fresult==FR_OK)&&(sz==uz))
						{	
							if(cObj->cacheMode & CACHE_MODE_LIMIT_UPPER)
							{
								/* truncate the file here */					
								fresult=f_truncate(&fvm);
								if(fresult==FR_OK)result=VM_OK;
							}
							else result=VM_OK;
						}
						else
						{
							
						}
					}
					else
					{
					
					}
				}
			}
			else
			{
			
			}
			f_close(&fvm);
			}
			else
			{
				result=VM_OK;
			}
			}
			cObj->cacheMode&=(~CACHE_DIRTY);
		}
		else
		{
			result=VM_OK;
		}
	}
	else
	{
		result=VM_OK;
	}
	return result;
}
#endif

unsigned char* getGlobalPtr(VIRTUAL_MACHINE* vm, unsigned short offset)
{
	/* Returns a pointer to the global variable, null otherwise */
	
	/* Must use unsigned int here for the PC host to work */
	unsigned int   index;
	unsigned int   loffs;
	unsigned char  sizeMode;
	unsigned char* ptr;
	
	if((offset>=0)&&(offset<GLOBAL_VM_UPPER))
	{
		index=(offset>>GLOBAL_ADDRESS_SHIFT);
		loffs=(offset & GLOBAL_ADDRESS_OFFSET);
			
		if((index>=0)&&(index<NUM_GLOBAL_VARS))
		{
			/* within range! */
			sizeMode=((GLOBAL_ADDRESS_OFFSET) & globalVariablesSizes[index]);
			if((sizeMode==0)||((loffs>=0)&&(loffs<(unsigned int)sizeMode)))
			{
				/* within size */
				ptr=(unsigned char*)globalVariablesAddresses[index];
				ptr+=loffs;		
			}
		}
		else
		{
			ptr=0;
		}
	}
	else
	{
		offset-=(unsigned short)GLOBAL_VM_UPPER;
		ptr=(unsigned char*)vm+(unsigned int)offset;
	}
	return ptr;
}

VM_RETURN_TYPE readCacheByte(CACHE_OBJECT* cObj, ADDRESS_TYPE offset, unsigned char* readByte)
{
	VM_RETURN_TYPE result;
	unsigned char* target;
	
	if(cObj)
	{
		if((offset>=(cObj->bottomAddress))&&(offset<(cObj->topAddress)))
		{
			/* a cache hit! */
			if(cObj->cacheMode & CACHE_MODE_STATS)
			{
				addHitMiss(cObj, 1, 0);
			}
			(*readByte)=*(cObj->cacheStart+(offset-cObj->bottomAddress));
			result=VM_OK;
		}
		else
		{
			/* a cache miss! */
			if(cObj->cacheMode & CACHE_MODE_STATS)
			{
				addHitMiss(cObj, 0, 1);
			}

			/* save the image */
			result=writeCacheToDisk(cObj, cObj->cacheStart, cObj->bottomAddress, cObj->topAddress);
			if(result==VM_OK)
			{	
				/* reload a new image */
				if(offset>=(cObj->cacheSize/2))
				{
						if(cObj->cacheMode & CACHE_MODE_FILENAME)
						{
							cObj->bottomAddress=offset;
						}
						else
						{
							cObj->bottomAddress=offset-(cObj->cacheSize/2); 
						}	
				} 
				else 
				{
					cObj->bottomAddress=0;
				}				
				
				if((cObj->cacheSize+cObj->bottomAddress)<=cObj->cacheMaxAddress)
				{
					cObj->topAddress=cObj->cacheSize+cObj->bottomAddress;
				} 
				else
				{
					cObj->topAddress=cObj->cacheMaxAddress;
				}
						
				result=readCacheFromDisk(cObj, cObj->cacheStart, cObj->bottomAddress, cObj->topAddress);
				if(result==VM_OK)
				{
					if((offset>=(cObj->bottomAddress))&&(offset<(cObj->topAddress)))
					{
						(*readByte)=*(cObj->cacheStart+(offset-cObj->bottomAddress));
						result=VM_OK;
					}
					else
					{
						result=VM_ADDRESS_ERROR;
					}
				}
			}
		}
	}
	else
	{
		/* Global Cache here */
		target=(unsigned char*)(unsigned short)offset;
		#if(IS_PC_HOST)
			
			/* Emulate the Global Memory In The putGlobalMemory function... */

		#else
			
			*readByte=*target;
		
		#endif
		result=VM_OK;
	}
	return result;
}

VM_RETURN_TYPE writeCacheByte(CACHE_OBJECT* cObj, ADDRESS_TYPE offset, unsigned char* writeByte)
{
	VM_RETURN_TYPE result;
	unsigned char* target;

	if(cObj)
	{
		if(cObj->cacheMode & CACHE_READ_ONLY)
		{
			result=VM_OK;	
		}
		else
		{
		if((offset>=(cObj->bottomAddress))&&(offset<(cObj->topAddress)))
		{
			/* a cache hit! */
			if(cObj->cacheMode & CACHE_MODE_STATS)
			{
				addHitMiss(cObj, 1, 0);
			}	

		target=(cObj->cacheStart+(offset-cObj->bottomAddress));
		if((*target)!=(*writeByte))
		{
			(*target)=(*writeByte);
			cObj->cacheMode|=CACHE_DIRTY;
		}
		result=VM_OK;
		}
		else
		{
		/* a cache miss! */
		if(cObj->cacheMode & CACHE_MODE_STATS)
		{
			addHitMiss(cObj, 0, 1);
		}
		/* save the image */
		result=writeCacheToDisk(cObj, cObj->cacheStart, cObj->bottomAddress, cObj->topAddress);
		if(result==VM_OK)
		{	
			/* reload a new image */
			if(offset>=(cObj->cacheSize/2))
			{
				if(cObj->cacheMode & CACHE_MODE_FILENAME)
				{
					cObj->bottomAddress=offset;
				}
				else
				{
					cObj->bottomAddress=offset-(cObj->cacheSize/2); 
				}	
			}
			else 
			{
				cObj->bottomAddress=0;
			}

			if((cObj->cacheSize+cObj->bottomAddress)<=cObj->cacheMaxAddress)
			{
				cObj->topAddress=cObj->cacheSize+cObj->bottomAddress;
			} 
			else
			{
				cObj->topAddress=cObj->cacheMaxAddress;
			}			
			result=readCacheFromDisk(cObj, cObj->cacheStart, cObj->bottomAddress, cObj->topAddress);
			if(result==VM_OK)
			{
				if((offset>=(cObj->bottomAddress))&&(offset<(cObj->topAddress)))
				{
					target=(cObj->cacheStart+(offset-cObj->bottomAddress));
					if((*target)!=(*writeByte))
					{
						(*target)=(*writeByte);
						cObj->cacheMode|=CACHE_DIRTY;
						result=VM_OK;
					}
				}
				else
				{
					result=VM_ADDRESS_ERROR;
				}
			}
		}
		}
	}
	}
	else
	{
		/* Global Cache here */
		target=(unsigned char*)(unsigned short)offset;		
		#if(IS_PC_HOST)

			/* Emulate the Global Memory In The getGlobalMemory function... */
			
		#else
			*target=*writeByte;
		#endif
		result=VM_OK;
	}
	return result;
}

VM_RETURN_TYPE readCache(CACHE_OBJECT* cObj, ADDRESS_TYPE offset, unsigned char* cStart, ADDRESS_TYPE sizeBytes)
{
	VM_RETURN_TYPE result;
	 
	result=VM_OK;
	while((result==VM_OK)&&(sizeBytes--))
	{
		result=readCacheByte(cObj, offset, cStart);
		cStart++;
		offset++;
	}
	return result;
}

VM_RETURN_TYPE writeCache(CACHE_OBJECT* cObj, ADDRESS_TYPE offset, unsigned char* cStart, ADDRESS_TYPE sizeBytes)
{
	VM_RETURN_TYPE result;

	result=VM_OK;
	while((result==VM_OK)&&(sizeBytes--))
	{
		result=writeCacheByte(cObj, offset, cStart);
		cStart++;
		offset++;
	}
	return result;
}

VM_RETURN_TYPE getInstructionAndArgument(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, VM_ARGUMENT* aG, unsigned char* bs)
{
	/* 
		get the instruction opcode and argtype at offset into opcd and its argument 
		returns the number of bytes in (*bs) 

	*/
	VM_RETURN_TYPE result;
	unsigned char  iArg[3];
	unsigned long  x;
	
	aG->opcode=OPCODE_NOP;
	aG->type=NO_ARG_TYPE;
	(*bs)=0;
	result=readCacheByte(&vm->ROMImage, offset, &aG->opcode);
	if(result==VM_OK)
	{  
		vm->IR=aG->opcode;
		(*bs)++;
		if(offset<(vm->ROMImage.cacheMaxAddress-1))offset++;
		else
		{
			return VM_ADDRESS_ERROR;
		}
		aG->type=opcodeArgTypeTable[aG->opcode];
		switch(aG->type)
		{
			case NO_ARG_TYPE:
				result=VM_OK;
				break;

			case BYTE_ARG_TYPE:
				result=readCacheByte(&vm->ROMImage, offset, &aG->arg.bArg);
				if(result==VM_OK)
				{
					(*bs)++;
				}
				break;

			case SHORT_ARG_TYPE:
				result=readCache(&vm->ROMImage, offset, &iArg[0], 2);
				if(result==VM_OK)
				{
					aG->arg.nArg=(short)(((unsigned short)iArg[0]<<8)+(unsigned short)(iArg[1]));
					(*bs)+=2;
				}
				break;
			
			case USHORT_ARG_TYPE:
				result=readCache(&vm->ROMImage, offset, &iArg[0], 2);
				if(result==VM_OK)
				{
					aG->arg.uArg=(unsigned short)(((unsigned short)iArg[0]<<8)+(unsigned short)(iArg[1]));
					(*bs)+=2;
				}
				break;
						
			case REF_ARG_TYPE:
				result=readCacheByte(&vm->ROMImage, offset, &iArg[0]);
				if(result==VM_OK)
				{
					switch(iArg[0] & 0xC0)
					{
					case 0x00:
						aG->addr=(ADDRESS_TYPE)(iArg[0] & 0x3F);
						switch(aG->opcode & 0xFC)
						{
							case OPCODE_MOVWR:
							case OPCODE_MOVWRB:
							case OPCODE_MOVSDW:
								break;
								
							case OPCODE_MOVWB:
								result=getLocalMemory(vm, aG->addr, &aG->arg.bArg, 1);
								break;
								
							default:
								result=getLocalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
								break;
						}
						aG->type=REF_ARG_LOCAL;
						(*bs)++;
						break;
 
					case 0x40:
						aG->addr=(ADDRESS_TYPE)(iArg[0] & 0x3F);
						switch(aG->opcode & 0xFC)
						{
							case OPCODE_MOVWR:
							case OPCODE_MOVWRB:
							case OPCODE_MOVSDW:
								break;
							
							default:
								result=getStackMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg);
								break;
						}
						aG->type=REF_ARG_STACK;
						(*bs)++;
						break;

					case 0x80:
						aG->addr=(ADDRESS_TYPE)(iArg[0] & 0x3F);
						switch(aG->opcode & 0xFC)
						{
							case OPCODE_MOVWR:
							case OPCODE_MOVWRB:
							case OPCODE_MOVSDW:
								break;
							
							case OPCODE_MOVWB:
								if(vm->DSLIMIT)
								{
									if(vm->DSLIMIT>sizeof(x))vm->DSLIMIT=sizeof(x);
									x=0;
									result=getGlobalMemory(vm, aG->addr, (unsigned char*)&x, (unsigned char)vm->DSLIMIT);
									aG->arg.fArg=(float)(x & 0x00FFFFFF);
								}
								else
								{	
									result=getGlobalMemory(vm, aG->addr, &aG->arg.bArg, 1);
								}
								break;
								
							default:
								result=getGlobalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
								break;
						}
						aG->type=REF_ARG_GLOBAL;
						(*bs)++;
						break;

					case 0xC0:			
						if(offset<(vm->ROMImage.cacheMaxAddress-1))offset++;
						else
						{
							return VM_ADDRESS_ERROR;
						}
						result=readCacheByte(&vm->ROMImage, offset, &iArg[1]);
						switch(iArg[0] & 0xF0)
						{
						case 0xC0:
							aG->addr=(((ADDRESS_TYPE)((ADDRESS_TYPE)iArg[0] & 0x0F)<<8)+((ADDRESS_TYPE)iArg[1]));
							aG->addr+=64;
							(*bs)+=2;
							aG->type=REF_ARG_LOCAL;
							switch(aG->opcode & 0xFC)
							{
							case OPCODE_MOVWR:
							case OPCODE_MOVWRB:
							case OPCODE_MOVSDW:
								break;
								
							case OPCODE_MOVWB:
								result=getLocalMemory(vm, aG->addr, &aG->arg.bArg, 1);
								break;
								
							default:
								result=getLocalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
								break;
							}
							break;

						case 0xD0:
							aG->addr=(((ADDRESS_TYPE)((ADDRESS_TYPE)iArg[0] & 0x0F)<<8)+((ADDRESS_TYPE)iArg[1]));
							aG->addr+=64;
							(*bs)+=2;
							aG->type=REF_ARG_STACK;
							switch(aG->opcode & 0xFC)
							{
								case OPCODE_MOVWR:
								case OPCODE_MOVWRB:
								case OPCODE_MOVSDW:
									break;
								
								default:
									result=getStackMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg);
									break;
							}
							break;

						case 0xE0:
							aG->addr=(((ADDRESS_TYPE)((ADDRESS_TYPE)iArg[0] & 0x0F)<<8)+((ADDRESS_TYPE)iArg[1]));
							aG->addr+=64;
							(*bs)+=2;
							aG->type=REF_ARG_GLOBAL;
							switch(aG->opcode & 0xFC)
							{
								case OPCODE_MOVWR:
								case OPCODE_MOVWRB:
								case OPCODE_MOVSDW:
									break;
								
								case OPCODE_MOVWB:
									if(vm->DSLIMIT)
									{
										if(vm->DSLIMIT>sizeof(x))vm->DSLIMIT=sizeof(x);
										x=0;
										result=getGlobalMemory(vm, aG->addr, (unsigned char*)&x, (unsigned char)vm->DSLIMIT);
										aG->arg.fArg=(float)(x & 0x00FFFFFF);
									}
									else
									{	
										result=getGlobalMemory(vm, aG->addr, &aG->arg.bArg, 1);
									}	
									break;
								
								default:
									result=getGlobalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
									break;
							}
							break;

						case 0xF0:
							if(offset<(vm->ROMImage.cacheMaxAddress-1))offset++;
							else
							{
								return VM_ADDRESS_ERROR;
							}
						    result=readCacheByte(&vm->ROMImage, offset, &iArg[2]);
							switch(iArg[0] & 0x0C)
							{
								default:
									result=VM_ARG_ERROR;
									break;
								
								case 0x00:
									aG->addr=(((ADDRESS_TYPE)(((ADDRESS_TYPE)(iArg[0] & 0x03))<<16)+((ADDRESS_TYPE)iArg[1]<<8)+((ADDRESS_TYPE)iArg[2])));
									aG->addr+=64+4096;
									(*bs)+=3;
									aG->type=REF_ARG_LOCAL;
									switch(aG->opcode & 0xFC)
									{
										case OPCODE_MOVWR:
										case OPCODE_MOVWRB:
										case OPCODE_MOVSDW:
											break;
										
										case OPCODE_MOVWB:
											result=getLocalMemory(vm, aG->addr, &aG->arg.bArg, 1);
											break;
											
										default:
											result=getLocalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
											break;
									}
									break;
										
								case 0x04:
									aG->addr=(((ADDRESS_TYPE)(((ADDRESS_TYPE)(iArg[0] & 0x03))<<16)+((ADDRESS_TYPE)iArg[1]<<8)+((ADDRESS_TYPE)iArg[2])));
									aG->addr+=64+4096;
									(*bs)+=3;
									aG->type=REF_ARG_STACK;
									switch(aG->opcode & 0xFC)
									{
										case OPCODE_MOVWR:
										case OPCODE_MOVWRB:
										case OPCODE_MOVSDW:
											break;
											
										default:
											result=getStackMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg);
											break;
									}
									break;
																	
								case 0x08:
									aG->addr=(((ADDRESS_TYPE)(((ADDRESS_TYPE)(iArg[0] & 0x03))<<16)+((ADDRESS_TYPE)iArg[1]<<8)+((ADDRESS_TYPE)iArg[2])));
									aG->addr+=64+4096;
									(*bs)+=3;
									aG->type=REF_ARG_GLOBAL;
									switch(aG->opcode & 0xFC)
									{
										case OPCODE_MOVWR:
										case OPCODE_MOVWRB:
										case OPCODE_MOVSDW:
											break;
										
										case OPCODE_MOVWB:
											if(vm->DSLIMIT)
											{
												if(vm->DSLIMIT>sizeof(x))vm->DSLIMIT=sizeof(x);
												x=0;
												result=getGlobalMemory(vm, aG->addr, (unsigned char*)&x, (unsigned char)vm->DSLIMIT);
												aG->arg.fArg=(float)(x & 0x00FFFFFF);
											}
											else
											{	
												result=getGlobalMemory(vm, aG->addr, &aG->arg.bArg, 1);
											}	
											break;
											
										default:
											result=getGlobalMemory(vm, aG->addr, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
											break;
									}
									break;
							}
							break;
						}
						break;
				}
				break;
				
			case LITI_ARG_TYPE:
				/* integer constant */
				switch(aG->opcode & 0xFC)
				{
					case OPCODE_MOVSDW:
						result=VM_OK;
						break;
							
					default:
						result=readCache(&vm->ROMImage, offset, &iArg[0], 2);
						if(result==VM_OK)
						{
						aG->arg.fArg=(float)(short)(((unsigned short)iArg[0]<<8)+(unsigned short)(iArg[1]));
						(*bs)+=2;
						}
						break;
				}
				break;
				
			case LIT_ARG_TYPE:
				switch(aG->opcode & 0xFC)
				{
					case OPCODE_MOVSDW:
						result=VM_OK;
						break;
							
					default:
						result=readCache(&vm->ROMImage, offset, (unsigned char*)&aG->arg.fArg, SIZE_OF_FLOAT);
						if(result==VM_OK)
						{
							(*bs)+=SIZE_OF_FLOAT;
						}						
						break;
				}
				break;
				
			default:		
			case ERR_ARG_TYPE:
				result=VM_ARG_ERROR;
				break;
		}
	}
	}
	return result;
}

VM_RETURN_TYPE executeInstructionVM(VIRTUAL_MACHINE* vm)
{
	/* actually execute the instruction at PC */

	unsigned char	bytes;
	VM_RETURN_TYPE  result;

	result=getInstructionAndArgument(vm, vm->PC, &argument, &bytes);
	if(result==VM_OK)
	{
		while(bytes--)
		{
			/* advance the program counter */
	 		if(vm->PC<(vm->ROMImage.cacheMaxAddress-1))vm->PC++;
			else
			{
				return VM_ADDRESS_ERROR;
			}
		}
		/* execute the instruction */
		result=opcodes[argument.opcode](vm, &argument);
	}
	vm->lastError=result;
	vm->execDone++;
	
	if(result!=VM_OK)
	{
		/* Halt On Any Error, this could indicate rogue code is executing... */
		changeVMState(vm, VM_HALT_MODE);
	}
	
	if((vm->execState==VM_RUN_MODE)&&(vm->execLimit>0))
	{
			vm->execLimit--;
			vm->IR=VMENV_NOP;
			if(vm->execLimit<=0)changeVMState(vm, VM_LIMIT_MODE);
	}		
	return result;
}

VM_RETURN_TYPE stepVM(VIRTUAL_MACHINE* vm)
{
	/* step the VM */
	switch(vm->execState)
	{
	default:
	case VM_RUN_MODE:
			executeInstructionVM(vm);
			break;

	case VM_SLEEP_MODE:
			/* do nothing */
			break;
 
	case VM_HALT_MODE:
	  		/* do nothing */
	 		break;
	
	case VM_LIMIT_MODE:
			/* do nothing */
			break;
	}
	return VM_OK;
} 
       
#if(IS_PC_HOST)

VM_RETURN_TYPE saveVM(VIRTUAL_MACHINE* vm)
{       
	/* save VM state to disk */
	VM_RETURN_TYPE 	result;
	unsigned int 	uz; 
	unsigned int 	sz;     
	unsigned char  	fileName[256];
	unsigned char*	ptr;  
	unsigned char  	CRC;

	CRC=0;
	if(vm->execID!=INVALID_ID)
	{	
	result=writeCacheToDisk(&vm->ROMImage, vm->ROMImage.cacheStart, vm->ROMImage.bottomAddress, vm->ROMImage.topAddress);
	result=writeCacheToDisk(&vm->RAMImage, vm->RAMImage.cacheStart, vm->RAMImage.bottomAddress, vm->RAMImage.topAddress);
	result=writeCacheToDisk(&vm->STACKImage, vm->STACKImage.cacheStart, vm->STACKImage.bottomAddress, vm->STACKImage.topAddress);
	result=writeCacheToDisk(&vm->FILEImage, vm->FILEImage.cacheStart, vm->FILEImage.bottomAddress, vm->FILEImage.topAddress);
	if(result==VM_OK)
	{ 
		getFileNameVM(vm, fileName);
		result=CACHE_ERROR_WRITE;
		sz=sizeof(VIRTUAL_MACHINE);
		ptr=(unsigned char*)vm;
		vm->CRC=0;
 		for(uz=0; uz<sz; uz++)
		{
			CRC^=ptr[uz];
		}
		vm->CRC=CRC;
		WriteFile(getStringFromUnsignedCharArrayAuto(fileName), 0, (unsigned char*)vm, sz, &uz);
		if(sz==uz)result=VM_OK;
	}
	}
	else
	{
		result=VM_OK;
	}
	return result;
}
 
VM_RETURN_TYPE restoreVM(VIRTUAL_MACHINE* vm)
{
	/* get VM state from disk */
	VM_RETURN_TYPE  result;
	unsigned int 	uz;
	unsigned int 	sz;
	unsigned char 	fileName[256];
	unsigned char   CRC;
	unsigned char*  ptr;
	
	CRC=0;
	getFileNameVM(vm, fileName);
	result=CACHE_ERROR_READ;
	sz=sizeof(VIRTUAL_MACHINE);
	ReadFile(getStringFromUnsignedCharArrayAuto(fileName), 0, (unsigned char*)vm, sz, &uz);
	if(sz==uz)
	{
		ptr=(unsigned char*)vm;
		for(uz=0; uz<sz; uz++)
		{
			CRC^=ptr[uz];
		}
		result=VM_OK;
	}
	if(CRC!=0)result=VM_CRC_ERROR;
	return result;
}

#else

VM_RETURN_TYPE saveVM(VIRTUAL_MACHINE* vm)
{
	/* save VM state to disk */
	VM_RETURN_TYPE result;
	FRESULT 	   fresult;
	unsigned int   uz;
	unsigned int   sz;
	unsigned char  CRC;
	unsigned char* ptr;
	CRC=0; 	
	if(vm->execID!=INVALID_ID)
	{
	result=writeCacheToDisk(&vm->ROMImage, 	 vm->ROMImage.cacheStart,   vm->ROMImage.bottomAddress,    vm->ROMImage.topAddress);
	result=writeCacheToDisk(&vm->RAMImage,   vm->RAMImage.cacheStart,   vm->RAMImage.bottomAddress,    vm->RAMImage.topAddress);
	result=writeCacheToDisk(&vm->STACKImage, vm->STACKImage.cacheStart, vm->STACKImage.bottomAddress,  vm->STACKImage.topAddress);
	result=writeCacheToDisk(&vm->FILEImage,  vm->FILEImage.cacheStart,  vm->FILEImage.bottomAddress,   vm->FILEImage.topAddress);
	if(result==VM_OK)
	{
		getFileNameVM(vm, fileName);
		result=VM_ERROR;
		f_close(&fvm);
		if((memoryCardSystemUpMacro())&&(f_open(&fvm, (char*)fileName, FA_WRITE | FA_OPEN_ALWAYS)==FR_OK))
		{
			result=CACHE_ERROR_WRITE;
			sz=sizeof(VIRTUAL_MACHINE);
			ptr=(unsigned char*)vm;
			vm->CRC=0;
			for(uz=0; uz<sz; uz++)
			{
				CRC^=ptr[uz];
			}
			vm->CRC=CRC;
			fresult=f_write(&fvm, (void*)vm, sz, &uz);
			if(fresult==FR_OK)
 	 		{	
				if(sz==uz)result=VM_OK;
	 			f_close(&fvm);
	 		}
		}
	} 
	}
	else
	{
		result=VM_OK;
	}
	return result;
}

VM_RETURN_TYPE restoreVM(VIRTUAL_MACHINE* vm)
{ 
	/* get VM state from disk */
	VM_RETURN_TYPE  result;
	FRESULT 		fresult;
	unsigned int 	uz;
	unsigned int 	sz;
	unsigned char 	CRC;
	unsigned char*  ptr;
	
	CRC=0;
	getFileNameVM(vm, fileName);
	result=VM_ERROR;	
	f_close(&fvm);
	if((memoryCardSystemUpMacro())&&(f_open(&fvm, (char*)fileName, FA_READ | FA_OPEN_EXISTING)==FR_OK))
	{
		result=CACHE_ERROR_READ;
		sz=sizeof(VIRTUAL_MACHINE);
		ptr=(unsigned char*)vm;
		fresult=f_read(&fvm, (void*)vm, sz, &uz);
		if(fresult==FR_OK)
		{	
			if(sz==uz)
			{
				for(uz=0; uz<sz; uz++)
				{
					CRC^=ptr[uz];
				}
				result=VM_OK;
			}
			f_close(&fvm);
		}
	}
	if(CRC!=0)result=VM_CRC_ERROR;
	
	return result;
}

#endif

VM_RETURN_TYPE clearCaches(VIRTUAL_MACHINE* vm)
{
	unsigned char result;
	
	result=VM_OK;
	result|=clearCache(&vm->STACKImage, 	STACK_CACHE_ID + (VM_ID_MASK & vm->execID),   STACK_CACHE_SIZE,  MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_STACK_OFFSET]); 
	result|=clearCache(&vm->ROMImage,   	ROM_CACHE_ID   + (VM_ID_MASK & vm->execID),   ROM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_ROM_OFFSET]); 
	result|=clearCache(&vm->RAMImage,   	RAM_CACHE_ID   + (VM_ID_MASK & vm->execID),   RAM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_RAM_OFFSET]); 
	result|=clearCache(&vm->FILEImage,  	FILE_CACHE_ID  + (VM_ID_MASK & vm->execID),   FILE_CACHE_SIZE,   MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_FILE_OFFSET]);
	return result;
}

#if 0
VM_RETURN_TYPE saveCaches(VIRTUAL_MACHINE* vm)
{
	VM_RETURN_TYPE result;
	
	result=VM_OK;
	if(vm->execID!=INVALID_ID)
	{	
	result=writeCacheToDisk(&vm->ROMImage, 	 vm->ROMImage.cacheStart,   vm->ROMImage.bottomAddress,    vm->ROMImage.topAddress);
	result=writeCacheToDisk(&vm->RAMImage,   vm->RAMImage.cacheStart,   vm->RAMImage.bottomAddress,    vm->RAMImage.topAddress);
	result=writeCacheToDisk(&vm->STACKImage, vm->STACKImage.cacheStart, vm->STACKImage.bottomAddress,  vm->STACKImage.topAddress);
	result=writeCacheToDisk(&vm->FILEImage,  vm->FILEImage.cacheStart,  vm->FILEImage.bottomAddress,   vm->FILEImage.topAddress);
	clearCaches(vm);
	}
	return result;
}
#endif
	
VM_RETURN_TYPE changeVMState(VIRTUAL_MACHINE* vm, unsigned char newState)
{
	/* 
		The VM can be in a number of states, as follows:

		VM_RUN_MODE:	VM is running code...
		VM_SLEEP_MODE:	VM state is stored in non volatile memory...
		VM_HALT_MODE:	VM has finished executing. VM requires a reset to start again...
		VM_LIMIT_MODE:	VM has been suspended (perhaps ran too long)...
		
	*/

	switch(newState)
	{
	default:
	/* Change to Run Mode */
	case VM_RUN_MODE:
		switch(vm->execState)
		{
		default:
		case VM_RUN_MODE:
			/* do nothing */
			break;

		case VM_SLEEP_MODE:
			/* Change to Run Mode from Sleep Mode*/
			restoreVM(vm);
			break;

		case VM_HALT_MODE:
			restoreVM(vm);
			break;

		case VM_LIMIT_MODE:
			break;
		}
		vm->execState=VM_RUN_MODE;
		break;

	case VM_SLEEP_MODE:
		/* Change to Sleep Mode */
		switch(vm->execState)
		{
		case VM_RUN_MODE:
			vm->execState=VM_SLEEP_MODE;
			saveVM(vm);
			break;
		
		default:
		case VM_SLEEP_MODE:
			break;
		
		case VM_HALT_MODE:
			break;

		case VM_LIMIT_MODE:
			break;
		}
		vm->execState=newState;
		break;

	case VM_HALT_MODE:
		/* Change to Halt Mode */
		switch(vm->execState)
		{
		
		case VM_RUN_MODE:
			vm->execState=VM_HALT_MODE;
			saveVM(vm);
			break;
		
		case VM_SLEEP_MODE:
			break;
		
		default:
		case VM_HALT_MODE:
			break;

		case VM_LIMIT_MODE:
			break;
		}
		vm->execState=newState;
		break;

	case VM_LIMIT_MODE:
		/* Change to Limited Mode */
		switch(vm->execState)
		{
		
		case VM_RUN_MODE:
			break;
		
		case VM_SLEEP_MODE:
			break;
	 	
		case VM_HALT_MODE:
			break;
		
		default:
		case VM_LIMIT_MODE:
			break;
		}
		vm->execState=newState;
		break;
	}
	return VM_OK;
}

VM_RETURN_TYPE destroyVM(VIRTUAL_MACHINE* vm, unsigned char exID, unsigned char por)
{
	VM_RETURN_TYPE result;
	
	vm->resetVector=0;
	resetVM(vm, exID);
	vm->firstPor=por;
	result=saveVM(vm);
	return result;
}

#if 0

#if(IS_PC_HOST)

VM_RETURN_TYPE destroyVMFiles(unsigned char exID)
{
	VM_RETURN_TYPE result;
	unsigned char  fileName[256];
		
	igetFileNameVM(exID, fileName);
	if(System::IO::File::Exists(getStringFromUnsignedCharArrayAuto(fileName)))
	{
		try
		{
			System::IO::File::Delete(getStringFromUnsignedCharArrayAuto(fileName));
		}
		catch(...)
		{
		
		}
	}
	result=VM_OK;
	return result;
}

#else

VM_RETURN_TYPE destroyVMFiles(unsigned char exID)
{
	VM_RETURN_TYPE result;
	FRESULT fresult;
		
	igetFileNameVM(exID, fileName);
	fresult=f_unlink((char*)fileName);
	if(fresult==FR_OK)result=VM_OK; else result=VM_ERROR;
	return result;
}

#endif

#endif

VM_RETURN_TYPE resetVM(VIRTUAL_MACHINE* vm, unsigned char exID)
{
	/* perform a hard reset of the VM */
	resetFunction(vm, 0);
	
	vm->execID=exID;
	
	#if(IS_PC_HOST)
	
	#else

		closeSessionHardware(vm);
	
	#endif	
	vm->SPI.mode=SPI_NO_MODE;
	initCache(&vm->STACKImage,CACHE_MODE_NONE, STACK_CACHE_ID + (VM_ID_MASK & vm->execID),   STACK_CACHE_SIZE,  MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_STACK_OFFSET], 0); 
	initCache(&vm->ROMImage,  CACHE_MODE_NONE | CACHE_READ_ONLY, ROM_CACHE_ID   + (VM_ID_MASK & vm->execID),   ROM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_ROM_OFFSET], 0); 
	initCache(&vm->RAMImage,  CACHE_MODE_NONE, RAM_CACHE_ID   + (VM_ID_MASK & vm->execID),   RAM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_RAM_OFFSET], 0); 
	initCache(&vm->FILEImage, FILECACHE_MODE, FILE_CACHE_ID + (VM_ID_MASK & vm->execID), FILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_FILE_OFFSET], &vm->fName[0]);
	return VM_OK;
}

VM_RETURN_TYPE initVM(VIRTUAL_MACHINE* vm, unsigned char exID, ADDRESS_TYPE runStart)
{
	vm->resetVector=runStart;
	(void)resetVM(vm, exID);
	changeVMState(vm, VM_RUN_MODE);
	return VM_OK;
}

VM_RETURN_TYPE restartVM(VIRTUAL_MACHINE* vm)
{
	/* 
		If the VM is in SLEEP MODE or in LIMIT MODE, it restarts it... 
	    If the VM is in RUN MODE, this has no effect ...
	    If the VM is in HALT MODE, it resets it... 
	 
	 */ 
	 switch(vm->execState)
	 {
		default:
 	 	case VM_RUN_MODE:
	 		break;
	  
	 	case VM_HALT_MODE:
			if(vm->execMode & VM_EXEC_MODE_NO_RESTART)
			{
				vm->IR=VMENV_STOP+(vm->execID & VM_ID_MASK);
			}
			else
			{
				resetFunction(vm, 0);
			}
			break;
	 		
	 	case VM_LIMIT_MODE:
	 	case VM_SLEEP_MODE:
	 		vm->execState=VM_RUN_MODE;
	 		break;
	 }
	 return VM_OK;
}

/* 
	Virtual Machine Environment

	This allows multiple VMs to run, in a time sharing arrangement, with minimal memory footprint...
	
	By Mauro Grassi, October 2010.
*/

VM_RETURN_TYPE resetVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	ven->vmNum=0;
	ven->vmPtr=0;
	ven->cVM.execID=INVALID_ID;
	return VM_OK;
}

VM_RETURN_TYPE initVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	/* Initialize the complete environment */
	resetVMEnvironment(ven);
	ven->vmRecoveryTime=(unsigned char)VM_RECOVERY_TIME;
	ven->vmMinimumSleepPeriod=(unsigned char)VM_MINIMUM_SLEEP_PERIOD;
	clearVMEnvironment(ven);
	return VM_OK;
}

VM_RETURN_TYPE rewindVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	ven->vmMode=VME_DEFAULT_MODE | VME_REWIND;
	ven->vmMinimumPeriod=0;
	powerVMTimer(1);
	stopVMTimer();	
	ven->vmState=VME_START_CYCLE;
	ven->vmExecLimit=DEFAULT_VM_EXEC_LIMIT;
	ven->vmSelected=MAX_NUM_VM;
	return VM_OK;
}

VM_RETURN_TYPE clearVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	/* restart on every POR */
	int i;
	
	#if(IS_PC_HOST)

	if(isExtension((unsigned char*)&ven->vmLogFileName[0], LOG_FILE_NAME_SIZE, (const unsigned char*)"txt"))

	#else

	if(isExtension((unsigned char*)&ven->vmLogFileName[0], LOG_FILE_NAME_SIZE, (const rom unsigned char*)"txt"))

	#endif
	{
	
	
	
	}
	else
	{
		#if(IS_PC_HOST)

			ven->vmLogFileName[0]='s';
			ven->vmLogFileName[1]='y';
			ven->vmLogFileName[2]='s';
			ven->vmLogFileName[3]='l';
			ven->vmLogFileName[4]='o'; 
			ven->vmLogFileName[5]='g';
			ven->vmLogFileName[6]='.';
			ven->vmLogFileName[7]='t';
			ven->vmLogFileName[8]='x';
			ven->vmLogFileName[9]='t';
			ven->vmLogFileName[10]='\0';
	
		#else
	
		copyStringRom(&ven->vmLogFileName[0], (const rom unsigned char*)"syslog.txt");
	
		#endif
	}
	systemTime.show=(SHOW_TIME_DEFAULT);
	initCache(&ven->vmLogFileCache, CACHE_MODE_NONE | CACHE_MODE_FILENAME | CACHE_APPEND_ALWAYS | CACHE_MODE_LIMIT_UPPER, FILE_CACHE_ID + 0, VM_SYSTEM_LOGFILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_SYSTEM_OFFSET], &ven->vmLogFileName[0]);

	#if(IS_PC_HOST)
	
	#else
		clearCapture();
		initCapture();
	#endif
	
	clearHardwareDescriptor(&ven->systemState);
	for(i=0; i<MAX_NUM_VM; i++)
	{
		ven->vmSleepPeriod[i]=0;
	}
	
	(void)rewindVMEnvironment(ven);
	
	ven->timeSpeed=1.0;
	ven->timeScaling=0;
	ven->timeOffset=0;
	
	#if(IS_PC_HOST)
		initSerialUSBLocal();
	#else
		initSerialUSB();
	#endif
	
	return VM_OK;
}

unsigned char getNumVMRunning(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	unsigned char result;
	unsigned char  index;
	
	result=0;
	if(ven->vmNum>0)
	{
			if((ven->vmState==VME_HOLDING_ACK_STATE)||(ven->vmState==VME_HOLDING_STATE))
			{
			
			}
			else
			{
				index=0;
				while((index<ven->vmNum)&&(index<MAX_NUM_VM))
				{
						if(ven->vmID[index] & VM_ID_HOLD)
						{
							/* this script is paused, so it doesn't contribute to sleep quantities... */
						}
						else
						{
							result++;
						}					
						index++;			
				}
			}
	}
	return result;
}

unsigned short findVMInEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven, unsigned char vmID)
{
	/* returns the index of the VM in the environment ven, MAX_NUM_VM if not found */
	unsigned short i;
	i=0;
	while(i<ven->vmNum)
	{
		if((VM_ID_MASK & ven->vmID[i])==(VM_ID_MASK & vmID))
		{
			return i;
		}
		i++;
	}
	return MAX_NUM_VM;
}

VM_RETURN_TYPE destroyAllVM(VIRTUAL_MACHINE_ENVIRONMENT* ven, unsigned char* por)
{
	/* destroys all VMs in environment */
	unsigned short i;
	VM_RETURN_TYPE result;
	
	i=0;
	result=VM_OK;
	while(i<ven->vmNum)
	{
		if(destroyVM(&ven->cVM, ven->vmID[i], *por)==VM_OK)
		{
		
		}
		else
		{
			result=VM_ERROR;
		}
		i++;
	}
	*por=0;
	return result;
}

VM_RETURN_TYPE addVMToEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven, unsigned char vmID)
{
	VM_RETURN_TYPE result;
	unsigned short index;
	
	index=findVMInEnvironment(ven, vmID);
	if(index<MAX_NUM_VM)
	{
		/* replaces existing item */	
		ven->vmSleepPeriod[index]=0;
		result=VM_OK;
	}
	else
	if(ven->vmNum<MAX_NUM_VM)
	{
		/* add a new item */
		ven->vmID[ven->vmNum]=vmID;
		ven->vmSleepPeriod[ven->vmNum]=0;
		ven->vmNum++;
		result=VM_OK;
	}
	else
	{
		result=VM_OUT_MEM_ERROR;
	}
	return result;
}

unsigned char getFreeVMID(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	/* 
		Get a new, unique VM ID 
		This assumes that it is always possible, and this will be true
		whenever the number of IDs is greater than the number of elements
		in the queue...
	
	*/
	unsigned char  result;
	unsigned char  index;
	unsigned char  max;
	unsigned char  min;
	
	if(ven->vmNum>0)
	{
		index=0;
		max=0;
		min=VM_ID_MASK;
		while(index<(ven->vmNum))
		{
			if(ven->vmID[index]>max)max=ven->vmID[index];
			if(ven->vmID[index]<min)min=ven->vmID[index];
			index++;	
		}
		/* 
			min & max now hold the maximum & minimum of all the IDs in the queue 
			our insert candidate is going to be max+1 as long as it is not equal to
			min (due to wrapping), otherwise go for the next available...
			
		*/
		if(((max+1) & VM_ID_MASK)!=(min & VM_ID_MASK))
		{
			result=max+1;
		}
		else
		{
			result=min+1;
			index=0;
			while((index<MAX_NUM_VM)&&(result<max))
			{
				index=(unsigned char)findVMInEnvironment(ven, result);
				result++;
			}	
		}
	}
	else
	{
		result=0;
	}
	return result;
}

VM_RETURN_TYPE deleteVMFromEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven, unsigned char vmID)
{
	unsigned short i;
	unsigned short index;
	VM_RETURN_TYPE result;
	
	result=VM_ERROR;
	index=findVMInEnvironment(ven, vmID);
	if(index<MAX_NUM_VM)
	{
			if(index==ven->vmSelected)
			{
				/* unselect, if it was selected */
				ven->vmSelected=MAX_NUM_VM;
			}
			i=index+1;
			while(i<ven->vmNum)
			{
				ven->vmID[i-1]=ven->vmID[i];
				ven->vmSleepPeriod[i-1]=ven->vmSleepPeriod[i];
				i++;
			}
			ven->vmPtr=0;
			ven->vmNum--;
			result=VM_OK;
	}
	return result;
}

#if(IS_PC_HOST)

void powerVMTimer(unsigned char on)
{
	
}	

void setVMTimer(unsigned short x, unsigned char remainder)
{
	vmTimer=x;
}

void startVMTimer(void)
{
	
}

void stopVMTimer(void)
{
	vmTimer=0;
}

#else

void powerVMTimer(unsigned char on)
{
	if(on)
	{
		#if(USE_PMDIS)
			PMDIS2bits.TMR8MD=0;
		#endif
	}
	else
	{
		#if(USE_PMDIS)
			PMDIS2bits.TMR8MD=1;
		#endif	
	}
}

void setVMTimer(unsigned short x, unsigned char remainder)
{ 
	unsigned char c;
	if(PIE5bits.TMR8IE)c=1; else c=0;
	PIE5bits.TMR8IE=0;
	TMR8=remainder;
	vmTimer=x;
	PIR5bits.TMR8IF=0;
	if(c)PIE5bits.TMR8IE=1;
}

void startVMTimer(void)
{
	/* Assuming a FOSC=48 MHz set it for 1 tick every 1 ms */
	IPR5bits.TMR8IP=0;
	vmTimer=1;
	PR8=0xFF;				/* 183 times per second... */
	T8CON=0x7F;				/* 1:16 and 1:16 scaling */
	PIR5bits.TMR8IF=0;
	PIE5bits.TMR8IE=1;
}

void stopVMTimer(void)
{
	PIE5bits.TMR8IE=0;
	PIR5bits.TMR8IF=0;
	T8CON=0;
	TMR8=0;
	vmTimer=0;
}

#endif

unsigned char* getSizeGPSVM(VIRTUAL_MACHINE* vm, unsigned short* sizeOut)
{
	int size;
	unsigned char* ptr;
	
	ptr=(unsigned char*)&vm->GPS.GPSPipe[0];
	ptr+=(unsigned int)vm->GPS.GPSPipeGetPtr;
	size=(int)vm->GPS.GPSPipePutPtr-(int)vm->GPS.GPSPipeGetPtr;
	if(size<0)size+=MAX_GPS_PIPE_LENGTH;
	(*sizeOut)=(unsigned short)size;
	return ptr;
}

void printGPSVM(VIRTUAL_MACHINE* vm)
{
	unsigned char igetPtr;
	
	igetPtr=vm->GPS.GPSPipeGetPtr;
	while(igetPtr!=vm->GPS.GPSPipePutPtr)
	{
		outputCharVM(vm, vm->GPS.GPSPipe[igetPtr]);
		if(igetPtr<(MAX_GPS_PIPE_LENGTH-1))igetPtr++; else igetPtr=0;
	}
}

void printGPSTempStringVM(VIRTUAL_MACHINE* vm)
{
	unsigned char igetPtr;
	
	igetPtr=0;
	while((igetPtr<vm->GPS.GPSTempPtr)&&(igetPtr<MAX_GPS_TEMPSTRING))
	{
		outputCharVM(vm, vm->GPS.GPSTempString[igetPtr]);
		igetPtr++;
	}
}

#if (IS_PC_HOST)

#else

void autoMatchGPRMC(VIRTUAL_MACHINE* vm)
{
	unsigned char	i;
	unsigned short 	size;
	unsigned char* 	ptr;
	unsigned char	cInd;
	unsigned char	sInd;
	float			x;
	float			l;

	ptr=getSizeGPSVM(vm, &size);
	(void)matchNMEAString(0, (const rom unsigned char*)"sGPRMC,d2d2f,cA,d2f,$,d3f,$,f,f,d2d2d2,", ptr, (unsigned char*)&vm->GPS.GPSOutput[0], &cInd, &sInd, (unsigned char*)&vm->GPS.GPSOutPtr, size, MAX_GPS_OUTPUT);
	if(vm->GPS.GPSOutPtr>=AUTO_MATCH_NUM_BYTES_GPRMC)
	{
		#if 0
			showBuffer((unsigned char*)&vm->GPS.GPSOutput[0], (unsigned int)vm->GPS.GPSOutPtr);
		#endif

		ptr=(unsigned char*)&vm->GPS.GPSOutput[0];
		ptr=getNMEAX(ptr, &i, 1);
		vm->time.hours=(unsigned char)i;
		ptr=getNMEAX(ptr, &i, 1);
		vm->time.mins=(unsigned char)i;
		ptr=getNMEAX(ptr, (unsigned char*)&x, 4);
		vm->time.secs=(unsigned char)x;
		vm->time.updated=TIME_UPDATED;
			
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		ptr=getNMEAX(ptr, (unsigned char*)&x, 4);
		x=x/60;
		x+=(float)(int)i;
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		if(i=='S')x=-x;
		vm->GPS.GPSLatitude=x;	
		
		ptr=getNMEAX(ptr, (unsigned char*)&l, 4);
		ptr=getNMEAX(ptr, (unsigned char*)&x, 4);
		x=x/60;
		x+=l;
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		if(i=='W')x=-x;
		vm->GPS.GPSLongitude=x;
		
		ptr=getNMEAX(ptr, (unsigned char*)&x, 4);
		vm->GPS.GPSSpeed=x;
		ptr=getNMEAX(ptr, (unsigned char*)&x, 4);
		vm->GPS.GPSCourse=x;
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		vm->time.day=(unsigned char)i;
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		vm->time.month=(unsigned char)i;
		ptr=getNMEAX(ptr, (unsigned char*)&i, 1);
		vm->time.year=(unsigned short)i+EPOCH_YEAR;
		vm->time.wday=(unsigned char)getWeekDay(vm->time.day, vm->time.month, vm->time.year);
		vm->time.show=(SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS | SHOW_YEAR | SHOW_MONTH | SHOW_DAY | SHOW_WDAY);
		}
}
#endif

void receiveCharVM(VIRTUAL_MACHINE* vm)
{
	#if(IS_PC_HOST)
	
	#else
	
	unsigned char  	c;
	/* Check For Errors... */
	
	if(RCSTA2bits.OERR)
	{
		vm->GPS.GPSError|=GPS_OVERRUN_ERROR;
		RCSTA2bits.CREN=0;
		RCSTA2bits.CREN=1;
		c=RCREG2;
	}
	else
	{
	
	c=RCREG2;
	vm->GPS.GPSLastRx=c;
	vm->GPS.GPSNewRx|=1;
	
	if(vm->hardwareState.serialPort.mode & UART_GPS_MODE)
	{
		switch(vm->GPS.GPSPipeState)
		{
			default:
				vm->GPS.GPSPipeState=GPS_IDLE_STATE;
			case GPS_IDLE_STATE:
				/* drop any characters until a valid start character is received! */
				if(vm->GPS.GPSPipePutPtr!=0)
				{
					clearUARTRx(vm);
				}
				
				if(c==GPS_START_CHAR)
				{
					vm->GPS.GPSPipeState=GPS_PAYLOAD_STATE;
				}
				break;
			
			case GPS_PAYLOAD_STATE:
				if(c==GPS_START_CHAR)
				{
					/* reset! , this is not supposed to happen */
					vm->GPS.GPSPipeState=GPS_IDLE_STATE;
				}
				else
				if(c==GPS_END_PAYLOAD_CHAR)
				{
					vm->GPS.GPSPipeState=GPS_FIRST_CRC_CHAR_STATE;
				}
				else
				if(c==GPS_END_CHAR)
				{
					/* this is reached when no CRC is being used, force correct CRC... */
					vm->GPS.GPSRxCRC=0;
					vm->GPS.GPSPipeState=GPS_ALL_RECEIVED_STATE;
				}
				else
				{
					/* add to the CRC */
					UARTRxPipePut(vm, c);
					if(vm->GPS.GPSError & GPS_OVERFLOW_ERROR)
					{
						vm->GPS.GPSPipeState=GPS_IDLE_STATE;
					}
				}	
				break;
			
			case GPS_FIRST_CRC_CHAR_STATE:
				if(((c>='0')&&(c<='9'))||((c>='A')&&(c<='F')))
				{
					vm->GPS.GPSRxCRC^=((unsigned char)disFix(c)<<4);
					vm->GPS.GPSPipeState=GPS_SECOND_CRC_CHAR_STATE;
				}
				else
				{
					vm->GPS.GPSPipeState=GPS_IDLE_STATE;
				}
				break;
							
			case GPS_SECOND_CRC_CHAR_STATE:
				if(((c>='0')&&(c<='9'))||((c>='A')&&(c<='F')))
				{
					vm->GPS.GPSRxCRC^=(unsigned char)disFix(c);
					vm->GPS.GPSPipeState=GPS_ALL_RECEIVED_STATE;
				}
				else
				{
					/* reset! , this is not supposed to happen... */
					vm->GPS.GPSPipeState=GPS_IDLE_STATE;
				}
				break;
			
			case GPS_ALL_RECEIVED_STATE:
				/* Try to automatically match GPRMC sentences... */
				autoMatchGPRMC(vm);
				vm->GPS.GPSPipeState=GPS_ALL_DONE_STATE;
				break;
			
			case GPS_ALL_DONE_STATE:
				break;
	}
	}
	else
	{
		UARTRxPipePut(vm, c);
	}
	}			
	#endif
}

unsigned char notEmptyCharVM(VIRTUAL_MACHINE* vm)
{
	unsigned char* ptr;
	unsigned short size;
	
 	ptr=getSizeGPSVM(vm, &size);
	if(size>0)return 1; else return 0;
}

unsigned char newRxUART(VIRTUAL_MACHINE* vm)
{
	unsigned char c;
	c=vm->GPS.GPSNewRx;
	vm->GPS.GPSNewRx=0;
	return c;
}

VM_RETURN_TYPE initSessionHardware(VIRTUAL_MACHINE* vm)
{
	/* Initialize any hardware resources for the time slice session that the VM is running in */
	
	#if(IS_PC_HOST)

	#else

	if(vm->hardwareState.serialPort.mode & (UART_NO_RX | NO_UART_MODE))	
	{

	}
	else
	{
		autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, &vm->hardwareState, OPEN_SERIAL_PORT);
	}
	
	#endif
	
	return VM_OK;
}
 
VM_RETURN_TYPE closeSessionHardware(VIRTUAL_MACHINE* vm)
{
	/* Close any hardware resources for the time slice session that the VM is running in */
	
	#if(IS_PC_HOST)

	#else

	if(vm->hardwareState.serialPort.mode & (UART_NO_RX | NO_UART_MODE))
	{
	
	
	}
	else
	{
		PIE3bits.RC2IE=0;
		PIR3bits.RC2IF=0;	
	}
	
	#endif
	
	return VM_OK;
}

VM_RETURN_TYPE hardwareSessionTask(VIRTUAL_MACHINE* vm)
{
		/* Do any session hardware tasks for this VM */
		#if(IS_PC_HOST)
				
		#else
	
		if(vm->hardwareState.serialPort.mode & (UART_NO_RX | NO_UART_MODE))	
		{
				/* No UART receiving... */	
		}
		else
		{
		if(vm->hardwareState.serialPort.mode & UART_RX_INT)
		{
		    
		}
		else
		{
	  	if(PIR3bits.RC2IF)
	   	 {
	   	  	PIR3bits.RC2IF=0;
		   	receiveCharVM(vm);
	      }	
		}
	}	
	#endif
	return VM_OK;
}

#if(IS_PC_HOST)
long getDurationTotalSecondsFutureArgumentVM(VIRTUAL_MACHINE* vm, TIME_T* inTime)
{
	TIME_T argTime;

	argTime.show=(SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);
	/* get the current time! */
	if(vm->timeScaling)getTimeVM(vm, &argTime);
	else igetTime(&argTime);

	getFutureArgumentTime(inTime, &argTime);
	return getDurationTotalSeconds(inTime, &argTime);
}
#else
long getDurationTotalSecondsFutureArgument(TIME_T* inTime)
{
	TIME_T argTime;

	argTime.show=(SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);
	/* get the current time! */
	igetTime(&argTime);
	getFutureArgumentTime(inTime, &argTime);
	return getDurationTotalSeconds(inTime, &argTime);
}

#endif

#if(IS_PC_HOST)

VM_RETURN_TYPE getTimeVM(VIRTUAL_MACHINE* vm, TIME_T* time)
{
	getTime(time);
	if(vm->timeScaling)scaleTime((double)vm->timeSpeed, vm->timeOffset, time);
	return VM_OK;
}

int setAlarmTimeVM(VIRTUAL_MACHINE* vm, TIME_T* sysTime)
{
	alarmTime.year=sysTime->year;
	alarmTime.month=sysTime->month;
	alarmTime.day=sysTime->day;
	alarmTime.hours=sysTime->hours;
	alarmTime.mins=sysTime->mins;
	alarmTime.secs=sysTime->secs;
	alarmTime.show=sysTime->show;
	alarmEvent|=ALARM_SET_EVENT;
	return 0;
}

#endif

#if(IS_PC_HOST)
VM_RETURN_TYPE VMEnvironmentTask(VIRTUAL_MACHINE_ENVIRONMENT* ven, std::string &instring)
#else
VM_RETURN_TYPE VMEnvironmentTask(VIRTUAL_MACHINE_ENVIRONMENT* ven)
#endif
{
	unsigned char  index;
	unsigned char  minIndex;
	VM_RETURN_TYPE result;
	unsigned long absolute;
			 
	 #if 0

 	 static unsigned char ostate;
	 	 	 
	 if(ostate!=ven->vmState)
	 {
	 	//showVMEnvironment(ven);	 
	 	#if(IS_PC_HOST)
	 	
	 	#else
	 	putcUART('S');
	 	putcUART('0'+ven->vmState);
	 	putcUART('S');
	 	#endif
	 	ostate=ven->vmState;
	 }
	 
	 #endif
	 
	switch(ven->vmState)
	{ 
		default:
		case VME_START_CYCLE:
			ven->vmPtr=0;
			if((ven->vmNum>0)&&(!(ven->vmMode & VME_HOLD))&&(ven->vmNum<=MAX_NUM_VM))
			{
				ven->vmState=VME_GET_NEXT_VM;
			}
			else
			{
				ven->vmState=VME_HOLDING_STATE;
			}
			break;
					
		case VME_GET_NEXT_VM:
			if(ven->vmPtr<ven->vmNum)
			{		
				#if(VM_ENVIRONMENT_DEBUG)
				
				#if(IS_PC_HOST)
				
				if(ven->vmMode & VME_VEDEBUG)
				{
				instring.append("Get Next VM, Index: ");
				instring.append(getStdStringFromString(HexToString(ven->vmPtr, 2)));
				instring.append(" VID: ");
				instring.append(getStdStringFromString(HexToString(ven->vmID[ven->vmPtr], 2)));
				instring.append(" Sleep Period: ");
				instring.append(getStdStringFromString(HexToString(ven->vmSleepPeriod[ven->vmPtr], 2)));
				instring.append(" Minimum Period: ");
				instring.append(getStdStringFromString(HexToString(ven->vmMinimumPeriod, 2)));
				instring.append("\r\n");
				}
				
				#else
				
				
				#endif
				
				#endif
				
				/* run the next VM in the queue */
				if(ven->vmID[ven->vmPtr] & VM_ID_HOLD)
				{
					/* this script is paused, so ignore it for this cycle... */
					ven->vmState=VME_END_CYCLE;
				}
				else
				{
				if((ven->vmSleepPeriod[ven->vmPtr])<=(ven->vmMinimumPeriod))
				{
					/* This VM's period has elapsed, so run it */
					index=(ven->vmID[ven->vmPtr] & VM_ID_MASK);
					ven->cVM.execID=index;	
					result=restoreVM(&ven->cVM);
					ven->cVM.execID=index;	
					ven->vmSleepPeriod[ven->vmPtr]=0;
					
					clearCache(&ven->cVM.STACKImage, STACK_CACHE_ID + (VM_ID_MASK & ven->cVM.execID),   STACK_CACHE_SIZE,  MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_STACK_OFFSET]); 
					clearCache(&ven->cVM.ROMImage,   ROM_CACHE_ID   + (VM_ID_MASK & ven->cVM.execID),   ROM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_ROM_OFFSET]); 
					clearCache(&ven->cVM.RAMImage,   RAM_CACHE_ID   + (VM_ID_MASK & ven->cVM.execID),   RAM_CACHE_SIZE,    MAX_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_RAM_OFFSET]); 
					clearCache(&ven->cVM.FILEImage,  FILE_CACHE_ID  + (VM_ID_MASK & ven->cVM.execID),   FILE_CACHE_SIZE,   MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_FILE_OFFSET]);
			
					#if(VM_ENVIRONMENT_DEBUG)
					
					#if(IS_PC_HOST)
					
					if(ven->vmMode & VME_VEDEBUG)
					{
					instring.append("Restoring VID: ");
					instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
					instring.append(" Result: ");
					instring.append(getStdStringFromString(HexToString(result, 1)));
					instring.append("\r\n");
					}
									
					#else
					
					#endif
					
					#endif
					
					if(result==VM_OK)
					{
						/* the VM was restored successfully, so continue to the next stage */
						
						if(ven->vmNum<=1)
						{
							/* Lax execution limit, if only one VM in the environment */
							ven->cVM.execLimit=0;
						}
						else
						{
							/* Limit the execution for multitasking */
							if(ven->cVM.execPriority==VM_EXEC_PRIORITY_DEFAULT)
							{
								ven->cVM.execLimit=ven->vmExecLimit;
							}
							else
							{
								ven->cVM.execLimit=ven->cVM.execPriority;
							}
						}	
						
						restartVM(&ven->cVM);
						
						if(ven->cVM.execState!=VM_RUN_MODE)
						{
							ven->vmState=VME_SUSPEND_VM;
						}
						else
						{
							#if(IS_PC_HOST)
							
							/* Before Running Update The Time Scaling From Environment */
							if(ven->timeScaling)
							{
								ven->cVM.timeSpeed=ven->timeSpeed;
								ven->cVM.timeOffset=ven->timeOffset;	
							}
							else
							{
						
							}
							ven->cVM.timeScaling=ven->timeScaling;
							
							#else
							
							
							#endif
							
							/* Auto Configure the Serial Port , for receiving! */
							initSessionHardware(&ven->cVM);
					
							ven->vmState=VME_STEP_VM;
						}
						
						#if(VM_ENVIRONMENT_DEBUG)
						
						#if(IS_PC_HOST)
						
						if(ven->vmMode & VME_VEDEBUG)
						{
						instring.append("Restarted VID: ");
						instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
						//showVM(&ven->cVM, instring);
						}
						
						#else
						
						#endif
						#endif
						
					}
					else
					{
						/* there was an error restoring the VM, so create it... */
						result=destroyVM(&ven->cVM, ven->cVM.execID, 0);

						#if(VM_ENVIRONMENT_DEBUG)

						#if(IS_PC_HOST)
						
						if(ven->vmMode & VME_VEDEBUG)
						{
						instring.append("Destroy VID: ");
						instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
						instring.append(" Result: ");
						instring.append(getStdStringFromString(HexToString(result, 1)));
						instring.append("\r\n");
						}
						
						#else
						
						#endif						
						#endif
						
						ven->vmState=VME_END_CYCLE;
					}
				}
				else
				{
					/* this VM's period has not yet elapsed... */
					ven->vmSleepPeriod[ven->vmPtr]-=ven->vmMinimumPeriod;
					ven->vmState=VME_END_CYCLE;
				}
				}	
			}
			else
			{
				/* not supposed to be here */
				ven->vmState=VME_END_CYCLE;
			}	
			break;
		
		
		case VME_STEP_VM:
			/* we now step the appropriate VM */
			if(ven->cVM.execState==VM_RUN_MODE)
			{
				
				#if(VM_ENVIRONMENT_DEBUG)
			
				#if(IS_PC_HOST)
			
				if(ven->vmMode & VME_VEDEBUG)
				{
				instring.append("Stepping VM #: ");
				instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
				instring.append(" Exec Limit: ");
				instring.append(getStdStringFromString(HexToString(ven->cVM.execLimit, 1)));
				instring.append(" PC: ");
				instring.append(getStdStringFromString(HexToString(ven->cVM.PC, 4)));
				instring.append(" W: ");
				instring.append(getStdStringFromString(formatStringFloat((double)ven->cVM.W, 3)));
				instring.append("\r\n");
				}
				
				#else
				
				
				#endif
				
				#endif
				
				if(ven->vmID[ven->vmPtr] & VM_ID_HOLD)
				{
					/* the script is paused */
					result=VM_OK;
					ven->vmState=VME_END_CYCLE;
				}
				else
				if(ven->vmMode & VME_HOLD)
				{
					result=VM_OK;
					ven->vmState=VME_END_CYCLE;
				}
				else
				{
					hardwareSessionTask(&ven->cVM);
					result=stepVM(&ven->cVM);
				}
				
				#if(VM_ENVIRONMENT_DEBUG)
				
				#if(IS_PC_HOST)
				
				if(ven->vmMode & VME_DEBUG)
				{
					instring.append(" Result: ");
					instring.append(getStdStringFromString(HexToString(result, 1)));
					instring.append("\r\n");
					//showVM(&ven->cVM, instring);
				}
				
				#else
				
				#endif
				
				#endif
				
				if(result!=VM_OK)
				{
					/* a run time error! */
					
					result=destroyVM(&ven->cVM, ven->cVM.execID, 0);

					#if(VM_ENVIRONMENT_DEBUG)
					
					#if(IS_PC_HOST)
					
					if(ven->vmMode & VME_VEDEBUG)
					{
						instring.append("Destroy VID: ");
						instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
						instring.append(" Result: ");
						instring.append(getStdStringFromString(HexToString(result, 1)));
						instring.append("\r\n");
					}
						
					#else
					
					#endif						
					#endif
					ven->vmState=VME_END_CYCLE;
				}
			}
			else
			{
				/* suspend the VM */
				//closeSessionHardware(&ven->cVM);
				ven->vmState=VME_SUSPEND_VM;
			}
			break;
			
		case VME_SUSPEND_VM:
				
				#if(VM_ENVIRONMENT_DEBUG)
				
				#if(IS_PC_HOST)
				
				if(ven->vmMode & VME_VEDEBUG)
				{
				instring.append("Suspending VM ExecState: ");
				instring.append(getStdStringFromString(HexToString(ven->cVM.execState, 1)));
				instring.append("\r\n");
				}
				
				#else
				
				#endif
				
				#endif				
						
			/* put the default next state */
			ven->vmState=VME_END_CYCLE;				
			switch(ven->cVM.execState)
			{
				default:
				case VM_RUN_MODE:
					/* this should not occur, except if the script is paused */
					ven->vmSleepPeriod[ven->vmPtr]=0;
					break;
				
				case VM_HALT_MODE:
					/* 
						The VM ran and halted in its time slice... 
						Update the period...
					*/
				case VM_LIMIT_MODE:
					/* 
						The VM ran and did not halt or sleep in its time slice 
						Update the period...
					*/
					ven->vmSleepPeriod[ven->vmPtr]=0;
					
					#if(VM_ENVIRONMENT_DEBUG)
					#if(IS_PC_HOST)
					if(ven->vmMode & VME_VEDEBUG)
					{
					instring.append("Saved VM ID: ");
					instring.append(getStdStringFromString(HexToString(ven->cVM.execID, 1)));
					instring.append("\r\n");
					//showVM(&ven->cVM, instring);
					}
					#else
					#endif
					#endif
											
					if(ven->cVM.execState!=VM_HALT_MODE)
					{
							saveVM(&ven->cVM);		
					}
						
					switch(ven->cVM.IR & 0xC0)
					{
						case VMENV_START:
							/* try to add it and restart! */
							if(addVMToEnvironment(ven, ven->cVM.IR & VM_ID_MASK)==VM_OK)
							{
								/* restart if success ! */
								ven->vmState=VME_TIMEOUT_CYCLE;
								ven->vmMinimumPeriod=0;
							}
							else
		 					{
								/* ignore otherwise */
							
							}
							break;
				
						case VMENV_STOP:
							/* try to delete it */
							if(deleteVMFromEnvironment(ven, ven->cVM.IR & VM_ID_MASK)==VM_OK)
							{
								/* restart if success ! */
								destroyVM(&ven->cVM, ven->cVM.IR & VM_ID_MASK, 0);
								ven->vmState=VME_TIMEOUT_CYCLE;
								ven->vmMinimumPeriod=0;
							}
							else
							{
								/* ignore otherwise */
							
							}	
							break;
						
						case VMENV_PAUSE:
							/*  
								try to pause it, the ID is registered in cVM.IR 
								and the action (ie, whether to pause or unpause)
								is indicated in vm->temp

							*/
							index=(unsigned char)findVMInEnvironment(ven, ven->cVM.IR);
							if(index<MAX_NUM_VM)
							{	
								if(ven->cVM.temp & VM_ID_HOLD)
								{
									ven->vmID[index]=VM_ID_HOLD + (VM_ID_MASK & ven->cVM.IR);
								}
								else
								{
									ven->vmID[index]=(VM_ID_MASK & ven->cVM.IR);
								}	
							}
							ven->cVM.temp=0;
							break;
							
						default:
							/* NOP */
							break;
					}
					break;
					
				case VM_SLEEP_MODE:
					/* 
						The sleep command stopped execution of the current VM
						Therefore the indicated argument was recorded...
						
					*/
					ven->vmSleepPeriod[ven->vmPtr]=ven->cVM.sleepMode;
					break;	
			
			}
			closeSessionHardware(&ven->cVM);
			break;
					
		case VME_END_CYCLE:
		
			#if(IS_PC_HOST)
			
			#else
			
			
		 	#endif
		 	
		 	if((ven->vmNum>1)&&(ven->vmPtr<(ven->vmNum-1)))
			{
				ven->vmPtr++;
				ven->vmState=VME_GET_NEXT_VM;
			}
			else
			{
				/* 
			
				This is the end of the VM environment cycle, when all the VMs in the queue have been serviced,
				We assume that they all registered their SLEEP periods through the execution of a SLEEP command...
				
				*/
				
				ven->vmMinimumPeriod=MAX_VM_PERIOD;
				index=0;
				minIndex=MAX_NUM_VM;
				while(index<ven->vmNum)
				{
					if(ven->vmID[index] & VM_ID_HOLD)
					{
						/* this script is paused, so it doesn't contribute to sleep quantities... */
					}
					else
					if(ven->vmSleepPeriod[index]<ven->vmMinimumPeriod)
					{
						ven->vmMinimumPeriod=ven->vmSleepPeriod[index];
						minIndex=index;
					}		
					index++;			
				}		
				
				/* We now have the minimum period among all the VMs in this environment ... */
				#if(VM_ENVIRONMENT_DEBUG)
				
				#if(IS_PC_HOST)
				
				if(ven->vmMode & VME_VEDEBUG)
				{
				instring.append("New Minimum: ");
				instring.append(getStdStringFromString(HexToString(ven->vmMinimumPeriod, 4)));
				instring.append("\r\n");
				}
				#else
				#endif						
				#endif
				
					if(minIndex<MAX_NUM_VM)
					{
						ven->vmState=VME_TIMEOUT_CYCLE;
					}
					else
		 			{
						ven->vmState=VME_HOLDING_STATE;
					}	
				}
				break;		
			
			case VME_TIMEOUT_CYCLE:
				if(ven->vmMinimumPeriod>0)
				{
					#if(IS_PC_HOST)
						stopVMTimer();
					#else
						if(T8CON!=0)stopVMTimer();
					#endif
				}
				else
				{
					/* 
						
						If the vmMinimumPeriod==0, then it means that a task has not been
						suspended due to a SLEEP instruction in its execution time slice...
						In this case, we need to estimate the time that has elapsed since
	 					the tasks were first run...
	 					
	 				*/
					
					#if(USE_VM_REALTIME)
	 	 			
					#if(IS_PC_HOST)
		 			
		 				/* automatically increment to 1 */
						vmTimer+=REALTIME_PERIOD;
					
					#endif
					
					#if(IS_PC_HOST)
					
					if(vmTimer==0)
					{
						startVMTimer();
					}
					
					#else
					
					if(T8CON==0)
					{
						startVMTimer();
					}
					#endif
					
					else
					{
						if(vmTimer>=REALTIME_PERIOD)
						{
							ven->vmMinimumPeriod+=(unsigned long)(vmTimer/REALTIME_PERIOD);
							setVMTimer(1, vmTimer % REALTIME_PERIOD);
	 					}
					}
					
					#else
						ven->vmMinimumPeriod=1;
					#endif
					}
				
					#if(IS_PC_HOST)
					if(ven->vmMode & VME_VEDEBUG)
					{
						if(ven->vmMinimumPeriod>0)
						{
						instring.append("\r\n WAIT: ");
						instring.append(getStdStringFromString(formatStringFloat((double)ven->vmMinimumPeriod, 0)));
						instring.append("\r\n");
						}
					}
					#else
					#endif
				
					/* time to retrigger ! */
	     			ven->vmState=VME_TIMEOUT_WAIT_STATE;
				
					break;
				
				case VME_TIMEOUT_WAIT_STATE:
						if(ven->vmMode & VME_HOLD)
						{
							ven->vmState=VME_HOLDING_STATE;	
						}
						else
						{	
						
						#if(IS_PC_HOST)
						if(ven->vmMinimumPeriod>0)
						#else
						if((ven->vmMinimumPeriod>0)&&(T8CON==0))
						#endif
						{
							#if(IS_PC_HOST)
							
							ven->timeScaling=localSettings.timeScaling;
							ven->timeOffset=localSettings.timeOffset;
							ven->timeSpeed=localSettings.timeSpeed;
														
							if((ven->vmMinimumPeriod)>=((unsigned long)ven->timeSpeed))
							{
						
							}
							else
							{
								ven->timeSpeed=(float)ven->vmMinimumPeriod;							
							}
							ven->cVM.timeSpeed=ven->timeSpeed;							
							getTimeVM(&ven->cVM, (TIME_T*)&systemTime);
							
							#else
							
							igetTime((TIME_T*)&systemTime);
							
							#endif
							
							absolute=getTotalSeconds((TIME_T*)&systemTime);
							
							
							#if(IS_PC_HOST)
								if((ven->vmMode & VME_USE_SLEEP)&&(ven->vmMinimumPeriod>=(unsigned long)ven->vmMinimumSleepPeriod))
							#else
							
								timeAbsolute=absolute;
								if((hostAlive==0)&&(ven->vmMode & VME_USE_SLEEP)&&(ven->vmMinimumPeriod>=(unsigned long)ven->vmMinimumSleepPeriod))
							#endif	
							{							
								absolute+=(ven->vmMinimumPeriod-(unsigned long)ven->vmRecoveryTime);
								/* indicate that we are going to sleep */
								index=1;
						 	}	
						 	else
						 	{
						 		absolute+=(ven->vmMinimumPeriod);
						 		/* indicate that we are not going to sleep */
						 		index=0;
						 	}
						 	
						 	convertEpoch2TimeInternal(absolute, (TIME_T*)&systemTime);
						    
						    #if(IS_PC_HOST)
						    
						    setAlarmTimeVM(&ven->cVM, (TIME_T*)&systemTime);
							
						    #else
						    
						    alarmAbsolute=absolute;
							setAlarmTime((TIME_T*)&systemTime);
							
							#endif
							
							if(index)
							{
									#if(IS_PC_HOST)
									
									#else
								   	if(memoryCardSystemUpMacro())
									{	
										flushLog();	
										saveSettingsFileTask();
									}
									#endif
									
									goToSleep(1);
									goToSleep(0);
									absolute+=(unsigned long)ven->vmRecoveryTime;
									
									convertEpoch2TimeInternal(absolute, (TIME_T*)&systemTime);
									
									#if(IS_PC_HOST)
								    
								    setAlarmTimeVM(&ven->cVM, (TIME_T*)&systemTime);
									
								    #else
								    
								    setAlarmTime((TIME_T*)&systemTime);
									
									#endif
									ven->vmState=VME_WAITING_STATE;
							}
							else
							{
								ven->vmState=VME_WAITING_STATE;
							}
						}
						else
						{
								ven->vmState=VME_START_CYCLE;
						}
				  	}	      
				    break;
				  
				 case VME_HOLDING_STATE:
				 	if(ven->vmNum>0)
				 	{
				 	
				 	}
				 	else
				 	{
				 		ven->vmState=VME_HOLDING_ACK_STATE;
				 	}
					break;
				 	
				 case VME_HOLDING_ACK_STATE:
				 	break;
				 
				 case VME_WAITING_STATE:
				 	if(ven->vmMode & VME_HOLD)
				 	{
				 		ven->vmState=VME_HOLDING_STATE;
				 	}
				 	else
				 	{
					 	#if(IS_PC_HOST)
				   		
				   		if((alarmEvent & ALARM_TRIGGER_EVENT)||(keyFull!=0))
						{
							alarmEvent=CLEAR_ALARM_EVENT;
				   			ven->vmState=VME_START_CYCLE;
				   			keyFull=0;
				   		}
				   		
				   		#else
				   			
				 		if((alarmEvent & ALARM_TRIGGER_EVENT)||(keyFull!=0))
						{
							PIE3bits.RTCCIE=0;
							alarmEvent=CLEAR_ALARM_EVENT;
				   			ven->vmState=VME_START_CYCLE;
				   		
				   		}
				   		#endif
				   
				   }
				   break;
	}
	return VM_OK;
}

#if(IS_PC_HOST)
VM_RETURN_TYPE VMEnvironmentSuperTask(VIRTUAL_MACHINE_ENVIRONMENT* ven, std::string &instring)
{
	int 			execLimit;
	unsigned char 	cont;
	VM_RETURN_TYPE  result;
	
	execLimit=VME_SUPER_TASK_LIMIT;
	cont=1;
	result=VM_OK;
	while((cont)&&(result==VM_OK)&&(execLimit>0))
	{
		 execLimit--;
		 #if(IS_PC_HOST)
		  
		 #else
		 	if(idleTask!=0)idleTask();
		 #endif
			
		#if(IS_PC_HOST)
			result=VMEnvironmentTask(ven, instring);
		#else
			result=VMEnvironmentTask(ven);
		#endif
		
		switch(ven->vmState)
		{
			default:		
			case VME_START_CYCLE:
			case VME_GET_NEXT_VM:
			case VME_STEP_VM:
			case VME_SUSPEND_VM:
			case VME_END_CYCLE:
				cont=1;
				break;
				
			case VME_TIMEOUT_CYCLE:
			case VME_TIMEOUT_WAIT_STATE:
			case VME_HOLDING_STATE:
			case VME_HOLDING_ACK_STATE:
			case VME_WAITING_STATE:
				cont=0;
				break;
		}	
	}
	return result;
}
#else
VM_RETURN_TYPE VMEnvironmentSuperTask(VIRTUAL_MACHINE_ENVIRONMENT* ven, void (*idleTask)(void))
{

}
#endif
	    
#if(IS_PC_HOST)
void outputCVM(VIRTUAL_MACHINE* vm, unsigned char c)
{
	if(c!='\0')myVMOutput+=(char)c;
}

void outputSVM(VIRTUAL_MACHINE* vm, unsigned char* s)
{
	while((*s)!='\0')
	{
		outputCharVM(vm, *s++);
	}
}

void outputSROMVM(VIRTUAL_MACHINE* vm, const unsigned char *s)
{
	while((*s)!='\0')
	{
		outputCharVM(vm, *s++);
	}
}


void outputFVM(VIRTUAL_MACHINE* vm, float f, unsigned short numdec)
{
	unsigned char tempString[16];
	ftoa(tempString, f, numdec);
	outputSVM(vm, tempString);
}

void outputUFVM(VIRTUAL_MACHINE* vm, float f, unsigned short numdec)
{
	unsigned char tempString[16];
	uftoa(tempString, f, numdec);
	outputSVM(vm, tempString);
}

void showCache(CACHE_OBJECT* cObj, std::string &instring)
{
	instring.append("Cache Size: ");
	instring.append(getStdStringFromString(HexToString(cObj->cacheSize, 2)));
	instring.append(" Max: ");
	instring.append(getStdStringFromString(HexToString(cObj->cacheMaxAddress, 2)));
	instring.append(" B: ");
	instring.append(getStdStringFromString(HexToString(cObj->bottomAddress, 4)));
	instring.append(" T: ");
	instring.append(getStdStringFromString(HexToString(cObj->topAddress, 4)));
	instring.append(" EndPtr: ");
	instring.append(getStdStringFromString(HexToString(cObj->cacheEndPointer, 4)));
	instring.append("\r\nH: ");
	instring.append(getStdStringFromString(formatStringFloat((double)cObj->cacheHits, 0)));
	instring.append(" M: ");
	instring.append(getStdStringFromString(formatStringFloat((double)cObj->cacheTotal-cObj->cacheHits, 0)));
	instring.append(" P: ");
	instring.append(getStdStringFromString(formatStringFloat((double)cObj->cacheArgs.cachePerformance, 2)));
	instring.append("\r\n");
}

void showArgument(VM_ARGUMENT* varg, std::string &instring)
{
	//instring.append("Arg. Opcode: ");
	//instring.append(getStdStringFromString(HexToString(varg->opcode, 1)));
	instring.append("Type : ");
	instring.append(getStdStringFromString(HexToString(varg->type, 1)));
	instring.append("\tbArg : ");
	instring.append(getStdStringFromString(HexToString(varg->arg.bArg, 2)));
	instring.append("\tnArg: ");
	instring.append(getStdStringFromString(HexToString(varg->arg.nArg, 2)));
	instring.append("\tfArg: ");
	instring.append(getStdStringFromString(formatStringFloat((double)varg->arg.fArg, 3)));
	instring.append("\r\n");
}

void showVM(VIRTUAL_MACHINE* vm, std::string &instring)
{
	std::string    temp;
	instring.append("VM ID: ");
 	instring.append(getStdStringFromString(HexToString(vm->execID, 1)));
	instring.append("\tState: ");
	instring.append(getStdStringFromString(HexToString(vm->execState, 1)));
	instring.append("\tErr : ");
	instring.append(getStdStringFromString(HexToString(vm->lastError, 1)));
	instring.append("\tPC  : ");
	instring.append(getStdStringFromString(HexToString(vm->PC, 4)));	
	instring.append("\r\nIR   : ");
	instring.append(getStdStringFromString(HexToString(vm->IR, 1)));
	instring.append("\tIns. : "+getStdStringFromString(getStringFromUnsignedCharArrayAuto((unsigned char*)&opcodeNames[vm->IR][0])));
	instring.append("\tST  : ");
	instring.append(getStdStringFromString(HexToString(vm->StackSizePtr, 4)));
	instring.append("\tW   : ");
	instring.append(getStdStringFromString(formatStringFloat((double)vm->W, 6)));
  	instring.append("\r\nDS   : ");
	instring.append(getStdStringFromString(HexToString(vm->DS, 4)));
	instring.append("\tSS   : ");
	instring.append(getStdStringFromString(HexToString(vm->SS, 4)));
	instring.append("\tDSL : ");
	instring.append(getStdStringFromString(HexToString(vm->DSLIMIT, 4)));	
	instring.append("\r\nExL  : ");
	instring.append(getStdStringFromString(HexToString(vm->execLimit, 2)));
	instring.append("\tExD  : ");
	instring.append(getStdStringFromString(HexToString(vm->execDone, 2)));
	instring.append("\tInd : ");
	instring.append(getStdStringFromString(HexToString(vm->lastIndirect, 4)));
	instring.append("\r\n");
	showArgument(&argument, temp);
	instring.append(temp);
	//instring.append("STACK CACHE:\r\n");
	//showCache(&vm->STACKImage, instring);
	//instring.append("RAM CACHE:\r\n");
	//showCache(&vm->RAMImage, instring);
	//instring.append("ROM CACHE:\r\n");
	//showCache(&vm->ROMImage, instring);
	//instring.append("FILE CACHE:\r\n");
	//showCache(&vm->FILEImage, instring);
}

void showVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven, std::string &instring)
{
	unsigned short i;

	
	instring.append("VM Environment: \r\n");
	instring.append("Num VMs: ");
	instring.append(getStdStringFromString(formatStringFloat((double)ven->vmNum, 0)));
	instring.append(" VM Ptr: ");
	instring.append(getStdStringFromString(formatStringFloat((double)ven->vmPtr, 0)));
	
	instring.append(" VM Min: ");
	instring.append(getStdStringFromString(HexToString(ven->vmMinimumPeriod, 4)));
	
	instring.append(" VME State: ");
	instring.append(getStdStringFromString(HexToString(ven->vmState, 1)));
	
	instring.append(" Exec Limit: ");
	instring.append(getStdStringFromString(HexToString(ven->vmExecLimit, 2)));
	
	instring.append(" vmRecovery: ");
	instring.append(getStdStringFromString(HexToString(ven->vmRecoveryTime, 1)));
	instring.append(" vmMinimumSleepPeriod: ");
	instring.append(getStdStringFromString(HexToString(ven->vmMinimumSleepPeriod, 1)));

	instring.append("\r\nVM IDs: [");
	i=0; 
	while(i<ven->vmNum)
	{
		instring.append(getStdStringFromString(HexToString(ven->vmID[i], 1)));
		instring.append(".");
		i++;
	}
	instring.append("]\r\nVM Sleep Periods: [");
	i=0;
	while(i<ven->vmNum)
	{
		instring.append(getStdStringFromString(HexToString(ven->vmSleepPeriod[i], 4)));
		instring.append(".");
		i++;
	}
	instring.append("]\r\n");

}

#else

void outputCVM(VIRTUAL_MACHINE* vm, unsigned char c)
{
	/* output a character to the VM Serial Port, not neccessarily the same as the system UART */
	if(autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, &vm->hardwareState, OPEN_SERIAL_PORT))
	{

	}
	putcUART(c);
}

#if 0
void showCache(CACHE_OBJECT* cObj)
{ 
	putrsUART((const rom char*)"Cache Sz: ");
	disWUART(cObj->cacheSize);
	putrsUART((const rom char*)" Max: ");
	disWUART(cObj->cacheMaxAddress);
	putrsUART((const rom char*)" B: ");
	disWUART(cObj->bottomAddress);
	putrsUART((const rom char*)" T: ");
	disWUART(cObj->topAddress);
	putrsUART((const rom char*)" EP: ");
	disDWUART(cObj->cacheEndPointer);
	putrsUART((const rom char*)"\r\nH: ");
	ftoa(tempString, cObj->cacheHits, 0);
	putsUART((char*)tempString);
	putrsUART((const rom char*)" M: ");
	ftoa(tempString, cObj->cacheTotal-cObj->cacheHits, 0);
	putsUART((char*)tempString);
	putrsUART((const rom char*)" P: ");
	ftoa(tempString, cObj->cacheArgs.cachePerformance, 2);
	putsUART((char*)tempString);
	putrsUART((const rom char*)" %\r\n");
}
#endif

#if 0

void showArgument(VM_ARGUMENT* varg)
{
	#if(VM_ENVIRONMENT_DEBUG)
	putrsUART((const rom char*)"Arg. Opcode: ");
	disWUART(varg->opcode);
	putrsUART((const rom char*)" Type: ");
	disWUART(varg->type);
	putrsUART((const rom char*)" bArg: ");
	disWUART(varg->arg.bArg);
	putrsUART((const rom char*)" nArg: ");
	disWUART(varg->arg.nArg);
	putrsUART((const rom char*)"\r\nfArg: ");
	ftoa(tempString, varg->arg.fArg, 3);
	putsUART((char*)tempString);
	putrsUART((const rom char*)"\r\n");
	#endif		
}

void showVM(VIRTUAL_MACHINE* vm)
{
	#if(VM_ENVIRONMENT_DEBUG)
	putrsUART((const rom char*)"\r\n-------------------------------------------------------------------------------\r\nVM ID: ");
	disWUART(vm->execID);
	putrsUART((const rom char*)" State: ");
	disWUART(vm->execState);
	putrsUART((const rom char*)" Error: ");
	disWUART(vm->lastError);
	putrsUART((const rom char*)"\r\nPC: ");
	disDWUART(vm->PC);
	putrsUART((const rom char*)" W: ");
	ftoa(tempString, vm->W, 3);
	putsUART((char*)tempString);
	putrsUART((const rom char*)" IR: ");
	disWUART(vm->IR);
	putrsUART((const rom char*)" ST: ");
	disWUART(vm->StackSizePtr);
	putrsUART((const rom char*)"\r\nExecLimit: ");
	disWUART(vm->execLimit);
	putrsUART((const rom char*)" ExecDone: ");
	disWUART(vm->execDone);
	putrsUART((const rom char*)"\r\n");
	showArgument(&argument);
	putrsUART((const rom char*)"STACK CACHE:\r\n");
	showCache(&vm->STACKImage);
	putrsUART((const rom char*)"RAM CACHE:\r\n");
	showCache(&vm->RAMImage);
	putrsUART((const rom char*)"ROM CACHE:\r\n");
	showCache(&vm->ROMImage);
	putrsUART((const rom char*)"FILE CACHE:\r\n");
	showCache(&vm->FILEImage);
	putrsUART((const rom char*)"\r\n-------------------------------------------------------------------------------\r\n");
	#endif
}

void showVMEnvironment(VIRTUAL_MACHINE_ENVIRONMENT* ven)
{
	unsigned short i;
	
	putrsUART((const rom char*)"VM Environment: \r\n");
	putrsUART((const rom char*)"Num VMs: ");
	disDecUART((unsigned long)ven->vmNum);	
	putrsUART((const rom char*)" VM Ptr: ");
	disDecUART(ven->vmPtr);	
	putrsUART((const rom char*)" VM Min: ");
	disDWUART(ven->vmMinimumPeriod);
	putrsUART((const rom char*)" VME State: ");
	disWUART(ven->vmState);
	putrsUART((const rom char*)" Exec Limit: ");
	disWUART(ven->vmExecLimit);
	putrsUART((const rom char*)" VM Mode: ");
	disWUART(ven->vmMode);
	putrsUART((const rom char*)" vmRecovery: ");
	disWUART(ven->vmRecoveryTime);
	putrsUART((const rom char*)" vmMinimumSleepPeriod: ");
	disWUART(ven->vmMinimumSleepPeriod);
	putrsUART((const rom char*)"\r\nVM IDs: [");
	i=0;
	while(i<ven->vmNum)
	{
		disAUART(ven->vmID[i]);
		putrsUART((const rom char*)".");
		i++;
	}
	putrsUART((const rom char*)"]\r\nVM Sleep Periods: [");
	i=0;
	while(i<ven->vmNum)
	{
		disDWUART(ven->vmSleepPeriod[i]);
		putrsUART((const rom char*)".");
		i++;
	}
	putrsUART((const rom char*)"]\r\n");
}
#endif

void outputSVM(VIRTUAL_MACHINE* vm, unsigned char* s)
{
	while((*s)!='\0')
	{
		outputCharVM(vm, *s++);
	}
}
 
void outputSROMVM(VIRTUAL_MACHINE* vm, rom const unsigned char *s)
{
	while((*s)!='\0')
	{
		outputCharVM(vm, *s++);
	}
}

void outputFVM(VIRTUAL_MACHINE* vm, float f, unsigned short numdec)
{ 
	ftoa(tempString, f, numdec);
	outputSVM(vm, tempString);
} 

void outputUFVM(VIRTUAL_MACHINE* vm, float f, unsigned short numdec)
{
	uftoa(tempString, f, numdec);
	outputSVM(vm, tempString);
}
 
#endif

void outputCharVM(VIRTUAL_MACHINE* vm, unsigned char c)
{
	unsigned char result;

	if(vm)
	{	

	if(vm->DS==0)
	{

	if(vm->pipeMode & INTERNAL_PIPES_MASK)
	{
		/* Internal Pipes */
		if(vm->pipeMode & FILENAME_PIPE_MODE)
		{
			if(((c>='a')&&(c<='z'))||((c>='A')&&(c<='Z'))||((c>=0x21)&&(c<='9')))
			{
				if(vm->tempPipe<(VM_FILENAME_MAX-1))
				{
					vm->fName[vm->tempPipe++]=c;
				}
			}	
		}
		
		if(vm->pipeMode & NMEA_MATCH_STRING_MODE)
		{
			if(vm->GPS.GPSTempPtr<(MAX_GPS_TEMPSTRING-1))
			{
				vm->GPS.GPSTempString[vm->GPS.GPSTempPtr++]=c;
			}	
		}
		
		if(vm->pipeMode & NMEA_STRING_MODE)
		{
			#if(IS_PC_HOST)
			
			#else
				vm->GPS.GPSTxCRC^=c;
				outputCVM(vm, c);
			#endif
		}	
	}
	else
	{
		/* General Pipes */	
		
		if(vm->pipeMode & SERIAL_PIPE_MODE)
		{
			outputCVM(vm, c);
		}	
		
		if(vm->pipeMode & LOGFILE_PIPE_MODE)
		{
			result=writeCacheByte(&vm->FILEImage, (ADDRESS_TYPE)vm->FILEImage.cacheEndPointer, &c);
			vm->FILEImage.cacheEndPointer++;	
		}
		
		if(vm->pipeMode & SERIAL_USB_PIPE_MODE)
		{
			#if(IS_PC_HOST)
		        sendCharSerialUSBLocal(c);
			#else
				sendCharSerialUSB(c);
			#endif
		}
	}
	}
	else
	{
		/* vm->DS!=0 so it is output to memory */
		result=writePointerMemory(vm, MODE_DS_POST_INC, &c);
		if(result==VM_LIMIT)
		{
			c='\0';
			result=writePointerMemory(vm, MODE_DS, &c);
		}	
	}
	}
	
	#if(IS_PC_HOST)
	
	
	#else
	
	if((!vm)||(vm->pipeMode & SYSTEM_LOGFILE_PIPE_MODE))
	{
		if((vm)&&(vm->DS!=0))
		{
		
		}
		else
		{
			if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_MODE)
	        {
       			#if 1
       			if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_TO_SERIAL_USB)
       			{
		       		#if(IS_PC_HOST)
			       	 	sendCharSerialUSBLocal(c);
					#else
						sendCharSerialUSB(c);
					#endif
	       		}
	       		#endif
	       		
	       		if(thisUDL.dataLogger.modes & SYSTEM_LOGGING_TO_UART)
	       		{
	       			putcUART(c);
	       		}
	       		writeCacheByte(&thisUDL.dataLogger.ven.vmLogFileCache, (ADDRESS_TYPE)thisUDL.dataLogger.ven.vmLogFileCache.cacheEndPointer, &c);
				thisUDL.dataLogger.ven.vmLogFileCache.cacheEndPointer++;
			}
		}
	}

	#endif
}

unsigned char* outputFunctionVMToString(VIRTUAL_MACHINE *vm, unsigned char* outptr, unsigned short function)
{
	TIME_T*			lTime;		
	TIME_T*         aTime;
	unsigned char*	inptr;
	long   			duration;
	
	
	inptr=outptr;
	
	/* Preprocessing... */
	
	lTime=0;
	aTime=0;
	switch(function)
	{
		default:
		case PF_NOP:
		case PF_GPS_PIPE:
		case PF_GPS_TEMPSTRING:
			break;
		
		case PF_LOCAL_DURATION_SINCE_LAST_SYNC:
			#if(IS_PC_HOST)
				aTime=&localSettings.timeHostStarted;		
			#else
			if(thisUDL.dataLogger.lastSyncTime.updated & TIME_UPDATED)
			{
				aTime=&thisUDL.dataLogger.lastSyncTime;
			}
			else
			{
				
			}
			#endif
		
		case PF_LOCAL_TIME_OR_DURATION:
		case PF_LOCAL_TIME_IF_SET:			
		case PF_LOCAL_TIME:
		case PF_LOCAL_TIME_FILE_STRING:
		case PF_LOCAL_DATE_FILE_STRING:
		case PF_LOCAL_TIME_FILE_STRING_NUMERIC:
	 	case PF_LOCAL_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_FUTURE:
				#if(IS_PC_HOST)
					getTimeVM(vm, (TIME_T*)&systemTime);
				#else
					getTime((TIME_T*)&systemTime);
				#endif
				lTime=(TIME_T*)&systemTime;
				lTime->updated=timeUp;
				break;
		
		case PF_VM_DURATION_SINCE_LAST_SYNC:
			#if(IS_PC_HOST)
				aTime=&localSettings.timeHostStarted;		
			#else
			if(thisUDL.dataLogger.lastSyncTime.updated & TIME_UPDATED)
			{
				aTime=&thisUDL.dataLogger.lastSyncTime;
			}
			else
			{
				
			}
			#endif
			
		case PF_VM_TIME:
		case PF_VM_TIME_IF_SET:
		case PF_VM_TIME_FILE_STRING:
		case PF_VM_DATE_FILE_STRING:
		case PF_VM_TIME_FILE_STRING_NUMERIC:
		case PF_VM_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_OR_DURATION:
			if(vm)lTime=&vm->time;
			break;
					
		case PF_VM_FILENAME:
		case PF_NMEA:
			break;			
	}
	
	switch(function)
	{
		default: 
		case PF_NOP:
			break;	
							
		case PF_GPS_PIPE:
		case PF_GPS_TEMPSTRING:
			#if(IS_PC_HOST)
			#else
				//outptr=printGPSVM(vm, outptr);
			#endif
			break;
			
		case PF_LOCAL_DURATION_SINCE_LAST_SYNC:
			duration=(long)getTotalSeconds((TIME_T*)lTime);
			if(aTime)
			{
				duration-=(long)getTotalSeconds(aTime);
			}
			else
			{
				duration-=DEFAULT_DURATION_SECS;
			}
			#if(IS_PC_HOST)
				outptr+=(int)getUnsignedCharArrayFromStringMax(outptr, getDurationString((float)duration, 0), 32);
			#else
				outptr=getDurationString(outptr, (float)duration, 0);
			#endif
			break;
			
		case PF_LOCAL_TIME_OR_DURATION:
		case PF_LOCAL_TIME_IF_SET:			
		case PF_LOCAL_TIME:
		case PF_VM_TIME_OR_DURATION:
		case PF_VM_TIME_IF_SET:
		case PF_VM_TIME:
			if(lTime)
			{
			if((function==PF_LOCAL_TIME)||(function==PF_VM_TIME)||(lTime->updated & TIME_UPDATED))
			{
				outptr=ascTime(outptr, (TIME_T*)lTime); 
			}
			else
			{
				if((function==PF_LOCAL_TIME_IF_SET)||(function==PF_VM_TIME_IF_SET))
				{
				
				}
				else
				{
					duration=(long)getTotalSeconds((TIME_T*)lTime)-(long)DEFAULT_DURATION_SECS;
					#if(IS_PC_HOST)
						outptr+=(int)getUnsignedCharArrayFromStringMax(outptr, getDurationString((float)duration, 0), 32);
					#else
						outptr=getDurationString(outptr, (float)duration, 0);
					#endif
				}
			}
			}	
			break;
	 	
	 	case PF_LOCAL_TIME_FILE_STRING:
		case PF_LOCAL_DATE_FILE_STRING:
		case PF_VM_TIME_FILE_STRING:
		case PF_VM_DATE_FILE_STRING:
			if(lTime)
			{
			if(function==PF_LOCAL_TIME_FILE_STRING)lTime->show=(SHOW_YEAR | SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);
			else lTime->show=(SHOW_YEAR | SHOW_MONTH | SHOW_DAY);
			(void)ascTime(&tempString[0], lTime);
			outptr=copyFileStringString(inptr, &tempString[0]);
	 		} 	
	 	 	break;
	 	
	 	case PF_LOCAL_TIME_FILE_STRING_NUMERIC:
	 	case PF_LOCAL_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_FILE_STRING_NUMERIC:
		case PF_VM_DATE_FILE_STRING_NUMERIC:
			if(lTime)
			{
			if(function==PF_LOCAL_TIME_FILE_STRING_NUMERIC)lTime->show=(SHOW_YEAR | SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);
			else lTime->show=(SHOW_YEAR | SHOW_MONTH | SHOW_DAY);
	 		outptr=ascTimeFileName(outptr, lTime);
	 		}	
	 		break;
	 		
		case PF_VM_TIME_FUTURE:
			if((vm)&&(lTime))
			{
			lTime->show=(SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);
			/* get the current time! */
			getFutureArgumentTime(&vm->time, lTime);
			outptr=ascTimeNormal(outptr, (TIME_T*)&vm->time);
			}
			break;
		
		case PF_VM_FILENAME:
			inptr=&vm->fName[0];
			break;
		
		case PF_NMEA:
			*outptr++=GPS_END_PAYLOAD_CHAR;
			*outptr++=disFixVM(vm->GPS.GPSTxCRC>>4);
			*outptr++=disFixVM(vm->GPS.GPSTxCRC);
			*outptr++=0x0D;
			*outptr++=0x0A;
			break;
	}
	
	/* Post processing */
	
	switch(function)
	{
		default:
		case PF_NOP:
		case PF_GPS_PIPE:
		case PF_GPS_TEMPSTRING:
			break;
		
		case PF_LOCAL_DURATION_SINCE_LAST_SYNC:
		case PF_LOCAL_TIME_OR_DURATION:
		case PF_LOCAL_TIME_IF_SET:			
		case PF_LOCAL_TIME:
		case PF_LOCAL_TIME_FILE_STRING:
		case PF_LOCAL_DATE_FILE_STRING:
		case PF_LOCAL_TIME_FILE_STRING_NUMERIC:
	 	case PF_LOCAL_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_FUTURE:
		case PF_VM_DURATION_SINCE_LAST_SYNC:
		case PF_VM_TIME:
		case PF_VM_TIME_IF_SET:
		case PF_VM_TIME_FILE_STRING:
		case PF_VM_DATE_FILE_STRING:
		case PF_VM_TIME_FILE_STRING_NUMERIC:
		case PF_VM_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_OR_DURATION:
			if(inptr==outptr)
			{
			#if(IS_PC_HOST)
					outptr=copyStringRom(outptr, (const unsigned char*)"Time Unavailable");
			#else
					outptr=copyStringRom(outptr, (const rom char*)"Time Unavailable");
			#endif
			}
			break;
					
		case PF_VM_FILENAME:
		case PF_NMEA:
			break;			
	}	
	*outptr='\0';
	return inptr;
}

void outputFunctionVM(VIRTUAL_MACHINE* vm, unsigned short function)
{
	unsigned char* ptr;
	
	switch(function)
	{
		default:
		case PF_NOP:
			break;
			
		case PF_GPS_PIPE:
			printGPSVM(vm);
			break;
		
		case PF_GPS_TEMPSTRING:
			printGPSTempStringVM(vm);
			break;
		
		case PF_LOCAL_DURATION_SINCE_LAST_SYNC:
		case PF_LOCAL_TIME_OR_DURATION:
		case PF_LOCAL_TIME_IF_SET:			
		case PF_LOCAL_TIME:
		case PF_LOCAL_TIME_FILE_STRING:
		case PF_LOCAL_DATE_FILE_STRING:
		case PF_LOCAL_TIME_FILE_STRING_NUMERIC:
	 	case PF_LOCAL_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_FUTURE:
		case PF_VM_DURATION_SINCE_LAST_SYNC:
		case PF_VM_TIME:
		case PF_VM_TIME_IF_SET:
		case PF_VM_TIME_FILE_STRING:
		case PF_VM_DATE_FILE_STRING:
		case PF_VM_TIME_FILE_STRING_NUMERIC:
		case PF_VM_DATE_FILE_STRING_NUMERIC:
		case PF_VM_TIME_OR_DURATION:
		case PF_VM_FILENAME:
		case PF_NMEA:
			ptr=outputFunctionVMToString(vm, tempString, function);
			outputSVM(vm, ptr);
			break;
	}		
	
	switch(function)
	{
		default:
			break;
		
		case PF_NMEA:
			vm->GPS.GPSTxCRC=GPS_START_CHAR;
			break;
	}		
}
