/* $Id: scanner.ll 44 2008-10-23 09:03:19Z tb $ -*- mode: c++ -*- */
/** \file scanner.ll Define the example Flex lexical scanner */

%{ /*** C/C++ Declarations ***/

#include <string>

#include "scanner.h"

/* import the parser's token type into a local typedef */
typedef example::Parser::token token;
typedef example::Parser::token_type token_type;

/* By default yylex returns int, we use token_type. Unfortunately yyterminate
 * by default returns 0, which is not of token_type. */
#define yyterminate() return token::END

/* This disables inclusion of unistd.h, which is not available under Visual C++
 * on Win32. The C++ scanner uses STL streams instead. */
#define YY_NO_UNISTD_H

%}

/*** Flex Declarations and Options ***/

/* enable c++ scanner class generation */
%option c++

/* change the name of the scanner class. results in "ExampleFlexLexer" */
%option prefix="Example"

/* the manual says "somewhat more optimized" */
%option batch

/* enable scanner to generate debug output. disable this for release
 * versions. */
%option debug

/* no support for include files is planned */
%option yywrap nounput

/* enables the use of start condition stacks */
%option stack

/* The following paragraph suffices to track locations accurately. Each time
 * yylex is invoked, the begin position is moved onto the end position. */
%{
#define YY_USER_ACTION  yylloc->columns(yyleng);
%}

%% /*** Regular Expressions Part ***/

 /* code to place at the beginning of yylex() */
%{
    // reset location
    yylloc->step();
%}

 /*** BEGIN EXAMPLE - Change the example lexer rules below ***/

[0-9]+ {
    yylval->integerVal = atoi(yytext);
    return token::INTEGER;
}

"0x"[0-9A-Fa-f][0-9A-Fa-f]* {
    char* yx;
    yylval->integerVal = strtol(yytext+2, &yx, 16);
    return token::HEXINTEGER;
}

"0b"[0-1][0-1]* {
    char* yx;
    yylval->integerVal = strtol(yytext+2, &yx, 2);
    return token::BINARYINTEGER;
}

[0-9]+"."[0-9]* {
    yylval->doubleVal = atof(yytext);
    return token::DOUBLE;
}

"SCRIPT" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSCRIPT;
}

"script" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSCRIPT;
}

"SLEEP" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSLEEP;
}

"sleep" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSLEEP;
}

"SLEEPUNTIL" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSLEEPUNTIL;
}

"sleepUntil" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSLEEPUNTIL;
}

"TIMEUNTIL" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDTIMEUNTIL;
}

"timeUntil" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDTIMEUNTIL;
}

"TIME" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDTIME;
}

"time" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDTIME;
}

"PRECISION" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPRECISION;
}

"precision" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPRECISION;
}

"NEWLINE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::NEWLINE;
}

"newline" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::NEWLINE;
}

"PF" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPFUNCTION;
}

"pf" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPFUNCTION;
}

"IF" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDIF;
}

"if" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDIF;
}

"HEADER" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDHEADER;
}

"header" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDHEADER;
}

"PRINT" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPRINT;
}

"print" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDPRINT;
}

"OPENPIPES" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOPENPIPE;
}

"openPipes" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOPENPIPE;
}

"CLOSEPIPES" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCLOSEPIPE;
}

"closePipes" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCLOSEPIPE;
}

"CLEARFILE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCLEARFILE;
}

"clearFile" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCLEARFILE;
}

"RESET" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDRESET;
}

"reset" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDRESET;
}

"OPENFILE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOPENFILE;
}

"openFile" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOPENFILE;
}

"SERIAL" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSERIAL;
}

"serial" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDSERIAL;
}

"NMEA" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDNMEA;
}

"nmea" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDNMEA;
}

"MATCHNMEA" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDMATCHNMEA;
}

"matchNMEA" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDMATCHNMEA;
}

"ELSE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDELSE;
}

"else" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDELSE;
}

"FUNCTION" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDFUNCTION;
}

"function" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDFUNCTION;
}

"DECIMAL" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDDECIMAL;
}

"decimal" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDDECIMAL;
}

"BASE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDBASE;
}

"base" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDBASE;
}

"CHAR" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCHAR;
}

"char" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDCHAR;
}

"WHILE" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDWHILE;
}

"while" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDWHILE;
}

"AND" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDAND;
}

"and" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDAND;
}

"XOR" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDXOR;
}

"xor" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDXOR;
}

"OR" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOR;
}

"or" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDOR;
}

"==" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDEQ;
}

"!=" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDNEQ;
}

">=" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDGTE;
}

"<=" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDLTE;
}

"<" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDLT;
}

">" {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::KEYWORDGT;
}

[a-zA-Z][a-zA-Z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::OBJECT_NAME;
}

"$"[1-9][0-9]* {
   yylval->integerVal = atoi(yytext+1);
   return token::STACK_VARIABLE;
}

"$"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::LOCAL_VARIABLE;
}

"$$"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::GLOBAL_VARIABLE;
}

"*"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::LOCAL_STR_VARIABLE;
}

"**"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::GLOBAL_STR_VARIABLE;
}

"#"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::DEFINE_CONSTANT;
}

"@@"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::GLOBAL_FUNCTIONNAME;
}

"@"[A-Za-z][A-Za-z0-9_.]* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::LOCAL_FUNCTIONNAME;
}

"//".* {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::COMMENT;
}

["][^"]*["] {
    yylval->stringVal = new std::string(yytext, yyleng);
    return token::STRING;
}

 /* gobble up white-spaces */
[ \t\r]+ {
    yylloc->step();
}

 /* gobble up end-of-lines */
\n {
    yylloc->lines(yyleng); yylloc->step();
}

 /* pass all other characters up to bison */
. {
    return static_cast<token_type>(*yytext);
}

 /*** END EXAMPLE - Change the example lexer rules above ***/

%% /*** Additional Code ***/

namespace example {

Scanner::Scanner(std::istream* in,
         std::ostream* out)
    : ExampleFlexLexer(in, out)
{
}

Scanner::~Scanner()
{
}

void Scanner::set_debug(bool b)
{
    yy_flex_debug = b;
}

}

/* This implementation of ExampleFlexLexer::yylex() is required to fill the
 * vtable of the class ExampleFlexLexer. We define the scanner's main yylex
 * function via YY_DECL to reside in the Scanner class instead. */

#ifdef yylex
#undef yylex
#endif

int ExampleFlexLexer::yylex()
{
    std::cerr << "in ExampleFlexLexer::yylex() !" << std::endl;
    return 0;
}

/* When the scanner receives an end-of-file indication from YY_INPUT, it then
 * checks the yywrap() function. If yywrap() returns false (zero), then it is
 * assumed that the function has gone ahead and set up `yyin' to point to
 * another input file, and scanning continues. If it returns true (non-zero),
 * then the scanner terminates, returning 0 to its caller. */

int ExampleFlexLexer::yywrap()
{
    return 1;
}
