/* 
            
	Opcode Implementations for VM engine...
   
	by Mauro Grassi, October 2010. 
   
	Latest Version... 
  
*/
 
#include "vm.h"
#include "opcodes.h"
#include <math.h>
#include "random.h"
#include <stddef.h>

#if(IS_PC_HOST)

#include "common.h"
DATA_LOGGER_CONTAINER	localUDL;

void deleteLocalFile(unsigned char* filename);
float returnGlobal;

#else

#pragma udata opcodedata

float returnGlobal;

#endif
 
#if(IS_PC_HOST)
 
#include "clock.h"

const instructionFunction opcodes[256]=

#else 

#include "clock.h"
#include "adc.h"
#include "uart.h"
#include "led.h"
#include "i2c.h"
#include "main.h" 
#include "capture.h"
#include "oneWire.h"
#include "io.h"
#include "hardwarestate.h"
#include "mystring.h"
#include "gps.h"
#include "dee.h"

const rom instructionFunction opcodes[256]=
#endif
{
	haltFunction, 	negFunction, 	notFunction, 	rcpFunction, 	clrFunction, 	0, 				0, 				0, 
	0, 				0, 				0, 				0, 				0, 				0, 				0, 				0,

	modFunction, 	modFunction, 	modFunction, 	modFunction, 	mulFunction, 	mulFunction, 	mulFunction, 	mulFunction,
	orFunction,		orFunction,		orFunction,		orFunction,		powFunction,	powFunction,	powFunction,	powFunction,

	subFunction,	subFunction,	subFunction,	subFunction,	xorFunction,	xorFunction,	xorFunction,	xorFunction,
	addFunction,	addFunction,	addFunction,	addFunction,	andFunction,	andFunction,	andFunction,	andFunction,

	divFunction,	divFunction,	divFunction,	divFunction, 	0,				0,				0,				0,
	0, 				0, 				0, 				0, 				0, 				0, 				0, 				0,

	movwFunction, 	movwFunction,   movwFunction,	movwFunction,	movwbFunction, 	movwbFunction,  movwbFunction,	movwbFunction,	
	0, 				0, 				0, 				0, 				0, 				0, 				0, 				0,

	movwrFunction,	movwrFunction,  movwrFunction,  movwrFunction, 	movwrbFunction,	movwrbFunction, movwrbFunction, movwrbFunction,
	0, 				0, 				0, 				0, 				0, 				0, 				0, 				0,

	printdFunction, printdFunction, printdFunction, printdFunction, printbFunction, printbFunction, printbFunction, printbFunction,
	printcFunction, 0,		 		0, 				0,				0,		 		0,				0,				0,
	
	openFFunction,  clearFFunction, serialFunction,	mnmeaFunction, 	nmeaFunction,	0,		 		0,				0,
	0, 				0, 				0,		 		0, 				0,				0,		 		0,				0,

	isequFunction,	isequFunction,	isequFunction,	isequFunction,	isneqFunction,	isneqFunction,	isneqFunction,	isneqFunction,
	isgtFunction,	isgtFunction,	isgtFunction,	isgtFunction,	isgteFunction,	isgteFunction,	isgteFunction,	isgteFunction,

	isltFunction,	isltFunction,	isltFunction,	isltFunction,	islteFunction,	islteFunction,	islteFunction,	islteFunction,
	0, 				0, 				0,		 		0, 				0,				0,		 		0,				0,

	openPpFunction, closePpFunction,0,			 	0,		 		0,				0,				0,				0,					
	0, 				0, 				0,		 		0, 				0,				0,		 		0,				0,
   
	0, 				0, 				0,		 		0, 				0,				0,		 		0,				0,
	0, 	    		0, 	 			0,		 		0, 				0,				0,		 		0,				0,
	
 	braFunction,	brzFunction,	brnzFunction,	callFunction,	returnFunction,	glCallFunction,	hdrWrFunction,	0,
	setDLFunction, 	setDLFunction,  setDLFunction,  setDLFunction,   0,				0,		 		0,				0,

	movslwFunction, movsgwFunction, movdlwFunction, movdgwFunction,	movsdwFunction, movsdwFunction, movsdwFunction, movsdwFunction,
	movsFunction, 	clrDSFunction, 	0, 				0,				0,		 		0,				0,              0,                
	
	printWFunction, printStFunction,sleepFunction,	precFunction,	printfFunction,	sleepUFunction,	clearTFunction, movTWFunction,
	timeUFunction, 	0,				0, 				0,				0,		 		0,				0,				0,

	pushFunction, 	pushnFunction,	popFunction,	popindFunction, popindbFunction,szShortFunction,szLongFunction,	0,
	0, 				0, 				0, 				0,		 		0,				nopFunction, 	resetFunction,	haltFunction
};

#if(IS_PC_HOST)
const unsigned char opcodeNames[256][16]
=
{
	"HALT",			"NEG", 			"NOT", 			"RCP", 			"CLR", 			"-", 			"-", 			"-", 
	"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-",

	"MOD", 			"MOD", 			"MOD", 			"MOD", 			"MUL", 			"MUL", 			"MUL", 			"MUL",
	"OR",			"OR",			"OR",			"OR",			"POW",			"POW",			"POW",			"POW",

	"SUB",			"SUB",			"SUB",			"SUB",			"XOR",			"XOR",			"XOR",			"XOR",
	"ADD",			"ADD",			"ADD",			"ADD",			"AND",			"AND",			"AND",			"AND",

	"DIV",			"DIV",			"DIV",			"DIV", 			"-",			"-",			"-",			"-",
	"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-",

	"MOV W", 		"MOV W",   		"MOV W", 		"MOV W",		"MOV W B", 		"MOV W B",  	"MOV W B",		"MOV W B",	
	"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-",

	"MOVR W",		"MOVR W",  		"MOVR W",  		"MOVR W", 		"MOVR W B",		"MOVR W B", 	"MOVR W B", 	"MOVR W B",
	"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-", 			"-",

	"PRINT D",		"PRINT D", 		"PRINT D", 		"PRINT D",		"PRINT B",		"PRINT B",		"PRINT B",		"PRINT B",
	"PRINT C",		"-",		 	"-", 			"-",			"-",		 	"-",			"-",			"-",
	
	"OPEN FILE", 	"CLEAR FILE", 	"SERIAL",	 	"MATCH NMEA", 	"NMEA",			"-",			"-",			"-",
	"-", 			"-", 			"-",	 		"-", 			"-",			"-",	 		"-",			"-",

	"IS EQU",		"IS EQU",		"IS EQU",		"IS EQU",		"IS NEQ",		"IS NEQ",		"IS NEQ",		"IS NEQ",
	"IS GT",		"IS GT",		"IS GT",		"IS GT",		"IS GTE",		"IS GTE",		"IS GTE",		"IS GTE",

	"IS LT",		"IS LT",		"IS LT",		"IS LT",		"IS LTE",		"IS LTE",		"IS LTE",		"IS LTE",
	"-", 			"-", 			"-",			"-", 			"-",			"-",		 	"-",			"-",

	"OPEN PIPE",	"CLOSE PIPE",	"-",	 		"-",	 		"-",			"-",			"-",			"-",					
	"-", 			"-",			"-",	 		"-", 			"-",			"-",		 	"-",			"-",
   
	"-", 			"-",			"-", 			"-", 			"-",			"-",			"-",			"-",
	"-", 	   		"-", 	 		"-",	 		"-", 			"-",			"-",	 		"-",			"-",
	
 	"BRA",			"BRZ",			"BRNZ",			"CALL",			"RETURN",		"GLOBAL",		"HDR WR",		"-",
	"SET DL",		"SET DL",		"SET DL",		"SET DL",   	"-",			"-",	 		"-",			"-",

	"MOVS LW",		"MOVS GW",		"MOVD LW",		"MOVD GW",		"MOVSDW",		"MOVSDW",		"MOVSDW",		"MOVSDW",	
	"MOVS",			"CLR DS,SS",	"-", 			"-",			"-",		 	"-",			"-",            "-",                
	
	"PRINT W",		"PRINT STR",	"SLEEP",		"PREC",			"PRINTF",		"SLEEP UNTIL",	"CLR T",		"MOV T",
	"TIME UNTIL",	"-",			"-", 			"-",			"-",	 		"-",			"-",			"-",

	"PUSH",			"PUSH N",		"POP",			"POP IND",		"POP IND B",	"SIZE SHORT",	"SIZE LONG",	"-",
	"-", 			"-", 			"-", 			"-",		 	"-",			"NOP",			"RESET",		"HALT"
};
#endif

#if(IS_PC_HOST)
const unsigned char opcodeArgTypeTable[256]=
#else
const rom unsigned char opcodeArgTypeTable[256]=
#endif
{
	NO_ARG_TYPE, 	NO_ARG_TYPE, 	NO_ARG_TYPE, 	NO_ARG_TYPE, 	NO_ARG_TYPE, 	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE, 
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE, 	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE, 	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	NO_ARG_TYPE, 	BYTE_ARG_TYPE,	REF_ARG_TYPE, 	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	
	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE,	LIT_ARG_TYPE,	REF_ARG_TYPE,	LITI_ARG_TYPE,	NO_ARG_TYPE,	LIT_ARG_TYPE,	REF_ARG_TYPE,	LITI_ARG_TYPE,
	NO_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	
	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE, 	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,

	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE, 	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE,	NO_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

 	SHORT_ARG_TYPE,	SHORT_ARG_TYPE,	SHORT_ARG_TYPE,	USHORT_ARG_TYPE,BYTE_ARG_TYPE,	USHORT_ARG_TYPE,USHORT_ARG_TYPE,ERR_ARG_TYPE,
	NO_ARG_TYPE,	LIT_ARG_TYPE,	REF_ARG_TYPE,	LITI_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
 
	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE, 	LIT_ARG_TYPE,	REF_ARG_TYPE, 	LITI_ARG_TYPE,
	NO_ARG_TYPE,	NO_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,

	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	BYTE_ARG_TYPE,	NO_ARG_TYPE,
	NO_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,
 
	NO_ARG_TYPE, 	BYTE_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,    NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	ERR_ARG_TYPE,
	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	ERR_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE,	NO_ARG_TYPE
};
   
#if(IS_PC_HOST)
const int globalFunctionArgTable[]
#else
const rom char globalFunctionArgTable[]
#endif
=
{
//	0	1	2	3	4	5	6	7	8	9	A	B	C	D	E	F
//0
	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//1
	0,	1,	1,	1,	0,	1,	0,	1,	0,	1,	0,	1,	0,	0,	0,	0,
//2
	1,	0,	2,	3,	1,	2,	2,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//3  
	4,	0,	0,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//4
	1,	1,	1,	1,	1,	1,	1,	1,	1,	1,	1,	1,	1,	2,	2,	2,
//5
	2,	1,	1,	1,	1,	1,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//6    
	2,	2,	1,	1,	1,	1,	1,	1,	1,	2,	0,	0,	0,	0,	0,	0,
//7      
	1,	1,	1,	1,	1,	1,	2,	1,	1,	0,	0,	0,	0,	0,	0,	0,
//8       
	1,	1,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//9
	1,	2,	1,	2,	1,	2,	1,	2,	0,	0,	0,	0,	0,	0,	0,	0,
//A
	0,	1,	1,	1,	1,	1,	1,	1,	2,	2,	2,	2,	2,	2,	2,	2,
//B 
	2,	2,	2,	1,	1,	1,	2,	1,	1,	2,	2,	2,	4,	0,	0,	0,
//C
	0,	4,	2,	0,	1,	1,	1,	4,	0,	0,	0,	0,	0,	0,	0,	0,
//D
	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//E
	5,	0,	1,	1,	1,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
//F
	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,	0,
};

/* 

	The address of the global variable is calculated as follows. 
	Suppose the global address is G.
	
	Compute:
	
		index = G>>(GLOBAL_ADDRESS_SHIFT);
		offset= G & (GLOBAL_ADDRESS_OFFSET);
		
		ptr = globalAddressPtrs[index]+offset;

*/

#if(IS_PC_HOST)
const char globalVariablesSizes[NUM_GLOBAL_VARS]
#else
const rom char globalVariablesSizes[NUM_GLOBAL_VARS]
#endif
=
{
	0x00 + 1,
	0x00 + sizeof(tempString),
	/* 0x80 indicates floating point variable */
	0x80 + sizeof(float),
	0x00 + 0x7F,
	0x00 + sizeof(thisUDL.dataLogger.ven.vmPtr),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmNum),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmState),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmMode),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmID),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmSelected),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmLogFileCache),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmLogFileName),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmSleepPeriod),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmMinimumPeriod),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmExecLimit),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmRecoveryTime),
	0x00 + sizeof(thisUDL.dataLogger.ven.vmMinimumSleepPeriod),
	0x00 + 0
};

#if(IS_PC_HOST)

#if 0
/* For the PC Host, this is defined in the global.cpp file... */
unsigned char* globalVariablesAddresses[NUM_GLOBAL_VARS]
=
{
	(const unsigned char*)INDIRECT_OFFSET,
	(const unsigned char*)&tempString[0],
	(const unsigned char*)&returnGlobal,
	(const unsigned char*)&localUDL.dataLogger.ven,
	(const unsigned char*)&localUDL.dataLogger.ven.vmPtr,
	(const unsigned char*)&localUDL.dataLogger.ven.vmNum,
	(const unsigned char*)&localUDL.dataLogger.ven.vmState,
	(const unsigned char*)&localUDL.dataLogger.ven.vmMode,
	(const unsigned char*)&localUDL.dataLogger.ven.vmID,
	(const unsigned char*)&localUDL.dataLogger.ven.vmSelected,
	(const unsigned char*)&localUDL.dataLogger.ven.vmLogFileCache,
	(const unsigned char*)&localUDL.dataLogger.ven.vmLogFileName,
	(const unsigned char*)&localUDL.dataLogger.ven.vmSleepPeriod,
	(const unsigned char*)&localUDL.dataLogger.ven.vmMinimumPeriod,
	(const unsigned char*)&localUDL.dataLogger.ven.vmExecLimit,
	(const unsigned char*)&localUDL.dataLogger.ven.vmRecoveryTime,
	(const unsigned char*)&localUDL.dataLogger.ven.vmMinimumSleepPeriod,
	0
};
#endif

#else

const rom unsigned int globalVariablesAddresses[NUM_GLOBAL_VARS]

=
{
	(unsigned int)INDIRECT_OFFSET,
	(unsigned int)(unsigned char*)&tempString[0],
	(unsigned int)(unsigned char*)&returnGlobal,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmPtr,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmNum,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmState,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmMode,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmID,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmSelected,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmLogFileCache,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmLogFileName,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmSleepPeriod,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmMinimumPeriod,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmExecLimit,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmRecoveryTime,
	(unsigned int)(unsigned char*)&thisUDL.dataLogger.ven.vmMinimumSleepPeriod,
	0
};

#endif

/* Helper functions */
#if(IS_PC_HOST)

#else

#pragma code usercode

#endif

unsigned char  convertWToB(float w)
{
	unsigned char c;
	unsigned int  x;
	
	x=(unsigned int)w;
	c=(unsigned char)x;
	return c;
}

VM_RETURN_TYPE pushStack(VIRTUAL_MACHINE* vm, unsigned char* value)
{
	VM_RETURN_TYPE result;
	/* push the value onto the stack */
	result=writeCache(&vm->STACKImage, vm->StackSizePtr, (unsigned char*)value, SIZE_OF_FLOAT);
	if(result==VM_OK)
	{
		vm->StackSizePtr+=SIZE_OF_FLOAT;
	}
	return result;
}

VM_RETURN_TYPE popStack(VIRTUAL_MACHINE* vm, unsigned char* value)
{
	VM_RETURN_TYPE result;
	/* pop the stack into the value pointer */
	if(vm->StackSizePtr>=SIZE_OF_FLOAT)
	{
		vm->StackSizePtr-=SIZE_OF_FLOAT;
		result=readCache(&vm->STACKImage, vm->StackSizePtr, (unsigned char*)value, SIZE_OF_FLOAT);
	}
	else     
	{  
	 	result=VM_STACK_ERROR;
	}
	return result;
}

VM_RETURN_TYPE popStackN(VIRTUAL_MACHINE* vm, unsigned char* values, unsigned short n)
{
	VM_RETURN_TYPE result;
	
	result=VM_OK;
	while((result==VM_OK)&&(n--))
	{
		result=popStack(vm, values);
		values+=SIZE_OF_FLOAT;
	}
	return result;
} 

VM_RETURN_TYPE unlinkStack(VIRTUAL_MACHINE* vm, unsigned char times)
{
	VM_RETURN_TYPE result;
	ADDRESS_TYPE   amount;

	amount=((ADDRESS_TYPE)SIZE_OF_FLOAT*(ADDRESS_TYPE)times);
	if(vm->StackSizePtr>=amount)
	{
		vm->StackSizePtr-=amount;
		result=VM_OK;
	}
	else
	{
		result=VM_STACK_ERROR;
	}
	return result;
}

VM_RETURN_TYPE getLocalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size)
{
	VM_RETURN_TYPE result;
	if(offset!=INDIRECT_OFFSET)
	{
		result=readCache(&vm->RAMImage, offset, (unsigned char*)value, size);
	}
	else
	{
		/* indirect reference */
		offset=(ADDRESS_TYPE)vm->W;
		/* align the address perfectly on a 4 byte boundary */
		#if(LONG_ADDRESSES)
		vm->lastIndirect=(offset+0xC0000000);
		#else
		vm->lastIndirect=(offset+0xC000);
		#endif
		result=readCache(&vm->RAMImage, offset, (unsigned char*)value, size);
	}
	return result;
}

VM_RETURN_TYPE getGlobalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size)
{
	VM_RETURN_TYPE result;
	unsigned char* ptr;
	
	if(offset!=INDIRECT_OFFSET)
	{
		
	}
	else
	{
		/* indirect reference */		
		offset=(ADDRESS_TYPE)vm->W;
		#if(LONG_ADDRESSES)
		vm->lastIndirect=(offset+0xE0000000);
		#else
		vm->lastIndirect=(offset+0xE000);
		#endif
	}
	
	/* We now translate addresses above the GLOBAL_LIMIT to the FILE CACHE! */
	result=VM_OK;
	
	if((offset>=0)&&(offset<GLOBAL_VARS_UPPER))
	{
		ptr=getGlobalPtr(vm, (unsigned short)offset);
		
		if(ptr)
		{
			#if(IS_PC_HOST)
			
			/* Emulate the global memory on the host here */
			while(size>0)
			{
				*value++=*ptr++;
				size--;
			}
				
			#else	
			
			offset=(unsigned short)ptr;
			result=readCache(0, offset, (unsigned char*)value, size);
			
			#endif
	
		}
		else
		{
			while(size--)
			{
				*value++=0;
			}		
		}	
	}
	else
	{
		result=readCache(&vm->FILEImage, (offset-GLOBAL_VARS_UPPER), (unsigned char*)value, size);
	}
	return result;
}

VM_RETURN_TYPE getStackMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value)
{
	ADDRESS_TYPE a;
	VM_RETURN_TYPE result;
	
	if((offset>=1)&&(offset<=255))
	{
		a=(ADDRESS_TYPE)((offset+1)*(ADDRESS_TYPE)SIZE_OF_FLOAT);
		if(vm->StackSizePtr>=a)
		{
			result=readCache(&vm->STACKImage, vm->StackSizePtr-a, (unsigned char*)value, SIZE_OF_FLOAT);
		}
		else
		{
			result=VM_STACK_REF_ERROR;
			(*value)=0;
		}
	}
	else
	{
		result=VM_STACK_REF_ERROR;
		(*value)=0;
	}
	return VM_OK;
}

VM_RETURN_TYPE putLocalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size)
{
	VM_RETURN_TYPE result;
	if(offset!=INDIRECT_OFFSET)
	{
		result=writeCache(&vm->RAMImage, offset, (unsigned char*)value, size);
	}
	else
	{
		result=VM_ADDRESS_ERROR;
	}
	return result;
}

VM_RETURN_TYPE putGlobalMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value, unsigned char size)
{
	VM_RETURN_TYPE result;
	unsigned char* ptr;
	
	result=VM_OK;
	if(offset!=INDIRECT_OFFSET)
	{
		/* We now translate addresses above the GLOBAL_LIMIT to the FILE CACHE! */
		if((offset>=0)&&(offset<GLOBAL_VARS_UPPER))
		{			
			ptr=getGlobalPtr(vm, (unsigned short)offset);
			if(ptr)
			{
				#if(IS_PC_HOST)
				
				/* Emulate the global memory on the host here */
				while(size>0)
				{
					*ptr++=*value++;
					size--;
				}
							
				#else
	
				offset=(unsigned short)ptr;	
			 	result=writeCache(0, offset, (unsigned char*)value, size);
				
				#endif
				
			}
			else
			{
				/* do nothing */		
			}	
		}
		else
		{
			result=writeCache(&vm->FILEImage, offset-GLOBAL_VARS_UPPER, (unsigned char*)value, size);
		}
	}
	else
	{
		result=VM_ADDRESS_ERROR;
	}
	return result;	
}

VM_RETURN_TYPE putStackMemory(VIRTUAL_MACHINE* vm, ADDRESS_TYPE offset, unsigned char* value)
{
	return VM_STACK_REF_ERROR;
}

/* Instruction Functions */

VM_RETURN_TYPE nopFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* no operation */
	return VM_OK;
}

VM_RETURN_TYPE negFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* negate the accumulator */
	vm->W=-vm->W;
	return VM_OK;
}

VM_RETURN_TYPE notFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* not function */
	if(vm->W==0)vm->W=(float)1.0; else vm->W=(float)0.0;
	return VM_OK;
}

VM_RETURN_TYPE rcpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* reciprocal function */
	if(vm->W!=0)vm->W=(float)1.0/vm->W;
	return VM_OK;
}

VM_RETURN_TYPE clrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* clear function */
	vm->W=0;
	return VM_OK;
}

VM_RETURN_TYPE preludeBinaryArg(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;

	result=VM_OK;
	if(varg->type==NO_ARG_TYPE)
	{
		/* implicitly swap the arguments! */
		varg->arg.fArg=vm->W;
		result=popStack(vm, (unsigned char*)&vm->W);
	}
	return result;
}

VM_RETURN_TYPE mulFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->W*=varg->arg.fArg;
	}
	return result;
}

VM_RETURN_TYPE orFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->W=(float)(((unsigned long)vm->W) | ((unsigned long)varg->arg.fArg));
	}
	return result;
}

VM_RETURN_TYPE powFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
			/* normal */
			vm->W=(float)pow(vm->W, varg->arg.fArg);
	} 
	return result;
}

VM_RETURN_TYPE modFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
			/* normal */
			vm->W=(float)fmod(vm->W, varg->arg.fArg);
	}
	return result;
}

VM_RETURN_TYPE subFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		vm->W-=varg->arg.fArg;
	}
	return result;
}

VM_RETURN_TYPE xorFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->W=(float)(((unsigned long)vm->W) ^ ((unsigned long)varg->arg.fArg));
	}
	return result;
}

VM_RETURN_TYPE addFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->W+=varg->arg.fArg;
	}
	return result;
}

VM_RETURN_TYPE setDLFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->DSLIMIT=(ADDRESS_TYPE)varg->arg.fArg;
	}
	return result;
}

VM_RETURN_TYPE andFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		vm->W=(float)(((unsigned long)vm->W) & ((unsigned long)varg->arg.fArg));
	}
	return result;
}

VM_RETURN_TYPE divFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		if(varg->arg.fArg!=0)vm->W/=varg->arg.fArg;
	}
	return result;
}

VM_RETURN_TYPE movwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->W=varg->arg.fArg;
	return VM_OK;
}

VM_RETURN_TYPE movwbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	if(vm->DSLIMIT)
	{
		vm->W=(float)varg->arg.fArg;
		vm->DSLIMIT=0;
	}
	else
	{
		vm->W=(float)varg->arg.bArg;
	}	
	return VM_OK;
}

VM_RETURN_TYPE movwrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;

	switch(varg->type)
	{
	
	case NO_ARG_TYPE:
	case LIT_ARG_TYPE:
	case LITI_ARG_TYPE:
	case ERR_ARG_TYPE:
		result=VM_ARG_ERROR;	
		break;

	case REF_ARG_LOCAL:
		result=putLocalMemory(vm, varg->addr, (unsigned char*)&vm->W, SIZE_OF_FLOAT);
		break;

	case REF_ARG_GLOBAL:
		result=putGlobalMemory(vm, varg->addr, (unsigned char*)&vm->W, SIZE_OF_FLOAT);
		break;
	
	case REF_ARG_STACK:
		result=VM_ARG_ERROR;
		break;		

	default:
		result=VM_ARG_ERROR;
		break;
	}

	return result;
}

VM_RETURN_TYPE movwrbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* byte version */
	VM_RETURN_TYPE result;
	unsigned char c;
	unsigned long x;

	c=(unsigned char)convertWToB(vm->W);
		
	switch(varg->type)
	{
	
	case NO_ARG_TYPE:
	case LIT_ARG_TYPE:
	case LITI_ARG_TYPE:
	case ERR_ARG_TYPE:
		result=VM_ARG_ERROR;	
		break;

	case REF_ARG_LOCAL:
		result=putLocalMemory(vm, varg->addr, &c, 1);
		break;

	case REF_ARG_GLOBAL:
		if(vm->DSLIMIT)
		{
			if(vm->DSLIMIT>sizeof(x))vm->DSLIMIT=sizeof(x);
			x=(unsigned long)(long)vm->W;
			result=putGlobalMemory(vm, varg->addr, (unsigned char*)&x, (unsigned char)vm->DSLIMIT);
			vm->DSLIMIT=0;	
		}
		else
		{
			result=putGlobalMemory(vm, varg->addr, &c, 1);
		}	
		break;
	
	case REF_ARG_STACK:
		result=VM_ARG_ERROR;
		break;		

	default:
		result=VM_ARG_ERROR;
		break;
	}
	return result;
}

VM_RETURN_TYPE isequFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		if(vm->W==varg->arg.fArg)vm->W=1.0; else vm->W=0;
	}
	return result;
}

VM_RETURN_TYPE isneqFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		if(vm->W!=varg->arg.fArg)vm->W=1.0; else vm->W=0;
	}
	return result;
}

VM_RETURN_TYPE isgtFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		if(vm->W>varg->arg.fArg)vm->W=1.0; else vm->W=0;
	}
	return result;
}

VM_RETURN_TYPE isgteFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		if(vm->W>=varg->arg.fArg)vm->W=1.0; else vm->W=0;		
	}
	return result;
}

VM_RETURN_TYPE isltFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
			/* normal */
	 		if(vm->W<varg->arg.fArg)vm->W=1.0; else vm->W=0;
	}
	return result;
}

VM_RETURN_TYPE islteFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		if(vm->W<=varg->arg.fArg)vm->W=1.0; else vm->W=0;
	}
	return result;
}

VM_RETURN_TYPE braFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->PC+=(ADDRESS_TYPE)(varg->arg.nArg);
	return VM_OK;
}

VM_RETURN_TYPE brzFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	if(vm->W==0)vm->PC+=(ADDRESS_TYPE)(varg->arg.nArg);
	return VM_OK;
}

VM_RETURN_TYPE brnzFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	if(vm->W!=0)vm->PC+=(ADDRESS_TYPE)(varg->arg.nArg);
	return VM_OK;
}

VM_RETURN_TYPE callFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	float f;

	/* save address */
	f=(float)vm->PC;
	result=pushStack(vm, (unsigned char*)&f);
	/* jump to calling address */
	vm->PC=(ADDRESS_TYPE)varg->arg.uArg;
	return result;
}

VM_RETURN_TYPE returnFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	float f;

	result=popStack(vm, (unsigned char*)&f);
	if(result==VM_OK)
	{
		/* go to return address */
		vm->PC=(ADDRESS_TYPE)f;
		/* now unlink stack frame */
		result=unlinkStack(vm, varg->arg.bArg);
	}
	return result;
}

VM_RETURN_TYPE hdrWrFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;
	unsigned char numBytes;
	unsigned char offset;
	unsigned char  c;
	unsigned char* ptr;
	/* 
		varg->arg.uArg holds the number of bytes to write & offset 
		information, and the next byte is the start of the bytes to write 
	*/
	numBytes=(unsigned char)(varg->arg.uArg & 0xFF);
	offset=(unsigned char)((varg->arg.uArg>>8) & 0x00FF);
	result=VM_OK;
	ptr=(unsigned char*)vm;
	while((result==VM_OK)&&(numBytes--))
	{
		result=getNextROMByte(vm, &c);
		ptr[offset++]=c;
	}
	return result;
}

VM_RETURN_TYPE pushFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	return pushStack(vm, (unsigned char*)&vm->W);
}

VM_RETURN_TYPE pushnFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;

	result=VM_OK;
	vm->temp=varg->arg.bArg;
	while((result==VM_OK)&&(varg->arg.bArg--))
	{
		result=pushStack(vm, (unsigned char*)&vm->W);
	}
	return result;
}

VM_RETURN_TYPE popFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	return popStack(vm, (unsigned char*)&vm->W);
} 

VM_RETURN_TYPE popindFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/*  
		pops the contents of the stack onto the memory position pointed to
		by the lastIndirect register... 
	*/ 
	VM_RETURN_TYPE result;

	result=popStack(vm, (unsigned char*)&varg->arg.fArg);
	
	#if(LONG_ADDRESSES)
	switch(vm->lastIndirect & 0xF0000000)
	#else
	switch(vm->lastIndirect & 0xF000)
	#endif
	{
	default:
		/* do nothing */
		break;

	#if(LONG_ADDRESSES)
	case 0xC0000000:
	#else
	case 0xC000:
	#endif
	
		/* Local Reference */
		result=putLocalMemory(vm, vm->lastIndirect & SIZE_OF_FLOAT_MASK, (unsigned char*)&varg->arg.fArg, SIZE_OF_FLOAT);
		break;

	#if(LONG_ADDRESSES)
	case 0xE0000000:
	#else
	case 0xE000:
	#endif
		/* Global */
		result=putGlobalMemory(vm, vm->lastIndirect & SIZE_OF_FLOAT_MASK, (unsigned char*)&varg->arg.fArg, SIZE_OF_FLOAT);
		break;
	}
	return result;  
} 

VM_RETURN_TYPE popindbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/*
		pops the contents of the stack onto the memory position pointed to
		by the lastIndirect register... 
	*/
	unsigned char c;
	VM_RETURN_TYPE result;
	
	result=popStack(vm, (unsigned char*)&varg->arg.fArg);
	c=(unsigned char)convertWToB(varg->arg.fArg);
	
	#if(LONG_ADDRESSES)
	switch(vm->lastIndirect & 0xF0000000)
	#else
	switch(vm->lastIndirect & 0xF000)
	#endif
	{
	default:
		/* do nothing */
		break;

	#if(LONG_ADDRESSES)
	case 0xC0000000:
	#else
	case 0xC000:
	#endif
	
		/* Local Reference */
		result=putLocalMemory(vm, vm->lastIndirect & SIZE_OF_BYTE_MASK, (unsigned char*)&c, 1);
		break;

	#if(LONG_ADDRESSES)
	case 0xE0000000: 
	#else
	case 0xE000:
	#endif
		/* Global */
		result=putGlobalMemory(vm, vm->lastIndirect & SIZE_OF_BYTE_MASK, (unsigned char*)&c, 1);
		break;
	}
	return result;
} 
 
VM_RETURN_TYPE resetFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{ 
	vm->execState=VM_RUN_MODE;
	vm->execMode=VM_EXEC_MODE_DEFAULT;
	vm->firstPor=0;
	vm->PC=vm->resetVector;
	vm->DS=0;
	vm->SS=0;
	vm->DSLIMIT=0;
	vm->time.secs=0;
	vm->time.mins=0;
	vm->time.hours=0;
	vm->time.day=1;
	vm->time.month=1;
	vm->time.year=2011;
	vm->time.wday=5;
	vm->time.show=SHOW_TIME_DEFAULT;
	vm->time.updated=0;
	vm->addressModes=0;
	vm->W=0.0; 
	vm->StackSizePtr=0;	
	vm->lastError=VM_OK;
	vm->IR=VMENV_NOP;
	vm->execLimit=0;
	vm->execPriority=VM_EXEC_PRIORITY_DEFAULT;
	vm->execDone=0;
	vm->lastIndirect=0;
	vm->pipeMode=DEFAULT_PIPE_MODE;
	vm->typeMode=DEFAULT_TYPE_MODE;
	vm->temp=0;
	vm->tempPipe=0;	
	vm->sleepMode=0;
	vm->fName[0]='\0';
	vm->timeScaling=0;
	vm->timeSpeed=1.0;
	vm->timeOffset=0;
	clearHardwareDescriptor(&vm->hardwareState);
	#if(IS_PC_HOST)
	#else
		vm->GPS.GPSTempPtr=0;
		vm->GPS.GPSTempString[0]='\0';
		vm->GPS.GPSOutPtr=0;
		vm->GPS.GPSLongitude=0;
		vm->GPS.GPSLatitude=0;
		vm->GPS.GPSTxCRC=0;
		vm->GPS.GPSSpeed=0;
		vm->GPS.GPSCourse=0;
		clearUARTRx(vm);
	#endif	
	return VM_OK;
}

VM_RETURN_TYPE haltFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	changeVMState(vm, VM_HALT_MODE);
	return VM_OK;
} 

VM_RETURN_TYPE printWFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	short numdec;
	
	switch(vm->typeMode)
	{
	
	case TIME_TYPE:
		outputFunctionVM(vm, (unsigned short)PF_VM_TIME_FUTURE);
		vm->typeMode=NORMAL_TYPE;
		break;
	
	default:
		vm->typeMode=NORMAL_TYPE;
	case NORMAL_TYPE:
		numdec=(short)(vm->pipeMode & NUMDEC_MASK);
		outputUFVM(vm, vm->W, numdec);
		break;	
	}
	return VM_OK;
}

VM_RETURN_TYPE printfFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* internal printing functions ! */
	unsigned short ifunction;
	ifunction=(unsigned short)vm->W;
	outputFunctionVM(vm, ifunction);
	return VM_OK;
}

VM_RETURN_TYPE printdFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned short numdec;
	
	VM_RETURN_TYPE result;
	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		numdec=(unsigned short)(varg->arg.fArg);
		outputFVM(vm, vm->W, numdec);
	}
	return result;
}

VM_RETURN_TYPE printbFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	VM_RETURN_TYPE result;

	result=preludeBinaryArg(vm, varg);
	if(result==VM_OK)
	{
		/* normal */
		myultoabase((unsigned long)vm->W, (unsigned char*)&tempString[0], ((((unsigned int)varg->arg.fArg) & 0xF0) >> 4), (((unsigned int)varg->arg.fArg) & 0x0F)+1, 31, '0');
		outputSVM(vm, (unsigned char*)&tempString[0]);
	}
	return result;
}

VM_RETURN_TYPE printcFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	outputCharVM(vm, (unsigned char)vm->W);
	return VM_OK;
}

VM_RETURN_TYPE getNextROMByte(VIRTUAL_MACHINE* vm, unsigned char *c)
{
	VM_RETURN_TYPE result;
	
	result=readCacheByte(&vm->ROMImage, vm->PC, c);
	if(vm->PC<(vm->ROMImage.cacheMaxAddress-1))vm->PC++;
	else
	{  
		result=VM_ADDRESS_ERROR;
	}
	return result;
}

VM_RETURN_TYPE printStFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{ 
	unsigned char c;
	VM_RETURN_TYPE result;
	ADDRESS_TYPE prev;

	result=VM_OK;
	c=' ';
	while((c!='\0')&&(result==VM_OK))
	{
		prev=vm->DS;
		result=getNextROMByte(vm, &c);
		if((c!='\0')||(vm->DS!=0))
		{
			outputCharVM(vm, c);
		}
	}
	if(c=='\0')vm->DS=prev;
	return result;
}

unsigned char fileNamePipeFunctionVM(VIRTUAL_MACHINE* vm, unsigned short mode)
{
	unsigned char result;
	
	result=0;
	if(!(vm->pipeMode & mode))
	{
		/* begin the command */
		vm->pipeMode|=mode;
		vm->tempPipe=0;
	}
	else
	{
		/* done! */
		if(vm->tempPipe==0)
		{
			/* default name */
			vm->fName[VM_FILENAME_MAX-1]='\0';
			if(vm->tempPipe>VM_FILENAME_MAX)vm->tempPipe=VM_FILENAME_MAX;
		}
		else
		{ 
	 		vm->fName[vm->tempPipe]='\0';
		}	
		vm->pipeMode&=~mode;
		vm->tempPipe=0;
		result=1;
	}
	return result;	
}

unsigned char forcePipeFunctionVM(VIRTUAL_MACHINE* vm, unsigned short mode)
{
	unsigned char result;	
	result=0;
	if(vm->tempPipe==0)
	{
		/* begin the command */
		vm->tempPipe=(unsigned long)vm->pipeMode;
		vm->pipeMode=mode;
	}
	else
	{
		/* done! */
		vm->pipeMode=(unsigned short)vm->tempPipe;
		vm->tempPipe=0;
		result=1;
	}
	return result;	
}

VM_RETURN_TYPE serialFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char i;
	VM_RETURN_TYPE result;
	
	i=forcePipeFunctionVM(vm, SERIAL_PIPE_MODE);
	result=VM_OK;
	if(i)
	{ 
		/* finished, so execute the command */
	}
	return result;
}

VM_RETURN_TYPE mnmeaFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char i;
	VM_RETURN_TYPE result;
	
	i=forcePipeFunctionVM(vm, NMEA_MATCH_STRING_MODE);
	result=VM_OK;
	if(i)
	{ 
		/* finished, so execute the command */
		vm->GPS.GPSTempString[vm->GPS.GPSTempPtr]='\0';
	}
	else
	{
		/* Begin */
		vm->GPS.GPSTempPtr=0;
	}
	return result;
}

VM_RETURN_TYPE nmeaFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char i;
	VM_RETURN_TYPE result;
	
	i=forcePipeFunctionVM(vm, NMEA_STRING_MODE);
	result=VM_OK;
	if(i)
	{ 
		/* finished, so execute the command */
	}
	else
	{
		vm->GPS.GPSTxCRC=GPS_START_CHAR;
	}
	return result;
}

VM_RETURN_TYPE openFFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char i;
	VM_RETURN_TYPE result;
	
	i=fileNamePipeFunctionVM(vm, FILENAME_PIPE_MODE);
	result=VM_OK;
	if(i)
	{ 
		/* finished, so execute the command */
		#if(IS_PC_HOST)		
			clearFileCache(vm);
		#else
			initCache(&vm->FILEImage, FILECACHE_MODE, FILE_CACHE_ID + (VM_ID_MASK & vm->execID), FILE_CACHE_SIZE, MAX_FILE_CACHE_SIZE, (unsigned char*)&thisUDL.generalMemory[CACHE_FILE_OFFSET], &vm->fName[0]);
		#endif
	}
	return result;
}
 
VM_RETURN_TYPE clearFFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char i;
	VM_RETURN_TYPE result;
	
	i=fileNamePipeFunctionVM(vm, FILENAME_PIPE_MODE);
	result=VM_OK;
	if(i)
	{ 
		/* finished, so execute the command */
		#if(IS_PC_HOST)
			deleteLocalFile(&vm->fName[0]);
		#else
			vm->W=(float)f_unlink((char*)vm->fName);
		#endif
			vm->FILEImage.cacheEndPointer=0;
			clearCache(&vm->FILEImage, vm->FILEImage.cacheID, vm->FILEImage.cacheSize, vm->FILEImage.cacheMaxAddress, vm->FILEImage.cacheStart);
	} 
	return result;
}   

VM_RETURN_TYPE openPpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->pipeMode|=(unsigned short)vm->W;
	return VM_OK;
}

VM_RETURN_TYPE closePpFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->pipeMode&=~((unsigned short)vm->W);
	return VM_OK;
}

VM_RETURN_TYPE sleepFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->sleepMode=(unsigned long)vm->W;
	changeVMState(vm, VM_SLEEP_MODE);
	return VM_OK;
}
 
VM_RETURN_TYPE sleepUFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
 	long dt;
 	TIME_T argTime;

	argTime.show=(SHOW_MONTH | SHOW_DAY | SHOW_HOURS | SHOW_MINUTES | SHOW_SECONDS);

	/* get the current time! */

	#if(IS_PC_HOST)

	if(vm->timeScaling)getTimeVM(vm, &argTime);
	else igetTime(&argTime);

	#else
	
	igetTime(&argTime);
	
	#endif

	getFutureArgumentTime(&vm->time, &argTime);
	dt=getDurationTotalSeconds(&vm->time, &argTime);
   	if(dt>0)
	{
		vm->sleepMode=(unsigned long)dt; 
	}
	else 
	{
		vm->sleepMode=0;
	}	
	vm->typeMode=NORMAL_TYPE;
	changeVMState(vm, VM_SLEEP_MODE);
	return VM_OK;
}

VM_RETURN_TYPE timeUFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
 	long dt;
 	
 	#if(IS_PC_HOST)
 	
    dt=getDurationTotalSecondsFutureArgumentVM(vm, &vm->time);
	
	#else
	
	dt=getDurationTotalSecondsFutureArgument(&vm->time);
	
	#endif
	
	if(dt>0)
	{
		vm->W=(float)dt; 
	}
	else 
	{
		vm->W=0;
	}
	vm->typeMode=NORMAL_TYPE;	
	return VM_OK;
}

VM_RETURN_TYPE clearTFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* takes one argument which is the number of args to clear in order */
	TIME_T 	iTime;
	
	#if(IS_PC_HOST)
	
	getTimeVM(vm, &iTime);

	#else
	
	igetTime(&iTime);
	
	#endif
	
	vm->time.secs=iTime.secs;
	vm->time.mins=iTime.mins;
	vm->time.hours=iTime.hours;
	vm->time.day=iTime.day;
	vm->time.month=iTime.month;
	vm->time.year=iTime.year;
	vm->time.show=0;
	/* advance the pointer ! */
	vm->time.wday=varg->arg.bArg;
	return VM_OK;
}

VM_RETURN_TYPE movTWFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	/* wday is used as an internal pointer to the time structure in the VM */
	unsigned char c;

	c=convertWToB(vm->W);
	switch(vm->time.wday)
	{
		case 0:
			/* month, yes the American way makes more sense here! */
			if((c>=1)&&(c<=12))
			{
				vm->time.month=c;
				vm->time.show|=SHOW_MONTH;
			}		
			else
			{
				vm->time.month=1;
				vm->time.show|=SHOW_MONTH;
			}
			vm->time.wday++;
			break;
			
		case 1:
			/* day */
			if((c>=1)&&(c<=31))
			{
				vm->time.day=c;
				vm->time.show|=SHOW_DAY;
			}
			else
			{
				vm->time.day=1;
				vm->time.show|=SHOW_DAY;
			}
			vm->time.wday++;
			break;
			
		case 2:
			/* hour */
			if((c>=0)&&(c<=23))
			{
				vm->time.hours=c;
				vm->time.show|=SHOW_HOURS;
			}
			else
			if(c==24)
			{
				vm->time.hours=0;
				vm->time.show|=SHOW_HOURS;
			}
			vm->time.wday++;
			break;
			
		case 3:
			/* mins */
			if((c>=0)&&(c<=59))
			{
				vm->time.mins=c;
				vm->time.show|=SHOW_MINUTES;
			}
			else
			if(c==60)
			{
				vm->time.mins=0;
				vm->time.show|=SHOW_MINUTES;
			}
			vm->time.wday++;
			break;
			
		default:
		case 4:
			/* secs */
			if((c>=0)&&(c<=59))
			{
				vm->time.secs=c;
				vm->time.show|=SHOW_SECONDS;
			}
			else
			if(c==60)
			{
				vm->time.secs=0;
				vm->time.show|=SHOW_SECONDS;
			}
			/* load the non standard type ! */
			vm->typeMode=TIME_TYPE;
			break;
	}
	return VM_OK;
}

VM_RETURN_TYPE precFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->pipeMode&=~NUMDEC_MASK;
	vm->pipeMode|=(NUMDEC_MASK & (unsigned short)vm->W);
	return VM_OK;
}

VM_RETURN_TYPE movslwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->SS=(ADDRESS_TYPE)vm->W;
	vm->addressModes&=0xF3;
	vm->addressModes|=0x00;
	return VM_OK;
}

VM_RETURN_TYPE movsgwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->SS=(ADDRESS_TYPE)vm->W;
	vm->addressModes&=0xF3;
	vm->addressModes|=0x08;
	return VM_OK;
}

VM_RETURN_TYPE clrDSFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->DS=0;
	vm->SS=0;
	vm->DSLIMIT=0;
	return VM_OK;
}

VM_RETURN_TYPE movdlwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->DS=(ADDRESS_TYPE)vm->W;
	vm->addressModes&=0xFC;
	vm->addressModes|=0x00;
	return VM_OK;
}
 
VM_RETURN_TYPE movdgwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->DS=(ADDRESS_TYPE)vm->W;
	vm->addressModes&=0xFC;
	vm->addressModes|=0x02;
	return VM_OK;
}

VM_RETURN_TYPE readPointerMemory(VIRTUAL_MACHINE* vm, unsigned char mode, unsigned char* c)
{
	unsigned char adMode;
	VM_RETURN_TYPE result;
		
	switch(mode)
	{
		case MODE_SS:
		case MODE_SS_POST_INC:
			adMode=((vm->addressModes & 0x0C)>>2);
			break;
		
		default:
		case MODE_DS:
		case MODE_DS_POST_INC:
			adMode=(vm->addressModes & 0x03);
			break;
	}
	    
	switch(adMode)
	{
		case 0x00:
			/* D or S points to a local variable */
			switch(mode)
			{
				default:
				case MODE_DS:
					result=readCacheByte(&vm->RAMImage, vm->DS, c);
					break;
				
				case MODE_DS_POST_INC:
					result=readCacheByte(&vm->RAMImage, vm->DS, c);
					if(vm->DS<(vm->RAMImage.cacheMaxAddress-1))vm->DS++;
					else
					{
						result=VM_ADDRESS_ERROR;					
					}
					break;
				
				case MODE_SS:
					result=readCacheByte(&vm->RAMImage, vm->SS, c);
					break;
					
				case MODE_SS_POST_INC:
					result=readCacheByte(&vm->RAMImage, vm->SS, c);
					if(vm->SS<(vm->RAMImage.cacheMaxAddress-1))vm->SS++;
					else
					{
						result=VM_ADDRESS_ERROR;					
					}
					break;
			}
			break;
			
		case 0x01:
			/* D or S points to the stack */
			result=VM_OK;
			break;
			
		case 0x02:
			/* D or S points to a global reference */
			switch(mode)
				{
					default:
					case MODE_DS:
						result=readCacheByte(0, vm->DS, c);
						break;
					
					case MODE_DS_POST_INC:
						result=readCacheByte(0, vm->DS, c);
						vm->DS++;
						break;
					
					case MODE_SS:
						result=readCacheByte(0, vm->SS, c);
						break;
						
					case MODE_SS_POST_INC:
						result=readCacheByte(0, vm->SS, c);
						vm->SS++;
						break;
				}
				break;
					
		default:
		case 0x03:
			result=VM_ARG_ERROR;
			break;		
	}
	return result;
}

VM_RETURN_TYPE writePointerMemory(VIRTUAL_MACHINE* vm, unsigned char mode, unsigned char* c)
{
	unsigned char adMode;
	VM_RETURN_TYPE result;
	unsigned char limit;
	
	result=VM_OK;
	limit=0;
	switch(mode)
	{
		case MODE_SS:
		case MODE_SS_POST_INC:
			adMode=((vm->addressModes & 0x0C)>>2);
			break;
		
		default:
		case MODE_DS_POST_INC:
			limit=1;
		case MODE_DS:
			adMode=(vm->addressModes & 0x03);		
			break;
	}
	    
	switch(adMode)
	{
		case 0x00:
			/* D or S points to a local variable */
			switch(mode)
			{
				default:
				case MODE_DS:
					if(vm->DS==0)
					{
						outputCharVM(vm, *c);
					}
					else
					{
						result=writeCacheByte(&vm->RAMImage, vm->DS, c);
					}
					break;
				
				case MODE_DS_POST_INC:
					if(vm->DS==0)
					{
						outputCharVM(vm, *c);	
					}
					else
					{
						if((!limit)||(vm->DS<vm->DSLIMIT))
						{
						result=writeCacheByte(&vm->RAMImage, vm->DS, c);
						if(vm->DS<(vm->RAMImage.cacheMaxAddress-1))vm->DS++;
						else
						{
								result=VM_ADDRESS_ERROR;					
						}
						}
						else
						{
							result=VM_LIMIT;
						}			
					}
					break;
				
				case MODE_SS:
					result=writeCacheByte(&vm->RAMImage, vm->SS, c);
					break;
					
				case MODE_SS_POST_INC:
					result=writeCacheByte(&vm->RAMImage, vm->SS, c);
					if(vm->SS<(vm->RAMImage.cacheMaxAddress-1))vm->SS++;
					else
					{
						result=VM_ADDRESS_ERROR;					
					}
					break;
			}
			break;
			
		case 0x01:
			/* D or S points to the stack */
			result=VM_OK;
			break;
			
		case 0x02:
			/* D or S points to a global reference */
			switch(mode)
				{
					default:
					case MODE_DS:
						result=putGlobalMemory(vm, vm->DS, c, 1);
						break;
					
					case MODE_DS_POST_INC:
						if((!limit)||(vm->DS<vm->DSLIMIT))
						{
							result=putGlobalMemory(vm, vm->DS, c, 1);
							vm->DS++;	
						}
						else
						{
							result=VM_LIMIT;
						}
						break;
					
					case MODE_SS:
						result=putGlobalMemory(vm, vm->SS, c, 1);
						break;
						
					case MODE_SS_POST_INC:
						result=putGlobalMemory(vm, vm->SS, c, 1);
						vm->SS++;
						break;
			 	}
				break;
					
		default:
		case 0x03:
			result=VM_ARG_ERROR;
			break;		
	}
	return result;
}

VM_RETURN_TYPE movsdwFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
#if 0
	unsigned char c;
	VM_RETURN_TYPE result;
	
	result=VM_ERROR;	

	/* not implemented to save program space, not currently used by the compiler */
	
	switch(varg->type)
	{
	case NO_ARG_TYPE:
		/* no argument, copies the value of W to the destination address in D */
		c=convertWToB(vm->W);
		result=writePointerMemory(vm, MODE_DS, &c);
		break;
		
	case LIT_ARG_TYPE:
	case LITI_ARG_TYPE:
	case REF_ARG_LOCAL:
	case REF_ARG_GLOBAL:
		result=VM_OK;
		c=' ';
		while((c!='\0')&&(result==VM_OK))
		{
			switch(varg->type)
			{
				case LIT_ARG_TYPE:
					result=readCacheByte(&vm->ROMImage, vm->PC, &c);
					if(vm->PC<(vm->ROMImage.cacheMaxAddress-1))vm->PC++;
					else
				 	{  
						result=VM_ADDRESS_ERROR;
					}
					break;
					
				case REF_ARG_LOCAL:
					result=readCacheByte(&vm->RAMImage, varg->addr, &c);
					if(varg->addr<(vm->RAMImage.cacheMaxAddress-1))varg->addr++;
					else
				 	{  
						result=VM_ADDRESS_ERROR;
					}
					break;
				
				case REF_ARG_GLOBAL:
					result=readCacheByte(0, varg->addr, &c);
					varg->addr++;
					break;			
			}
						
			if(c!='\0')
			{
				result=writePointerMemory(vm, MODE_DS_POST_INC, &c);								
			}
		}
		c='\0';
		result=writePointerMemory(vm, MODE_DS, &c);
		break;
			
	case ERR_ARG_TYPE:
		result=VM_ARG_ERROR;	
		break;

	
	case REF_ARG_STACK:
		result=VM_ARG_ERROR;
		break;		

	default:
		result=VM_ARG_ERROR;
		break;
	}
	return result;
#else
	return VM_ERROR;
#endif
}

VM_RETURN_TYPE movsFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	unsigned char c;
	VM_RETURN_TYPE result;
	unsigned short maxTransfers;
	
	result=VM_OK;
	c=' ';
	/* provide a maximum transfer quota, if no proper string termination is present! */
	maxTransfers=MAX_LOCAL_ALLOCATION_SIZE-1;
	while((c!='\0')&&(result==VM_OK)&&(maxTransfers--))
	{
		result=readPointerMemory(vm, MODE_SS_POST_INC, &c);
		if((result==VM_OK)&&(c!='\0'))
		{
			result=writePointerMemory(vm, MODE_DS_POST_INC, &c);
		}
	}	
	c='\0';
	result=writePointerMemory(vm, MODE_DS, &c);
	return result;
}

VM_RETURN_TYPE glCallFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{ 
	/* Global Function Executors */
	VM_RETURN_TYPE 		result;
	unsigned short 		nArgs;
	unsigned short		i;
	ADDRESS_TYPE		offset;
	TIME_T*				ptime;
	unsigned char  		k;
		
	#if(IS_PC_HOST)
	
	#else
	 
	HARDWARE_DESCRIPTOR* hd;
	unsigned char   l;
	unsigned char*	ptr;
	unsigned char*  dptr;
	
	#endif
	
	result=VM_OK;
	nArgs=0;
	if((varg->arg.uArg>=0)&&(varg->arg.uArg<256))
	{
		nArgs=(unsigned short)globalFunctionArgTable[varg->arg.uArg];
	}
	
	if(nArgs<=VM_MAX_ARGUMENTS)
	{
		if(vm)
		/* 	get the required number of arguments from the stack
			arguments in reversed order to the actual function */
		{
			result=popStackN(vm, (unsigned char*)&varg->f[0], nArgs);
			varg->w=vm->W;
		}
	}
	
	switch(varg->arg.uArg)
	{
	
	/* Maths Functions */
		
	case GL_FUNCTION_INT:
			varg->w=(float)(unsigned int)varg->f[0];
			break;
		
	case GL_FUNCTION_SQRT:
			if(varg->f[0]>=0)varg->w=(float)sqrt(varg->f[0]); else varg->w=0;
			break;
	
	case GL_FUNCTION_SIN:
			varg->w=(float)sin(varg->f[0]);
			break;
			
	case GL_FUNCTION_COS:
			varg->w=(float)cos(varg->f[0]);
			break;
	
	case GL_FUNCTION_TAN:
			varg->w=(float)tan(varg->f[0]);
			break;
	
	case GL_FUNCTION_ACOS:
			if((varg->f[0]>=-1)&&(varg->f[0]<=1))varg->w=(float)acos(varg->f[0]);
			else varg->w=0;
			break;
	
	case GL_FUNCTION_ASIN:
			if((varg->f[0]>=-1)&&(varg->f[0]<=1))varg->w=(float)asin(varg->f[0]);
			else varg->w=0;
			break;
	
	case GL_FUNCTION_ATAN:
			varg->w=(float)atan(varg->f[0]);
			break;
	
	case GL_FUNCTION_EXP:
			varg->w=(float)exp(varg->f[0]);
			break;
	
	case GL_FUNCTION_LN:
			if(varg->f[0]>0)varg->w=(float)log(varg->f[0]);
			else varg->w=0;
			break;
	
	case GL_FUNCTION_LOG10:
			if(varg->f[0]>0)varg->w=(float)log10(varg->f[0]);
			else varg->w=0;
			break;
	
	case GL_FUNCTION_FPART:
			varg->w=(float)varg->f[0]-(float)(int)varg->f[0];
			break;
	
	case GL_FUNCTION_ABS:
			varg->w=(float)fabs(varg->f[0]);
			break;
	
	case GL_FUNCTION_SIGMA:
			/* adds up a number of local floats */
			varg->w=0;
			if(vm)
			{
			if((varg->f[0]>=1)&&(varg->f[0]<=65535))
			{
				i=(unsigned short)varg->f[0];
				offset=(ADDRESS_TYPE)varg->f[1];
				while((i--)&&(result==VM_OK))
				{
					result=getLocalMemory(vm, offset, (unsigned char*)&varg->f[0], SIZE_OF_FLOAT);
					offset+=(ADDRESS_TYPE)SIZE_OF_FLOAT;
					varg->w+=(float)varg->f[0];				
				}	
			} 
			}
			break;
		 	
	case GL_FUNCTION_SIGMA_SQUARES:
			/* adds up a number of squares of local floats */
			varg->w=0;
			if(vm)
			{
			if((varg->f[0]>=1)&&(varg->f[0]<=65535))
			{
				i=(unsigned short)varg->f[0];
				offset=(ADDRESS_TYPE)varg->f[1];
				while((i--)&&(result==VM_OK))
				{
					result=getLocalMemory(vm, offset, (unsigned char*)&varg->f[0], SIZE_OF_FLOAT);
					offset+=(ADDRESS_TYPE)SIZE_OF_FLOAT;
					varg->w+=(float)varg->f[0]*varg->f[0];				
				}	
			}
			}			
			break;

	case GL_FUNCTION_GCD:
			varg->w=(float)gcd((unsigned long)varg->f[1], (unsigned long)varg->f[0]);
			break;
		
	case GL_FUNCTION_INIT_RANDOM:
			varg->w=(float)initRandomAutomatic((unsigned long)varg->f[1], (unsigned long)varg->f[0]);
			break;

	case GL_FUNCTION_RANDOM:
			varg->w=(float)getRandom((unsigned long)varg->f[0]);
			break;
			
	case GL_FUNCTION_NUM_DIVISORS:
			varg->w=(float)numDivisors((unsigned long)varg->f[0]);
			break;

	case GL_FUNCTION_IS_PRIME:
			varg->w=(float)isPrime((unsigned long)varg->f[0]);
			break;

	case GL_FUNCTION_PRIME_PRODUCT:
			varg->w=(float)getPrimeProduct((unsigned long)varg->f[0]);
			break;

	case GL_FUNCTION_CONVERT_BCD_TO_DECIMAL:
			varg->w=(float)convertBCDToDecimal((unsigned int)varg->f[0]);
			break;
			
	case GL_FUNCTION_CONVERT_DECIMAL_TO_BCD:
			varg->w=(float)convertDecimalToBCD((unsigned int)varg->f[0]);
			break;

	/* Process Control Functions */
	
	case GL_FUNCTION_START_PROCESS:
			if(vm)
			{
			if((varg->f[0]>=0)&&(varg->f[0]<=(float)VM_ID_MASK))
			{
				i=((unsigned short)varg->f[0]) & VM_ID_MASK;
				vm->IR=(unsigned char)(VMENV_START+i);
				changeVMState(vm, VM_LIMIT_MODE);
			}
			}
			break; 
		
	case GL_FUNCTION_STOP_PROCESS:
			if(vm)
			{
			if((varg->f[0]>=0)&&(varg->f[0]<=(float)VM_ID_MASK))
			{
				i=((unsigned short)varg->f[0]) & VM_ID_MASK;
				vm->IR=(unsigned char)(VMENV_STOP+i);
				changeVMState(vm, VM_LIMIT_MODE);
			}
			}			
			break;

	case GL_FUNCTION_PAUSE_PROCESS:
			if(vm)
			{
				i=(unsigned short)varg->f[0];
				if(i & VM_ID_HOLD)
				{
					vm->temp=VM_ID_HOLD;
				}
				else
				{
					vm->temp=0;
				}
				i&=VM_ID_MASK;
				vm->IR=(unsigned char)(VMENV_PAUSE+i);
				changeVMState(vm, VM_LIMIT_MODE);
				/* to prevent the vm->temp being cleared again */
				vm=0;
			}		
			break;

	case GL_FUNCTION_SET_TIME_W:
			/* do nothing, since the time would have already been copied */
			break;
			
	case GL_FUNCTION_SET_SHOW_TIME:
			if(vm)
			{
				vm->time.show=convertWToB(varg->f[0]);		
			}
			break;
			
	case GL_FUNCTION_GET_LOCAL_TIME:
			if(vm)
			{
				
				#if(IS_PC_HOST)
				
					getTimeVM(vm, (TIME_T*)&vm->time);
				
				#else
				
					igetTime((TIME_T*)&vm->time);
				
				#endif
			
			
			}	
			break;
	
	case GL_FUNCTION_SET_LOCAL_TIME:
			/* If an agument was given, that sets both TIME W and the Local Time */
			if(vm)
			{
				setTime((TIME_T*)&vm->time);
			}	
			break;

	case GL_FUNCTION_SET_YEAR:
	case GL_FUNCTION_SET_MONTH:
	case GL_FUNCTION_SET_DAY:
	case GL_FUNCTION_SET_HOUR:
	case GL_FUNCTION_SET_MINUTES:
	case GL_FUNCTION_SET_SECONDS:
	case GL_FUNCTION_ADD_MONTHS:
	case GL_FUNCTION_ADD_DAYS:
	case GL_FUNCTION_ADD_HOURS:
	case GL_FUNCTION_ADD_MINUTES:
	case GL_FUNCTION_ADD_SECONDS:
				/* Note that vm->temp holds the number of default arguments pushed onto the stack before this call */
				i=(unsigned short)varg->f[1];
				if(i>0x7FFF)i=0;
				k=1;
				goto continueHere;
				
	case GL_FUNCTION_GET_YEAR:
	case GL_FUNCTION_GET_MONTH:
	case GL_FUNCTION_GET_DAY:
	case GL_FUNCTION_GET_HOUR:
	case GL_FUNCTION_GET_MINUTES:
	case GL_FUNCTION_GET_SECONDS:
				k=0;
				goto continueHere;
				
	case GL_FUNCTION_GET_TOTAL_SECONDS:
				k=2;
				goto continueHere;

	case GL_FUNCTION_GET_TOTAL_SECONDS_DIV:
				k=5;
				if(varg->f[1]<=0)varg->f[1]=1;
				goto continueHere;

	case GL_FUNCTION_GET_TOTAL_SECONDS_MOD:
				k=6;
				i=(unsigned short)varg->f[1];
				if(i>0x7FFF)i=0; else if(i<=0)i=1;
				goto continueHere;
				
	case GL_FUNCTION_SET_TOTAL_SECONDS:
				k=3;
				goto continueHere;

	case GL_FUNCTION_GET_WEEKDAY:
				k=4;
				goto continueHere;
	
	case GL_FUNCTION_SET_TOTAL_SECONDS_DIV_MOD:
				k=7;
				goto continueHere;
								
continueHere:
			
			/* Note that vm->temp holds the number of default arguments pushed onto the stack before this call */
			/* If no argument is given, gets the local time, otherise the TIME W */
	
			if((!vm)||(vm->temp==1))
			{
				/* No Argument was given */
				
				#if(IS_PC_HOST)
					
					getTimeVM(vm, (TIME_T*)&systemTime);
				
				#else
				
					igetTime((TIME_T*)&systemTime);
				
				#endif
				
				ptime=&systemTime;
			}
			else
			{
				ptime=&vm->time;
			}
			
			switch(varg->arg.uArg)
			{
					default:
						break;
						
					case GL_FUNCTION_GET_YEAR:
						i=(unsigned short)ptime->year;
						break;
						
					case GL_FUNCTION_GET_MONTH:
						i=(unsigned short)ptime->month;
						break;
						
					case GL_FUNCTION_GET_DAY:
						i=(unsigned short)ptime->day;
						break;
						
					case GL_FUNCTION_GET_HOUR:
						i=(unsigned short)ptime->hours;
						break;
						
					case GL_FUNCTION_GET_MINUTES:
						i=(unsigned short)ptime->mins;	
						break;
						
					case GL_FUNCTION_GET_SECONDS:
						i=(unsigned short)ptime->secs;	
						break;				
						
					case GL_FUNCTION_SET_YEAR:
						ptime->year=(unsigned short)i;
						break;
					
					case GL_FUNCTION_SET_MONTH:
						ptime->month=(unsigned char)i;
						break;
					
					case GL_FUNCTION_SET_DAY:
						ptime->day=(unsigned char)i;
						break;
					
					case GL_FUNCTION_SET_HOUR:
						ptime->hours=(unsigned char)i;
						break;
					
					case GL_FUNCTION_SET_MINUTES:
						ptime->mins=(unsigned char)i;
						break;
					
					case GL_FUNCTION_SET_SECONDS:
						ptime->secs=(unsigned char)i;
						break;

					case GL_FUNCTION_ADD_MONTHS:
						addMonths(ptime, i);
						break;
					
					case GL_FUNCTION_ADD_DAYS:
						addDays(ptime, i);
						break;
					
					case GL_FUNCTION_ADD_HOURS:
						addHours(ptime, i);
						break;
					
					case GL_FUNCTION_ADD_MINUTES:
						addMinutes(ptime, i);
						break;
					
					case GL_FUNCTION_ADD_SECONDS:
						addSeconds(ptime, i);
						break;
			}
			
			switch(k)
			{
				default:
					break;
					
				case 0:
					varg->w=(float)i;
					break;
		
				case 1:
					checkTime(ptime);
					if((!vm)||(vm->temp==1))
					{
						setTime((TIME_T*)ptime);
					}
					else
					{
			
					}
					break;
				
				case 2:
					varg->w=(float)(long)((long)getTotalSeconds(ptime)-(long)DEFAULT_DURATION_SECS);
					break;
				
				case 3:
					convertEpoch2TimeInternal((unsigned long)((long)varg->f[1]+(long)DEFAULT_DURATION_SECS), ptime);
					break;	
				
				case 4:
					varg->w=(float)getWeekDay(ptime->day, ptime->month, ptime->year);
					break;

				case 5:
					varg->w=((float)(long)(((long)getTotalSeconds(ptime)-(long)DEFAULT_DURATION_SECS)))/varg->f[1];
					break;

				case 6:
					varg->w=(float)(long)(((long)getTotalSeconds(ptime)-(long)DEFAULT_DURATION_SECS)%i);
					break;
				
				case 7:
					convertEpoch2TimeInternal((unsigned long)((long)(((long)varg->f[3]*(long)varg->f[2])+(long)varg->f[1])+(long)DEFAULT_DURATION_SECS), ptime);
					break;
			}
			break;

	case GL_FUNCTION_IS_LEAP_YEAR:
				/* Note that vm->temp holds the number of default arguments pushed onto the stack before this call */
				if((!vm)||(vm->temp==1))
				{
					#if(IS_PC_HOST)
					
					getTimeVM(vm, (TIME_T*)&systemTime);
				
					#else
				
					igetTime((TIME_T*)&systemTime);
				
					#endif
					varg->w=(float)isLeapYear((unsigned short)systemTime.year);
				}
				else
				{
					varg->w=(float)isLeapYear((unsigned short)varg->f[0]);
				}	
				break;
		
	case GL_FUNCTION_GET_DAYS_IN_MONTH_YEAR:
				if((!vm)||(vm->temp>=2))
				{
					#if(IS_PC_HOST)
						
						getTimeVM(vm, (TIME_T*)&systemTime);
					
					#else
					
						igetTime((TIME_T*)&systemTime);
					
					#endif
					varg->w=(float)daysInMonth((unsigned short)systemTime.month, (unsigned short)systemTime.year);				
				}
				else
				{
					varg->w=(float)daysInMonth((unsigned short)varg->f[1], (unsigned short)varg->f[0]);				
				}
				break;

		case GL_FUNCTION_GET_PIPE:
				if(vm)
				{
					varg->w=(float)vm->pipeMode;
				}
				break;	

	#if(IS_PC_HOST)
	
	#else
	
	/* ADC Functions */
	
	case GL_FUNCTION_OPEN_ADC:
			varg->w=(float)openADCObject((int)varg->f[0]);
			break;
			
	case GL_FUNCTION_CLOSE_ADC:
			varg->w=(float)closeADCObject((int)varg->f[0]);
			break;
			
	case GL_FUNCTION_READ_ADC:
			varg->w=(float)readADC((unsigned int)varg->f[0]);
			break;
	
	case GL_FUNCTION_GET_ADC_REFERENCE:
			varg->w=(float)getADCReference(ADC_DENOMINATOR, BAND_GAP_VOLTAGE);
			break;

	case GL_FUNCTION_READ_ADCP:						
			varg->w=(float)getADCP();
			break;
			
	case GL_FUNCTION_SET_ADC_SUPPLY_VOLTAGE:
			varg->w=(float)setADCSupplyVoltage((float)varg->f[0]);
			break;
			
	case GL_FUNCTION_GET_ADC_SUPPLY_VOLTAGE:
			varg->w=(float)getADCSupplyVoltage();
			break;
	
	case GL_FUNCTION_GET_ADC_REFERENCE_OPTIONS:
			varg->w=(float)getADCReference(ADC_DENOMINATOR, varg->f[0]);
			break;
 
	case GL_FUNCTION_SELF_CALIBRATE_ADC:
			(void)calibrateADC();
			break;
	
	case GL_FUNCTION_SET_VBG_IO:
			(void)setVGBIO((unsigned char)varg->f[0]);
			break;
	
	case GL_FUNCTION_READ_COMPARATOR:
			varg->w=(float)CMSTATbits.COUT1;
			break;

	case  GL_FUNCTION_READ_V:
			varg->w=(float)readV((unsigned short)varg->f[0]);
			break;
							
	/* I2C Functions */
	
	case GL_FUNCTION_INIT_I2C:
			if(vm)
			{
				/* open the port */
				hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				hd->i2cPort.busRate=(unsigned short)varg->f[0];	
			}
			break;
			
	case GL_FUNCTION_CLOSE_I2C:
			if(vm)
			{
				hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				clearI2CDescriptor(&hd->i2cPort);
				autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, hd, OPEN_I2C_PORT);
			}	
			break;
			
	case GL_FUNCTION_PUT_I2C_ONE_BYTE:
			varg->w=(float)putI2COneByte(vm, (unsigned char)(unsigned short)varg->f[1], (unsigned char)(unsigned short)varg->f[0]);
			break;

	case GL_FUNCTION_PUT_I2C_TWO_BYTE:
			varg->w=(float)putI2CTwoBytes(vm, (unsigned char)(unsigned short)varg->f[2], (unsigned char)(unsigned short)varg->f[1], (unsigned char)varg->f[0]);
			break;

	case GL_FUNCTION_GET_I2C_TWO_BYTE:
			varg->w=(float)getI2CTwoBytes(vm, (unsigned char)(unsigned short)varg->f[0]);
			break;
	
	case GL_FUNCTION_GET_I2C_BYTE_ARRAY:
			if((varg->f[0]>=0)&&(varg->f[0]<I2C_BUFFER_SIZE))
			{
			
			}
			else
			{
				varg->f[0]=I2C_BUFFER_SIZE;
			}
			varg->w=(float)getI2C(vm, (unsigned char)(unsigned short)varg->f[1], &vm->hardwareState.i2cPort.I2CBuffer[0], (unsigned short)varg->f[0]);
			break; 	

	case GL_FUNCTION_PUT_I2C_BYTE_ARRAY:
			if((varg->f[0]>=0)&&(varg->f[0]<I2C_BUFFER_SIZE))
			{
			
			}
			else
			{
				varg->f[0]=I2C_BUFFER_SIZE;
			}
			varg->w=(float)getI2C(vm, (unsigned char)(unsigned short)varg->f[1], &vm->hardwareState.i2cPort.I2CBuffer[0], (unsigned short)varg->f[0]);
			break; 	
			
			
		/* SPI related functions */
		
	case GL_FUNCTION_INIT_SPI:
			if(vm)
			{
				/* open the port */
				vm->SPI.mode=(unsigned char)(unsigned int)varg->f[4];
				k=(unsigned char)(unsigned int)varg->f[3];
				if((k>=0)&&(k<=5))
				{
				
				}
				else
				{
					k=0;
				}
				vm->SPI.ckdidocspin=k;
				
				k=(unsigned char)(unsigned int)varg->f[2];
				if((k>=0)&&(k<=5))
				{
				
				}
				else
				{
					k=0;
				}
				vm->SPI.ckdidocspin=vm->SPI.ckdidocspin<<4;
				vm->SPI.ckdidocspin|=k;
				
				k=(unsigned char)(unsigned int)varg->f[1];
				if((k>=0)&&(k<=5))
				{
				
				}
				else
				{
					k=0;
				}
				vm->SPI.ckdidocspin=vm->SPI.ckdidocspin<<4;
				vm->SPI.ckdidocspin|=k;
				
				k=(unsigned char)(unsigned int)varg->f[0];
				if((k>=0)&&(k<=5))
				{
				
				}
				else
				{
					k=0;
				}
				vm->SPI.ckdidocspin=vm->SPI.ckdidocspin<<4;
				vm->SPI.ckdidocspin|=k;
			}
			break;
	
	case GL_FUNCTION_CLOSE_SPI:
			if(vm)
			{
				vm->SPI.mode=SPI_NO_MODE;
				initSPIState(&vm->SPI);
			}
			break;
	
	case GL_FUNCTION_WRITE_SPI:
			if(vm)
			{
				/* set script parameters */
				k=(unsigned char)(unsigned int)varg->f[0];
				initSPIState(&vm->SPI);
				varg->w=(float)WriteSPIState(&vm->SPI, (unsigned char)k);
				/* revert to normal SPI config for the memory card access */
				closeSPIState(&vm->SPI);
			}
			break;
	
	case GL_FUNCTION_WRITE_SPI_WITHOUT_CS:
			if(vm)
			{
				/* set script parameters */
				k=(unsigned char)(unsigned int)varg->f[0];
				initSPIState(&vm->SPI);
				varg->w=(float)WriteSPIWithoutSS((unsigned char)k);
				/* revert to normal SPI config for the memory card access */
				closeSPIState(&vm->SPI);
			}
			break;
			
	case GL_FUNCTION_SET_SPI_CS:
			if(vm)
			{
				k=(unsigned char)(unsigned int)varg->f[0];
				if(k)k=1; else k=0;
				setIOObject((unsigned char)vm->SPI.ckdidocspin & 0x0F, k);
			}
			break;
			
		/* One Wire Related Functions */
	
		/* One Wire Globals */
		case GL_FUNCTION_SEND_ONE_WIRE_COMMAND_USING_ROM_CODE_BUFFER:
		case GL_FUNCTION_RESET_ONE_WIRE:
		case GL_FUNCTION_SEND_ONE_WIRE_COMMAND:
			if(vm)
			{
				/* configure the port automatically */
				hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, hd, OPEN_ONE_WIRE_PORT);
			}
			
			switch(varg->arg.uArg)
			{
				default:
					break;
				
				case GL_FUNCTION_RESET_ONE_WIRE:
					varg->w=(float)putResetPulse(&vm->hardwareState);
					break;
					
				case GL_FUNCTION_SEND_ONE_WIRE_COMMAND:
					varg->w=(float)sendOneWireCommand(&vm->hardwareState, (unsigned char)(unsigned int)varg->f[3], (unsigned int)varg->f[2], (unsigned char*)&vm->hardwareState.oneWirePort.oneWireBuffer[0], (unsigned char)(unsigned int)varg->f[1], (unsigned int)varg->f[0]);
					break;
				
				case GL_FUNCTION_SEND_ONE_WIRE_COMMAND_USING_ROM_CODE_BUFFER:
					varg->w=(float)sendOneWireCommand(&vm->hardwareState, (unsigned char)(unsigned int)varg->f[3], (unsigned int)varg->f[2], (unsigned char*)&vm->hardwareState.oneWirePort.romCode[0], (unsigned char)(unsigned int)varg->f[1], (unsigned int)varg->f[0]);
					break;
			}
			break;
			
		case GL_FUNCTION_COPY_ONE_WIRE_BUFFER_TO_ROM_CODE_BUFFER:
				/* k is the offset */
				k=convertWToB(varg->f[0]);
				copyOneWireRomCode(&vm->hardwareState, k, 0);
				break;
				
		case GL_FUNCTION_COPY_ROM_CODE_BUFFER_TO_ONE_WIRE_BUFFER:
				/* k is the offset */
				k=convertWToB(varg->f[0]);
				copyOneWireRomCode(&vm->hardwareState, k, 1);
				break;
					
		case GL_FUNCTION_INIT_ONE_WIRE:
			if(vm)
			{
				/* open the one wire port */
				k=(unsigned char)(unsigned int)varg->f[1];
				k=k & (~ONE_WIRE_PIN_MASK);
				k+=(unsigned char)(unsigned int)varg->f[0];
				vm->hardwareState.oneWirePort.mode=k;
				getOneWireState(&vm->hardwareState);					
			}
			break;	
		
		case GL_FUNCTION_CLOSE_ONE_WIRE:
			if(vm)
			{
				 vm->hardwareState.oneWirePort.mode=NO_ONE_WIRE_MODE;
				 getOneWireState(&vm->hardwareState);
				 hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				 autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, hd, OPEN_ONE_WIRE_PORT);
			}
			break;
			
		case GL_FUNCTION_ONE_WIRE_CRC:
			if(vm)
			{
				varg->w=(float)(CRCOneWire((unsigned char*)&vm->hardwareState.oneWirePort.oneWireBuffer[0], (unsigned int)varg->f[0]));
			}
			break;
			
	/* UART related Functions */			

	case GL_FUNCTION_INIT_UART:
			if(vm)
			{
				/* open the port */
				hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				hd->serialPort.mode=(unsigned char)(unsigned int)varg->f[3];
				hd->serialPort.baudRate=(unsigned short)(varg->f[2]/BAUD_RATE_FACTOR);
				hd->serialPort.txrxpin=(unsigned char)((((unsigned char)varg->f[1])<<4)+(unsigned char)varg->f[0]);	
			}
			break;
	
	case GL_FUNCTION_CLOSE_UART:
			if(vm)
			{
				hd=(HARDWARE_DESCRIPTOR*)&vm->hardwareState;
				clearUARTDescriptor(&hd->serialPort);
				autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, hd, OPEN_SERIAL_PORT);
			}	
			break;
			
	case GL_FUNCTION_GET_UART:
			if(vm)
			{
				varg->w=(float)UARTRxPipeGet(vm);
			}
			break;
	
	case GL_FUNCTION_PUT_UART:
			if(vm)
			{
				outputCVM(vm, (unsigned char)convertWToB(varg->f[0]));
			}
			break;		
	
	case GL_FUNCTION_NOT_EMPTY_UART:
			if(vm)
			{
				varg->w=(float)notEmptyCharVM(vm);			
			}
			break;
	
	case GL_FUNCTION_GET_SIZE_UART:
			if(vm)
			{
				ptr=getSizeGPSVM(vm, &i);
				varg->w=(float)i;
			}
			break;
	
	case GL_FUNCTION_LAST_RX_UART:
			if(vm)
			{
				varg->w=(float)vm->GPS.GPSLastRx;
			}
			break;
			
	case GL_FUNCTION_CLEAR_RX_UART:
			if(vm)
			{
				#if(IS_PC_HOST)
				#else
					clearUARTRx(vm);
				#endif
			}
			break;
	
	case GL_FUNCTION_GET_UART_ERROR:
			if(vm)
			{
				#if(IS_PC_HOST)
				#else
					varg->w=(float)getUARTError(vm);				
				#endif
			}
			break;
			
	case GL_FUNCTION_RECEIVED_NMEA_UART:
		 if(vm)
		 {
			 #if(IS_PC_HOST)
			 #else
				varg->w=(float)receivedNMEAUART(vm);
			 #endif
		 }
		 break;
	
	case GL_FUNCTION_DO_NMEA_MATCH:
		if(vm)
		{
			#if(IS_PC_HOST)
			#else
				ptr=getSizeGPSVM(vm, &i);
				varg->w=(float)matchNMEAString((unsigned char*)&vm->GPS.GPSTempString[0], (const rom unsigned char*)0, (unsigned char*)&vm->GPS.GPSPipe[0], (unsigned char*)&vm->GPS.GPSOutput[0], &k, &l, (unsigned char*)&vm->GPS.GPSOutPtr, i, MAX_GPS_OUTPUT);
			#endif
		}
		break;
	
	case GL_FUNCTION_NEW_RX_UART:
		if(vm)
		{
			#if(IS_PC_HOST)
			#else
			  varg->w=(float)newRxUART(vm);
			#endif
		}
		break;
								
	/* General IO Functions */
	
	case GL_FUNCTION_OPEN_IO:
			varg->w=(float)openIOObject((unsigned char)varg->f[1], (unsigned char)varg->f[0]);
			break;
			
	case GL_FUNCTION_SET_IO:
			varg->w=(float)setIOObject((unsigned char)varg->f[1], (unsigned char)varg->f[0]);
			break;
			
	case GL_FUNCTION_GET_IO:
			varg->w=(float)getIOObject((unsigned char)varg->f[0]);
			break;
	
	case GL_FUNCTION_CLOSE_IO:
			varg->w=(float)getIOObject((unsigned char)varg->f[0]);
			break;
	
	case GL_FUNCTION_TOGGLE_IO:
			k=(unsigned char)varg->f[0];
			if(getIOObject(k))l=0; else l=1;
			varg->w=(float)setIOObject(k, l);
			break;
			
	case GL_FUNCTION_IF_IO_HIGH:
			if(getIOObject((unsigned char)varg->f[0])==1)varg->w=1; else varg->w=0;
			break;
			
	case GL_FUNCTION_IF_IO_LOW:
			if(getIOObject((unsigned char)varg->f[0])==0)varg->w=1; else varg->w=0;
			break;
			
	case GL_FUNCTION_SET_LED:
			varg->w=(float)setLED((unsigned int)varg->f[0]);
			break;
			
	case GL_FUNCTION_FLASH_LED_TIMES:
			internalSetLED((unsigned char)convertWToB(varg->f[0]), STANDARD_LED_ON_MS, STANDARD_LED_OFF_MS); 
			break;

	case GL_FUNCTION_INTERNAL_SET_LED:
			internalSetLED((unsigned char)convertWToB(varg->f[1]), (unsigned int)varg->f[0], STANDARD_LED_OFF_MS); 
			break;
	
	/* Capture globals */
		case GL_FUNCTION_OPEN_CAPTURE_LF:
		case GL_FUNCTION_OPEN_CAPTURE_MF:
		case GL_FUNCTION_OPEN_CAPTURE_HF:
		case GL_FUNCTION_READ_CAPTURE:
		case GL_FUNCTION_CLOSE_CAPTURE:
		case GL_FUNCTION_OPEN_CAPTURE_AUTO_FREQUENCY:
		case GL_FUNCTION_OPEN_CAPTURE_COUNTER_RISING:
		case GL_FUNCTION_OPEN_CAPTURE_COUNTER_FALLING:
		case GL_FUNCTION_OPEN_CAPTURE_OPTIONS:			
				
				if(varg->arg.uArg==GL_FUNCTION_OPEN_CAPTURE_OPTIONS)
				{
					k=(unsigned char)varg->f[1];
				}
				else
				{
					k=(unsigned char)varg->f[0];
				}
				
				if((k>=0)&&(k<CAPTURE_LOGICAL_CHANNELS))
				{
					/* translate from logical to physical capture channel */
					k=captureChannels[k];
				}
				else
				{
					/* choose an invalid channel */
					k=1;
				}
				switch(varg->arg.uArg)
				{
					case GL_FUNCTION_OPEN_CAPTURE_LF:
							varg->w=(float)openCaptureObject(k, LOW_FREQUENCY_MODE);
							break;
							
					case GL_FUNCTION_OPEN_CAPTURE_MF:
							varg->w=(float)openCaptureObject(k, MEDIUM_FREQUENCY_MODE);
							break;
						
					case GL_FUNCTION_OPEN_CAPTURE_HF:
							varg->w=(float)openCaptureObject(k, HIGH_FREQUENCY_MODE);
							break;
					 	 
					case GL_FUNCTION_READ_CAPTURE:
							varg->w=(float)readCaptureObjectChannel(k, &k);
							capture_FullTask();  
							break;
							
					case GL_FUNCTION_CLOSE_CAPTURE:
							varg->w=(float)closeCaptureObject((unsigned char)k);
							break;
						
					case GL_FUNCTION_OPEN_CAPTURE_AUTO_FREQUENCY:
							varg->w=(float)openCaptureObject((unsigned char)k, ALLOW_FREQUENCY_SCALING | LOW_FREQUENCY_MODE);
							break;
				
					case GL_FUNCTION_OPEN_CAPTURE_COUNTER_RISING:
							varg->w=(float)openCaptureObject((unsigned char)k, COUNT_RISING_EDGES | NO_FREQUENCY_MODE);
							break;
					
					case GL_FUNCTION_OPEN_CAPTURE_COUNTER_FALLING:
							varg->w=(float)openCaptureObject((unsigned char)k, NO_FREQUENCY_MODE);
							break;
			
					case GL_FUNCTION_OPEN_CAPTURE_OPTIONS:
							varg->w=(float)openCaptureObject((unsigned char)k, (int)varg->f[0]);
							break;
				}
				break;
		
	/* System functions */
	
	case GL_FUNCTION_GLOBAL_READ:
			ptr=(unsigned char*)(unsigned int)varg->f[0];
			varg->w=(float)(*ptr);
			break;
	
	case GL_FUNCTION_GLOBAL_WRITE:
			ptr=(unsigned char*)(unsigned int)varg->f[1];
			*ptr=(unsigned char)convertWToB(varg->f[0]);
			varg->w=varg->f[0];
			break;

	case GL_FUNCTION_READ_EE:
			varg->w=(float)DataEERead((unsigned int)(EE_GENERAL_ADDR+(int)varg->f[0]));
			break;
			
	case GL_FUNCTION_WRITE_EE:
			varg->w=(float)DataEEWrite((unsigned char)convertWToB(varg->f[0]), (unsigned int)(EE_GENERAL_ADDR+(int)varg->f[1]));
			break;
			
	case GL_FUNCTION_GLOBAL_READ_F:
			ptr=(unsigned char*)&varg->w;
			dptr=(unsigned char*)(unsigned int)varg->f[0];;
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			break;
						
	case GL_FUNCTION_GLOBAL_WRITE_F:
			ptr=(unsigned char*)(unsigned int)varg->f[1];
			dptr=(unsigned char*)&varg->f[0];
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			*ptr++=*dptr++;
			break;
			
	case GL_FUNCTION_READ_EE_F:
			ptr=(unsigned char*)&varg->w;
			i=(unsigned int)(EE_GENERAL_ADDR+(int)varg->f[0]);
			*ptr++=(unsigned char)DataEERead(i++);
			*ptr++=(unsigned char)DataEERead(i++);
			*ptr++=(unsigned char)DataEERead(i++);
			*ptr++=(unsigned char)DataEERead(i++);
			break;
			
	case GL_FUNCTION_WRITE_EE_F:
			ptr=(unsigned char*)&varg->f[0];
			i=(unsigned int)(EE_GENERAL_ADDR+(int)varg->f[1]);
			DataEEWrite(*ptr++, i++);
			DataEEWrite(*ptr++, i++);
			DataEEWrite(*ptr++, i++);
			DataEEWrite(*ptr++, i++);
			break;
	
	case GL_FUNCTION_SIZE_EE:
			varg->w=(DATA_EE_TOTAL_SIZE-EE_GENERAL_ADDR);
			break;
				
	#endif
	
	default:
		/* default behaviour, in case IS_PC_HOST and some commands are not implemented! */
		varg->w=0;
		break;
	}
		
	if(vm)
	{
		vm->W=varg->w;
		vm->temp=0;
		vm->typeMode=DEFAULT_TYPE_MODE;
	}
	return result;
}

VM_RETURN_TYPE szShortFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->DSLIMIT=sizeof(unsigned short);
	return VM_OK;
}

VM_RETURN_TYPE szLongFunction(VIRTUAL_MACHINE* vm, VM_ARGUMENT* varg)
{
	vm->DSLIMIT=sizeof(unsigned long);
	return VM_OK;
}
