//10k resistors
#define A_DRIVE_1 3
#define B_DRIVE_1 1
#define C_DRIVE_1 A5

//1k resistors
#define A_DRIVE_2 A1
#define B_DRIVE_2 2
#define C_DRIVE_2 0

//sense pins (must be analog)
#define A_SENSE A2
#define B_SENSE A3
#define C_SENSE A4

//test resistor values
#define R_DRIVE_1 10000.0
#define R_DRIVE_2 1000.0

const char drivePins1[3]={A_DRIVE_1,B_DRIVE_1,C_DRIVE_1};
const char drivePins2[3]={A_DRIVE_2,B_DRIVE_2,C_DRIVE_2};
const char sensePins[3]={A_SENSE,B_SENSE,C_SENSE};

//voltage calcs, assume 5V
#define VCC 5000
#define HIGHZ -1
int vA,vB,vC;
#define PN_LOWER 200
#define PN_UPPER 750
#define LED_LOWER 800
#define LED_UPPER 4000
#define RAIL_MARGIN 100

int j[6]={0,0,0,0,0,0};
char jj1,jj2;
char pIndex[6]={0,0,1,1,2,2};
char nIndex[6]={1,2,2,0,0,1};
int iDS=0;  //DS current for testing

// include the library code:
#include <LiquidCrystal.h>
LiquidCrystal LCD(8,9,4,5,6,7); //display
#define BETA_SYMBOL "\xE2"
#define MU_SYMBOL "\xE4"
#define OMEGA_SYMBOL "\xF4"
#include "lcdkeys.h"

char* pinMap;

void setup() {
  Serial.begin(115200);
  pinMode(A_SENSE,INPUT);
  pinMode(B_SENSE,INPUT);
  pinMode(C_SENSE,INPUT);
  allOff();  //all drive Z
  LCD.begin(16, 2);
  LCD.clear();
  LCD.setCursor(0,0);
  LCD.print("SC and Jaycar   ");
  LCD.setCursor(0,1);
  LCD.print("TransistorTester");
  delay(2000);
  LCD.clear();
}

void loop() {
  lcdKeys_t k;
  char pnRes=0;
  char ledRes=0;
  LCD.setCursor(0,0); //home
  //scan
  allOff();  //all drive Z
  getJVPN();  
  //check keys
  k=getLCDkey();
  if(k==LCD_LEFT){  //bipolar
    pnRes=checkPNs();
    LCD.setCursor(0,0);
    if(pnRes==2){   //NPN
      NPNreport();
    }else if(pnRes==3){ //PNP
      PNPreport();
    }else{  //something else
      LCD.print("Not NPN or PNP. ");LCD.setCursor(0,1);LCD.print("Restart scan.   ");
    }    
  }else if(k==LCD_RIGHT){  //mosfet
    pnRes=checkPNs();
    if(pnRes==1){ //needs to find a silicon junction (ie body diode)
      mosReport();  
    }else{
      LCD.print("No body diode.  ");LCD.setCursor(0,1);LCD.print("Restart scan.   ");
    }    
  }else if(k==LCD_UP){  //diode/LED1
    if(jj1!=-1){
      diodeReport(jj1);
    }else{
      LCD.print("Q Tester        ");LCD.setCursor(0,1);LCD.print("Nothing detected");
    }
  }else if(k==LCD_DOWN){ //diode/LED2
    if(jj2!=-1){
      diodeReport(jj2);
    }else{
      if(jj1!=-1){
        LCD.print("Q Tester      UP");LCD.setCursor(0,1);LCD.print("Restart scan.   ");
      }else{
        LCD.print("Q Tester        ");LCD.setCursor(0,1);LCD.print("Nothing detected");
      }
    }
  }else{
    pnRes=checkPNs();
    if(pnRes==0){
      ledRes=checkLEDs();
    }  
    //show results    
    if(pnRes){
      switch(pnRes){
        case 1: LCD.print("Q Tester   UP/RT");LCD.setCursor(0,1);LCD.print("DIODE/MOSFET ");LCD.print(pinMap); break;
        case 2: LCD.print("Q Tester    LEFT");LCD.setCursor(0,1);LCD.print("NPN BIPOLAR  ");LCD.print(pinMap); break;
        case 3: LCD.print("Q Tester    LEFT");LCD.setCursor(0,1);LCD.print("PNP BIPOLAR  ");LCD.print(pinMap); break;
      }
    }else if(ledRes){
      switch(ledRes){
        case 1: LCD.print("Q Tester UP/DOWN");LCD.setCursor(0,1);LCD.print("single LED   ");LCD.print(pinMap); break;
        case 2: LCD.print("Q Tester UP/DOWN");LCD.setCursor(0,1);LCD.print("CA dual LED  ");LCD.print(pinMap); break;
        case 3: LCD.print("Q Tester UP/DOWN");LCD.setCursor(0,1);LCD.print("CC dual LED  ");LCD.print(pinMap); break;
        case 4: LCD.print("Q Tester UP/DOWN");LCD.setCursor(0,1);LCD.print("BP dual LED  ");LCD.print(pinMap); break;
      }
    }else if(checkShorts()){
        LCD.print("Q Tester        ");LCD.setCursor(0,1);LCD.print("Short!       ");LCD.print(pinMap);
    }else{
        LCD.print("Q Tester        ");LCD.setCursor(0,1);LCD.print("Nothing detected");
    }
  }
  delay(300);
}

void diodeReport(char n){ //junction tested in j[]
  int va,vk;
  float vf,cf;
  static char akMap[4]="---";
  akMap[0]='-'; //reset every time
  akMap[1]='-';
  akMap[2]='-';
  akMap[3]=0;
  akMap[pIndex[n]]='A';
  akMap[nIndex[n]]='K';
  allOff();  //all Z
  digitalWrite(drivePins2[pIndex[n]],HIGH);
  digitalWrite(drivePins2[nIndex[n]],LOW);
  pinMode(drivePins2[pIndex[n]],OUTPUT);
  pinMode(drivePins2[nIndex[n]],OUTPUT);
  delay(1);
  //this time we leave everything on to light up LEDs etc
  va=ADC2V(analogRead(sensePins[pIndex[n]]));
  vk=ADC2V(analogRead(sensePins[nIndex[n]]));
  vf=(va-vk); 
  vf=vf/1000;   //V
  if(vf<0){vf=0;}
  cf=((float)vk)/R_DRIVE_2; //mA
  LCD.print("Vf=");
  LCD.print(vf,2); //show in V
  LCD.print("V     ");
  LCD.setCursor(13,0);LCD.print(akMap);
  LCD.setCursor(0,1);
  if((millis()/2000)%2){  //cycle between I/R
    LCD.print("If=");
    LCD.print(cf,2);  //show in mA
    LCD.print("mA    ");
    LCD.setCursor(13,1);LCD.print(pinMap);  
  }else{
    if(cf>0.001){
      LCD.print("R=");
      LCD.print(vf/cf,2);  //show in k#
      LCD.print("k" OMEGA_SYMBOL "     ");
    }else{
      LCD.print("High R       ");
    }
  }
  LCD.setCursor(13,1);LCD.print(pinMap);    
}

void mosReport(void){ //scan for conduction as the three terminals are driven by
  //pnRes=1 => getPNmap '-PN' or something similar in pinMap[] where it could be GSD nMOS or GDS pMOS
  int vp,vn;
  vn=nMosVt();
  vp=pMosVt();
  if(vn>0){
    LCD.print("NMOS   Vt=");
    LCD.print(vn/1000.0,2);
    LCD.print("V  ");
    LCD.setCursor(0,1);
    LCD.print("Ids=");
    LCD.print(iDS);
    LCD.print(MU_SYMBOL "A      ");
    LCD.setCursor(13,1);
    LCD.print(pinMap);
  }else if(vp>0){
    LCD.print("PMOS   Vt=");
    LCD.print(vp/1000.0,2);
    LCD.print("V  ");
    LCD.setCursor(0,1);
    LCD.print("Ids=");
    LCD.print(iDS);
    LCD.print(MU_SYMBOL "A      ");
    LCD.setCursor(13,1);
    LCD.print(pinMap);
  }else{
    LCD.print("MOS not detected");LCD.setCursor(0,1);LCD.print("Check connection");
  }
}

int nMosVt(void){  //find Vt(mV), assume NMOS and PN junction is SD
  int res;
  char gPin=3; //can all be determined, if NMOS is assumed
  char sPin=3;
  char dPin=3;
  int vg,vs,vd;
  char i;
  static char qMap[4]="---";  
  qMap[0]='-'; //reset every time
  qMap[1]='-';
  qMap[2]='-';
  qMap[3]=0;
  for(i=0;i<3;i++){
    if(pinMap[i]=='-'){gPin=i;qMap[i]='G';}
    else if(pinMap[i]=='P'){sPin=i;qMap[i]='S';}
    else if(pinMap[i]=='N'){dPin=i;qMap[i]='D';}
  }
  if((gPin==sPin)||(gPin==3)){return -1;} //error of some sort
  if((sPin==dPin)||(sPin==3)){return -1;}
  if((dPin==gPin)||(dPin==3)){return -1;}
  allOff();
  digitalWrite(drivePins1[gPin],HIGH);  //via 10k
  digitalWrite(drivePins2[dPin],HIGH);  //via 1k
  digitalWrite(drivePins1[sPin],LOW); //via 10k
  pinMode(drivePins1[gPin],OUTPUT);
  pinMode(drivePins2[dPin],OUTPUT);
  pinMode(drivePins1[sPin],OUTPUT);
  delay(1);
  vg=ADC2V(analogRead(sensePins[gPin]));
  vd=ADC2V(analogRead(sensePins[dPin]));
  vs=ADC2V(analogRead(sensePins[sPin]));
  //showStat();
  allLow();
  delay(1);  
  allOff();
  if(vg<(VCC-RAIL_MARGIN)){return -1;}  //no current should be drawn through gate, no matter what
  if(vd>(VCC-RAIL_MARGIN)){return -1;}  //if no current through drain, then not conducting NMOS
  if(vs<RAIL_MARGIN){return -1;}  //if no current through source, then not conducting NMOS
  //vt is g-s voltage
  res=vg-vs;
  if(res<0){return -1;}
  pinMap=qMap;  //OK to update
  iDS=(VCC-vs)/(1000000UL/R_DRIVE_1);
  return res;  
}

int pMosVt(void){  //find Vt(mV), assume PMOS and PN junction is DS
  int res;
  char gPin=3; //can all be determined, if PMOS is assumed
  char sPin=3;
  char dPin=3;
  int vg,vs,vd;
  char i;
  static char qMap[4]="---";  
  qMap[0]='-'; //reset every time
  qMap[1]='-';
  qMap[2]='-';
  qMap[3]=0;
  for(i=0;i<3;i++){
    if(pinMap[i]=='-'){gPin=i;qMap[i]='G';}
    else if(pinMap[i]=='N'){sPin=i;qMap[i]='S';}
    else if(pinMap[i]=='P'){dPin=i;qMap[i]='D';}
  }
  if((gPin==sPin)||(gPin==3)){return -1;} //error of some sort
  if((sPin==dPin)||(sPin==3)){return -1;}
  if((dPin==gPin)||(dPin==3)){return -1;}
  allOff();
  digitalWrite(drivePins1[gPin],LOW);  //via 10k
  digitalWrite(drivePins2[dPin],LOW);  //via 1k
  digitalWrite(drivePins1[sPin],HIGH); //via 10k
  pinMode(drivePins1[gPin],OUTPUT);
  pinMode(drivePins2[dPin],OUTPUT);
  pinMode(drivePins1[sPin],OUTPUT);
  delay(1);
  vg=ADC2V(analogRead(sensePins[gPin]));
  vd=ADC2V(analogRead(sensePins[dPin]));
  vs=ADC2V(analogRead(sensePins[sPin]));
  showStat();
  allLow();
  delay(1);  
  allOff();
  if(vg>RAIL_MARGIN){return -1;}  //no current should be drawn through gate, no matter what
  if(vd<RAIL_MARGIN){return -1;}  //if no current through drain, then not conducting PMOS
  if(vs>(VCC-RAIL_MARGIN)){return -1;}  //if no current through source, then not conducting PMOS
  //vt is g-s voltage
  res=vs-vg;
  if(res<0){return -1;}
  pinMap=qMap;  //OK to update
  iDS=vs/(1000000UL/R_DRIVE_1);
  return res;  
}

void NPNreport(void){
  static char qMap[4]="---";
  char bPin=3;  //flag value to detect error
  char cePin1=0;  //need to find which 
  char cePin2=0;  //is which
  int b1,b2;
  qMap[0]='-'; //reset every time
  qMap[1]='-';
  qMap[2]='-';
  qMap[3]=0;
  if(pinMap[0]=='P'){bPin=0;cePin1=1;cePin2=2;} //BEC or BCE
  else if(pinMap[1]=='P'){bPin=1;cePin1=0;cePin2=2;}  //EBC or CBE
  else if(pinMap[2]=='P'){bPin=2;cePin1=0;cePin2=1;}  //ECB or CEB
  else{LCD.print("Scanning error  ");LCD.setCursor(0,1);LCD.print("Check connection");return;}
  b1=getBeta(1,bPin,cePin1,cePin2); //BEC, EBC, ECB (ie cePin1=E)
  b2=getBeta(1,bPin,cePin2,cePin1); //BCE, CBE, CEB (ie cePin2=E)
  qMap[bPin]='B';
  if(b1>b2){
    qMap[cePin1]='E';
    qMap[cePin2]='C';
    LCD.setCursor(13,0);
    LCD.print(b1);
    LCD.print("   ");
    LCD.setCursor(5,1);
    LCD.print(b2);
    LCD.print("   ");
  }else{
    qMap[cePin1]='C';
    qMap[cePin2]='E';    
    LCD.setCursor(13,0);
    LCD.print(b2);
    LCD.print("   ");
    LCD.setCursor(5,1);
    LCD.print(b1);
    LCD.print("   ");
  }
  LCD.setCursor(0,0);
  LCD.print("NPN DETECT:" BETA_SYMBOL "=");
  LCD.setCursor(0,1);
  LCD.print("(" BETA_SYMBOL "r)=");
  LCD.setCursor(8,1);
  LCD.print("     ");
  LCD.print(qMap);
}

int getBeta(char isNPN, char bPin, char ePin, char cPin){ //
  float vb,ve;
  float ib,ie;
  float beta=0; //use floats but return int
  allOff();
  if(isNPN){  //c,b high, e low
    digitalWrite(drivePins1[bPin],HIGH);  //via 10k
    digitalWrite(sensePins[cPin],HIGH);  //direct drive via sense pin
    digitalWrite(drivePins2[ePin],LOW); //via 1k
    pinMode(drivePins1[bPin],OUTPUT);
    pinMode(sensePins[cPin],OUTPUT);
    pinMode(drivePins2[ePin],OUTPUT);
  }else{  //reverse of above
    digitalWrite(drivePins1[bPin],LOW);  //via 10k
    digitalWrite(sensePins[cPin],LOW);  //direct drive via sense pin
    digitalWrite(drivePins2[ePin],HIGH); //via 1k
    pinMode(drivePins1[bPin],OUTPUT);
    pinMode(sensePins[cPin],OUTPUT);
    pinMode(drivePins2[ePin],OUTPUT);
  }
  delay(1);  
  vb=ADC2V(analogRead(sensePins[bPin]));
  ve=ADC2V(analogRead(sensePins[ePin]));
  //showStat();
  allLow();
  delay(1);  
  allOff();
  if(isNPN){ //base referred to VCC, emitter to GND
    if(((vb-ve)>PN_LOWER)&&((vb-ve)<PN_UPPER)){
      ib=(VCC-vb)/R_DRIVE_1;
      ie=ve/R_DRIVE_2;
      if(ib>1e-6){
        beta=ie/ib;    
      }      
    }
  }else{  //emitter referred to GND, base to VCC
    if(((ve-vb)>PN_LOWER)&&((ve-vb)<PN_UPPER)){
      ib=vb/R_DRIVE_1;
      ie=(VCC-ve)/R_DRIVE_2;
      if(ib>1e-6){
        beta=ie/ib;    
      }      
    }
  }
  if(beta>999){beta=999;} //bound for display
  if(beta<0){beta=0;}
  return beta;
}

void showStat(void){
  int v0,v1,v2;
  v0=ADC2V(analogRead(sensePins[0]));
  v1=ADC2V(analogRead(sensePins[1]));
  v2=ADC2V(analogRead(sensePins[2]));
  Serial.println("STAT:");
  Serial.print("1:");
  Serial.print(v0);
  Serial.print("mV\r\n2:");
  Serial.print(v1);
  Serial.print("mV\r\n3:");
  Serial.print(v2);
  Serial.print("mV\r\n");
}

void PNPreport(void){
  static char qMap[4]="---";
  char bPin=3;  //flag value to detect error
  char cePin1=0;  //need to find which 
  char cePin2=0;  //is which
  int b1,b2;
  qMap[0]='-'; //reset every time
  qMap[1]='-';
  qMap[2]='-';
  qMap[3]=0;
  if(pinMap[0]=='N'){bPin=0;cePin1=1;cePin2=2;} //BEC or BCE
  else if(pinMap[1]=='N'){bPin=1;cePin1=0;cePin2=2;}  //EBC or CBE
  else if(pinMap[2]=='N'){bPin=2;cePin1=0;cePin2=1;}  //ECB or CEB
  else{LCD.print("Scanning error  ");LCD.setCursor(0,1);LCD.print("Check connection");return;}
  b1=getBeta(0,bPin,cePin1,cePin2); //BEC, EBC, ECB (ie cePin1=E)
  b2=getBeta(0,bPin,cePin2,cePin1); //BCE, CBE, CEB (ie cePin2=E)
  qMap[bPin]='B';
  if(b1>b2){
    qMap[cePin1]='E';
    qMap[cePin2]='C';
    LCD.setCursor(13,0);
    LCD.print(b1);
    LCD.print("   ");
    LCD.setCursor(5,1);
    LCD.print(b2);
    LCD.print("   ");
  }else{
    qMap[cePin1]='C';
    qMap[cePin2]='E';    
    LCD.setCursor(13,0);
    LCD.print(b2);
    LCD.print("   ");
    LCD.setCursor(5,1);
    LCD.print(b1);
    LCD.print("   ");
  }
  LCD.setCursor(0,0);
  LCD.print("PNP DETECT:" BETA_SYMBOL "=");
  LCD.setCursor(0,1);
  LCD.print("(" BETA_SYMBOL "r)=");
  LCD.setCursor(8,1);
  LCD.print("     ");
  LCD.print(qMap);  
}

char* getAKmap(char n){   //n is same as in checkLEDs
  static char res[4]="---";
  char i,k;
  res[0]='-'; //reset every time
  res[1]='-';
  res[2]='-';
  res[3]=0;
  k=1;
  for(i=0;i<6;i++){
    if(k&n){
      res[pIndex[i]]='A';
      res[nIndex[i]]='K';
    }
    k=k<<1;
  }
  return res;
}

char* getPNmap(char n){   //n is same as in checkPNs
  static char res[4]="---";
  char i,k;
  res[0]='-'; //reset every time
  res[1]='-';
  res[2]='-';
  res[3]=0;
  k=1;
  for(i=0;i<6;i++){
    if(k&n){
      res[pIndex[i]]='P';
      res[nIndex[i]]='N';
    }
    k=k<<1;
  }
  return res;
}

void getJVPN(){ //scan pairs of pins
  char n;
  jj1=-1;
  jj2=-1;
  for(n=0;n<6;n++){
    j[n]=getVPN(pIndex[n],nIndex[n]);
    if(j[n]<(VCC-RAIL_MARGIN)){
      if(jj1==-1){jj1=n;}
      else if(jj2==-1){jj2=n;}
    }
  }
}

char checkShorts(void){ //anything this low is probably a short
  char i,r;
  static char res[4]="---";  
  r=0;
  res[0]='-'; //reset every time
  res[1]='-';
  res[2]='-';
  res[3]=0;
  for(i=0;i<6;i++){
    if(j[i]<PN_LOWER){
      r=1;
      res[pIndex[i]]='S'; //flag those involved
      res[nIndex[i]]='S';
      pinMap=res;   //set pointer to use this
    }
  }
  if(r){Serial.println("Short detected");}
  return r;
}

char checkLEDs(void){  
  char i,n,k;  
  char res=0;
  n=0;
  k=1;
  for(i=0;i<6;i++){
    if((j[i]>LED_LOWER)&&(j[i]<LED_UPPER)){
      n=n|k;
    }
    k=k<<1;
  }
  Serial.print((int)n);
  switch(n){
    case 0:
     Serial.println(": No LED device detected.");
     res=0;
     break;
    case 1:
    case 2:
    case 4:
    case 8:
    case 16:
    case 32:
     Serial.println(": 1 LED detected");
     pinMap=getAKmap(n);
     Serial.println(pinMap);
     res=1;
     break;
    case 3:
    case 12:
    case 48:
     Serial.println(": 2 LEDs detected, common anode device");
     pinMap=getAKmap(n);
     Serial.println(pinMap);
     res=2;
     break;
    case 6:
    case 24:
    case 33:
     Serial.println(": 2 LEDs detected, common cathode device");
     pinMap=getAKmap(n);
     Serial.println(pinMap);
     res=3;
     break;
    case 9:
    case 18:
    case 36:
     Serial.println(": 1 Bipolar LED detected");
     pinMap=getAKmap(n);
     if((pinMap[0]=='A')||(pinMap[0]=='K')){pinMap[0]='B';}
     if((pinMap[1]=='A')||(pinMap[1]=='K')){pinMap[1]='B';}
     if((pinMap[2]=='A')||(pinMap[2]=='K')){pinMap[2]='B';}
     Serial.println(pinMap);
     res=4;
     break;
    default:
     Serial.println(": Multiple LED junctions detected; cannot identify");
     res=0;
     break;
  }
  return res;
}

char checkPNs(void){  //count and report on PN junctions
  char i,n,k;  
  char res=0;
  n=0;
  k=1;
  for(i=0;i<6;i++){
    if((j[i]>PN_LOWER)&&(j[i]<PN_UPPER)){
      n=n|k;
    }
    k=k<<1;
  }
  Serial.print((int)n);
  switch(n){
    case 0:
     Serial.println(": No silicon device detected.");
     res=0;
     break;
    case 1:
    case 2:
    case 4:
    case 8:
    case 16:
    case 32:
     Serial.println(": 1 PN junction detected");
     pinMap=getPNmap(n);
     Serial.println(pinMap);
     res=1;
     break;
    case 3:
    case 12:
    case 48:
     Serial.println(": 2 PN junctions detected, NPN device");
     pinMap=getPNmap(n);
     Serial.println(pinMap);
     res=2;
     break;
    case 6:
    case 24:
    case 33:
     Serial.println(": 2 PN junctions detected, PNP device");
     pinMap=getPNmap(n);
     Serial.println(pinMap);
     res=3;
     break;
    case 5:
    case 9:
    case 10:
    case 17:
    case 18:
    case 20:
    case 34:
    case 36:
    case 40:
     Serial.println(": 2 PN junctions detected, not NPN or PNP, can't analyse");     
     res=0;
     break;
    default:
     Serial.println(": Multiple PN junctions detected; cannot identify");
     res=0;
     break;
  }
  return res;
}

int ADC2V(int n){
  return (((long)n)*VCC)/1024;
}

int getVPN(int a, int b){
  int va,vb;
  allOff();
  allLow();    
  delay(1);
  allOff();  //all Z
  digitalWrite(drivePins1[a],HIGH);
  digitalWrite(drivePins1[b],LOW);
  pinMode(drivePins1[a],OUTPUT);
  pinMode(drivePins1[b],OUTPUT);
  delay(1);
  va=analogRead(sensePins[a]);
  vb=analogRead(sensePins[b]);
  allOff();
  Serial.print(a);
  Serial.print(b);
  Serial.print("\t");
  Serial.println(ADC2V(va-vb));
  return ADC2V(va-vb);
}

void allOff(void){  //off, but doesn't change variables
  pinMode(A_SENSE,INPUT);
  pinMode(B_SENSE,INPUT);
  pinMode(C_SENSE,INPUT);
  pinMode(A_DRIVE_1,INPUT);
  pinMode(B_DRIVE_1,INPUT);
  pinMode(C_DRIVE_1,INPUT);
  pinMode(A_DRIVE_2,INPUT);
  pinMode(B_DRIVE_2,INPUT);
  pinMode(C_DRIVE_2,INPUT);
}

void allLow(void){  //in case gate capacitance is holding a MOSFET on
  digitalWrite(A_DRIVE_1,LOW);
  digitalWrite(B_DRIVE_1,LOW);
  digitalWrite(C_DRIVE_1,LOW);
  pinMode(A_DRIVE_1,OUTPUT);
  pinMode(B_DRIVE_1,OUTPUT);
  pinMode(C_DRIVE_1,OUTPUT);
}

