; ***************************************************************************
; file: 			SolarTrak.asm
; processor:		PIC16F88
; clock frequency:	4 MHz
; clock type:		internal RC
; Last revision: 	23 September, 2011
; Author:			Herman Nacinovich
; comments:
;
;	This programme provides two operating modes: tracking (auto)
;	mode and manual mode.
;
;	Tracking mode is selected (by the user) by closing a switch
;	(mode_sw = 1).
;
;	In tracking mode a motor automatically tracks the sun according
; 	to the input signals from a pair of photovoltaic cells.
;	Motor stops if limit switch west_sw opens while motor is moving 
;	in forward mode or if limit switch east_sw opens while motor is
;   moving in reverse mode. In either case the motor will not restart
;	unless the direction is changed, either manually, or as a result
;	of a change in light levels. For example, if the motor stops, while
;	in forward mode, as a result of west_sw = 1 and the ambient light
;	level falls below dark threshold (THRESH_DARK) then the motor will
;	restart in reverse mode. Conversely, if the motor stops in reverse
;	mode due to east_sw = 1 and the sun moves to a position west of
; 	the photosensors, then the motor restarts in forward mode.
;
; 	The motor stops if the ambient light level lies between THRESH_DARK  
;	and THRESH_CLOUD. This prevents hunting due to light reflections 
;	off clouds.
;	
;	In tracking mode the motor duty cycle is proportional to the 
;	difference between signal voltages from the photocells. The 
;	direction of the motor depends on which of these signals is 
;	greater. Thus if the west sensor output is greater than the east
;	sensor output then the motor tracks toward the west, and if the 
;	east sensor output is greater than the west sensor output then the
;	motor tracks to the east.
;
; 	By default, motor tracks in either forward and reverse directions,
;	depending upon the position of the sun. This works fine with the
;	prototype. If hunting occurs try setting TRACK_MODE in <constants.inc>
;	to UNIDIRECTIONAL. Otherwise setting TRACK_MODE to BIDIRECTIONAL
;
; 	Manual mode is selected when mode_sw = 0.
; 	Tracking mode is selected when mode_sw = 1.
;
;	In manual mode the motor is controlled by a potentiometer (pot). In 
;	this mode forward or reverse movement of the motor is controlled by 
;   moving the pot respectively above or below the centre position of 
;	the pot. The speed (or, more accurately, the duty cycle) of the 
;	motor is controlled by the position of the wiper of the pot, the
;	duty cycle increasing with distance from the centre position.
;
;	As in tracking mode, the motor will stop, in forward mode, if 
;	west_sw opens or, in reverse mode, if east_sw is open. To restart
;	motor the user must rotate the pot toward the opposite direction.
;
;	On start up, or when mode is switched from tracking to manual mode, 
;	the pot must first be set to centre position before the motor will 
;	run. This prevents the situation in which the motor moves as a
;	result of having being set to some random position.

;
;	Other features of the programme include:
;		- motor current limiting
;		- trimpot adjustable maximum motor duty cycle
;  		- proportional speed control in tracking mode
;
;	Operating parameters are defined in the file <constants.inc>. 
;	These can be modified as necessary to suit the particular
;	application.
;
;	motor stops on timeout while ERR_DARK = 1
; ****************************************************************

	; ****************************************************************
	; See file <constants.inc> for definitions, constants, etc.
	; ****************************************************************

	; ----------------------------------------------------------------
	list	p=pic16f88			; selects chip type
	; ----------------------------------------------------------------
	#include <p16f88.inc>		; includes definitions
	#include <registers.inc>
	#include <constants.inc>
	#include <macros.inc>
	; ----------------------------------------------------------------
	; set configuration options
	; ----------------------------------------------------------------
	__CONFIG    _CONFIG1, _CP_OFF & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _MCLR_OFF & _PWRTE_ON & _WDT_ON & _INTRC_IO &_BODEN_ON & _CCP1_RB0 & _LVP_OFF
	__CONFIG    _CONFIG2, _IESO_ON & _FCMEN_OFF
	; ----------------------------------------------------------------
start
	call	initialise
main
	clrwdt
	call 	timer
	call	get_mode
	call	get_trimpot
	call	get_pot
	call	get_east
	call	get_west
	call	calc_duty
	call	get_amps
	call	chk_thresh
	call	chk_limits	
	call	dark_handler
	call	motor_handler
	call	led_handler
	goto 	main				
	; ---------------------------------------------------------------
	#include <timers.inc>	
; =======================================================================
; function:		dark_handler
; comments:				
;				When it goes dark motor runs for a maximum preset 
;				then turns off when timeout flag (rtmrf = 1)
;				
;				When it goes light run timer is disabled (rtrme = 1)
;				and run timeout flag is cleared (rtmrf = 0)
; ---------------------------------------------------------------
dark_handler
	btfss	FLAG_TRAK	; manual mode?
	bcf		ERR_DARK	; yes; 
	btfsc	ERR_DARK	; dark?
	goto	dark		; yes
	; ---------------------------------------------------------------
	; ERR_DARK = 0
	; ---------------------------------------------------------------
light
	bcf		rtmre		; disable run timer
	bcf		rtmrf		; clear timeout flag
	return				; done
	; ---------------------------------------------------------------
dark
	btfsc	rtmre		; run timer enabled?
	return				; yes
	; ---------------------------------------------------------------
	btfsc	rtmrf		; timeout flag set?
	return				; yes
	; ---------------------------------------------------------------
	preset	rtmr,RUN_TIMEOUT ; preset run timer
	return
; ---------------------------------------------------------------
; function:		led_handler
; description:	turn led on/off
; ---------------------------------------------------------------
led_handler
	banksel	PORTB
	btfss	FLAG_RUN		; run mode?
	goto	$+3				; no
	bsf		led
	return
	; ---------------------------------------------------------------
	btfsc	ltmre			; led timer active?
	return					; yes
	; ---------------------------------------------------------------
	btfsc	led				; led is on?
	goto	led_off			; yes
led_on
	bsf		led
	preset	ltmr,LEDON_TIME
	return
led_off
	bcf		led
	preset	ltmr,LEDOFF_TIME
	return
; ---------------------------------------------------------------
; function:		initialise
; description:	initialise registers on start-up
; comments:		see file <constants.inc> for definitions
; ---------------------------------------------------------------
initialise
	banksel	bank0
	movlw	T2CON_VAL
	movwf	T2CON
	movlw	CCP1CON_VAL
	movwf	CCP1CON
	clrf	CCPR1L
	movlw	PORTB_INIT
	movwf	PORTB
	movlw	ADCON0_VAL
	movwf	ADCON0
	; --------------------------------------------------------------
	banksel	bank1
	movlw	OPTION_VAL
	movwf	OPTION_REG
	movlw	TRISB_VAL
	movwf	TRISB
	movlw	TRISA_VAL
	movwf	TRISA
	movlw	OSCCON_VAL
	movwf	OSCCON
	movlw	PR2_VAL
	movwf	PR2
	movlw	ANSEL_VAL
	movwf	ANSEL
	movlw	ADCON1_VAL
	movwf	ADCON1
	; --------------------------------------------------------------
	banksel	0
	clrf	tmrcon
	clrf	tmrflags
	movlw	THRESH_CLOUD
	movwf	eastup
	movwf	westup	
	clrf	CCPR1L
	clrf	err
	clrf	flags	
	clrf	motorcon
	return
; ------------------------------------------------------------------
get_mode					; checks mode_sw for current mode
	banksel	PORTA
	bcf		FLAG_TRAK		; default
	btfss	mode_sw			; switch closed?
	return					; yes
	bcf		FLAG_INIT		; else...
	bsf		FLAG_TRAK
	return
; ------------------------------------------------------------------
get_pot						; gets speed pot reading
	variable temp = A
	; ------------------------------------------------------------
	; get potentiometer setting -> ADRESH
	; ------------------------------------------------------------
	sel_pot
	call	convert				; WREG <- A/D result
	banksel	flags
	btfsc	FLAG_TRAK			; tracking mode?
	return						; yes
	; ------------------------------------------------------------
	banksel	ADRESH
	; ------------------------------------------------------------
	; pot setting > upper limit neutral rannge?
	; ------------------------------------------------------------
	bcf		FLAG_DIRN			; default
	movlw	NEUTRAL_HI			; upper limit neutral range, run mode
	btfss	FLAG_RUN			
	movlw	NEUTRAL_HI+DELTA	; upper limit neutral range, stop mode
	subwf	ADRESH,W
	movwf	duty
	skpnc						; pot setting =< upper limit?
	return						; yes
	; ------------------------------------------------------------
	; pot setting >= lower limit neutral range?
	; ------------------------------------------------------------
	clrf	duty				; default
	movlw	NEUTRAL_LO			; lower limit neutral range, run mode
	btfss	FLAG_RUN			; run mode?
	movlw	NEUTRAL_LO - DELTA	; lower limit neutral range, stop mode 
	subwf	ADRESH,W
	skpc						; pot >= lower limit?
	goto	$+3					; no
	bsf		FLAG_INIT
	return						
	; ------------------------------------------------------------
	bsf		FLAG_DIRN			; reverse mode
	sublw	0
	movwf	duty
	return
; ------------------------------------------------------------------
; function:		get_trimpot
; description:	gets maximum duty setting => maxduty
; comments:		
;				Conversion algorithm is as follows:
;					- add 1 to trimpot reading
;					- divide by 4
;					- multiply by 3
;					- divide by 4
;					- add (PWM period + 1) / 4
; ------------------------------------------------------------------
get_trimpot
	; -------------------------------------------------------------
	; get trimpot reading 
	; -------------------------------------------------------------
	sel_trimpot
	call	convert		; WREG <- [trimpot]
	banksel	ADRESH
	; -------------------------------------------------------------
	; add 1 to trimpot reading
	; -------------------------------------------------------------
	setc				; assume ADRESH = 0ffh
	movfw	ADRESH
	movwf	maxduty
	incfsz	maxduty,F	; ADRESH = 0ffh?
	clrc				; no
	; -------------------------------------------------------------
	; divide by 4
	; -------------------------------------------------------------
	rrf		maxduty,F	; / 2
	clrc
	rrf		maxduty,W	; / 4 -> W
	; -------------------------------------------------------------
	; multiply [WREG] by 3
	; -------------------------------------------------------------
	addwf	maxduty,F	; x 3
	; -------------------------------------------------------------
	; divide by 4
	; -------------------------------------------------------------
	clrc
	rrf		maxduty,F	; / 2
	clrc
	rrf		maxduty,F	; / 4
	; -------------------------------------------------------------
	; add (MAX_DUTY + 1)/4
	; -------------------------------------------------------------
	movlw	(MAX_DUTY+1)/4
	addwf	maxduty,F
	return			
; ---------------------------------------------------------------
; function:		get_east
; description:	east (photocell output) to ADC
; registers:	eastup:easthi:eastlo - 3 byte working register
; comments:		
;				Successive sensor readings are averaged
;				according to the formula:
;
;				A(n+1) = A(n) * (n - 1) + B(n+1)
;
;				where
;					n is the oversampling (or averaging) count (n = 256)				
;					A(n) is the previous average (3 bytes)
;					B(n+1) is the current measurement (2 bytes
;				
;				The calculation is based on the following
;				algorithm:
;
;				- divide (shift by 1 byte) current average and 
;				subtract from the latter.
;				- add current measurement to previous average
;
; EXAMPLE:		previous average = 0x38749a
;				 new measurement = 0x004b1c
;
;				calculate:	  0x38749a	; previous average (3 bytes
;							- 0x003874	; subtract upper 2 bytes 
;							-----------      of previous average
;							= 0x383c26	; intermediate result
;							+ 0x004b1c	; add new measurent
;							-----------      of previous average
;							= 0x388742	; final result
;
; Note:			Upper byte (most significant byte) is used
;				for comparison purposes. High (middle) byte and
;				low (least significant) are working registers that
;				store required values for successive average
;				calculations.
; ---------------------------------------------------------------
get_east
	sel_east
	call	convert
  if debug
	banksel	ADRESH
	movfw	ADRESH
	movwf	east
	return
  endif	
	movlw	low easthi	
	call	calc_avge
	return
; ---------------------------------------------------------------
; function:		get_west
; description:	see {get_east}
; ---------------------------------------------------------------
get_west
	sel_west
	call	convert
  if debug
	banksel	ADRESH
	movfw	ADRESH
	movwf	west
	return
  endif	
	movlw	low westhi	; point to middle byte of previous average
	call	calc_avge	; calculate new average based on previous
	return				; average and current measurement.
; ---------------------------------------------------------------
; function:		calc_avge
; description:	calculate current average based on current 
;				measurement and previous average.
; inputs:		FSR		- initialised to address of middle byte, current average
;				ADRESH	- high byte, current measurement
;				ADRESL	- low byte, current measurement
; comments:		this function implements the equivalent of averaging
;				up to 256 samples of consecutive readings. This number
;				may be referred to as the "sample count"
;				Note: if a sample count of greater or less than 256 is
;				required then it will be necessary to modify the function.
;
;				In most cases, 8 bit resolution is sufficient, in which 
;				case the lower 2 bytes can be ingored by the calling routine.
;				However, these bytes should not (normally) be modified outside 
;				this function.
; ---------------------------------------------------------------
calc_avge
	bankisel east
	; ---------------------------------------------------------------
	; subtract upper 2 bytes from lower 2 bytes, current average
	; ---------------------------------------------------------------
	movwf	FSR
	movfw	INDF		; high (middle) byte, previous average
	decf	FSR,F		; point to low byte, previous average
	subwf	INDF,F		; subtract high byte from low byte
	movlw	1			; in case of borrow
	incf	FSR,F		; point to high byte
	skpc				; borrow from previous subtract?
	subwf	INDF,F		; yes
	incf	FSR,F		; point ot upper byte, previous average
	movfw	INDF		; upper (most significant) byte of average
	skpc				; borrow from previous subtract?
	decf	INDF,F		; yes
	decf	FSR			; point to high byte
	subwf	INDF,F		; subtract from high (middle) byte
	incf	FSR,F		; point to upper byte
	skpc				; borrow?
	decf	INDF,F		; Yes
	; ---------------------------------------------------------------
	; add current measurement to lower 2 bytes of previous result
	; ---------------------------------------------------------------
	banksel	ADRESL		
	movfw	ADRESL		; low byte, measurement
	banksel ADRESH
	decf	FSR,F
	decf	FSR,F		; point to low byte
	addwf	INDF,F		; add to low byte, previous result
	movlw	1			; in case of carry
	incf	FSR,F		; point to high byte
	skpnc				; carry?
	addwf	INDF,F		; yes
	movfw	ADRESH		; high byte, measurement
	incf	FSR,F		; point to upper byte
	skpnc				; carry from previous add?
	incf	INDF,F		; yes
	decf	FSR,F		; point to high byte
	addwf	INDF,F		; add high byte measurement to middle byte average
	incf	FSR,F		; point to upper byte
	skpnc				; carry?
	incf	INDF,F		; yes
	return
; ---------------------------------------------------------------
; function:		get_amps
; desctription:	motor current to 10-bit ADC
; registers:	ampshi:ampslo	(right justified)
; ---------------------------------------------------------------
get_amps
	variable temp = A
	sel_amps
	call	convert
	; ---------------------------------------------------------------
	banksel	ADRESL
	movfw	ADRESL
	banksel	bank0
	movwf	ampslo	
	; ---------------------------------------------------------------
	movfw	ADRESH
	movwf	ampshi
	; ---------------------------------------------------------------
	movf	CCPR1L,F
	skpnz				; CCPR1L = 0?
	return				; yes
	; ---------------------------------------------------------------
	; ampshi:ampslo < MAX_AMPS?
	; ---------------------------------------------------------------
	bsf		FLAG_Z
	movlw	low	(MAX_AMPS)
	subwf	ampslo,W
	skpz
	bcf		FLAG_Z
	movlw	high MAX_AMPS
	skpc
	addlw	1
	subwf	ampshi,W
	skpz
	bcf		FLAG_Z
	skpc					; ampshi:ampslo < AVGE_AMPS?				' 
	return					; yes
	; ---------------------------------------------------------------
	; ampshi:ampslo = MAX_AMPS?
	; ---------------------------------------------------------------
	btfsc	FLAG_Z			; ampshi:ampslo = AVGE_AMPS?
	return					; yes
	; ---------------------------------------------------------------
	; ampshi:ampslo > AVGE_AMPS
	; ---------------------------------------------------------------
	movlw	high MAX_AMPS
	movwf	E				
	movlw	low	(MAX_AMPS)
	movwf	D
	movfw	ampshi
	movwf	E1
	movfw	ampslo
	movwf	D1
loop=$
	movfw	E
	iorwf	E1,W
	skpnz			; ms bytes both = 0?
	goto	x_lp1	; yes
	clrc			; else...shift bytes
	rrf		E,F		; 	
	rrf		D,F		 
	clrc	
	rrf		E1,F
	rrf		D1,F
	goto	loop
x_lp1
	movfw	CCPR1L
	movwf	E
	call	mult
	movfw	D1
	movwf	D
	bcf		ERR_OVRFLO
	call	div			; L = CCPR1L x MAX_AMPS / ampshi:ampslo
	subwf	maxduty,W	
	skpc				; WREG > maxduty?
	return				; yes
	movfw	L			; WREG <- result of previous division
	movwf	maxduty
	return
;---------------------------------------------------------------
; function:		chk_thresh	
; description:	check if photsensor outputs below threshold levels
; flags:		ERR_DARK		; 1 = "both east & west < THRESH_DARK"
;								; 0 = "[either east or west >= THRESH_DARK}
;				ERR_CLOUD		; 1 = both east & west < THRESH_CLOUD
;								; 0 = "[either east or west >= THRESH_CLOUD}
; ---------------------------------------------------------------
chk_thresh
	banksel	flags
	btfss	FLAG_TRAK		; manual mode?
	return					; yes
	; ---------------------------------------------------------------
	; check if both east and west outputs less than cloud threshold
	; ---------------------------------------------------------------
	bsf		ERR_CLOUD		; default
	movfw	east
	sublw	THRESH_CLOUD	
	skpc
	bcf		ERR_CLOUD
	movfw	west
	sublw	THRESH_CLOUD	
	skpc
	bcf		ERR_CLOUD
	; ---------------------------------------------------------------
	; check if both east and west outputs less than dark threshold
	; ---------------------------------------------------------------
	bcf		ERR_DARK		; default
	movfw	east
	sublw	THRESH_DARK	
	skpc					; east > THRESH_DARK?
	return					; yes
	movfw	west
	sublw	THRESH_DARK
	skpc					; west > THRESH_DARK?
	return					; yes
	; ---------------------------------------------------------------
	bsf		FLAG_DIRN
	bsf		ERR_DARK
	movfw	maxduty
	movwf	duty
 	return
; ---------------------------------------------------------------
; function:		chk_limits
; description:	check if limit switches are open
; flags:		ERR_EAST <- 1 if east_sw = 1
;				ERR_WEST <- 1 if west_sw = 1
;				ERR_EAST <- 0 if east_sw = 0 and FLAG_REV = 0
;				ERR_WEST <- 0 if west limit swich closed AND FLAG_DIRN = 1
; ---------------------------------------------------------------
chk_limits
	banksel	PORTA
	; ---------------------------------------------------------------
	; set error flags if either east_sw or west_sw = 1
	; ---------------------------------------------------------------
	btfsc	west_sw		; west limit switch open?
	bsf		ERR_WEST	; yes
	btfsc	east_sw		; east switch open?
	bsf		ERR_EAST	; yes
	; ---------------------------------------------------------------
	btfsc	FLAG_REV	; currently reverse mode?
	goto	chk_rev		; yes
	; ---------------------------------------------------------------
	; is east switch closed (forward mode)
	; ---------------------------------------------------------------
chk_fwd
	btfss	east_sw		; east limit switch closed?
	bcf		ERR_EAST	; yes
	return
	; ---------------------------------------------------------------
	; is west switch closed (forward mode)
	; ---------------------------------------------------------------
chk_rev
	btfss	west_sw		; west limit switch closed?
	bcf		ERR_WEST	; yes
	return
; ------------------------------------------------------------------
; function:		convert
; description:	perform an adc conversion
; notes:		Before calling this function:
;				1. select analogue input.
;				2. allow minimum 12 us acquisition time
; 				3. Do not disable interrupts in a/d conversion 
;				   because this might interfere with remote 
;				   control timing
;				Conversion time ~ 24 us
; 				(excludes acquisition time)
;				for setting acquisition time assume MCU clock = 4 MHz
; ------------------------------------------------------------------
convert	
	movlw	2
loop1=$
	banksel	ADCON0				; loop 10 us
	addlw	-1
	skpz
	goto	loop1
	; --------------------------------------------------------------
	bsf		ADCON0,GO			; start conversion
	; --------------------------------------------------------------
loop2=$
	btfsc	ADCON0,NOT_DONE		; ADC complete?
	goto 	loop2 				; No
	; --------------------------------------------------------------
	bcf		ADCON0,ADON
	return
; ------------------------------------------------------------------
; function:		init_dtmr
; description:	initialises debounce timer (dtmr)
; ------------------------------------------------------------------
init_dtmr
	preset	dtmr,DBNCE_TIME
	return
; ------------------------------------------------------------------
; function:		calc_duty
; description:	calculates duty cycle while in tracking mode
; comments:
;				In manual mode:
;					FLAG_MODE = 1
;					duty cycle and motor direction set by pot
;					 
;				In tracking mode:
;					duty cycle = |east - west| * GAIN_FACTOR
;					FLAG_MODE = 0
;					FLAG_DIRN = 0 if west >= east	
;					FLAG_DIRN = 1 if east > west
; ------------------------------------------------------------------
calc_duty
	variable	temp = A
	banksel	0
	btfss	FLAG_TRAK 		; manual mode?
	return					; yes
	; ------------------------------------------------------------------
	; calculate: west-east
	; ------------------------------------------------------------------
	bcf		FLAG_DIRN		; default (set for forward direction)
	movfw	east
	subwf	west,W
	skpnc					; result is positive?
	goto	calc_duty1		; yes
	; ------------------------------------------------------------------
	; ******************************************************************
	; the following code segment depends upon 
	; whether TRACKING_MODE == BIIDIRECTIONAL
	; or TRACKING_MODE == UNIDIRECTIONAL
	; ******************************************************************
	; ------------------------------------------------------------------
  if TRACKING_MODE == UNIDIRECTIONAL
	clrf	duty
	return
  else
	if TRACKING_MODE == BIDIRECTIONAL	
		bsf		FLAG_DIRN		; set for reverse direction
		sublw	0				; convert negative result to positive
	else
		error"TRACKING_MODE undefined"
	endif
  endif	
	; ------------------------------------------------------------------
calc_duty1
	movwf	D
	; ------------------------------------------------------------------
	; duty = duty - HYSTERESIS
	; ------------------------------------------------------------------
	movlw	HYSTERESIS
	subwf	D,F				; adjust for hysteresis
	skpc					; D < 0?
	clrf	D				; yes
	; ------------------------------------------------------------------
	movlw	GAIN_FACTOR
	movwf	E
	call	mult			; HL = |east - west| * GAIN_FACTOR
	; ------------------------------------------------------------------
	movfw	L				; low byte, previous result
	movf	H,F
	skpz					; high byte, previous result > 0?
	movfw	maxduty			; yes
	movwf	duty				
	; ------------------------------------------------------------------
	movf	duty,F			; duty = 0?
	skpnz
	return					; yes
	; ------------------------------------------------------------------
	movlw	MIN_DUTY
	addwf	duty,F
	skpc					; previous result > 0ffh?
	return					; no
	; ------------------------------------------------------------------
	movfw	maxduty
	movwf	duty			; yes
	return					
; ------------------------------------------------------------------
; function:		motor_handler
; description:	speed -> duty cycle
; comments:
; 	
; External circuitry arranged according to the logic diagram below:
;
;  en_fwd ---------------o----------------------------------------
;		                 |            + 12V                       |      
;                        |        ______|______                   |
;	            __ 	     |       |             |           __     |
;              |   \     |    |--	            --|      /   |    | 
;            --| G1 o----)----| Q1 <-  p-ch -> Q2 |-----o G2 |----
;		 	|  |__ /     |    |--               --|      \ __|  
;	        |    _ _ _ _/        |	           |               
;	         \ /                 |--- MOTOR ---|               
;	         / \ _ _ _ _         |	           |
;           |            \       |             |
;           |    __       |      |             |          __
; 		     ---|   \     |   |--			    --|     /   |-------  
;               | G3 |----)---|	Q3 <- n-ch -> Q4  |----| G4 |       | 
;	   pwm -----|__ /     |   |--			    --|     \ __|----   | 
;	        |       	  |      |_____________|                 |  | 
;	        |       	  |		        |                        |  | 
;           | 			  |            0V                        |  | 
;           | _ _ _ _ _ _ |_ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ |  |
;						  |                                         |
;  en_rev ----------------o-----------------------------------------
;
; Modes of operation as follows:
;
; FORWARD RUN MODE:
;
; 	- PWM = modulated pulse 
; 	- en_fwd = 1;	    Turns on  Q2, enables Q3
;	- en_rev = 0;		Turns off Q1, disables G4
; 
; REVERSE RUN MODE:
;	- PWM = modulated pulse
; 	- en_fwd = 0;	    Turns off Q2, disables G3
;	- en_rev = 1;		Turns on Q1, enables G4
; 
; STOP MODE:
;
;	- PWM = 0
; 	- en_fwd = 0;	    Turns off Q2, disables G3
;	- en_rev = 0;		Turns off Q1, disables G4
; ------------------------------------------------------------------
motor_handler
	banksel	bank0
	; ------------------------------------------------------------------
	movf	duty,F		
	skpnz				; duty = 0?
	goto	motor_stop	; yes
	; ------------------------------------------------------------------
	btfss	FLAG_TRAK	; manual mode?
	goto	motor_start	; yes
	; ------------------------------------------------------------------
	btfsc	ERR_DARK	; dark?
	goto	mtr_strt1	; yes
	; ------------------------------------------------------------------
	btfsc	ERR_CLOUD	; cloud?
	goto	motor_stop	; yes
	goto	mtr_strt1	; else...	
	; ------------------------------------------------------------------
motor_start
	btfss	FLAG_INIT	; manual mode initialised?
	goto	motor_stop	; no
	; ------------------------------------------------------------------
mtr_strt1
	btfsc	stmre		; stmr active?
	return				; yes
	; ------------------------------------------------------------------
	btfsc	FLAG_DIRN	; direction currently reverse?
	goto	motor_rev	; yes
	; ------------------------------------------------------------------
motor_fwd
	btfss	FLAG_REV	; direction previously forward?
	goto	$+3			; yes
	; ------------------------------------------------------------------
	btfsc	FLAG_RUN	; currently in run mode?
	goto	motor_stop	; yes (cannot be in both reverse and run modes)
	; ------------------------------------------------------------------
	btfsc	ERR_WEST	; west switch open?
	goto	motor_stop	; yes
	; ------------------------------------------------------------------
	btfsc	en_rev		; reverse driver enabled?
	goto	motor_stop	; yes: should be disabled
	; ------------------------------------------------------------------
	bsf		en_fwd		; enable forward drivers
	btfss	en_fwd		; drivers enabled?
	goto	motor_stop	; no (could be a hardware problem)
	bsf		FLAG_RUN	
	bcf		FLAG_REV
	; ------------------------------------------------------------------
	movfw	duty		; desired speed (duty) setting
	call	set_duty	; adjust duty cycle
	return
	; ------------------------------------------------------------------	
motor_rev
	btfsc	rtmrf		; run timeout?
	goto	motor_stop	; yes
	; ------------------------------------------------------------------	
	btfsc	FLAG_REV	; direction previously reverse?
	goto	$+3			; yes
	; ------------------------------------------------------------------	
	btfsc	FLAG_RUN	; currently in run mode?
	goto	motor_stop	; yes (cannot be in both forward and run modes)
	; ------------------------------------------------------------------
	btfsc	ERR_EAST	; east switch open?
	goto	motor_stop	; yes
	; ------------------------------------------------------------------	
	btfsc	en_fwd		; forward driver enabled?
	goto	motor_stop	; yes: should be disabled
	; ------------------------------------------------------------------	
	bsf		en_rev		; enable reverse driver
	btfss	en_rev		; driver enabled?	
	goto	motor_stop	; no (could be hardware fault)
	bsf		FLAG_RUN	
	bsf		FLAG_REV
	; ------------------------------------------------------------------	
	movfw	duty		; desired speed (duty) setting	
	call	set_duty	; adjust duty cycle
	return
	; ------------------------------------------------------------------	
motor_stop
	movf	CCPR1L,F
	skpnz				; motor already stopped?
	goto	motor_stop1	; yes
	; ------------------------------------------------------------------	
	clrw					 
	call	set_duty	; decelerate motor
	return				; no
	; ------------------------------------------------------------------	
motor_stop1
	bcf		en_fwd		; disable forward driver
	bcf		en_rev		; disable reverse driver
	; ------------------------------------------------------------------	
	bcf		FLAG_REV	; default
	btfsc	FLAG_DIRN	; set to reverse mode?
	bsf		FLAG_REV
	; ------------------------------------------------------------------	
	btfss	FLAG_RUN	; run mode?
	return				; no
	; ------------------------------------------------------------------	
	bcf		FLAG_RUN
	preset 	stmr,STOP_TIME	; initialise delay before restart motor
	return
; ------------------------------------------------------------------	
; function:		set_duty
; description:	duty cycle is adjusted up or down, if necessary
;				according to speed setting to give a controlled
;				acceleration or deceleration rate.
;				
;				TO increase motor speed, add |input value-CCPR1L| / 8
;				or 1, whichever is greater to CCPR1L.
;
;				TO decrease motor speed subtract |speed-CCPR1L| / 8
;				or 1, whichever is greater from CCPR1L
;
; rates of acceleration and deceleration are set by the constants
; ACEL_TIME or DECEL_TIME, respectively, in <constants.inc>
;
; At current revision: 
;
; 	maximum duty cycle is achieved, starting
; 	from zero duty, after approx. 250 ms
;
; 	zero duty cycle is achieved, starting
; 	from maximum duty, after approx. 80 ms
; ------------------------------------------------------------------	
set_duty
	variable inval = A, temp = A
	banksel	bank0
	btfsc	ctmre		; ctmr active?
	return				; yes
	movwf	inval
 	movfw	maxduty
	subwf	inval,W
	movfw	maxduty
	skpnc				; inval > maxduty?
	movwf	temp		; yes
	movfw	CCPR1L
	subwf	temp,W
	skpc				; inval < CCPR1L?
	goto	dec_duty	; yes
	skpnz				; inval = CCPR1L?
	return				; yes
inc_duty
	movwf	temp		
	clrc	
	rrf		temp,F		; |temp - CCPR1L} / 2  
	clrc				
	rrf		temp,F		; |temp - CCPR1L} / 4 
	clrc				
	rrf		temp,W		; |temp - CCPR1L} / 8  -> WREG
	addlw	0
	skpnz
	movlw	1
	addwf	CCPR1L,F	; increment CCPR1L by 25% 
	preset	ctmr,ACEL_TIME
	return
dec_duty
	movfw	temp
	subwf	CCPR1L,W
	movwf	temp
	clrc	
	rrf		temp,F		; |temp - CCPR1L} / 2 
	clrc	
	rrf		temp,F		; |temp - CCPR1L} / 4 
	clrc				
	rrf		temp,W		; |speed - CCPR1L} / 8 -> WREG
	addlw	0
	skpnz
	movlw	1
	subwf	CCPR1L,F	; decrement CCPR1L by 25% 
	preset	ctmr,DCEL_TIME
	return

; *************************************************
; mult subroutine - 8 bit x 8 bit -> 16 bit result
;
; ACTION:	multiplier * multiplicand -> product
;
; INPUTS:
;
;	D	(multiplier)
;	E	(multiplicand)
;
; OUTPUTS:
;
;	H	<- high byte (product)
;	L	<- low byte (product)
;	W	<- low byte (product)
;
; REGISTERS MODIFIED:
;
;	E, H, L, count
; *************************************************

mult 	; declare variables

	variable mulplr = D
	variable mulcnd = E
	variable h_byte = H	
	variable l_byte = L
	variable count  = X

	banksel	0

	; initialise variables

	clrf	h_byte
	clrf	l_byte
	movlw	8
	movwf	count

	; multiply
	
	movfw	mulcnd
	bcf		STATUS,C
loop=$
	rrf		mulplr,F
	btfsc	STATUS,C
	addwf	h_byte,F	
	rrf		h_byte,F
	rrf		l_byte,F
	decfsz	count,F
	goto	loop
	movfw	l_byte
	return		
; _______________________________________________________________
; ---------------------------------------------------------------
; function:		div
; description:	16-bit / 8-bit divide
; inputs:		HL		16-bit number to be divided
;				D		8-bit divisor
; outputs:		W, L	dividend
;				H		remainder
; flags:		ERR_OVRFLO = 1 if overflow error
; registers:	X, E, STATUS
; comments:		overflow error occurs if divisor (D) is smaller
;				than high byte, input value to be divided, i.e.
;				if result of division is greater than 0xff. This
;				could be, but is not always, an indicator of a
;				divide by zero.
; ---------------------------------------------------------------
div
	; declare variables
	variable	divisor = D
	variable	msb	= E
	variable	h_byte	= H
	variable	l_byte	= L
	variable	result 	= l_byte
	variable	count	= X	
	; test for overflow (H >= D)
	banksel	0
	bcf		ERR_OVRFLO	; default, no error
	movfw	D			; divisor
	subwf	H,W			; h_byte-divisor
	btfsc	STATUS,C	; h_byte < divisor?
	bsf		ERR_OVRFLO	; No? Then set flag
	clrf	msb			; extra byte attached to 16-bit input
	movlw	d'8'
	movwf	count	
dloop	
	rlf		l_byte,F
	rlf		h_byte,F	
	rlf		msb,F	
	rrf		result,F	; this line is NOT superfluous
	movfw	divisor
	subwf	h_byte,F
	movlw	1
	btfss	STATUS,C
	subwf	msb,F		
	rlf		result,F	
	btfss	result,C	
	incf	msb,F
	movfw	divisor
	btfss	result,C
	addwf	h_byte,F
	decfsz	count,F
	goto 	dloop
	movfw	result	
	return	
; ======================================================================
	end




