; ***************************************************************************
; file: 			constants.inc
; Last revision: 	5 October, 2011
; Author:			Herman Nacinovich
; ****************************************************************
	; -----------------------------------------------------------------------------------------
	; general constants
	; -----------------------------------------------------------------------------------------
	#define	FALSE	0
	#define	false	FALSE
	#define	TRUE 	!false
	#define	true 	TRUE
	#define	debug	false
	#define	bank0	0x00
	#define bank1	0x80
	; -----------------------------------------------------------------------------------------
	; tracking mode definitions
	; -----------------------------------------------------------------------------------------
	#define BIDIRECTIONAL	FALSE
	#define UNIDIRECTIONAL	TRUE
	#define TRACKING_MODE	BIDIRECTIONAL
	; -----------------------------------------------------------------------------------------
	; OPTION_REG assignment			- disable port B pullups, set TMR0 prescale
	; -----------------------------------------------------------------------------------------
	#define	OPTION_VAL	b'10000001'	; port b pullups disabled, prescaler 1:4 -> TMR0
	; -----------------------------------------------------------------------------------------
	; OSCCON assignment				- sets MCU clock frequency
	; -----------------------------------------------------------------------------------------
	#define	OSCCON_VAL	b'01100000'	; int. RC osc = 4 MHz
	; -----------------------------------------------------------------------------------------
	; T2CON assignment				- set PWM frequency (also sets periodic time signals)
	; -----------------------------------------------------------------------------------------
	#define	T2CON_VAL	b'01111100'	; TMR2 prescale 1:1, postscale 1:16
	; -----------------------------------------------------------------------------------------
	; PR2 assignment				- set PWM frequency
	; -----------------------------------------------------------------------------------------
	#define	PR2_VAL		0x3f		; sets PWM frequency = 19.5 kHz (in conjunction with OSCCON)
	; -----------------------------------------------------------------------------------------
	; CCP1CON assignment			- set PWM mode
	; -----------------------------------------------------------------------------------------
	#define CCP1CON_VAL	b'00001100'	; sets PWM mode
	; -----------------------------------------------------------------------------------------
	; ADCON0 assignment				- set A/D clock, mode
	; -----------------------------------------------------------------------------------------
	#define	ADCON0_VAL	b'01000001'	; clock = Fosc/8, A/D = on
	; -----------------------------------------------------------------------------------------
	; ADCON1 assignment				- set A/D justify, voltage reference
	; -----------------------------------------------------------------------------------------
	#define	ADCON1_VAL	b'00000000'	; left justify, Vref -> Vcc
	; -----------------------------------------------------------------------------------------
	; ANSEL assignment				- select analog/digital	
	; -----------------------------------------------------------------------------------------
	#define	ANSEL_VAL	b'00001111'	; AN7-AN4 = digital, AN3-AN0 analogue
	; -----------------------------------------------------------------------------------------
	; threshold definitions
	; -----------------------------------------------------------------------------------------
	#define	CF0	d'51' / d'1000'				; calibration factor, input voltage (mV) to hex. units
	#define	THRESH_CLOUD (d'800' * CF0)		; cloud threshold (mV * calibration factor)
	#define	THRESH_DARK  (d'20' * CF0)		; dark threshold  (mV * calibration factor)
	#define	HYSTERESIS	(D'40'* CF0)		; minimum difference (mV) between east and west before motor will track
	#define	GAIN_FACTOR	d'8'				; multiplier used in calculating duty cycle
	; -------------------------------------------------------------
	; ****************************************************************************
	; *		Current is measured by using a current sensing 
	; *     resistance (R_SENSE), which is defined in units 
	; *		of milliohms
	; *
	; *		The current limit (CURRENT_LIMIT) is defined in 
	; *		units of milliamps, and is converted to the
	; *		equivalent hexadecimal value by multiplying by
	; *		a convertion factor (calculated from the sensing
	; *		resistance).
	; ****************************************************************************
	; -------------------------------------------------------------
	#define	R_SENSE			d'100'				; current sensing resistance (milliohms)
	#define CURRENT_LIMIT	d'2000'				; current limit (milliamps)
	#define	CFR				R_SENSE / d'6700'	; conversion factor
	#define	MAX_AMPS		(CURRENT_LIMIT * CFR) 	
	; -------------------------------------------------------------
	#define	NEUTRAL		0x80
	#define	NEUTRAL_HI	NEUTRAL + 0x20		; upper end of neutral range
	#define	NEUTRAL_LO	NEUTRAL - 0x20		; lower end of neutral range
	#define	DELTA		0x01				; hysteresis in neutral range
	; -----------------------------------------------------------------------------------------
	; acceleration and deceleration rate definitions
	; -----------------------------------------------------------------------------------------
	#define MAX_DUTY	PR2_VAL/2	; PWM period (microseconds) 
	#define	MIN_DUTY	PR2_VAL/4	; minimum duty in tracking mode
	#define	ACEL_TIME	d'50'	; time (ms) between duty cycle updates
	#define DCEL_TIME	d'5'	; time (ms) between duty cycle updates
	#define	STOP_TIME	d'250'	; time (ms) between motor stop and motor restart
	#define	DBNCE_TIME	d'30'	; time (ms) for switch debounce
	#define	LEDON_TIME	d'50'	; led on time (in standby mode)
	#define	LEDOFF_TIME	d'1950'	; led off time (in standby mode)
	#define	RUN_TIMEOUT	d'600000'	; run timeout when dark
	; -----------------------------------------------------------------------------------------
	; east, west register definitions
	; -----------------------------------------------------------------------------------------
	#define	east		eastup	; upper byte of 3-byte working register
	#define	west		westup	; upper byte of 3-byte working register
	; -----------------------------------------------------------------------------------------
	; timer definitions	- there are 8 general purpose (2-byte)
	;					  timers (only 3 are actually used)
	; -----------------------------------------------------------------------------------------
	#define	tmr1	tmr1lo
	#define	tmr2	tmr2lo
	#define	tmr3	tmr3lo
	#define	tmr4	tmr4lo
	#define	tmr5	tmr5lo
	#define	tmr6	tmr6lo
	#define	tmr7	tmr7lo
	#define	tmr8	tmr8lo		; initialisation timer
	; -----------------------------------------------------------------------------------------
	; timer flag definitions
	; -----------------------------------------------------------------------------------------
	#define	timers	tmr1lo
	#define	tmr1e	tmrcon,(tmr1-timers)/2		
	#define	tmr1f	tmrflags,(tmr1-timers)/2	
	#define	tmr2e	tmrcon,(tmr2-timers)/2		
	#define	tmr2f	tmrflags,(tmr2-timers)/2	
	#define	tmr3e	tmrcon,(tmr3-timers)/2		
	#define	tmr3f	tmrflags,(tmr3-timers)/2	
	#define	tmr4e	tmrcon,(tmr4-timers)/2		
	#define	tmr4f	tmrflags,(tmr4-timers)/2	
	#define	tmr5e	tmrcon,(tmr5-timers)/2		
	#define	tmr5f	tmrflags,(tmr5-timers)/2	
	#define	tmr6e	tmrcon,(tmr6-timers)/2		
	#define	tmr6f	tmrflags,(tmr6-timers)/2	
	#define	tmr7e	tmrcon,(tmr7-timers)/2	
	#define	tmr7f	tmrflags,(tmr7-timers)/2
	#define	tmr8e	tmrcon,(tmr8-timers)/2	
	#define	tmr8f	tmrflags,(tmr8-timers)/2
	; -----------------------------------------------------------------------------------------
	; alternative timer definitions
	; -----------------------------------------------------------------------------------------
	#define	ctmr	tmr1		; acceleration and deceleration times
	#define	dtmr	tmr2		; switch debounce
	#define	stmr	tmr3		; motor start delay
	#define	ltmr	tmr4
	#define	rtmr	tmr8		; run timeout (used when dark)
	; -----------------------------------------------------------------------------------------
	; alternative timer flag definitions
	; -----------------------------------------------------------------------------------------
	#define	ctmre	tmr1e
	#define	ctmrf	tmr1f
	; -----------------------------------------------------------------------------------------
	#define	dtmre	tmr2e
	#define	dtmrf	tmr3f
	; -----------------------------------------------------------------------------------------
	#define	stmre	tmr3e
	#define	stmrf	tmr3f
	; --------------------------------------------------------------------
	#define	ltmre	tmr4e
	#define	ltmrf	tmr4f
	; --------------------------------------------------------------------
	#define	rtmre	tmr8e
	#define	rtmrf	tmr8f
	; --------------------------------------------------------------------
	; error flag assignments
	; -----------------------------------------------------------------------------------------
	#define	ERR_OVRFLO		err,0		; overflow in division
	#define ERR_EEWRITE		err,1		; eeprom write error
	#define ERR_LOBAT		err,2		; low battery voltage
	#define	ERR_CLOUD		err,3
	#define	ERR_DARK		err,4
	#define	ERR_EAST		err,5
	#define	ERR_WEST		err,6
	#define ERR_OVL			err,7		; motor overload
	; -----------------------------------------------------------------------------------------
	; miscellaneous flags definitions
	; -----------------------------------------------------------------------------------------
	#define	FLAG_GP			flags,0		; general purpose flag
	#define	FLAG_Z			flags,1		; used by get_amps function
	#define	FLAG_SW			flags,2		; used by get_limits function
	#define	FLAG_TRAK 		flags,3		; 0 = manual mode, 1 = tracking mode
	#define	FLAG_INIT	 	flags,4		; 0 = "manual mode not initialised"
 										; 1 = "manual mode initialised"
	; -----------------------------------------------------------------------------------------
	; motorcon control flags
	; -----------------------------------------------------------------------------------------
	#define FLAG_RUN	 	motorcon,0		; 0 = stop mode, 1 = run mode
	#define	FLAG_DIRN	 	motorcon,1		; 0 = forward, 1 = reverse
	#define	FLAG_REV	 	motorcon,2		; 0 = forward, 1 = reverse
	; -----------------------------------------------------------------------------------------
	; porta assignments				- set port A direction
	; -----------------------------------------------------------------------------------------
	#define	TRISA_VAL	b'11111111'	; port A, all inputs
	#define	mode_sw		PORTA,5		; 0 = tracking mode, 1 = manual mode
	; -----------------------------------------------------------------------------------------
	; portb assignment				- set port b direction
	; -----------------------------------------------------------------------------------------
	#define	TRISB_VAL	b'11010010'	 ; 0 = output, 1 = input
	#define	PORTB_INIT	b'00000000'	 ; initial value port B	
	; -----------------------------------------------------------------------------------------
	#define	RB0		PORTB,0
	#define	RB1		PORTB,1
	#define	RB2		PORTB,2	
	#define	RB3		PORTB,3	
	#define	RB4		PORTB,4
	#define	RB5		PORTB,5
	; ---------------------------------------------------------------------------------------------
	#define	PWM			RB0		; output: pwm output - see notes on motor_handler function for more details
	#define	east_sw		RB1		; input:  0 = closed, 1 = open
	#define	en_fwd		RB2		; output: 0 = disable, 1 = enable (bottom forward mosfet driver)
	#define	en_rev		RB3		; output: 0 = disable, 1 = enable (bottom reverse mosfet driver)
	#define	west_sw		RB4		; input:  0 = closed, 1 = open		
	#define	led			RB5		; 0 = led on, 1 = led off
	; ---------------------------------------------------------------------------------------------
