; *************************************************************************
; File:				timers.inc
; Description:		contains timer routines
; Last revision:	16 November, 2007
; *************************************************************************
; ______________________________________________________________________________________________________
; function:		timer
; description:	Provides basic timing ~ 1.024 ms - based on TMR2
;				use for timing intervals up to 4.6 hours
; inputs:		tmr	= tmrup:tmrhi:tmrlo	(master timer, 3 byte)
;				tmr1 - tmr7			 	(slave timer, 2 bytes)
;				tmr8					(slave timer, 3 byte)
;				tmrcon					(timer enable flags)
;				tmrflags				(timeout flags)
; comments:		each of up to 8 gp timers can be preset for timing 
;				intervals as specified above. A gp timer is actived
;				by clearing the corresponding timeout flag in tmrflags
;				and setting the corresponding enable flag in tmrcon.
; 				When activating a timer, also preset the timer according
;				to the formula:
;		
;				preset value = (tmr) + t (ms)
; 
;				The master timer (tmr) is incremented asynchronously
;				each time the TMR2 timeout flag (PIR1<TMR2IF> is set.
;				On every update each slave timer, if active, is compared
;				with the master timer. If their values match then the
;				corresponding enable bit is cleared and timeout flag is set.
;				In the case of the 2-byte slave timers only the lower
;				two bytes of the master timer are used for the compare
;				operation. In the case of the 3-byte slave timer (tmr8)
;				all 3 bytes are used for the compare operation.	
; -----------------------------------------------------------------------------
timer
	banksel	PIR1
	btfss	PIR1,TMR2IF
	return
	bcf		PIR1,TMR2IF		; clear timeout flag till next timeout
	incfsz	tmrlo,F			; increment master timer, low byte
	goto	timer1a			; low byte rollover?
	incf	tmrhi,F			; yes
	skpnz					
	incf	tmrup,F	
timer1a		
	clrf	mask
	incf	mask,F
	movlw	low	timers
	movwf	FSR
	bankisel timers
loop=$
	movfw	mask
	andwf	tmrcon,W	
	skpnz					; timer active?
	goto	tx2				; no (don't compare bytes)
	movfw	tmrlo
	xorwf	INDF,W
	skpz					; low bytes match?
	goto	tx2				; no
	incf	FSR,F			; point to slave timer, high byte
	movfw	tmrhi
	xorwf	INDF,W
	skpz					; high bytes match?
	goto	tx2+1			; no
	btfss	mask,7			; currently comparing tmr8?
	goto	tx1				; no
	incf	FSR,F			; point to upper byte (tmr8)
	movfw	tmrup
	xorwf	INDF,W
	skpz					; upper bytes match?
	goto	tx2+2			; no
tx1							; all bytes match!
	movfw	mask
	iorwf	tmrflags		; set timeout flag
	comf	mask,W			; complement all bits
	andwf	tmrcon			; clear enable flag
	goto	tx2+1
tx2
	incf	FSR,F			; point to high byte (current timer)
	incf	FSR,F			; point to low byte  (next timer)
	clrc					; clear carry for shift
	rlf		mask,F			; shift mask bit
	skpc					; all done?
	goto	loop

	return			

; __________________________________________________________________
; ------------------------------------------------------------------
; function:		init_tmrx
; description:	initialises 16-bit timer
; inputs:		tmr  = master timer
;				tmrx = time interval
;				WREG = address tmrx
;				mask = mask byte for setting/clearing flags
; outputs:		tmrx = tmr0 + time interval
;				tmrcon, tmrflags - updated flags
; comments:		use preset_tmr macro to initialise timer
; ------------------------------------------------------------------
init_tmrx
	banksel		timers
	bankisel	timers
	; ------------------------------------------------------------
	movwf	FSR					; point to slave timer
	; ------------------------------------------------------------
	; init. low byte tmrx
	; ------------------------------------------------------------
	movfw	tmrlo				; add master timer, low byte
	addwf	INDF,F				; to time offset, low byte
	; ------------------------------------------------------------
	incf	FSR,F				; point to time offset, high byte
	; ------------------------------------------------------------
	skpc						; carry on previous add?
	goto	tmrx1				; no
	; ------------------------------------------------------------
	incf	INDF,F				 ; add carry to time offset, high byte
	; ------------------------------------------------------------
	btfss	mask,7				; tmrx == tmr8?
	goto	tmrx1				; no
	; ------------------------------------------------------------
	skpnz						; zero on previous increment?
	incf	tmr8up				; yes: add carry to time offset, upper byte
	; ------------------------------------------------------------
	; init. high byte tmrx
	; ------------------------------------------------------------
tmrx1
	movfw	tmrhi				; master timer, high byte
	addwf	INDF,F				; add time offset, high byte
	; ------------------------------------------------------------
	btfss	mask,7				; tmrx == tmr8?
	goto	tmrx2				; no
	; ------------------------------------------------------------
	; tmrx == tmr8
	; ------------------------------------------------------------
	skpnc						; carry from previous add?
	incf	tmr8hi				; yes
	; ------------------------------------------------------------
	movfw	tmrup
	addwf	tmr8up,F			; yes
	; ------------------------------------------------------------
	; update tmrcon, tmrflags
	; ------------------------------------------------------------
tmrx2
	movfw	mask
	iorwf	tmrcon,F			; set timer enable flag
	comf	mask,W				; complement all mask bits
	andwf	tmrflags,F			; clear timeout flag
	return
; ----------------------------------------------------------------

