/******************************************************************************
 *
 *		M M A 8 4 5 1 Q    A C C E L E R O M E T E R   C O D E  
 *
 */
#include "accel.h"
#include "display.h"
#include <math.h>

#define BATT_LEVEL_MV		2800			// 2.8 volts
#define CAL_ADDRESS 		0x00
#define FLAG_ADDRESS		0x04
#define MODE_ADDRESS            0x08

unsigned char rawdata[6];
float theta, phi, rho;
float thetaraw, thetaoffset;
unsigned int battlevel;
char percentmode;

void accInitialise(void) {
    dispInitialise();
    // Set up I2C
    i2cInitialise();
    // Set up pins
    CAL_TRIS = 1;
    INT1_TRIS = 1;
    INT2_TRIS = 1;
    INTCON2bits.INTEDG1 = 0;                // Int 1 falling edge
    INTCON3bits.INT1IP = 0;                 // Int 1 low prio
    INTCON2bits.INTEDG2 = 0;                // Int 2 falling edge
    INTCON3bits.INT2IP = 0;                 // Int 2 low prio
    CAL_PULLUP = 1;
    INTCON2bits.RABPU = 0;                  // enable pullups
    INTCON2bits.RABIP = 0;                  // Cal button low priority
    CAL_IOC = 1;                            // Enable interrupt on change
    i2cSetByte(ACC_ADDR, CTRL_REG1, i2cGetByte(ACC_ADDR, CTRL_REG1) & 0xfe);
    // Set Up Timer 0
    T0CONbits.TMR0ON = 0;                   // Off
    T0CONbits.T08BIT = 0;                   // 16 bit
    T0CONbits.T0CS = 0;                     // Internal 4MHz clock
    T0CONbits.PSA = 0;                      // Prescaler assigned
    T0CONbits.T0PS = 7;                     // Divide by 256
    INTCON2bits.TMR0IP = 0;                 // low prioity interrupt
    // Set up Battery Monitor
    ADCON0bits.CHS = 0xf;                   // convert FVR
    ADCON1bits.PVCFG = 0;                   // +ve ref Vdd
    ADCON1bits.NVCFG = 0;                   // -ve ref Vss
    ADCON2bits.ADFM = 1;                    // Result right justified
    ADCON2bits.ACQT = 0x4;                  // 8 Tad acquisition time
    ADCON2bits.ADCS = 0x5;                  // Tad = Fosc/16 = 1MHz
    VREFCON0bits.FVR1S0 = 1;                // FVR = 1.024V
    VREFCON0bits.FVR1EN = 1;                // FVR on
    while (VREFCON0bits.FVR1ST == 0);       // wait till FVR is stable
    ADCON0bits.ADON = 1;                    // turn ADC on
    battlevel = (unsigned int) (1047552 / BATT_LEVEL_MV);
}

void accStart(void) {
    uchar temp;
    // Put into active mode
    i2cSetByte(ACC_ADDR, CTRL_REG1, i2cGetByte(ACC_ADDR, CTRL_REG1) | 0x01);
    // get calibration and mode value
    thetaoffset = 0.0;
    percentmode = 0;
    if (eeReadChar(FLAG_ADDRESS) == 0x00) {  // Good values
        thetaoffset = eeReadFloat(CAL_ADDRESS);
        percentmode = eeReadChar(MODE_ADDRESS);
    }
    else {
        eeWriteFloat(CAL_ADDRESS, 0.00);
        eeWriteChar(MODE_ADDRESS, 0x00);
        eeWriteChar(FLAG_ADDRESS, 0x00);
    }
    // Enable interrupts
    INTCON3bits.INT1IF = 0;                 // Clear int1 flag
    INTCON3bits.INT1IE = 1;                 // enable int1
    INTCON3bits.INT2IF = 0;                 // Clear int2 flag
    INTCON3bits.INT2IE = 1;                 // enable int2
    temp = CAL_PIN;                         // clear any mismatch and
    INTCONbits.RABIF = 0;                   // Clear IOC flag
    INTCONbits.RABIE = 1;                   // Enable interrupt on change
}

void accSet(uchar reg, uchar value) {
    i2cSetByte(ACC_ADDR, reg, value);
}

uchar accGet(uchar reg) {
    return i2cGetByte(ACC_ADDR, reg);
}

void accInt1isr(void) {
    int accx, accy, accz;

    i2cRead(ACC_ADDR, OUTPUT_REGS, 6, rawdata);
    accx = ((unsigned int) rawdata[0] << 8) + rawdata[1];
    accy = ((unsigned int) rawdata[2] << 8) + rawdata[3];
    accz = ((unsigned int) rawdata[4] << 8) + rawdata[5];
    theta = atan2(-accx, accy) * 57.2957795 + 180;
    thetaraw = theta;
    theta += thetaoffset;
    if (theta < 0.00) theta += 360.0;
    if (theta >= 360.0) theta -= 360.0;
    phi = fabs(atan2(accy, -accz)) * 57.2957795;
    rho = fabs(atan2(accz, accx)) * 57.2957795;
    if ((phi >= 45.0 && phi <= 135.0) || (rho <= 45.0 || rho >= 135.0)) {
        if (percentmode) dispDec(100.0 * tan(theta / 57.2958), 1);
        else dispDec(theta, 0);
    } else dispDot();

    ADCON0bits.GO = 1;
    while (ADCON0bits.GO);
    if (ADRES > battlevel) dispBatt();
}

void accInt2isr() {
    if (accGet(INT_SRC_REG) & 0x80) {       // Wake or sleep transition
        if (accGet(SYSMOD) & 0x02) {        // Sleep
            INTCONbits.GIEH = 0;            // disable all interrupts while asleep
            INTCON3bits.INT1IE = 0;         // Disallow data updates to wake me
            INTCONbits.RABIE = 0;           // Disallow Cal to wake me
            dispOff();
            _asm
            sleep
            nop
            nop
            reset
            _endasm
        }
    }
}

void accCalisr() {
    unsigned char temp;

    if (CAL_PIN == 0) {                     // Pressed
        TMR0H = 0;                          // start timer
        TMR0L = 0;
        T0CONbits.TMR0ON = 1;
        INTCONbits.TMR0IE = 1;
        INTCONbits.TMR0IF = 0;
    } else {                                // Released
        temp = TMR0L;
        temp = TMR0H;
        T0CONbits.TMR0ON = 0;               // stop timer
        INTCONbits.TMR0IE = 0;
        if (temp < 2) return;               // less that 30 ms -> ignore
        if (temp < 122) {                   // less than 2sec -> cal
            if (thetaraw < 45.0) thetaoffset = -thetaraw;
            else if (thetaraw > 315.0) thetaoffset = 360.0 - thetaraw;
            eeWriteFloat(CAL_ADDRESS, thetaoffset);
        }
    }
    temp = CAL_PIN;
    INTCONbits.RABIF = 0;
}

void accT0isr(void) {
    percentmode = eeReadChar(MODE_ADDRESS);
    percentmode = percentmode == 0 ? 1 : 0;
    eeWriteChar(MODE_ADDRESS, percentmode);
    T0CONbits.TMR0ON = 0;                   // stop timer
    INTCONbits.TMR0IE = 0;
}
