
/*
  This is the software for the Silicon Chip DAC with
  the updated analog board based on the CS4398 IC.

             1 RESET     PC5 28---> ZERO In
  BLED0 <--- 2 PD0       PC4 27---> Reset (active low)
  BLED1 <--- 3 PD1       PC3 26---> MC
  BLED2 <--- 4 PD2       PC2 25---> MD
  AuLED <--- 5 PD3   M   PC1 24---> MS
  SiLED <--- 6 PD4   e   PC0 23---> 3.3V Monitor
             7 VCC   g   GND 22
             8 GND   a  AREF 21---> cap
  IR In <--- 9 PB6   4  AVCC 20
  Btn1  <---10 PB7   8   PB5 19---> AUDIO In
  Btn2  <---11 PD5       PB4 18---> FSOUT0 In
  Btn0  <---12 PD6       PB3 17---> FSOUT1 In
  Mux1  <---13 PD7       PB2 16---> EMPH In
  Mux0  <---14 PB0       PB1 15---> ERROR In

  fuses: CKDIV8=0 SUT=00 CKSEL=0010 BODLEVEL=100
  hfuse=11001100 (0xcc) lfuse=01000010 (0x42)
*/


#define F_CPU 1000000UL
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include <string.h>

#define ALL_LEDS_AND_MUX_ENA()  DDRD  |=   _BV(PD0)|_BV(PD1)|_BV(PD2)|_BV(PD3)|_BV(PD4)|_BV(PD7)
#define CHANNEL_LEDS_ENA()      DDRD  |=  _BV(PD0)|_BV(PD1)|_BV(PD2)
#define CHANNEL_LEDS_OFF()      PORTD |=  _BV(PD0)|_BV(PD1)|_BV(PD2)
#define CHANNEL_LEDS_ON()       PORTD &= ~(_BV(PD0)|_BV(PD1)|_BV(PD2))
#define CHANNEL_LED_0_ENA()     DDRD  |=  _BV(PD0)
#define CHANNEL_LED_0_OFF()     PORTD |=  _BV(PD0)
#define CHANNEL_LED_0_ON()      PORTD &= ~_BV(PD0)
#define CHANNEL_LED_1_ENA()     DDRD  |=  _BV(PD1)
#define CHANNEL_LED_1_OFF()     PORTD |=  _BV(PD1)
#define CHANNEL_LED_1_ON()      PORTD &= ~_BV(PD1)
#define CHANNEL_LED_2_ENA()     DDRD  |=  _BV(PD2)
#define CHANNEL_LED_2_OFF()     PORTD |=  _BV(PD2)
#define CHANNEL_LED_2_ON()      PORTD &= ~_BV(PD2)
#define AUDIO_SPDIF_LEDS_ENA()  DDRD  |= (_BV(PD3)|_BV(PD4))
#define AUDIO_SPDIF_LEDS_OFF()  PORTD &= ~(_BV(PD3)|_BV(PD4))
#define   DATA_LED_ENA()        DDRD  |=  _BV(PD4)
#define   DATA_LED_OFF()        PORTD &= ~_BV(PD4)
#define   DATA_LED_ON()         PORTD |=  _BV(PD4)
#define  SPDIF_LED_ENA()        DDRD  |=  _BV(PD3)
#define  SPDIF_LED_OFF()        PORTD &= ~_BV(PD3)
#define  SPDIF_LED_ON()         PORTD |=  _BV(PD3)
#define CHANNEL_BTN_2_IN()     (PIND&_BV(PD6))
#define CHANNEL_BTN_1_IN()     (PIND&_BV(PD5))
#define CHANNEL_BTN_0_IN()     (PINB&_BV(PB7))

#define IR_IN()                 (PINB&_BV(PB6))

#define MUX_ENA()               DDRB |= _BV(PB0)
#define SET_MUX(x)              if( (x)&1 ) PORTB |= _BV(PB0); else PORTB &= ~_BV(PB0); if( (x)&2 ) PORTD |= _BV(PD7); else PORTD &= ~_BV(PD7)

#define ERROR_IN()             (PINB&_BV(PB1))
#define EMPH_IN()              (PINB&_BV(PB2))
#define FSOUT_IN()           (((PINB&_BV(PB4))?1:0)|((PINB&_BV(PB3))?2:0))
#define AUDIO_IN()           (!(PINB&_BV(PB5)))
#define ZERO_IN()           (!!(PINC&_BV(PC5)))

#define MC_ENA()                DDRC  |=  _BV(PC3)
#define MC_HI()                 PORTC |=  _BV(PC3)
#define MC_LO()                 PORTC &= ~_BV(PC3)
#define MD_ENA()                DDRC  |=  _BV(PC2)
#define MD_HI()                 PORTC |=  _BV(PC2)
#define MD_LO()                 PORTC &= ~_BV(PC2)
#define MS_ENA()                DDRC  |=  _BV(PC1)
#define MS_HI()                 PORTC |=  _BV(PC1)
#define MS_LO()                 PORTC &= ~_BV(PC1)
#define MC_MD_MS_ENA()          DDRC  |=  _BV(PC3)

#define RESET_ENA()             DDRC  |=  _BV(PC4)
#define RESET_ON()              PORTC &= ~_BV(PC4)
#define RESET_OFF()             PORTC |=  _BV(PC4)

#define PWRSENSE_ADC_CHAN       0

#define IR_IN_PCMSK             PCMSK0
#define IR_IN_PCMSK_PIN         PCINT6
#define IR_IN_PCIE              PCIE0
#define IR_IN_VECT              PCINT0_vect
#define IR_CFG_INTERRUPT()      sei(); IR_IN_PCMSK = _BV(IR_IN_PCMSK_PIN); TCCR1A = 0; TCCR1B = _BV(CS10)
#define IR_ENA_INTERRUPT()      PCICR = _BV(IR_IN_PCIE)
#define IR_DIS_INTERRUPT()      PCICR = 0
#define IR_READ()              (PINB&_BV(PB6))

/* Set ADC divider to 8 (ADPS1, ADPS0), 1MHz/8 = 125kHz which is <200kHz maximum frequency */
/* Set voltage reference to AVCC (REFS0) */
static inline void sampleADC() {
  ADCSRA = (1 << ADEN)|(1 << ADPS1)|(1 << ADPS0)|(1 << ADSC);
  loop_until_bit_is_set(ADCSRA, ADIF);
}

void delay_1_ms() {
  _delay_ms(1);
}

static inline unsigned short readADC(unsigned char channel) {
  ADCSRA = _BV(ADEN);

  if( ADMUX != (channel|(1<<REFS0)) ) {
    ADMUX = channel|(1<<REFS0);
    sampleADC();
    delay_1_ms();
    ADCSRA = _BV(ADEN);
  }
  sampleADC();
  return ADCL|(ADCH<<8);
}

static inline unsigned char is_power_good() {
  unsigned short adc_val = readADC(PWRSENSE_ADC_CHAN);
  /* We're trying to determine when VDD has risen high enough to successfully reset the DIR9001.
     This needs to be at least 2.7V. VDD is nominally 3.3V. However, the LM3940IT-3.3 specifies
     an output range of 3.13V - 3.47V over the full temperature range. Also, we're using the
     nominally 5V VCC as the reference voltage, but the 7805 regulator outputs 4.8V - 5.2V.
     So, ignore ADC inaccuracies, worst case once VDD has stabilised we should read
     3.13 * 1024 / 5.2 = 616. At the other extreme once VDD reaches 2.7V we should read no more
     than 2.7 * 1024 / 4.8 = 598. So, let's pick 607 as it is halfway between these two worst
     case values and that should be enough room to work around ADC inaccuracy. */
  return adc_val >= 607;
}

static void dac_register_write(unsigned char reg, unsigned char value) {
  unsigned char data[3];
  unsigned char* ptr;
  unsigned char bits;

  data[0] = 0x98 /* chip address & write bit */;
  data[1] = reg;
  data[2] = value;
  ptr = data;
  bits = sizeof(data)*8;

  MS_LO();

  while( bits-- ) {
    if( ptr[0]&0x80 )
      MD_HI();
    else
      MD_LO();

    if( bits&7 )
      ptr[0] <<= 1;
    else
      ++ptr;

    MC_LO();
    asm volatile("nop");
    MC_HI();
    asm volatile("nop");
  }

  MS_HI();
  MD_HI();
}

union {
  unsigned long l;
  unsigned short s;
  unsigned char c;
} timer;

ISR(TIMER0_COMPA_vect) {
  ++timer.l;
}

///////////////////////////////////////////////////////////////////////
// IR

unsigned short rc5_code, rc5_timing;
volatile unsigned short rc5_final_code;
volatile unsigned char rc5_bit, rc5_reset_timer, g_ir_leds;

static void cancel_rc5_reception() {
  TIMSK1 &= ~_BV(TOIE1);
  rc5_bit = 0;
}

unsigned char g_ir_timer;
ISR(TIMER1_OVF_vect) {
  if( rc5_bit == 1 || rc5_reset_timer ) {
    cancel_rc5_reception();
  } else {
    unsigned char level = IR_READ();
    rc5_code <<= 1;
    rc5_code |= !level;
    if( ++rc5_bit == 15 ) {
      rc5_final_code = (rc5_code&0x7F00)|((rc5_code>>1)&0x00FF)|0x8000;
      cancel_rc5_reception();
      if( g_ir_leds )
        DATA_LED_ON();
      else
        g_ir_timer = 24;
    } else {
      TCNT1 = 0 - (rc5_timing>>1);
      rc5_reset_timer = 1;
    }
  }
}

#define RC5_PERIOD (2*0.00088889)
#define RC5_CYCLES (F_CPU*RC5_PERIOD)
#define RC5_MIN    (RC5_CYCLES / 1.2)
#define RC5_MAX    (RC5_CYCLES * 1.2)

ISR(IR_IN_VECT) {
  if( rc5_bit == 0 ) {
    if( IR_READ() ) {
      // start reception
      TCNT1 = 0 - (unsigned short)RC5_MAX;
      TIFR1 |= _BV(TOV1);
      TIMSK1 |= _BV(TOIE1);
      rc5_reset_timer = 0;
      rc5_bit = 1;
      rc5_code = 0;
      if( g_ir_leds ) {
        DATA_LED_OFF();
      }
    }
  } else if( rc5_bit == 1 ) {
    if( IR_READ() ) {
      rc5_timing = TCNT1;
      cancel_rc5_reception();
      rc5_timing += (unsigned short)RC5_MAX;
      if( rc5_timing >= (unsigned short)RC5_MIN ) {
        TCNT1 = 0 - (rc5_timing>>1) - (rc5_timing>>2);
        TIMSK1 |= _BV(TOIE1);
        rc5_bit = 2;
      } else {
        cancel_rc5_reception();
      }
    }
  } else if( rc5_reset_timer ) {
    TCNT1 = 0 - (rc5_timing>>1) - (rc5_timing>>2);
    rc5_reset_timer = 0;
  }
  if( g_ir_leds ) {
    if( IR_READ() )
      SPDIF_LED_ON();
    else
      SPDIF_LED_OFF();
  }
}

///////////////////////////////////////////////////////////////////////

#define NUM_IR_CODES  8
#define IR_CODE_CHAN0 0
#define IR_CODE_CHAN1 1
#define IR_CODE_CHAN2 2
#define IR_CODE_NEXT  3
#define IR_CODE_PREV  4
#define IR_CODE_MUTE  5
#define IR_CODE_VOLUP 6
#define IR_CODE_VOLDN 7

#define RC5_CODE_0      0x000
#define RC5_CODE_1      0x001
#define RC5_CODE_2      0x002
#define RC5_CODE_3      0x003
#define RC5_CODE_POWER  0x00C
#define RC5_CODE_MUTE   0x00D
#define RC5_CODE_VOLUP  0x010
#define RC5_CODE_VOLDN  0x011
#define RC5_CODE_CHANUP 0x020
#define RC5_CODE_CHANDN 0x021

unsigned short g_codes[NUM_IR_CODES] = { RC5_CODE_1, RC5_CODE_2, RC5_CODE_3, RC5_CODE_CHANUP, RC5_CODE_CHANDN, RC5_CODE_MUTE, RC5_CODE_VOLUP, RC5_CODE_VOLDN };

static void set_on_led(unsigned char which) {
  if( which == 0 )
    CHANNEL_LED_0_ON();
  else
    CHANNEL_LED_0_OFF();
  if( which == 1 )
    CHANNEL_LED_1_ON();
  else
    CHANNEL_LED_1_OFF();
  if( which == 2 )
    CHANNEL_LED_2_ON();
  else
    CHANNEL_LED_2_OFF();
}

static unsigned short setup_ir_codes() {
  unsigned short codes[NUM_IR_CODES];
  unsigned short receive_window[8];
  unsigned short ret = 0;
  unsigned char i, window_len, last_ir_timer, waiting, which;

  SPDIF_LED_OFF();
  DATA_LED_OFF();
  g_ir_leds = 1;

  which = 0;
  for( i = 0; i < NUM_IR_CODES; ++i ) {
    window_len = 0;
    waiting = 1;
    last_ir_timer = timer.c;

    while( CHANNEL_BTN_0_IN() || CHANNEL_BTN_1_IN() || CHANNEL_BTN_2_IN() ) {
      asm volatile("wdr");
    }

    while(1) {
      unsigned short temp = rc5_final_code;
      asm volatile("wdr");
      if( temp ) {
        rc5_final_code = 0;
        last_ir_timer = timer.c;
        if( !waiting ) {
          memmove(receive_window, receive_window+1, 7 * sizeof(unsigned short));
          receive_window[7] = temp;
          if( window_len < 8 ) {
            ++window_len;
          } else {
            unsigned char j;
            for( j = 0; j < 7; ++j )
              if( receive_window[i] != temp )
                break;
            if( j == 7 ) {
              codes[i] = temp&0xFFF;
              if( i == NUM_IR_CODES-1 )
                ret = temp;
              break;
            }
          }
        }
      } else if( waiting && (timer.c - last_ir_timer) >= 64 ) {
        waiting = 0;
        DATA_LED_OFF();
      }
      set_on_led( (timer.c&16) ? which : 3 );
      if( CHANNEL_BTN_0_IN() ) {
        codes[i] = 0xFFFF;
        break;
      } else if( CHANNEL_BTN_1_IN() ) {
        if( i > 0 ) {
          i -= 2;
          break;
        }
      } else if( CHANNEL_BTN_2_IN() ) {
        g_ir_leds = 0;
        return 0;
      }
    }
    if( ++which == 3 )
      which = 0;
  }
  for( i = 0; i < NUM_IR_CODES; ++i ) {
    asm volatile("wdr");
    eeprom_write_word((uint16_t*)(i<<1), codes[i]);
    g_codes[i] = codes[i];
  }
  g_ir_leds = 0;
  return ret;
}

static void load_ir_codes() {
  unsigned char i;
  for( i = 0; i < NUM_IR_CODES; ++i ) {
    unsigned short code = eeprom_read_word((uint16_t*)(i<<1));
    if( code != 0xFFFF )
      g_codes[i] = code;
  }
}

static void set_mux(unsigned char channel) {
  if( channel == 0 ) {
    SET_MUX(1);
  } else if( channel == 1 ) {
    SET_MUX(2);
  } else {
    SET_MUX(0);
  }
}

static const unsigned char deemphasis_map[4] = { 1<<2, 2<<2, 0<<2, 3<<2 };

static unsigned short used_reg_map = (1<<2)|(1<<3)|(1<<4)|(1<<5)|(1<<7)|(1<<8);
unsigned char current_reg_state[10];
static void setup_DAC(unsigned char muted, unsigned char deemphasis, unsigned char sample_rate, unsigned char volume) {
  unsigned char regs[10];
  unsigned char i;

  // sample_rate, 0 = 44.1kHz, 1 = 48kHz, 2 = out of range, 3 = 32kHz
  regs[2] = 0x10|(sample_rate == 2 ? 1 : 0)|(deemphasis ? deemphasis_map[sample_rate] : 0);
  regs[3] = 0x89;
  regs[4] = 0xC0 | (muted ? 0x18 : 0x00);
  regs[5] = 0xFF - volume;
  regs[7] = 0x70; // zero cross muting/volume changing (0xb0 for soft ramp, the default)
  regs[8] = 0x40 | (sample_rate == 2 ? 0x10 : 0x00) | (current_reg_state[8] == 0 ? 0x80 : 0x00);
  for( i = 9; i != (unsigned char)-1; --i ) {
    if( (used_reg_map&((unsigned short)1<<i)) && regs[i] != current_reg_state[i] ) {
      dac_register_write(i, regs[i]);
      current_reg_state[i] = regs[i];
    }
  }
}

static unsigned char wait_for_button(unsigned char flash_which) {
  unsigned char ret;
  while(1) {
    asm volatile("wdr");
    if( CHANNEL_BTN_0_IN() ) {
      ret = 0;
      break;
    } else if( CHANNEL_BTN_1_IN() ) {
      ret = 1;
      break;
    } else if( CHANNEL_BTN_2_IN() ) {
      ret = 2;
      break;
    }
    set_on_led( (timer.c&16) ? flash_which : 3 );
  }
  while( CHANNEL_BTN_0_IN() || CHANNEL_BTN_1_IN() || CHANNEL_BTN_2_IN() ) {
    asm volatile("wdr");
  }
  return ret;
}

#define NUM_DELAYS 4
unsigned long delays[NUM_DELAYS] = { 10 * 100, 60 * 100, 5 * 60 * 100, 0 };
unsigned char startup_behaviour;
#define DELAY_AUTO_NO_SIGNAL   0
#define DELAY_AUTO_NO_AUDIO    1
#define DELAY_MANUAL_NO_SIGNAL 2
#define DELAY_MANUAL_NO_AUDIO  3

unsigned char store_last_channel, last_channel_stored;
static unsigned short setup_config(unsigned short which) {
  while(1) {
    asm volatile("wdr");
    if( CHANNEL_BTN_0_IN() && CHANNEL_BTN_1_IN() && CHANNEL_BTN_2_IN() ) {
      return setup_ir_codes();
    } else if( !CHANNEL_BTN_0_IN() && !CHANNEL_BTN_1_IN() && !CHANNEL_BTN_2_IN() ) {
      break;
    }
  }

  /*
    which = 0: set no signal timeout for auto-selected channels
    which = 1: set no audio  timeout for auto-selected channels
    which = 2: set no signal timeout for manually-selected channels
    which = 3: set no audio  timeout for manually-selected channels
    which = 4: set default channel on startup
    which = 5: set startup scan behaviour
  */
  if( which < 4 ) {
    /*
      0 => 10*msec    1 => sec    2 => mins
      0, 0 => 1       0, 1 => 2   0, 2 => 3
      1, 0 => 5       1, 1 => 10  1, 2 => 20
      2, 0 => 30      2, 1 => 40  2, 2 => 50
    */
    unsigned char mult;
    unsigned long val;
    static const unsigned short options[9] = { 1, 2, 3, 5, 10, 20, 30, 40, 50 };
    mult = wait_for_button(0);
    val = options[wait_for_button(1)*3+wait_for_button(2)];
    switch(mult) {
    case 0:
      break;
    case 1:
      val *= 100;
      break;
    case 2:
      val *= 60*100;
      break;
    }
    delays[which] = val;
    asm volatile("wdr");
    eeprom_write_dword((uint32_t*)(NUM_IR_CODES*2+(which<<2)), delays[which]);
  } else {
    if( which == 5 ) {
      // startup behaviour
      // 0 => immediately scan
      // 1 => scan after auto scan delay
      // 2 => scan after manual press delay
      startup_behaviour = (startup_behaviour&0xFC)|wait_for_button(0);
    } else {
      // set default channel on startup
      // 0 => remember last   1 => preset button (following press)
      unsigned char selection = wait_for_button(0);
      if( selection != 0 ) {
        selection = wait_for_button(1)+1;
      }
      startup_behaviour = (startup_behaviour&0xF3)|(selection<<2);
      store_last_channel = (selection == 0);
    }
    asm volatile("wdr");
    eeprom_write_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4), startup_behaviour);
  }
  return 0;
}

static void load_delays() {
  unsigned char i;
  for( i = 0; i < NUM_DELAYS; ++i ) {
    asm volatile("wdr");
    unsigned long delay = eeprom_read_dword((uint32_t*)(NUM_IR_CODES*2+(i<<2)));
    if( delay != 0xFFFFFFFF )
      delays[i] = delay;
  }
}

static unsigned char time_exceeded(unsigned long time, unsigned long last_time, unsigned long delay) {
  if( delay <= 1 )
    return 0;
  return time - last_time >= delay;
}

void main(void) __attribute((noreturn));

void main(void) {
  unsigned char running = 0;
  unsigned char cur_channel;
  unsigned short last_ir_code = 0;
  unsigned char ir_code_timeout = 0;
  unsigned char muted = 0;
  unsigned char button_depressed = 0;
  unsigned char scanning = 0, scan_timer = 0, manually_selected = 0;
  unsigned long last_signal_time = 0, last_audio_time = 0;
  unsigned char volume;
  unsigned char on_hold = 0, hold_timer = 0;
  unsigned char volume_memory_timer = 0;
  unsigned char last_speed = 0, this_speed, mute;
  unsigned char mute_timer = 0, speed_reset_timer = 0, scan_ignore = 130;


  asm volatile("wdr");
  WDTCSR |= _BV(WDCE);
  WDTCSR = _BV(WDE);

  DIDR0 = 1;

  ALL_LEDS_AND_MUX_ENA();
  CHANNEL_LEDS_OFF();
  MUX_ENA();
  SET_MUX(1);

  MS_ENA();
  MS_HI();
  MC_ENA();
  MC_HI();
  MD_ENA();
  MD_HI();

  RESET_ENA();
  RESET_ON();
  delay_1_ms();

  load_ir_codes();
  load_delays();

  IR_CFG_INTERRUPT();
  IR_ENA_INTERRUPT();

  TCCR0A = _BV(WGM01);
  TCCR0B = _BV(CS01)|_BV(CS00);
  OCR0A = 156;
  TIMSK0 = _BV(OCIE0A);

  startup_behaviour = eeprom_read_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4));
  if( startup_behaviour == 0xFF )
    startup_behaviour = 5;
  switch( startup_behaviour&3 ) {
  case 0:
    scanning = 1;
    scan_timer = timer.c+30;
    last_signal_time = last_audio_time = 0x7FFFFFFF;
    break;
  case 2:
    manually_selected = 1;
    break;
  }

  switch(startup_behaviour&12) {
  case 0:
    store_last_channel = 1;
    cur_channel = eeprom_read_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4+1));
    if( cur_channel > 2 )
      cur_channel = 0;
    last_channel_stored = cur_channel;
    break;
  default:
    cur_channel = ((startup_behaviour>>2)&3)-1;
    break;
  }
  volume = eeprom_read_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4+2));
  set_mux(cur_channel);

  while(1) {
    asm volatile("wdr");
    if( is_power_good() ) {
      unsigned char chan_changed = 0;
      unsigned char btn1, btn2, btn3;

      delay_1_ms();
      if( !running ) {
        RESET_OFF();
        running = 1;
        if( !speed_reset_timer )
          mute_timer = 100;
        _delay_ms(0.01);
      }

      mute = muted || mute_timer || !AUDIO_IN() || scan_ignore > 130 || (scanning && scan_timer - timer.c < 2);
      if( !mute ) {
        this_speed = FSOUT_IN() == 2 ? 2 : 1;
        if( this_speed != last_speed ) {
          if( last_speed != 0 && !speed_reset_timer ) {
            speed_reset_timer = 25;
            goto do_reset;
          } else {
            last_speed = this_speed;
          }
        }
      }
      setup_DAC(mute, EMPH_IN(), FSOUT_IN(), volume);
      if( mute_timer )
        --mute_timer;
      else if( speed_reset_timer )
        --speed_reset_timer;
      if( scan_ignore )
        --scan_ignore;

      btn1 = CHANNEL_BTN_0_IN();
      btn2 = CHANNEL_BTN_1_IN();
      btn3 = CHANNEL_BTN_2_IN();
      if( btn1 && button_depressed != 1 ) {
        if( button_depressed != 0 ) {
          last_ir_code = setup_config(button_depressed == 3 ? 4 : 2);
          ir_code_timeout = 0;
          continue;
        }
        cur_channel = 0;
button_pressed:
        button_depressed = cur_channel+1;
        manually_selected = 1;
        chan_changed = 1;
        on_hold = 0;
        hold_timer = timer.c + 64;
      }
      else if( btn2 && button_depressed != 2 ) {
        if( button_depressed != 0 ) {
          last_ir_code = setup_config(button_depressed == 1 ? 0 : 5);
          ir_code_timeout = 0;
          continue;
        }
        cur_channel = 1;
        goto button_pressed;
      }
      else if( btn3 && button_depressed != 3 ) {
        if( button_depressed != 0 ) {
          last_ir_code = setup_config(button_depressed == 1 ? 1 : 3);
          ir_code_timeout = 0;
          continue;
        }
        cur_channel = 2;
        goto button_pressed;
      } else {
        if( btn1 ) {
          if( cur_channel == 0 && hold_timer == timer.c )
            on_hold = 1;
        } else if( btn2 ) {
          if( cur_channel == 1 && hold_timer == timer.c )
            on_hold = 1;
        } else if( btn3 ) {
          if( cur_channel == 2 && hold_timer == timer.c )
            on_hold = 1;
        } else {
          button_depressed = 0;
        }
      }

      unsigned short temp = rc5_final_code;
      if( temp ) {
        rc5_final_code = 0;
        unsigned char i;
        for( i = 0; i < NUM_IR_CODES; ++i ) {
          if( !(g_codes[i]&0x8000) ) {
            if( (temp&0xFFF) == g_codes[i] ) {
              // don't allow holding down of buttons except for volume control
              if( i < IR_CODE_VOLUP && temp == last_ir_code )
                continue;

              switch(i) {
              case IR_CODE_CHAN0:
                cur_channel = 0;
                manually_selected = 1;
                chan_changed = 1;
                break;
              case IR_CODE_CHAN1:
                cur_channel = 1;
                manually_selected = 1;
                chan_changed = 1;
                break;
              case IR_CODE_CHAN2:
                cur_channel = 2;
                manually_selected = 1;
                chan_changed = 1;
                break;
              case IR_CODE_NEXT:
                if( ++cur_channel == 3 )
                  cur_channel = 0;
                manually_selected = 1;
                chan_changed = 1;
                break;
              case IR_CODE_PREV:
                if( --cur_channel == 0xFF )
                  cur_channel = 2;
                manually_selected = 1;
                chan_changed = 1;
                break;
              case IR_CODE_MUTE:
                muted = !muted;
                break;
              case IR_CODE_VOLUP:
                if( volume < 255 ) {
                  ++volume;
                  volume_memory_timer = 255;
                }
                break;
              case IR_CODE_VOLDN:
                if( volume > 195 ) {
                  --volume;
                  volume_memory_timer = 255;
                }
                break;
              }
            }
          }
        }
        if( temp != last_ir_code ) {
          last_ir_code = temp;
          ir_code_timeout = 0;
        }
      } else if( ++ir_code_timeout == 0 ) {
        last_ir_code = 0;
      }
      if( chan_changed ) {
        set_mux(cur_channel);
        muted = 0;
        cli();
        last_signal_time = last_audio_time = timer.l;
        sei();
        scanning = 0;
      }

      if( muted ) {
        if( timer.c&16 ) {
          CHANNEL_LEDS_ON();
        } else {
          CHANNEL_LEDS_OFF();
        }
      } else {
        set_on_led( on_hold && !(timer.c&240) ? 3 : cur_channel );
      }
      if( ERROR_IN() ) {
        unsigned long time;

        if( g_ir_timer ) {
          --g_ir_timer;
          SPDIF_LED_ON();
        } else {
          SPDIF_LED_OFF();
        }
        DATA_LED_OFF();
        cli();
        time = timer.l;
        sei();
        if( !muted &&
            (time_exceeded(time, last_signal_time, delays[manually_selected ? DELAY_MANUAL_NO_SIGNAL : DELAY_AUTO_NO_SIGNAL]) ||
             time_exceeded(time, last_audio_time,  delays[manually_selected ? DELAY_MANUAL_NO_AUDIO  : DELAY_AUTO_NO_AUDIO ])) ) {
          if( !scanning && !on_hold )
            goto scan_now;
        } else {
          scanning = 0;
        }
      } else if( ZERO_IN() || scan_ignore ) {
        unsigned long time;

        cli();
        time = timer.l;
        sei();
        last_signal_time = time;
        if( g_ir_timer ) {
          --g_ir_timer;
          SPDIF_LED_OFF();
        } else {
          SPDIF_LED_ON();
        }
        DATA_LED_OFF();
        if( !muted &&
            time_exceeded(time, last_audio_time,   delays[manually_selected ? DELAY_MANUAL_NO_AUDIO  : DELAY_AUTO_NO_AUDIO ]) ) {
          if( !scanning && !on_hold )
            goto scan_now;
        } else {
          scanning = 0;
        }
      } else {
        unsigned long time;
        unsigned char deemphasis_active = EMPH_IN() && FSOUT_IN() != 2;

        cli();
        time = timer.l;
        sei();
        last_signal_time = last_audio_time = time;

        if( g_ir_timer ) {
          --g_ir_timer;
          if( deemphasis_active ) {
            SPDIF_LED_OFF();
          } else {
            SPDIF_LED_ON();
          }
        } else {
          if( deemphasis_active ) {
            SPDIF_LED_ON();
          } else {
            SPDIF_LED_OFF();
          }
        }
        DATA_LED_ON();
        scanning = 0;
      }
      if( scanning && scan_timer == timer.c ) {
      scan_now:
        scanning = 1;
        scan_ignore = 140;
        scan_timer = timer.c+30;
        if( ++cur_channel == 3 )
          cur_channel = 0;
        manually_selected = 0;
        set_mux(cur_channel);
        muted = 0;
      }
      if( !scanning && store_last_channel && last_channel_stored != cur_channel ) {
        last_channel_stored = cur_channel;
        asm volatile("wdr");
        eeprom_write_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4+1), cur_channel);
      }
    } else {
      if( running ) {
        CHANNEL_LEDS_OFF();
        AUDIO_SPDIF_LEDS_OFF();
        SET_MUX(0);
do_reset:
        RESET_ON();
        memset(current_reg_state, 0, sizeof(current_reg_state));
        delay_1_ms();
        running = 0;
      }
    }

    if( volume_memory_timer && !--volume_memory_timer ) {
      asm volatile("wdr");
      eeprom_write_byte((uint8_t*)(NUM_IR_CODES*2+NUM_DELAYS*4+2), volume);
    }
  }
}
