;	************************************************************
;	DiscreteSemiTester.asm - firmware for PIC-based
;	Discrete Semiconductor Test Set
;
;	Written by Jim Rowe for Silicon Chip, using a PIC16F877A
;	processor running from an external crystal at 8MHz,
;	so one machine cycle (mc) = 0.5us. (Version 1.4)

;	Program last revised 05/04/2012 @ 3:40pm
;
;	This version of the program provides tests for bipolar junction
;	transistors (BJTs, both NPN and PNP), diodes (including zener,
;	Schottky & LEDs), MOSFETs (depletion mode, N-channel & P-channel),
;	SCRs and PUTs. It allows a choice of four operating voltages for
;	testing (10V, 25V, 50V and 100V), plus a fifth voltage (600V) for
;	breakdown/avalanche voltage testing. 
;
;	Note: Program makes use of 24-bit and 32-bit floating point &
;	fixed point maths routines for Microchip Technology Inc's
; 	8-bit PIC processors, written by Frank J. Testa and
;	described in MTI's App Notes AN575, AN617, AN660 and AN670,
; 	downloadable from the MTI website at www.microchip.com
;	(Routines used here are all in FPRF24.TXT)
;
;	**********************************************************
; 
;	CPU configuration
;

	list P=16F877A
	#include "p16f877A.inc"
	__config h'3F3A'	; HS osc selected, WDT disabled, PWRT disabled,
						; BOR disabled, RB3 is digital I/O, code prot off,
						; Flash prog mem write protection off, RB6 & RB7 are
						; general purpose I/O pins (not used for debugging)
;
;**********************************************************
;
;	First define some handy macros
;
;	Select Register Bank 0
bank0	macro
	errorlevel	+302	; Re-enable bank warning
	BCF	STATUS,RP0		; Select Bank 0
	BCF STATUS, RP1
	BCF STATUS, IRP		; and also indir addr banks 0,1
	endm

;	Select Register Bank 1
bank1	macro
	BSF	STATUS,RP0		; Select Bank 1
	BCF	STATUS,RP1
	BCF STATUS,IRP
	errorlevel	-302	; disable warning
	endm

;	Copy bytes in register file via W
copy	macro	from,to
	MOVF	from,W
	MOVWF	to
	endm

; 	convert a numeral byte in w into ASCII code (in w)
GetASCII	macro
	ANDLW h'0F'			; first mask off upper nibble, if any
	IORLW h'30'			; now make upper nibble h'30'
	endm

;	Prepare to call or goto page3 (1800-1FFFh)
Ppage3	macro
	BSF PCLATH,4		; set bits 3 & 4 of PCLATH
	BSF PCLATH,3
	endm

;	Prepare to call or goto page2 (1000-17FFh)
Ppage2	macro
	BSF PCLATH,4		; set only bit4 of PCLATH
	BCF PCLATH,3
	endm

;	Prepare to call or goto page1 (800-FFFh)
Ppage1	macro
	BCF PCLATH,4		; set only bit3 of PCLATH
	BSF PCLATH,3
	endm

;	Prepare to call or goto page0 (000-7FFh)
Ppage0	macro
	BCF PCLATH,4		; clear bits 3 & 4 of PCLATH
	BCF PCLATH,3
	endm

; 	set BARG regs to 10d, before an FP operation
; 	(10d = 82 20 00 00, precalc using Fprep)
SetBARG10	macro
	MOVLW h'82'			; exponent first
	MOVWF BEXP
	MOVLW h'20'			; then MSB/sign byte
	MOVWF BARGB0
	CLRF BARGB1			; then remaining two bytes
	CLRF BARGB2
	endm

;	Swap bytes in register file via W
swap	macro	this,that
	MOVF	this,w		; get this
	XORWF	that,f		; Swap using Microchip
	XORWF	that,w		; Tips'n Tricks
	XORWF	that,f		; #18
	MOVWF	this
	endm

;	toggle EN pin of LCD module to write an instr or data nibble
ToggleEN	macro
	BSF PORTE,0			; take LCD's EN line high (RE0)
	GOTO $+1			; pause 2us (4mc) to let it stabilise
	GOTO $+1
	BCF PORTE,0			; then low again, to write into LCD controller
	endm

;	set ADCON0 to use ADC channel AN0
UseAN0	macro
	MOVLW h'41'		; turn on ADC, make Tad = Toscx16, reset ADC
	MOVWF ADCON0	; and set AN0 as current analog input channel
	endm

;	set ADCON0 to use ADC channel AN1
UseAN1	macro
	MOVLW h'49'		; turn on ADC, make Tad = Toscx16, reset ADC
	MOVWF ADCON0	; and set AN1 as current analog input channel
	endm

;	set ADCON0 to use ADC channel AN2
UseAN2	macro
	MOVLW h'51'		; turn on ADC, make Tad = Toscx16, reset ADC
	MOVWF ADCON0	; and set AN0 as current analog input channel
	endm
;
;	**********************************************************
;     STATUS bit definitions (used in FPRF24.TXT)

#define	_C	STATUS,0
#define	_Z	STATUS,2
;
;**************************************************************
;
;	storage register declarations:

Counter1	equ	h'20'	; gp counter variable 1
Counter2	equ h'21'	; gp counter variable 2
Counter3	equ h'22'	; gp counter variable 3
Counter4	equ h'23'	; gp counter variable 4
Counter5	equ h'24'	; gp counter variable 5
MLoopCtr	equ h'25'	; measurement loop counter
Temp1		equ	h'26'	; working storage location 1
Temp2		equ h'27'	; working storage location 2
Temp3		equ h'28'	; working storage location 3
Flags1		equ h'29'	; relay flag bits 0-7 (-> PORTB)
;						bit0: RLY15 (0 = SCR, 1 = PUT)
; 						bit1: RLY9 (0 = diode FWD, 1 = diode REV)
;						bit2: RLY16 (0 = gate shorted to K or A,
;								1 = gate connected to +/-Ibias)
;						bit3: RLY4 (0 = Ibias > 20uA, 1 = Ibias > 500uA)
;						bit4: RLY3 (0 = Ibias > 20uA, 1 = Ibias > 100uA)
;						bit5: RLY5 (0 = +Ibias OFF, 1 = +Ibias ON)
;						bit6: RLY7 (0 = Idevice 0-200uA, 1 = Idevice 0-50mA)
;						bit7: RLY1 (0 = DC/DC conv OFF, 1 = DC/DC conv ON)
Flags2		equ h'2A'	; relay flag bits 8-16 (-> PORTD)
;						bit0: RLY2 (0 = 10/25/50/100V, 1 = 600V for BDV test)
;						bit1: RLY8 (0 = Vdev FSD>1000V, 1 = Vdev FSD>100V)
;						bit2: RLY10 (0 = NPN BJT, 1 = PNP BJT)
;						bit3: RLY11 (0 = BVceo/Iceo/hFE test, 1 = BVcbo/Icbo)
;						bit4: RLY12 (0 = N-ch MOSFET, 1 = P-ch MOSFET)
;						bit5: RLY14 (0 = MOSFET +Vgs/N-ch, 1 = -Vgs/P-ch)
;						bit6: RLY13 (0 = MOSFET G to S, 1 = G to +/-Vgs)
;						bit7: RLY6 (0 = -Ibias OFF, 1 = -Ibias ON)
;
;	storage for device current reading display digits	
IDig1		equ h'2B'	; first digit 
IDig2		equ h'2C'	; second digit
IDig3		equ h'2D'	; third digit
IDig4		equ h'2E'	; fourth digit
IDig5		equ h'2F'	; fifth digit (multiplier)
;
;	storage for device voltage reading display digits
VdevDig1	equ h'30'	; first digit (hundreds)
VdevDig2	equ h'31'	; second digit (tens)
VdevDig3	equ h'32'	; third digit (units)
VdevDig4	equ h'33'	; fourth digit (tenths)
;
;	storage for Vgs voltage reading display digits
VgsDig1		equ h'34'	; first digit (tens)
VgsDig2		equ h'35'	; second digit (units)
VgsDig3		equ h'36'	; third digit (tenths)

;	storage for ADC reading in 24-bit form, after calculation
ADCEXP		equ h'37'	; exponent
ADCB0		equ h'38'	; MSB & sign bit
ADCB1		equ h'39'	; LSB

;	storage for other variables 
DevNum		equ h'3A'	; device menu number (1 = Diodes, 2 = LEDs,
						; 3 = NPN BJTs, 4 = PNP BJTs, 5 = N-ch MOSFETs,
						; 6 = P-ch MOSFETs, 7 = SCRs, 8 = PUTs
TestNum		equ h'3B'	; ID number of selected test to perform
						; 01 = Diode Irev (BV), 02 = Diode Irev (OPV),
						; 03 = Diode Vforward, 04 = Diode Vrev (BV),
						; 05 = BJT BVcbo; 06 = BJT BVceo; 07 = BJT Icbo,
						; 08 = BJT Icbo, 09 = BJT hFE(Ib=50uA),
						; 0A = BJT hFE(Ib=200uA), 0B = BJT hFE(Ib=1mA),
						; 0C = MOSFET BVdss, 0D = MOSFET Idss (OPV),
						; 0E = MOSFET Id v. Vgs (gm)
						; 0F = SCR BVaks (g-k shorted)
						; 10 = SCR or PUT Iaks (OPV)
						; 11 = SCR or PUT Iak(Ig=50uA)
						; 12 = SCR or PUT Iak(Ig=200uA)
						; 13 = SCR or PUT Iak(Ig=1mA)
						; 14 = SCR or PUT Vak ON (OPV)
						; 15 = PUT BVaks (g-a shorted)
TNummin		equ h'3C'	; lowest TestNum for selected device
TNummax		equ h'3D'	; highest TestNum for selected device
Keycode		equ h'3E'	; valid button press code stored here
Devcode		equ h'3F'	; code for current device to be tested

;	regs for storage of Byte10 and Byte1 (output bytes of ByteSplit routine)
Byte10		equ h'40'	; decimal tens output
Byte1		equ h'41'	; decimal units output

;	regs for storage of variables used by Bin16toBCD conversion routine
Count		equ h'42'	; counter
Temp		equ h'43'	; temp storage
H_byte		equ h'44'	; higher input byte
L_byte		equ h'45'	; lower input byte
R0			equ h'46'	; storage for output MSD (in rightmost nibble)
R1			equ h'47'	; storage for next two output digit nibbles
R2			equ h'48'	; storage for last two output digit nibbles

Ones		equ h'49'	; storage for least signif BCD digit in ASCII
Tens		equ h'4A'	; storage for next signif BCD digit in ASCII
Hundreds	equ h'4B'	; and the next digit
Thousands	equ h'4C'	; and the next digit

;	retained only to keep assembler happy (float_ascii no longer used)
digit_count	equ h'4D'	; digit counter used by float_ascii
tenthous	equ h'4E'	; also needed by float_ascii

;	Display string char offset pointer
DSpointer	equ h'4F'

;	***********************************************************
;   Floating Point Stack & other locations used by FPRF24.TXT
;	routines
;
AARGB7	equ h'50'	; AARGB7 byte for FP argument A
AARGB6	equ h'51'	; AARGB6 byte
AARGB5	equ h'52'	; AARGB5 byte
AARGB4	equ	h'53'	; AARGB4 byte
AARGB3	equ h'54'	; AARGB3
AARGB2	equ h'55'	; AARGB2
AARGB1	equ h'56'	; AARGB1
AARGB0	equ h'57'	; AARGB0
AEXP	equ h'58'	; 8 bit biased exponent for argument A

BARGB3	equ h'59'	; BARGB3 byte for argument B
BARGB2	equ h'5A'	; BARGB2
BARGB1	equ h'5B'	; BARGB1
BARGB0	equ h'5C'	; BARGB0
BEXP	equ h'5D'	; 8 bit biased exponent for argument B

TEMPB3	equ h'5E'	; TEMPB3 byte
TEMPB2	equ h'5F'	; TEMPB2 byte
TEMPB1	equ h'60'	; TEMPB1 byte
TEMPB0	equ h'61'	; TEMPB0 byte

CARGB1	equ h'62'	; CARGB1 byte for argument C
CARGB0	equ h'63'	; CARGB0 byte
CEXP	equ h'64'	; 8 bit biased exponent for argument C

DARGB3	equ h'65'	; DARGB3 byte for argument D
DARGB2	equ h'66'	; DARGB2 byte
DARGB1	equ h'67'	; DARGB1 byte
DARGB0	equ	h'68'	; DARGB0 byte
DEXP	equ	h'69'	; 8-bit biased exponent for argument D

EARGB3	equ h'6A'	; needed by EXP1024, it seems

JARGB1	equ h'6B'	; ARGB1 byte for argument J
JARGB0	equ h'6C'	; ARGB0 byte
JEXP	equ h'6D'	; 8 bit biased exponent for argument J


SIGN	equ	h'6E'	; location for saving sign in MSB
FPFLAGS	equ h'6F'	; floating point library exception flags
FPError	equ h'70'	; floating point routine error code (FFh = error)

;	Fixed point storage locations

REMB3	equ h'71'	; remainder LSB
REMB2	equ h'72'	; remainder middle byte
REMB1	equ h'73'	; remainder upper middle byte
REMB0	equ h'74'	; remainder MSB
LOOPCOUNT	equ h'75'	; loop counter

;	storage for current shunt voltage drop in 24-bit FP form
;
KARGB1	equ	h'78'	; ARGB1 for argument K
KARGB0	equ	h'79'	; ARGB0
KEXP	equ h'7A'	; 8-bit biased exponent

;       floating point library exception flag bits
;
IOV     equ     0	; bit0 = integer overflow flag
FOV     equ     1   ; bit1 = FP overflow flag
FUN     equ     2   ; bit2 = FP underflow flag
FDZ     equ     3   ; bit3 = FP divide by zero flag
NAN		equ		4	; bit4 = not-a-number exception flag
DOM		equ		5	; bit5 = domain error exception flag
RND     equ     6   ; bit6 = FP rounding flag
					; 0 = truncation
                    ; 1 = unbiased rounding to nearest LSB
SAT     equ     7   ; bit7 = FP saturate flag
					; 0 = term on exception w/out saturation
					; 1 = term on exception with saturation
					; to appropriate value

EXP		equ	AEXP 
TEMP	equ	TEMPB0

;	define assembler constants
B0		equ	0
B1		equ	1
B2		equ	2
B3		equ	3
B4		equ	4
B5		equ	5
B6		equ	6
B7		equ	7

MSB		equ	7
LSB		equ	0

;       Floating point literal constants
;
EXPBIAS   equ h'7F'		; = 127d, bias for FP exponents

;*************************************************************

;	Program now begins
;
	org	h'00'		; normal start vector
	GOTO Initialise
	org h'04'		; interrupt service vector
	GOTO Initialise	; (interrupts not used in this prog)

	org h'0008'		; program proper begins here
Initialise:
	; first we set up CPU and INTOSC, I/O ports and ADC module
	Ppage0			; first ensure PCLATH is set for page0
	bank0			; also make sure we're set for bank 0
	CLRF PORTA		; then clear PORTA
	CLRF PORTB		; also PORTB, C, D and E
	CLRF PORTC
	CLRF PORTD
	CLRF PORTE
	CLRF INTCON		; disable interrupts
	CLRF RCSTA		; and also the serial port
	CLRF T1CON		; and timer 1
	CLRF TMR1H		; and its H & L registers
	CLRF TMR1L
	CLRF T2CON		; and timer 2
	CLRF CCP1CON	; disable the CCP module
	CLRF SSPCON		; and the SSP module
	bank1			; then switch to bank1
	MOVLW h'80'		; set Option reg to disable PORTB pullups
	MOVWF OPTION_REG
	CLRF PIE1		; turn off peripheral interrupts
	CLRF PIE2
	CLRF CVRCON		; now disable comparator Vref module
	MOVLW h'07'		; and disable the comparators
	MOVWF CMCON		; (by setting them for mode 111)
	MOVLW h'3F'		; next set RA0-RA5 as inputs
	MOVWF TRISA
	CLRF TRISB		; then set RB0-RB7 as outputs
	MOVLW h'F0'		; also set RC4-7 as inputs, RC0-3 as outputs
	MOVWF TRISC
	CLRF TRISD		; also set RD0-RD7 as outputs
	MOVLW h'04'		; and finally set RE2 as an input, RE0-1 as outputs
	MOVWF TRISE		; and PSP mode disabled
	MOVLW h'C3'		; then set RA0-RA2 as AN0-AN2, RA3 as Vref+,
	MOVWF ADCON1	; RA4 as AN4, result right justified
	bank0			; then down to bank 0 again, so we can
					; set defaults during startup
	MOVLW "D"		; first set device type for Diodes
	MOVWF Devcode
	MOVLW h'01'
	MOVWF DevNum	; DevNum = 1 (default setting)
	MOVWF TestNum	; also set for Test01 (Diode Irev test)
	CLRF Flags1		; clear all relay flags
	CLRF Flags2
	CLRF PIR1		; now clear all periph int flags
	CALL DispInit	; then initialise the LCD module
	CALL Display1	; and show initial greeting display
	CALL Wait2pt4sec	; then pause for 2.4 seconds

Mainloop:
;	Main operating loop begins here. We first check if the S3
;	button is pressed, and if so pick a device to test etc

	CALL Display2		; now display 'MenuSelect' information
	CALL DoAScan		; go check if a button was pressed
	MOVLW "3"			; check if it was "3"
	XORWF Keycode,0		; W will now have 0 if it was a match...
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a 3, and W -> 0)
	GOTO Mainloop		; Z=0 (because it wasn't a 3), so try again
	Ppage3
	CALL PickADevice	; it was a 3, so go choose a device to test
	Ppage3
	CALL ChooseTest		; going to choose the test wanted
	Ppage1				; then prepare to go to page 1, so we can
	CALL SetupTest		; setup for chosen test (both relay flags
						; and LCD display, ready to go
	Ppage0
	CALL Wait1sec		; now pause a second
	CALL DoAScan		; then go look for a button press again
	MOVLW "7"			; check if it was a "7"
	XORWF Keycode,0		; w will now = 0 if we have a match
	BTFSS STATUS,Z		; skip if Z=1 (ie., it was a 7)
	GOTO Mainloop		; otherwise just go back to try all over
OKtoTest:
	copy Flags1,PORTB	; we're OK to test, so set up relays
	copy Flags2,PORTD	; (all except RLY1, that is)
	MOVLW h'01'			; first	check if TestNum = 01
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1 
	GOTO $+3			; otherwise keep looking
	CALL Test01			; is Test01, so go do it
	GOTO OffnBack		; key pressed, so abort
 
	MOVLW h'02'			; not 01, so now check if TestNum = 02
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking
	CALL Test02			; is Test02, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'03'			; not 02, so now check if TestNum = 03
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test03			; is Test03, so go do it
	GOTO OffnBack		; key pressed, so abort
 
	MOVLW h'04'			; not 03, so now check if TestNum = 04
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test04			; is Test04, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'05'			; not 04, so now check if TestNum = 05
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test05			; is Test05, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'06'			; not 05, so now check if TestNum = 06
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test06			; is Test06, so go do it
	GOTO OffnBack		; key pressed, so abort 

	MOVLW h'07'			; not 06, so now check if TestNum = 07
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test07			; is Test07, so go do it
	GOTO OffnBack		; key pressed, so abort  

	MOVLW h'08'			; not 07, so now check if TestNum = 08
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test08			; is Test08, so go do it
	GOTO OffnBack		; key pressed, so abort 

	MOVLW h'09'			; not 08, so now check if TestNum = 09
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test09			; is Test09, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'0A'			; not 09, so now check if TestNum = 0A
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0A			; is Test0A, so go do it
	GOTO OffnBack		; key pressed, so abort 

	MOVLW h'0B'			; not 0A, so now check if TestNum = 0B
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0B			; is Test0B, so go do it
	GOTO OffnBack		; key pressed, so abort 

	MOVLW h'0C'			; not 0B, so now check if TestNum = 0C
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0C			; is Test0C, so go do it
	GOTO OffnBack		; key pressed, so abort 

	MOVLW h'0D'			; not 0C, so now check if TestNum = 0D
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0D			; is Test0D, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'0E'			; not 0D, so now check if TestNum = 0E
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0E			; is Test0E, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'0F'			; not 0E, so now check if TestNum = 0F
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test0F			; is Test0F, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'10'			; not 0F, so now check if TestNum = 10
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test10			; is Test10, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'11'			; not 10, so now check if TestNum = 11
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test11			; is Test11, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'12'			; not 11, so now check if TestNum = 12
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test12			; is Test12, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'13'			; not 12, so now check if TestNum = 13
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test13			; is Test13, so go do it
	GOTO OffnBack		; key pressed, so abort

	MOVLW h'14'			; not 13, so now check if TestNum = 14
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise keep looking 
	CALL Test14			; is Test14, so go do it
	GOTO OffnBack		; key pressed, so abort

	CALL Test15			; must be Test15, so go do it
OffnBack:
	BCF PORTB,7			; key pressed, so turn off RLY1
	Ppage3
	CALL HVSPTest		; then check for HV still present
	Ppage0				; (only returns when it has gone <30V)
	GOTO Mainloop		; then loop back to begin again
 
;	main program loop ends -- subroutines follow
;
;	***********************************************************
;
Bin16toBCD:
;	routine to convert a 16-bit fixed point binary number in H_byte
;	and L_byte into a 5-digit BCD number, left in R0, R1 and R2
;	(Adapted from Microchip Technology B16TOBCD routine -- AN526)

	BCF STATUS,0			; first clear the carry bit
	MOVLW h'10'				; set count for 16d/10h
	MOVWF Count
	CLRF R0					; and clear the output registers
	CLRF R1
	CLRF R2
Loop16:
	RLF L_byte,F			; now rotate L_byte left through carry
	RLF H_byte,F			; and do the same with H_byte, from carry
	RLF R2,F				; then continue through R2
	RLF R1,F				; and R1
	RLF R0,F				; and finally R0

	DECFSZ Count,F			; decrement count, skip if it -> 0
	GOTO adjDEC				; otherwise keep going by going to adjDEC
	RETLW h'0'				; done all 16 bits, so leave

adjDEC:
	MOVLW R2				; 
	MOVWF FSR				;
	CALL adjBCD
; 
	MOVLW R1				;
	MOVWF FSR				;
	CALL adjBCD				;
; 
	MOVLW R0				;
	MOVWF FSR				;
	CALL adjBCD				;
;
	GOTO Loop16
;
adjBCD:
	MOVLW h'03'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,3			; test if result > 7
	MOVWF 0					; 
	MOVLW h'30'				; 
	ADDWF 0,W				;
	MOVWF Temp 
	BTFSC Temp,7			; test if result > 7
	MOVWF 0					; save as MSD
	RETLW h'0'
;	
;**************************************************************
;
BringbackADC2AARG:
;	subr to copy ADC regs back into AARG regs
	MOVF ADCEXP,W		; first copy ADC EXP
	MOVWF AEXP
	MOVF ADCB0,W		; then ADCB0
	MOVWF AARGB0
	MOVF ADCB1,W		; and finally ADCB1
	MOVWF AARGB1
	RETURN				; then return
;
;	***********************************************************
;
ByteSplit:
	; routine to split a byte in w (value 0-99) into two decimal
	; digits which are placed into Byte10 and Byte1
	ANDLW h'7F'			; first make sure byte value is less than 128
	MOVWF Temp3			; then save it in Temp3
	CLRF Byte10			; initialise output digit values
	CLRF Byte1
Beg10:
	MOVLW h'0A'			; subtract 10d from input byte in Temp3
	SUBWF Temp3,0		; w = Temp3 - 0Ah
	BTFSS STATUS,C		;skip if we got a carry (Temp3 >= 10d)
	GOTO Beg1			;no carry, so Temp3 < 10d. Skip to Beg1
	BTFSS STATUS,Z		;carry was set, but check for zero
	GOTO $+3			;no zero, so continue
	INCF Byte10,1		;0 and C, so must have been 10 exactly
	RETURN				;so just increment 10s & leave
	INCF Byte10,1		;positive result, so increment 10s
	MOVWF Temp3			;save remainder in w back to Temp3
	GOTO Beg10			;and continue looping
Beg1:
	INCF Byte1,1		;increment units -- must have at least one
	DECF Temp3,1		;now decrement Temp3
	BTFSS STATUS,Z		;skip if zero bit is set now
	GOTO Beg1			;not zero yet, so continue
	RETURN				;zero, so we're done -- leave
;
;	**************************************************************
;
CalcBeta:
	; routine to calculate BJT hFE, knowing the base current Ig
	; and using the 24b FP version of Idev (scaled 0-5298 on high range
	; or 0-2066 on low range) left in KARG regs by ReadIdev.
	; Result is left in Idig1-4 for display

	copy KEXP,AEXP		; first copy Idev back into AARG regs
	copy KARGB0,AARGB0	; so AARG regs now have 
	copy KARGB1,AARGB1	; Idev (scaled 0-5298 or 0-2066) in 24-bit FP form
	SetBARG10			; so load BARG regs with 10d
	BTFSS Flags1,6		; now which current range was used for Idev?
	GOTO LowIdr			; low range, so go to divide by 10
	CALL FPM24			; hi range, so multiply by 10 (-> 0-52980uA)
	CALL Check4FPE		; (duck away to check for any FP errors)
	GOTO WhichTnum		; then go to look for test number
LowIdr:
	CALL FPD24			; low range, so divide by 10 (-> 0-206.6uA)
	CALL Check4FPE		; (duck away to check for any FP errors)
WhichTnum:
;	AARG regs now have Idev in microamps, in 24-bit FP form
	MOVLW h'09'			; now check if TestNum = 09 (Ib=20uA)
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+7			; otherwise keep looking 
	MOVLW h'83'			; is Test09, so load BARG regs with 
	MOVWF BEXP			; 20d in 24-bit form: 83 20 00
	MOVLW h'20'			; (precalc using Fprep)
	MOVWF BARGB0
	CLRF BARGB1
	GOTO Dividem		; then go do the division
	MOVLW h'0A'			; TestNum not = 09, so try 0A (Ib=100uA)
	XORWF TestNum,0		; w will = 0 if we got a match
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+7			; otherwise skip (must be 0B)
	MOVLW h'85'			; is Test0A, so load BARG regs with 
	MOVWF BEXP			; 100d in 24-bit form: 85 48 00
	MOVLW h'48'			; (precalc using Fprep)
	MOVWF BARGB0
	CLRF BARGB1
	GOTO Dividem		; then go do the division
	MOVLW h'87'			; must be Testnum 0B so, load BARG regs
	MOVWF BEXP			; with 500d in 24-bit form: 87 7A 00
	MOVLW h'7A'			; (precalc using Fprep)
	MOVWF BARGB0
	CLRF BARGB1			; then drop thru into division
Dividem:
	; Idev in AARG regs, Ib divisor in BARG regs, so we can divide
	CALL FPD24			; so call FPD24 to divide (AARG <- AARG/BARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	result of division now in AARG regs, in 24-bit FP form
	CALL FloatIntoBCD	; now convert into BCD digits
;	finally turn back into digits Idig1-4 for display as hFE
	MOVLW h'30'				; now first see if Thousands is zero
	XORWF Thousands,W		; Z will = 1 if it is
	BTFSS STATUS,Z			; so skip if that's what you find
	GOTO $+3				; otherwise go to process IDig1 normally
	MOVLW h'20'				; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Thousands,W		; not a leading zero, so fetch Thousands into W
	MOVWF IDig1				; and save Thousands (or blank) as IDig1
	copy Hundreds, IDig2	; then copy Hundreds into IDig2 (even if zero)
	copy Tens, IDig3		; copy Tens into IDig3 (even if zero)
	copy Ones, IDig4		; and copy Ones into IDig4
	RETURN				; then return to display, etc
;
;	***********************************************************
;
Check4FPE:
	; routine to check if floating point functions had errors
	IORLW h'00'			; check if w came back with 00h or FFh
	BTFSC STATUS,Z		; if Z=0, must have been FFh (= an error)
	RETURN				; if Z=1, must have been 00h (= no error)
	CALL ClearLCD		; was an error, so prepare to show it
	MOVLW h'80'			; next set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW "F"			; then send "FPE!"
	CALL DisplayData
	MOVLW "P"
	CALL DisplayData
	MOVLW "E"
	CALL DisplayData
	MOVLW "!"
	CALL DisplayData
	CALL Wait2pt4sec	; now pause for 2.4 seconds for user to see
	CALL ClearLCD		; then wipe away again
	CALL Display2		; restore normal display
	RETURN				; and then return to resume anyway
;
;	************************************************************
;
ClearLCD:
	;routine to clear LCD and reset address counter
	MOVLW h'01'			; clear display & reset addr ptr
	CALL DispAddress
	CALL Delay200ms		; pause 200ms to give it time to clear
	CALL Delay200ms		; and again, for really sloooow LCDs
	RETURN				; then return
;
;	************************************************************
;	
Delay1ms:
	;routine to delay approx 1ms (2058 x 0.5us = 1029us) before return
	MOVLW h'0A'			; set Counter1 for 10 outer loops
	MOVWF Counter1
OuterLoop:
	MOVLW h'42'			; and Counter2 for 66 inner loops
	MOVWF Counter2		; (66 x 3mc = 198mc, + 7mc = 205)
	DECFSZ Counter2, 1	; decrement Counter2 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	DECFSZ Counter1, 1	; did reach zero, so decrement Counter1
	GOTO OuterLoop		; didn't hit zero, so loop back
	RETURN				; reached zero (10 x 66 loops) so return

Delay200ms:
	;routine to delay approx 200ms before returning
	MOVLW h'C8'			; load w for 200 outer loops
	GOTO $+2			; then go set Counter3
Delay10ms:
	;routine to delay approx 10ms before returning
	MOVLW h'0A'			; load w for 10 outer loops
	MOVWF Counter3		; now set Counter3
	CALL Delay1ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
;
;	***********************************************************
;	
DispAddress:
	;routine to translate & load display address (in w reg) into LCD
	BCF PORTE,1			; first set RS pin of LCD low, for instr/addr
	CALL Nibbles2LCD	; then send addr/cmd nibbles to LCD
	BCF PORTE,1			; make sure RS is is left low
	GOTO BusyWait		; then jump to delay 250us before return
	
DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTE,1			; RS pin of LCD high, for sending data
	CALL Nibbles2LCD	; then send data nibbles to LCD

BusyWait:
	; routine to wait until LCD module is not busy, after writing
	MOVLW h'7D'			; set delay counter for 125 loops
	MOVWF Counter5		; (should give about 125 x 4 x 0.5 = 250us)
	NOP
	DECFSZ Counter5,1	; decrement counter & skip when zero
	GOTO $-2			; loop back until we reach zero
	RETURN				; then return
;
;	***********************************************************
;
DispInit:
	; routine to initialise LCD display module
	BCF PORTE,0			; first make sure EN and RS lines are low
	BCF PORTE,1
	CALL Delay200ms		; then wait about 200ms before proceeding
	BSF PORTC,3			; now load initialisn code 03h into RC0-3 
	BSF PORTC,2			; (= DB7 to DB4, so 0C -> 03h)
	BCF PORTC,1
	BCF PORTC,0
	ToggleEN			; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms
	ToggleEN			; then toggle EN to write to LCD again
	CALL Delay10ms		; then wait about 10ms again
	ToggleEN			; then toggle EN to write to LCD again
	CALL Delay10ms		; then wait about 10ms a third time
	BCF PORTE,1			; make sure RS is still low
	BCF PORTC,3			; now change code in RC to 04 (-> 20h)
	ToggleEN			; then toggle EN to write to LCD
	CALL Delay10ms		; then wait about 10ms one last time

	MOVLW h'28'			; now set LCD functions (4 bit i/f, 2 lines, 5x10 chars)
	CALL DispAddress	; (also delays for 200us)
	MOVLW h'0C'			; also set display mode (disp on, no blink or cursor)
	CALL DispAddress	; (also delays for 200us)
	CALL ClearLCD		; then clear LCD screen (& delay 400ms)
	MOVLW h'06'			; set entry mode (increm addr, no shift)
	CALL DispAddress	; (also delays for 200us)
	RETURN				; should now be set up & ready to go, so leave
;
;	***********************************************************
;
Display1:
	; routine to display initial greeting text on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW h'00'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in disp string char pointer
	CALL DispString01	; then go send that string to LCD
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW h'12'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in disp string char pointer
	CALL DispString01	; and go send that string to LCD
	RETURN				; before leaving
;
;	***********************************************************
;
Display2:
	; routine to display Menu Select text on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL DispAddress	; (also delays for 160us)
	MOVLW h'24'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL DispString01	; then go send string to LCD
	MOVLW h'C0'			; now move down to line 2
	CALL DispAddress
	MOVLW h'36'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL DispString01	; and go send that string to LCD
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayH01:
; 	routine to display measured & calculated hFE on the second
;	line of the LCD, with digits in the right positions
;	for Tests 09, 0A, 0B
	MOVLW h'CC'			; set LCD address to line 2, char C
	CALL DispAddress	; (also delays for 160us)
	MOVF IDig1,0		; then send IDig1,
	CALL DisplayData
	MOVF IDig2,0		; IDig2
	CALL DisplayData
	MOVF IDig3,0		; IDig3
	CALL DisplayData
	MOVF IDig4,0		; & finally IDig4
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayI01:
; 	routine to display measured Idev on the second line
;	of the LCD, with digits in the right positions
;	for Tests 01, 02, 07, 08, 0D, 10
	MOVLW h'CA'			; set address to line 2, char A
	CALL DispAddress	; (also delays for 160us)
	MOVF IDig1,0		; then send IDig1,
	CALL DisplayData
	MOVF IDig2,0		; IDig2
	CALL DisplayData
	MOVF IDig3,0		; IDig3
	CALL DisplayData
	MOVF IDig4,0		; IDig4
	CALL DisplayData
	MOVF IDig5,0		; and Idig5 ("m" or mu)
	CALL DisplayData
	MOVLW "A"			; and finally "A"
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayI02:
; 	routine to display measured Idev on the second line
;	of the LCD, with digits in the right positions
;	for Tests 11, 12, 13
	MOVLW h'CC'			; set address to line 2, char C
	CALL DispAddress	; (also delays for 160us)
	MOVF IDig1,0		; then send IDig1
	CALL DisplayData
	MOVF IDig2,0		; IDig2
	CALL DisplayData
	MOVF IDig5,0		; and Idig5 ("m" or mu)
	CALL DisplayData
	MOVLW "A"			; and finally "A"
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayV01:
;	routine to display measured Vdev on the second line
;	of the LCD, with digits inserted in the right positions
;	for Tests 04, 05, 06 and 0C
	MOVLW h'C9'			; set address to line 2, char 9
	CALL DispAddress	; (also delays for 160us)
	MOVF VdevDig1,0		; then send VdevDig1
	CALL DisplayData
	MOVF VdevDig2,0		; and VdevDig2
	CALL DisplayData
	MOVF VdevDig3,0		; and VdevDig3
	CALL DisplayData
	MOVF VdevDig4,0		; and VdevDig4
	CALL DisplayData
	MOVLW "V"			; followed by "V"
	CALL DisplayData
	MOVLW " " 			; and two spaces
	CALL DisplayData
	MOVLW " " 
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayV02:
;	routine to display measured Vdev on the second line
;	of the LCD, with digits inserted in the right positions
;	for Tests 0F, 14 and 15
	MOVLW h'CB'			; set address to line 2, char B
	CALL DispAddress	; (also delays for 160us)
	MOVF VdevDig1,0		; then send VdevDig1
	CALL DisplayData
	MOVF VdevDig2,0		; and VdevDig2
	CALL DisplayData
	MOVF VdevDig3,0		; and VdevDig3
	CALL DisplayData
	MOVF VdevDig4,0		; and VdevDig4
	CALL DisplayData
	MOVLW "V"			; followed by "V"
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayV03:
;	routine to display measured Vdev and Idev on the second
;	line of the LCD, with digits inserted in the right
;	positions for Test03
	MOVLW h'C3'			; set address to line 2, char 3
	CALL DispAddress	; (also delays for 160us)
	MOVF VdevDig2,0		; then send VdevDig2
	CALL DisplayData
	MOVF VdevDig3,0		; and VdevDig3 (dec pt)
	CALL DisplayData
	MOVF VdevDig4,0		; and VdevDig4
	CALL DisplayData
	MOVLW "V"			; followed by "V"
	CALL DisplayData
	MOVLW "/"			; and "/If="
	CALL DisplayData
	MOVLW "I"
	CALL DisplayData
	MOVLW "f"
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVF IDig1,0		; then Idig1
	CALL DisplayData
	MOVF IDig2,0		; and Idig2
	CALL DisplayData
	MOVF IDig3,0		; then Idig3 (".")
	CALL DisplayData
	MOVF IDig4,0		; then Idig4
	CALL DisplayData
	MOVF IDig5,0		; and finally Idig5 ("m" or mu)
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DisplayIvVgs:
;	routine to display measured Vgs and Idev on the second
;	line of the LCD, with digits inserted in the right
;	positions for Test0E (shows Vgs vs Idev, for MOSFETs)
	MOVLW h'C4'			; set address to line 2, char 4
	CALL DispAddress	; (also delays for 160us)
	MOVF VgsDig1,0		; then send VgsDig1
	CALL DisplayData
	MOVF VgsDig2,0		; and VgsDig2
	CALL DisplayData
	MOVLW "."			; and the decimal pt
	CALL DisplayData
	MOVF VgsDig3,0		; then VgsDig3
	CALL DisplayData
	MOVLW "V"			; followed by a 'V'
	CALL DisplayData
	MOVLW "I"			; and "I="
	CALL DisplayData
	MOVLW "="
	CALL DisplayData
	MOVF IDig1,0		; then IDig1
	CALL DisplayData
	MOVF IDig2,0		; then IDig2
	CALL DisplayData
	MOVF IDig3,0		; then IDig3
	CALL DisplayData
	MOVF IDig4,0		; and IDig4
	CALL DisplayData
	MOVF IDig5,0		; then IDig5 (m or u)
	CALL DisplayData
	RETURN				; before leaving
;
;	***********************************************************
;
DispString01:
	; routine to display a 16-char string from the first set of
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called
 
	MOVLW h'10'			; first prepare to call Dstring01
	MOVWF PCLATH		; (which starts at 1000h in page2)
	MOVF DSpointer,0	; next load next char's offset into w
	CALL Dstring01		; & call lookup table to get char
	Ppage0				; then swing back to current page
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer for next char
	GOTO DispString01	; and loop back to continue
;
;	***********************************************************
;
DispString02:
	; routine to display a 16-char string from the second set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called

	MOVLW h'11'			; first prepare to call Dstring02
	MOVWF PCLATH		; (which starts at 1100h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring02		; & call lookup table to get char
	Ppage0				; then return to page0
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO DispString02	; and loop back to continue
;
;	***********************************************************
;
DispString03:
	; routine to display a 16-char string from the third set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called

	MOVLW h'12'			; first prepare to call Dstring03
	MOVWF PCLATH		; (which starts at 1200h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring03		; & call lookup table to get char
	Ppage0				; then return to page0
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO DispString03	; and loop back to continue
;
;	***********************************************************
;
DoAScan:
	; routine to scan front panel buttons via Scankeys, re-check
	; after bounce delay and save in Keycode if valid key press
	; detected, before returning
	; (keeps looping if no valid keypress detected)
	CALL ScanKeys		; go scan FP buttons
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not 0)
	GOTO DoAScan		; otherwise loop back & keep scanning
	MOVWF Keycode		; Z=0, so a button was pressed - save code
	CALL Delay200ms		; then wait for 200ms to allow for bounce
	CALL Delay200ms		; and again, just to make sure
	CALL ScanKeys		; & scan again to verify
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS,Z		; skip if Z=0 (ie w not 0, because it has a keycode)
	GOTO DoAScan		; Z=1, so just a false alarm - loop back to start 
	XORWF Keycode, 0	; Z=0, so compare with first code (w -> 0 if match)
	BTFSS STATUS,Z		; skip if Z=1 (because codes did match)
	GOTO DoAScan		; no match - loop back to start
	RETURN				; matched, so exit (with valid char in Keycode)
;
;	*************************************************************
;
FloatIntoBCD:
;	subr to convert 24-bit FP reading (in AARG regs on entry)
;	into a four-digit BCD number (left in Ones-Tens-Hundreds-Thousands)
;	
	CALL INT2416		; first convert into a 16-bit fixed point number
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARGB0 and AARGB1 now have value as a 16-bit fixed-pt number
;
	copy AARGB0, H_byte		; next transfer these bytes to H_byte
	copy AARGB1, L_byte		; and L_byte, as input for the 
	CALL Bin16toBCD			; binary to BCD conversion routine
;
	MOVF R2,W			; now fetch LSD BCD digit into W
	GetASCII			; convert to ASCII
	MOVWF Ones			; and save in Ones
;
	SWAPF R2,W			; then repeat for the other signif digits
	GetASCII
	MOVWF Tens
;
	MOVF R1,W
	GetASCII
	MOVWF Hundreds
;
	SWAPF R1,W
	GetASCII
	MOVWF Thousands

	RETURN
;
;	*************************************************************
;
Nibbles2LCD:
	; routine to test bits of data byte (passed in w) then send
	; to LCD module as two nibbles (high nibble first)
	MOVWF Temp2			; first save byte to be sent in Temp2
	BTFSC Temp2,7		; now test bit 7 (upper nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,6		; now test bit 6
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,5		; now test bit 5
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,4		; now test bit 4
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; now toggle EN to write hi nibble to LCD

	BTFSC Temp2,3		; next test bit 3 (lower nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,2		; now test bit 2
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,1		; now test bit 1
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,0		; now test bit 0
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; toggle EN again to write low nibble to LCD
	RETURN				; and return, since both nibbles sent
;
;	************************************************************
;
ReadIdev:
;	routine to use the PIC's ADC module (via AN1) to take 10
;	readings, which are added together and then divided by 10
;	to get an average. After getting the first average it checks to
;	see if an uprange is needed (ADC average >=1023). If so, it
;	switches to the hi range and goes back for another 10 readings.
;	It also checks if a downrange is needed (ADC average <10), in
;	which case it downranges and takes another set of readings.
;	Otherwise it calculates the current (Idev) ready for display.

	UseAN1			; first set ADC for AN1 channel, to read current 

Take10Readings:
	MOVLW h'0A'			; next set MLoopCtr for a loop/sequence
	MOVWF MLoopCtr 		; of 10 readings
	MOVLW h'6E'			; then set ADCEXP, ADCB0 and ADCB1 for a near
	MOVWF ADCEXP		; zero initial number (6E 17 03 = 0.000009001,
	MOVLW h'17'			; which is close enough to zero for our purposes)
	MOVWF ADCB0
	MOVLW h'03'
	MOVWF ADCB1

MeasLoop:
	BCF PIR1,6			; first clear ADIF bit in PIR1, just in case
	BSF ADCON0,2		; then start ADC conversion by setting GO bit
	BTFSC ADCON0,2		; then wait until conversion is complete by
	GOTO $-1			; looping back until GO bit cleared again
	MOVF ADRESH,0		; done, so fetch high byte of ADC result into w
	MOVWF AARGB0		; and place into AARGB0
	bank1
	MOVF ADRESL,0		; then fetch low byte of result as well
	bank0
	MOVWF AARGB1		; and place it into AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy ADCEXP,BEXP	; now AARG reg have ADC reading, so bring
	copy ADCB0,BARGB0	; accumulated readings into BARG regs
	copy ADCB1,BARGB1
	CALL FPA24			; now add the two together using FPA24
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,ADCEXP	; then save sum back into ADC reading regs
	copy AARGB0,ADCB0	; (so sum of readings will be accumulating
	copy AARGB1,ADCB1	;  in these regs)
	DECFSZ MLoopCtr,1	; decrement loop counter, skip if it -> 0
	GOTO MeasLoop		; otherwise loop back for another reading
	SetBARG10			; done 10, so load dec10 (24bFP) into BARG regs
	CALL BringbackADC2AARG	; bring sum of the 10 readings into AARG regs
	CALL FPD24			; then call FPD24 to divide the sum by 10
	CALL Check4FPE		; (duck away to check for any FP errors)
	copy AEXP,ADCEXP	; then save average back into ADC reading regs
	copy AARGB0,ADCB0	; (so these regs will now have the average
	copy AARGB1,ADCB1	; of the 10 readings, 0000-1023, in 24b FP form)

;	AARG regs still have 24b version of average reading, so now check
;	for full scale reading, in case upranging may be needed 
	MOVLW h'88'			; now load 24b FP version of 1023
	MOVWF BEXP			; (pre calc using Fprep) into BARG regs
	MOVLW h'7F'			; so it can be used as subtractant
	MOVWF BARGB0		; (to see if we should uprange)
	MOVLW h'C0'
	MOVWF BARGB1
	CALL FPS24			; now subtract BARG from AARG (-> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;						  AARG regs now have (ADC result - 1023)
	BTFSS AARGB0,7		; so check sign bit of AARGB0, skip if set (neg result)
	GOTO ChekHirange	; positive result, so we do have a FS+ reading
	CALL BringbackADC2AARG	; neg result, so bring back average ADC reading
	MOVLW h'80'			; then load BARG regs with 3.8618 in 24b FP form
	MOVWF BEXP			; (precalc using fprep)
	MOVLW h'77'			; (3.8618 = 0.2mA reading on high range) 
	MOVWF BARGB0
	MOVLW h'28'
	MOVWF BARGB1
	CLRF BARGB2	
	CALL FPS24			; now subtract BARG from AARG (-> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;						  AARG regs now have (ADC result - 3.8618)
	BTFSS AARGB0,7		; so check sign bit of AARGB0, skip if set (neg result)
	GOTO ChekRange		; otherwise continue to process (reading > 3.8618)
	BTFSS Flags1,6		; reading is <3.8618, but are we in low range already?
	GOTO KGlow			; yes (flag clear), so OK to continue in low range
	BCF Flags1,6		; flag is set, so clear it for 206uA range
	BCF PORTB,6			; also clear RB6 to turn off RLY7
	GOTO Take10Readings	; then go back for another 10 readings on low range

ChekHirange:
	BTFSC Flags1,6		; FS reading, but see if we are in hi range already
	GOTO KGhigh			; flag is set (hi range already), so just keep going
	BSF PORTB,6			; not set, so set for 50mA range (RLY7 on)
	BSF Flags1,6		; and set Flags1,6 to show it
	GOTO Take10Readings	; then go back for another 10 readings on hi range

ChekRange:
	BTFSS Flags1,6		; now are we on high range or low range?
	GOTO KGlow			; flag clear, so we're on low range. Go thataway

KGhigh:
;	high range, so proceed to process reading & calc device current
	CALL BringbackADC2AARG ; first bring back ADC reading into AARG regs
	MOVLW h'81'			; now set FP multiplier for scaling to
	MOVWF BEXP			; 0-5298 (81 25 B9 = 5.178833)
	MOVLW h'25'			; (cf theoretical multiplier = 5.17888563)
	MOVWF BARGB0
	MOVLW h'B9'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs now have scaled Idev reading (0000 - 5298) in 24b FP format
	copy AEXP,KEXP		; so save args in K regs for use in Vdev correction
	copy AARGB0,KARGB0	; (also in CalcBeta, and in Vgs correction for
	copy AARGB1,KARGB1	;  N-channel MOSFETs)
;
	CALL FloatIntoBCD	; then go turn into four BCD digit number
;
	MOVLW h'30'				; now first see if Thousands is zero
	XORWF Thousands,W		; Z will = 1 if it is
	BTFSS STATUS,Z			; so skip if that's what you find
	GOTO $+3				; otherwise go to process IDig1 normally
	MOVLW h'20'				; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Thousands,W		; not a leading zero, so fetch Thousands into W
	MOVWF IDig1				; and save Thousands (or blank) as IDig1

	copy Hundreds, IDig2	; then copy Hundreds into IDig2 (even if zero)
	MOVLW h'2E'				; IdevDig3 becomes the decimal point
	MOVWF IDig3
	copy Tens, IDig4		; and just copy Tens into IDig4
	MOVLW h'6D'				; making IDig5 an 'm'
	MOVWF IDig5				; (we discard Ones on this range)
	RETURN					; then return to display, etc

KGlow:
;	low range, so proceed to process reading & calc device current
	CALL BringbackADC2AARG	; first bring back ADC rdg into AARG regs
	MOVLW h'80'			; now set FP multiplier for scaling to
	MOVWF BEXP			; 0-2066 (80 01 40 = 2.0195312)
	MOVLW h'01'			; (cf theoretical multiplier = 2.019550342)
	MOVWF BARGB0
	MOVLW h'40'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs now have scaled Idev reading (0000 - 2066) in 24b FP format
	copy AEXP,KEXP		; so save args in K regs for use in Vdev correction
	copy AARGB0,KARGB0	; (also in CalcBeta, and in Vgs correction for
	copy AARGB1,KARGB1	;  N-channel MOSFETs)

	CALL FloatIntoBCD	; then go turn into four BCD digit number

	MOVLW h'20'			; low range, so first make IDig1 a blank
	MOVWF IDig1

	MOVLW h'30'			; now check if Thousands is zero
	XORWF Thousands,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process IDig2 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Thousands,W	; not a leading zero, so fetch Thousands into W
	MOVWF IDig2			; and save Thousands (or blank) as IDig2

	MOVLW h'30'			; now check if Hundreds is zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+7			; otherwise go to process IDig3 normally
	MOVLW h'20'			; Hundreds = 0, but check if IDig2 is blank
	XORWF IDig2,W		; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process IDig3 normally
	MOVLW h'20'			; IDig2 is blank, so make IDig3 blank also
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch Hundreds into W
	MOVWF IDig3			; then save Hundreds (or blank) as IDig3
	copy Tens, IDig4	; and just copy Tens into IDig4 (even if zero)
	MOVLW h'E4'			; making IDig5 a 'mu'
	MOVWF IDig5
	RETURN				; then return to display, etc
;
;	***********************************************************
;
ReadVdev:
;	routine to take a reading of device voltage via analog input
;	AN0, to be displayed as VDig1-3. Measures 0-1028V on hi range
;	(RLY8 off, Flags2,1 = 0) or 0-102.8V on low range (RLY8 on, 
;	Flags2,1 = 1).

	UseAN0				; first set ADC for AN0 channel, to read Vdev 
	BCF PIR1,6			; then clear ADIF bit in PIR1, just in case
	CALL Delay1ms		; allow 1ms delay to cover acquisition	
	BSF ADCON0,2		; now start conversion by setting GO bit
	BTFSC ADCON0,2		; then wait until conversion complete by
	GOTO $-1			; looping back until GO bit cleared again
	MOVF ADRESH,0		; done, so fetch high byte of ADC result into w
	MOVWF AARGB0		; and place into AARGB0
	bank1
	MOVF ADRESL,0		; then fetch low byte of result as well
	bank0
	MOVWF AARGB1		; and place it into AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	MOVLW h'7F'			; AARG regs have ADC reading in 24b FP form,
	MOVWF BEXP			; so now load BARG regs with scaling factor
	MOVLW h'00'			; 7F00A3 = 1.0049744d (for 1023 FSD -> 1028.098)
	MOVWF BARGB0		; (cf theoretical factor of 1.004983957)
	MOVLW h'A3'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs have Vdev reading (000 - 1028) in 24b FP format
	copy AEXP,JEXP		; now save copy in JARG regs in case needed
	copy AARGB0,JARGB0	; by Vgs calculation for a P-channel MOSFET
	copy AARGB1,JARGB1
	Ppage1				; now jump up to page 1
	CALL VdevshFix		; to correct for volts across current shunt
	Ppage0
	CALL INT2416		; next convert into a 16-bit fixed point number
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARGB0 and AARGB1 now have scaled Vdev as a 16-bit fixed-pt number
;
	copy AARGB0, H_byte		; next transfer these bytes to H_byte
	copy AARGB1, L_byte		; and L_byte, as input for the 
	CALL Bin16toBCD			; binary to BCD conversion routine
;
	MOVF R2,W			; now fetch LSD BCD digit into W
	GetASCII			; convert to ASCII
	MOVWF Ones			; and save in Ones
;
	SWAPF R2,W			; then repeat for the other digits
	GetASCII
	MOVWF Tens
;
	MOVF R1,W
	GetASCII
	MOVWF Hundreds
;
	SWAPF R1,W
	GetASCII
	MOVWF Thousands
;
ChekVRange:
	BTFSC Flags2,1		; now are we on high range or low range?
	GOTO LowVRng		; flag set, so we're on low range. Go thataway
HiVRng:
	MOVLW h'30'			; high range (0-1028), so see if Thousands=0
	XORWF Thousands,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process VdevDig1 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Thousands,W	; not a leading zero, so fetch Thousands into W
	MOVWF VdevDig1		; and save Thousands (or blank) as VdevDig1
	copy Hundreds, VdevDig2		; then copy Hundreds as VdevDig2
	copy Tens, VdevDig3	; copy Tens into VdevDig3
	copy Ones, VdevDig4	; and copy Ones into VdevDig4
	RETURN				; then return to display, etc
LowVRng:
	MOVLW h'30'			; low range (0-102.8), so check if Hundreds=zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process VdevDig1 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch Hundreds into W
	MOVWF VdevDig1		; and save Hundreds (or blank) as VdevDig1
	copy Tens,VdevDig2	; then copy Tens into VdevDig2
	MOVLW "."			; make VdevDig3 the decimal point
	MOVWF VdevDig3
	copy Ones, VdevDig4	; and copy Ones into VdevDig4
	RETURN				; then return to display, etc
;
;	***********************************************************
;
ReadVgs:
;	routine to read Vgs for MOSFETs, via AN2. For N-ch MOSFETs,
;	where Vgs is positive, reads it directly but then subtracts
;	Vsh (volts across current shunt). For P-ch MOSFETs where Vgs
;	is nominally negative, reads & then subtracts from Vdev.

	UseAN2				; first set ADC for AN2 channel, to read Vgs 
	BCF PIR1,6			; then clear ADIF bit in PIR1, just in case
	CALL Delay1ms		; allow 1ms delay to cover acquisition	
	BSF ADCON0,2		; now start conversion by setting GO bit
	BTFSC ADCON0,2		; then wait until conversion complete by
	GOTO $-1			; looping back until GO bit cleared again
	MOVF ADRESH,0		; done, so fetch high byte of ADC result into w
	MOVWF AARGB0		; and place into AARGB0
	bank1
	MOVF ADRESL,0		; then fetch low byte of result as well
	bank0
	MOVWF AARGB1		; and place it into AARGB1
	CALL FLO24			; then go convert it into 24-bit FP form
	CALL Check4FPE		; (duck away to check for any FP errors)
	MOVLW h'7F'			; AARG regs have ADC reading in 24b FP form,
	MOVWF BEXP			; so now load BARG regs with scaling factor
	MOVLW h'01'			; 7F 01 40 = 1.0097656d (for 1033 FSD)
	MOVWF BARGB0		; (cf theoretical factor of 1.00977517)
	MOVLW h'40'
	MOVWF BARGB1
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARG regs have scaled Vgs reading (0000 - 1033) in 24b FP format
	MOVLW h'06'			; now do we have an N-ch or P-ch device?
	XORWF DevNum,0		; w will = 0 if it's a P-ch (DevNum = 6)
	BTFSS STATUS,Z		; so skip if Z=1 (P-ch)
	GOTO NchDev		; otherwise go do Vsh subtraction
	;	P-ch, so we have to save Vgs and subtract from Vdev,
	;	then leave result in AARG regs...
	copy AEXP,BEXP		; first move scaled Vgs over into BARG regs
	copy AARGB0,BARGB0
	copy AARGB1,BARGB1
	copy JEXP,AEXP		; now bring back Vdev from JARG regs
	copy JARGB0,AARGB0	; into the AARG regs
	copy JARGB1,AARGB1
	CALL FPS24			; then subtract Vgs from Vdev (AARG <- AARG-BARG)
	CALL Check4FPE		; (duck away to check for any FP errors)
	GOTO ProcessVgs		; go process corrected Vgs
NchDev:
;	N-ch, but we have to subtract Vsh to get true effective Vgs
;	(AARG regs still have scaled Vgs reading, 0000-1033)
	Ppage1
	CALL VgsshFix		; (code had to be moved up to page1)
	Ppage0
ProcessVgs:
;	AARG regs have corrected Vgs reading (0000 - 1033V) in 24b FP format	
	CALL INT2416		; next convert into a 16-bit fixed point number
	CALL Check4FPE		; (duck away to check for any FP errors)
;	AARGB0 and AARGB1 now have Vgs as a 16-bit fixed-pt number
;
	copy AARGB0, H_byte		; next transfer these bytes to H_byte
	copy AARGB1, L_byte		; and L_byte, as input for the 
	CALL Bin16toBCD			; binary to BCD conversion routine
;
	MOVF R2,W			; now fetch LSD BCD digit into W
	GetASCII			; convert to ASCII
	MOVWF Ones			; and save in Ones
;
	SWAPF R2,W			; then repeat for the other digits
	GetASCII
	MOVWF Tens
;
	MOVF R1,W
	GetASCII
	MOVWF Hundreds
;
	SWAPF R1,W
	GetASCII
	MOVWF Thousands
;
	MOVWF h'30'			; now check if Hundreds=zero
	XORWF Hundreds,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if that's what you find
	GOTO $+3			; otherwise go to process VgsDig1 normally
	MOVLW h'20'			; we do have a leading zero, so make it blank
	GOTO $+2
	MOVF Hundreds,W		; not a leading zero, so fetch Hundreds into W
	MOVWF VgsDig1		; and save Hundreds (or blank) as VgsDig1
	copy Tens,VgsDig2	; then copy Tens into VgsDig2
	copy Ones, VgsDig3	; and copy Ones into VgsDig3
	RETURN				; then return to display, etc
;
;	***********************************************************
;		
ScanKeys:
	; routine to look for a pressed key, return with ASCII code
	; in w (or with a null in w if no key is pressed)
	BTFSC PORTC, 4		; first check if RC4 = 0 (S3 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "3"			; S3 pressed, so return with "3" in w
	BTFSC PORTC, 7		; next check if RC7 = 0 (S4 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "4"			; S4 pressed, so return with "4" in w
	BTFSC PORTC, 5		; next check if RC5 = 0 (S5 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "5"			; S5 pressed, so return with "5" in w
	BTFSC PORTC, 6		; next check if RC6 = 0 (S6 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "6"			; S6 pressed, so return with "6" in w
	BTFSC PORTE, 2		; now check if RE2 = 0  (S7 pressed)
	RETLW h'00'			; no key pressed, so return with null in w
	RETLW "7"			; S7 pressed, so return with "7" in w
;
;	***********************************************************
;	The actual test routines follow

Test01:
;	routine to perform Test01 (Diode/Zener Irev with BV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test01			; Z=1, so no key pressed. Test again
	RETURN				; key was pressed, so return

Test02:
;	routine to perform Test02 (Diode/LED Irev with OPV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test02			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test03:
;	routine to perform Test03 (Diode/LED Vforward with OPV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev
	CALL DisplayV03		; and display both, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test03			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test04:
;	routine to perform Test04 (Diode/Zener Vrev with BV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev
	CALL DisplayV01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test04			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test05:
;	routine to perform Test05 (BJT BVcbo)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev
	CALL DisplayV01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test05			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test06:
;	routine to perform Test06 (BJT BVceo)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev
	CALL DisplayV01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test06			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test07:
;	routine to perform Test07 (BJT Icbo)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test07			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test08:
;	routine to perform Test08 (BJT Iceo)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test08			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test09:
;	routine to perform Test09 (BJT hFE with Ib = 20uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL CalcBeta		; then work out hFE from it 
	CALL DisplayH01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test09			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0A:
;	routine to perform Test0A (BJT hFE with Ib = 100uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev 
	CALL CalcBeta		; then work out hFE from it 
	CALL DisplayH01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0A			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0B:
;	routine to perform Test0B (BJT hFE with Ib = 500uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev
	CALL CalcBeta		; then work out hFE from it 
	CALL DisplayH01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0B			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0C:
;	routine to perform Test0C (MOSFET BVdss)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev  
	CALL DisplayV01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0C			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0D:
;	routine to perform Test0D (MOSFET Idss, with OPV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev  
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0D			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0E:
;	routine to perform Test0E (MOSFET Ids vs Vgs, i.e., gm)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then take a reading of Vdev
	CALL ReadVgs		; also take a reading of Vgs
	CALL DisplayIvVgs	; and display both Iv & Vgs
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0E			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test0F:
;	routine to perform Test0F (SCR BVaks)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev  
	CALL DisplayV02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test0F			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test10:
;	routine to perform Test10 (SCR or PUT Iaks with OPV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev  
	CALL DisplayI01		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test10			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test11:
;	routine to perform Test11 (SCR or PUT Iak with Ig = 20uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev  
	CALL DisplayI02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test11			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test12:
;	routine to perform Test12 (SCR or PUT Iak with Ig = 100uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev  
	CALL DisplayI02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test12			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test13:
;	routine to perform Test13 (SCR or PUT Iak with Ig = 500uA)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; then go take a reading of Idev  
	CALL DisplayI02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test13			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test14:
;	routine to perform Test14 (SCR or PUT Vak ON with OPV)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev  
	CALL DisplayV02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test14			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return

Test15:
;	routine to perform Test15 (PUT BVaks with g-a short)	
	BSF PORTB,7			; first turn on RB7 (RLY1)
	CALL ReadIdev		; take a reading of Idev first
	CALL ReadVdev		; then go take a reading of Vdev  
	CALL DisplayV02		; and display it, then 
	CALL Wait1sec		; wait a second so user can read
	CALL ScanKeys		; then look for a button press
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not = 0)
	GOTO Test15			; Z=1, so no key pressed. Test again
	RETURN				; key pressed, so return
;
;	***********************************************************
;
Wait2pt4sec:
;	routine to pause for about 2.4 seconds (12 x 200ms = 2.4s)
	MOVLW h'0C'			; set Counter4 for 12d loops
	GOTO $+2
Wait1sec:
;	routine to pause for about 1 second (5 x 200ms = 1s)
	MOVLW h'05'			; set Counter4 for 7d loops
	MOVWF Counter4
	CALL Delay200ms		; then wait for 200ms
	DECFSZ Counter4,1	; now decrement Counter4, skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; return when done

;	**********************************************************
;	now we move up to page 1, above the upper routines
;	in FPRF24.TXT
;	**********************************************************
;
	org h'0D50'
;	these routines were moved up here to make space on page 0
;	(Many are duplicates of p0 routines, so minimise page crossings)
;	***********************************************************
;	
P1DispAddress:
	;routine to translate & load display address (in w reg) into LCD
	BCF PORTE,1			; first set RS pin of LCD low, for instr/addr
	CALL P1Nibbles2LCD	; then send addr/cmd nibbles to LCD
	BCF PORTE,1			; make sure RS is is left low
	GOTO P1BusyWait		; then jump to delay 250us before return
	
P1DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTE,1			; RS pin of LCD high, for sending data
	CALL P1Nibbles2LCD	; then send data nibbles to LCD

P1BusyWait:
	; routine to wait until LCD module is not busy, after writing
	MOVLW h'7D'			; set delay counter for 125 loops
	MOVWF Counter5		; (should give about 125 x 4 x 0.5 = 250us)
	NOP
	DECFSZ Counter5,1	; decrement counter & skip when zero
	GOTO $-2			; loop back until we reach zero
	RETURN				; then return
;
;	***********************************************************
;
P1Nibbles2LCD:
	; routine to test bits of data byte (passed in w) then send
	; to LCD module as two nibbles (high nibble first)
	MOVWF Temp2			; first save byte to be sent in Temp2
	BTFSC Temp2,7		; now test bit 7 (upper nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,6		; now test bit 6
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,5		; now test bit 5
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,4		; now test bit 4
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; now toggle EN to write hi nibble to LCD

	BTFSC Temp2,3		; next test bit 3 (lower nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,2		; now test bit 2
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,1		; now test bit 1
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,0		; now test bit 0
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; toggle EN again to write low nibble to LCD
	RETURN				; and return, since both nibbles sent
;
;	***********************************************************
;
SetupTest:
;	routine to set up relay flags, LCD display lines etc
;	for chosen test (but we don't turn on any relays just yet.
;	This is done only for the actual test)

	MOVLW h'01'		; first check TestNum - is it 01?
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest01	; did match, so go setup for Test01
	MOVLW h'02'		; not 01, so try 02
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest02	; did match, so go setup for Test02
	MOVLW h'03'		; not 02, so try 03
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest03	; did match, so go setup for Test03
	MOVLW h'04'		; not 03, so try 04
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest04	; did match, so go setup for Test04
	MOVLW h'05'		; not 04, so try 05
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest05	; did match, so go setup for Test05
	MOVLW h'06'		; not 05, so try 06
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest06	; did match, so go setup for Test06
	MOVLW h'07'		; not 06, so try 07
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest07	; did match, so go setup for Test07
	MOVLW h'08'		; not 07, so try 08
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest08	; did match, so go setup for Test08
	MOVLW h'09'		; not 08, so try 09
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest09	; did match, so go setup for Test09
	MOVLW h'0A'		; not 09, so try 0A
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0A	; did match, so go setup for Test0A
	MOVLW h'0B'		; not 0A, so try 0B
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0B	; did match, so go setup for Test0B
	MOVLW h'0C'		; not 0B, so try 0C
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0C	; did match, so go setup for Test0C
	MOVLW h'0D'		; not 0C, so try 0D
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0D	; did match, so go setup for Test0D
	MOVLW h'0E'		; not 0D, so try 0E
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0E	; did match, so go setup for Test0E
	MOVLW h'0F'		; not 0E, so try 0F
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest0F	; did match, so go setup for Test0F
	MOVLW h'10'		; not 0F, so try 10
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest10	; did match, so go setup for Test10
	MOVLW h'11'		; not 10, so try 11
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest11	; did match, so go setup for Test11
	MOVLW h'12'		; not 11, so try 12
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest12	; did match, so go setup for Test12
	MOVLW h'13'		; not 12, so try 13
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest13	; did match, so go setup for Test13
	MOVLW h'14'		; not 13, so try 14
	XORWF TestNum,0	; (w will = 0 if it's a match
	BTFSC STATUS,Z	; skip if Z=0 (ie, not a match)
	GOTO SetTest14	; did match, so go setup for Test14
	GOTO SetTest15	; must be 15, so go setup for Test15
SetTest01:
	; routine to setup for Test01 (Diode/Zener Irev (BV))
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'12'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'42'		; next set Flags1 bits for Test01
	MOVWF Flags1
	MOVLW h'01'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	RETURN			; setup done, so return
SetTest02:
	; routine to setup for Test02 (Diodes/LEDs Irev (OPV))
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'24'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'42'		; next set Flags1 bits for Test02
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	RETURN			; setup done, so return
SetTest03:
	; routine to setup for Test03 (Diodes/LEDs Vfwd (OPV))
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'36'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test03
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	RETURN			; setup done, so return
SetTest04:
	; routine to setup for Test04 (Diode/Zener Vrev (BV))
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'48'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'42'		; next set Flags1 bits for Test04
	MOVWF Flags1
	MOVLW h'01'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	RETURN			; setup done, so return
SetTest05:
	; routine to setup for Test05 (BJT BVcbo)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'5A'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test05
	MOVWF Flags1
	MOVLW h'09'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	RETURN			; then return
SetTest06:
	; routine to setup for Test06 (BJT BVceo)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'6C'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test06
	MOVWF Flags1
	MOVLW h'01'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	RETURN			; then return
SetTest07:
	; routine to setup for Test07 (BJT Icbo)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'7E'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test07
	MOVWF Flags1
	MOVLW h'08'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	RETURN			; then return
SetTest08:
	; routine to setup for Test08 (BJT Iceo)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'90'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test08
	MOVWF Flags1
	MOVLW h'00'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	RETURN			; then return
SetTest09:
	; routine to setup for Test09 (BJT hFE with Ig=20uA)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'A2'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test09
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	CALL IgPolarity		; also Ig polarity check
	RETURN			; then return
SetTest0A:
	; routine to setup for Test0A (BJT hFE with Ig=100uA)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'B4'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'50'		; next set Flags1 bits for Test0A
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	CALL IgPolarity		; also Ig polarity check
	RETURN			; then return
SetTest0B:
	; routine to setup for Test0B (BJT hFE with Ig=500uA)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'C6'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'48'		; next set Flags1 bits for Test0B
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NPNorPNPCheck	; now check if it's NPN or PNP
	CALL IgPolarity		; also Ig polarity check
	RETURN			; then return
SetTest0C:
	; routine to setup for Test0C (MOSFET BVdss)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'D8'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test0C
	MOVWF Flags1
	MOVLW h'01'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NchorPch	; now check if it's Nch or Pch
	RETURN			; then return
SetTest0D:
	; routine to setup for Test0D (MOSFET Idss with OPV)
	CALL ShowTestL1	; first go and display correct L1
	MOVLW h'EA'		; then set correct offset for L2
	CALL ShowTestL2	; and display it also
	MOVLW h'40'		; next set Flags1 bits for Test0D
	MOVWF Flags1
	MOVLW h'02'		; and also Flags2 bits
	MOVWF Flags2	; (for relays setup)
	CALL NchorPch	; now check if it's Nch or Pch
	RETURN			; then return
SetTest0E:
	; routine to setup for Test0E (MOSFET Id v. Vgs)
	CALL ShowTestL1		; first go and display correct L1
	MOVLW h'00'			; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'40'			; next set Flags1 bits for Test0E
	MOVWF Flags1
	MOVLW h'42'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL NchorPch		; now check if it's Nch or Pch
	CALL VgsPolarity	; also set correct Vgs polarity
	RETURN				; then return
SetTest0F:
	; routine to setup for Test0F (SCR BVaks)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'12'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'40'			; next set Flags1 bits for Test0F
	MOVWF Flags1
	MOVLW h'01'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	RETURN				; then return
SetTest10:
	; routine to setup for Test10 (SCR or PUT Iaks with OPV)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'24'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'40'			; next set Flags1 bits for Test10
	MOVWF Flags1
	MOVLW h'02'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL SCRorPUT		; now check if it's an SCR or a PUT
	RETURN				; then return
SetTest11:
	; routine to setup for Test11 (SCR or PUT Iak with Ig=50uA)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'36'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'44'			; next set Flags1 bits for Test11
	MOVWF Flags1
	MOVLW h'02'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL SCRorPUT		; now check if it's an SCR or a PUT
	CALL SCRPUTIbPol	; also setting Ibias polarity
	RETURN				; then return
SetTest12:
	; routine to setup for Test12 (SCR or PUT Iak with Ig=200uA)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'48'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'54'			; next set Flags1 bits for Test12
	MOVWF Flags1
	MOVLW h'02'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL SCRorPUT		; now check if it's an SCR or a PUT
	CALL SCRPUTIbPol	; also setting Ibias polarity
	RETURN				; then return
SetTest13:
	; routine to setup for Test13 (SCR or PUT Iak with Ig=1mA)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'5A'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'4C'			; next set Flags1 bits for Test13
	MOVWF Flags1
	MOVLW h'02'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL SCRorPUT		; now check if it's an SCR or a PUT
	CALL SCRPUTIbPol	; also setting Ibias polarity
	RETURN				; then return
SetTest14:
	; routine to setup for Test14 (SCR or PUT Vak ON, (OPV))
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'6C'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'4C'			; next set Flags1 bits for Test14
	MOVWF Flags1
	MOVLW h'02'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	CALL SCRorPUT		; now check if it's an SCR or a PUT
	CALL SCRPUTIbPol	; also setting Ibias polarity
	RETURN				; then return
SetTest15:
	; routine to setup for Test15 (PUT BVaks)
	CALL ShowTestL1	    ; first go and display correct L1
	MOVLW h'7E'		 	; then set correct offset for L2
	CALL ShowTestaltL2	; and display it also
	MOVLW h'41'			; next set Flags1 bits for Test15
	MOVWF Flags1
	MOVLW h'01'			; and also Flags2 bits
	MOVWF Flags2		; (for relays setup)
	RETURN				; then return
;
;	***********************************************************
;
ShowTestL1:
;	routine to show correct line 1 for a test, according to the
;	device type chosen
	MOVLW h'80'			; first set address to line 1, char 0
	CALL P1DispAddress	; (also delays for 160us)
	MOVLW h'01'			; first see if DevNum = 01
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 01)
	GOTO $+3			; otherwise keep looking
	MOVLW h'7E'			; DevNum=01, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'02'			; DevNum not 01, so try 02
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 02)
	GOTO $+3			; otherwise keep looking
	MOVLW h'90'			; DevNum=02, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'03'			; DevNum not 02, so try 03
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 03)
	GOTO $+3			; otherwise keep looking
	MOVLW h'A2'			; DevNum=03, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'04'			; DevNum not 03, so try 04
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 04)
	GOTO $+3			; otherwise keep looking
	MOVLW h'B4'			; DevNum=04, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'05'			; DevNum not 04, so try 05
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 05)
	GOTO $+3			; otherwise keep looking
	MOVLW h'C6'			; DevNum=05, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'06'			; DevNum not 05, so try 06
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 06)
	GOTO $+3			; otherwise keep looking
	MOVLW h'D8'			; DevNum=06, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'07'			; DevNum not 06, so try 07
	XORWF DevNum,0		; w will = 0 if it's a match
	BTFSS STATUS,Z		; skip if Z=1 (ie, DevNum = 07)
	GOTO $+3			; otherwise keep looking
	MOVLW h'EA'			; DevNum=03, so load its L1 offset
	GOTO FoundL1		; and go to display it
	MOVLW h'00'			; DevNum must=08, so load its L1 offset
	GOTO FoundaltL1		; and go to display it

FoundL1:
	MOVWF DSpointer		; save offset in char pointer
	CALL P1DispString03	; then go send string to LCD
	MOVLW h'C0'			; finally move down to line 2
	CALL P1DispAddress	; of LCD 
	RETURN				; before returning
FoundaltL1:
	MOVWF DSpointer		; save offset in char pointer
	CALL DispString04	; then go send string to LCD
	MOVLW h'C0'			; finally move down to line 2
	CALL P1DispAddress	; of LCD 
	RETURN				; before returning
;
;	***********************************************************
;
ShowTestL2:
	MOVWF DSpointer		; save offset in char counter
	CALL DispString04	; then go send string to LCD
	RETURN

ShowTestaltL2:
	MOVWF DSpointer		; save offset in char counter
	CALL DispString05	; then go send string to LCD
	RETURN
;
;	***********************************************************
;
IgPolarity:
;	routine to check for NPN or PNP, set Ig polarity accordingly
	MOVLW h'04'		; first check if it's a PNP (DevNum=04)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because PNP)
	GOTO $+3		; must be NPN, so skip to fix
	BSF Flags2,7	; PNP, so set for RLY6 on
	RETURN			; then return
	BSF Flags1,5	; NPN, so set for RLY5 on
	RETURN			; then return
;
NchorPch:
;	routine to check if MOSFET is N-ch or P-ch, set Flags2,5
	MOVLW h'06'		; first check if it's a P-ch (DevNum=06)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because P-ch)
	RETURN			; must be NPN, so just return
	BSF Flags2,4	; P-ch, so set for RLY12 on
	RETURN			; then return;
;
NPNorPNPCheck:
;	routine to check if BJT is NPN or PNP, set Flags2,2 to suit
	MOVLW h'04'		; first check if it's a PNP (DevNum=04)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because PNP)
	RETURN			; must be NPN, so just return
	BSF Flags2,2	; PNP, so set for RLY10 on
	RETURN			; then return
;
SCRorPUT:
;	routine to check if device is an SCR or a PUT, set flags
	MOVLW h'08'		; first check if it's a PUT (DevNum=08)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because PUT)
	RETURN			; must be an SCR, so just return
	BSF Flags1,0	; PUT, so set for RLY15 on
	RETURN			; then return
;
SCRPUTIbPol;
;	routine to check for SCR or PUT, to set Ibias polarity
	MOVLW h'08'		; first check if it's a PUT (DevNum=08)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because PUT)
	GOTO $+3		; must be SCR, so skip over two
	BSF Flags2,7	; PUT, so set for RLY6 on
	RETURN			; then return
	BSF Flags1,5	; SCR, so set for RLY5 on instead
	RETURN			; then return
;
VgsPolarity;
;	routine to check for Nch or Pch MOSFET, set Vgs polarity
	MOVLW h'06'		; first check if it's a P-ch (DevNum=06)
	XORWF DevNum,0	; w will now = 0 if it is
	BTFSS STATUS,Z	; skip if Z=1 (ie match because P-ch)
	RETURN			; must be NPN, so just return
	BSF Flags2,5	; P-ch, so set for RLY14 on
	RETURN			; then return
;
;	***********************************************************
;
P1DispString03:
	; routine to display a 16-char string from the third set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called
	; (Duplicate of routine in page 0, to avoid page crossing)

	MOVLW h'12'			; first prepare to call Dstring03
	MOVWF PCLATH		; (which starts at 1200h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring03		; & call lookup table to get char
	Ppage1
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P1DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO P1DispString03	; and loop back to continue
;
;	***********************************************************
;
DispString04:
	; routine to display a 16-char string from the fourth set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called

	MOVLW h'13'			; first prepare to call Dstring04
	MOVWF PCLATH		; (which starts at 1300h in page2) 
	MOVF DSpointer,w	; then load next char's offset into w
	CALL Dstring04		; and call lookup table for char
	Ppage1
	ADDLW h'00'			; add zero to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P1DisplayData	; returned byte was a char, so send it off
	INCF DSpointer,1	; increment pointer
	GOTO DispString04	; then loop back to continue
;
;	***********************************************************
;
DispString05:
	; routine to display a 16-char string from the fifth set
	; of <14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called

	MOVLW h'14'			; first prepare to call Dstring05
	MOVWF PCLATH		; (which starts at 1400h in page2) 
	MOVF DSpointer,w	; then load next char's offset into w
	CALL Dstring05		; and call lookup table for char
	Ppage1
	ADDLW h'00'			; add zero to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1 (returned byte was 00h), so leave
	CALL P1DisplayData	; returned byte was a char, so send it off
	INCF DSpointer,1	; increment pointer
	GOTO DispString05	; then loop back to continue
;
;	***********************************************************
;
VdevshFix:
; 	routine to correct Vdev reading for volts across current shunt.
;	Scaled (0-1028) 24-bit FP Vdev value should be in JARG regs from
;	ReadVdev on entry, also scaled (0000-2066) 24-bit FP Vshunt in KARG
;	regs (from ReadIdev). Returns with corrected Vdev value saved in
;	both AARG regs and JARG regs (for use by Vgs correction)
;
	CALL VshuntNormalise	; first make sure Vshunt is scaled 0-2066
	copy KEXP,AEXP		; then bring it into AARG regs
	copy KARGB0,AARGB0
	copy KARGB1,AARGB1
	BTFSC Flags2,1		; now are we on high Vdev range or low range?
	GOTO LoVdevRng		; flag set, so we're on low range. Go thataway
HiVdevRng:
	MOVLW h'75'			; high range, so load BARG regs with 0.001d 
	MOVWF BEXP			; in 24-bit FP form (pre-calc using Fprep)
	MOVLW h'03'			; 75 03 12 = 0.0009999871
	MOVWF BARGB0
	MOVLW h'12'
	MOVWF BARGB1
	GOTO ScalenSubtract	; then go do the scaling
LoVdevRng:
	MOVLW h'78'			; low range, so prepare to multiply by 0.01d
	MOVWF BEXP			; by loading BARGregs with 0.01d in 24-bit
	MOVLW h'23'			; FP form (pre-calc using Fprep)
	MOVWF BARGB0		; 78 23 D7 = 0.00999999
	MOVLW h'D7'
	MOVWF BARGB1
ScalenSubtract:
	Ppage0
	CALL FPM24			; now do the scaling (AARG*BARG -> AARG)
	Ppage1
	CALL P1Check4FPE	; (duck away to check for any FP errors)
	copy AEXP,BEXP		; now move scaled Vshunt into BARG regs
	copy AARGB0,BARGB0
	copy AARGB1,BARGB1
	copy JEXP,AEXP		; and fetch Vdev into AARG regs
	copy JARGB0,AARGB0
	copy JARGB1,AARGB1
	Ppage0
	CALL FPS24			; then subtract Vsh from Vdev (AARG <- AARG-BARG)
	Ppage1
	CALL P1Check4FPE	; (duck away to check for any FP errors)
	copy AEXP,JEXP		; AARG regs now have corrected Vdev, but copy them
	copy AARGB0,JARGB0	; into JARG regs again before
	copy AARGB1,JARGB1 
	RETURN				; we return (with corrected Vdev in AARG regs)
;
;	***********************************************************
;
VgsshFix:
;	routine to correct Vgs reading for N-ch MOSFETs, to allow for
;	effect of Vsh - volts drop across current shunt. Enters with
;	uncorrected Vgs (0000 - 1033) in AARG regs, raw Vshunt
;	(0000 - 2066) in KARG regs. Leaves with corrected Vgs in AARG
;	regs after downscaling Vsh and then subtracting it from Vgs

	copy AEXP, JEXP		; first save Vgs temporarily in JARG regs
	copy AARGB0,JARGB0	; (copy of Vdev not needed for testing
	copy AARGB1,JARGB1	;  N-ch MOSFETs via Test0E, only for P-ch devs)
	CALL VshuntNormalise	; then make sure Vshunt is scaled 0-2066
	copy KEXP,AEXP		; now bring Vshunt into AARG regs
	copy KARGB0,AARGB0
	copy KARGB1,AARGB1
	MOVLW h'78'			; then load BARG regs with 0.01d 
	MOVWF BEXP			; in 24-bit FP form (pre-calc using Fprep)
	MOVLW h'23'			; 78 23 D7 = 0.00999999
	MOVWF BARGB0
	MOVLW h'D7'
	MOVWF BARGB1
	Ppage0
	CALL FPM24			; now multiply them (AARG<-AARG*BARG)
	Ppage1
	CALL P1Check4FPE	; (duck away to check for any FP errors)
	copy AEXP,BEXP		; now move downscaled Vshunt into BARG regs
	copy AARGB0,BARGB0
	copy AARGB1,BARGB1
	copy JEXP,AEXP		; then fetch Vgs back into AARG regs
	copy JARGB0,AARGB0
	copy JARGB1,AARGB1
	Ppage0
	CALL FPS24			; then do the subtraction (AARG<-AARG-BARG)
	Ppage1
	CALL P1Check4FPE	; (duck away to check for any FP errors)
	RETURN				; return with corrected Vgs in AARG regs
;
;	***********************************************************
;
VshuntNormalise:
	; routine to make sure Vshunt (in KARG regs) is scaled
	; 0000-2066 (0-2.066V) regardless of Idev range used to
	; derive it. So if high range was used, it is scaled back
	BTFSS Flags1,6		; first check if we were on high Idev range
	RETURN				; if not, just return (already OK)
	copy KEXP,AEXP		; but if yes, bring Vsh into AARG regs
	copy KARGB0,AARGB0
	copy KARGB1,AARGB1
	MOVLW h'7D'			; then load correction factor into BARG
	MOVWF BEXP			; regs. 7D47A9 = 0.38996125, which is very
	MOVLW h'47'			; close to 2066/5298 (= 7D47A8A3)
	MOVWF BARGB0
	MOVLW h'A9'
	MOVWF BARGB1		; now AARG & BARG regs have Vsh & CF
	Ppage0
	CALL FPM24			; so multiply them (AARG<-AARG*BARG)
	Ppage1
	CALL P1Check4FPE	; (duck away to check for any FP errors)
	copy AEXP,KEXP		; then return scaled back Vsh to K regs
	copy AARGB0,KARGB0
	copy AARGB1,KARGB1	; (also still in AARG regs)
	RETURN				; before returning
;
;	***********************************************************
;
P1Check4FPE:
	; routine to check if floating point functions had errors
	IORLW h'00'			; check if w came back with 00h or FFh
	BTFSC STATUS,Z		; if Z=0, must have been FFh (= an error)
	RETURN				; if Z=1, must have been 00h (= no error)
	CALL P1ClearLCD		; was an error, so prepare to show it
	MOVLW h'80'			; next set address to line 1, char 0
	CALL P1DispAddress	; (also delays for 160us)
	MOVLW "F"			; then send "FP Error!"
	CALL P1DisplayData
	MOVLW "P"
	CALL P1DisplayData
	MOVLW " "
	CALL P1DisplayData
	MOVLW "E"
	CALL P1DisplayData
	MOVLW "r"
	CALL P1DisplayData
	MOVLW "r"
	CALL P1DisplayData
	MOVLW "o" 
	CALL P1DisplayData
	MOVLW "r"
	CALL P1DisplayData
	MOVLW "!"
	CALL P1DisplayData
	CALL P1Wait2pt4sec	; now pause for 2.4 seconds for user to see
	CALL P1ClearLCD		; then wipe away again
	CALL P1Display2		; restore normal display
	RETURN				; and then return to resume anyway
;
;	************************************************************
;	More duplicates of routines in page 0, again to minimise
;	page crossings...
;
P1ClearLCD:
	;routine to clear LCD and reset address counter
	MOVLW h'01'			; clear display & reset addr ptr
	CALL P1DispAddress
	CALL P1Delay200ms	; pause 200ms to give it time to clear
	CALL P1Delay200ms	; and again, for really sloooow LCDs
	RETURN				; then return
;
;	************************************************************
;
P1Delay1ms:
	;routine to delay approx 1ms (2058 x 0.5us = 1029us) before return
	MOVLW h'0A'			; set Counter1 for 10 outer loops
	MOVWF Counter1
P1OuterLoop:
	MOVLW h'42'			; and Counter2 for 66 inner loops
	MOVWF Counter2		; (66 x 3mc = 198mc, + 7mc = 205)
	DECFSZ Counter2, 1	; decrement Counter2 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	DECFSZ Counter1, 1	; did reach zero, so decrement Counter1
	GOTO P1OuterLoop	; didn't hit zero, so loop back
	RETURN				; reached zero (10 x 66 loops) so return

P1Delay200ms:
	;routine to delay approx 200ms before returning
	MOVLW h'C8'			; load w for 200 outer loops
	GOTO $+2			; then go set Counter3
P1Delay10ms:
	;routine to delay approx 10ms before returning
	MOVLW h'0A'			; load w for 10 outer loops
	MOVWF Counter3		; now set Counter3
	CALL P1Delay1ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
;
P1Wait2pt4sec:
;	routine to pause for about 2.4 seconds (12 x 200ms = 2.4s)
	MOVLW h'0C'			; set Counter4 for 12d loops
	GOTO $+4
P1Wait1sec:
;	routine to pause for about 1 second (5 x 200ms = 1s)
	MOVLW h'05'			; set Counter4 for 5 loops
	GOTO $+2
P1Waithalfasec:
;	routine to pause for about 0.5 second (3 x 200ms = 0.6s)
	MOVLW h'03'			; set Counter4 for 3 loops
	MOVWF Counter4
	CALL P1Delay200ms	; then wait for 200ms
	DECFSZ Counter4,1	; now decrement Counter4, skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; return when done
;
;	***********************************************************
;
P1Display2:
	; routine to display Menu Select text on LCD
	MOVLW h'80'			; first set address to line 1, char 0
	CALL P1DispAddress	; (also delays for 160us)
	MOVLW h'24'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P1DispString01	; then go send string to LCD
	MOVLW h'C0'			; now move down to line 2
	CALL P1DispAddress
	MOVLW h'36'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P1DispString01	; and go send that string to LCD
	RETURN				; before leaving
;
P1DispString01:
	; routine to display a 16-char string from the first set of
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called
 
	MOVLW h'10'			; first prepare to call Dstring01
	MOVWF PCLATH		; (which starts at 1000h in page2)
	MOVF DSpointer,0	; next load next char's offset into w
	CALL Dstring01		; & call lookup table to get char
	Ppage1				; then swing back to current page
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P1DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer for next char
	GOTO P1DispString01	; and loop back to continue
;
;
;	***********************************************************
;	***********************************************************
;	all LCD display strings are stored up here in page 2,
;	(in computed GOTO lookup tables), so
;	all CALLs to them must be preceded by preloading PCLATH
;	***********************************************************
;
	org h'1000'
Dstring01:
; 	start of first display string lookup table (h'1000')
;	so offset = h'00', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "S"		; string "SC Discrete Semi" follows
	RETLW "C"
	RETLW " "
	RETLW "D"
	RETLW "i"
	RETLW "s"
	RETLW "c"
	RETLW "r"
	RETLW "e" 
	RETLW "t"
	RETLW "e"
	RETLW " "
	RETLW "S"
	RETLW "e"
	RETLW "m"
	RETLW "i"
	RETLW h'00'		; end of string marker
; 	start of second display string L/U table (h'1012')
;	so offset = h'12', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "c"		; string "conductor Tester" follows
	RETLW "o"
	RETLW "n"
	RETLW "d"
	RETLW "u"
	RETLW "c"
	RETLW "t"
	RETLW "o"
	RETLW "r"
	RETLW " "
	RETLW "T"
	RETLW "e"
	RETLW "s"
	RETLW "t"
	RETLW "e"
	RETLW "r"
	RETLW h'00'		; end of string marker
;	start of third display string L/U table (h'1024')
;	so offset = h'24', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "P"		; string "Press MenuSelect" follows
	RETLW "r"
	RETLW "e"
	RETLW "s"
	RETLW "s"
	RETLW " "
	RETLW "M"
	RETLW "e"
	RETLW "n" 
	RETLW "u"
	RETLW "S"
	RETLW "e"
	RETLW "l"
	RETLW "e"
	RETLW "c"
	RETLW "t"
	RETLW h'00'		; end of string marker
;	start of fourth display string L/U table (h'1036')
;	so offset = h'36', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "b"		; string "button to begin:" follows
	RETLW "u"
	RETLW "t"
	RETLW "t"
	RETLW "o"
	RETLW "n"
	RETLW " "
	RETLW "t"
	RETLW "o"
	RETLW " "
	RETLW "b"
	RETLW "e"
	RETLW "g"
	RETLW "i"
	RETLW "n"
	RETLW ":"
	RETLW h'00'		; end of string marker
;	start of fifth display string L/U table (h'1048')
;	so offset = h'48', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "D"		; string "Device to Test:^" follows
	RETLW "e"
	RETLW "v"
	RETLW "i"
	RETLW "c"
	RETLW "e"
	RETLW " "
	RETLW "t"
	RETLW "o" 
	RETLW " "
	RETLW "T"
	RETLW "e"
	RETLW "s"
	RETLW "t"
	RETLW ":"
	RETLW "^"
	RETLW h'00'		; end of string marker
;	start of sixth display string L/U table (h'105A')
;	so offset = h'5A', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "1"		; string "1:Diode/Zener  v" follows
	RETLW ":"
	RETLW "D"
	RETLW "i"
	RETLW "o"
	RETLW "d"
	RETLW "e"
	RETLW "/"
	RETLW "Z"
	RETLW "e"
	RETLW "n"
	RETLW "e"
	RETLW "r"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of seventh display string L/U table (h'106C')
;	so offset = h'6C', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "2"		; string "2.LED          v" follows
	RETLW ":"
	RETLW "L"
	RETLW "E"
	RETLW "D"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 8th display string L/U table (h'107E')
;	so offset = h'7E', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "3"		; string "3:NPN bipolar  v" follows
	RETLW ":"
	RETLW "N"
	RETLW "P"
	RETLW "N"
	RETLW " "
	RETLW "b"
	RETLW "i"
	RETLW "p"
	RETLW "o"
	RETLW "l"
	RETLW "a"
	RETLW "r"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 9th display string L/U table (h'1090')
;	so offset = h'90', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "4"		; string "4:PNP bipolar  v" follows
	RETLW ":"
	RETLW "P"
	RETLW "N"
	RETLW "P"
	RETLW " "
	RETLW "b"
	RETLW "i"
	RETLW "p"
	RETLW "o"
	RETLW "l"
	RETLW "a"
	RETLW "r"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 10th display string L/U table (h'10A2')
;	so offset = h'A2', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "5"		; string "5:N-ch MOSFET  v" follows
	RETLW ":"
	RETLW "N"
	RETLW "-"
	RETLW "c"
	RETLW "h"
	RETLW " "
	RETLW "M"
	RETLW "O"
	RETLW "S"
	RETLW "F"
	RETLW "E"
	RETLW "T"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 11th display string L/U table (h'10B4')
;	so offset = h'B4', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "6"		; string "6:P-ch MOSFET  v" follows
	RETLW ":"
	RETLW "P"
	RETLW "-"
	RETLW "c"
	RETLW "h"
	RETLW " "
	RETLW "M"
	RETLW "O"
	RETLW "S"
	RETLW "F"
	RETLW "E"
	RETLW "T"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 12th display string L/U table (h'10C6')
;	so offset = h'C6', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "7"		; string "7:SCR          v" follows
	RETLW ":"
	RETLW "S"
	RETLW "C"
	RETLW "R"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 13th display string L/U table (h'10D8')
;	so offset = h'D8', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "8"		; string "8:PUT          v" follows
	RETLW ":"
	RETLW "P"
	RETLW "U"
	RETLW "T"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of 14th display string L/U table (h'10EA')
;	so offset = h'EA', calling Dstring01
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "T"		; string "Test parameter:^" follows
	RETLW "e"
	RETLW "s"
	RETLW "t"
	RETLW " "
	RETLW "p"
	RETLW "a"
	RETLW "r"
	RETLW "a"
	RETLW "m"
	RETLW "e"
	RETLW "t"
	RETLW "e"
	RETLW "r"
	RETLW ":"
	RETLW "^"
	RETLW h'00'		; end of string marker (h'10FB')
;
;	***********************************************************
	org h'1100'
Dstring02:
;	start of second set of display string, also
;	start of Test 01 display string L/U table (h'1100')
;	so offset = h'00', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Irev (BV)      v" follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW " "
	RETLW "("
	RETLW "B"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 02 display string L/U table (h'1112')
;	so offset = h'12', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Irev (OPV)     v" follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW " "
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 03 display string L/U table (h'1124')
;	so offset = h'24', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vforward drop  v" follows
	RETLW "f"
	RETLW "o"
	RETLW "r"
	RETLW "w"
	RETLW "a"
	RETLW "r"
	RETLW "d"
	RETLW " "
	RETLW "d"
	RETLW "r"
	RETLW "o"
	RETLW "p"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 04 display string L/U table (h'1136')
;	so offset = h'36', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vrev (BV-aval) v" follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW " "
	RETLW "("
	RETLW "B"
	RETLW "V"
	RETLW "-"
	RETLW "a"
	RETLW "v"
	RETLW "a"
	RETLW "l"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 05 display string L/U table (h'1148')
;	so offset = h'48', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVcbo (e o/c)  v" follows
	RETLW "V"
	RETLW "c"
	RETLW "b"
	RETLW "o"
	RETLW " "
	RETLW "("
	RETLW "e"
	RETLW " "
	RETLW "o"
	RETLW "/"
	RETLW "c"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 06 display string L/U table (h'115A')
;	so offset = h'5A', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVceo (b o/c)  v" follows
	RETLW "V"
	RETLW "c"
	RETLW "e"
	RETLW "o"
	RETLW " "
	RETLW "("
	RETLW "b"
	RETLW " "
	RETLW "o"
	RETLW "/"
	RETLW "c"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 07 display string L/U table (h'116C')
;	so offset = h'6C', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Icbo (e o/c)   v" follows
	RETLW "c"
	RETLW "b"
	RETLW "o"
	RETLW " "
	RETLW "("
	RETLW "e"
	RETLW " "
	RETLW "o"
	RETLW "/"
	RETLW "c"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 08 display string L/U table (h'117E')
;	so offset = h'7E', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iceo (b o/c)   v" follows
	RETLW "c"
	RETLW "e"
	RETLW "o"
	RETLW " "
	RETLW "("
	RETLW "b"
	RETLW " "
	RETLW "o"
	RETLW "/"
	RETLW "c"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 09 display string L/U table (h'1190')
;	so offset = h'90', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE (Ib=20uA)  v" follows
	RETLW "F"
	RETLW "E"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "="
	RETLW "2"
	RETLW "0"
	RETLW h'E4'		; code for greek mu symbol
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 0A display string L/U table (h'11A2')
;	so offset = h'A2', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE (Ib=100uA) v" follows
	RETLW "F"
	RETLW "E"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "="
	RETLW "1"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 0B display string L/U table (h'11B4')
;	so offset = h'B4', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE (Ib=500uA) v" follows
	RETLW "F"
	RETLW "E"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "="
	RETLW "5"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 0C display string L/U table (h'11C6')
;	so offset = h'C6', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVdss (BV)     v" follows
	RETLW "V"
	RETLW "d"
	RETLW "s"
	RETLW "s"
	RETLW " "
	RETLW "("
	RETLW "B"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 0D display string L/U table (h'11D8')
;	so offset = h'D8', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Idss (OPV)     v" follows
	RETLW "d"
	RETLW "s"
	RETLW "s"
	RETLW " "
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 0E display string L/U table (h'11EA')
;	so offset = h'EA', calling Dstring02
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Id v. Vgs (gm) v" follows
	RETLW "d"
	RETLW " "
	RETLW "v"
	RETLW "."
	RETLW " "
	RETLW "V"
	RETLW "g"
	RETLW "s"
	RETLW " "
	RETLW "("
	RETLW "g"
	RETLW "m"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker (h'11FB')
;
;	***********************************************************
	org h'1200'
Dstring03:
;	start of third set of display strings, also
;	start of Test 0F display string L/U table (h'1200')
;	so offset = h'00', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVaks(g-k shrt)v" follows
	RETLW "V"
	RETLW "a"
	RETLW "k"
	RETLW "s"
	RETLW "("
	RETLW "g"
	RETLW "-"
	RETLW "k"
	RETLW " "
	RETLW "s"
	RETLW "h"
	RETLW "r"
	RETLW "t"
	RETLW ")"
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 10 display string L/U table (h'1212')
;	so offset = h'12', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iaks (OPV)     v" follows
	RETLW "a"
	RETLW "k"
	RETLW "s"
	RETLW " "
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 11 display string L/U table (h'1224')
;	so offset = h'24', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak (Ig=20uA)  v" follows
	RETLW "a"
	RETLW "k"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "="
	RETLW "2"
	RETLW "0"
	RETLW h'E4'		; code for greek mu symbol
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 12 display string L/U table (h'1236')
;	so offset = h'36', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak (Ig=100uA) v" follows
	RETLW "a"
	RETLW "k"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "="
	RETLW "1"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 13 display string L/U table (h'1248')
;	so offset = h'48', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak (Ig=500uA) v" follows
	RETLW "a"
	RETLW "k"
	RETLW " "
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "="
	RETLW "5"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 14 display string L/U table (h'125A')
;	so offset = h'5A', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vak on (OPV)   v" follows
	RETLW "a"
	RETLW "k"
	RETLW " "
	RETLW "o"
	RETLW "n"
	RETLW " "
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of Test 15 display string L/U table (h'126C')
;	so offset = h'6C', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVaks(g-a shrt)v" follows
	RETLW "V"
	RETLW "a"
	RETLW "k"
	RETLW "s"
	RETLW "("
	RETLW "g"
	RETLW "-"
	RETLW "a"
	RETLW " "
	RETLW "s"
	RETLW "h"
	RETLW "r"
	RETLW "t"
	RETLW ")"
	RETLW "v"
	RETLW h'00'		; end of string marker
;	start of DevNum1 TestLine1 string L/U table (h'127E')
;	so offset = h'7E', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "D"		; string "Diode/Zener:    " follows
	RETLW "i"
	RETLW "o"
	RETLW "d"
	RETLW "e"
	RETLW "/"
	RETLW "Z"
	RETLW "e"
	RETLW "n"
	RETLW "e"
	RETLW "r"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum2 TestLine1 string L/U table (h'1290')
;	so offset = h'90', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "L"		; string "LED:            " follows
	RETLW "E"
	RETLW "D"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum3 TestLine1 string L/U table (h'12A2')
;	so offset = h'A2', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "N"		; string "NPN bipolar:    " follows
	RETLW "P"
	RETLW "N"
	RETLW " "
	RETLW "b"
	RETLW "i"
	RETLW "p"
	RETLW "o"
	RETLW "l"
	RETLW "a"
	RETLW "r"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum4 TestLine1 string L/U table (h'12B4')
;	so offset = h'B4', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "P"		; string "PNP bipolar:    " follows
	RETLW "N"
	RETLW "P"
	RETLW " "
	RETLW "b"
	RETLW "i"
	RETLW "p"
	RETLW "o"
	RETLW "l"
	RETLW "a"
	RETLW "r"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum5 TestLine1 string L/U table (h'12C6')
;	so offset = h'C6', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "N"		; string "N-ch MOSFET:    " follows
	RETLW "-"
	RETLW "c"
	RETLW "h"
	RETLW " "
	RETLW "M"
	RETLW "O"
	RETLW "S"
	RETLW "F"
	RETLW "E"
	RETLW "T"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum6 TestLine1 string L/U table (h'12D8')
;	so offset = h'D8', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "P"		; string "P-ch MOSFET:    " follows
	RETLW "-"
	RETLW "c"
	RETLW "h"
	RETLW " "
	RETLW "M"
	RETLW "O"
	RETLW "S"
	RETLW "F"
	RETLW "E"
	RETLW "T"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of DevNum7 TestLine1 string L/U table (h'12EA')
;	so offset = h'EA', calling Dstring03
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "S"		; string "SCR:            " follows
	RETLW "C"
	RETLW "R"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker (h'12FB')
;
;	***********************************************************
	org h'1300'
Dstring04:
;	start of fourth set of display strings, also
;	start of DevNum8 TestLine1 string L/U table (h'1300')
;	so offset = h'00', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "P"		; string "PUT:            " follows
	RETLW "U"
	RETLW "T"
	RETLW ":"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of Test01 Line2 string L/U table (h'1312')
;	so offset = h'12', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Irev(BV) =00.0mA" follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW "("
	RETLW "B"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW "="
	RETLW "0"		; dummy digits start here (CA)
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "m"
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test02 Line2 string L/U table (h'1324')
;	so offset = h'24', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Irev(OPV)= 000uA" follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CA)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test03 Line2 string L/U table (h'1336')
;	so offset = h'36', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vf=0.0V/If=00.0m" follows
	RETLW "f"
	RETLW "="
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "V"
	RETLW "/"
	RETLW "I"
	RETLW "f"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "m"
	RETLW h'00'		; end of string marker
;	start of Test04 Line2 string L/U table (h'1348')
;	so offset = h'48', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vrev(BV)= 000V  " follows
	RETLW "r"
	RETLW "e"
	RETLW "v"
	RETLW "("
	RETLW "B"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (C9)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of Test05 Line2 string L/U table (h'135A')
;	so offset = h'5A', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW " "		; string " BVcbo =  000V  " follows
	RETLW "B"
	RETLW "V"
	RETLW "c"
	RETLW "b"
	RETLW "o"
	RETLW " "
	RETLW "="
	RETLW " "
	RETLW " "		; dummy digits start here (C9)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of Test06 Line2 string L/U table (h'136C')
;	so offset = h'6C', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW " "		; string " BVceo =  000V  " follows
	RETLW "B"
	RETLW "V"
	RETLW "c"
	RETLW "e"
	RETLW "o"
	RETLW " "
	RETLW "="
	RETLW " "
	RETLW " "		; dummy digits start here (C9)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of Test07 Line2 string L/U table (h'137E')
;	so offset = h'7E', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Icbo(OPV)= 000uA" follows
	RETLW "c"
	RETLW "b"
	RETLW "o"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CA)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test08 Line2 string L/U table (h'1390')
;	so offset = h'90', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iceo(OPV)= 000uA" follows
	RETLW "c"
	RETLW "e"
	RETLW "o"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CA)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test09 Line2 string L/U table (h'13A2')
;	so offset = h'A2', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE(Ib20uA)=0000" follows
	RETLW "F"
	RETLW "E"
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "2"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'00'		; end of string marker		
;	start of Test0A Line2 string L/U table (h'13B4')
;	so offset = h'B4', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE(Ib100u)=0000" follows
	RETLW "F"
	RETLW "E"
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "1"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW ")"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'00'		; end of string marker
;	start of Test0B Line2 string L/U table (h'13C6')
;	so offset = h'C6', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "h"		; string "hFE(Ib500u)=0000" follows
	RETLW "F"
	RETLW "E"
	RETLW "("
	RETLW "I"
	RETLW "b"
	RETLW "5"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW ")"
	RETLW " "
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'00'		; end of string marker		
;	start of Test0C Line2 string L/U table (h'13D8')
;	so offset = h'D8', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW " "		; string " BVdss =  000V  " follows
	RETLW "B"
	RETLW "V"
	RETLW "d"
	RETLW "s"
	RETLW "s"
	RETLW " "
	RETLW "="
	RETLW " "
	RETLW " "		; dummy digits start here (C9)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of Test0D Line2 string L/U table (h'13EA')
;	so offset = h'EA', calling Dstring04
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Idss(OPV)=00.0mA" follows
	RETLW "d"
	RETLW "s"
	RETLW "s"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW "0"		; dummy digits start here (CA)
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "m"
	RETLW "A"
	RETLW h'00'		; end of string marker (h'13FB')
;
;	***********************************************************
	org h'1400'								
Dstring05:
;	start of fifth set of display strings, also
;	start of Test0E Line2 string L/U table (h'1400')
;	so offset = h'00', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vgs=00.0V Id=00m" follows
	RETLW "g"
	RETLW "s"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW "I"
	RETLW "d"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "m"
	RETLW h'00'		; end of string marker
;	start of Test0F Line2 string L/U table (h'1412')
;	so offset = h'12', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVak(g-ks)= 000V" follows
	RETLW "V"
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "g"
	RETLW "-"
	RETLW "k"
	RETLW "s"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CB)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW h'00'		; end of string marker
;	start of Test10 Line2 string L/U table (h'1424')
;	so offset = h'24', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iaks(OPV)= 000uA" follows
	RETLW "a"
	RETLW "k"
	RETLW "s"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CA)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test11 Line2 string L/U table (h'1436')
;	so offset = h'36', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak(Ig20uA)=00mA" follows
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "2"
	RETLW "0"
	RETLW h'E4'
	RETLW "A"
	RETLW ")"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "m"
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test12 Line2 string L/U table (h'1448')
;	so offset = h'48', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak(Ig100u)=00mA" follows
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "1"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW ")"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "m"
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test13 Line2 string L/U table (h'145A')
;	so offset = h'5A', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "I"		; string "Iak(Ig500u)=00mA" follows
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "I"
	RETLW "g"
	RETLW "5"
	RETLW "0"
	RETLW "0"
	RETLW h'E4'
	RETLW ")"
	RETLW "="
	RETLW "0"
	RETLW "0"
	RETLW "m"
	RETLW "A"
	RETLW h'00'		; end of string marker
;	start of Test14 Line2 string L/U table (h'146C')
;	so offset = h'6C', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "V"		; string "Vak(OPV) = 00.0V" follows
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW ")"
	RETLW " "
	RETLW "="
	RETLW " "
	RETLW "0"		; dummy digits start here (CB)
	RETLW "0"
	RETLW "."
	RETLW "0"
	RETLW "V"
	RETLW h'00'		; end of string marker
;	start of Test15 Line2 string L/U table (h'147E')
;	so offset = h'7E', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "B"		; string "BVak(g-as)= 000V" follows
	RETLW "V"
	RETLW "a"
	RETLW "k"
	RETLW "("
	RETLW "g"
	RETLW "-"
	RETLW "a"
	RETLW "s"
	RETLW ")"
	RETLW "="
	RETLW " "		; dummy digits start here (CB)
	RETLW "0"
	RETLW "0"
	RETLW "0"
	RETLW "V"
	RETLW h'00'		; end of string marker
;	start of LED OPV warning top line L/U table (h'1490')
;	so offset = h'90', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "W"		; string "Warning: OPV not" follows
	RETLW "a"
	RETLW "r"
	RETLW "n"
	RETLW "i"
	RETLW "n"
	RETLW "g"
	RETLW ":"
	RETLW " "
	RETLW "O"
	RETLW "P"
	RETLW "V"
	RETLW " "
	RETLW "n"
	RETLW "o"
	RETLW "t"
	RETLW h'00'		; end of string marker
;	start of LED OPV warning line2 L/U table (h'14A2')
;	so offset = h'A2', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW " "		; string "   set to 10V   " follows
	RETLW " "
	RETLW " "
	RETLW "s"
	RETLW "e"
	RETLW "t"
	RETLW " "
	RETLW "t"
	RETLW "o"
	RETLW " "
	RETLW "1"
	RETLW "0"
	RETLW "V"
	RETLW " "
	RETLW " "
	RETLW " "
	RETLW h'00'		; end of string marker
;	start of 'HV Present' Line1 string L/U table (h'14B4')
;	so offset = h'B4', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW "W"		; string "WARNING: HiVolts" follows
	RETLW "A"
	RETLW "R"
	RETLW "N"
	RETLW "I"
	RETLW "N"
	RETLW "G"
	RETLW ":"
	RETLW " "
	RETLW "H"
	RETLW "i"
	RETLW "V"
	RETLW "o"
	RETLW "l"
	RETLW "t"
	RETLW "s"
	RETLW h'00'		; end of string marker
;	start of 'HV Present' Line2 string L/U table (h'14C6')
;	so offset = h'C6', calling Dstring05
	ADDWF PCL,F		; adds offset (in w) to PCL for compGOTO
	RETLW " "		; string " Still Present! " follows
	RETLW "S"
	RETLW "t"
	RETLW "i"
	RETLW "l"
	RETLW "l"
	RETLW " "
	RETLW "P"
	RETLW "r"
	RETLW "e"
	RETLW "s"
	RETLW "e"
	RETLW "n"
	RETLW "t"
	RETLW "!"
	RETLW " "

	RETLW h'00'		; end of string marker		
;
;	OTHER DISPLAY STRINGS COULD BE ADDED HERE (IF NEEDED)
;
;	***********************************************************
;	these routines moved up here to page 3 (above FPRF24.TXT
;	and the display srings) to relieve overcrowding in page 0
;	***********************************************************
;
	org h'1800'
ChooseTest:
;	routine to allow user to select which test to perform on selected
;	device. Returns with selected test ID number in TestNum.
;	This routine is called only from main loop

	copy TNummin, TestNum	; set TestNum to lowest number for device
ChTstloop:
	CALL DispCT2P		; to display choose-test screen
	CALL P3DoAScan		; now look for button press (S4,S5 or S6)
	MOVLW "4"			; then check if it was S4 ("4")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a 4)
	GOTO NotUpButton	; Z=0, so keep looking
	INCF TestNum,1		; was a 4, so increment TestNum
	MOVF TestNum,w		; now fetch TestNum into w and subtract
	SUBWF TNummax,0		; w now = TNummax - TestNum
	BTFSC STATUS,C		; skip if C=0 (result was negative)
	GOTO ChTstloop		; otherwise just loop back to display
	MOVF TNummin,w		; neg result, so swing around to TNummin
	MOVWF TestNum		; by saving TNummin in TestNum
	GOTO ChTstloop		; and then loop back
NotUpButton:
	MOVLW "5"			; not S4, so check if it was S5 ("5")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a 5)
	GOTO NotDownEither	; Z=0, so keep looking
	DECF TestNum,1		; was S5, so decrement TestNum
	MOVF TNummin,w		; now fetch min value into w and subtract
	SUBWF TestNum,0		; w now = TestNum - TNummin
	BTFSC STATUS,C		; skip if C=0 (result was negative)
	GOTO ChTstloop		; otherwise just loop back (0 or +)
	MOVF TNummax,w		; neg result, so swing around to TNummax
	MOVWF TestNum		; by saving TNummax in TestNum
	GOTO ChTstloop		; and then loop back
NotDownEither:
	MOVLW "6"			; not S5, so check if it was S6 ("6")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a 6)
	GOTO ChTstloop		; otherwise ignore & loop back
	MOVLW h'02'			; test selected, but check if DevNum = 2 (LED)
	XORWF DevNum,0		; w will be zero if DevNum = 2
	BTFSS STATUS,Z		; skip if that is so
	RETURN				; otherwise just return
	MOVLW h'02'			; DevNum = 2, but what about TestNum?
	XORWF TestNum,0		; w will be zero if TestNum = 2
	BTFSS STATUS,Z		; skip if so
	RETURN				; otherwise just return
	BTFSC PORTA,4		; yes, both LED & Test 02, so check S2 setting
	GOTO $+3			; if RA4 = 1, we need to warn user re OPV
	BTFSS PORTA,5		; RA4 was 0, so now check RA5
	RETURN				; it was 0 too, so S2 set for 10V - can leave
	CALL ShowWarning	; one of them was 1, so we need to warn user
	CALL P3Wait2pt4sec	; then pause a short time
	GOTO ChTstloop		; before looping back to allow another try
;
;	***********************************************************
;
DispCT2P:
	; routine to display choose-a-test info on LCD
	; (Second line changes according to test chosen)
	; This routine is called only by ChooseTest
;
	MOVLW h'80'			; first set address to line 1, char 0
	CALL P3DispAddress	; (also delays for 160us)
	MOVLW h'EA'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in string char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'C0'			; now move down to line 2
	CALL P3DispAddress
	; but then select correct version to suit current TestNum
	MOVLW h'01'			; first check if TestNum = 01
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 01)
	GOTO T01line2		; but if a match, go show T01line2
	MOVLW h'02'			; not 01, so try 02
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 02)
	GOTO T02line2		; but if a match, go show T02line2
	MOVLW h'03'			; not 02, so try 03
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 03)
	GOTO T03line2		; but if a match, go show T03line2	
	MOVLW h'04'			; not 03, so try 04
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 04)
	GOTO T04line2		; but if a match, go show T04line2
	MOVLW h'05'			; not 04, so try 05
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 05)
	GOTO T05line2		; but if a match, go show T05line2
	MOVLW h'06'			; not 05, so try 06
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 06)
	GOTO T06line2		; but if a match, go show T06line2
	MOVLW h'07'			; not 06, so try 07
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 07)
	GOTO T07line2		; but if a match, go show T07line2
	MOVLW h'08'			; not 07, so try 08
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 08)
	GOTO T08line2		; but if a match, go show T08line2
	MOVLW h'09'			; not 08, so try 09
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 09)
	GOTO T09line2		; but if a match, go show T09line2
	MOVLW h'0A'			; not 09, so try 0A
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0A)
	GOTO T0Aline2		; but if a match, go show T0Aline2
	MOVLW h'0B'			; not 0A, so try 0B
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0B)
	GOTO T0Bline2		; but if a match, go show T0Bline2
	MOVLW h'0C'			; not 0B, so try 0C
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0C)
	GOTO T0Cline2		; but if a match, go show T0Cline2
	MOVLW h'0D'			; not 0C, so try 0D
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0D)
	GOTO T0Dline2		; but if a match, go show T0Dline2
	MOVLW h'0E'			; not 0D, so try 0E
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0E)
	GOTO T0Eline2		; but if a match, go show T0Eline2
	MOVLW h'0F'			; not 0E, so try 0F
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 0F)
	GOTO T0Fline2		; but if a match, go show T0Fline2
	MOVLW h'10'			; not 0F, so try 10
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 10)
	GOTO T10line2		; but if a match, go show T10line2
	MOVLW h'11'			; not 10, so try 11
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 11)
	GOTO T11line2		; but if a match, go show T11line2
	MOVLW h'12'			; not 11, so try 12
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 12)
	GOTO T12line2		; but if a match, go show T12line2
	MOVLW h'13'			; not 12, so try 13
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 13)
	GOTO T13line2		; but if a match, go show T13line2
	MOVLW h'14'			; not 13, so try 14
	XORWF TestNum,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not 14)
	GOTO T14line2		; but if a match, go show T14line2
	GOTO T15line2		; must be 15, so go show T15line2
T01line2:
	; routine to display Test 01 second line
	MOVLW h'00'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in string char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T02line2:
	; routine to display Test 02 second line
	MOVLW h'12'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T03line2:
	; routine to display Test 03 second line
	MOVLW h'24'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T04line2:
	; routine to display Test 04 second line
	MOVLW h'36'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T05line2:
	; routine to display Test 05 second line
	MOVLW h'48'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T06line2:
	; routine to display Test 06 second line
	MOVLW h'5A'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T07line2:
	; routine to display Test 07 second line
	MOVLW h'6C'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T08line2:
	; routine to display Test 08 second line
	MOVLW h'7E'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T09line2:
	; routine to display Test 09 second line
	MOVLW h'90'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Aline2:
	; routine to display Test 0A second line
	MOVLW h'A2'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Bline2:
	; routine to display Test 0B second line
	MOVLW h'B4'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Cline2:
	; routine to display Test 0C second line
	MOVLW h'C6'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Dline2:
	; routine to display Test 0D second line
	MOVLW h'D8'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Eline2:
	; routine to display Test 0E second line
	MOVLW h'EA'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString02	; then go send string to LCD
	RETURN				; before leaving
T0Fline2:
	; routine to display Test 0F second line
	MOVLW h'00'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T10line2:
	; routine to display Test 10 second line
	MOVLW h'12'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T11line2:
	; routine to display Test 11 second line
	MOVLW h'24'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T12line2:
	; routine to display Test 12 second line
	MOVLW h'36'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T13line2:
	; routine to display Test 13 second line
	MOVLW h'48'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T14line2:
	; routine to display Test 14 second line
	MOVLW h'5A'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving
T15line2:
	; routine to display Test 15 second line
	MOVLW h'6C'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString03	; then go send string to LCD
	RETURN				; before leaving

;	************************************************************
;	
DelP31ms:
;   routine to delay approx 1ms (2058 x 0.5us = 1029us) before return
;	(duplicate of routine on page 0, to reduce page crossings)
	MOVLW h'0A'			; set Counter1 for 10 outer loops
	MOVWF Counter1
P3OuterLoop:
	MOVLW h'42'			; and Counter2 for 66 inner loops
	MOVWF Counter2		; (66 x 3mc = 198mc, + 7mc = 205)
	DECFSZ Counter2, 1	; decrement Counter2 & skip when zero
	GOTO $-1			; not zero yet, so loop back
	DECFSZ Counter1, 1	; did reach zero, so decrement Counter1
	GOTO P3OuterLoop	; didn't hit zero, so loop back
	RETURN				; reached zero (10 x 66 loops) so return

DelP3200ms:
;   routine to delay approx 200ms before returning
;	(again duplicates routine on page 0)
	MOVLW h'C8'			; load w for 200 outer loops
	GOTO $+2			; then go set Counter3
DelP310ms:
;   routine to delay approx 10ms before returning
;	(again duplicates routine on page 0)
	MOVLW h'0A'			; load w for 10 outer loops
	MOVWF Counter3		; now set Counter3
	CALL DelP31ms		; then wait 1ms
	DECFSZ Counter3, 1	; then decrement Counter3 & skip when zero
	GOTO $-2			; not zero yet, so keep going
	RETURN				; reached zero, so return	
;
;	***********************************************************
;	
P3DispAddress:
	;routine to translate & load display address (in w reg) into LCD
	BCF PORTE,1			; first set RS pin of LCD low, for instr/addr
	CALL P3Nibbles2LCD	; then send addr/cmd nibbles to LCD
	BCF PORTE,1			; make sure RS is is left low
	GOTO P3BusyWait		; then jump to delay 250us before return
	
P3DisplayData:
	;routine to display a data byte in w reg at the current LCD address
	BSF PORTE,1			; RS pin of LCD high, for sending data
	CALL P3Nibbles2LCD	; then send data nibbles to LCD

P3BusyWait:
	; routine to wait until LCD module is not busy, after writing
	MOVLW h'7D'			; set delay counter for 125 loops
	MOVWF Counter5		; (should give about 125 x 4 x 0.5 = 250us)
	NOP
	DECFSZ Counter5,1	; decrement counter & skip when zero
	GOTO $-2			; loop back until we reach zero
	RETURN				; then return
;
;	***********************************************************
;
DispPaD2T:
	; routine to display Pick-device-to-test info on LCD
	; (Second line changes according to device selected)
	; this routine is called only by PickADevice
	; To begin we display first line "Device to Test:^"

	MOVLW h'80'			; set address to line 1, char 0
	CALL P3DispAddress	; (also delays for 160us)
	MOVLW h'48'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in string char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'C0'			; now move down to start of line 2
	CALL P3DispAddress
	; but then select correct version to suit current DUT
	MOVLW "D"			; first check if Devcode = "D"
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not a D)
	GOTO Secline1		; but if a match, go show Secline1
	MOVLW "L"			; not a D, so try an L
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not an L)
	GOTO Secline2		; but if it matched, go show Secline2
	MOVLW "N"			; not an L, so try an N
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not an N)
	GOTO Secline3		; but if it matched, go show Secline3	
	MOVLW "P"			; not an N, so try a P
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not a P)
	GOTO Secline4		; but if it matched, go show Secline4
	MOVLW "M"			; not an P, so try an M
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not an M)
	GOTO Secline5		; but if it matched, go show Secline5
	MOVLW "F"			; not an M, so try an F
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not an F)
	GOTO Secline6		; but if it matched, go show Secline6
	MOVLW "S"			; not an F, so try an S
	XORWF Devcode,0
	BTFSC STATUS,Z		; skip if Z=0 (i.e., not an S)
	GOTO Secline7		; but if it matched, go show Secline7
	CLRW				; must be a T, so clear w
	GOTO Secline8		; then go show Secline8		
Secline1:
	; routine to display "1:Diode/Zener  v" second line
	MOVLW h'5A'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'01'			; then save 1 in DevNum
	MOVWF DevNum
	MOVLW h'01'			; also make lowest TestNum = 01
	MOVWF TNummin
	MOVLW h'04'			; and make highest TestNum = 04
	MOVWF TNummax
	RETURN				; before returning
Secline2:
	; routine to display "2:LED          v" second line
	MOVLW h'6C'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'02'			; then save 2 in DevNum
	MOVWF DevNum
	MOVLW h'02'			; also make lowest TestNum = 02
	MOVWF TNummin
	MOVLW h'03'			; and make highest TestNum = 03
	MOVWF TNummax
	RETURN				; before returning
Secline3:
	; routine to display "3:NPN bipolar  v" second line
	MOVLW h'7E'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'03'			; then save 3 in DevNum
	MOVWF DevNum
	MOVLW h'05'			; also make lowest TestNum = 05
	MOVWF TNummin
	MOVLW h'0B'			; and make highest TestNum = 0B
	MOVWF TNummax
	RETURN				; before returning
Secline4:
	; routine to display "4:PNP bipolar  v" second line
	MOVLW h'90'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'04'			; then save 4 in DevNum
	MOVWF DevNum
	MOVLW h'05'			; also make lowest TestNum = 05
	MOVWF TNummin
	MOVLW h'0B'			; and make highest TestNum = 0B
	MOVWF TNummax
	RETURN				; before returning
Secline5:
	; routine to display "5:N-ch MOSFET  v" second line
	MOVLW h'A2'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'05'			; then save 5 in DevNum
	MOVWF DevNum
	MOVLW h'0C'			; also make lowest TestNum = 0C
	MOVWF TNummin
	MOVLW h'0E'			; and make highest TestNum = 0E
	MOVWF TNummax
	RETURN				; before returning
Secline6:
	; routine to display "6:P-ch MOSFET  v" second line
	MOVLW h'B4'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'06'			; then save 6 in DevNum
	MOVWF DevNum
	MOVLW h'0C'			; also make lowest TestNum = 0C
	MOVWF TNummin
	MOVLW h'0E'			; and make highest TestNum = 0E
	MOVWF TNummax
	RETURN				; before returning
Secline7:
	; routine to display "7:SCR          v" second line
	MOVLW h'C6'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'07'			; was an S, so save 7 in DevNum
	MOVWF DevNum
	MOVLW h'0F'			; also make lowest TestNum = 0F
	MOVWF TNummin
	MOVLW h'14'			; and make highest TestNum = 14
	MOVWF TNummax
	RETURN				; before returning
Secline8:
	; routine to  display "8:PUT         v" second line
	MOVLW h'D8'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in char pointer
	CALL P3DispString01	; then go send string to LCD
	MOVLW h'08'			; then save 8 in DevNum
	MOVWF DevNum
	MOVLW h'10'			; also make lowest TestNum = 10
	MOVWF TNummin
	MOVLW h'15'			; and make highest TestNum = 15
	MOVWF TNummax
	RETURN				; before leaving

;	***********************************************************
;
P3DispString01:
; 	routine to display a 16-char string from the FIRST set of
; 	of 14 strings (in page2) on the LCD. Offset for first
; 	char in string must be in DSpointer when routine called
;	(Duplicate of routine on page 0, to avoid page crossings)
 
	MOVLW h'10'			; first prepare to call Dstring01
	MOVWF PCLATH		; (which starts at 1000h in page2)
	MOVF DSpointer,0	; next load next char's offset into w
	CALL Dstring01		; & call lookup table to get char
	Ppage3
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P3DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer for next char
	GOTO P3DispString01	; and loop back to continue
;
;	***********************************************************
;
P3DispString02:
; 	routine to display a 16-char string from the second set
; 	of 14 strings (in page2) on the LCD. Offset for first
; 	char in string must be in DSpointer when routine called
;	(Duplicate of routine on page 0, to avoid page crossings)

	MOVLW h'11'			; first prepare to call Dstring02
	MOVWF PCLATH		; (which starts at 1100h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring02		; & call lookup table to get char
	Ppage3				; then return to page3
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P3DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO P3DispString02	; and loop back to continue
;
;	***********************************************************
;
P3DispString03:
	; routine to display a 16-char string from the third set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called
;	(duplicate of routine in page 0, to avoid page crossings)

	MOVLW h'12'			; first prepare to call Dstring03
	MOVWF PCLATH		; (which starts at 1200h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring03		; & call lookup table to get char
	Ppage3				; then return to page3
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P3DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO P3DispString03	; and loop back to continue
;
;	***********************************************************
;
P3DispString05:
	; routine to display a 16-char string from the fifth set
	; of 14 strings (in page2) on the LCD. Offset for first
	; char in string must be in DSpointer when routine called
;	(duplicate of routine in page 1, to avoid page crossings)

	MOVLW h'14'			; first prepare to call Dstring05
	MOVWF PCLATH		; (which starts at 1400h in page2)
	MOVF DSpointer,0	; then load next char's offset into w
	CALL Dstring05		; & call lookup table to get char
	Ppage3				; then return to page3
	ADDLW h'00'			; add 0 to w, to test for null char
	BTFSC STATUS,Z		; skip if Z=0 (non-zero char)
	RETURN				; Z=1, so returned byte was 00h, leave
	CALL P3DisplayData	; non-zero char, so send it off to the LCD
	INCF DSpointer,1	; then increment pointer
	GOTO P3DispString05	; and loop back to continue
;
;	***********************************************************
;
P3DoAScan:
; 	routine to scan front panel buttons via Scankeys, re-check
; 	after bounce delay and save in Keycode if valid key press
; 	detected, before returning (only after key pressed).
;	(duplicate of routine on page 0, to avoid page crossings)

	CALL P3ScanKeys		; go scan FP buttons
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS, Z		; skip if Z=0 (ie w not 0)
	GOTO P3DoAScan		; otherwise loop back & keep scanning
	MOVWF Keycode		; Z=0, so a button was pressed - save code
	CALL DelP3200ms		; then wait for 200ms to allow for bounce
	CALL P3ScanKeys		; & scan again to verify
	ADDLW h'00'			; test return value in w reg
	BTFSC STATUS,Z		; skip if Z=0 (ie w not 0, because it has a keycode)
	GOTO P3DoAScan		; Z=1, so just a false alarm - loop back to start 
	XORWF Keycode, 0	; Z=0, so compare with first code (w -> 0 if match)
	BTFSS STATUS,Z		; skip if Z=1 (because codes did match)
	GOTO P3DoAScan		; no match - loop back to start
	RETURN				; matched, so exit (with valid char in Keycode)
;
;	*************************************************************
;
HVSPTest:
;	routine to check if HV still present at ZIF socket, after RELAY1
;	has been turned off at end of test. If HV still present and above
;	30V, displays a warning message and keeps checking. Only when Vdev
;	falls below 30V does it exit by returning to the main prog loop.
	BCF PORTD,1			; first turn off RELAY8 so we're on hi range
HVCheck:
	Ppage0				; then back to page 0 so we can
	CALL ReadVdev		; measure Vdev
	Ppage3
	MOVLW h'30'			; now is Thousands = 0 (30 in ASCII)
	XORWF Thousands,W	; Z will = 1 if it is
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO ShowHVWarning	; otherwise go show warning & check again
	MOVLW h'30'			; Thousands = 0, but now check Hundreds
	XORWF Hundreds,W
	BTFSS STATUS,Z
	GOTO ShowHVWarning	; if Hundreds not 0, go show warning etc
	MOVLW h'33'			; Hundreds = 0, but now see if Tens <3
	SUBWF Tens,W		; by subtracting 33 (3 in ASCII)
	BTFSS STATUS,C		; C=1 means that Tens = 3 or more
	RETURN				; but C=0 means Tens <3, so we can return
ShowHVWarning:
	MOVLW h'80'			; set LCD address to line 1, char 0
	CALL P3DispAddress	; (also delays for 160us)
	MOVLW h'B4'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in string char pointer
	CALL P3DispString05	; then go send string to LCD
	MOVLW h'C0'			; now move down to start of line 2
	CALL P3DispAddress
	MOVLW h'C6'			; then set offset for line 2 start
	MOVWF DSpointer
	CALL P3DispString05	; and go send that string to LCD also
	GOTO HVCheck		; then loop back to check again	 

;	*************************************************************
;
P3Nibbles2LCD:
; 	routine to test bits of data byte (passed in w) then send
; 	to LCD module as two nibbles (high nibble first)
; 	(duplicate of routine on page 0, to avoid page crossings)
	MOVWF Temp2			; first save byte to be sent in Temp2
	BTFSC Temp2,7		; now test bit 7 (upper nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,6		; now test bit 6
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,5		; now test bit 5
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,4		; now test bit 4
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; now toggle EN to write hi nibble to LCD

	BTFSC Temp2,3		; next test bit 3 (lower nibble)
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,0			; if it's a 0, clear RC0
	GOTO $+2			; and proceed to next bit
	BSF PORTC,0			; was a 1, so set RC0 instead
	BTFSC Temp2,2		; now test bit 2
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,1			; if it's a 0, clear RC1
	GOTO $+2			; and proceed to next bit
	BSF PORTC,1			; was a 1, so set RC1 instead
	BTFSC Temp2,1		; now test bit 1
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,2			; if it's a 0, clear RC2
	GOTO $+2			; and proceed to next bit
	BSF PORTC,2			; was a 1, so set RC2 instead
	BTFSC Temp2,0		; now test bit 0
	GOTO $+3			; if it's a 1, skip down
	BCF PORTC,3			; if it's a 0, clear RC3
	GOTO $+2			; and proceed to next bit
	BSF PORTC,3			; was a 1, so set RC3 instead
	ToggleEN			; toggle EN again to write low nibble to LCD
	RETURN				; and return, since both nibbles sent
;
;	*************************************************************
;
Number2Code:
;	routine to convert current DevNum to equiv Devcode
;	(returns with Devcode in w)
	MOVLW h'01'			; first see if DevNum = 1
	XORWF DevNum,0
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "D"			; was 1, so return with D in w

	MOVLW h'02'			; not 1, so now see if it's 2
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "L"			; was 2, so return with L in w

	MOVLW h'03'			; not 2, so now see if it's 3
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "N"			; was 3, so return with N in w

	MOVLW h'04'			; not 3, so now see if it's 4
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "P"			; was 4, so return with P in w

	MOVLW h'05'			; not 4, so now see if it's 5
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "M"			; was 5, so return with M in w

	MOVLW h'06'			; not 5, so now see if it's 6
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "F"			; was 6, so return with F in w

	MOVLW h'07'			; not 6, so now see if it's 7
	XORWF DevNum,0	
	BTFSS STATUS,Z		; skip if we get a match (Z=1)
	GOTO $+2			; otherwise move on
	RETLW "S"			; was 7, so return with S in w
	RETLW "T"			; must be 8, so return with T in w
;
;	************************************************************
;
PickADevice:
;	routine to allow user to select which type of device to be tested.
;	Returns with device code in Devcode: Diodes = "D", LEDs = "L",
;	NPN BJT = "N", PNP BJT = "P", N-ch MOSFET = "M", P-ch MOSFET = "F",
;	SCR = "S" and PUT = "T"
	CALL DispPaD2T		; display pick-a-device screen
	CALL P3DoAScan		; now look for button press (S4,S5 or S6)
	MOVLW "4"			; then check if it was S4 ("4")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (i.e., it was a 4)
	GOTO NotS4			; Z=0, so keep looking
	INCF DevNum,1		; was a 4, so increment DevNum
	MOVLW h'09'			; then check if DevNum = 9
	XORWF DevNum,0		; Z will =1 if it is
	BTFSS STATUS,Z		; so skip if Z=1
	GOTO $+3			; otherwise continue
	MOVLW h'01'			; was 9, so swing back to 1
	MOVWF DevNum		; by saving 1 in DevNum
	CALL Number2Code	; then convert back to Devcode
	MOVWF Devcode		; saving it also
	GOTO PickADevice	; then loop back to display
NotS4:
	MOVLW "5"			; not S4, so check if it was S5 ("5")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a 5)
	GOTO NotS5			; Z=0, so keep looking
	DECFSZ DevNum,1		; decrement DevNum, skip if it goes to zero
	GOTO $+3			; otherwise continue
	MOVLW h'08'			; was 0, so swing around to 8
	MOVWF DevNum		; by saving 8 in DevNum
	CALL Number2Code	; then convert back to Devcode
	MOVWF Devcode		; saving it also
	GOTO PickADevice	; then loop back to display
NotS5:
	MOVLW "6"			; not S5, so check if it was S6 ("6")
	XORWF Keycode,0
	BTFSS STATUS,Z		; skip if Z=1 (it was a 6)
	GOTO PickADevice	; otherwise ignore & loop back
	CALL DelP3200ms		; but it was a 6, so pause 200ms
	RETURN				; and then return
;
;	***********************************************************
;		
P3ScanKeys:
	; routine to look for a pressed key, return with ASCII code
	; in w (or with a null in w if no key is pressed)
;	(duplicate of routine on page0, to avoid page crossings)

	BTFSC PORTC, 4		; first check if RC4 = 0 (S3 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "3"			; S3 pressed, so return with "3" in w
	BTFSC PORTC, 7		; next check if RC7 = 0 (S4 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "4"			; S4 pressed, so return with "4" in w
	BTFSC PORTC, 5		; next check if RC5 = 0 (S5 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "5"			; S5 pressed, so return with "5" in w
	BTFSC PORTC, 6		; next check if RC6 = 0 (S6 pressed)
	GOTO $+2			; otherwise keep looking
	RETLW "6"			; S6 pressed, so return with "6" in w
	BTFSC PORTE, 2		; now check if RE2 = 0  (S7 pressed)
	RETLW h'00'			; no key pressed, so return with null in w
	RETLW "7"			; S7 pressed, so return with "7" in w
;
;	***********************************************************
;
ShowWarning:
	; routine to show warning about S2 not being set to '10V'
	; when DevNum=2 (LED) and TestNum=2 (Diode/LED Irev) chosen

	MOVLW h'80'			; set LCD address to line 1, char 0
	CALL P3DispAddress	; (also delays for 160us)
	MOVLW h'90'			; set offset for 1st char of string
	MOVWF DSpointer		; and save in string char pointer
	CALL P3DispString05	; then go send string to LCD
	MOVLW h'C0'			; now move down to start of line 2
	CALL P3DispAddress
	MOVLW h'A2'			; then set offset for line 2 start
	MOVWF DSpointer
	CALL P3DispString05	; and go send that string to LCD also
	RETURN				; then return - done
;
;	***********************************************************
;
P3Wait2pt4sec:
;	routine to pause for about 2.4 seconds (12 x 200ms = 2.4s)
	MOVLW h'0C'			; set Counter4 for 12d loops
	MOVWF Counter4
	CALL DelP3200ms		; then wait for 200ms
	DECFSZ Counter4,1	; now decrement Counter4, skip when zero
	GOTO $-2			; otherwise keep looping
	RETURN				; return when done
;
;	***********************************************************
;
;	finally we include floating point routines (in FPRF24.TXT)
;
;	***********************************************************
;
	#include 	<FPRF24.TXT> 

 	END

