//Display Module for Breadboard PSU
//see io.h for pinouts

#include "config.h"
#include <xc.h>
#include "io.h"
#include "hd44780.h"
#include "graphics.h"
#include "tmr0.h"
#include "nvm.h"

//data is little endian
#define V_FACTOR (7625)
#define I_FACTOR (625)
__EEPROM_DATA(low(V_FACTOR),high(V_FACTOR),low(I_FACTOR),high(I_FACTOR),low(V_FACTOR),high(V_FACTOR),low(I_FACTOR),high(I_FACTOR));
__EEPROM_DATA(low(V_FACTOR),high(V_FACTOR),low(I_FACTOR),high(I_FACTOR),low(V_FACTOR),high(V_FACTOR),low(I_FACTOR),high(I_FACTOR));
__EEPROM_DATA(low(V_FACTOR),high(V_FACTOR),low(V_FACTOR),high(V_FACTOR),low(V_FACTOR),high(V_FACTOR),low(V_FACTOR),high(V_FACTOR));
__EEPROM_DATA(low(I_FACTOR),high(I_FACTOR),low(I_FACTOR),high(I_FACTOR),low(V_FACTOR),high(V_FACTOR),0,0);

char buf[11]="";    //for outputting numbers
char dFlag=0;
char bFlag=0;
char t0cnt=0;
unsigned int aFactors[16];  //this comes from EEPROM on boot
unsigned char* t=(unsigned char*)aFactors;      //char array union/overlay
//debounced in interrupt
#define SDELAY 1
char s1press=0;
char s2press=0;
char s3press=0;
char buz1=0;
char buz2=0;
char error1=0;
char error2=0;
char screen=0;
char alert=0;
char saveDone=0;
unsigned int v1set,v2set,i1set,i2set;
unsigned int v1fb,v2fb,i1fb,i2fb;
unsigned long dv,power;
#define SCREENCOUNT (16)
#define CALCOUNT (15)
#define VISCALE (16)
const char screens[][21]={
    "********************", //dummy value
    "LEFT V SETPOINT     ",
    "LEFT I SETPOINT     ",
    "RIGHT V SETPOINT    ",
    "RIGHT I SETPOINT    ",
    "LEFT V ACTUAL       ",
    "LEFT I ACTUAL       ",
    "RIGHT V ACTUAL      ",
    "RIGHT I ACTUAL      ",
    "V1 INPUT            ",
    "V2 INPUT            ",
    "V3 INPUT            ",
    "V4 INPUT            ",
    "I1 INPUT            ",
    "I2 INPUT            ",
    "DC SUPPLY VOLTAGE   ",
    "SAVE TO EEPROM?     ",
};

const char adcChan[]={0,    //dummy for first
    V1SET,I1SET,V2SET,I2SET,
    V1FB,I1FB,V2FB,I2FB,
    VIN1,VIN2,VIN3,VIN4,
    IIN1,IIN2,DCINSENSE
};

const char units[]={' ',    //dummy for first
    'V','A','V','A',
    'V','A','V','A',
    'V','V','V','V',
    'A','A','V'
};

int main(void){    
    unsigned int i;
    for(i=0;i<sizeof(aFactors);i++){  //read from EEPROM
        t[i]=EEPROMread(i);
    }
    unsigned long d=0;
    ioinit();
    LCD_init();
    LCD_cgram(graphics);
    LCD_clear();
    tmr0init();
    GIE=1;
    while(1){
        //S3 handled independently of screen
        if(s3press){
            s3press=0;
            screen++;
            LCD_clear();
            if(screen>SCREENCOUNT){
                screen=0;
                s1press=0;
                s2press=0;
                saveDone=0; //reset
            }
            if(screen>0){
                LCD_setpos(LCD_LINE0+0);
                LCD_print("CALIBRATION:");
                LCD_setpos(LCD_LINE1+0);
                LCD_print(screens[screen]);
                if(screen==SCREENCOUNT){
                    LCD_setpos(LCD_LINE3+0);
                    LCD_print("PRESS S1 AND S2.");                                    
                }            
            }
        }
        //check S1 and S2 per screen
        if(screen==0){
            if(s1press){
                s1press=0;
                if(buz1){buz1=0;}else{buz1=1;}
            }
            if(s2press){
                s2press=0;
                if(buz2){buz2=0;}else{buz2=1;}
            }
        }else if(screen<=CALCOUNT){
            if(bFlag){
                bFlag=0;
                if(S1){
                    aFactors[screen-1]++;
                    if(aFactors[screen-1]>9999){aFactors[screen-1]=9999;}
                }
                if(S2){
                    aFactors[screen-1]--;
                    if(aFactors[screen-1]<100){aFactors[screen-1]=100;}
                }                
            }
        }else if(screen==SCREENCOUNT){
            if(S1 && S2 && (saveDone==0)){       //do save
                saveDone=1;
                for(i=0;i<sizeof(aFactors);i++){  //read from EEPROM
                    EEPROMmodify(i,t[i]);   
                }
                LCD_setpos(LCD_LINE3+0);
                LCD_print("SAVED TO EEPROM.");                
            }
        }
        //check for current limiting on channel 1
        v1set=adjust(getADC(V1SET),aFactors[0]);
        v1fb=adjust(getADC(V1FB),aFactors[4]);
        i1set=adjust(getADC(I1SET),aFactors[1]);
        i1fb=adjust(getADC(I1FB),aFactors[5]);        
        if((abs(i1set-i1fb)*VISCALE)<abs(v1set-v1fb)){  //current closer to setpoint> limiting
            LED1(LED_RED);
            error1=1;
        }else{
            LED1(LED_GREEN);            
            error1=0;
        }
        //check for current limiting on channel 2
        v2set=adjust(getADC(V2SET),aFactors[2]);
        v2fb=adjust(getADC(V2FB),aFactors[6]);
        i2set=adjust(getADC(I2SET),aFactors[3]);
        i2fb=adjust(getADC(I2FB),aFactors[7]);        
        if((abs(i2set-i2fb)*VISCALE)<abs(v2set-v2fb)){  //current closer to setpoint> limiting
            LED2(LED_RED);
            error2=1;
        }else{
            LED2(LED_GREEN);            
            error2=0;
        }
        //test to sound buzzer
        if((buz1 && error1) || (buz2 && error2)){
            alert=1;
        }else{
            alert=0;
        }
        if(screen==0){
            if(dFlag){
                dFlag=0;
                LCD_setpos(LCD_LINE0+0);
                LCD_data('S');        
                d=adjust(getADC(V1SET),aFactors[0]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(I1SET),aFactors[1]);
                showI(d);
                LCD_data(A_CHAR);        
                d=adjust(getADC(V2SET),aFactors[2]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(I2SET),aFactors[3]);
                showI(d);

                LCD_setpos(LCD_LINE1+0);
                LCD_data('A');        
                d=adjust(getADC(V1FB),aFactors[4]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(I1FB),aFactors[5]);
                showI(d);
                LCD_data(A_CHAR);        
                d=adjust(getADC(V2FB),aFactors[6]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(I2FB),aFactors[7]);
                showI(d);

                LCD_setpos(LCD_LINE2+0);
                d=adjust(getADC(VIN1),aFactors[8]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(VIN2),aFactors[9]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(VIN3),aFactors[10]);
                showV(d);
                LCD_data(V_CHAR);        
                d=adjust(getADC(VIN4),aFactors[11]);
                showV(d);
                LCD_data(V_CHAR);        

                LCD_setpos(LCD_LINE3+0);
                d=adjust(getADC(IIN1),aFactors[12]);
                showI(d);
                LCD_data(A_CHAR);        
                d=adjust(getADC(IIN2),aFactors[13]);
                showI(d);
                LCD_data(A_CHAR);        

                LCD_data(' ');        
                if(buz1){LCD_data(SPK_ON);}else{LCD_data(SPK_OFF);}
                LCD_data(' ');        
                if(buz2){LCD_data(SPK_ON);}else{LCD_data(SPK_OFF);}
                LCD_data(' ');        

                d=adjust(getADC(DCINSENSE),aFactors[14]);
                if(t0cnt&16){
                    showV(d);
                    LCD_data(V_CHAR);        
                }else{
                    if(d<=v1fb){    //handle negative
                        LCD_data('0');                                
                    }else{
                        dv=d-v1fb;
                        power=(dv*(long)i1fb);
                        power=power/1000000UL;
                        if(power>9){power=9;}
                        LCD_data('0'+power);                                
                    }
                    LCD_data('W');        
                    LCD_data(' ');        
                    if(d<=v2fb){    //handle negative
                        LCD_data('0');                                
                    }else{
                        dv=d-v2fb;
                        power=(dv*(long)i2fb);
                        power=power/1000000UL;
                        if(power>9){power=9;}
                        LCD_data('0'+power);                                
                    }
                    LCD_data('W');                            
                }
            }
        }else if(screen<=CALCOUNT){
            d=adjust(getADC(adcChan[screen]),aFactors[screen-1]);
            LCD_setpos(LCD_LINE2+0);
            showM(d);
            LCD_data(units[screen]);                    
            LCD_setpos(LCD_LINE3+0);
            d=aFactors[screen-1];
            showN(d);
        }else{
            //no other actions needed
        }
    }
}

void __interrupt() isr(void){
    static char s1tmout=0;
    static char s2tmout=0;
    static char s3tmout=0;    
    if(TMR0IE && TMR0IF){
        TMR0IF=0;               //clear
        t0cnt++;
        bFlag=1;                    //button hold update at 7Hz
        if(alert & ((t0cnt&7)==0)){
            PIEZO=1;
        }else{
            PIEZO=0;            
        }
        if((t0cnt&3)==0){dFlag=1;}  //display update at ~2Hz
        if(S1){
            if(s1tmout==0){s1press=1;}
            s1tmout=SDELAY;
        }else{
            if(s1tmout){s1tmout--;}
        }
        if(S2){
            if(s2tmout==0){s2press=1;}
            s2tmout=SDELAY;
        }else{
            if(s2tmout){s2tmout--;}
        }
        if(S3){
            if(s3tmout==0){s3press=1;}
            s3tmout=SDELAY;
        }else{
            if(s3tmout){s3tmout--;}
        }
    }
}