//interface for HD44780 LCD using 4 bit mode
#include "hd44780.h"

//pinout for RevA PCB:
//2     RD4 LCDD7
//3     RD5 LCDD6
//4     RD6 LCDD5
//5     RD7 LCDD4
//8     RB0 LCDEN
//9     RB1 LCDRS

void LCD_init(void){
    //turn on power
    //TRISDbits.TRISD6=0; //LCD_PWR
    //LCD_PWR=1;
    //set up pins and do LCD initialisation
    LCD_EN=0;
    LCD_RS=0;           //default to commands for init
    TRISDbits.TRISD4=0; //D7
    TRISDbits.TRISD5=0; //D6
    TRISDbits.TRISD6=0; //D5
    TRISDbits.TRISD7=0; //D4
    TRISBbits.TRISB0=0; //EN
    TRISBbits.TRISB1=0; //RS    
    __delay_ms(10);        //powerup delay
    LCD_D7=0;
    LCD_D6=0;
    LCD_D5=1;
    LCD_D4=0;   //0=> 4-bit mode
    LCD_EN=1;
    LCD_EN=0;   //clock
    __delay_ms(1);
    LCD_RS=1;           //default to data
    LCD_clear();
    LCD_cmd(0b00000110);         //increment on write, no shift
    LCD_cmd(0b00001100);         //display on, cursor off
    LCD_cmd(0b00101000);         //4 bit mode, 2-line font
    LCD_cmd(0b10000000);         //start of data ram (ready for output)
}

void LCD_deinit(void){      //for low power mode
    LCD_EN=0;
    LCD_RS=0;
    LCD_D7=0;
    LCD_D6=0;
    LCD_D5=0;
    LCD_D4=0;   //all low to avoid parasitic drain
    //LCD_PWR=0;  //off    
}

void LCD_clear(void){
    LCD_cmd(0b00000001);         //clear
    __delay_ms(10);    
}

void LCD_data(unsigned char d){
    LCD_D7=(d&128)?1:0;
    LCD_D6=(d&64)?1:0;
    LCD_D5=(d&32)?1:0;
    LCD_D4=(d&16)?1:0;
    LCD_EN=1;
    LCD_EN=0;   //clock
    LCD_D7=(d&8)?1:0;
    LCD_D6=(d&4)?1:0;
    LCD_D5=(d&2)?1:0;
    LCD_D4=(d&1)?1:0;   //needs 4 bytes less than the obvious (d&1)
    LCD_EN=1;
    LCD_EN=0;   //clock    
    __delay_ms(1);      //all commands/data take max 37us except return home (0x2)
}

void LCD_cmd(unsigned char c){
    LCD_RS=0;           //clear for command
    LCD_data(c);
    LCD_RS=1;           //default to data    
}

void LCD_print(const char *c){
    while(*c){
        LCD_data(*c++);
    }    
}

void LCD_setpos(unsigned char x){
    LCD_cmd((x & 0x7F)|0x80);
}

void LCD_cgram(const unsigned char* p){
    unsigned char i,j;
    LCD_cmd(0b01000000);    //start of CGRAM
    for(i=0;i<64;i++){
        j=p[i];
        LCD_data(j);     //fill entire CGRAM
    }
    LCD_cmd(0b10000000);    //start of data ram (ready for output)    
}

unsigned long places(signed char n){
    unsigned long r=1;
    while(n--){r=r*10;}
    return r;    
}

char getDigit(unsigned long n, signed char p){  //get digit at pth decimal place (0 = ones)
    signed char res,pstart,zf;
    zf=0;
    pstart=9;
    while(pstart>=p){
        res=0;
        while(n>=places(pstart)){
            n=n-places(pstart);
            res++;
            zf=1;   //non zero digit detected
        }
        pstart=pstart-1;
    }
    if((zf==0)&&(res==0)&&(p!=0)){return F_SPACE;}  //zero blanking, ones position is never blanked
    return res+'0';
}

char lzFix(char c){ //suppress leading zero suppression for display reasons
    if(c==F_SPACE){return '0';}
    return c;
}

void LCD_ulong(unsigned long n, char p){    //output unsigned long to p places using getDigit
    while(p--){
        LCD_data(getDigit(n,p));
    }
}

void LCD_clr(void){                 //separate function to include necessary delay, also does a home
    LCD_cmd(LCD_CLR);
    __delay_ms(2);
}

void getDigits(unsigned long n,char* t){
    char p;
    unsigned long d=1000000000UL;
    for(p=0;p<10;p++){t[p]='0';}    //clear
    t[10]=0;
    p=0;
    while(p<10){
        if(n>=d){
            n=n-d;
            t[p]=t[p]+1;            
        }else{
            d=d/10;
            p++;
        }
    }
}

void lzFixDigits(char* t,char s){
    char p=0;
    while(p<s){
        if(t[p]=='0'){
            t[p]=' ';
            p=p+1;
        }else{
            return;
        }
    }
    return;
}

void showV(unsigned int v){
    getDigits(v,buf);
    lzFixDigits(buf,6);
    LCD_data(buf[5]);
    LCD_data(buf[6]);
    LCD_data('.');
    LCD_data(buf[7]);
    //LCD_data(buf[8]);
    //LCD_data(buf[9]);
    //LCD_data('V');
}

void showI(unsigned int v){
    getDigits(v,buf);
    lzFixDigits(buf,6);
    LCD_data(buf[6]);
    LCD_data('.');
    LCD_data(buf[7]);
    LCD_data(buf[8]);
    //LCD_data(buf[9]);
    //LCD_data('I');
}

void showM(unsigned int n){
    getDigits(n,buf);
    lzFixDigits(buf,6);
    LCD_data(buf[5]);
    LCD_data(buf[6]);
    LCD_data('.');
    LCD_data(buf[7]);
    LCD_data(buf[8]);
    LCD_data(buf[9]);    
}

void showN(unsigned int n){
    getDigits(n,buf);
    lzFixDigits(buf,9);
    LCD_data(buf[5]);
    LCD_data(buf[6]);
    LCD_data(buf[7]);
    LCD_data(buf[8]);
    LCD_data(buf[9]);    
}