; 
; Uses assembly code for PIC16F88 microcontroller
; A 3-digit oxygen sensor display  
; Displays air/fuel ratio on three digit display . 


	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations


EEPROM1		equ	H'00'	; for display brightness minumum (BRIGHT_MIN)

EEPROM2		equ	H'01'	; Upper limit value for 5V (UPPER_LIM0) ms byte
EEPROM3		equ	H'02'	; Upper limit value for 5V (UPPER_LIM1) ls byte
EEPROM4		equ	H'03'	; Lower limit value for 0V (LOWER_LIM0) ms byte
EEPROM5		equ	H'04'	; Lower limit value for 0V (LOWER_LIM1) ls byte

EEPROM6		equ	H'05'	; Decimal Point value(DECIMAL_P)

; Bank 0 RAM

DISP1		equ	H'20'	; working storage for Display1 
DISP2		equ	H'21'	; working storage for Display2 
DISP3		equ	H'22'	; working storage for Display3 
BIN_0		equ	H'23'	; binary value ms byte
BIN_1		equ	H'24'	; binary value ls byte
BCD_0		equ	H'25'	; BCD value left digit
BCD_1		equ	H'26'	; BCD value middle digit
BCD_2		equ	H'27'	; BCD value right digit
TEMP		equ	H'28'	; temporary register
CNT_16		equ	H'29'	; counter for BCD conversion
BRIGHT		equ	H'2A'	; current brightness level
BRIGHT_DIM	equ	H'2B'	; brightness threshold
BRIGHT_MIN	equ	H'2C'	; brightness minimum 
STORE		equ	H'2D'	; delay counter for A/D conversion
INPUT0		equ	H'2E'	; input value ms
INPUT1		equ	H'2F'	; input value ls
TIMER0		equ	H'30'	; timer ms
TIMER1		equ	H'31'	; timer ls
SETTINGS	equ	H'32'	; display settings
FLAG_1		equ	H'33'	; alternate interrupt rate for brightness control
PORTA_STO	equ	H'34' 	; switch closed detect
MODE		equ	H'35'	; display mode
DISP1X		equ	H'36'	; multiplexing storage for Display1 
DISP2X		equ	H'37'	; multiplexing storage for Display2 
DISP3X		equ	H'38'	; multiplexing storage for Display3 
UPDATE0		equ	H'39'	; digits update counter	ms
UPDATE1		equ	H'3A'	; digits update counter ls
ZRO_FLG		equ	H'3B'	; leading zero flag
UPPER_LIM0	equ	H'3C'	; Upper limit value ms byte values
UPPER_LIM1	equ	H'3D'	; Upper limit value ls byte
LOWER_LIM0	equ	H'3E'	; Lower limit value ms byte
LOWER_LIM1	equ	H'3F'	; Lower limit value ls byte
DECIMAL_P	equ	H'40'	; decimal point value
SW_FLG		equ	H'41'	; switch flag
TRAN_FLG 	equ	H'42'	; allow transfer in interrupt flag
NEGATIVE	equ	H'43'	; negative flag 
SPAN0		equ	H'44'	; value span ms byte
SPAN1		equ	H'45'	; value span ls byte
		
; math routines
REMB0		equ	H'63'	; remainder
REMB1		equ	H'64'
TEMP1		equ H'65'	; temporary
TEMPD		equ	H'66'
AARGB3		equ	H'67'	; ls of argument A
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
TEMPB1      equ H'6B'	; temporary
TEMPB0      equ H'6C'
BARGB1      equ H'6D'	; 
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; loop counter

; All Banks RAM
STATUS_TMP 	equ H'70'	; temp storage for status during interrupt
W_TMP		equ	H'71'	; temporary storage for w during interrupt


; preprogram EEPROM DATA 
	
 	ORG     H'2100'
 
	DE		D'14'		; EEPROM1, for display brightness minimum (BRIGHT_MIN)

	DE		H'00'		; EEPROM2, Upper limit value (UPPER_LIM0) ms byte (lambda 1.84)
	DE		D'184'		; EEPROM3, Upper limit value (UPPER_LIM1) ls byte
	DE		H'00'		; EEPROM4, Lower limit value (LOWER_LIM0) ms byte (lambda 0.70)
	DE		D'70'		; EEPROM5, Lower limit value (LOWER_LIM1) ls byte

	DE		H'01'		; EEPROM6, Decimal Point position (DECIMAL_P)
	

; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4			; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT	; start of interrupt routine

; **************************************************************************
; lookup table subroutine to get seven segment display data. 
;  			RB76543210
;             fgae.cdb
;				  |-------------------------------- decimal point RB3	
;             01001000 ; for a '0'. other numbers as per below in SEVEN_SEG
; 0 is segment on, 1 is segment off

SEVEN_SEG
	addwf	PCL,f		; add value of display to program counter
; numbers 0-9
; segments    fgae.cdb	
	retlw 	B'01001000'	; 7-segment code 0 
	retlw 	B'11111010'	; 7-segment code 1
	retlw 	B'10001100'	; 7-segment code 2
	retlw 	B'10011000'	; 7-segment code 3
	retlw 	B'00111010'	; 7-segment code 4
	retlw 	B'00011001'	; 7-segment code 5 
	retlw 	B'00001001'	; 7-segment code 6
	retlw 	B'11011010'	; 7-segment code 7
	retlw 	B'00001000'	; 7-segment code 8
	retlw 	B'00011010'	; 7-segment code 9


; ******************************************************************************

; interrupt from counter used to multiplex display
; this sucessively switches on Disp1, Disp2, Disp3, Disp4 in sequence 

; start interrupt by saving w and status registers before altered by interrupt routine
INTERRUPT	
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; bank 0
	bcf		STATUS,RP1

INT_BY
	bcf		INTCON,TMR0IF
; update timer
	btfss	TRAN_FLG,0 	; if set allow transfer 
	goto	DELAY_TIMERS
	decfsz	UPDATE0,f	; update timer ms
	goto	DELAY_TIMERS
	decfsz	UPDATE1,f	; update timer for digits 
	goto	DELAY_TIMERS
	movlw	D'02'		; digits update time period
	movwf	UPDATE1	
; transfer values at each update
	movf	DISP1,w
	movwf	DISP1X		; multiplexing display value
	movf	DISP2,w
	movwf	DISP2X		; multiplexing display value
	movf	DISP3,w
	movwf	DISP3X		; multiplexing display value

DELAY_TIMERS
; delay timers
	decfsz	TIMER0,f	; ms byte timer
	goto	BRIGHTNESS
	movf	TIMER1,w
	btfss	STATUS,Z	; if zero do not decrease
	decf	TIMER1,f	; decrease to 0

BRIGHTNESS
; brightness control
	comf	BRIGHT,w	; if brightness at maximum run full drive
	btfsc	STATUS,Z
	goto	MULTIPLEX
; if not full brightness alternate
	btfsc	FLAG_1,0	; check bit 0
	goto	OFF_OUT
	bsf		FLAG_1,0	; set if clear
	movf	BRIGHT,w	; modulation off time to w
	movwf	TMR0		; 4MHz so off time is inverse of 4MHz/4/2/(FF-TMR0) 
	movlw	H'FF'
	movwf	PORTB		; displays off
	goto	RECLAIM
OFF_OUT
	comf	BRIGHT,w	; on time
	movwf	TMR0		; 4MHz so on time is inverse of 4MHz/4/2/(FF-TMR0) 
	bcf		FLAG_1,0	; clear if set

; multiplex display routine
MULTIPLEX
	btfss	PORTA,6		; skip if display 1 not lit
	goto	ALL_OFF		; all displays off for short period
	btfss	PORTA,1		; skip if display 2 not lit
	goto	LIT2		; display 2 lit
	btfss	PORTA,0		; skip if display 3 not lit				
	goto	LIT3		; display 3 lit
	btfss	PORTA,7		; skip if display 4 not lit
	goto	LIT4
LIT1
	bsf		PORTA,6		; disp1 off
	movf	DISP2X,w	; disp2 to w
	movwf	PORTB		; seven segment value to portB
	bcf		PORTA,1		; disp2 powered
	bsf		FLAG_1,6	; new multiplex flag
	goto	RECLAIM		; end of multiplex
LIT4
	bsf		PORTA,7		; disp4 off
	movf	DISP1X,w	; disp1	to w
	movwf	PORTB
	bcf		PORTA,6		; disp1 powered
	bsf		FLAG_1,5	; new multiplex flag
	goto 	RECLAIM
LIT2
	bsf		PORTA,1		; disp2 off
	movf	DISP3X,w	; disp3 to w
	movwf	PORTB		; portB has seven-segment code 
	bcf		PORTA,0		; disp3 powered
	bsf		FLAG_1,4	; new multiplex flag
	goto	RECLAIM		; end of multiplex
LIT3
	bsf		PORTA,0		; disp3 off
	movlw	H'FF'		; 
	movwf 	PORTB
	bcf		PORTA,7		; switch S4 powered
	bsf		FLAG_1,3	; new multiplex flag
	goto	RECLAIM
ALL_OFF
	bsf		PORTA,6		; disp1 off
	movlw	D'50'
	movwf	TMR0		; 4MHz so on time is inverse of 4MHz/4/2/(HFF-D200) 
	bcf		FLAG_1,0	; clear so multiplex dimming runs next
	bsf		FLAG_1,7	; displays are all off flag

; end of interrupt reclaim w and status 

RECLAIM
	btfsc	INTCON,TMR0IF; if interrupt flag set redo interrupt
	goto	INT_BY	
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;***************************************************************************************

; RESET		
; Set ports A & B

MAIN
; initial values
	clrf	FLAG_1		; multiplex flag
	clrf	MODE		; display mode
	clrf	SETTINGS	; settings
	clrf	TIMER1
	clrf	TIMER0		; delay timers
	movlw	D'01'		; update set at startup
	movwf	UPDATE1
	movlw	B'11111111'	; code for 7-segment displays off
	movwf 	DISP1		; display values
	movwf	DISP2		; display value
	movwf	DISP3		; display value

	movwf 	DISP1X		; multiplex initial display value
	movwf	DISP2X		; multiplex initial display value
	movwf	DISP3X		; multiplex initial display value

	movlw	D'200'		; value can be changed	
	movwf	BRIGHT_DIM	; set brightness threshold

	clrf	BRIGHT		; display brightness
	clrf	SW_FLG		; switch flag
	bsf		TRAN_FLG,0 	; allow transfer in interrupt

; set inputs/outputs
	movlw	H'FF'		; set all outputs high
	movwf	PORTA
	movwf	PORTB
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'00000000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'00000000'	; 
	movwf	OPTION_REG	; TMRO prescaler is 2, 
	movlw   B'00111100'	; I/O RA0,1,6,7 outputs
	movwf   TRISA		; port A data direction register

; analog inputs, A/D
	movlw	B'00011100'	;  AN2,3,4 are analog inputs 
	movwf	ANSEL
	movlw	B'10000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0

	movlw	B'01010000'	; Fosc, channel 2 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01101000'	; for 4MHz
	movwf	OSCCON		; osc
	bcf		STATUS,RP0	; select memory bank 0

; obtain EEPROM values

EEPROM_LOADING
; set display off first
	movlw	H'FF'
	movwf	PORTA
	movwf	PORTB		; set high 

; Brightness minimum 
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movwf	BRIGHT_MIN	; 

; Upper limit value
	movlw	EEPROM2
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIM0	; ms btye
	movlw	EEPROM3
	call	EEREAD		; sets EEADR
	movwf	UPPER_LIM1	; ls btye

; Lower limit value
	movlw	EEPROM4
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIM0	; ms btye
	movlw	EEPROM5
	call	EEREAD		; sets EEADR
	movwf	LOWER_LIM1	; ls btye

; decimal point value
	movlw	EEPROM6
	call	EEREAD		; sets EEADR
	movwf	DECIMAL_P

; interrupt enable and program now runs
	bsf		INTCON,TMR0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE		; set global interrupt enable for above

RUN ; program run
	movf	TIMER1,w	; switch delay timer if zero can recheck switches
	btfss	STATUS,Z
	goto	NO_SWITCH

; wait for new multiplex flag
; Switch 1
	bcf		FLAG_1,5
WAIT_SET5
	btfss	FLAG_1,5	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET5
	bcf		INTCON,GIE	; stop interrupt

; A/D conversion
; channel 2
; CH2 bits ADCON bits 5,4,3 are 0,1,0 for switch
	bcf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_5
	decfsz	STORE,f
	goto	WAIT20_5	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should be 3)	
	btfss	STATUS,C
	goto	FLG6
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG6; Switch 2
	btfss	MODE,0		; if mode is clear then bypass this switch
	goto	FLG3
; wait for new multiplex flag
	bcf		FLAG_1,6
WAIT_SET6
	btfss	FLAG_1,6	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET6
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_6
	decfsz	STORE,f
	goto	WAIT20_6
	
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt

	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	FLG4
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG4; Switch 3
; wait for new multiplex flag
	bcf		FLAG_1,4
WAIT_SET4
	btfss	FLAG_1,4	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET4
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_4
	decfsz	STORE,f
	goto	WAIT20_4	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt

	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	sublw	D'174'		; check if >174 (0.4V below 5V) (ignoring ADRESH as it should always be 3)	
	btfss	STATUS,C
	goto	FLG3
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	goto	SWITCH_ACT

FLG3; Switch 4
; wait for new multiplex flag
	bcf		FLAG_1,3
WAIT_SET3
	btfss	FLAG_1,3	; flag set in multiplex routine in interrupt signifying new multiplex
	goto	WAIT_SET3
	bcf		INTCON,GIE	; stop interrupt
	
; wait 20us to charge input capacitance
	movlw	D'5'
	movwf	STORE
WAIT20_3
	decfsz	STORE,f
	goto	WAIT20_3	
		
	call	ACQUIRE_AD
	bsf		INTCON,GIE	; allow interrupt
	movf	ADRESH,w
	sublw	D'1'		; check if above or below 2.5V) (ignoring ADRESL)	
	btfss	STATUS,C
	goto	NO_SWITCH
	movf	PORTA,w
	movwf	PORTA_STO	; get closed switch
	btfsc	MODE,0		; check mode
	goto	SWITCH_ACT	; act on switch closure

NO_SWITCH

; BRIGHTNESS level channel 3
; CH3 bits 5,4,3 are 0,1,1 LDR

	bsf		ADCON0,3
	bsf		ADCON0,4
	bcf		ADCON0,5
	
; wait > 20us to charge input capacitance
	movlw	D'20'
	movwf	STORE
WAIT20_CH3
	decfsz	STORE,f
	goto	WAIT20_CH3	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w
	bcf		STATUS,RP0	; select memory bank 0
	movwf	STORE		; store in bank0 RAM
	rrf		ADRESH,f	; ms byte. shift to ms bits in STORE 
	rrf		STORE,f
	rrf		ADRESH,f
	rrf		STORE,f		; shift to get ms 8-bits
	comf	STORE,f		; reverse sense as LDR gives low value at high brightness 	

; compare with threshold value

; if STORE > BRIGHT_DIM set at FF		
	movf	STORE,w
	subwf	BRIGHT_DIM,w; brightness threshold
	btfsc	STATUS,C
	goto	CALC_BRIGHT
SET_FF
	movlw	H'FF'		; or set at maximum
	goto	TRANSF

; calculate (STORE/BRIGHT_DIM) x FF for BRIGHT value for display
CALC_BRIGHT
	btfsc	STATUS,Z	; if STORE = BRIGHT_DIM set at FF 
	goto	SET_FF
; Multiply Store x FF (shift left by 1 byte (8-bits))
	clrf	AARGB0
	movf	STORE,w
	movwf	AARGB1
	clrf	AARGB2
; Divide by BRIGHT_DIM
	clrf	BARGB0		; 
	movf	BRIGHT_DIM,w
	movwf	BARGB1
	call	FXD2416U
; result in AARGB0,1,2
	movf	AARGB2,w
; compare with minimum. If under, set at minimum 	
	
	subwf	BRIGHT_MIN,w; minimum brightness
	movf	AARGB2,w	; keep value
	btfsc	STATUS,C
	movf	BRIGHT_MIN,w; or set at minimum
; transfer to bright register for interrupt multiplexing
TRANSF
	movwf	BRIGHT		; display brightness

INPUT
; check mode
	movf	MODE,w		; if zero normal display if not zero settings display
	btfss	STATUS,Z
	goto	SETTINGS_RUN

; INPUT value channel 4
; CH4 bits 5,4,3 are 1,0,0 input value
	bcf		ADCON0,3
	bcf		ADCON0,4
	bsf		ADCON0,5

; wait for all displays off flag to be set in interrupt for a quiet reading	of value
	bcf		FLAG_1,7
WAIT_SET
	btfss	FLAG_1,7	; flag set in multiplex routine in interrupt signifying displays all off
	goto	WAIT_SET
	
; wait > 20us to charge input capacitance after quiet period
	movlw	D'100'
	movwf	STORE
WAIT20_CH4
	decfsz	STORE,f
	goto	WAIT20_CH4	
		
	call	ACQUIRE_AD
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls byte	
	bcf		STATUS,RP0	; select memory bank 0
	movwf	INPUT1		; store in bank0 RAM
	movf	ADRESH,w	; ms byte.  
	movwf	INPUT0


; Display calculation

; Positive (When upper > lower)
; SPAN = (Upper Limit - Lower Limit) 
; Display value = (SPAN x Input Value/H3FF) + Lower limit 

; Negative (When Lower > upper)
; SPAN = (Lower Limit - Upper Limit) 
; Display value = (SPAN x Complement of Input Value/H3FF) + Upper limit 

SUBTRACT
; subtract Lower Limit from Upper Limit
	clrf	NEGATIVE		; negative flag 
	movf	LOWER_LIM0,w	; ms lower limit
	subwf	UPPER_LIM0,w	; ms upper limit 
	movwf	SPAN0
	movf	LOWER_LIM1,w	; ls byte lower limit
	subwf	UPPER_LIM1,w	; ls upper limit 
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry
	btfss	SPAN0,7			; if set then 0V Lower limit > 5V upper limit so rearrange calculation
	goto	SPAN_AARGB

; subtract Upper Limit from Lower Limit
	movf	UPPER_LIM0,w	; ms upper limit 
	subwf	LOWER_LIM0,w	; ms lower limit
	movwf	SPAN0
	movf	UPPER_LIM1,w	; ls upper limit corrected
	subwf	LOWER_LIM1,w	; ls byte lower limit
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry	
	bsf		NEGATIVE,0		; set negative flag

; place SPAN in AARGB
SPAN_AARGB
	movf	SPAN0,w
	movwf	AARGB0
	movf	SPAN1,w
	movwf	AARGB1

; SPAN x input value
	btfss	NEGATIVE,0	; if negative flag use complement
	goto	USE_INPUT
USE_COMPLEMENT
; subtract INPUT0,INPUT1 from 3FF
	movf	INPUT0,w	; ms INPUT 
	sublw	H'3'		; ms maximum
	movwf	BARGB0
	movf	INPUT1,w	; ls INPUT
	sublw	H'FF'		; ls 
	movwf	BARGB1
	btfss	STATUS,C
	decf	BARGB0,f	; decrease if carry	
	movlw	B'00000011'
	andwf	BARGB0,f	; remove bits 2-7
	goto	MULT_SPAN_INPUT		
USE_INPUT
	movf	INPUT1,w
	movwf	BARGB1
	movf	INPUT0,w
	movwf	BARGB0
MULT_SPAN_INPUT
	call	FXM1616U	; multiply
;result in AARGB0,1,2,3
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2	
; Divide
	movlw	H'03'
	movwf	BARGB0		; for 1023
	movlw	H'FF'
	movwf	BARGB1
	call	FXD2416U
; result in AARGB0,1,2

	btfsc	NEGATIVE,0	; if negative flag add Upper limit	
	goto	ADD_UPPER
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	LOWER_LIM0,w	; lower limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	LOWER_LIM1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry
	goto	VAL_CONV

; Upper Limit < Lower limit so add upper limit
ADD_UPPER
; add lower limit
	movf	AARGB1,w	; ms byte 
	addwf	UPPER_LIM0,w	; lower limit ms value
	movwf	BIN_0
	movf	AARGB2,w	; ls byte 
	addwf	UPPER_LIM1,w
	movwf	BIN_1
	btfsc	STATUS,C
	incf	BIN_0,f		; increase if carry

VAL_CONV
; check overrange
	btfss	BIN_0,2		; if set over 3FF
	goto	VAL_CONV_NO_OVER
	movlw	H'03'
	movwf	BIN_0
	movlw	H'FF'
	movwf	BIN_1

VAL_CONV_NO_OVER
; convert to packed BCD	
; do not increase past D999, H3E7
	movf	BIN_0,w			; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	NOT_999
	movf	BIN_1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	BIN_1			; E7 to BIN_1
NOT_999
	movf	DECIMAL_P,w		; decimal point
	movwf	ZRO_FLG			; set flag for 0 suppression and dP
	call	BCD_AND_LOAD	; convert and load
	bsf		TRAN_FLG,0 		; allow transfer in interrupt

	goto 	RUN

; ************************************************************
SWITCH_ACT
; switch closed. 
; is TIMER1 clear if not. out
	movf	TIMER1,w	; switch timer is clear after switch delay
	btfss	STATUS,Z
	goto	RUN
	bsf		SW_FLG,0

; Check which switch is closed 	
; PORTA_STO has closed switch. S1 is RA6 low, S2 is RA1 low, S3 is RA0 low and S4 is RA7 low
	btfss	PORTA_STO,6
	goto	SWITCH1		; Mode
	btfss	PORTA_STO,1
	goto	SWITCH2		; settings
	btfss	PORTA_STO,0
	goto	SWITCH3		; down
	btfsc	PORTA_STO,7	; up
	goto	RUN			; no switch closed

SWITCH4; up
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'3'		; ls timer
	movwf	TIMER1

; check settings

; 0
	movf	SETTINGS,w
	xorlw	D'0'
	btfsc	STATUS,Z
	goto	DP_UP
; 1
	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	LOWER_UP
; 2
	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	UPPER_UP
; 3
	movf	SETTINGS,w
	xorlw	D'3'
	btfss	STATUS,Z
	goto	RUN

DIMMING_MIN_UP
; EEPROM1 for display brightness minimum 
	incfsz	BRIGHT_MIN,f
	goto	LOAD_MIN
	movlw	H'FF'
	movwf	BRIGHT_MIN	; stop at FF	
LOAD_MIN
	movlw	EEPROM1
	call	EEREAD
	movf	BRIGHT_MIN,w
	call	EEWRITE		; store in eeprom
	goto	RUN

DP_UP
DP_DN

;EEPROM6 Decimal Point for value(DECIMAL_P)
	incf	DECIMAL_P,f
	btfss	DECIMAL_P,0		; when bits 0 and 1 set clear register
	goto	BY_CLEAR_DEC	; do not clear 
	btfsc	DECIMAL_P,1		; when bits 0 and 1 set clear register
	clrf	DECIMAL_P
BY_CLEAR_DEC
	movlw	EEPROM6
	call	EEREAD
	movf	DECIMAL_P,w
	call	EEWRITE		; store in eeprom
	goto	RUN


LOWER_UP
;EEPROM4 Lower limit value (LOWER_LIM0) ms byte
;EEPROM5 Lower limit value (LOWER_LIM1) ls byte
	incf	LOWER_LIM1,f
	btfsc	STATUS,Z
	incf	LOWER_LIM0,f

; do not increase past D999, H3E7
	movf	LOWER_LIM0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_LOWER
	movf	LOWER_LIM1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	LOWER_LIM1

LOAD_LOWER
	movlw	EEPROM4
	call	EEREAD
	movf	LOWER_LIM0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM5
	call	EEREAD
	movf	LOWER_LIM1,w
	call	EEWRITE		; store in eeprom
	goto	RUN

UPPER_UP

;EEPROM2 Upper limit value (UPPER_LIM0) ms byte
;EEPROM3 Upper limit value (UPPER_LIM1) ls byte

	incf	UPPER_LIM1,f
	btfsc	STATUS,Z
	incf	UPPER_LIM0,f

; do not increase past D999, H3E7
	movf	UPPER_LIM0,w	; ms byte
	xorlw	H'03'			; if 3 test for >E7 in ls byte
	btfss	STATUS,Z
	goto	LOAD_UPPER
	movf	UPPER_LIM1,w
	sublw	H'E7'
	movlw	H'E7'			; load E7 if > E7
	btfss	STATUS,C
	movwf	UPPER_LIM1

LOAD_UPPER
	movlw	EEPROM2		
	call	EEREAD
	movf	UPPER_LIM0,w
	call	EEWRITE		; store in eeprom
	movlw	EEPROM3
	call	EEREAD
	movf	UPPER_LIM1,w
	call	EEWRITE		; store in eeprom
	goto	RUN
	
SWITCH3; down
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'3'		; ls timer
	movwf	TIMER1
; 0
	movf	SETTINGS,w
	xorlw	D'0'
	btfsc	STATUS,Z
	goto	DP_DN
; 1
	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	LOWER_DN
; 2
	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	UPPER_DN
; 3
	movf	SETTINGS,w
	xorlw	D'3'
	btfss	STATUS,Z
	goto	RUN

DIMMING_MIN_DN
	decf	BRIGHT_MIN,f
	movf	BRIGHT_MIN,w
	xorlw	H'FF'			; test for past 00
	btfsc	STATUS,Z
	clrf	BRIGHT_MIN		; stop at 00
	goto	LOAD_MIN		; store in EEPROM

LOWER_DN

; if both zero do not decrease
	movf	LOWER_LIM1,w
	btfss	STATUS,Z
	goto	DECL
	movf	LOWER_LIM0,w
	btfsc	STATUS,Z
	goto	LOAD_LOWER
DECL
	movf	LOWER_LIM1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	LOWER_LIM0,f
	decf	LOWER_LIM1,f
	goto	LOAD_LOWER

UPPER_DN

; if both zero do not decrease
	movf	UPPER_LIM1,w
	btfss	STATUS,Z
	goto	DECU
	movf	UPPER_LIM0,w
	btfsc	STATUS,Z
	goto	LOAD_UPPER
DECU
	movf	UPPER_LIM1,w
	btfsc	STATUS,Z		; if ls byte zero decrease ms byte
	decf	UPPER_LIM0,f
	decf	UPPER_LIM1,f
	goto	LOAD_UPPER

SWITCH2; setting
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'5'		; ls timer
	movwf	TIMER1
	incf	SETTINGS,f
	movlw	D'4'		; number of settings 
	subwf	SETTINGS,w
	btfsc	STATUS,C	; 
	clrf	SETTINGS	; clear when reach end of settings 

	goto	RUN

SWITCH1; mode
; reset switch timers
	clrf	TIMER0		; ms timer
	movlw	D'5'		; ls timer
	movwf	TIMER1
	incf	MODE,f
	movlw	B'00000001'
	andwf	MODE,f		; keep ls bit only
; when Mode = 0 reset SETTINGS
	movf	MODE,w
	btfss	STATUS,Z
	goto	RUN
	clrf 	SETTINGS
	goto	RUN

; *****************************************************
SETTINGS_RUN

	btfsc	SW_FLG,0	; switch flag only update at each switch press
	goto	DO_SET
	
	movf	TIMER1,w	; switch timer is clear after switch delay
	btfss	STATUS,Z
	goto	RUN
	btfss	SW_FLG,0	; switch flag only update at each switch press
	goto	RUN
DO_SET
	bcf		SW_FLG,0

; find settings
	movf	SETTINGS,w
	xorlw	D'0'
	btfsc	STATUS,Z
	goto	DP_SHOW

	movf	SETTINGS,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	LOW_LIMIT_SHOW

	movf	SETTINGS,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	UPPER_LIMIT_SHOW

	movf	SETTINGS,w
	xorlw	D'3'
	btfss	STATUS,Z
	goto 	RUN

MIN_DIMMING
;  dimming minimum 
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt
	clrf	BIN_0
	movf	BRIGHT_MIN,w
DIM_LOAD
	movwf	BIN_1		; for BCD conversion
	bcf		ZRO_FLG,0	; clear flag for leading 0 suppression
	bcf		ZRO_FLG,1	; digit 2 '0' suppression when 0
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

DP_SHOW
;  Decimal point X.XX or XX.X 
	bcf		TRAN_FLG,0 		; prevent transfer in interrupt 

	movlw	B'10101000'		; 7-segment code d, 
	movwf	DISP1
	movlw	B'00001110'		; 7-segment code P, 
	movwf	DISP2			;
	movlw	B'11111111'		; segment off 
	movwf	DISP3

	btfsc	DECIMAL_P,0		; if set show in DISP1
	bcf		DISP1,3
	btfsc	DECIMAL_P,1		; if set show in DISP2
	bcf		DISP2,3
	bsf		TRAN_FLG,0 		; allow interrupt after transfer
	goto	RUN

LOW_LIMIT_SHOW
; 5. Lower limit value bar segments C and G, portb 2 and 6
	bcf		TRAN_FLG,0 	; prevent transfer in interrupt

	movf	LOWER_LIM0,w	; ms 
	movwf	BIN_0

	movf	LOWER_LIM1,w	; ls 
	movwf	BIN_1

	movf	DECIMAL_P,w	; decimal point
	movwf	ZRO_FLG		; set flag for 0 suppression and dP
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN

UPPER_LIMIT_SHOW
; 6. upper limit value segments C and B, portb 2 and 0
	bcf		TRAN_FLG,0 		; prevent transfer in interrupt

	movf	UPPER_LIM0,w	; ms
	movwf	BIN_0

	movf	UPPER_LIM1,w	; ls
	movwf	BIN_1

	movf	DECIMAL_P,w	; decimal point
	movwf	ZRO_FLG		; set flag for 0 suppression and dP
	call	BCD_AND_LOAD; convert and load
	bsf		TRAN_FLG,0 	; allow transfer in interrupt
	goto	RUN


; extra 7-segment codes
;   fgae.cdb
; B'00001010'	; 7-segment code A, 
; B'00101001'	; 7-segment code b, 
; B'01001101'	; 7-segment code C, 
; B'10101000'	; 7-segment code d, 
; B'00001101'	; 7-segment code E, 
; B'00001111'	; 7-segment code F, 
; B'00101010'	; 7-segment code H, 
; B'01101111'	; 7-segment code I, 
; B'11101000'	; 7-segment code J, 
; B'01101101'	; 7-segment code L,	
; B'10101011'	; 7-segment code n, 
; B'10001001'	; 7-segment code o, (note; for use with T in oT)
; B'00001110'	; 7-segment code P, 
; B'01001110'	; 7-segment code r, 
; B'01001111'	; 7-segment code T, (note: use with o in oT)
; B'01101000'	; 7-segment code U, 
; B'00111000'	; 7-segment code Y, 

; *****************************************************************************************
; Subroutines

DELAAY
	movlw	H'FF'
	movwf	LOOPCOUNT
DELAAY1
	call	DELAY
	decfsz	LOOPCOUNT,f
	goto	DELAAY1
	return
DELAY
	movlw	H'FF'
	movwf	STORE
DELAY1
	decfsz	STORE,f
	goto 	DELAY1
	return

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; Subroutine to convert from 16-bit binary to 3-digit BCD (packed)
; Binary value is in BIN0 & BIN1. BIN0 is high byte BIN1 is low byte
; Result in BCD is in BCD0, BCD1 & BCD2.  
; BCD0 is MSB, BCD2 is LSB

BCD_AND_LOAD	
	bcf		STATUS,C	; clear carry bit
	movlw	D'16'
	movwf	CNT_16		; 16 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
	clrf	BCD_2

LOOPBCD
	rlf		BIN_1,f
	rlf		BIN_0,f		; LSB shift left binary registers
	rlf		BCD_2,f		; LSB shift left BCD registers
	rlf		BCD_1,f
	rlf		BCD_0,f

	decfsz	CNT_16,f	; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
; LOAD into display
	movf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	bcf		TRAN_FLG,0 	; stop transfer in interrupt
	movwf	DISP3

; check ZRO_FLG bits
	btfsc	ZRO_FLG,0	; if set then X.XX
	goto	X_XX		; no leading 0 suppression decimal point at DP1
	btfsc	ZRO_FLG,1	; if set then -X.X where - is a 0 blanking or XX.X
	goto	XX_X
; both flags clear so no dp but DISP1 and DISP2 '0' suppression
XXX; or _XX or __X
	movf	BCD_1,w		; check ms byte
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP1_LDY
	movlw	H'FF'
	movwf	DISP1		; DISP1 unlit 
; check DISP2
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP2_LDX
	movlw	H'FF'
	movwf	DISP2		; DISP2 unlit 
	return
DISP2_LDX
	call	SEVEN_SEG	; seven segment value
	movwf	DISP2
	return

DISP1_LDY
	call	SEVEN_SEG	; seven segment value
	movwf	DISP1		; DISP1 lit with all but a 0
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; seven segment value
	movwf	DISP2		; DISP2 unlit 
	return

X_XX; no leading '0' suppression decimal at first digit X.XX
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP2
	movf	BCD_1,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP1
	bcf		DISP1,3		; decimal point on
	return

XX_X; leading zero suppression on ms digit, dp at second digit XX.X or _X.X
	movf	BCD_1,w		; check ms byte
	andlw	H'0F'		; get ls bit
	btfss	STATUS,Z
	goto	DISP1_LD
	movlw	H'FF'		; unlit DISP1
	goto	DISP1_LD1
DISP1_LD
	call	SEVEN_SEG	; seven segment value
DISP1_LD1
	movwf	DISP1		; DISP1 lit with all but a 0
; display 2
	swapf	BCD_2,w
	andlw	H'0F'		; get ls bit
	call	SEVEN_SEG	; get seven segment value
	movwf	DISP2
	bcf		DISP2,3		; decimal point on
	return

; subroutine decimal adjust
DECADJ
	movlw	BCD_2		; BCD LSB address
	movwf	FSR			; pointer for BCD2
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_1
	movwf	FSR
	call 	ADJBCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR			; pointer for BCD_0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD
	movlw	H'03'		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-2)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	H'30'		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine

; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP1	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 	INTCON,GIE	; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 


EEWRITE_X ; no interrupt enable	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR4	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR4			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITEX
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITEX		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 


; Math routines

;

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U    	CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1,F
                MOVF            BARGB1,W
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                RLF             AARGB0,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LA 		RLF             AARGB0,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L8         RLF             AARGB1,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LB         RLF             AARGB1,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L16        RLF             AARGB2,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LC 		RLF             AARGB2,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
UOK46L			RETURN

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0,1
;               16 bit unsigned fixed point multiplier in BARGB0,1

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0,1,2,3



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
		        RETURN



	end

	
	
