; Wideband Controller for LSU4.9 Bosch sensor
; Uses assembly code for PIC16F1507 microcontroller

; proven operation 11/4/2012

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1507
	#include p16f1507.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF 

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _STVREN_ON & _BORV_HI & _LPBOR_ON & _LVP_OFF 


; Define variables at memory locations

; Bank 0 RAM
IMPEDANCE	equ	H'20'	; impedance on/off flag
HEAT		equ	H'21'	; heater value
HEAT_COUNT	equ	H'22'	; heater counter for PWM
RICHLEAN	equ	H'23'	; rich lean flag for pressure correction
WIDEBAND0	equ	H'24'	; wideband value ms
WIDEBAND1	equ	H'25'	; wideband value ls
S_CURVE		equ	H'26'	; S curve output
CURRENT_IP0	equ	H'27'	; IP current (PWM output value) ms	
CURRENT_IP1	equ	H'28'	; IP current (PWM output value) ls
TWO_DIV		equ	H'29'	; impedance drive/2
REDUC_4		equ	H'2A'	; interrupt decrease rate counter
BATT0		equ	H'2B'	; battery volts ms byte
BATT1		equ	H'2C'	; battery volts ls byte
RAMP		equ	H'2D'	; heater ramp				
COUNT1		equ	H'2E'	; counter for interrupt counts
HALFSEC		equ	H'2F'	; half second counter
FLAG		equ	H'30'	; flag indicator
VH_EFF0		equ	H'31'	; heater effective voltage ms
VH_EFF1		equ	H'32'	; heater effective voltage ls
MIN_COUNT	equ	H'33'	; minute counter
ERROR_FLG	equ	H'34'	; error flag
HEAT_I		equ	H'35'	; transfer new heat value at end of cycle
SWINGH0		equ	H'36'	; voltage swing high across VS impedance ms byte
SWINGH1		equ	H'37'	; voltage swing high across VS impedance ls byte
SWINGL0		equ	H'38'	; voltage swing low across VS impedance ms byte
SWINGL1		equ	H'39'	; voltage swing low across VS impedance ls byte	
SPAN0		equ	H'3A'	; impedance measuremant voltage span ms byte
SPAN1		equ	H'3B'	; impedance measuremant voltage span ls byte	
HEATX		equ	H'3C'	; heater value temporary store
WORK0		equ	H'3D'	; working value divisor ms
WORK1		equ	H'3E'	; working value divisor ls
WORK3		equ	H'3F'	; working value multiplier ms
FOUR_SEC	equ	H'40'	; four seconds counter
CALC0		equ	H'41'	; calculation temporary value ms byte
RATE		equ	H'42'	; ramp rate counter
DIV_2		equ	H'43'	; divide by 2
VOLT0		equ	H'44'	; battery voltage startup threshold ms byte
VOLT1		equ	H'45'	; battery voltage threshold ls byte	
STARTFLAG	equ	H'46'	; after 2-seconds prehat flag for a dimmer startup indicator LED
TEMPK		equ	H'47'	; temperature offset
SENSE_IP0	equ	H'48'	; IP sense ms	
SENSE_IP1	equ	H'49'	; IP sense ls
VS_AVG0		equ	H'4A'	; Vs average ms
VS_AVG1		equ	H'4B'	; Vs average ls
TEMP_AV0	equ	H'4C'	; temporary ms
TEMP_AV1	equ	H'4D'	; temporary ls
TEMP_PWM0	equ H'4E'	; temp PWM ms
TEMP_PWM1	equ H'4F'	; temp PWM ls
VS_IP0		equ	H'50'	; VS/IP	ms
VS_IP1		equ	H'51'	; VS/IP ls
TWO_COUNT	equ	H'52'	; A/D AN2 counter
SWING_FLG	equ	H'53'	; swing flag
MAX0		equ	H'54'	; calculation for wideband PWM max value ms byte
MAX1		equ	H'55'	; calculation for wideband PWM max value ls byte
SUB0		equ	H'56'	; subtract value
SUB1		equ	H'57'	; subtract value
DIV1		equ	H'58'	; division value
MULT1		equ	H'59'	; multiplier value
RESPONSE	equ	H'5A'	; response rate
OFFSET		equ	H'5B'	; pressure sensor offset from VR6/64
PRESSL0		equ	H'5C'	; pressure value Low ms
PRESSL1		equ	H'5D'	; pressure value low ls
PRESSH0		equ	H'5E'	; pressure value High ms
PRESSH1		equ	H'5F'	; pressure value high ls
		
; math routines
TEMP160		equ	H'60'	; DIV1616U temp ms byte
TEMP161		equ	H'61'	; ls byte	
TEMP		equ	H'62'	; temp file
REMB0		equ	H'63'	; remainder ms
REMB1		equ	H'64'
TEMP1		equ H'65'	; temporary
TEMPD		equ	H'66'
AARGB3		equ	H'67'	; ls of argument A
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
TEMPB1      equ H'6B'	; temporary
TEMPB0      equ H'6C'
BARGB1      equ H'6D'	; 
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; loop counter

; All Banks RAM

TEMPX		equ	H'70'	; temporary
LOCK		equ	H'71'	; locked control for pump cell
ADDN_S0		equ	H'73'	; averaging addition ms byts
ADDN_S1		equ	H'74'	; averging addition ls byte
STORE0		equ	H'75'	; temp store
STORE1		equ	H'76'	; temp store
STOREH		equ	H'77'	; ms byte store
STOREL		equ	H'78'	; ls byte store
TEMPD0		equ	H'79'	; temp ms byte
TEMPD1		equ	H'7A'	; temp ls byte	
STORE3		equ	H'7B'	; timer for A/D
DIM			equ	H'7C'	; LED dimmer
STORE		equ	H'7D'	; storage
TEMP_AV0X	equ	H'7E'	; temporary ms
TEMP_AV1X	equ	H'7F'	; temporary ls 

; define reset and interrupt vector start addresses

	org		0  			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	nop
	org     4			; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT	; start of interrupt routine

; ******************************
; Lookup table for PWM max for duty cycle driven heater
; max varies with battery supply voltage Duty cycle = (VHeff/Vbatt)**2
; returns duty cycle/255
LOOK_PWM
	addwf	PCL,f		; add value of display to program counter
	retlw	D'163'		; when 15V supply for A-D value/4 = 255 and for 12V limit
	retlw	D'164'		; when 14.94V for A-D value/4 = 254 and for 12V limit
	retlw	D'165'		; when 14.88V for A-D value/4 = 253 and for 12V limit
	retlw	D'167'		; when 14.82V for A-D value/4 = 252 and for 12V limit
	retlw	D'168'		; when 14.76V for A-D value/4 = 251 and for 12V limit
	retlw	D'169'		; when 14.71V for A-D value/4 = 250 and for 12V limit
	retlw	D'171'		; when 14.65V for A-D value/4 = 249 and for 12V limit
	retlw	D'172'		; when 14.59V for A-D value/4 = 248 and for 12V limit
	retlw	D'174'		; when 14.53V for A-D value/4 = 247 and for 12V limit
	retlw	D'175'		; when 14.47V for A-D value/4 = 246 and for 12V limit
	retlw	D'176'		; when 14.41V for A-D value/4 = 245 and for 12V limit
	retlw	D'178'		; when 14.35V for A-D value/4 = 244 and for 12V limit
	retlw	D'179'		; when 14.29V for A-D value/4 = 243 and for 12V limit
	retlw	D'181'		; when 14.24V for A-D value/4 = 242 and for 12V limit
	retlw	D'182'		; when 14.18V for A-D value/4 = 241 and for 12V limit
	retlw	D'184'		; when 14.12V for A-D value/4 = 240 and for 12V limit
	retlw	D'185'		; when 14.06V for A-D value/4 = 239 and for 12V limit
	retlw	D'187'		; when 14.00V for A-D value/4 = 238 and for 12V limit
	retlw	D'188'		; when 13.94V for A-D value/4 = 237 and for 12V limit
	retlw	D'190'		; when 13.88V for A-D value/4 = 236 and for 12V limit
	retlw	D'192'		; when 13.82V for A-D value/4 = 235 and for 12V limit
	retlw	D'193'		; when 13.76V for A-D value/4 = 234 and for 12V limit
	retlw	D'195'		; when 13.71V for A-D value/4 = 233 and for 12V limit
	retlw	D'197'		; when 13.65V for A-D value/4 = 232 and for 12V limit
	retlw	D'198'		; when 13.59V for A-D value/4 = 231 and for 12V limit
	retlw	D'200'		; when 13.53V for A-D value/4 = 230 and for 12V limit
	retlw	D'202'		; when 13.47V for A-D value/4 = 229 and for 12V limit
	retlw	D'204'		; when 13.41V for A-D value/4 = 228 and for 12V limit
	retlw	D'205'		; when 13.35V for A-D value/4 = 227 and for 12V limit
	retlw	D'207'		; when 13.29V for A-D value/4 = 226 and for 12V limit
	retlw	D'209'		; when 13.24V for A-D value/4 = 225 and for 12V limit
	retlw	D'211'		; when 13.18V for A-D value/4 = 224 and for 12V limit
	retlw	D'213'		; when 13.12V for A-D value/4 = 223 and for 12V limit
	retlw	D'215'		; when 13.06V for A-D value/4 = 222 and for 12V limit
	retlw	D'217'		; when 13.00V for A-D value/4 = 221 and for 12V limit
	retlw	D'219'		; when 12.94V for A-D value/4 = 220 and for 12V limit
	retlw	D'221'		; when 12.88V for A-D value/4 = 219 and for 12V limit
	retlw	D'223'		; when 12.82V for A-D value/4 = 218 and for 12V limit
	retlw	D'225'		; when 12.76V for A-D value/4 = 217 and for 12V limit
	retlw	D'227'		; when 12.71V for A-D value/4 = 216 and for 12V limit
	retlw	D'229'		; when 12.65V for A-D value/4 = 215 and for 12V limit
	retlw	D'231'		; when 12.59V for A-D value/4 = 214 and for 12V limit
	retlw	D'233'		; when 12.53V for A-D value/4 = 213 and for 12V limit
	retlw	D'236'		; when 12.47V for A-D value/4 = 212 and for 12V limit
	retlw	D'238'		; when 12.41V for A-D value/4 = 211 and for 12V limit
	retlw	D'240'		; when 12.35V for A-D value/4 = 210 and for 12V limit
	retlw	D'242'		; when 12.29V for A-D value/4 = 209 and for 12V limit
	retlw	D'245'		; when 12.24V for A-D value/4 = 208 and for 12V limit
	retlw	D'247'		; when 12.18V for A-D value/4 = 207 and for 12V limit
	retlw	D'250'		; when 12.12V for A-D value/4 = 206 and for 12V limit
	retlw	D'252'		; when 12.06V for A-D value/4 = 205 and for 12V limit
	retlw	D'255'		; when 12.00V for A-D value/4 = 204 and for 12V limit
	
LOOKUP_S_CURVE
; wideband value/4 if <D22 then 890mV out
; if >D105 then 30mV output
; start at D22
; S-curve output is effectively a divide by 4 value as it uses the 10-bit PWM and ls 8-bits
; so a D255 value in S_CURVE is 5V/4 or 1.25V max
	addwf	PCL,f	; add value to program counter
	retlw	D'182'	; 890mV (0.89/5V) x 1024 ; wideband /4 = 22 	
	retlw	D'182'	;  wideband /4 = 23 
	retlw	D'182'	;  wideband /4 = 24 
	retlw	D'182'	;  wideband /4 = 25 
	retlw	D'181'	;  wideband /4 = 26 
	retlw	D'181'	;  wideband /4 = 27 
	retlw	D'181'	;  wideband /4 = 28 
	retlw	D'180'	;  wideband /4 = 29 
	retlw	D'180'	;  wideband /4 = 30 
	retlw	D'180'	;  wideband /4 = 31 
	retlw	D'179'	;  wideband /4 = 32 
	retlw	D'179'	; 870mV  ; wideband /4 = 33
	retlw	D'179'	;  wideband /4 = 34 
	retlw	D'179'	;  wideband /4 = 35 
	retlw	D'179'	;  wideband /4 = 36 
	retlw	D'178'	;  wideband /4 = 37 
	retlw	D'178'	;  wideband /4 = 38 
	retlw	D'178'	;  wideband /4 = 39 
	retlw	D'177'	;  wideband /4 = 40 
	retlw	D'177'	;  wideband /4 = 41 
	retlw	D'176'	;  wideband /4 = 42 
	retlw	D'175'	;  wideband /4 = 43 
	retlw	D'174'	;  wideband /4 = 44 
	retlw	D'173'	; 860mV  ; wideband /4 = 45  
	retlw	D'172'	;  wideband /4 = 46 
	retlw	D'171'	;  wideband /4 = 47 
	retlw	D'170'	;  wideband /4 = 48 
	retlw	D'169'	;  wideband /4 = 49 
	retlw	D'168'	;  wideband /4 = 50 
	retlw	D'167'	;  wideband /4 = 51 
	retlw	D'166'	;  wideband /4 = 52 
	retlw	D'165'	;  wideband /4 = 53 
	retlw	D'163'	;  wideband /4 = 54 
	retlw	D'161'	;  wideband /4 = 55 
	retlw	D'159'	;  wideband /4 = 56 
	retlw	D'158'	;  wideband /4 = 57 
	retlw	D'156'	;  wideband /4 = 58 
	retlw	D'154'	;  wideband /4 = 59 
	retlw	D'153'	;  wideband /4 = 60 
	retlw	D'151'	;  wideband /4 = 61  
	retlw	D'148'	;  wideband /4 = 62 
	retlw	D'146'	;  wideband /4 = 63 
	retlw	D'141'	;  700mV  ;wideband /4 = 64 
	retlw	D'136'	;  wideband /4 = 65  
	retlw	D'124'	;  wideband /4 = 66 
	retlw	D'114'	;  wideband /4 = 67 
	retlw	D'100'	;  wideband /4 = 68 
	retlw	D'92'	;  450mV; wideband /4 = 69 
	retlw	D'63'	;  wideband /4 = 70 
	retlw	D'35'	;  wideband /4 = 71
 	retlw	D'26'	;  130mV ; wideband /4 = 72 
	retlw	D'20'	;  wideband /4 = 73 
	retlw	D'18'	;  wideband /4 = 74 
	retlw	D'16'	;  wideband /4 = 75 
	retlw	D'14'	;  wideband /4 = 76 
	retlw	D'13'	;  wideband /4 = 77 
	retlw	D'12'	;  wideband /4 = 78 
	retlw	D'12'	;  wideband /4 = 79 
	retlw	D'11'	;  wideband /4 = 80 
	retlw	D'11'	;  wideband /4 = 81 
	retlw	D'10'	;  wideband /4 = 82 
	retlw	D'10'	;  wideband /4 = 83 
	retlw	D'10'	;  wideband /4 = 84 
	retlw	D'9'	;  wideband /4 = 85 
	retlw	D'9'	;  wideband /4 = 86 
	retlw	D'9'	;  wideband /4 = 87 
	retlw	D'8'	;  wideband /4 = 88 
	retlw	D'8'	;  wideband /4 = 89 
	retlw	D'8'	;  wideband /4 = 90 
	retlw	D'8'	;  wideband /4 = 91 
	retlw	D'8'	;  wideband /4 = 92 
	retlw	D'8'	;  wideband /4 = 93 
	retlw	D'8'	;  wideband /4 = 94 
	retlw	D'8'	;  wideband /4 = 95 
	retlw	D'8'	;  wideband /4 = 96 
	retlw	D'7'	;  wideband /4 = 97 
	retlw	D'7'	;  wideband /4 = 98 
	retlw	D'7'	;  wideband /4 = 99 
	retlw	D'7'	;  wideband /4 = 100 
	retlw	D'7'	;  wideband /4 = 101 
	retlw	D'6'	;  wideband /4 = 102 
	retlw	D'6'	;  wideband /4 = 103 
	retlw	D'6'	;  wideband /4 = 104 
	retlw	D'6'	;  wideband /4 = 105 
	retlw	D'6'	;  wideband /4 = 106

; lookup table for pressure compensation/correction
PRESS_LEAN ; lean 
	addwf	PCL,f	; add value to program counter
	retlw	D'100'	; 0
	retlw	D'99'	; 1
	retlw	D'98'	; 2
	retlw	D'97'	; 3
	retlw	D'96'	; 4
	retlw	D'95'	; 5
	retlw	D'94'	; 6
	retlw	D'93'	; 7
	retlw	D'92'	; 8
	retlw	D'91'	; 9
	retlw	D'90'	; 10
	retlw	D'90'	; 11
	retlw	D'89'	; 12
	retlw	D'89'	; 13
	retlw	D'88'	; 14
	retlw	D'88'	; 15
PRESS_RICH
	addwf	PCL,f	; add value to program counter
	retlw	D'100'	; 0
	retlw	D'99'	; 1
	retlw	D'98'	; 2
	retlw	D'97'	; 3
	retlw	D'96'	; 4
	retlw	D'95'	; 5
	retlw	D'95'	; 6
	retlw	D'95'	; 7
	retlw	D'94'	; 8
	retlw	D'94'	; 9
	retlw	D'93'	; 10
	retlw	D'93'	; 11
	retlw	D'92'	; 12
	retlw	D'92'	; 13
	retlw	D'91'	; 14
	retlw	D'91'	; 15

; ******************************************************************************

INTERRUPT	
	clrf	BSR			; bank 0
	bcf		PIR1,TMR1IF	; clear timer 1 flag
	bcf		T1CON,0		; stop timer1
	movlw	H'FF'
	movwf	TMR1H		; set ms byte of counter
	bsf		T1CON,0		; set timer to run
	
; heater PWM driver
HEATER
	incfsz	HEAT_COUNT,f	; heater counter
	goto	DO_PWM
	movf	HEAT,w			; transfer new pwm value at end of cycle
	movwf	HEAT_I			; prevents glitches in PWM
DO_PWM
	movf	HEAT_I,w		; heat value
	btfsc	STATUS,Z		; if zero clear
	goto	CLR_7
	subwf	HEAT_COUNT,w
	btfss	STATUS,C
	goto	SET_7
	btfsc	STATUS,Z
	goto	SET_7
CLR_7
	bcf		PORTB,7
	goto	DIV_4_INTERRUPT
SET_7 
	bsf		PORTB,7

DIV_4_INTERRUPT

; divide by 4
	decfsz	REDUC_4,f	; when zero continue
	goto	TEST_VALUE
	movlw	D'4'
	movwf	REDUC_4
	goto	DRIVE 		;(PWM drive)
TEST_VALUE ; split interrupt into 4-independent sections
	movf	REDUC_4,w
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	COUNT_RUN
	movf	REDUC_4,w
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	CK_IMP		; A/D for span and impedance measurement
;	goto	IMPEDANCE_DRIVE
; *******
	
IMPEDANCE_DRIVE; and negative supply driver
; impedance drive output set at start of interrupt to ensure a good square wave
; not affected by routine length
; check if impedance is on	
	btfss	IMPEDANCE,0	;
	goto	NEG_DRV		; 
	incf	TWO_DIV,f	; divide by 2 for ~2kHz (1.98kHz)
	btfss	TWO_DIV,0
	goto	NEG_DRV
	btfss	PORTB,6
	goto	SET6
	bcf		PORTB,6		; impedance drive 5V square wave

	goto	NEG_DRV
SET6
	bsf		PORTB,6		; impedance drive 5V square wave

NEG_DRV ; negative supply driver ~4kHz (3.906kHz)
	btfss	PORTA,1
	goto	SET1
	bcf		PORTA,1		; negative supply generation driver
	retfie
SET1
	bsf		PORTA,1		; negative supply generation driver	
	retfie
; **********

; check if impedance is on
CK_IMP
	
	btfss	IMPEDANCE,0
	retfie

	btfsc	FLAG,1		; if flag set no A/D allowed as it may occur in main routine
	retfie

; check impedance every ~10ms
		
	movlw	D'6'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH10_1
	movlw	D'86'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH10_1
	movlw	D'166'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	goto	CHECK_2

; measure voltage across impedance of Nernst cell
; for VS detection and impedance measurement
; A/D conversion
; channel 10
CH10_1
	
	clrf	SWING_FLG	; data not ready flag

	movlw	B'00101001'	; CH10
	call	ACQUIRE_AD

	movwf	SWINGL0
	movf	TEMPD1,w
	movwf	SWINGL1		; ls voltage when RB6 is low (or possible high instead)

	retfie

; measure voltage across impedance of Nernst cell
; A/D conversion
; channel 10
CHECK_2

; check impedance every ~10ms
	btfsc	SWING_FLG,0	; if swing flag set bypass
	goto	CH6
	movlw	D'3'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH10_2
	movlw	D'83'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH10_2
	movlw	D'163'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	goto	CH6
CH10_2

	movlw	B'00101001'	; channel 10
	call	ACQUIRE_AD

	movwf	SWINGH0
	movf	TEMPD1,w
	movwf	SWINGH1		; ls voltage when RB6 is high (or possible low instead)
	bsf		SWING_FLG,0
	
	retfie

CH6
; check Ip sense every ~10ms

	movlw	D'1'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'81'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfsc	STATUS,Z
	goto	CH6_2
	movlw	D'161'
	xorwf	COUNT1,w	; when count1 is at specified count run A/D
	btfss	STATUS,Z
	retfie

; A/D conversion
; channel 6
CH6_2
	movlw	B'00011001'	; channel 6
	
	call	ACQUIRE_AD
	movwf	SENSE_IP0	; ms byte
	movf	TEMPD1,w
	movwf	SENSE_IP1	
	bsf		FLAG,1		; set when end of acquisition set 
	retfie

; ********************

COUNT_RUN
	decfsz	COUNT1,f	; counter
	goto	POWER
	movlw	D'244'		
	movwf	COUNT1		; 32.768ms at 256 counts/ 31.23ms at count of 244
	decf	DIV_2,f
	btfss	DIV_2,0		; 62.4ms period
	goto	POWER
	decfsz	RATE,f		; ramp rate counter
	goto	HALF_DEC
	bsf		FLAG,0		; flag for each rate increment
	movlw	D'3'
	movwf	RATE		; reset rate counter
HALF_DEC
	decfsz	HALFSEC,f
	goto	POWER
	bsf		FLAG,2		; uncleared flag indicates when both voltage swings 
						; across Nernst cell have been made
	movf	FOUR_SEC,w	; four seconds counter
	btfss	STATUS,Z	; when zero do not decrease
	decf	FOUR_SEC,f
	movlw	D'8'
	movwf	HALFSEC
	movf	MIN_COUNT,w	; minute counter (120 half seconds)
	btfss	STATUS,Z	; do not decrease if zero
	decf	MIN_COUNT,f	; decrease to zero minute counter

POWER
; power/heat/error LED

; dim the LED if heater drive is reduced to 0
	btfsc	ERROR_FLG,5
	goto	DIM_LED
; error/data LED flash when error
	btfss	ERROR_FLG,7
	goto	ERR_OFF

; flash LED
FLASH2
	btfsc	HALFSEC,2	
	goto	LED_OFF
	goto	SET_LED
	
ERR_OFF
; flash data when heater ready otherwise stay lit
	btfsc	FLAG,7			; when set, heater ready
	goto	DATA_1
; power/heater LED. Lower brightness until after the 2Veff has finished (also after 13V if selected)
	btfsc	STARTFLAG,0	
	goto	SET_LED
DIM_LED
; DIM counter
	movf	DIM,w			; dim counter
	btfss	STATUS,Z
	goto	LED_DIM
	movlw	D'10'			; 1/10 duty cycle for LED
	movwf	DIM
	goto	SET_LED

; check new data flash indication required
DATA_1
	btfss	ERROR_FLG,6		; new data flag indication 
	retfie
;	bcf		ERROR_FLG,6		; data flag clear for next data ready. (changes state with new data)
; (remark out for constant flash rate)
	btfsc	HALFSEC,0
	goto	LED_OFF
SET_LED
	bsf		PORTC,4			; LED on
	retfie
LED_DIM
	decf	DIM,f
LED_OFF	
	bcf		PORTC,4			; error LED off
	retfie
; ****************

DRIVE ; PWM drive for IP, wideband and S curve output 
; check if timer2 close to overrange. TMR2 increments
	movlw	D'235'
	subwf	TMR2,w
	btfsc	STATUS,C		; if carry is clear then timer <=235
	retfie					; do not load PWMs if timer2 is close to overflow

; IP_OUT
; load pwm for 0-5V out
	movf	CURRENT_IP0,w	; ms byte first
	movwf	STOREH		; store
	movf	CURRENT_IP1,w	; ls byte
	movwf	STOREL			; store

; load PWM1
	movlb	D'12'			; bank12 PWM
	bcf		PWM1DCL,6		; ls bits PWM operation
	btfsc	STOREL,0		; wideband ls		
	bsf		PWM1DCL,6		; ls bits PWM operation
	bcf		PWM1DCL,7		; ls bits PWM operation
	btfsc	STOREL,1		; wideband ls		
	bsf		PWM1DCL,7		; ls bits PWM operation	
; move right to find ms byte
	rrf		STOREH,f
	rrf		STOREL,f
	rrf		STOREH,f
	rrf		STOREL,w
	movwf	PWM1DCH			; ms byte PWM
	clrf	BSR				; bank0

; load PWM4 for 0-5V out or impedance
; check if timer2 close to overrange. TMR2 increments
	movlw	D'228'
	subwf	TMR2,w
	btfsc	STATUS,C		; if carry is clear then timer <=228
	retfie

; If JP1 inserted show impedance at wideband output
; place impedance in PWM
	btfsc	PORTA,5
	goto	WIDE_OUT_1
	movf	SPAN0,w
	movwf	WIDEBAND0		; ms PWM byte for wideband output	
	movf	SPAN1,w
	movwf	WIDEBAND1		; ls PWM byte for wideband output

; WIDE_OUT
WIDE_OUT_1
	movf	WIDEBAND0,w		; ms byte first
	movwf	STOREH			; store
	movf	WIDEBAND1,w		; ls byte
	movwf	STOREL			; store

; load PWM4
LOAD_WIDE
	movlb	D'12'			; bank12 PWM
	bcf		PWM4DCL,6		; ls bits PWM operation
	btfsc	STOREL,0		; wideband ls		
	bsf		PWM4DCL,6		; ls bits PWM operation
	bcf		PWM4DCL,7		; ls bits PWM operation
	btfsc	STOREL,1		; wideband ls		
	bsf		PWM4DCL,7		; ls bits PWM operation	
; move right to find ms byte
	rrf		STOREH,f
	rrf		STOREL,f
	rrf		STOREH,f
	rrf		STOREL,w
	movwf	PWM4DCH			; ms byte PWM
	clrf	BSR				; bank0

; S_CURVE
	movlw	D'236'
	subwf	TMR2,w
	btfsc	STATUS,C		; if carry is clear then timer <=236)
	retfie
; ls bits
	movf	S_CURVE,w
	movwf	STOREL
	movlb	D'12'		; bank12 PWM
	bcf		PWM3DCL,6	; ls bits PWM operation
	btfsc	STOREL,0	; S-curve value	
	bsf		PWM3DCL,6	; ls bits PWM operation
	bcf		PWM3DCL,7	; ls bits PWM operation
	btfsc	STOREL,1	; s curve value		
	bsf		PWM3DCL,7	; ls bits PWM operation	
	bcf		STATUS,C	; carry clear
	rrf		STOREL,f
	bcf		STATUS,C
	rrf		STOREL,w	
	movwf	PWM3DCH		; ms byte PWM
	clrf	BSR			; bank0

; end of interrupt  

	retfie					; return from interrupt

;***************************************************************************************

; RESET		
; Set ports A & B

MAIN
; initial values
	
; set inputs/outputs
	clrf	BSR			; bank0
	movlw	H'00'		; set all outputs low
	movwf	PORTA
	movwf	PORTB
	movwf	PORTC
; weak pullups off/on
	movlb	D'4'		; bank	WPUA/B
	clrf	WPUA
	bsf		WPUA,5		; PORTA,5 pullup
	clrf	WPUB
; set I/O
	movlb	D'1'		; bank1	TRISA/B/C
	movlw   B'00111001'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00110000'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'11001101'
	movwf	TRISC
; option
	movlw	B'00000000'	; weak pullups set via WPUA/B
	movwf	OPTION_REG	; 
; analog inputs, A/D
	movlb	D'3'		; bank3	ANSELA/B/C
	movlw	B'00010001'	; AN0,AN3 are analog inputs 
	movwf	ANSELA
	movlw	B'00110000' ; AN10,AN11
	movwf	ANSELB
	movlw	B'11001101'	; AN4,6,7,8,9
	movwf	ANSELC
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00000000'	; channel 
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlw	B'10100000'	; right justified A/D result, fosc /32, Vdd to Vss A/D
	movwf	ADCON1
; oscillator	
	movlw	B'01111000'	; for 16MHz
	movwf	OSCCON		; osc
; timer2 set
	movlb	D'0'		; bank0	
	movlw	H'FE'
	movwf	PR2			; PWM period register
	movlw	B'00000100'
	movwf	T2CON
; timer1 set
	movlw	B'01010001'	; 8MHz counter
	movwf	T1CON
; PWM set
	movlb	D'12'		; bank12 PWM	
	movlw	B'11000000'	; set PWM mode
	movwf	PWM1CON		; enable PWM1 operation
	movwf	PWM3CON		; enable PWM3 operation
	movwf	PWM4CON		; enable PWM4 operation
	movlb	D'0'		; bank0	
; Initialise
	clrf	IMPEDANCE	; impedance measurement off
	movlw	H'0'		; initial VH effective for heater 2V effective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	H'88'		; initial VH effective for heater 
	movwf	VH_EFF1		; heater effective voltage ls
	movlw	D'4'
	movwf	REDUC_4		; interrupt decrease rate counter

; Temperature of sensor degrees C -40, -10, 20,  25,  50
; VH effective			standard  7.4, 7.8, 8.2, 8.4, 8.5(max)
; initially set at 2VHeff then to 7.4VHeff
; for 7.4V Veff is D504 or H1F8
; other values calculated by multiplying required effective voltage by 68.2 
; and converting to a hexadecimal value (2-byte)
;(actual calculation: /3 for battery voltage divider then /5 for value compared to A/D reference,
; multiply by 1023 for 10-bit range). Result is the same as x 68.2

	bsf		LOCK,0		; data in lock 
	clrf	ERROR_FLG	; error flag
	clrf	HEAT		; heater PWM calculated value
	clrf	HEAT_I		; interrupt heat value
	clrf	SWING_FLG	; amplitude swing flag
	clrf	TEMPK		; heat temperature
; set PWM 
	clrf	WIDEBAND0	; wideband value ms
	clrf	WIDEBAND1	; wideband value ls

; set S-CURVE ouput to 0
	clrf	S_CURVE
; start flag
	clrf	STARTFLAG	; startup flag
	clrf	DIM			; LED dimming counter

; measure VS/IP offset and so set current to pump cell to 0mA	
; A/D conversion
; channel 0

; Channel 0; VS/Ip
CH0
	movlw	B'00000001'	; CH0
	
	call	ACQUIRE_AD
	movwf	VS_IP0		; VS/IP ms byte
	movwf	CURRENT_IP0	; IP current ms
	movf	TEMPD1,w
	movwf	VS_IP1		; VS/IP ls byte
	movwf	CURRENT_IP1	; IP current ls 

; timers
	movlw	D'3'
	movwf	RATE		; ramp rate 187.5ms	
	movlw	D'244'		
	movwf	COUNT1		; 62.5ms counter (x 8 for 500ms)
	movlw	D'60'
	movwf	MIN_COUNT	; minute counter to 60 for 1/2 minute (30 seconds) countdown
	movlw	D'1'		; set at .5 seconds initially
	movwf	FOUR_SEC	; four seconds (8-half seconds)
	movlw	D'8'
	movwf	HALFSEC		; half second
	
; flags
	clrf	FLAG		; indicator flags
	bsf		FLAG,0		; 187.5ms flag

; interrupt enable 
	bcf		PIR1,TMR1IF	; timer1 flag off
	movlb	D'1' 
	bsf		PIE1,TMR1IE	; timer 1 interrupt enable
	movlb	D'0'
	bsf		INTCON,PEIE	; peripheral interrupt
	bsf		INTCON,GIE	; set global interrupt enable 
	bsf		IMPEDANCE,0	; impedance on

; wait for voltages to settle
CYC_DELAY
	movf	FOUR_SEC,w
	btfss	STATUS,Z	; when 0 continue on (counter decreased in interrupt)	
	goto	CYC_DELAY
	
	bcf		INTCON,GIE	; stop interrupt

; measure pressure sensor offset at VR6
; 0.5v/500m
; divide by 64 for about 7mV/500m
; pressure sensor is 1.25mV/kPa or ~7mV/500m (5.5kPa/500m)

	movlw	B'00101101'	; CH11
	
	call	ACQUIRE_AD
	movwf	TEMPD0		; ms
	movlw	D'6'
	movwf	TEMP
; divide by 64
DIV64P
	rrf		TEMPD0,f	; ms
	rrf		TEMPD1,f	; ls
	decfsz	TEMP,f
	goto	DIV64P
	
	movf	TEMPD1,w
	movwf	OFFSET		; 
	
; measure input voltage startup voltage set by VR2
; A/D conversion
; channel 8

	movlw	B'0100001'	; CH8
	
	call	ACQUIRE_AD

	movwf	VOLT0		; ms
	movf	TEMPD1,w	; ls
	movwf	VOLT1	
	
; measure VS/IP offset and so set current to pump cell to 0mA	
; A/D conversion
; channel 0

; Channel 0; VS/Ip
; CH0
	movlw	B'00000001'	; CH0
	
	call	ACQUIRE_AD
	movwf	VS_IP0		; VS/IP ms byte
	movwf	CURRENT_IP0	; IP current ms
	movf	TEMPD1,w
	movwf	VS_IP1		; VS/IP ls byte
	movwf	CURRENT_IP1	; IP current ls 

	bsf		INTCON,GIE	; interrupt on

; measure battery	
; A/D conversion
; channel 3

MEAS_BATT

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_YZ
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_YZ

	call	CH3			; measures battery voltage

; check battery voltage. Only start at above setup voltage at TP2 (VOLT0,VOLT1)
	movf	VOLT0,w			; battery voltage threshold for start ms byte typically at 13V
	subwf	BATT0,w			; ms battery  
	movwf	CALC0
	movf	VOLT1,w			; threshold ls byte 
	subwf	BATT1,w			; ls 
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then battery V < Volt0,Volt1
	goto	MEAS_BATT		; measure battery
	
HEATER_RUN1
; reset four second timer
	movlw	D'4'		; set at 2 seconds for 2Veff for 2-seconds 
	movwf	FOUR_SEC	; four seconds counter (8-half seconds)

; measure battery	
; A/D conversion
; channel 3
MEAS_BATT1

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_XYZ
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_XYZ

	call	CH3			; measures battery voltage
		
; start preheat at 2V Veff
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT

; run 2V Veff startup for 2 seconds

	movf	FOUR_SEC,w
	btfss	STATUS,Z	; when 0 continue on (counter decreased in interrupt)	
	goto	MEAS_BATT1

	movlw	D'8'		; set at 4 seconds 
	movwf	FOUR_SEC	; four seconds counter (8-half seconds)
	bsf		STARTFLAG,0	; set flag after 2-seconds preheat

; set Veff startup for ramp up 
	movlw	H'1'		; initial VH effective for heater D504 or H1F8 for 7.2V effective
	movwf	VH_EFF0		; heater effective voltage ms
	movlw	H'F8'		; initial VH effective for heater 
	movwf	VH_EFF1		; heater effective voltage ls
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT

; *********************************************************************************************
; program returns here. Normal program run
HEATER_RUN

; HEATER Control
; initial power up phase starts at VH effective of 7.4V. Ramps up at 0.4V per second 
; to 12V where the impedance of the Nernst cell is checked
; Current must not exceed 4A
; Duty cycle = (VH effective/ Vbattery ) squared
; battery voltage : H3FF from A/D is 15V

; check for ~10ms
	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_Z
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_Z
	
MEASURE_BATT
; measure battery	
; A/D conversion
; channel 3

	call	CH3			; battery voltage measured

	decf	TWO_COUNT,f	; decrease counter 
	movf	TWO_COUNT,w
	btfsc	STATUS,Z
	goto	MEAS_P
	movf	TWO_COUNT,w	; measure when count is 127
	xorlw	D'127'
	btfss	STATUS,Z
	goto	HEAT_I_AD

MEAS_P; measure pressure every ~1.25s
; Plus terminal
	movlw	B'00011101'	; CH7
	call	ACQUIRE_AD

	movwf	PRESSH0		; ms
	movf	TEMPD1,w
	movwf	PRESSH1
; minus terminal	
	movlw	B'00100101'	; CH9
	call	ACQUIRE_AD

	movwf	PRESSL0		; ms
	movf	TEMPD1,w
	movwf	PRESSL1

HEAT_I_AD

	movf	TWO_COUNT,w	; measure when count is 180
	xorlw	D'180'
	btfss	STATUS,Z
	goto	RAMP_UP

; measure heater current	
; A/D conversion
; channel 4

	movlw	B'00010001'	; CH4

	call	ACQUIRE_AD
	movwf	STORE0		; ms byte
	movf	TEMPD1,w
	movwf	STORE1		; ls byte

; check for over 4A (0.4V across 0.1 ohm) equivalent to D82 in A/D
; compare with (D82). If over set error flag (ERROR_FLG,7)and stop heater.

	movf	STORE0,w	; high byte if zero check ls byte
	btfss	STATUS,Z
	goto	SET_ERR		; not zero so set error
; check for over D82
	movf	STORE1,w
	sublw	D'82'
	btfss	STATUS,C	; if > error
	goto	SET_ERR

; check if <8 for open circuit
; check after first 4-seconds
	movf	FOUR_SEC,w 	; 
	btfss	STATUS,Z	; 
	goto	RAMP_UP
	btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	RAMP_UP		; bypass o/c test after ramp up
	movf	STORE1,w
	sublw	D'8'
	btfsc	STATUS,C
	goto	SET_ERR
	bcf		ERROR_FLG,7	; no error so clear flag
	goto	RAMP_UP

; under or over-current so off
SET_ERR
	clrf	VH_EFF0
	clrf	VH_EFF1		; effective Voltage at 0
	bsf		ERROR_FLG,7	; set error flag
	clrf	IMPEDANCE	; impedance measurement off
	clrf	HEAT		; heater off
; set PWM 
; wideband to lambda 1 (1.35V)
	movlw	H'1'
	movwf	WIDEBAND0	; wideband value ms
	movlw	H'15'
	movwf	WIDEBAND1	; wideband value ls
; S curve to 0.45V (D92)
	movlw	D'92'
	movwf	S_CURVE		; S curve output

; set current pump off 
; set with VS/IP offset	
; A/D conversion
; channel 0

	movf	VS_IP0,w	; VS/IP ms byte
	movwf	CURRENT_IP0	; IP current ms
	movf	VS_IP1,w	; VS/IP ls byte
	movwf	CURRENT_IP1	; IP current ls 
	
	bcf		FLAG,1		; clear ready for next ~10ms
SELF
	goto	SELF		; cycle without heater control

; check for ramp up phase
RAMP_UP

	btfsc	FLAG,7		; ramp up phase finished if flag is set
	goto	HEAT_CON	; control loop of temperature
	bcf		FLAG,1		; clear ready for next ~10ms
; check 187.5ms flag
	btfss	FLAG,0		; if set 187.5ms flag
	goto	DUTY_CALC	
	bcf		FLAG,0

; ramp up at 73.3mV/187.5ms (0.4V per second maximum ramp rate spec.) 390mV/s
; equivalent to digital value of 5
; starts at 7.4V
; this is a Veff ramp up that needs the PWM to be calculated

	movlw	D'5'		; 73.3mV x 68.2 see above for calculation of required value under Veff section
	addwf	VH_EFF1,f	; heater effective voltage
	btfsc	STATUS,C
	incf	VH_EFF0,f	; increase ms byte when carry
	call	CK_LIMIT	; 
DUTY_CALC
	call	DUTY		; calculate duty cycle for Veff depending on battery Voltage
	movwf	HEAT
	goto	HEAT_CON 

CK_LIMIT
; stop increment at Veff = 13V or Veff = 12V. Use Veff = 13V at start and Veff = 12V after 30s.
; if MIN_COUNT is clear 30 seconds have elapsed and use Veff = 12V as the maximum
; 30 seconds allows 2 x 60 x 200h (max allowable hours at 13V is 200h) or 24000 cold starts
; equivalent to more than 240,000kms if average trip is 10kms or more. Note that warm starts would effectively bypass
; the 13V heating
; A Veff of 12V is allowable continuously

; check minute counter
	movf	MIN_COUNT,w	; minute counter
	btfsc	STATUS,Z
	goto	COMP12		; 12V maximum	

; set 13V as maximum
; compare with 13V [D886, H376]
	movlw	H'03'			; 13V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	H'76'			; 13V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 13V
	return
; Veff > 13V so fix at 13V
	movlw	H'03'
	movwf	VH_EFF0			; set at 13V max
	movlw	H'76'			;
	movwf	VH_EFF1			; fix at 13V max 
	return

COMP12
; set 12V as maximum after 60-seconds
; compare with 12V [D818, H332]
	movlw	H'03'			; 12V max V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movlw	H'32'			; 12V max 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	btfsc	CALC0,7			; if set then V eff < 12V
	return
; Veff > 12V so fix at 12V
	movlw	H'03'
	movwf	VH_EFF0			; set at 12V max
	movlw	H'32'
	movwf	VH_EFF1			; fix at 12V max 
	return

DUTY
; calculation of PWM duty cycle during ramp up
; calculate duty cycle based on Vbatt and Veff
; Vbatt is 15V with H3FF measured 
; duty cycle is (VH effective/VBatt)squared. Result is a fraction between 1 and 0 for a heat value of 
; between 255 and 0
; Method: multiply VH effective by 255 (to gain resolution in result) and divide by VBatt. 
; Then square the value and divide by 255 to get a range of 0-255.

; if Vbatt <= Veff then set HEAT at maximum 

	movf	BATT0,w			; ms battery V 
	subwf	VH_EFF0,w		; ms V effective  
	movwf	CALC0
	movf	BATT1,w			; ls batt 
	subwf	VH_EFF1,w		; ls eff  
	btfss	STATUS,C
	decf	CALC0,f			; decrease if carry
	
	btfsc	CALC0,7			; if set then battery V > V eff
	goto	DO_CALC
	movlw	H'FF'			; set heat at max
	return

DO_CALC

; calculate duty cycle based on Vbatt and Veff
	movf	VH_EFF0,w
	movwf	AARGB0
	movf	VH_EFF1,w
	movwf	AARGB1
	clrf	BARGB0
	movlw	H'FF'
	movwf	BARGB1
	call	FXM1616U	; multiply Veff x 255
	
	movf	BATT0,w		; battery volts divisor
	movwf	BARGB0
	movf	BATT1,w		; min of 4 for 16 bit result
	movwf	BARGB1	
	call	FXD3216U	; divide by battery Voltage

; shift
	movf	AARGB2,w	; numerator
	movwf	AARGB0
	movwf	BARGB0
	movf	AARGB3,w
	movwf	AARGB1
	movwf	BARGB1
	call	FXM1616U	; square the value

; divide by 255
	clrf	BARGB0
	movlw	H'FF'	
	movwf	BARGB1
	call	FXD3216U
;	result in AARGB3
	movf	AARGB3,w
	return
	
HEAT_CON
; two loops check FLAG,7
; 1. during ramp up do not increase duty, but let ramp up process occur
; 2. after ramp up can alter heater drive at will. Uses lookup table to check if 
; the 12V Veff has been reached and alters the PWM value for control rather than Veff 
; set FLAG,7 when Nernst cell is 300 ohms indicating end of heatup phase
; when driving Nernst cell from RA7 with +/-2.5V ac via a 22k ohm impedance, 
; there is a 316mV swing from x 4.7 amplifier both high and low for 300 ohm on Nernst cell. 
; Equivalent to a change of D65 from A/D

 
; check if first swing values have been measured
	btfss	FLAG,2		; cycle ready flag 
	goto	CURRENT_CONTROL

; take SWINGH0,1 from SWINGL0,1 (or vice versa if negative) to get difference in swing across Nernst cell
; (voltages measured in interrupt)

WAIT_SWING1 ; wait for data
	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING1
	bcf		INTCON,GIE		; stop interrupt
	
; subtract Lower swing from Upper swing
	movf	SWINGL0,w		; ms lower 
	subwf	SWINGH0,w		; ms upper  
	movwf	SPAN0
	movf	SWINGL1,w		; ls lower 
	subwf	SWINGH1,w		; ls upper  
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry
	btfss	SPAN0,7			; if set then Lower swing > upper swing so rearrange calculation
	goto	COMPARE80

; subtract Upper swing from Lower swing
	movf	SWINGH0,w		; ms upper  
	subwf	SWINGL0,w		; ms lower 
	movwf	SPAN0
	movf	SWINGH1,w		; ls upper 
	subwf	SWINGL1,w		; ls lower
	movwf	SPAN1
	btfss	STATUS,C
	decf	SPAN0,f			; decrease if carry	
	
COMPARE80
	bsf		INTCON,GIE		; re allow interrupt
	movf	SPAN0,w
	btfss	STATUS,Z		; when ms byte of span is not 0, cell is >>300 ohms
	goto	SET_FF			; maximum heating allowed (either ramp up voltage if FLAG,7 not set
							; or at maximum (12V Veff) for FLAG,7 set.
; check if ramp up or control mode
	btfsc	FLAG,7			; when set use control loop for temperature control
	goto	HEAT_LOOP

CK_SPAN 
; D'140' is span for 300 ohms using 10k drive and 4.7 amplifier gain
	movf	SPAN1,w
	sublw	D'150'			; >300 ohms, set at higher than 300 ohms for controller to take over
	btfss	STATUS,C		; if minus (c=0) >300 ohms
	goto	SET_FF			; maximum heating allowed (either ramp up voltage if FLAG,7 not set
							; or at maximum (12V Veff) for FLAG,7 set.

; when < = 300 ohms set FLAG,7 (ramp up complete)
	movlw	D'120'			; set heat value at typical heat value required to maintain temperature at end of ramp up
	subwf	HEATX,w			; 
	btfss	STATUS,C
	clrw
	movwf	HEATX			; revised heat value
	bsf		FLAG,7			; end of ramp up, start of heat control
	goto	LOOP1

HEAT_LOOP

	movf	RESPONSE,w		; response to value
	btfsc	STATUS,Z
	goto	LOOP1			; check when response is clear
	decfsz	RESPONSE,f
	goto	CHECK_MAX
LOOP1
; check span values

	movf	SPAN1,w
	sublw	D'140'			; check offset (error value) from 300 ohms
	btfss	STATUS,Z
	goto	CK_Hi_LO
	clrf	TEMPK			; = 300 ohms so no temperature adjustment
	goto	CHECK_MAX		; = 300 ohms
CK_Hi_LO
	btfsc	STATUS,C		; if negative, span> 300 ohms
	goto	COOL
	movlw	D'140'
	subwf	SPAN1,w			; check offset from 300 ohms

HEATING
	movwf	TEMPK			; temperature correction value
	sublw	D'150'			; response rate adjust
	btfss	STATUS,C		; if negative set at 0	
	clrw
	movwf	RESPONSE		; response rate

; Heat change adjustment
	movf	TEMPK,w
	addwf	HEATX,w
	btfsc	STATUS,C
	movlw	H'FF'
	movwf	HEATX			; heat value change
	bsf		TEMPK,7			; set flag for lower temperature than required so heat required (temperature correction flag)
	goto	CHECK_MAX

COOL
	movwf	TEMPK			
	sublw	D'150'			; response rate adjust
	btfss	STATUS,C		; if negative set at 0	
	clrw
	movwf	RESPONSE

; Heat value change adjustment
	movf	TEMPK,w
	subwf	HEATX,w
	btfss	STATUS,C
	clrw
	movwf	HEATX
	goto	CHECK_MAX
	
SET_FF ;(heat)
	btfss	FLAG,7			; if heatup ramp bypass
	goto	BYPASS_FF
	movlw	H'FF'			; set heat at max	
	movwf	HEATX
	movlw	D'115'

	movwf	TEMPK
	bsf		TEMPK,7
	clrf	RESPONSE
	goto	CHECK_MAX
BYPASS_FF
	movf	HEAT,w			; get heat value
	movwf	HEATX			; place in working register

; check 12V Veff limit
; check if maximum duty for Veff of 12V has been reached
CHECK_MAX
; if in ramp up heat mode bypass max check
	btfss	FLAG,7		
	goto	CURRENT_CONTROL	
	rrf		BATT0,f		; divide battery volts by 4
	rrf		BATT1,f
	rrf		BATT0,w
	rrf		BATT1,f

	movf	BATT1,w
	sublw	D'204'
	movlw	D'204'		; load w ready
	btfsc	STATUS,C	; if plus then set at 204 (12V)
	movwf	BATT1
	comf	BATT1,w		; reverse the order
	call	LOOK_PWM	; get PWM value for 12V Veff max for the given battery voltage

	movwf	STORE		; temp value
	subwf	HEATX,w
	movf	STORE,w		; load limit to w ready 
	btfsc	STATUS,C	; if negative value was ok	
	movwf	HEATX		; limit PWM

	movf	HEATX,w
	btfss	STATUS,Z	; when zero show error
	goto	BY_ERRA	
; error/data LED flash when error
	bsf		ERROR_FLG,5 ; heater error when heat reduced to 0
	movwf	HEAT
	goto	CURRENT_CONTROL
BY_ERRA
	bcf		ERROR_FLG,5	; no error
	movwf	HEAT
	
CURRENT_CONTROL 
; keep current at zero until heater is in closed loop maintaining 780 degrees when FLAG,7 is set

	btfsc	FLAG,7	; 
	goto	CURRENT_LOOP

; use measured VS/IP offset to set current to pump cell to 0mA	
; A/D conversion

	bcf		FLAG,1	; clear ready for next ~10ms
CK_FLG_1
	btfss	FLAG,1	; ~10ms flag
	goto	CK_FLG_1
	goto	HEATER_RUN

CURRENT_LOOP

; VS detect assumes 2.5V (average) at AN0 for 450mV Nernst Cell Voltage. 
; Above 2.5 is rich. 
; Below 2.5V is lean

; VS is (SWINGL0,1 + SWINGH0,1)/2
; add

WAIT_SWING2 ; wait for data
	btfss	SWING_FLG,0		; if swing flag set can use data
	goto	WAIT_SWING2
	bcf		INTCON,GIE		; stop changes in interrupt
	bcf		SWING_FLG,0
	bcf		FLAG,1			; IP sense flag
	movf	SWINGL0,w
	addwf	SWINGH0,w
	movwf	VS_AVG0			; Ip sense average ms
	movf	SWINGL1,w
	addwf	SWINGH1,w
	bsf		INTCON,GIE		; allow changes in interrupt
	movwf	VS_AVG1			; Ip sense average ls
	btfsc	STATUS,C		; check carry	
	incf	VS_AVG0,f		; increase when ls byte overflows
; divide by 2
	bcf		STATUS,C		; carry clear
	rrf		VS_AVG0,f		; ms byte
	rrf		VS_AVG1,f		; ls byte

; if above 2.5V then mixture is rich so reduce CURRENT_IP0,IP1 in value
; if below 2.5V then mixture is lean so increase CURRENT_IP0,IP1 in value
; allow a hysteresis range above and below 2.5V because of sharp narrow band sensor output

; compare with 2.5V (H1FF)
; subtract VS_AVG0,1 from H1FF

	movf	VS_AVG0,w		; ms byte 
	sublw	H'1'			; ms 
	movwf	TEMP_AV0
	movf	VS_AVG1,w		; ls 
	sublw	H'FF'			; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
	iorwf	TEMP_AV0,w		; check if both TEMP_AV0 and TEMP_AV1 are at 0
	btfsc	STATUS,Z		; if zero bypass PWM change
	goto	SET_OUTPUT
	btfsc	TEMP_AV0,7		; if set then VS_AVG0,1 > H1FF so rearrange calculation
	goto	REDUCE

	movf	TEMP_AV0,w
	movwf	TEMP_AV0X
	movf	TEMP_AV1,w
	movwf	TEMP_AV1X

DEC_HYS_INC	
; add hysteresis
; value of 1 = (4.8mV) from VS_AVG0,1 equivalent to 1.02mV on VS
	movlw	D'5'			;  
	subwf	TEMP_AV1,w		; to 'w' for comparison with 0 (ls byte)
	movwf	TEMP_AV1		; equivalent to subwf TEMP_AV1,f but with value in w as well as f 
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry

	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	MULT_INC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	PROP_INC		; bypass multiply when 0

MULT_INC
; multiply by 2
;	bcf		STATUS,C
;	rlf		TEMP_AV1X,f
;	rlf		TEMP_AV0X,f
;	movf	TEMP_AV0X,w
;	sublw	D'3'			; if >3 set at 3
;	btfss	STATUS,C
;	goto	SET_MAX_INC
; multiply by 2
	bcf		STATUS,C
	rlf		TEMP_AV1X,f
	rlf		TEMP_AV0X,f
	movf	TEMP_AV0X,w
	sublw	D'3'			; if >3 set at 3
	btfsc	STATUS,C
	goto	PROP_INC
SET_MAX_INC
	movlw	D'3'
	movwf	TEMP_AV0X
	movlw	H'FF'
	movwf	TEMP_AV1X

; setup a proportional increase rate
PROP_INC 
; INCREASE
; add value to PWM
	movf	CURRENT_IP0,w	; ms byte 
	addwf	TEMP_AV0X,w
	movwf	TEMP_PWM0

	movf	TEMP_AV1X,w		; ls byte
	addwf	CURRENT_IP1,w
	movwf	TEMP_PWM1	
	btfsc	STATUS,C		; if carry increase ms byte if 2 or less
	incf	TEMP_PWM0,f

	movf	TEMP_PWM0,w
	sublw	D'3'			; if >3 do not increase
	btfsc	STATUS,C
	goto	TRANS_PWM		; place into current drive PWM

SET_MAX
; set at 3FF
	bcf		INTCON,GIE
	movlw	H'FF'
	movwf	CURRENT_IP1
	movwf	WIDEBAND1
	movlw	H'03'
	movwf	CURRENT_IP0		; ms 
	movwf	WIDEBAND0
	bsf		ERROR_FLG,7		; error LED on at limit
	goto	SET_OUTPUT

REDUCE
; subtract H1FF (2.5V) from VS_AVG0,1
	movlw	H'1'
	subwf	VS_AVG0,w		; ms byte 
	movwf	TEMP_AV0
	movlw	H'FF'			; ls 
	subwf	VS_AVG1,w		; ls 
	movwf	TEMP_AV1
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry

	movf	TEMP_AV0,w
	movwf	TEMP_AV0X
	movf	TEMP_AV1,w
	movwf	TEMP_AV1X

DEC_HYS_DEC
; add hysteresis
; Value of 1 = (4.8mV) from VS_AVG0,1 equivalent to 1.02mV on Vs
	movlw	D'5'			; 5 = 5.1mV on Vs 
	subwf	TEMP_AV1,w		; w for comparison (ls byte)
	movwf	TEMP_AV1		;  
	btfss	STATUS,C
	decf	TEMP_AV0,f		; decrease if carry
	
	btfss	TEMP_AV0,7		; if set then negative so set at 0
	goto	MULT_DEC
	clrf	TEMP_AV0
	clrf	TEMP_AV1
	goto	PROP_DEC		; bypass multiply when 0

MULT_DEC
; multiply by 2
;	bcf		STATUS,C
;	rlf		TEMP_AV1X,f
;	rlf		TEMP_AV0X,f
;	movf	TEMP_AV0X,w
;	sublw	D'3'			; if >3 set at 3
;	btfss	STATUS,C
;	goto	SET_MAX_DEC
; multiply by 2
	bcf		STATUS,C
	rlf		TEMP_AV1X,f
	rlf		TEMP_AV0X,f
	movf	TEMP_AV0X,w
	sublw	D'3'			; if >3 set at 3
	btfsc	STATUS,C
	goto	PROP_DEC
SET_MAX_DEC
	movlw	D'3'
	movwf	TEMP_AV0X
	movlw	H'FF'
	movwf	TEMP_AV1X

; setup a proportional decrease rate
PROP_DEC
; REDUCE VALUE
	movf	TEMP_AV0X,w		; ms byte 
	subwf	CURRENT_IP0,w
	movwf	TEMP_PWM0

	movf	TEMP_AV1X,w
	subwf	CURRENT_IP1,w	; ls 
	movwf	TEMP_PWM1
	btfss	STATUS,C
	decf	TEMP_PWM0,f		; decrease ms if carry
	btfsc	TEMP_PWM0,7		; if set then negative so set at 0
	goto	ZERO_SET
; transfer to PWM
TRANS_PWM
	bcf		INTCON,GIE	
	movf	TEMP_PWM0,w		; ms byte 
	movwf	CURRENT_IP0
	movf	TEMP_PWM1,w		; ls byte 
	movwf	CURRENT_IP1
	goto	SET_OUTPUT

ZERO_SET
	bcf		INTCON,GIE
	clrf	CURRENT_IP0
	clrf	CURRENT_IP1	

; allow 10ms for pump action
; Nernst cell at 450mV is set to 2.5V

SET_OUTPUT 
	
; When AN10 (Vs - 0.45V x 4.7V) is close to 2.5V (0.45V across Nernst cell)
; VS_AVG0,1 is the average AN0 value
; TEMP_AV1 is the offset from 2.5V 
; so if TEMP_AV1 is zero load values

	bsf		INTCON,GIE		; resume interrupt
	movf	TEMP_AV0,w
	btfss	STATUS,Z
	goto	NEW_RUN 		; if ms byte is zero check ls byte
	
	movf	TEMP_AV1,w		; if zero, load curve outputs
	btfss	STATUS,Z		; if 0 value 
	goto	NEW_RUN			; clears LOCK when value not suitable		
	btfsc	LOCK,0			; if in lock last time 
	goto	CK_CURRENT
	bsf		LOCK,0
	goto	HEATER_RUN
NEW_RUN
	clrf	LOCK
	goto	HEATER_RUN

CK_CURRENT 
; when data is correct, flash DATA LED 
	bsf		LOCK,0
	bsf		ERROR_FLG,6		; data flag

READINGS

; get (amplified) IP sense value and subtract from VS_IP
; SENSE_IP0,1 is IP sense at AN6
; subtract VS_IP from IP Sense

	movf	VS_IP0,w		; ms VS_IP 
	subwf	SENSE_IP0,w		; ms IP sense  
	movwf	TEMPB0
	movf	VS_IP1,w		; ls  
	subwf	SENSE_IP1,w		; ls  
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry
	btfss	TEMPB0,7		; if set then VS_IP > IP sense so rearrange calculation
	goto	SET_WIDE_LEAN

; subtract 
	movf	SENSE_IP0,w		; ms IP sense 
	subwf	VS_IP0,w		; ms VS_IP 
	movwf	TEMPB0
	movf	SENSE_IP1,w		; ls  
	subwf	VS_IP1,w		; ls 
	movwf	TEMPB1
	btfss	STATUS,C
	decf	TEMPB0,f		; decrease if carry	

SET_WIDE_RICH ;(Lambda <1.016)
; pressure correction
	bsf		RICHLEAN,0		; sets for rich correction
	call	TEMPERATURE		; temperature correction
	call	PRESSURE 

; values obtained from calculated table. See wideband table at end of file
; <= D161?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'161'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 > 151 
	goto	RICH1
; <= 245?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'245'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 245
	goto	RICH2
; <= D358?
	movf	TEMPB0,w 		; ms 
	sublw	D'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'66'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > D348
	goto	RICH3
; <= D587?
	movf	TEMPB0,w 		; ms 
	sublw	D'2'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'4B'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 596
	goto	RICH4
; above D587 is out of range so error
	bcf		INTCON,GIE		; stop changes in interrupt
	clrf	WIDEBAND0		; clear ms PWM byte for wideband output	
	clrf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		INTCON,GIE		; allow changes in interrupt
	bsf		ERROR_FLG,7		; error LED on at limit
	goto	NARROW_BAND

RICH1
; 1.016-0.9
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)

				
; TEMPB0,1 between D0 and D161, Lambda is between 1.016 and 0.9, output for wideband is from D283 to D180 
; 1.39V to 0.88V.
; Calculation for wideband PWM output is D277 - (((TEMPB0,1- D'0')/D161) x D97). Note D97 = D277-D180

	movlw	H'01'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'1B'
	movwf	MAX1			; max value D269 = H010D
	movlw	D'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'161'			; divide value
	movwf	DIV1
	movlw	D'97'

	goto	CALC_PWM_WIDE_R

RICH2
; 0.9-0.85
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)
;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
				
; TEMPB0,1 between D161 and D245, Lambda is between 0.9 and 0.85, output for wideband is from D180 to D135 
; 0.88V to 0.66V.
; Calculation for wideband PWM output is D180 - (((TEMPB0,1-D161)/D84) x D45).
; Note D45 = D180-D135; D94 is D245-D161

	movlw	D'00'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'180'
	movwf	MAX1			; max value 
	movlw	D'161'			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'84'			; divide value
	movwf	DIV1
	movlw	D'45'

	goto	CALC_PWM_WIDE_R

RICH3
; 0.85-0.8

;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)

				
; TEMPB0,1 between D245 and D358, Lambda is between 0.85 and 0.8, output for wideband is from D135 to D90 
; 0.66V to 0.44V.
; Calculation for wideband PWM output is D135 - (((TEMPB0,1-D245)/D113) x D45).
; Note D45 = D135-D90; D113 is D358-D245

	movlw	D'00'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'135'
	movwf	MAX1			; max value 
	movlw	D'245'			; subtract value from TEMPB0,1
	clrf	SUB0			; subtract value ms
	movwf	SUB1			; subtract value ls
	movlw	D'113'			; divide value
	movwf	DIV1
	movlw	D'45'

	goto	CALC_PWM_WIDE_R

RICH4
; 0.8-0.7

;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)
;*0.7		-1.82 mA	2.87V (D587)	0V (D0)			
				
; TEMPB0,1 between D358 and D596, Lambda is between 0.8 and 0.7, output for wideband is from D90 to D0 
; 0.44V to 0.0V.
; Calculation for wideband PWM output is D90 - (((TEMPB0,1-D358)/D238) x D90).
; Note D90 = D90-D0; D248 is D596-D358

	movlw	H'0'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	D'90'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'66'
	movwf	SUB1			; subtract value ls
	movlw	D'238'			; divide value
	movwf	DIV1
	movlw	D'90'

	goto	CALC_PWM_WIDE_R

SET_WIDE_LEAN
; pressure correction
	bcf		RICHLEAN,0		; sets for lean correction
	call	TEMPERATURE		; temperature correction
	call	PRESSURE

; <= D18?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'18'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP > 18 
	goto	LEAN1
; <= D37?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'37'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN2
; <= D55?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'55'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN3
; <= D73?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'73'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN4
; <= D91?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'91'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN5
; <= D109?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'109'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN6
; <= D128?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'128'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN7
; <= D146?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'146'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN8
; <= D164?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'164'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN9
; <= D183?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'183'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN10
; <= D201?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'201'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN11
; <= D219?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'219'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN12
; <= D238?
	movf	TEMPB0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	D'238'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN13
; <= D257?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'1'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN14
; <= D276?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'14'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN15
; <= D295?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'27'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN16
; <= D306?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'32'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN17
; <= D314?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'3A'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN18
; <= D334?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'4E'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN19
; <= D345?
	movf	TEMPB0,w 		; ms 
	sublw	H'1'
	movwf	TEMP
	movf	TEMPB1,w		; ls  
	sublw	H'59'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then TEMPB0,1 VS_IP >  
	goto	LEAN20
; above D345 is out of range so set at 5V
; if VS/Ip is 2V show Ip
; Check for Air Calibration mode (VS/IP = 2V)or normal mode.

	movlw	D'02'			; VS/IP ms
	subwf	VS_IP0,w	
	btfss	STATUS,C		; if positive normal mode
	goto	IN_AIR

	bcf		INTCON,GIE		; stop changes in interrupt
	btfss	PORTA,5
	goto 	IN_AIR
	movlw	H'03'
	movwf	WIDEBAND0		; set ms PWM byte for max wideband output
	movlw	H'FF'	
	movwf	WIDEBAND1		; set ls PWM byte for max wideband output
	bsf		INTCON,GIE		; allow changes in interrupt
	bsf		ERROR_FLG,7		; error LED on at limit
	goto	NARROW_BAND

IN_AIR
	movf	TEMPB0,w		; TEST to show Ip for in air measurement			; 
	movwf	WIDEBAND0		; set ms PWM byte for wideband output
	movf	TEMPB1,w		; TEST to show Ip for in air measurement
	movwf	WIDEBAND1		; set ls PWM byte for wideband output
	bsf		INTCON,GIE		; allow changes in interrupt
	goto	NARROW_BAND

LEAN1
; 1.016 to 1.026
; TEMPB0,1 between D0 and D18, Lambda is between 1 and 1.026, output for wideband is from D277 to D292 
; 1.43V to 1.39V.
; Calculation for wideband PWM output is D292 - (((D18-TEMPB0,1)/D18) x D15).
; Note D15 = D292-D277; D18 is D18-D0

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'24'
	movwf	MAX1			; max value D285
	movlw	D'18'			; subtract value from TEMPB0,1
	movwf	SUB1			; subtract value ls	
	clrf	SUB0			; subtract value ms
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'15'

	goto	CALC_PWM_WIDE
LEAN2
; 1.026 to 1.0534
; TEMPB0,1 between D18 and D37, output for wideband is from D292 to D317 
; Calculation for wideband PWM output is D317 - (((D37-TEMPB0,1)/D19) x D25).
; Note D25 = D317-D292; D19 is D37-D18

	movlw	D'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'3D'
	movwf	MAX1			; max value D317
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'37'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'25'

	goto	CALC_PWM_WIDE
LEAN3
; 1.0534 to 1.0823
; TEMPB0,1 between D37 and D55, output for wideband is from  D317 to D343
; Calculation for wideband PWM output is D343 - (((D55-TEMPB0,1)/D18) x D26).
; Note D26 = D343-D317; D18 is D55-D37

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'57'
	movwf	MAX1			; max value D343
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'55'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'26'

	goto	CALC_PWM_WIDE
LEAN4
; 1.0823 to 1.112
; TEMPB0,1 between D55 and D73, output for wideband is from D343 to D370
; Calculation for wideband PWM output is D370 - (((D73-TEMPB0,1)/D18) x D27).
; Note MULT D27 = D370-D343; DIV D18 is D73-D55

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'72'
	movwf	MAX1			; max value D370
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'73'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'27'

	goto	CALC_PWM_WIDE
LEAN5
; 1.112 to 1.1445
; TEMPB0,1 between D73 and D91, output for wideband is from D399 to D370
; Calculation for wideband PWM output is D399 - (((D91-TEMPB0,1)/D18) x D29).
; Note MULT D29 = D399-D370; DIV D18 is D91-D73

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'8F'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'91'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'29'

	goto	CALC_PWM_WIDE
LEAN6 
; 1.1445 to 1.18
; TEMPB0,1 between D91 and D109, output for wideband is from D399 to D431
; Calculation for wideband PWM output is D431 - (((D109-TEMPB0,1)/D18) x D32).
; Note MULT D32 = D431-D399; DIV D18 is D109-D91

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'AF'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'109'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'32'

	goto	CALC_PWM_WIDE
LEAN7 
; 1.18 to 1.214
; TEMPB0,1 between D109 and D128, output for wideband is from D431 to D462
; Calculation for wideband PWM output is D462 - (((D128-TEMPB0,1)/D19) x D31).
; Note MULT D31 = D462-D431; DIV D19 is D128-D109

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'CE'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'128'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'31'

	goto	CALC_PWM_WIDE
LEAN8 
; 1.214 to 1.2516
; TEMPB0,1 between D128 and D146, output for wideband is from D462 to D495
; Calculation for wideband PWM output is D495 - (((D146-TEMPB0,1)/D18) x D33).
; Note MULT D33 = D495-D462; DIV D18 is D146-D128

	movlw	H'1'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'EF'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'146'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'33'

	goto	CALC_PWM_WIDE
LEAN9 
; 1.2516 to 1.2916
; TEMPB0,1 between D146 and D164, output for wideband is from D531 to D495
; Calculation for wideband PWM output is D531 - (((D164-TEMPB0,1)/D18) x D36).
; Note MULT D36 = D531-D495; DIV D18 is D164-D146

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'13'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'164'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'36'

	goto	CALC_PWM_WIDE
LEAN10 
; 1.2916 to 1.33
; TEMPB0,1 between D164 and D183, output for wideband is from D531 to D566
; Calculation for wideband PWM output is D566 - (((D183-TEMPB0,1)/D19) x D35).
; Note MULT D35 = D566-D531; DIV D19 is D183-D164

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'36'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'183'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'35'

	goto	CALC_PWM_WIDE
LEAN11 
; 1.33 to 1.38
; TEMPB0,1 between D183 and D201, output for wideband is from D611 to D566
; Calculation for wideband PWM output is D611 - (((D201-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D611-D566; DIV D18 is D201-D183

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'63'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'201'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'45'

	goto	CALC_PWM_WIDE
LEAN12 
; 1.38 to 1.43
; TEMPB0,1 between D201 and D219, output for wideband is from D611 to D656
; Calculation for wideband PWM output is D656 - (((D219-TEMPB0,1)/D18) x D45).
; Note MULT D45 = D656-D611; DIV D18 is D219-D201

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'90'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'219'
	movwf	SUB1			; subtract value ls
	movlw	D'18'			; divide value
	movwf	DIV1
	movlw	D'45'

	goto	CALC_PWM_WIDE
LEAN13 
; 1.43 to 1.48
; TEMPB0,1 between D219 and D238, output for wideband is from D701 to D656
; Calculation for wideband PWM output is D701 - (((D238-TEMPB0,1)/D19) x D45).
; Note MULT D45 = D701-D656; DIV D19 is D238-D219

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'BD'
	movwf	MAX1			; max value 
	movlw	H'0'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	D'238'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'45'

	goto	CALC_PWM_WIDE
LEAN14 
; 1.48 to 1.5346
; TEMPB0,1 between D238 and D257, output for wideband is from D701 to D750
; Calculation for wideband PWM output is D750 - (((D257-TEMPB0,1)/D19) x D49).
; Note MULT D49 = D750-D701; DIV D19 is D257-D238

	movlw	H'2'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'EE'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'1'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'49'

	goto	CALC_PWM_WIDE
LEAN15 
; 1.5346 to 1.594
; TEMPB0,1 between D257 and D276, output for wideband is from D803 to D750
; Calculation for wideband PWM output is D803 - (((D276-TEMPB0,1)/D19) x D53).
; Note MULT D53 = D803-D750; DIV D19 is D276-D257

	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'23'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'14'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'53'

	goto	CALC_PWM_WIDE
LEAN16 
; 1.594 to 1.658
; TEMPB0,1 between D276 and D295, output for wideband is from D803 to D861
; Calculation for wideband PWM output is D861 - (((D295-TEMPB0,1)/D19) x D53).
; Note MULT D58 = D861-D803; DIV D19 is D295-D276

	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'5D'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'27'
	movwf	SUB1			; subtract value ls
	movlw	D'19'			; divide value
	movwf	DIV1
	movlw	D'58'

	goto	CALC_PWM_WIDE
LEAN17 
; 1.658 to 1.7
; TEMPB0,1 between D295 and D306, output for wideband is from D898 to D861
; Calculation for wideband PWM output is D898 - (((D306-TEMPB0,1)/D11) x D31).
; Note MULT D37 = D898-D861; DIV D11 is D306-D295

	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'82'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'32'
	movwf	SUB1			; subtract value ls
	movlw	D'11'			; divide value
	movwf	DIV1
	movlw	D'37'

	goto	CALC_PWM_WIDE
LEAN18 
; 1.7 to 1.73
; TEMPB0,1 between D306 and D314, output for wideband is from D898 to D925
; Calculation for wideband PWM output is D925 - (((D314-TEMPB0,1)/D8) x D27).
; Note MULT D27 = D925-D898; DIV D8 is D314-D306
	
	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'9D'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'3A'
	movwf	SUB1			; subtract value ls
	movlw	D'8'			; divide value
	movwf	DIV1
	movlw	D'27'

	goto	CALC_PWM_WIDE
LEAN19 
; 1.73 to 1.80
; TEMPB0,1 between D314 and D334, output for wideband is from D925 to D988
; Calculation for wideband PWM output is D988 - (((D334-TEMPB0,1)/D20) x D63).
; Note MULT D63 = D988-D925; DIV D20 is D334-D314

	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'DC'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'4E'
	movwf	SUB1			; subtract value ls
	movlw	D'20'			; divide value
	movwf	DIV1
	movlw	D'63'

	goto	CALC_PWM_WIDE
LEAN20 
; 1.80 to 1.84
; TEMPB0,1 between D334 and D345, output for wideband is from D1023 to D988
; Calculation for wideband PWM output is D1023 - (((D345-TEMPB0,1)/D11) x D35).
; Note MULT D35 = D1023-D988; DIV D11 is D345-D334

	movlw	H'3'			; ms byte of max value
	movwf	MAX0			; max ms byte
	movlw	H'FF'
	movwf	MAX1			; max value 
	movlw	H'1'			; subtract value from TEMPB0,1
	movwf	SUB0			; subtract value ms
	movlw	H'59'
	movwf	SUB1			; subtract value ls
	movlw	D'11'			; divide value
	movwf	DIV1
	movlw	D'35'

	goto	CALC_PWM_WIDE

CALC_PWM_WIDE_R
; RICH Calculation
	movwf	MULT1
	bcf		ERROR_FLG,7		; ERROR LED off	
; take subtract value from TEMPB0,1
	movf	SUB0,w 			; ms 
	subwf	TEMPB0,w		; subtract value
	movwf	AARGB0
	movf	SUB1,w			; ls  
	subwf	TEMPB1,w		; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	
	goto	MULT_ALL

CALC_PWM_WIDE
; lean calculation
	movwf	MULT1
	bcf		ERROR_FLG,7		; ERROR LED off	
; take  TEMPB0,1 from subtract value 
	movf	TEMPB0,w 		; ms 
	subwf	SUB0,w			; subtract value
	movwf	AARGB0
	movf	TEMPB1,w		; ls  
	subwf	SUB1,w			; subtract value ls  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry	

MULT_ALL
; multiply
	clrf	BARGB0
	movf	MULT1,w			; multiply value
	movwf	BARGB1
	call	FXM1616U		; multiply

; divide
	movf	DIV1,w
	movwf	BARGB1
	clrf	BARGB0
	call	FXD3216U		; divide

; subtract from maximum 
	
	movf	AARGB2,w 		; ms 
	subwf	MAX0,f			; subtract value
	movf	AARGB3,w		; ls  
	subwf	MAX1,f			; subtract value ls  
	btfss	STATUS,C
	decf	MAX0,f	

; place in PWM

	bcf		INTCON,GIE		; stop changes in interrupt
	movf	MAX0,w
	movwf	WIDEBAND0		; ms PWM byte for wideband output	
	movf	MAX1,w
	movwf	WIDEBAND1		; ls PWM byte for wideband output
	bsf		INTCON,GIE		; allow changes in interrupt

NARROW_BAND

; S_CURVE output
; Calculation for narrowband S-CURVE
; divide WIDEBAND0,1 by 4 
WIDE_4
	bcf		STATUS,C
	rrf		WIDEBAND0,w		; divide by 2
	movwf	ADDN_S0			; store
	rrf		WIDEBAND1,w
	movwf	ADDN_S1			; store
	bcf		STATUS,C
	rrf		ADDN_S0,f		; divide by 2 again		
	rrf		ADDN_S1,f		; 

; check if range is < 22 or > if 105 
; less than 22
	movlw	D'22'
	subwf	ADDN_S1,f
	btfss	STATUS,C		; if minus then less than 22
	clrf	ADDN_S1			; set at 0 if less
; compare to 83 (105 -22) (max range)
	movlw	D'83'
	subwf	ADDN_S1,w
	movlw	D'83'			; maximum
	btfsc	STATUS,C
	movwf	ADDN_S1			; set at 83 if >83
	
; set S_CURVE value accordingly	
	movf	ADDN_S1,w		; get value
	call	LOOKUP_S_CURVE	; lookup table to set up S-curve output

	movwf	S_CURVE
	goto	HEATER_RUN

TEMPERATURE
	
; temperature correction for variation in sensor temperature
; TEMPK has offset of span (impedance)Span is 140 for 300 ohms. span of 70 is 150 ohms and span of 280 is 600 ohms.
; These limits are 100 degrees C changes and 4% correction required
; if TEMPK,7 is set then TEMPB0,TEMPB1 should be increased,(3500 + TEMPK)/3500 for 4%/100 deg C change (up to 600 ohms)
; if TEMPK,7 is clear then TEMPB0,TEMPB1 should be decreased,(1750 - TEMPK)/1750 for 4%/100 deg C change
; typically corrects for -82 degrees under the 300 ohms temperature and up to 100 degrees over.

	btfsc	TEMPK,7		; if clear
	goto	CALC4
 	
; set divisor at 1750
	movlw	H'6'
	movwf	WORK0	; divisor
	movwf	WORK3	; multiplier
	movlw	H'D6'
	movwf	WORK1	; divisor

; take TEMPK from 1750 (H'6D6')
	movf	TEMPK,w
	sublw	H'D6'
	btfss	STATUS,C		; if negative decrease ms
	decf	WORK3,f
	goto	MULT_CORR		; multiply then divide/1750
CALC4
	
	bcf		TEMPK,7	
; multiply and then divide by 3500 (H'DAC')
; set divisor
	
	movlw	H'D'
	movwf	WORK0	; divisor
	movwf	WORK3	; multiplier
	movlw	H'AC'
	movwf	WORK1	; divisor

; add TEMPK to 3500 (H'DAC')
	movf	TEMPK,w	
	addlw	H'AC'
	btfsc	STATUS,C	; if over increase ms byte
	incf	WORK3,f
	goto	MULT_CORR

PRESSURE; pressure correction
	
; check if no sensor
; OUT+ (PRESSH0,1) (at AN7) is at 0V
; compare if >1V
	movf	PRESSH0,w 		; ms 
	sublw	D'0'
	movwf	TEMP
	movf	PRESSH1,w		; ls  
	sublw	D'204'			; ls  
	btfss	STATUS,C
	decf	TEMP,f			; decrease if carry
	btfss	TEMP,7			; if set then pressh>1V 
	return					; no sensor	

; set divisor to 100
	movlw	D'100'
	movwf	WORK1
	clrf	WORK0
; multiplier MS byte
	clrf	WORK3
; subtract Lower value from Upper value
	movf	PRESSL0,w		; ms lower 
	subwf	PRESSH0,w		; ms upper  
	movwf	AARGB0
	movf	PRESSL1,w		; ls lower 
	subwf	PRESSH1,w		; ls upper  
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f		; decrease if carry
	btfss	AARGB0,7		; if set then rearrange calculation
	goto	POSITIVE_PRESSURE

; subtract Upper value from Lower value
	movf	PRESSH0,w		; ms upper  
	subwf	PRESSL0,w		; ms lower 
	movwf	AARGB0
	movf	PRESSH1,w		; ls upper 
	subwf	PRESSL1,w		; ls lower
	movwf	AARGB1
	btfss	STATUS,C
	decf	AARGB0,f
; add offset
	movf	AARGB1,w
	addwf	OFFSET,w
	addlw	D'100'	
	goto	MULT_CORR

POSITIVE_PRESSURE
; take away OFFSET
	movf	OFFSET,w
	subwf	AARGB1,w
	btfss	STATUS,C		; if negative
	goto	REV_CALC		; negative

	movwf	AARGB1
	sublw	D'15'
	movlw	D'15'			; set at 15 max	
	btfss	STATUS,C
	movwf	AARGB1			; if >15 set at 15
	movf	AARGB1,w	
	btfsc	RICHLEAN,0		; check rich or lean
	goto	PRESSURE_RICH_SET
	call	PRESS_LEAN		; percentage calculation in W
	goto	MULT_CORR
PRESSURE_RICH_SET
	call	PRESS_RICH

; mult (Ip: TEMPB0, TEMPB1 by w plus divide)
MULT_CORR
; multiply
	movwf	AARGB1
	movf	WORK3,w
	movwf	AARGB0
	movf	TEMPB0,w
	movwf	BARGB0
	movf	TEMPB1,w		; multiply value
	movwf	BARGB1
	
	call	FXM1616U		; multiply

; divide
	movf	WORK1,w
	movwf	BARGB1
	movf	WORK0,w
	movwf	BARGB0
	call	FXD3216U		; divide 

; result to TEMPB0,1	
	movf	AARGB2,w 		; ms 
	movwf	TEMPB0
	movf	AARGB3,w
	movwf	TEMPB1
	return					; correction completed

REV_CALC
	movf	AARGB1,w
	subwf	OFFSET,w
	addlw	D'100'
	goto	MULT_CORR
	
; *****************************************************************************************
; Subroutines

; subroutine to wait for A/D conversion
ACQUIRE_AD
	movlb	D'1'		; bank1	ADCON0
	movwf	ADCON0		; set channel
; wait >8us to charge input capacitance/ 
	movlw	D'15'
	movwf	STORE3
WAIT2C1
	decfsz	STORE3,f
	goto	WAIT2C1	
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	
	movf	ADRESL,w
	movwf	TEMPD1		; ls
	movf	ADRESH,w
	movlb	D'0'		; bank0
	return

; battery channel3
CH3
	movlw	B'00001101'	; CH3
	
	call	ACQUIRE_AD
	movwf	BATT0		; ms
	movf	TEMPD1,w
	movwf	BATT1		; ls
	return
	
; Math routines
; 32/16 Bit Unsigned Fixed Point Divide 32/16 -> 32.16
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 16 bit unsigned fixed point divisor in BARGB0, BARGB1
; Use: CALL FXD3216U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 16 bit unsigned fixed point remainder in REMB0, REMB1
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 2+699+2 = 703 clks
; Min Timing: 2+663+2 = 667 clks

FXD3216U 
	CLRF REMB0
	CLRF REMB1

	CLRF TEMP
	RLF AARGB0,W
	RLF REMB1, F
	MOVF BARGB1,W
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	RLF AARGB0, F
	MOVLW H'7'
	MOVWF LOOPCOUNT
LOOPU3216A 
	RLF AARGB0,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB0,0
	GOTO UADD26LA
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26LA
UADD26LA 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26LA
	RLF AARGB0, F
	DECFSZ LOOPCOUNT, F
	GOTO LOOPU3216A
	RLF AARGB1,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB0,0
	GOTO UADD26L8
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26L8
UADD26L8 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26L8
	RLF AARGB1, F
	MOVLW H'7'
	MOVWF LOOPCOUNT
LOOPU3216B
	RLF AARGB1,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB1,0
	GOTO UADD26LB
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26LB
UADD26LB
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26LB 
	RLF AARGB1, F
	DECFSZ LOOPCOUNT, F
	GOTO LOOPU3216B
	RLF AARGB2,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB1,0
	GOTO UADD26L16
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26L16
UADD26L16 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26L16 
	RLF AARGB2, F
	MOVLW H'7'
	MOVWF LOOPCOUNT
LOOPU3216C 
	RLF AARGB2,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB2,0
	GOTO UADD26LC
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26LC
UADD26LC 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26LC 
	RLF AARGB2, F
	DECFSZ LOOPCOUNT, F
	GOTO LOOPU3216C
	RLF AARGB3,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB2,0
	GOTO UADD26L24
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26L24
UADD26L24 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F

UOK26L24 	
	RLF AARGB3, F
	MOVLW H'7'
	MOVWF LOOPCOUNT
LOOPU3216D 
	RLF AARGB3,W
	RLF REMB1, F
	RLF REMB0, F
	RLF TEMP, F
	MOVF BARGB1,W
	BTFSS AARGB3,0
	GOTO UADD26LD
	SUBWF REMB1, F
	MOVF BARGB0,W
	BTFSS STATUS,C
	INCFSZ BARGB0,W
	SUBWF REMB0, F
	CLRW
	BTFSS STATUS,C
	MOVLW H'1'
	SUBWF TEMP, F
	GOTO UOK26LD
UADD26LD 
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
	CLRW
	BTFSC STATUS,C
	MOVLW H'1'
	ADDWF TEMP, F
UOK26LD 
	RLF AARGB3, F
	DECFSZ LOOPCOUNT, F
	GOTO LOOPU3216D
	BTFSC AARGB3,0
	GOTO UOK26L
	MOVF BARGB1,W
	ADDWF REMB1, F
	MOVF BARGB0,W
	BTFSC STATUS,C
	INCFSZ BARGB0,W
	ADDWF REMB0, F
UOK26L
	return



; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0,1
;               16 bit unsigned fixed point multiplier in BARGB0,1

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0,1,2,3

FXM1616U 	
			    CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMP160
                MOVF    AARGB1,W
                MOVWF   TEMP161
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
				RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMP161,W
                ADDWF   AARGB1, F
                MOVF    TEMP160,W
                BTFSC   STATUS,C
                INCFSZ  TEMP160,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMP161,W
                ADDWF  AARGB1, F
                MOVF   TEMP160,W
                BTFSC  STATUS,C
                INCFSZ TEMP160,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
				RETURN


	end

;Wideband data	
	
;Lambda	O2 concentration %	Ip mA	A/D value of Span relative to Vs/Ip @ AN0 and Ip sense @ AN6	0-5V curve output value(PWM decimal value)
				
;CALIBRATE	For Air 20.9%		Vs/Ip = 2V, TP4=2.343 IC5b has 560k resistors paralleled for 560k/2
; and gain of 12.73 for current measurement at AN6
; Vs/Ip = 2V(gain is 12.73 with 560k in parallel)	


;				
;				
;Standard range 0.7 to 1.84				
; lambda O2%	Ip		AN0-AN6			Wideband out				
;1.84	9.24%	1.07 mA	1.685V (D345)	5V (D1023)
;1.80	9%	1.036 mA	1.633V (D334)	4.83V (D988)
;1.73	8.5%	0.975 mA	1.536V (D314)	4.52V (D925)
;*1.7	8.29%	0.95mA mA	1.50V (D306)	4.39V (D898)
;1.658	8%	0.9158 mA	1.443V (D295)	4.21V (D861)
;1.594	7.5%	0.8568 mA	1.35V (D276)	3.92V (D803)
;1.5346	7%	0.7979 mA	1.257V (D257)	3.67V (D750)
;1.48	6.5%	0.73895 mA	1.164V (D238)	3.43V (D701)
;*1.43	6.0%	0.68 mA	1.07V (D219)	3.21V (D656)
;1.38	5.5%	0.6233 mA	0.982V (D201)	2.99V (D611)
;1.33	5%	0.5666 mA	0.893V (D183)	2.77V (D566)
;1.2916	4.5%	0.510 mA	0.804V (D164)	2.60V (D531)
;1.216	4%	0.45338 mA	0.714V (D146)	2.42V (D495)
;1.214	3.5%	0.39666 mA	0.625V (D128)	2.26V (D462)
;*1.18	3%	0.34 mA	0.54V (D109)	2.11V (D431)
;1.1445	2.5%	0.28333 mA	0.446V (D91)	1.95V (D399)
;1.112	2%	0.22666 mA	0.3571V (D73)	1.81V (D370)
;1.08227	1.5%	0.170 mA	0.2678V (D55)	1.68V (D343)
;1.0534	1%	0.1133 mA	0.1786V (D37)	1.55V (D317)
;1.026	0.5%	0.0566 mA	0.0893V (D18)	1.43V (D292)

;*1.016	0%	0 mA	0V (D0)	1.35V (D277)
				
;*0.9		-0.50 mA	0.79V (D161)  0.88V (D180)
;*0.85		-0.76 mA	1.2V (D245)	0.66V (D135)
;*0.8 		-1.11 mA	1.75V (D358)	0.44V (D90)
;*0.7		-1.82 mA	2.87V (D587)	0V (D0)			
				


;*denotes tabled values Intermediate values use interpolation using the O2 concentration 
;against Ip graph and the lambda calculation ((XO2/3) +1)/1-4.76XO2.

;Span is Ip x 61.9 ohm x gain (560k/22k)

