; ***************************************************************************
; file: 			remote.asm
; project file:		remote.mcp
; processor:		PIC12F617
; clock frequ:		8 MHz
; clock type:		internal RC
; Last revision: 	5 July, 2012
; Author:			Herman Nacinovich
; ****************************************************************
	; ----------------------------------------------------------------
	list	p=pic12f617			; selects chip type
	; ----------------------------------------------------------------
	#include <p12f617.inc>		; includes definitions
	#include <registers.inc>
	#include <constants.inc>
	; ------------------------------------------------------------------------------------------------------------------	#include <macros.inc>
	; ----------------------------------------------------------------
	radix	hex				; selects hexadecimal as default radix
	; ----------------------------------------------------------------
	; set configuration options
	; ------------------------------------------------------------------------------------------------------------------
	__CONFIG     _CP_OFF  &  _PWRTE_ON & _WDT_ON & _IOSCFS_8MHZ & _MCLRE_OFF & _INTRC_OSC_NOCLKOUT & _BOR_ON & _WRT_HALF
	; ----------------------------------------------------------------
	org		0x00
   	goto	start	
	org		0x04
; ---------------------------------------------------------------------------------
interrupt_handler
	; ------------------------------------------------
	; Save critical registers for interrupt routines
	; ------------------------------------------------
	movwf	_W				; save W
	swapf	_W,F			; 
	swapf	STATUS,W		;
	movwf	_status			; save STATUS
	movfw	PCLATH
	movwf	_pclath
	banksel	0
	movlw	tmr0_offset
	movwf	TMR0
	call	remote_handler
	bcf		INTCON,GPIF
	movfw	_pclath
	movwf	PCLATH
	swapf	_status,W		; 
	movwf	STATUS			; restore STATUS
	swapf	_W,W			; restore W
	retfie
; ------------------------------------------------
; Initialise registers
; ------------------------------------------------
start	
	clrf	GPIO
	banksel	TRISIO
	movlw	TRIS_VAL
	movwf	TRISIO
	clrf	PMCON1
	clrf	ANSEL			; digital I/O
	movlw	OPTION_VAL
	movwf	OPTION_REG
	movlw	PR2_VAL
	movwf	PR2
	bsf		PIE1,TMR1IE		; enable TMR1 interrupts (subject to INTCON,PEIE = 1)
	movlw	IOC_VAL
	movwf	IOC
	banksel	0
	movlw	T2CON_VAL
	movwf	T2CON
	movlw	T1CON_VAL
	movwf	T1CON
	movlw	T2CON_VAL
	movwf	T2CON
	clrf	TMR1H
	clrf	TMR1L
	bcf		PIR1,TMR1IF	
	bsf		INTCON,GPIE
	clrf	statecntr
	clrf	bitcntr
	clrf	flags1
	clrf	flags2
	clrf	tmrcon	
	bsf		led				; led on
	call	wait_250		; wait 0.25s
	bcf		led				; led off
	movlw	low code1lo
	variable loopcntr=X
	movwf	FSR
	banksel	PMADRH
	movlw	high	PMADR_CODE1
	movwf	PMADRH				; start address in flash, high byte
	movlw	low		PMADR_CODE1
	movwf	PMADRL				; start address in flash, low byte	
	movlw	4
	movwf	loopcntr
	bsf		flag_nocode			; default
	; -----------------------------------------------------------
	; load code from flash memory
	; -----------------------------------------------------------
loop=$
	clrwdt
	call	read_flash			; H = high byte, WREG = low byte
	banksel	0
	movwf	INDF				; copy low byte to RAM
	xorlw	0ffh				; blank code?
	skpz
	bcf		flag_nocode			; no
	incf	FSR,F				; point to high byt in RAM
	movfw	H					; high byte (flash)
	movwf	INDF				; copy to RAM	
	incf	FSR,F				; point to next
	decfsz	loopcntr,F			; all words read from flash memory?
	goto	loop				; no
	; -----------------------------------------------------------
	movlw	CCP1CON_VAL
	movwf	CCP1CON
	clrf	tableptr
	clrf	CCPR1L
	btfsc	flag_nocode			; code in flash memory blank?	
	goto	main				; yes
	; -----------------------------------------------------------
	movlw	0fh	
	movwf	tableptr
	movlw	0ffh
	movwf	CCPR1L
	bsf		pwr				
	goto	main
; ----------------------------------------------------------------
main
	clrwdt
	bsf		INTCON,GIE	
	call	timer
	call	keyhandler
	call	codehandler	
	goto 	main
; =======================================================================
; Functions
; -------------------------------------------------------------------
wait_250	
	preset_tmr3	d'250'
loop=$
	call	timer
	btfsc	tmr3e		; loop while tmr3e = 1
	goto	loop
	return
; ------------------------------------------------------------------------
; timer
;		This function utilises a master timer and 2 slave timers.
;
;		Master timer MTMR is a 24-bit based on TMR2 and is updated every 1.000 milliseconds.
;
; 		Slave timer tmr3 is a 16-bit timer and is used for timing periods to
;		65,535 ms
;
;		Slave timer tmr4 is a 24-bit timer and is used for timeing periods to
;		4.66 hours
;
;		Associated with each slave timer is control register tmrcon, whose bits
;		are defined as follows:
;
;				tmr3e 	: 1 = timer 3 enabled (cleared automatically on timeout)
;				tmr3f 	: 1 = timer 3 timeout (cleared on timer initialisation) 
;				tmr4e 	: 1 = timer 4 enabled (cleared automatically on timeout)
;				tmr4f 	: 1 = timer 4 timeout (cleared on timer initialisation) 
;
;		To initialise timer3, add timer offset to current value of MTMRH:MTMRL
;		and save result in tmr3hi:tmr3lo. Clear tmr3f and set tmr3e
;

;		To initialise timer4, add timer offset to current value of MTMRU:MTMRH:MTMRL
;		and save result in tmr4hi:tmr4lo. Clear tmr4f and set tmr4e
; -----------------------------------------------------------------
timer
	banksel	0
	clrwdt
	btfss	PIR1,TMR2IF		; TMR2 timeout?
	return					; no
	; ------------------------------------------------------------
	bcf		PIR1,TMR2IF
	; ------------------------------------------------------------
	incfsz	MTMRL,F
	goto	$+4
	; ------------------------------------------------------------
	incfsz	MTMRH,F
	goto	$+2
	; ------------------------------------------------------------
	incf	MTMRU,F
	; ------------------------------------------------------------
chk_tmr3
	btfss	tmr3e			; tmr3 enabled?
	goto	chk_tmr4		; no
	; ------------------------------------------------------------
	movfw	MTMRL
	xorwf	tmr3lo,W		; low bytes match?
	skpz
	return
	; ------------------------------------------------------------
	movfw	MTMRH
	xorwf	tmr3hi,W		; high bytes match?
	skpz
	return
	; ------------------------------------------------------------
	bsf		tmr3f
	bcf		tmr3e
chk_tmr4
	; ------------------------------------------------------------
	movfw	MTMRL
	xorwf	tmr4lo,W		; low bytes match?
	skpz
	return
	; ------------------------------------------------------------
	movfw	MTMRH
	xorwf	tmr4hi,W		; high bytes match?
	skpz
	return
	; ------------------------------------------------------------
	movfw	MTMRU
	xorwf	tmr4up,W		; upper bytes match?
	skpz
	return
	; ------------------------------------------------------------
	bsf		tmr4f
	bcf		tmr4e
	return
; ----------------------------------------------------------------
keyhandler
	banksel	GPIO
	btfsc	pb1			; key pressed?
	return				; no
	; -------------------------------------------------------------
	btfsc	tmr3e		; timer enabled?
	return				; yes
	; -------------------------------------------------------------
	bsf		flag_pb1	
	return
; ------------------------------------------------------------------------
; codehandler
; ------------------------------------------------------------------------
codehandler
	banksel	0
	btfsc	flag_nocode		; code stored in flash?
	goto	begin_learn		; yes
	btfsc	flag_learn		; currently in learn mode?
	goto	learn			; yes
	; ------------------------------------------------------------------------
	btfsc	flag_pb1		; learn button pressed?
	goto	begin_learn		; yes
	; ------------------------------------------------------------------------
	btfss	flag_valid		; code received?
	return					; no
	; ------------------------------------------------------------------------
	bcf	flag_valid			; flag no longer needed
	; ------------------------------------------------------------------------
normal_mode
	; ------------------------------------------------------------------------
	; check if input received matches code1
	; ------------------------------------------------------------------------
	setc					; default (assume codes match)
	movfw	rxdatlo			; input low byte
	xorwf	code1lo,W		; code1 low byte
	skpz					; low bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	movfw	rxdathi			; input high byte
	xorwf	code1hi,W		; code1 high byte
	skpz					; high bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	skpnc					; bytes match?
	goto	inc_pwm			; yes
	; ------------------------------------------------------------
	; check if input matches code2
	; ------------------------------------------------------------------------
	setc					; default (assume codes match)
	movfw	rxdatlo			; input low byte
	xorwf	code2lo,W		; code2 low byte
	skpz					; low bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	movfw	rxdathi			; input high byte
	xorwf	code2hi,W		; code2 high byte
	skpz					; high bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	skpnc					; bytes match?
	goto	dec_pwm			; yes	
	; ------------------------------------------------------------
	; check if input matches code3
	; ------------------------------------------------------------------------
	setc					; default (assume codes match)
	movfw	rxdatlo			; input low byte
	xorwf	code3lo,W		; code3 low byte
	skpz					; low bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	movfw	rxdathi			; input high byte
	xorwf	code3hi,W		; code3 high byte
	skpz					; high bytes match?	
	clrc					; no
	; ------------------------------------------------------------------------
	skpnc					; bytes match?
	goto	toggle_pwr		; yes
	; ------------------------------------------------------------------------
	return
	; ------------------------------------------------------------------------
begin_learn
	; ------------------------------------------------------------------------
	banksel	0
	bcf		flag_nocode		; flag  no longer needed
	bcf		flag_pb1
	btfsc	flag_learn		; learn mode already?
	return					; Yes: then ignore button press
	; ------------------------------------------------------------------------
	preset_tmr3	DBNCE_TIME
	bsf		led
	bsf		codesel1		; select code1
	bcf		codesel2
	bcf		codesel3
	bsf		flag_learn
	bcf		pwr				; turn off main power
	return
	; ------------------------------------------------------------------------
learn
	btfss	flag_valid		; valid code received?
	return					; no
	bcf		flag_valid
	; ------------------------------------------------------------------------
	btfsc	codesel3
	goto	update_code3
	; ------------------------------------------------------------------------
	btfsc	codesel2
	goto	update_code2
	; ------------------------------------------------------------------------
	btfsc	codesel1
	goto	update_code1
	; ------------------------------------------------------------------------
	return
	; ------------------------------------------------------------
update_code1
	movfw	rxdatlo			
	movwf	code1lo			; update code1 low byte
	movfw	rxdathi
	movwf	code1hi			; update code1 high byte
	; ------------------------------------------------------------
	bcf		codesel1
	bsf		codesel2
	bcf		led
	call	wait_250
	bsf		led
	bcf		flag_valid
	return
	; ------------------------------------------------------------
update_code2
	; ------------------------------------------------------------
	; check if input matches code1, low byte
	; ------------------------------------------------------------
	movfw	rxdatlo			; input (code)
	xorwf	code1lo,W		; low bytes match?
	skpz
	goto	$+5				; no
	; ------------------------------------------------------------
	; check if input matches code1, high byte
	; ------------------------------------------------------------
	movfw	rxdathi			; input (code)
	xorwf	code1hi,W		; high bytes match?
	skpnz
	return					; yes
	; ------------------------------------------------------------
	; update code2
	; ------------------------------------------------------------
	movfw	rxdatlo			
	movwf	code2lo			; update code2 low byte
	movfw	rxdathi
	movwf	code2hi			; update code2 high byte
	; ------------------------------------------------------------
	bcf		codesel2
	bsf		codesel3		; select code3
	bcf		led
	call	wait_250
	bsf		led
	bcf		flag_valid
	return
	; ------------------------------------------------------------
update_code3
	; ------------------------------------------------------------
	; check if input matches code1, low byte
	; ------------------------------------------------------------
	movfw	rxdatlo			; input (code)
	xorwf	code1lo,W		; low bytes match?
	skpz
	goto	$+5				; no
	; ------------------------------------------------------------
	; check if input matches code1, high byte
	; ------------------------------------------------------------
	movfw	rxdathi			; input (code)
	xorwf	code1hi,W		; high bytes match?
	skpnz
	return					; yes
	; ------------------------------------------------------------
	; check if input matches code2, low byte
	; ------------------------------------------------------------
	movfw	rxdatlo			; input (code)
	xorwf	code2lo,W		; low bytes match?
	skpz
	goto	$+5				; no
	; ------------------------------------------------------------
	; check if input matches code2, high byte
	; ------------------------------------------------------------
	movfw	rxdathi			; input (code)
	xorwf	code2hi,W		; high bytes match?
	skpnz
	return					; yes
	; ------------------------------------------------------------
	; update code3
	; ------------------------------------------------------------
	movfw	rxdatlo			
	movwf	code3lo			; update code3 low byte
	movfw	rxdathi
	movwf	code3hi			; update code3 high byte
	bcf		codesel3
	; ------------------------------------------------------------
	call	write_code		; update codes in flash memory
	; ------------------------------------------------------------
	banksel	0
	bcf		codesel1		; reset code select pointer
	bcf		codesel2		; reset code select pointer
	bcf		codesel3		; reset code select pointer
	bcf		codesel4		; reset code select pointer
	bcf		flag_learn
	call	wait_250		; wait 0.25s
	bcf		led			
	bcf		flag_valid
	bsf		pwr
	return	
; ---------------------------------------------------------------------------
inc_pwm
	btfsc	pwr				; main power turned on?
	incf	tableptr,F		; yes
	bsf		pwr				; ensure power turned on
	movfw	tableptr
	andlw	0f0h
	skpz					; tableptr > 0fh?
	decf	tableptr,F		; yes: cancel previous <incf> instruction
	movfw	tableptr		
	call	table1
	movwf	CCPR1L
	goto	exit_codehandler
	; ------------------------------------------------------------
dec_pwm
	btfsc	pwr				; main power turned on?
	decf	tableptr,F		; yes
	bsf		pwr				; ensure power turned on
	movlw	0f0h
	andwf	tableptr,W
	skpz					; tableptr > 15?
	clrf	tableptr		; yes
	movfw	tableptr
	call	table1
	movwf	CCPR1L
	; ------------------------------------------------------------
exit_codehandler
	bsf		led
	call	wait_250	
	bcf		led
	bcf		flag_valid
	return
; ------------------------------------------------------------------------
; function:		toggle_pwr
;				This function assumes that mode pin is pulled low
;				by an external resistor. The function toggles the state
;				of the corresponding TRISIO bit regardless of the
;				state of the inhibit pin. 
;	-----------------------------------------------------------------
toggle_pwr
	banksel	0
	; ------------------------------------------------------------------------
	btfsc	pwr				; pwr on?
	goto	toggle_off		; yes
	; ------------------------------------------------------------------------
toggle_on
	bsf		pwr				; turn on main power
	movfw	tableptr		
	call	table1			; get duty cycle from table1
	movwf	CCPR1L			; set PWM duty cycle
	goto	exit_codehandler
	; ------------------------------------------------------------------------
toggle_off
	clrf	CCPR1L			; optional
	bcf		pwr				; turn off main power
	goto	exit_codehandler
; ------------------------------------------------------------------------
; function:		read_flash
; description:	reads 14-bit word in flash memory
; inputs:		PMADRH = upper 5 bits address in flash
;				PMADRH = lower 8 bits address in flash
; outputs:		H = upper 6 bits of word stored in flash 	
;				WREG, L = lower 8 bits of word stored in flash
;				
; NOTE:			registers PMADRH:PMADRL are incremented to point
;				to next address.
; ------------------------------------------------------------------------
read_flash	
	banksel PMCON1 
	bsf 	PMCON1, RD ; flash Read
	nop 				; First instruction after BSF PMCON1,RD executes normally
	nop					; Any instructions here are ignored as program
						; memory is read in second cycle after BSF PMCON1,RD
	movfw 	PMDATH	 	; W = MS Byte of Program PMDATL	
	movwf	H
	movfw 	PMDATL	 	; W = LS Byte of Program PMDATL
	movwf	L
	incf	PMADRL,F
	skpnz
	incf	PMADRH,F
	banksel	0
	return
; ------------------------------------------------------------------------
; function:		write_code
; description:	writes 4 consecutive bytes in codeblock to programme  memory
; inputs:		PMADRH = upper 5 bits address in flash
;				PMADRH = lower 8 bits address in flash
; outputs:		none
; ------------------------------------------------------------------------
write_code
	movlw	 low codeblock
	movwf	 FSR
	; --------------------------------------------------------	
	bankisel codeblock
	banksel PMADRL
	; --------------------------------------------------------	
	movlw 	low PMADR_CODE1 	; Load initial address, low
	movwf 	PMADRL 				;
write_flash
	movlw 	high PMADR_CODE1 	; Load initial address, high
	movwf 	PMADRH 				;
	; --------------------------------------------------------	
	bcf		INTCON,GIE 			; Disable interrupts (if using)
	btfsc	INTCON,GIE 			; ensure interrupts disabled
	goto	$-2
	; --------------------------------------------------------	
	bankisel codeblock
	banksel PMDATL
	clrwdt
	; --------------------------------------------------------	
loop=$	
	movfw	INDF
	movwf	PMDATL				; load low byte of word in RAM
	incf	FSR,F
	movfw	INDF
	movwf	PMDATH				; load upper byte of word in RAM
	incf	FSR,F
	bsf	 	PMCON1,WREN 		; Enable writes
	; --------------------------------------------------------	
	; Required Sequence
	; --------------------------------------------------------	
	movlw 	55h 				; Start of required write sequence:
	movwf 	PMCON2 
	movlw 	0AAh
	movwf 	PMCON2
	bsf 	PMCON1,WR 			; Set WR bit to begin write
	nop 						; Required to transfer data to the buffer
	nop 						; registers
	; --------------------------------------------------------	
	bcf 	PMCON1,WREN 		; Disable writes
	; --------------------------------------------------------		
	movfw 	PMADRL
	incf 	PMADRL,F
	andlw 	0x03 				; Indicates when sixteen words have been programmed
	xorlw	0x03				; 0x00 = 1 word
								; 0x0F = 16 words
								; 0x0B = 12 words
								; 0x07 = 8 words
								; 0x03 = 4 words
	skpz 						; Exit on a match,
	goto 	loop 				; Continue if more data needs to be written
	clrwdt
	; --------------------------------------------------------		
	bsf 	INTCON,GIE 			
	banksel	0
	return
; ------------------------------------------------------------------------
	org		0x280
table1
	variable temp = A
	movwf	temp
	movlw	high $
	movwf	PCLATH
	movfw	temp
	andlw	0fh
	addwf	PCL,F
	retlw	002h
	retlw	004h
	retlw	008h
	retlw	00eh
	retlw	015h
	retlw	01ch
	retlw	024h
	retlw	02dh
	retlw	035h
	retlw	03fh
	retlw	04ch
	retlw	05fh
	retlw	073h
	retlw	091h
	retlw	0bch
	retlw	0ffh
	org		0x300			; make sure next line starts on 256-byte boundary
; ---------------------------------------------------------------------------------
remote_handler
	banksel	0
	btfss	flag_valid			
	goto	$+4
	; ------------------------------------------------------------
	; if flag set then ignore input (do not modify rxdata...)
	; ------------------------------------------------------------
	bcf		INTCON,PEIE		; disable timer1 interrupts
	bcf		INTCON,GPIF		; disable interrupt on change	
	return
	; ------------------------------------------------------------
	btfss	INTCON,PEIE		; peripheral (TMR1) interrupts enabled?
	goto	$+3				; no
	; ------------------------------------------------------------
	btfsc	PIR1,TMR1IF		; TMR1 timeout?	
	goto	validate		; yes
	; ------------------------------------------------------------
	btfss	INTCON,GPIF		; interrupt on portb change?
	return					; no
	; ------------------------------------------------------------
	movlw	high $
	movwf	PCLATH
	; ------------------------------------------------------------
	movlw	07h				; 
	andwf	statecntr,W		; insurance against invalid statecntr
	addwf	PCL,F			; jump to routine specifed by statecntr
	; ------------------------------------------------------------
	goto	begin
	goto	lowplse
 	goto	highplse
	goto	restart
	goto 	restart
	goto 	restart
	goto 	restart
	goto 	restart
	; ----------------------------------------------------------------
restart
	banksel	0
	clrf	bitcntr
	clrf	statecntr
	clrf	lowwdth
	clrf	highwdth
	bcf		INTCON,PEIE		; disable TMR1 interrupts
	bcf		PIR1,TMR1IF		; clear TMR1 interrupt
	return
	; ----------------------------------------------------------------
	; *************************************************************
	; * NOTE: Ensure: Fosc = 8 MHz
	; * 			  TMR1 preset = 1:8
	; *************************************************************
	; *****************************************************************
	; Pulse widths vary between protocols. Usually, a data block 
	; begins with a low then high pulse of widths greater than 1 ms
	; and less than 10 ms; followed by a stream of low/high pulses
	; of duration greater than 200 us and less than 1 ms. The block 
	; typically ends with a high pulse and a low pulse of more than
	; 1 ms and less than 10 ms.
	; 
	; Tosimplify programming, any data pulse of width less than 18 ms
	; is accepted as valid.
	;
	; Data blocks typically consist of about 33-34 pulses. This programme
	; accepts any data block not more than 66 pulses. 
	; Block ends when the input line goes high for more than 65 ms.
	;
	; With some protocols, the first 8 or 16 pulses following the preamble
	; are header pulses of equal high and low duration. For this reason
	; the first 16 pulses do not constitute part of the coding - they
	; are checked to ensure that pulse widths are within specs but that
	; is all. All the rest are assumed to be of nonequal duration but
	; are not checked for this. This might result in a false positive or
	; a false negative on rare occasions but the possibility is thought
	; to be slight and any deleterious effects to be negligible.
	; *****************************************************************

	; *****************************************************************
	; Most remote control protocols allow an interval of 32 ms or 
	; more between transmissions. However Sony Trinitron allows an
	; interval of less than 32 ms and more than 16 ms (about 24 ms)
	; Timing intervals exceeding some preset interval are detected
	; using the TMR1 interrupt flag. With Fosc = 8 MHz and TMR1
	; prescale set to 1:1, TMR1 normally rolls over at 32 ms intervals.
	; Therefore, to detect intervals of 16 ms it is necessary to 
	; preset TMR1 with an offset (TMR1_OFFSET) of 16 ms.
	; *****************************************************************

	; ----------------------------------------------------------------
	; start of data stream (input = 0)
	; ----------------------------------------------------------------
begin
	banksel	0
	btfsc	input	; input valid?
	goto	restart	; no
	; ----------------------------------------------------------------
	incf	statecntr,F
	; ----------------------------------------------------------------
	; iniialise TMR1
	; ----------------------------------------------------------------
	bcf		T1CON,TMR1ON	; turn off TMR1
	movfw	TMR0			
	movwf	TMR1L
	movlw	TMR1_OFFSET	
	movwf	TMR1H 	
	clrf	rxdathi
	clrf	rxdatlo
	clrf	highwdth
	clrf	bitcntr
	bsf		T1CON,TMR1ON	; turn on TMR1
	bcf		PIR1,TMR1IF		; clear interrupt flag
	bsf		INTCON,PEIE		; enable peripheral (TMR1) interrupts
	return
	; ----------------------------------------------------------------
	; Analyze low pulse (input = 1)
	; ----------------------------------------------------------------
lowplse
	variable	variance = A
	banksel	0
	; ----------------------------------------------------------------
	btfss	input			; input valid?
	goto	restart			; no
	; ----------------------------------------------------------------
	; subtract TMR1_OFFSET from TMR1H
	; ----------------------------------------------------------------
	bcf		T1CON,TMR1ON	; turn off TMR1
	movlw	TMR1_OFFSET
	subwf	TMR1H,F
	; ----------------------------------------------------------------
	; check that pulse width < 16 ms
	; ----------------------------------------------------------------
	movlw	080h
	andwf	TMR1H,W			; low pulse > 16.384 ms?
	skpz	
	goto	restart			; yes
	; ----------------------------------------------------------------
	; shift TMR1H:TMR1L 2 places to the left (for 64 us resolution)
	; ----------------------------------------------------------------
	rlf		TMR1L,F
	rlf		TMR1H,F
	rlf		TMR1L,F
	rlf		TMR1H,F			
	; ----------------------------------------------------------------
	; calculate: variance ={highwdth (+) TMR1H}/4 : note (+) = "inclusive OR"
	; ----------------------------------------------------------------
	movfw	TMR1H
	movwf	variance	
	movfw	lowwdth	
	iorwf	variance,F		; variance = highwdth (+) TMR1H
	clrc
	rrf		variance,F
	clrc
	rrf		variance,F		; variance = {highwdth (+) TMR1H}/4
	; ----------------------------------------------------------------
	; calculate: delta = abs(TMR1H - lowwdth)
	; ----------------------------------------------------------------
	movfw	TMR1H
	subwf	lowwdth,W
	skpc					; W = signed (TMR1H - lowwdth)
	sublw	0
	; ----------------------------------------------------------------
	; Is delta > variance?	
	; ----------------------------------------------------------------
	bsf		flag_gp1		; default
	subwf	variance,W		; W = variance - delta	
	skpc
	bcf		flag_gp1		; 0 = "delta > variance, 1 = "delta <= variance)
	; ----------------------------------------------------------------
	; calculate:	highwdth - TMR1H
	; ----------------------------------------------------------------
	movfw	TMR1H
	subwf	lowwdth,W
	btfsc	flag_gp1		; delta <= variance?
	setc					; yes
	; ----------------------------------------------------------------
	; update rxdat2lo, rxdat2hi, lowwdth
	; ----------------------------------------------------------------
	rlf		rxdat2lo,F
	rlf		rxdat2hi,F
	movfw	TMR1H
	movwf	lowwdth
	incf	statecntr,F
	; ----------------------------------------------------------------
	goto	update_tmr1
	; ----------------------------------------------------------------
	; Analyze high pulse (input = 0)
	; ----------------------------------------------------------------
highplse
	banksel	0
	btfsc	input			; input valid?
	goto	restart			; no
	; ----------------------------------------------------------------
	decf	statecntr,F
	; ----------------------------------------------------------------
	bcf		T1CON,TMR1ON	; turn off TMR1
	; ----------------------------------------------------------------
	; subtract TMR1_OFFSET from TMR1H
	; ----------------------------------------------------------------
	bcf		T1CON,TMR1ON	; turn off TMR1
	movlw	TMR1_OFFSET
	subwf	TMR1H,F
	; ----------------------------------------------------------------
	movlw	080h
	andwf	TMR1H,W			; low pulse > 16.384 ms?
	skpz	
	goto	restart			; yes
	; ----------------------------------------------------------------
	; shift TMR1H:TMR1L 2 places to the left (for 64 us resolution)
	; ----------------------------------------------------------------
	rlf		TMR1L,F
	rlf		TMR1H,F
	rlf		TMR1L,F
	rlf		TMR1H,F			
	; ----------------------------------------------------------------
	; calculate: variance ={highwdth (+) TMR1H}/4 : note (+) = "inclusive OR"
	; ----------------------------------------------------------------
	movfw	TMR1H
	movwf	variance	
	movfw	highwdth	
	iorwf	variance,F		; variance = highwdth (+) TMR1H
	clrc
	rrf		variance,F
	clrc
	rrf		variance,F		; variance = {highwdth (+) TMR1H}/4
	; ----------------------------------------------------------------
	; calculate: delta = abs(TMR1H - highwdth)
	; ----------------------------------------------------------------
	movfw	TMR1H
	subwf	highwdth,W
	skpc					; W = signed (TMR1H - highwdth)
	sublw	0
	; ----------------------------------------------------------------
	; Is delta > variance?	
	; ----------------------------------------------------------------
	bsf		flag_gp1		; default
	subwf	variance,W		; W = variance - delta	
	skpc
	bcf		flag_gp1		; 0 = "delta > variance, 1 = "delta <= variance)
	; ----------------------------------------------------------------
	; calculate:	highwdth - TMR1H
	; ----------------------------------------------------------------
	movfw	TMR1H
	subwf	highwdth,W
	btfsc	flag_gp1		; delta <= variance?
	setc					; yes
	; ----------------------------------------------------------------
	; update rxdatlo, rxdathi, highwdth, bitcntr
	; ----------------------------------------------------------------
	rlf		rxdatlo,F
	rlf		rxdathi,F
	movfw	TMR1H
	movwf	highwdth
	incf	bitcntr,F
	; ----------------------------------------------------------------
	; initialise TMR1H:TMR1L
	; ----------------------------------------------------------------
update_tmr1
	movfw	TMR0
	movwf	TMR1L
	movlw	TMR1_OFFSET
	movwf	TMR1H
	bsf		T1CON,TMR1ON	; turn on TMR1
	bcf		INTCON,GPIF
	return
	; ----------------------------------------------------------------
	; Validate data stream
	; ----------------------------------------------------------------
validate
	movlw	MAXBITS
	subwf	bitcntr,W		; #bits > MAXBITS?
	skpnc
	goto	restart			; yes
	; ----------------------------------------------------------------
	movlw	MINBITS
	subwf	bitcntr,W		; #bits < MINBITS?
	skpc
	goto	restart			; yes
	; ----------------------------------------------------------------
	bcf		INTCON,PEIE		; disable TMR1 interrupts
	bcf		PIR1,TMR1IF		; clear interrupt
	bsf		flag_valid		; flag result
	clrf	statecntr
	; *****************************************************************
	; Till now low pulse widths and high pulse widths were treated
	; as separate and independent. Now the two sets of data are 
	; combined by XORing the values stored in rxdatlo:rxdathi
	; and rxdat2lo:rxdat2hi, respectively. The reason is that some
	; remote control use low pulse widths as data while others use
	; high pulse widths. By XORing these sets of date it becomes
	; possible to discriminate between codes regardless of which
	; protocol is being used.
	; *****************************************************************
	movfw 	rxdat2lo
	addlw	07fh				; add arbitrary value
	xorwf	rxdatlo,F
	movfw	rxdat2hi
	addlw	017h				; add arbitrary value
	xorwf	rxdathi,F
	; ----------------------------------------------------------------
	movlw	03fh
	andwf	rxdathi,F		; discard ms bits 6, 7 since only 14 bits
	return					; will be stored in flash memory
; _____________________________________________________________________________________________
	end





