
/* USB Sound Effects Generator Module source code, Copyright 2012 SILICON CHIP Publications Pty. Ltd.
   Written by Nicholas Vinen, based on the Microchip CDC (serial) USB sample code.
   Target device: PIC18F27J53 */

#include "p18F27J53.h"
#include "spi.h"
#include "config.h"
#include <stdlib.h>

/*
  pin mapping:
    2  RA0/AN0/RP0:   PWM audio output (low bits)
    3  RA1/AN1/RP1:   LM4819 shutdown pin (active high)
    7  RA5/AN4/RP2:   PWM audio output (high bits)
   15  RC4/D-:        USB D-
   16  RC5/D+:        USB D+
   17  RC6/RP17/TX1:  external power detection (active high)
   18  RC7/RP18/SDO1: SDO for serial flash chip(s)
   23  RB2/AN8/RP5:   CS for serial flash chip #2
   24  RB3/AN9/RP6:   CS for serial flash chip #1
   25  RB4/RP7/SCK1:  SCK for serial flash chip(s)
   26  RB5/RP8/SDI1:  SDI for serial flash chip(s)
   27  RB6/RP9/PGD:   PGD/trigger input #1
   28  RB7/RP10/PGC:  PGC/trigger input #2
*/

#define FLASH_CHIP_CMD_GETID         0x9F
#define FLASH_CHIP_CMD_SLEEP         0xb9
#define FLASH_CHIP_CMD_WAKE          0xab
#define FLASH_CHIP_CMD_READ_STATUS   0x05
#define FLASH_CHIP_CMD_WRITE_STATUS  0x01
#define FLASH_CHIP_CMD_WRITE_ENABLE  0x06
#define FLASH_CHIP_CMD_WRITE_DISABLE 0x04
#define FLASH_CHIP_CMD_ERASE_4K      0x20
#define FLASH_CHIP_CMD_ERASE_32K     0x52
#define FLASH_CHIP_CMD_ERASE_64K     0xD8
#define FLASH_CHIP_CMD_BYTE_PROGRAM  0x02
#define FLASH_CHIP_CMD_READ_SLOW     0x03
#define FLASH_CHIP_CMD_READ_FAST     0x0b

#define FLASH_CHIP_RESPONSE_ID0 0x1F
#define FLASH_CHIP_RESPONSE_ID1 0x44
#define FLASH_CHIP_RESPONSE_ID2 0x01

#define FLASH_CHIP_STATUS_BSY (1   )
#define FLASH_CHIP_STATUS_WEL (1<<1)
#define FLASH_CHIP_STATUS_SWP (3<<2)
#define FLASH_CHIP_STATUS_WPP (1<<4)
#define FLASH_CHIP_STATUS_EPE (1<<5)
#define FLASH_CHIP_STATUS_SPM (1<<6)
#define FLASH_CHIP_STATUS_SPRL (1<<7)

typedef union {
  unsigned short s;
  unsigned char c[2];
} cs;

typedef union {
  unsigned short long l;
  unsigned char c[3];
} csl;

void spi_init() {
  ANCON1bits.PCFG8 = 1;
  TRISBbits.TRISB2 = 0;
  LATBbits.LATB2 = 1;
  ANCON1bits.PCFG9 = 1;
  TRISBbits.TRISB3 = 0;
  LATBbits.LATB3 = 1;

  TRISCbits.TRISC7 = 0;
  TRISBbits.TRISB4 = 0;
  LATBbits.LATB4 = 0;
  TRISBbits.TRISB5 = 1;
  RPOR7 = 11;  //  RB4/RP7/SCK1 = SCK2 (output)
  RPOR18 = 10; // RC7/RP18/SDO1 = SDO2 (output)
  RPINR21 = 8; //  RB5/RP8/SDI1 = SDI2 (input)
  RPINR22 = 7; //  RB4/RP7/SCK1 = SCK2 (input)

  SSP2STATbits.SMP = 0;
  SSP2STATbits.CKE = 0;
//SSP2CON1bits.SSPM = 1; // SPI master mode, clock = Fosc/16
//SSP2CON1bits.CKP = 1;
//SSP2CON1bits.SSPEN = 1;
  SSP2CON1 = 0x31;
}

unsigned char spi_send_recv_byte(unsigned char byte) {
  PIR3bits.SSP2IF = 0;
  SSP2BUF = byte;
  while( !PIR3bits.SSP2IF )
    ;
  return SSP2BUF;
}

static void spi_send_addr(csl paddr) {
  spi_send_recv_byte(paddr.c[2]);
  spi_send_recv_byte(paddr.c[1]);
  spi_send_recv_byte(paddr.c[0]);
}

static void spi_assert_cs(unsigned char index, unsigned char state) {
  if( index == 0 )
    LATBbits.LATB3 = state;
  else
    LATBbits.LATB2 = state;
}

unsigned char spi_get_flash_chip_presence(unsigned char index) {
  unsigned char ret;

  spi_assert_cs(index, 0);

  spi_send_recv_byte(FLASH_CHIP_CMD_GETID);
  ret = spi_send_recv_byte(0) == FLASH_CHIP_RESPONSE_ID0 && spi_send_recv_byte(0) == FLASH_CHIP_RESPONSE_ID1 && spi_send_recv_byte(0) == FLASH_CHIP_RESPONSE_ID2;

  spi_assert_cs(index, 1);

  return ret;
}

unsigned char spi_get_flash_chips_presence() {
  spi_deep_sleep(0);
  return spi_get_flash_chip_presence(0) | (spi_get_flash_chip_presence(1)<<1);
}

void spi_deep_sleep(unsigned char enabled) {
  spi_assert_cs(0, 0);
  spi_assert_cs(1, 0);
  spi_send_recv_byte(enabled ? FLASH_CHIP_CMD_SLEEP : FLASH_CHIP_CMD_WAKE);
  spi_assert_cs(0, 1);
  spi_assert_cs(1, 1);
}

static void spi_send_command(unsigned char index, unsigned char cmd) {
  spi_assert_cs(index, 0);
  spi_send_recv_byte(cmd);
  spi_assert_cs(index, 1);
}

unsigned char spi_get_status(unsigned char index) {
  unsigned char status;

  spi_assert_cs(index, 0);
  spi_send_recv_byte(FLASH_CHIP_CMD_READ_STATUS);
  status = spi_send_recv_byte(0);
  spi_assert_cs(index, 1);

  return status;
}

void spi_set_status(unsigned char index, unsigned char status) {
  spi_send_command(index, FLASH_CHIP_CMD_WRITE_ENABLE);
  spi_assert_cs(index, 0);
  spi_send_recv_byte(FLASH_CHIP_CMD_WRITE_STATUS);
  spi_send_recv_byte(status);
  spi_assert_cs(index, 1);
}

void spi_protect_all(unsigned char index) {
  spi_set_status(index, 0xFF);
}

void spi_unprotect_all(unsigned char index) {
  spi_set_status(index, 0x00);
  spi_set_status(index, 0x00);
}

static void spi_wait_busy(unsigned char index) {
  while( spi_get_status(index) & FLASH_CHIP_STATUS_BSY )
    ;
}

unsigned char spi_erase_flash_chip_4k(unsigned char index, unsigned short long addr) {
  csl laddr = { addr };
  unsigned char status;

  spi_wait_busy(index);
  spi_send_command(index, FLASH_CHIP_CMD_WRITE_ENABLE);
  if( !(spi_get_status(index) & FLASH_CHIP_STATUS_WEL) )
    return 0;

  spi_assert_cs(index, 0);
  spi_send_recv_byte(FLASH_CHIP_CMD_ERASE_4K);
  spi_send_addr(laddr);
  spi_assert_cs(index, 1);
  spi_wait_busy(index);

  status = spi_get_status(index);
  return !(status & FLASH_CHIP_STATUS_EPE);
}

unsigned char spi_write_flash_chip_64b(unsigned char index, unsigned short long addr, unsigned char num_64b_blocks, const void* data) {
  csl laddr = { addr };
  unsigned char status, i;
  const unsigned char* ptr = (const unsigned char*)data;

  spi_wait_busy(index);

  do {
    spi_send_command(index, FLASH_CHIP_CMD_WRITE_ENABLE);
    if( !(spi_get_status(index) & FLASH_CHIP_STATUS_WEL) )
      return 0;
    spi_assert_cs(index, 0);
    spi_send_recv_byte(FLASH_CHIP_CMD_BYTE_PROGRAM);
    spi_send_addr(laddr);
    do {
      for( i = 0; i < 64; ++i ) {
        spi_send_recv_byte(*ptr++);
        ++laddr.c[0];
      }
      --num_64b_blocks;
    } while( num_64b_blocks && laddr.c[0] );
    if( laddr.c[0] == 0 )
      if( ++laddr.c[1] == 0 )
        ++laddr.c[2];
    spi_assert_cs(index, 1);
    spi_wait_busy(index);
  } while( num_64b_blocks );

  status = spi_get_status(index);
  return !(status & FLASH_CHIP_STATUS_EPE);
}

void spi_initiate_read(unsigned char index, unsigned short long addr) {
  csl laddr = { addr };

  spi_wait_busy(index);
  spi_assert_cs(index, 0);
  spi_send_recv_byte(FLASH_CHIP_CMD_READ_SLOW);
  spi_send_addr(laddr);
}

void spi_read(unsigned char index, unsigned char* dest, unsigned short long addr, unsigned short length) {
	spi_initiate_read(index, addr);
	while( length ) {
		*dest = spi_send_recv_byte(0);
		++dest;
		--length;
	}
	spi_terminate_read(index);
}

void spi_copy_page(unsigned char index, unsigned short long dest_addr, unsigned short long src_addr, unsigned char max_blocks) {
	unsigned char temp[64];
	while( max_blocks ) {
		spi_read(index, temp, src_addr, 64);
		src_addr += 64;
		spi_write_flash_chip_64b(index, dest_addr, 1, (const void*)temp);
		dest_addr += 64;
		--max_blocks;
	}
}

void spi_setup_dma(void* dest) {
  cs destaddr = { (unsigned short)dest };
  DMACON1 = 0x10;
//DMACON1bits.SSCON1 = 0;
//DMACON1bits.SSCON0 = 0;
//DMACON1bits.RXINC = 1;
//DMACON1bits.DUPLEX1 = 0; // receive only
//DMACON1bits.DUPLEX0 = 0; // receive only
//DMACON1bits.DLYINTEN = 0;
  DMACON2 = 0;
  RXADDRH = destaddr.c[1];
  RXADDRL = destaddr.c[0];
}
/*
void spi_initiate_read_dma(unsigned char index, unsigned short long addr, void* dest) {
  spi_initiate_read(index, addr);
  spi_setup_dma(dest);
}
*/
void spi_terminate_read(unsigned char index) {
  spi_assert_cs(index, 1);
}

unsigned char spi_truncate_flash_chip_page_4k(unsigned char index, unsigned short long addr) {
	// addr should be a multiple of 64.
	// Due to limited RAM, this will:
	// (a) erase the following flash page if the address provided is in the second half of
	//     the page (ie, >2KB in)
	// (b) fail (return zero) if the address provided is in the second half of the page (ie,
	//     >2KB in) and it's the last page on the chip.
	// This assumes that whenever you're truncating, you don't care about the contents of
	// any pages following the one being truncated.

	unsigned short addr_within_page = addr & 4095;
	unsigned char blocks = (addr_within_page+63)/64;
	addr &= ~4095;
	if( addr_within_page >= 2048 ) {
		if( addr >= 512UL * 1024 - 4096 )
			return 0;
		if( !spi_erase_flash_chip_4k(index, addr+4096) )
			return 0;

		spi_copy_page(index, addr+4096, addr, blocks);
		if( !spi_erase_flash_chip_4k(index, addr) )
			return 0;
		spi_copy_page(index, addr, addr+4096, blocks);
		return 1;
	} else if( addr_within_page ) {
		spi_read(index, (unsigned char ram far*)FREE_RAM_ADDR, addr, (unsigned short)blocks*64);
		if( !spi_erase_flash_chip_4k(index, addr) )
			return 0;
		spi_write_flash_chip_64b(index, addr, blocks, (const void ram far*)FREE_RAM_ADDR);
		return 1;
	} else {
		return spi_erase_flash_chip_4k(index, addr);
	}

	return 1;
}
