
/* USB Sound Effects Generator Module source code, Copyright 2012 SILICON CHIP Publications Pty. Ltd.
   Written by Nicholas Vinen, based on the Microchip CDC (serial) USB sample code.
   Target device: PIC18F27J53 */

#include "p18F27J53.h"
#include "audio.h"
#include "spi.h"
#include "config.h"

static void wait_15ms(void) {
  unsigned short samples_to_wait = 2816;
  do {
    PIR1bits.TMR2IF = 0;
    while( !PIR1bits.TMR2IF )
      ;
  } while( samples_to_wait-- );
}

void audio_play(unsigned char index, unsigned char portBmask, unsigned char portBval, unsigned char portBendmask, unsigned char portBendval, unsigned char loop) {
  csl caddr, sample_rate;
  cs phase = { 0 };
  unsigned char speed, _12bit;
  csl samples_left, real_samples_left;
  unsigned short long data_to_end_of_mem;
  unsigned char flash_chips_present;
  unsigned char flashchip;
  audio_info* psound;

  if( index >= 8 )
    return;
  psound = &g_config.sounds[index];
  *((unsigned short*)&sample_rate.c[1]) = *((unsigned short*)&psound->sample_rate.c[0]);
  sample_rate.c[0] = 0;
  speed = sample_rate.sl / PWM_RATE;
  if( speed < 4 )
    speed = 4;

  _12bit = psound->flags&AUDIO_FLAG_8BIT;

  flash_chips_present = spi_get_flash_chips_presence();

  LATAbits.LATA1 = 0;   // enable LM4819 (IC2)
  CCPR1L = 31;          // PWM at mid-scale
  CCPR2L = 31;
  T2CONbits.TMR2ON = 1; // enable PWM outputs
  INTCONbits.GIE = 0;   // no interruptions

play_it_again:
  caddr.sl = psound->start_addr.sl;
  samples_left.sl = psound->length_samples.sl;
  if( caddr.c[2] == 0xFF )
    goto abt;
  flashchip = get_flash_chip(&caddr);
  if( flashchip )
      spi_initiate_read(flashchip-1, caddr.sl);

  real_samples_left.sl = samples_left.sl;

  if( flashchip == 0 ) {
    data_to_end_of_mem = INT_MEM_DATA_ADDR + MAX_INT_MEM_BYTES - caddr.sl;

    TBLPTRL = caddr.c[0];
    TBLPTRH = caddr.c[1];
    TBLPTRU = caddr.c[2];

    if( !_12bit ) {
      unsigned char nextval;
      cs lastval, val;
      css delta;

      if( samples_left.sl > data_to_end_of_mem )
        samples_left.sl = data_to_end_of_mem;
      real_samples_left.sl -= samples_left.sl;

      lastval.s = 0x8000;
      do {
        _asm TBLRDPOSTINC _endasm
        nextval = TABLAT;
#ifdef INTERPOLATION
        delta.s = (signed short)nextval - (signed short)lastval.c[1];

        if( delta.s < 0 ) {
          delta.c[0] = -(signed char)delta.c[0];
          do {
            val.s = lastval.s - (delta.c[0] * phase.c[0], PROD);
            val.s >>= 2;
            CCPR1L = val.c[1];
            val.c[0] >>= 2;
            CCPR2L = val.c[0];

            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
         } while( !phase.c[1] );
        } else {
          do {
            val.s = lastval.s + (delta.c[0] * phase.c[0], PROD);
            val.s >>= 2;
            CCPR1L = val.c[1];
            val.c[0] >>= 2;
            CCPR2L = val.c[0];

            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
        }
        lastval.c[1] = nextval;
#else
        do {
          CCPR1L = (nextval>>2);
          CCPR2L = (nextval&3)<<4;

          while( !PIR1bits.TMR2IF )
            ;
          PIR1bits.TMR2IF = 0;
          phase.s += speed;
        } while( !phase.c[1] );
#endif

        phase.c[1] = 0;
        --samples_left.sl;
      } while( samples_left.sl && !INTCON3bits.INT1IF && (PORTB&portBmask) == portBval );
    } else {
      // 12 bit data
      cs lastval;
      cs nextval;
      css raw_delta;
      cs dest;

      two_thirds(&data_to_end_of_mem);
      if( samples_left.sl > data_to_end_of_mem )
        samples_left.sl = data_to_end_of_mem;
      real_samples_left.sl -= samples_left.sl;

      lastval.s = 2048;
      while(1) {
// packing: 23  22  21  20  19  18  17  16  | 15  14  13  12  11  10  09  08  | 07  06  05  04  03  02  01  00
//          bh1 bh0 bl5 bl4 bl3 bl2 bl1 bl0   bh5 bh4 bh3 bh2 ah5 ah4 ah3 ah2   ah1 ah0 al5 al4 al3 al2 al1 al0
        _asm TBLRDPOSTINC _endasm
        nextval.c[0] = TABLAT;
        _asm TBLRDPOSTINC _endasm
        nextval.c[1] = TABLAT&0x0F;

        raw_delta.s = nextval.s - lastval.s;
        if( raw_delta.s < 0 ) {
          do {
#ifdef INTERPOLATION
            cs neg_delta = { 0 };

            neg_delta.s -= raw_delta.s;

            dest.s = lastval.s - (neg_delta.c[1] * phase.c[0], PROD) - (neg_delta.c[0] * phase.c[0], PRODH);
#else
            dest.s = lastval.s;
#endif

            CCPR2L = dest.c[0]&63;
            dest.s <<= 2;
            CCPR1L = dest.c[1]&63;
            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
        } else {
          do {
#ifdef INTERPOLATION
            dest.s = lastval.s + (raw_delta.c[1] * phase.c[0], PROD) + (raw_delta.c[0] * phase.c[0], PRODH);
#else
            dest.s = lastval.s;
#endif

            CCPR2L = dest.c[0]&63;
            dest.s <<= 2;
            CCPR1L = dest.c[1]&63;
            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
        }
        phase.c[1] = 0;
        lastval.s = nextval.s;

		if( !samples_left.c[0]-- ) {
			if( !*((unsigned short*)(samples_left.c+1)) )
				break;
			--*((unsigned short*)(samples_left.c+1));
		} else if( (PORTB&portBmask) != portBval ) {
          break;
		}

        nextval.c[1] = TABLAT>>4;
        _asm TBLRDPOSTINC _endasm
        nextval.c[0] = TABLAT;

        raw_delta.s = nextval.s - lastval.s;
        if( raw_delta.s < 0 ) {
          do {
#ifdef INTERPOLATION
            cs neg_delta = { 0 };

            neg_delta.s -= raw_delta.s;

            dest.s = lastval.s - (neg_delta.c[1] * phase.c[0], PROD) - (neg_delta.c[0] * phase.c[0], PRODH);
#else
            dest.s = lastval.s;
#endif

            CCPR2L = dest.c[0]&63;
            dest.s <<= 2;
            CCPR1L = dest.c[1]&63;
            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
        } else {
          do {
#ifdef INTERPOLATION
            dest.s = lastval.s + (raw_delta.c[1] * phase.c[0], PROD) + (raw_delta.c[0] * phase.c[0], PRODH);
#else
            dest.s = lastval.s;
#endif

            CCPR2L = dest.c[0]&63;
            dest.s <<= 2;
            CCPR1L = dest.c[1]&63;
            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
        }
        phase.c[1] = 0;
        lastval.s = nextval.s;

		if( !samples_left.c[0]-- ) {
			if( !*((unsigned short*)(samples_left.c+1)) )
				break;
			--*((unsigned short*)(samples_left.c+1));
		} else if( INTCON3bits.INT1IF ) {
          break;
		}
      }
    }
    if( real_samples_left.sl && flash_chips_present ) {
      flashchip = (flash_chips_present&1) ? 1 : 2;
      caddr.sl = 0;
      samples_left.sl = real_samples_left.sl;

      spi_initiate_read(flashchip-1, 0/*addr*/);
    }
  }

  if( flashchip ) {
    while( !INTCON3bits.INT1IF && (PORTB&portBmask) == portBval ) {
      data_to_end_of_mem = MAX_EXT_MEM_BYTES - caddr.sl;

      if( !_12bit ) {
        unsigned char nextval;
        cs lastval, val;
        css delta;

        if( samples_left.sl > data_to_end_of_mem )
          samples_left.sl = data_to_end_of_mem;
        real_samples_left.sl -= samples_left.sl;

        PIR3bits.SSP2IF = 0;
        SSP2BUF = 0;

        lastval.s = 0x8000;
        do {
          while( !PIR3bits.SSP2IF )
            ;
          nextval = SSP2BUF;
          PIR3bits.SSP2IF = 0;
          SSP2BUF = 0;
#ifdef INTERPOLATION
          delta.s = (signed short)nextval - (signed short)lastval.c[1];

          if( delta.s < 0 ) {
            delta.c[0] = -(signed char)delta.c[0];
            do {
              val.s = lastval.s - (delta.c[0] * phase.c[0], PROD);
              val.s >>= 2;
              CCPR1L = val.c[1];
              val.c[0] >>= 2;
              CCPR2L = val.c[0];

              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          } else {
            do {
              val.s = lastval.s + (delta.c[0] * phase.c[0], PROD);
              val.s >>= 2;
              CCPR1L = val.c[1];
              val.c[0] >>= 2;
              CCPR2L = val.c[0];

              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          }
          lastval.c[1] = nextval;
#else
          do {
            CCPR1L = (nextval>>2);
            CCPR2L = (nextval&3)<<4;

            while( !PIR1bits.TMR2IF )
              ;
            PIR1bits.TMR2IF = 0;
            phase.s += speed;
          } while( !phase.c[1] );
#endif

          phase.c[1] = 0;
          --samples_left.sl;
        } while( samples_left.sl && !INTCON3bits.INT1IF && (PORTB&portBmask) == portBval );
      } else {
        // 12 bit data
        cs lastval;
        cs nextval;
        css raw_delta;
        cs dest;
        unsigned char buf[3];
        cs bufaddr;

        two_thirds(&data_to_end_of_mem);
        if( samples_left.sl > data_to_end_of_mem )
          samples_left.sl = data_to_end_of_mem;
        real_samples_left.sl -= samples_left.sl;

        spi_setup_dma(buf);
        bufaddr.s = (unsigned short)buf;

        DMABCH = 0;
        DMABCL = 2;
        RXADDRH = bufaddr.c[1];
        RXADDRL = bufaddr.c[0];
        DMACON1bits.DMAEN = 1;
        while( !PIR3bits.SSP2IF )
          ;
        PIR3bits.SSP2IF = 0;

        lastval.s = 2048;
        while(1) {
// packing: 23  22  21  20  19  18  17  16  | 15  14  13  12  11  10  09  08  | 07  06  05  04  03  02  01  00
//          bh1 bh0 bl5 bl4 bl3 bl2 bl1 bl0   bh5 bh4 bh3 bh2 ah5 ah4 ah3 ah2   ah1 ah0 al5 al4 al3 al2 al1 al0
          nextval.c[0] = buf[0];
          nextval.c[1] = buf[1]&0x0F;

          raw_delta.s = nextval.s - lastval.s;
          if( raw_delta.s < 0 ) {
            do {
#ifdef INTERPOLATION
              cs neg_delta = { 0 };

              neg_delta.s -= raw_delta.s;

              dest.s = lastval.s - (neg_delta.c[1] * phase.c[0], PROD) - (neg_delta.c[0] * phase.c[0], PRODH);
#else
              dest.s = lastval.s;
#endif

              CCPR2L = dest.c[0]&63;
              dest.s <<= 2;
              CCPR1L = dest.c[1]&63;
              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          } else {
            do {
#ifdef INTERPOLATION
              dest.s = lastval.s + (raw_delta.c[1] * phase.c[0], PROD) + (raw_delta.c[0] * phase.c[0], PRODH);
#else
              dest.s = lastval.s;
#endif

              CCPR2L = dest.c[0]&63;
              dest.s <<= 2;
              CCPR1L = dest.c[1]&63;
              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          }
          phase.c[1] = 0;
          lastval.s = nextval.s;

			if( !samples_left.c[0]-- ) {
				if( !*((unsigned short*)(samples_left.c+1)) )
					break;
				--*((unsigned short*)(samples_left.c+1));
			} else if( (PORTB&portBmask) != portBval ) {
	          break;
			}

          nextval.c[1] = buf[1]>>4;
          nextval.c[0] = buf[2];

          RXADDRH = bufaddr.c[1];
          RXADDRL = bufaddr.c[0];
          DMABCL = 2;
          DMACON1bits.DMAEN = 1;

          raw_delta.s = nextval.s - lastval.s;
          if( raw_delta.s < 0 ) {
            do {
#ifdef INTERPOLATION
              cs neg_delta = { 0 };

              neg_delta.s -= raw_delta.s;

              dest.s = lastval.s - (neg_delta.c[1] * phase.c[0], PROD) - (neg_delta.c[0] * phase.c[0], PRODH);
#else
              dest.s = lastval.s;
#endif

              CCPR2L = dest.c[0]&63;
              dest.s <<= 2;
              CCPR1L = dest.c[1]&63;
              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          } else {
            do {
#ifdef INTERPOLATION
              dest.s = lastval.s + (raw_delta.c[1] * phase.c[0], PROD) + (raw_delta.c[0] * phase.c[0], PRODH);
#else
              dest.s = lastval.s;
#endif

              CCPR2L = dest.c[0]&63;
              dest.s <<= 2;
              CCPR1L = dest.c[1]&63;
              while( !PIR1bits.TMR2IF )
                ;
              PIR1bits.TMR2IF = 0;
              phase.s += speed;
            } while( !phase.c[1] );
          }
          phase.c[1] = 0;
          lastval.s = nextval.s;

			if( !samples_left.c[0]-- ) {
				if( !*((unsigned short*)(samples_left.c+1)) )
					break;
				--*((unsigned short*)(samples_left.c+1));
			} else if( INTCON3bits.INT1IF ) {
	          break;
			}
        }
      }

      spi_terminate_read(flashchip-1);
      if( real_samples_left.sl && flashchip == 1 && (flash_chips_present&2) && !INTCON3bits.INT1IF && (PORTB&portBmask) == portBval ) {
        flashchip = 2;
        caddr.sl = 0;
        samples_left.sl = real_samples_left.sl;
        spi_initiate_read(1/*flashchip-1*/, 0/*caddr*/);
      } else {
        break;
      }
    }
  }

  if( flashchip )
    spi_terminate_read(flashchip-1);

  if( loop && !INTCON3bits.INT1IF && (PORTB&portBendmask) == portBendval )
    goto play_it_again;

abt:
  CCPR1L = 31;          // PWM at mid-scale
  CCPR2L = 31;
  LATAbits.LATA1 = 1;   // turn LM4819 (IC2) off

  // wait 15ms for the amplifier to shut down
  wait_15ms();
  T2CONbits.TMR2ON = 0; // PWMs off
  INTCONbits.GIE = 1;   // interrupts back on

  TBLPTRU = 0; // bad things happen if we don't change this back!

  return;
}
