
/* USB Sound Effects Generator Module source code, Copyright 2012 SILICON CHIP Publications Pty. Ltd.
   Written by Nicholas Vinen, based on the Microchip CDC (serial) USB sample code.
   Target device: PIC18F27J53 */

#include "p18F27J53.h"
#include "flash.h"
#include "config.h"
#include "spi.h"
#include "audio.h"
#include <string.h>

#pragma udata config_mem
config_info g_config;
#pragma udata string_udata

static void FlashWriteSeq(void) {
	unsigned char flag=0;
	EECON1bits.WREN = 1;
	if(INTCONbits.GIE) {
		INTCONbits.GIE = 0;
		flag=1;
	}		  
	EECON2 = 0x55;
	EECON2 = 0xAA;
	EECON1bits.WR = 1;
	EECON1bits.WREN = 0; 
	if(flag)
		INTCONbits.GIE = 1;	
}

void MyWriteBlockFlash(csl startaddr, unsigned char num_blocks, unsigned char *flash_array) {
	unsigned char write_byte=0,flag=0;

	while(num_blocks--) {
		TBLPTRU = startaddr.c[2];
		TBLPTRH = startaddr.c[1];
		TBLPTRL = startaddr.c[0];

		write_byte = FLASH_WRITE_BLOCK;
		while(write_byte--) {
			TABLAT = *(flash_array++);
			_asm  TBLWTPOSTINC 	_endasm
		}

		TBLPTRU = startaddr.c[2];
		TBLPTRH = startaddr.c[1];
		TBLPTRL = startaddr.c[0];
		FlashWriteSeq();

		startaddr.sl += FLASH_WRITE_BLOCK;
	}
	TBLPTRU = 0;
}

void EraseFlashPage(csl startaddr) {
	DWORD_VAL flash_addr;

	TBLPTRU = startaddr.c[2];
	TBLPTRH = startaddr.c[1];
	TBLPTRL = startaddr.c[0];
	EECON1bits.FREE = 1;
	FlashWriteSeq();
	EECON1bits.FREE = 0;
}

void config_load(void) {
	memcpypgm2ram(&g_config, (const rom far void*)INT_MEM_CFG_ADDR, sizeof(g_config));
}

void config_save(void) {
	csl addr = { INT_MEM_CFG_ADDR };

	g_config.saved = 0xff;
	memcpypgm2ram((unsigned char ram far*)FREE_RAM_ADDR, (const rom far void*)INT_MEM_CFG_ADDR, 1024);
    memmove((unsigned char ram far*)FREE_RAM_ADDR, (const void*)&g_config, sizeof(g_config));

	EraseFlashPage(addr);
	MyWriteBlockFlash(addr, g_config.sounds[0].start_addr.c[2] == 0xFF ? sizeof(g_config)/64 : 1024/64, (unsigned char ram far*)FREE_RAM_ADDR);
}

void config_erase_all_sounds(void) {
	memset(g_config.sounds, 0xFF, sizeof(g_config.sounds));
	config_save();
}

// addr must be 64-byte aligned
static unsigned short long TruncateFlashPage(unsigned short long addr) {
	csl page_addr = { addr & ~1023 };
	unsigned short sub_addr = addr & 1023;

	memcpypgm2ram((unsigned char ram far*)FREE_RAM_ADDR, (const rom far void*)page_addr.sl, 1024);
	EraseFlashPage(page_addr);
	if( sub_addr )
		MyWriteBlockFlash(page_addr, sub_addr/64, (unsigned char ram far*)FREE_RAM_ADDR);
}

unsigned char config_erase_last_sound(void) {
	signed char i;

	for( i = 7; i >= 0; --i ) {
		csl start_addr;
		audio_info* psound = &g_config.sounds[i];

		start_addr.sl = psound->start_addr.sl;
		if( start_addr.c[2] != 0xFF ) {
			unsigned char flashchip;

			if( i == 0 ) {
				config_erase_all_sounds();
			} else {
				unsigned char mask;
				psound[-1].finish_addr.c[0] |= 63;
				start_addr.sl = (start_addr.sl+63)/64*64;
				flashchip = get_flash_chip(&start_addr);
				if( flashchip ) {
					if( (start_addr.sl & 4095) ) {
				        spi_unprotect_all(flashchip-1);
						if( !spi_truncate_flash_chip_page_4k(flashchip-1, start_addr.sl) ) {
							psound[-1].finish_addr.sl |= 4095;
						}
				        spi_protect_all(flashchip-1);
					}
				} else {
					if( start_addr.sl & 1023 )
						TruncateFlashPage(start_addr.sl);
				}
				memset(psound, 0xFF, sizeof(g_config.sounds[i]));
				mask = 1<<i;
				g_config.triggers[0].which_sounds_inv |= mask;
				g_config.triggers[1].which_sounds_inv |= mask;
				config_save();
			}
			return i+1;
		}
	}
	return 0;
}
