
/* USB Sound Effects Generator Module source code, Copyright 2012 SILICON CHIP Publications Pty. Ltd.
   Written by Nicholas Vinen, based on the Microchip CDC (serial) USB sample code.
   Target device: PIC18F27J53 */

/*
  pin mapping:
    2  RA0/AN0/RP0:   PWM audio output (low bits)
    3  RA1/AN1/RP1:   LM4819 shutdown pin (active high)
    4  RA2/AN2/VREF-: Sleep output (active high; not documented)
    7  RA5/AN4/RP2:   PWM audio output (high bits)
   15  RC4/D-:        USB D-
   16  RC5/D+:        USB D+
   17  RC6/RP17/TX1:  external power detection (active high)
   18  RC7/SP18/SDO1: SDO for serial flash chip(s)
   23  RB2/AN8/RP5:   CS for serial flash chip #2
   24  RB3/AN9/RP6:   CS for serial flash chip #1
   25  RB4/SCK1/RP7:  SCK for serial flash chip(s)
   26  RB5/SDI1/RP8:  SDI for serial flash chip(s)
   27  RB6/PGD/RP9:   PGD/trigger input #1
   28  RB7/PGC/RP10:  PGC/trigger input #2
*/

#include <string.h>
#include "flash.h"
#include "WAV.h"
#include "audio.h"
#include "spi.h"
#include "config.h"
#include "mysprintf.h"

// Local includes
#include "picConfiguration.h"
#include "hardwareProfile.h"

// Microchip Application Library
#include "USB/usb.h"
#include "USB/usb_function_cdc.h"

#pragma code code_section=0x1FC00

/** V A R I A B L E S ********************************************************/
#if defined(__18CXX)
    #pragma udata
#endif

#pragma udata usb_in_buf
char USB_In_Buffer[256];
#pragma udata usb_out_buf
char USB_Out_Buffer[64];
unsigned char usb_attached;

/** P R I V A T E  P R O T O T Y P E S ***************************************/
static void InitializeSystem(void);
void ProcessIO(void);
void USBDeviceTasks(void);
void YourHighPriorityISRCode();
void YourLowPriorityISRCode();
void UserInit(void);

static void ClearRBIF(void) {
  // reg 0 = W
  _asm MOVF 0xf81, 0, ACCESS _endasm
  _asm NOP _endasm
  _asm BCF 0xff2, 0, ACCESS _endasm
}

//void Timer0Int();

/** VECTOR REMAPPING ***********************************************/
#if defined(__18CXX)
	//On PIC18 devices, addresses 0x00, 0x08, and 0x18 are used for
	//the reset, high priority interrupt, and low priority interrupt
	//vectors.  However, the current Microchip USB bootloader 
	//examples are intended to occupy addresses 0x00-0x7FF or
	//0x00-0xFFF depending on which bootloader is used.  Therefore,
	//the bootloader code remaps these vectors to new locations
	//as indicated below.  This remapping is only necessary if you
	//wish to program the hex file generated from this project with
	//the USB bootloader.  If no bootloader is used, edit the
	//usb_config.h file and comment out the following defines:
	//#define PROGRAMMABLE_WITH_USB_HID_BOOTLOADER
	//#define PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x1000
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x1008
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x1018
	#elif defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x800
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x808
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x818
	#else	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x00
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x08
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x18
	#endif
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	extern void _startup (void);        // See c018i.c in your C18 compiler dir
	#pragma code REMAPPED_RESET_VECTOR = REMAPPED_RESET_VECTOR_ADDRESS
	void _reset (void)
	{
	    _asm goto _startup _endasm
	}
	#endif
	#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS
	void Remapped_High_ISR (void)
	{
	     _asm goto YourHighPriorityISRCode _endasm
	}
	#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS
	void Remapped_Low_ISR (void)
	{
		_asm retfie 0 _endasm
	}
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	//Note: If this project is built while one of the bootloaders has
	//been defined, but then the output hex file is not programmed with
	//the bootloader, addresses 0x08 and 0x18 would end up programmed with 0xFFFF.
	//As a result, if an actual interrupt was enabled and occured, the PC would jump
	//to 0x08 (or 0x18) and would begin executing "0xFFFF" (unprogrammed space).  This
	//executes as nop instructions, but the PC would eventually reach the REMAPPED_RESET_VECTOR_ADDRESS
	//(0x1000 or 0x800, depending upon bootloader), and would execute the "goto _startup".  This
	//would effective reset the application.
	
	//To fix this situation, we should always deliberately place a 
	//"goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS" at address 0x08, and a
	//"goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS" at address 0x18.  When the output
	//hex file of this project is programmed with the bootloader, these sections do not
	//get bootloaded (as they overlap the bootloader space).  If the output hex file is not
	//programmed using the bootloader, then the below goto instructions do get programmed,
	//and the hex file still works like normal.  The below section is only required to fix this
	//scenario.
	#pragma code HIGH_INTERRUPT_VECTOR = 0x08
	void High_ISR (void)
	{
	     _asm goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#pragma code LOW_INTERRUPT_VECTOR = 0x18
	void Low_ISR (void)
	{
	     _asm goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#endif	//end of "#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER)"

	#pragma code
	
	
	//These are your actual interrupt handling routines.
	#pragma interrupt YourHighPriorityISRCode
	void YourHighPriorityISRCode()
	{
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.
        #if defined(USB_INTERRUPT)
		if( usb_attached )
	        USBDeviceTasks();
        #endif
		if( INTCONbits.RBIF && INTCONbits.RBIE )
			ClearRBIF();
		if( INTCON3bits.INT1IE )
			INTCON3bits.INT1IF = 0;
		if( INTCON3bits.INT2IE )
			INTCON3bits.INT2IF = 0;
	}	//This return will be a "retfie fast", since this is in a #pragma interrupt section 

#elif defined(__C30__)
    #if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
        /*
         *	ISR JUMP TABLE
         *
         *	It is necessary to define jump table as a function because C30 will
         *	not store 24-bit wide values in program memory as variables.
         *
         *	This function should be stored at an address where the goto instructions 
         *	line up with the remapped vectors from the bootloader's linker script.
         *  
         *  For more information about how to remap the interrupt vectors,
         *  please refer to AN1157.  An example is provided below for the T2
         *  interrupt with a bootloader ending at address 0x1400
         */
//        void __attribute__ ((address(0x1404))) ISRTable(){
//        
//        	asm("reset"); //reset instruction to prevent runaway code
//        	asm("goto %0"::"i"(&_T2Interrupt));  //T2Interrupt's address
//        }
    #endif
#endif


/** DECLARATIONS ***************************************************/
#if defined(__18CXX)
    #pragma code
#endif

unsigned char round_robin[2] = { 0, 0 };
unsigned char which_sound(unsigned char sounds, unsigned char round_robin, unsigned char* round_robin_last) {
	unsigned char last;
	if( round_robin ) {
		last = *round_robin_last;
		while(1) {
			if( sounds&(1<<last) ) {
				*round_robin_last = (last+1)&7;
				return last;
			}
			last = (last+1)&7;
		}
	} else {
		unsigned char val = TMR0L;
		unsigned char num = 0, temp = sounds;
		do {
			if( temp&1 )
				++num;
			temp >>= 1;
		} while( temp );
		while( val >= num )
			val -= num;
		temp = 1;
		num = 0;
		while(1) {
			if( sounds&temp && val-- == 0 ) {
				return num;
			}
			temp <<= 1;
			++num;
		}
	}
}

static unsigned char trigger_sound(unsigned char triga, unsigned char trigb, unsigned char maskbits) {
	unsigned char pria = g_config.triggers[0].flags&TRIGGER_FLAG_PRIORITY;
	unsigned char prib = g_config.triggers[1].flags&TRIGGER_FLAG_PRIORITY;
	unsigned char val = ((g_config.triggers[0].flags&TRIGGER_FLAG_NC)<<6)|((g_config.triggers[0].flags&TRIGGER_FLAG_NC)<<7);
	unsigned char mask, endmask, sounds, mode, index;
	unsigned short delay, maxdelay;

	if( triga && (!trigb || prib) && (sounds = ~g_config.triggers[0].which_sounds_inv) ) {
		index = which_sound(sounds, g_config.triggers[0].flags&TRIGGER_FLAG_RANDOM, &round_robin[0]);
		endmask = 0x40;
		mask = (prib ? 0 : 0x80);
	} else if( trigb && (sounds = ~g_config.triggers[1].which_sounds_inv) ) {
		index = which_sound(sounds, g_config.triggers[1].flags&TRIGGER_FLAG_RANDOM, &round_robin[1]);
		endmask = 0x80;
		mask = (pria ? 0 : 0x40);
	} else {
		return 0;
	}
	val ^= endmask;
	endmask |= mask;
    if( !(g_config.sounds[index].flags&AUDIO_FLAG_PARTIAL) )
		mask |= endmask;
	mask &= maskbits;
	endmask &= maskbits;
	audio_play(index, mask, val&mask, endmask, val&endmask, !(g_config.sounds[index].flags&AUDIO_FLAG_LOOP));

	// short delay to let contacts settle if sound playback was ended due to switch opening, etc.
	delay = 1024 /* 20ms */, maxdelay = 2560 /* 50ms */;
	do {
		// reset timer if trigger conditions re-occur (ie, contacts bounce)
		if( mask && (PORTB&mask) == val )
			delay = 1024;
		INTCONbits.TMR0IF = 0;
		while( !INTCONbits.TMR0IF )
			;
	} while( --maxdelay &&--delay );

	return 1;
}

unsigned char usb_suspend = 0;

static void DischargeRegulatorInputCap(unsigned char dly) {
	cs delay;

	// When switching from external power to battery, there's a critical regulator input voltage
	// just below its output voltage (ie, the battery voltage) where its current draw increases
	// dramatically and it will remain in this condition until something happens to discharge
	// the input capacitor further. To avoid this, once external power goes away, we use the
	// pin designed to detect the presence of external power, tied to the gate of Q1, to turn
	// Q1 off and thus isolate the battery temporarily. The circuit then runs off the charge
	// left in the regulator input and output caps until either they have discharged
	// sufficiently for the micro to reset (due to brown-out) or until Q1 switches back on due
	// to the drop in its gate voltage and the 50ms time-out occurs.

	// This can also happen after sound playback due to drops in the regulator output supply
	// voltage due to battery internal impedance and so on.

	TRISCbits.TRISC6 = 0;
	LATCbits.LATC6 = 1;
	delay.c[1] = dly;
	delay.c[0] = 0;
	do {
		INTCONbits.TMR0IF = 0;
		while( !INTCONbits.TMR0IF )
			;
	} while( --delay.s );

	LATCbits.LATC6 = 0;
	TRISCbits.TRISC6 = 1;
}

#if defined(__18CXX)
void main(void) {
#else
int main(void) {
#endif
    unsigned char triga, trigb, lasttriga = 0, lasttrigb = 0;

    InitializeSystem();

    while(1) {
  		if( PORTCbits.RC6 && !usb_attached ) {
			usb_suspend = 0;
			USBDeviceAttach();
			usb_attached = 1;
		} else if( !PORTCbits.RC6 ) {
            if( usb_attached ) {
				unsigned short delay;

                USBDeviceDetach();
				usb_attached = 0;
				RCONbits.IPEN = 0;
				INTCONbits.PEIE = 0;
				DischargeRegulatorInputCap(10);
			}
		}

		triga = PORTBbits.RB6^(g_config.triggers[0].flags&TRIGGER_FLAG_NC);
		trigb = PORTBbits.RB7^(g_config.triggers[1].flags&TRIGGER_FLAG_NC);
		if( usb_attached ) {
	        #if defined(USB_POLLING)
    	    USBDeviceTasks();
        	#endif
    				  
			// Application-specific tasks.
        	ProcessIO();
		} else if( (triga || trigb) && (triga != lasttriga || trigb != lasttrigb) ) {
			lasttriga = triga;
			lasttrigb = trigb;

			if( trigger_sound(!PORTBbits.RB6, 0, 0xFF) ) {
				DischargeRegulatorInputCap(1);
	            continue;
			}
		}

		lasttriga = triga;
		lasttrigb = trigb;
		if( !usb_attached || (usb_suspend && PORTCbits.RC6) ) {
            spi_deep_sleep(1);
			RPINR1 = 17; // INT1 = RP17 (non-battery power detection), rising edge
			INTCON2bits.INTEDG1 = 1;
			RPINR2 = 17; // INT2 = RP17 (non-battery power detection), falling edge
			INTCON2bits.INTEDG2 = 0;
			OSCCONbits.IDLEN = 0;
			DSCONHbits.DSEN = 0;
			INTCONbits.GIE = 1;
			INTCONbits.RBIE = 1;
			INTCON3bits.INT1IE = 1;
			INTCON3bits.INT2IE = 1;
			Sleep();
			INTCONbits.RBIE = 0;
//			INTCON3bits.INT1IE = 0;
//			INTCON3bits.INT1IF = 0;
//			INTCON3bits.INT2IE = 0;
//			INTCON3bits.INT2IF = 0;
			INTCON3 = 0xC0;
			ClearRBIF();
		}
    }
}


static void InitializeSystem(void) {
	config_load();

    UserInit();

    USBDeviceInit();

	T0CONbits.T0CS = 0;

	usb_attached = 0;
}


void UserInit(void) {
	TRISAbits.TRISA1 = 0;
	LATAbits.LATA1 = 1;
	TRISBbits.TRISB2 = 0;
	LATBbits.LATB2 = 1;
	TRISBbits.TRISB3 = 0;
	LATBbits.LATB3 = 1;

	INTCON2bits.RBPU = 0;
	audio_init();
	spi_init();
}

#define SOH 0x01
#define EOT 0x04
#define ACK 0x06
#define NAK 0x15
#define CAN 0x18
#define SUB 0x1a

#pragma udata xmodem_buffer
unsigned char XMODEM_buffer[128];
#pragma udata data_buffer
unsigned char data_buffer[128];
#pragma udata wav_header
WAVHeader WAVheader;
#pragma udata string_udata

unsigned char g_wavcpy_extra;
unsigned char g_wavcpy_extrabuf[2];

typedef union {
  unsigned short s;
  unsigned char c[2];
} cs;

void wavcpyinit(void) {
	g_wavcpy_extra = 0;
}
// returns number of bytes written to destination
// packing: 23  22  21  20  19  18  17  16  | 15  14  13  12  11  10  09  08  | 07  06  05  04  03  02  01  00
//          bh1 bh0 bl5 bl4 bl3 bl2 bl1 bl0   bh5 bh4 bh3 bh2 ah5 ah4 ah3 ah2   ah1 ah0 al5 al4 al3 al2 al1 al0
static unsigned char wavcpy(void* dest, const void* dest_end, const void** src, const void* src_end, unsigned char _8bit) {
	if( _8bit ) {
		unsigned char len1 = (const unsigned char*)src_end - (const unsigned char*)*src;
		unsigned char len2 = (const unsigned char*)dest_end - (unsigned char*)dest;
		if( len2 < len1 )
			len1 = len2;
		memmove(dest, *src, len1);
		*src += len1;
		return len1;
	} else {
		const signed short* ssrc = (const signed short*)*src;
		unsigned char* ddest = (unsigned char*)dest;
		cs val;

		while( (g_wavcpy_extra&15) > 2 && ddest != (const unsigned char*)dest_end ) {
			ddest[0] = g_wavcpy_extrabuf[(g_wavcpy_extra - 3)&3];
			++ddest;
			++g_wavcpy_extra;
			if( g_wavcpy_extra == 5 )
				g_wavcpy_extra = 2;
			else if( (g_wavcpy_extra&15) == 8 )
				g_wavcpy_extra = (g_wavcpy_extra&0xF0)|1;
		}

        while( ddest < (const unsigned char*)dest_end && ssrc < (const signed short*)src_end ) {
			val.s = ((ssrc[0] + 32768) >> 4);
			switch(g_wavcpy_extra&3) {
			default:
//			case 0:
				++ssrc;
            	*ddest = val.c[0];
				++ddest;
            	g_wavcpy_extra = (val.c[1]<<4) | 1;
            	break;
          	case 1:
            	*ddest = (g_wavcpy_extra>>4) | (val.c[1]<<4);
				++ddest;
            	g_wavcpy_extra ^= 3;
            	break;
          	case 2:
				++ssrc;
            	*ddest = val.c[0];
				++ddest;
            	g_wavcpy_extra = 0;
            	break;
		  	}
		}
		*src = ssrc;
		return ddest - (const unsigned char*)dest;
	}
}

static unsigned char wavcpy_rollback(const unsigned char* buf_end, unsigned char _8bit) {
	if( !_8bit ) {
		switch(g_wavcpy_extra&3) {
       	case 1:
			g_wavcpy_extra = (g_wavcpy_extra&0xF0) | 7;
			g_wavcpy_extrabuf[0] = buf_end[-1];
           	break;
       	case 2:
			g_wavcpy_extra = 3;
			g_wavcpy_extrabuf[0] = buf_end[-2];
			g_wavcpy_extrabuf[1] = buf_end[-1];
           	break;
	  	}
	}
}

#ifdef __DEBUG
static void PrintHex(char* dest, unsigned char val) {
	unsigned char high = val >> 4;
	if( high > 9 )
		dest[0] = 'A' + high - 10;
	else
		dest[0] = '0' + high;
	val &= 15;
	if( val > 9 )
		dest[1] = 'A' + val - 10;
	else
		dest[1] = '0' + val;
}
#endif//__DEBUG

static void TransferError(const rom char* msg, unsigned char* pmsglen, unsigned char* pXMODEM, unsigned char WAVDestination) {
	USB_In_Buffer[0] = CAN;
	USB_In_Buffer[1] = CAN;
	*pmsglen = mystrcpypgm2ram(USB_In_Buffer+2, (const rom char far*)msg) - USB_In_Buffer;
	*pXMODEM = 0;
	if( WAVDestination != 0 )
		spi_protect_all(WAVDestination-1);
}

static unsigned char mystrnicmppgm2ram(const rom far char* a, const char* ptr, const char* end, unsigned char exact) {
    TBLPTRL = ((csl*)&a)->c[0];
    TBLPTRH = ((csl*)&a)->c[1];

	while(1) {
		unsigned char c, d;
	    _asm TBLRDPOSTINC _endasm
		c = TABLAT;
		if( !c )
			return exact ? ptr == end-1 : 1;
		d = *ptr;
		if( c >= 'a' && c <= 'z' )
			c += 'A' - 'a';
		if( d >= 'a' && d <= 'z' )
			d += 'A' - 'a';
		if( ptr == end || c != d )
			return 0;
		++ptr;
	}
}

unsigned char CmdBuf[64];
int CmdBufPos = 0;

static unsigned char CmdIs(const rom far char* cmd, unsigned char exact) {
	return mystrnicmppgm2ram(cmd, (const char*)CmdBuf, (const char*)CmdBuf + CmdBufPos, exact);
}

static unsigned char mystricmpram2pgm(const char* a, const rom far char* b) {
	return !mystrnicmppgm2ram(b, a, a + strlen(a), 0);
}

static unsigned char AppendUSBInBuffer(unsigned char l, const rom char far* msg) {
	return mysprintf( USB_In_Buffer+l, msg ) - USB_In_Buffer;
}

static unsigned short get_sound_time(unsigned char index) {
	return (g_config.sounds[index].length_samples.sl * 10) / g_config.sounds[index].sample_rate.s;
}

unsigned short bytes_to_tens_of_kb(unsigned short long bytes) {
	return (bytes * 10 + 512) / 1024;
}

static char* PrintSoundInfo(char* dest, unsigned char index) {
	return mysprintf( dest, (const rom far char*)"Sound #%d: %dHz, %d-bit, %fs, %fKB, %sloop, %s\r\n",
                            (unsigned short)(index+1),
                             g_config.sounds[index].sample_rate,
                            (unsigned short)((g_config.sounds[index].flags&AUDIO_FLAG_8BIT) ? 12 : 8),
                             get_sound_time(index),
                            bytes_to_tens_of_kb(g_config.sounds[index].length_bytes.sl),
                            (g_config.sounds[index].flags&AUDIO_FLAG_LOOP) ? "no " : "",
                            (g_config.sounds[index].flags&AUDIO_FLAG_PARTIAL) ? "stop after playback" : "stop immediately" );
}

static char* PrintTriggerInfo(char* dest, unsigned char index) {
	unsigned char s = g_config.triggers[index].which_sounds_inv;
	dest = mysprintf( dest, (const rom far char*)"Trigger #%d: %s, ", (unsigned short)(index+1), (g_config.triggers[index].flags&TRIGGER_FLAG_NC) ? "NO" : "NC" );
	if( s == 0xFF ) {
		dest = mysprintf(dest, (const rom far char*)"no sounds");
	} else {
		unsigned char j, c = 0, t = s, n;
		for( j = 0; j < 8; ++j ) {
			if( !(t&1) ) {
				++c;
				n = j;
			}
			t >>= 1;
		}
		if( c == 1 ) {
			dest = mysprintf(dest, (const rom far char*)"sound #%d", (unsigned short)(n+1));
		} else {
			t = s;
			n = 0;
			for( j = 0; j < 8; ++j ) {
				if( !(t&1) ) {
					dest = mysprintf(dest, (const rom far char*)"%s #%d", (n == 0 ? "sounds" : ","), (unsigned short)(j+1));
					++n;
				}
				t >>= 1;
			}
		}
	}
	if( !(g_config.triggers[index].flags&TRIGGER_FLAG_PRIORITY) )
		dest = mysprintf(dest, (const rom far char*)", priority");
	if( !(g_config.triggers[index].flags&TRIGGER_FLAG_RANDOM) )
		dest = mysprintf(dest, (const rom far char*)", random\r\n");
	else
		dest = mysprintf(dest, (const rom far char*)", round robin\r\n");

	return dest;
}

static unsigned char SendInfoString(unsigned char l, unsigned char* pcurrentstate) {
	unsigned char sound_index;
	unsigned short long sound_addr;
	unsigned char i;

	if( *pcurrentstate == 0 ) {
		spi_deep_sleep(0);

		l = mysprintf( USB_In_Buffer+l, (const rom far char*)"SILICON CHIP Sound Effects Module v1.0\r\n"
		                                "Total memory: %fKB\r\n"
        	                            "Free memory: %fKB\r\n",
	        	                        bytes_to_tens_of_kb(audio_get_sound_memory_total()),
	            	                    bytes_to_tens_of_kb(audio_get_sound_memory_free()) ) - USB_In_Buffer;
	} else if( *pcurrentstate <= 8 ) {
		i = *pcurrentstate - 1;
		if( g_config.sounds[i].start_addr.c[2] != 0xFF ) {
			l = PrintSoundInfo( USB_In_Buffer+l, i ) - USB_In_Buffer;
		} else {
			if( i == 0 )
				l = AppendUSBInBuffer(l, (const rom far char*)"No sounds present.\r\n");
			*pcurrentstate = 8;
		}
	} else {
		for( i = 0; i < 2; ++i ) {
			l = PrintTriggerInfo( USB_In_Buffer+l, i ) - USB_In_Buffer;
		}
		if( g_config.saved != 0xFF )
			l = AppendUSBInBuffer(l, (const rom far char*)"Unsaved configuration changes\r\n");
		*pcurrentstate = -1;
	}
	++*pcurrentstate;
	return l;
}

unsigned char ClearMemory(unsigned char l, unsigned char all) {
	if( g_config.sounds[0].start_addr.c[2] != 0xFF ) {
		if( all ) {
			config_erase_all_sounds();
			l = AppendUSBInBuffer(l, (const rom far char*)"Memory cleared, ");
		} else {
			unsigned char i = config_erase_last_sound();
			l = mysprintf( USB_In_Buffer+l, (const rom far char*)"Sound #%d cleared, ", (unsigned short)i ) - USB_In_Buffer;
		}
	} else {
		l = AppendUSBInBuffer(l, (const rom far char*)"Memory already clear, ");
	}

	return mysprintf( USB_In_Buffer+l, (const rom far char*)"%fKB free\r\n", bytes_to_tens_of_kb(audio_get_sound_memory_free()) ) - USB_In_Buffer;
}

#ifdef __DEBUG
unsigned char strotousl(unsigned short long* tohere, char** ptr, const char* end) {
	unsigned char digits = 0, val;
	char c;

	*tohere = 0;
	while( *ptr < end && digits < 6 ) {
		c = **ptr;
		if( c >= 'a' && c <= 'f' )
			val = c - 'a' + 10;
		else if( c >= 'A' && c <= 'F' )
			val = c - 'A' + 10;
		else if( c >= '0' && c <= '9' )
			val = c - '0';
		else
			break;

		*tohere = (*tohere << 4) + val;
		++digits;
		++*ptr;
	}

	return digits > 0;
}
#endif//__DEBUG

void ProcessIO(void) {   
    BYTE numBytesRead;
	static unsigned char XMODEM = 0, XMODEM_packet, XMODEM_byte, XMODEM_checksum, XMODEM_waiting = 0, XMODEM_Start_Sub_Timer = 0, flash_chips_present, XMODEM_ignore_packet_data;
	static unsigned short XMODEM_Start_Timer = 0;
	static csl FlashAddress, WAVStartAddress;
	static unsigned short long WAVBytesLeft;
	static unsigned char WAVHeaderRead, WAVDataOffset, WAVBitDepth8, WAVDestination;
	static unsigned char current_state = 0;
#ifdef __DEBUG
	static unsigned short long mem_read_start = 0, mem_read_finish = 0;
#endif//__DEBUG

    // User Application USB tasks
    if((USBDeviceState < CONFIGURED_STATE)||(USBSuspendControl==1))
		return;

    if(USBUSARTIsTxTrfReady()) {
		numBytesRead = getsUSBUSART(USB_Out_Buffer,64);
		if(numBytesRead == 0) {
			if( current_state ) {
				unsigned char l = SendInfoString(0, &current_state);
				if( l ) {
					putUSBUSART(USB_In_Buffer, l);
				}
#ifdef __DEBUG
			} else if( mem_read_start != mem_read_finish ) {
				unsigned char buf[16], i, l = 0;

				USB_In_Buffer[l++] = '[';
				PrintHex(USB_In_Buffer+l, mem_read_start>>16);
				l += 2;
				PrintHex(USB_In_Buffer+l, mem_read_start>>8);
				l += 2;
				PrintHex(USB_In_Buffer+l, mem_read_start);
				l += 2;
				USB_In_Buffer[l++] = ']';
				USB_In_Buffer[l++] = ' ';

				if( mem_read_start < 128 * 1024UL ) {
					memcpypgm2ram(buf, (const rom far void*)mem_read_start, 16);
				} else {
					unsigned short long addr = mem_read_start - 128 * 1024UL;
					unsigned char flashchip = 0;
					if( addr > 512 * 1024UL ) {
						addr -= 512 * 1024UL;
						flashchip = 1;
					}
					spi_deep_sleep(0);
					spi_read(flashchip, buf, addr, 16);
				}
				for( i = 0; i < 16; ++i ) {
					PrintHex(USB_In_Buffer+l, buf[i]);
					l += 2;
					USB_In_Buffer[l++] = ' ';
				}
				for( i = 0; i < 16; ++i ) {
					if( buf[i] == '\r' || buf[i] == '\n' || buf[i] == '\t' || buf[i] == 8 || buf[i] == 0xb )
						USB_In_Buffer[l++] = '.';
					else
						USB_In_Buffer[l++] = (char)buf[i];
				}
				USB_In_Buffer[l++] = '\r';
				USB_In_Buffer[l++] = '\n';
				putUSBUSART(USB_In_Buffer, l);
				mem_read_start += 16;
				if( mem_read_start > mem_read_finish )
					mem_read_start = mem_read_finish = 0;
#endif//__DEBUG
			}
		} else {
			BYTE i, c;

			for(i=0;i<numBytesRead;i++) {
				c = USB_Out_Buffer[i];

				if( XMODEM ) {
					XMODEM_Start_Timer = 0;
					XMODEM_Start_Sub_Timer = 0;
					if( XMODEM_byte < 3 ) {
						switch(XMODEM_byte) {
						case 0:
							if( c == SOH ) {
								++XMODEM_byte;
							} else if( c == EOT ) {
								unsigned char msglen, ok = 1;

								// end of transmission
								if( WAVBytesLeft ) {
									TransferError("Incomplete transfer\r\n", &msglen, &XMODEM, WAVDestination);
								} else {
									if( WAVDataOffset ) {
										if( g_wavcpy_extra ) {
											signed short zero = 0;
											signed short* pzero = &zero;
											WAVDataOffset += wavcpy((void*)(data_buffer + WAVDataOffset), (const void*)(data_buffer + sizeof(data_buffer)), (const void**)&pzero, (const void*)(&zero + 1), WAVBitDepth8);
										}
										if( WAVDestination == 0 ) {
											if( !(FlashAddress.sl&1023) )
												EraseFlashPage(FlashAddress);
											MyWriteBlockFlash(FlashAddress, 2, data_buffer);
										} else {
											if( !(FlashAddress.sl&4095) ) {
									            spi_deep_sleep(0);
												if( !spi_erase_flash_chip_4k(WAVDestination-1, FlashAddress.sl) ) {
													TransferError("Flash erase error\r\n", &msglen, &XMODEM, WAVDestination);
													ok = 0;
												}
											}
											if( !spi_write_flash_chip_64b(WAVDestination-1, FlashAddress.sl, 2, (const void*)data_buffer) ) {
												TransferError("Flash write error\r\n", &msglen, &XMODEM, WAVDestination);
												break;
											}
										}
	
										FlashAddress.sl += 128;
									}
									if( ok ) {
										unsigned char index;
										audio_info* info = audio_get_available_slot(&index);
	
										info->start_addr.sl = WAVStartAddress.sl;
										if( FlashAddress.sl == 0 ) {
											if( WAVDestination == 2 ) {
												info->finish_addr.sl = (MAX_EXT_MEM_BYTES - 1) | 0x400000UL;
											} else {
												info->finish_addr.sl = INT_MEM_DATA_ADDR + MAX_INT_MEM_BYTES - 1;
											}
										} else {
											info->finish_addr.sl = FlashAddress.sl-1;
											info->finish_addr.c[2] |= WAVDestination<<6;
										}
										info->length_samples.sl = info->length_bytes.sl = WAVheader.dataHeader.data.chunkSize;
										if( WAVBitDepth8 ) {
											info->flags = ~AUDIO_FLAG_8BIT;
										} else {
											three_quarters(&info->length_bytes.sl);
											info->length_samples.sl >>= 1;
											info->flags = ~AUDIO_FLAG_16BIT;
										}
										info->sample_rate.s = WAVheader.fmtHeader.sampleRate;
										g_config.saved = 0;
										config_save();

										msglen = mysprintf( USB_In_Buffer, (const rom char far*)" Saved to index #%d\r\n", (unsigned short)(index+1) ) - USB_In_Buffer;
										USB_In_Buffer[0] = ACK;
									}
								}

								putUSBUSART(USB_In_Buffer, msglen);
								XMODEM = 0;
								if( WAVDestination != 0 )
									spi_protect_all(WAVDestination-1);
							} else {
							CanCan:
								USB_In_Buffer[0] = CAN;
								USB_In_Buffer[1] = CAN;
								putUSBUSART(USB_In_Buffer, 2);
								XMODEM = 0;
								if( WAVDestination != 0 )
									spi_protect_all(WAVDestination-1);
							}
							break;
						case 1:
							if( c == XMODEM_packet ) {
								XMODEM_ignore_packet_data = 0;
								++XMODEM_byte;
							} else if( c == XMODEM_packet-1 && WAVHeaderRead ) {
								// seems like last packet is being re-sent, ignore it since we will have already written it to flash
								XMODEM_ignore_packet_data = 1;
								++XMODEM_byte;
							} else {
								goto CanCan;
							}
							break;
						default:
//						case 2:
							if( c == 255-XMODEM_packet || (c == 0-XMODEM_packet && XMODEM_ignore_packet_data) ) {
								++XMODEM_byte;
							} else {
								goto CanCan;
							}
							break;
						}
					} else if( XMODEM_byte < 128+3 ) {
						XMODEM_buffer[XMODEM_byte-3] = c;
						XMODEM_checksum += c;
						++XMODEM_byte;
					} else {
						unsigned char msglen = 1;
						if( XMODEM_checksum == c ) {
							if( !WAVHeaderRead ) {
								memset(&WAVheader, 0, sizeof(WAVheader));
								if( WAVreadWAVHeader(XMODEM_buffer, &WAVheader, sizeof(XMODEM_buffer)) && WAVheader.dataHeader.dataPtr >= XMODEM_buffer && WAVheader.dataHeader.dataPtr < XMODEM_buffer+sizeof(XMODEM_buffer) ) {
									if( WAVheader.fmtHeader.audioFormat != AUDIO_FMT_PCM || WAVheader.fmtHeader.numChannels != 1 || (WAVheader.fmtHeader.bitsPerSample != 8 && WAVheader.fmtHeader.bitsPerSample != 16) || WAVheader.fmtHeader.sampleRate < 8000 || WAVheader.fmtHeader.sampleRate > 48000 ) {
										TransferError("Bad WAV file format\r\n", &msglen, &XMODEM, WAVDestination);
									} else {
										csl bytes_required;
										bytes_required.sl = WAVheader.dataHeader.data.chunkSize;
										WAVBitDepth8 = (WAVheader.fmtHeader.bitsPerSample == 8);
										if( !WAVBitDepth8 )
											three_quarters(&bytes_required.sl);
										if( bytes_required.sl > audio_get_sound_memory_free() ) {
											TransferError("WAV file too big\r\n", &msglen, &XMODEM, WAVDestination);
										} else {
											WAVDataOffset = WAVheader.dataHeader.dataPtr - XMODEM_buffer;
											WAVBytesLeft = WAVheader.dataHeader.data.chunkSize;
											wavcpyinit();
											if( WAVDataOffset == sizeof(XMODEM_buffer) ) {
												WAVDataOffset = 0;
											} else {
												unsigned char* ptr = XMODEM_buffer + WAVDataOffset;
												if( WAVBytesLeft > sizeof(XMODEM_buffer) - WAVDataOffset )
													WAVBytesLeft -= sizeof(XMODEM_buffer) - WAVDataOffset;
												else
													WAVBytesLeft = 0;
												WAVDataOffset = wavcpy((void*)data_buffer, (const void*)(data_buffer + sizeof(data_buffer)), (const void**)&ptr, (const void*)(XMODEM_buffer + sizeof(XMODEM_buffer)), WAVBitDepth8);
											}

											USB_In_Buffer[0] = ACK;
											WAVHeaderRead = 1;
										}
									}
								} else {
									TransferError("Not WAV file\r\n", &msglen, &XMODEM, WAVDestination);
								}
							} else {
								if( !XMODEM_ignore_packet_data ) {
									unsigned char* ptr = XMODEM_buffer;

									do {
										WAVDataOffset += wavcpy((void*)(data_buffer + WAVDataOffset), (const void*)(data_buffer + sizeof(data_buffer)), (const void**)&ptr, (const void*)(XMODEM_buffer + sizeof(XMODEM_buffer)), WAVBitDepth8);
										if( WAVDataOffset == sizeof(data_buffer) ) {
											if( WAVDestination == 0 ) {
												if( !(FlashAddress.sl&1023) )
													EraseFlashPage(FlashAddress);
												MyWriteBlockFlash(FlashAddress, 2, data_buffer);
											} else {
												if( !(FlashAddress.sl&4095) ) {
									    	        spi_deep_sleep(0);
													if( !spi_erase_flash_chip_4k(WAVDestination-1, FlashAddress.sl) ) {
														TransferError("Flash erase error\r\n", &msglen, &XMODEM, WAVDestination);
														continue;
													}
												}
												if( !spi_write_flash_chip_64b(WAVDestination-1, FlashAddress.sl, 2, (const void*)data_buffer) ) {
													TransferError("Flash write error\r\n", &msglen, &XMODEM, WAVDestination);
													break;
												}
											}
											FlashAddress.sl += 128;
											WAVDataOffset = 0;

									        if( FlashAddress.sl == (WAVDestination == 0 ? INT_MEM_DATA_ADDR + MAX_INT_MEM_BYTES : MAX_EXT_MEM_BYTES) ) {
		                                        if( WAVDestination == 0 && flash_chips_present ) {
											        // continue writing samples to external flash chip
											        WAVDestination = (flash_chips_present&1) ? 1 : 2;
											        FlashAddress.sl = 0;
											        spi_unprotect_all(WAVDestination-1);
											        wavcpy_rollback(data_buffer + sizeof(data_buffer), WAVBitDepth8);
										        } else if( WAVDestination == 1 && (flash_chips_present&2) ) {
											        // continue writing samples to other external flash chip
											        WAVDestination = 2;
											        FlashAddress.sl = 0;
										    	    spi_unprotect_all(WAVDestination-1);
										        	wavcpy_rollback(data_buffer + sizeof(data_buffer), WAVBitDepth8);
										        } else {
											        TransferError("WAV file too big\r\n", &msglen, &XMODEM, WAVDestination);
												    CDCTxService();
											        return;
										        }
									        }
										}
									} while( ptr != XMODEM_buffer + sizeof(XMODEM_buffer) );

									if( WAVBytesLeft > sizeof(XMODEM_buffer) )
										WAVBytesLeft -= sizeof(XMODEM_buffer);
									else
										WAVBytesLeft = 0;
								}
								USB_In_Buffer[0] = ACK;
							}
						} else {
							USB_In_Buffer[0] = NAK;
						}
						putUSBUSART(USB_In_Buffer,1);
						if( !XMODEM_ignore_packet_data )
							++XMODEM_packet;
						XMODEM_byte = 0;
						XMODEM_checksum = 0;
					}
				} else {
					unsigned char l = 1;
					USB_In_Buffer[0] = c;
					if( c == '\r' ) {
						USB_In_Buffer[0] = '\r';
						USB_In_Buffer[1] = '\n';
						l = 2;
					}

					if( c == 8 ) {
						if( CmdBufPos > 0 )
							--CmdBufPos;
						USB_In_Buffer[1] = ' ';
						USB_In_Buffer[2] = c;
						l = 3;
					} else if( CmdBufPos == sizeof(CmdBuf) ) {
						memmove(CmdBuf+0, (const void*)(CmdBuf+1), sizeof(CmdBuf)-1);
						CmdBuf[sizeof(CmdBuf)-1] = c;
					} else {
						CmdBuf[CmdBufPos++] = c;
					}
					if( CmdBufPos >= 3 && CmdBuf[CmdBufPos-3] == SOH && CmdBuf[CmdBufPos-2] == 1 && CmdBuf[CmdBufPos-1] == 254 ) {
						// start of XMODEM transmission
						XMODEM_waiting = 0;
						XMODEM = 1;
						XMODEM_packet = 1;
						XMODEM_byte = 3;
						XMODEM_checksum = 0;
						XMODEM_ignore_packet_data = 0;
						CmdBufPos = 0;
						FlashAddress.sl = audio_get_first_free_address();
						WAVStartAddress.sl = FlashAddress.sl;
						WAVDestination = get_flash_chip(&FlashAddress);
						if( WAVDestination ) {
							spi_deep_sleep(0);
					        spi_unprotect_all(WAVDestination-1);
						}
						WAVHeaderRead = 0;
						l = 0;
					} else if( XMODEM_waiting && CmdBuf[CmdBufPos-1] == NAK ) {
						XMODEM_waiting = 0;
					} else if( CmdBufPos > 0 && (CmdBuf[CmdBufPos-1] == '\r' || CmdBuf[CmdBufPos-1] == '\n') ) {
						if( CmdIs((const rom char far*)"Info", 1) ) {
							l = SendInfoString(l, &current_state);
						} else if( CmdIs((const rom char far*)"Clear all", 1) ) {
							l = ClearMemory(l, 1);
						} else if( CmdIs((const rom char far*)"Clear last", 1) ) {
							l = ClearMemory(l, 0);
						} else if( CmdIs((const rom char far*)"Send", 1) ) {
							if( g_config.sounds[7].start_addr.c[2] != 0xFF ) {
								l = AppendUSBInBuffer(l, (const rom far char*)"No free slots\r\n");
							} else if( !audio_get_first_free_address() ) {
								l = AppendUSBInBuffer(l, (const rom far char*)"No free memory\r\n");
							} else {
								if( XMODEM ) {
									XMODEM = 0;
									l = AppendUSBInBuffer(l, (const rom far char*)"Re-starting XMODEM transfer\r\n");
								} else {
									l = AppendUSBInBuffer(l, (const rom far char*)"Ready for file via XMODEM\r\n");
								}
								XMODEM_waiting = 1;
								XMODEM_Start_Timer = 0;
								XMODEM_Start_Sub_Timer = 0;
                            	flash_chips_present = spi_get_flash_chips_presence();
							}
						} else if( CmdIs((const rom char far*)"Abort", 1) ) {
							if( XMODEM || XMODEM_waiting ) {
								XMODEM_waiting = 0;
								XMODEM = 0;
								if( WAVDestination != 0 )
									spi_protect_all(WAVDestination-1);
								l = AppendUSBInBuffer(l, (const rom char far*)"Transfer aborted\r\n");
							} else {
								l = AppendUSBInBuffer(l, (const rom char far*)"No transfer in progress\r\n");
							}
						} else if( CmdIs((const rom char far*)"Play ", 0) && CmdBufPos == 7 && CmdBuf[5] >= '1' && CmdBuf[5] <= '8' ) {
							unsigned char index = CmdBuf[5] - '1';
							if( g_config.sounds[index].start_addr.c[2] == 0xFF ) {
								l = AppendUSBInBuffer(l, (const rom char far*)"No sound at that index\r\n");
							} else {
								l = mysprintf( USB_In_Buffer + l, (const rom char far*)"Playing sound #%d (%fs)...\r\n", (unsigned short)(index+1), get_sound_time(index) ) - USB_In_Buffer;
								putUSBUSART(USB_In_Buffer, l);
							    CDCTxService();
								audio_play(index, 0, 0, 0, 0, 0);
								l = AppendUSBInBuffer(0, (const rom char far*)"Playback complete\r\n");
							    CDCTxService();
							}
						} else if( CmdIs((const rom char far*)"Options ", 0) && CmdBufPos > 10 && CmdBuf[8] >= '1' && CmdBuf[8] <= '8' && CmdBuf[9] == ' ' ) {
							unsigned char index = CmdBuf[8] - '1';
							const char* pos = (const char*)CmdBuf+10;
							audio_info* info = &g_config.sounds[index];
							if( info->start_addr.c[2] == 0xFF ) {
								l = AppendUSBInBuffer(l, (const rom char far*)"No sound at that index\r\n");
							} else {
								unsigned char loop = 0, partial = 0;
								while( *pos == ' ' )
									++pos;
								while(1) {
									if( !mystricmpram2pgm(pos, (const rom char far*)"loop") ) {
										loop = 2;
										pos += 4;
									} else if( !mystricmpram2pgm(pos, (const rom char far*)"once") ) {
										loop = 1;
										pos += 4;
									} else if( !mystricmpram2pgm(pos, (const rom char far*)"partial") ) {
										partial = 2;
										pos += 7;
									} else if( !mystricmpram2pgm(pos, (const rom char far*)"complete") ) {
										partial = 1;
										pos += 8;
									} else {
										break;
									}
									if( pos[0] == ',' ) {
										while( *++pos == ' ' )
											;
									} else {
										break;
									}
								}
								if( pos - (const char*)CmdBuf == CmdBufPos-1 ) {
									if( loop && !(info->flags&AUDIO_FLAG_LOOP) != (loop-1) ) {
										info->flags ^= AUDIO_FLAG_LOOP;
										g_config.saved = 0;
									}
									if( partial && !(info->flags&AUDIO_FLAG_PARTIAL) != (partial-1) ) {
										info->flags ^= AUDIO_FLAG_PARTIAL;
										g_config.saved = 0;
									}
									l = PrintSoundInfo(USB_In_Buffer + l, index) - USB_In_Buffer;
								} else {
									l = AppendUSBInBuffer(l, (const rom char far*)"Invalid Sound Options\r\n");
								}
							}
						} else if( CmdIs((const rom char far*)"Sounds ", 0) && CmdBufPos > 8 && CmdBuf[7] >= '1' && CmdBuf[7] <= '2' && (CmdBufPos == 9 || CmdBuf[8] == ' ') ) {
							unsigned char index = CmdBuf[7] - '1';
							const char* pos = (const char*)CmdBuf+9;
							unsigned char sounds = 0;

							while( pos - (const char*)CmdBuf < CmdBufPos-1 ) {
								if( *pos >= '1' && *pos <= '8' ) {
									unsigned char sound = *pos - '1';
									if( g_config.sounds[sound].start_addr.c[2] == 0xFF ) {
										l = mysprintf( USB_In_Buffer+l, (const rom far char*)"Sound #%d is not present\r\n", (unsigned short)(sound+1) ) - USB_In_Buffer;
										goto sounds_err;
									}
									sounds |= 1<<sound;
									if( ++pos < (const char*)CmdBuf + CmdBufPos && *pos == ',' ) {
										while( *++pos == ' ' )
											;
									} else {
										break;
									}
								} else {
									break;
								}
							}
							if( pos - (const char*)CmdBuf >= CmdBufPos-1 ) {
								g_config.triggers[index].which_sounds_inv = ~sounds;
								g_config.saved = 0;
								l = PrintTriggerInfo(USB_In_Buffer + l, index) - USB_In_Buffer;
							} else {
								l = AppendUSBInBuffer(l, (const rom char far*)"Invalid Sounds list\r\n");
							}
						} else if( CmdIs((const rom char far*)"Trigger ", 0) && CmdBufPos > 10 && CmdBuf[8] >= '1' && CmdBuf[8] <= '2' && CmdBuf[9] == ' ' ) {
							unsigned char index = CmdBuf[8] - '1';
							const char* pos = CmdBuf+10;
							trigger_info* info = &g_config.triggers[index];
							unsigned char nc = 0, priority = 0, random = 0;
							while(1) {
								if( !mystricmpram2pgm(pos, (const rom char far*)"NC") ) {
									nc = 2;
									pos += 2;
								} else if( !mystricmpram2pgm(pos, (const rom char far*)"NO") ) {
									nc = 1;
									pos += 2;
								} else if( !mystricmpram2pgm(pos, (const rom char far*)"priority") ) {
									priority = 2;
									pos += 8;
								} else if( !mystricmpram2pgm(pos, (const rom char far*)"nopriority") ) {
									priority = 1;
									pos += 10;
								} else if( !mystricmpram2pgm(pos, (const rom char far*)"random") ) {
									random = 2;
									pos += 6;
								} else if( !mystricmpram2pgm(pos, (const rom char far*)"roundrobin") ) {
									random = 1;
									pos += 10;
								} else {
									break;
								}
								if( pos[0] == ',' ) {
									++pos;
									while( *pos == ' ' )
										++pos;
								} else {
									break;
								}
							}
							if( pos - (const char*)CmdBuf == CmdBufPos-1 ) {
								if( nc && !(info->flags&TRIGGER_FLAG_NC) != (nc-1) ) {
									info->flags ^= TRIGGER_FLAG_NC;
									g_config.saved = 0;
								}
								if( priority && !(info->flags&TRIGGER_FLAG_PRIORITY) != (priority-1) ) {
									info->flags ^= TRIGGER_FLAG_PRIORITY;
									// Triggers can't both have priority, if setting trigger x to have priority,
									// other trigger must have priority flag unset (ie, set to 1).
									if( priority == 2 && !(g_config.triggers[index^1].flags&TRIGGER_FLAG_PRIORITY) )
										g_config.triggers[index^1].flags ^= TRIGGER_FLAG_PRIORITY;
									g_config.saved = 0;
								}
								if( random && !(info->flags&TRIGGER_FLAG_RANDOM) != (random-1) ) {
									info->flags ^= TRIGGER_FLAG_RANDOM;
									g_config.saved = 0;
								}
								l = PrintTriggerInfo(USB_In_Buffer + l, index) - USB_In_Buffer;
							} else {
								l = AppendUSBInBuffer(l, (const rom char far*)"Invalid Trigger Options\r\n");
							}
						} else if( CmdIs((const rom char far*)"Save", 1) ) {
							if( g_config.saved == 0xff ) {
								l = AppendUSBInBuffer(l, (const rom char far*)"No changes to save\r\n");
							} else {
								config_save();
								l = AppendUSBInBuffer(l, (const rom char far*)"Configuration saved\r\n");
							}
#ifdef __DEBUG
						} else if( CmdIs((const rom char far*)"Trig ", 0) && CmdBufPos == 7 && CmdBuf[5] >= '1' && CmdBuf[5] <= '2' ) {
//							l = mysprintf( USB_In_Buffer + l, (const rom char far*)"Triggering #%d\r\n", (unsigned short)(CmdBuf[5]-'0') ) - USB_In_Buffer;
//							putUSBUSART(USB_In_Buffer, l);
//						    CDCTxService();
							trigger_sound(CmdBuf[5] == '1', CmdBuf[5] == '2', 0);
//							l = AppendUSBInBuffer(0, (const rom char far*)"Finished\r\n");
//						    CDCTxService();
						} else if( CmdIs((const rom char far*)"Read ", 0) && CmdBufPos > 5 ) {
							unsigned short long start, finish, maxmem = audio_get_sound_memory_total() + 128 * 1024UL - MAX_INT_MEM_BYTES;
							char* ptr = CmdBuf + 5;
							char* end = CmdBuf + CmdBufPos - 1;
							if( strotousl(&start, &ptr, end) && ptr < end && *ptr++ == ' ' && strotousl(&finish, &ptr, end) && ptr == end && start < maxmem && finish < maxmem && start < finish ) {
								mem_read_start = start & ~0xf;
								mem_read_finish = finish |= 0xf;
							} else {
								l = AppendUSBInBuffer(l, (const rom char far*)"Invalid read arguments\r\n" );
							}
#endif//__DEBUG
						} else if( CmdBufPos > 1 ) {
							l = AppendUSBInBuffer(l, (const rom char far*)"Unrecognised command\r\n" );
						}
					sounds_err:
						CmdBufPos = 0;
					}

					if( l )
						putUSBUSART(USB_In_Buffer, l);
				}
			}
		}

		if( (XMODEM_waiting || XMODEM) && ++XMODEM_Start_Timer == 0 && (XMODEM_waiting || ++XMODEM_Start_Sub_Timer == 10) ) {
			XMODEM_Start_Sub_Timer = 0;
			USB_In_Buffer[0] = NAK;
			USB_In_Buffer[1] = '.';
			putUSBUSART(USB_In_Buffer, 2);
		}
	}

    CDCTxService();
}

void USBCBCheckOtherReq(void)
{
    USBCheckCDCRequest();
}
void USBCBInitEP(void) {
    CDCInitEP();
}

#if defined(ENABLE_EP0_DATA_RECEIVED_CALLBACK)
void USBCBEP0DataReceived(void) {
}
#endif

BOOL USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, WORD size)
{
    switch( (INT)event )
    {
/*
        case EVENT_TRANSFER:
            //Add application specific callback task or callback function here if desired.
            break;
*/
/*
        case EVENT_SOF:
            USBCB_SOF_Handler();
            break;
*/
        case EVENT_SUSPEND:
			usb_suspend = 1;
            break;
        case EVENT_RESUME:
			usb_suspend = 0;
            break;
        case EVENT_CONFIGURED: 
            USBCBInitEP();
            break;
/*
        case EVENT_SET_DESCRIPTOR:
            USBCBStdSetDscHandler();
            break;
*/
        case EVENT_EP0_REQUEST:
            USBCBCheckOtherReq();
            break;
/*
        case EVENT_BUS_ERROR:
            USBCBErrorHandler();
            break;
*/
        case EVENT_TRANSFER_TERMINATED:
            //Add application specific callback task or callback function here if desired.
            //The EVENT_TRANSFER_TERMINATED event occurs when the host performs a CLEAR
            //FEATURE (endpoint halt) request on an application endpoint which was 
            //previously armed (UOWN was = 1).  Here would be a good place to:
            //1.  Determine which endpoint the transaction that just got terminated was 
            //      on, by checking the handle value in the *pdata.
            //2.  Re-arm the endpoint if desired (typically would be the case for OUT 
            //      endpoints).
            break;
        default:
            break;
    }      
    return TRUE; 
}
