#ifndef MYLCONTROL_H
#define MYLCONTROL_H
/*
Single instrument control for Load
*/

void onOff(int8_t channel, bool status);
short SCPIsendGrpMessage(uint8_t tgrp, char *buf);
uint8_t setMode(uint8_t aMode);
uint8_t setModeB(uint8_t bMode);
void errorI(bool shutOff, bool DACzero, float reading, char errMsg[]);
int modeCalc(float readV, float readI, uint8_t c_speed, bool guessFirst, bool printMe);
bool getESPviReadings(void);
int dacRead(void);

bool changedLocal = true; // last value changed was local (or remote)
//bool shutDown = false;
bool firstGuess = true; // deal with non-linear cases where first guess is outside the C_VARCOARSE window
float openV;

// coarse and fine control for main output with limiting and tracking
char estop[] = ":TRAC:ESTO";
int cVal = 0;
// must be non-blocking
// quick test routine
int controlx(bool c_speed)
{ int val;	
		val = ampsToDAC(dynSet.current * cVal / 4.0);
		val = constrain(val, 0, 400);
		dacSet(val);     
		//Serial.printf("Control: DAC %i\n", val);

		//setFan(50 + cVal * 15);
		//Serial.printf("Fan %i, %i\n", j, val);

	//Serial.println("Done Control");
	return val;	
}
bool SOAcheck(float vRead, float iRead)
{	
	// do we want to turn off or just limit? 
	if (vRead * iRead > PMAXOP)
	{
		errorI(true, true, vRead * iRead, "Maximum power exceeded");
		//Serial.print("Exceeded SOA\n");
		return true;
	}
	if (iRead > IMAXOP)
	{		
		errorI(true, true, iRead, "Maximum operating current exceeded");
		Serial.printf("Exceeded current limit\n");
		return true;
	}
	return false;
}
bool limitCheck(float vRead, float iRead)
{
	// errorI turns off output
	if (vRead > halCal[ADS_V].maxErr)
	{
		errorI(true, true, vRead, "Maximum voltage exceeded");
		return true;
	}
	if (iRead > halCal[ADS_I].maxErr)
	{		
		errorI(true, true, iRead, "Maximum current exceeded");
		return true;
	}
	return false;
}

// returns setting error (set - val)
float settingsGap(int adcType)
{
	float iRead, vRead;
	iRead = (adcType == ADC_ADS) ? meas.ADSamps : meas.ESPamps;
	vRead = (adcType == ADC_ADS) ? meas.ADSvolts : meas.ESPvolts;
	switch (dynSet.mode)
	{
		case MODE_CC : // CR mode
			return dynSet.current - iRead;
			
		case MODE_CR : // CR mode
			return dynSet.resistance - vRead/iRead;
		
		case MODE_CP : // CP mode
		//Serial.printf(" <SG-P %2.3f = %2.3f * %2.3f> ", dynSet.power, vRead, iRead);
			return dynSet.power - vRead * iRead;
			
		case MODE_CV : // CV mode
			return dynSet.voltage - vRead;		
	}
	return -1.0;
}
// returns setting error (set - val)/set
float settingsRatioGap(int adcType)
{
	float iRead, vRead;
	iRead = (adcType == ADC_ADS) ? meas.ADSamps : meas.ESPamps;
	vRead = (adcType == ADC_ADS) ? meas.ADSvolts : meas.ESPvolts;
	switch (dynSet.mode)
	{
		case MODE_CC : 
			if(dynSet.current < SMALL_DIFF)
				return 0;
			else
				return 1.0 - (iRead)/dynSet.current;
			
		case MODE_CR : 
			if(dynSet.resistance < 1)
				return 0;
			else
				return 1.0 - (vRead/iRead)/dynSet.resistance;
		
		case MODE_CP : 
		//Serial.printf(" <SRG-P %2.3f = %2.3f * %2.3f> ", dynSet.power, vRead, iRead);
			if(dynSet.power < MEDIUM_DIFF)
				return 0;
			else
				return 1.0 - (vRead * iRead)/dynSet.power;
			
		case MODE_CV : 
		
			if(dynSet.voltage < SMALL_DIFF)
				return 0;
			else
				return 1.0 - vRead/dynSet.voltage;		
	}
	return -1.0;
}
float iLast, vLast;
// slow (ADS readings, fine tune) - once through only
// or fast (ESP readings, coarse only) - up to COARSE_ITER iterations 
int control(uint8_t c_speed)
{
	float vSetpoint, iSetpoint, iTrack;
	float vRead, vReadS, iRead, tRead, vGap;
float iGap, iGap2;	
	int jump, dacVal;
	bool rLimit;
	short fjump;
	bool guessFirst = false;	// *************need to fix this ************	

	if (!outOn) // don't control if output is OFF
	{
		Serial.print("!");
	  return _lastDAC;
	}
	if (c_speed == ESP_CONTROL)
	{		 
	  vRead  = meas.ESPvolts;   
	  iRead  = meas.ESPamps;
	  limitCheck(vRead, iRead);	  // overvoltage / short circuit: error - disconnect
	  SOAcheck(vRead, iRead);  // reached limits: error - disconnect???
	  // Check for readings that are too small for ESP control. CC mode independent of vRead.
	  if (iRead < RESPMIN_I || ((dynSet.mode != MODE_CC) && vRead < RESPMIN_V)  )
	  {
#ifdef C_DEBUG   
	if(_printMe) Serial.printf("Low FAST ESP counts: A %i C, V %i C\n", ADCcount.ESPa, ADCcount.ESPv);
#endif
		  return _lastDAC;	// can't FAST control when ESP readings are invalid, wait for ADS reading
	  }
	  // is gap enough to warrant a control cycle?
	  iGap = settingsGap(ADC_ESP);
	  iGap2 = settingsRatioGap(ADC_ESP);
	  if(abs(iGap) < C_VARCOARSE && abs(iGap2) < C_VARCOARSER)
	  {
#ifdef C_DEBUG   
	if(_printMe) Serial.printf("FAST not required, gap is : %2.3f\n", iGap, iGap2);
#endif
		  return _lastDAC; // no need for coarse control
	  }
	}
	else // ADS FINE control 
	{
	  vRead  = meas.ADSvolts;   
	  iRead  = meas.ADSamps;
	  limitCheck(vRead, iRead);	  // overvoltage / short circuit
	  SOAcheck(vRead, iRead);  // reached limits: error - disconnect???
	  iGap = settingsGap(ADC_ADS);
	  iGap2 = settingsRatioGap(ADC_ADS);
	  
	  // gap too small to iterate again
	  // test duplicated in main line
	  if(abs(iGap) < C_VARFINE || abs(iGap2) < C_VARFINER)
	  {
#ifdef C_DEBUG   
	if(_printMe) Serial.printf("SLOW not required, gap is  %2.3f : %2.3f\n", iGap, iGap2);
#endif
		return _lastDAC;
	  }

	  if (vRead < RADSMIN_V && dynSet.mode != MODE_CV)
	  {
#ifdef C_DEBUG   
	if(_printMe) Serial.printf("Low SLOW ADS volts: %2.3fV [%i]\n", meas.ADSvolts, ADCcount.ADSv);
#endif		  
		 // return _lastDAC;	// can't control when readings are invalid, wait for ADS reading, slow creep up of current
	  }
	}
	float dv;
#ifdef C_DEBUG   
   if(_printMe) Serial.printf("\nControl %c iRread %5.3f", (c_speed == ADS_CONTROL) ? 'A' : 'E', iRead);
   if(_printMe) Serial.printf(", vRread %5.3f, MC {\n", vRead);
#endif
	dacVal = modeCalc(vRead, iRead, c_speed, guessFirst, _printMe);
	dv = dacSet(dacVal); 
#ifdef C_DEBUG   
	if(_printMe) Serial.printf("\n} dacVal set %i ", dacVal);
	if(_printMe) Serial.printf(", Gaps %2.3f %2.3f \n", iGap, iGap2);
#endif
    return dacVal;
}
// for the mode, calculate the optimum DAC setpoint given current DAC, volts and amps  
/*
 * return DAC value for desired setpoint in mode
 *  R (CR) mode set amps = volts / desired resistance
 *  P (CP) mode set amps = desired power / volts
 *  V (CV) mode set amps to achieve correct volts (difference between current and desired /volts
 *  
 *  iteration if value is less than C_VARCOARSE (needs to be big enough to capture calculation after first guess)
 * setPoint is V, A, R or P.
 * return DACval
 */
float rDUT;	// estimated from open circuit and lightly loaded voltages
int modeCalc(float readV, float readI, uint8_t c_speed, bool guessFirst, bool printMe)
{
  float setI, curVal, delta; 
  int DACval;
  float setPoint;
  int mode = dynSet.mode;
  switch (mode) // all modes except CC
  {
    // have no idea of correct values at switch on as I reading is Zero.
	// at a new setting, we make the assumption that the DUT has a linear V/I characteristic (resistive)
    // give a small kick initially, to get an initial I reading, then go from there.
    case MODE_CR : 
    case MODE_CP :		
    case MODE_CV : 
      if(readI < IDROPOUT)    
      {          
           DACval = KICK; // provide a small current to get things started
           if (printMe) Serial.printf(" Kick: not enough current. GF %i [%i]\n", guessFirst, DACval ); 
           firstGuess = true;		   
           return DACval;
      }         
	  // CV mode - I setting has brought V down to Zero  
	  // what about other modes?
      if (readV < VDROPOUT & mode == MODE_CV) // avoid stupid answers
      {
         DACval = KICK;
#ifdef C_DEBUG   
         if(printMe) Serial.printf("Kick: Not enough Voltage to guess %3.2f [%i]\n ", readV, DACval);  
#endif
         firstGuess = true;    
         return DACval;
      }        
      break;
    default:	// CC
      break;     
  }
  switch (mode)
  {	   
    case MODE_CC :	
	  setPoint = dynSet.current;
      if(setPoint < IDROPOUT) // avoid divide by zero
      {
        DACval = 0;
        if(printMe) Serial.printf(" C value too small %3.2f [%i]\n ", setPoint, DACval);
        break;
      }
      curVal = readI; 
      if(firstGuess) //abs(curVal - setPoint)/setPoint < C_VARCOARSE) // damped increment
      {     
	     DACval = ampsToDAC(setPoint);
         firstGuess = false;
         if(printMe) Serial.printf(" CC. %3.2f [%i] ",setPoint, DACval); 
	  }
	  else // iterate
      {	  
         delta =   setPoint - curVal;
		 // ampsToDAC is inaccurate at small values, so use HI cal values for delta.
		 float d2 = delta * CDACHI_I / RDACHI_I;
         int dd = C_DAMPING * d2;
         DACval = _lastDAC + dd;     
         if(printMe) 
			 Serial.printf(" CC~ SP %2.2f DE %2.2f [%i + %i = %i]\n ", setPoint, delta, _lastDAC, dd, DACval); 
      }              
      break; // End CC
	  
    case MODE_CR : // CR mode
	/*
	  if (readI > dynSet.current)// setPoint current limiting
	  {
		Serial.print("CP: Limiting to set current");
		DACval = ampsToDAC(dynSet.current);
		break;
	  }
	  */
	  setPoint = dynSet.resistance;
      if(setPoint < RDROPOUT ||  readV / setPoint < IDROPOUT) // avoid divide by zero
      {
        DACval = 0;
        if(printMe) Serial.printf(" R value too small %3.2f [%i]\n ", setPoint, DACval);
        break;
      }
      curVal = readV / readI; // dynamic resistance
      // if(abs(curVal - setPoint)/setPoint < C_VARCOARSE) // damped increment      
      if(firstGuess)
      {
          setI = readV / setPoint;  // assume linear DUT (Not true for PSU)
          DACval = ampsToDAC(setI);		  
          firstGuess = false; 
          if(printMe) Serial.printf(" CR. %3.2f [%i]\n",setI, DACval);
      }     
      else    
      { 
         delta = (readV / setPoint) - (readV / curVal)  ; // delta I
		 // ampsToDAC is inaccurate at small values, so use cal values directly.
		 float d2 = delta * CDACHI_I / RDACHI_I;
         DACval = _lastDAC + C_DAMPING * d2; 
         DACval = constrain(DACval, 0, ampsToDAC(dynSet.current)); // keep below the current limit.
         if(printMe) Serial.printf(" CR~ %2.2f [%i + %i = %i]\n", delta, _lastDAC ,d2, DACval); 
      } 
      break; // End CR
      
    case MODE_CP :	  
	/*
	  if (readI > dynSet.current) // setPoint current limiting
	  {
		Serial.print("CP: Limiting to set current");
		DACval = ampsToDAC(dynSet.current) - 10; go a bit below
		break;
	  }
	  */
	  setPoint = dynSet.power;
      if(setPoint < PDROPOUT) // avoid divide by zero
      {
        DACval = 0;
        if(printMe) Serial.printf(" P value too small %3.2f [%i]\n ", setPoint, DACval);
        break;
      }
	  if(c_speed == ADC_ADS)
	  {  
		if(readV < RADSMIN_V) // too much current, can't estimate accurately at low voltage
		{
			DACval = _lastDAC - DAC_BIGJUMP;
			break;
		}
	  }
	  else // esp
	  {
		if(readV < RESPMIN_V) // active control here causes instability
		{
			DACval = _lastDAC; 
			break;
		}
	  }
      curVal = readV * readI; 

     if(firstGuess)
     {
           setI =  setPoint / readV;   
           DACval = ampsToDAC(setI);  
           firstGuess = false;
           if(printMe) Serial.printf(" CP. read %3.2fW, set %3.2fA [%i]", curVal, setI, DACval);                 
      }    
      else // iterate
      {	
         delta = (setPoint - curVal) / readV; // in amps
		 // ampsToDAC is inaccurate at small values, so use cal values directly.
		 // also makes an assumption of 0 ESR
		 /*
		 float d2 = delta * CDACHI_I / RDACHI_I;
         DACval = _lastDAC + C_DAMPING2 * d2;   
		  */
		 int d2 = C_DAMPING * (ampsToDAC(setPoint/readV) - ampsToDAC(curVal/readV)); // non-linear so need two conversions
		 int d3 = (SIGN(delta)) * ((abs(delta) < 0.1) ? DAC_SMALLJUMP : DAC_BIGJUMP);		
		 DACval = _lastDAC + d2;
		 DACval = constrain(DACval, 0, ampsToDAC(dynSet.current)); // keep below the current limit.
         if(printMe) Serial.printf(" CP~ read %3.2fW, delta %3.2f, SGN %i, set [%i + %i [%i] = %i] ", curVal, delta, SIGN(delta) , _lastDAC , d2, d3,DACval);   
      } 
	  
      break; // End CP

    case MODE_CV  : // serial calculation of rDUT to determine deltaI
	  setPoint = dynSet.voltage;	  
      if(setPoint < VDROPOUT) // avoid divide by zero
      {
        DACval = 0;
        if(printMe) Serial.printf(" V value too small %3.2f [%i]\n ", setPoint, DACval);
        break;
      }     
     
      if (readV < VDROPOUT) // turn off if no viable DUT voltage
		  return 0;
	  
	  // no sensible first guess for non-linear systems like a PSU?
	  // estimate only works as a first guess - or when V/I reading gaps are large, better to just use search function?
	  rDUT = abs((readV - _lastVolts)/(readI - _lastAmps)); // estimate DUT resistance from open circuit voltage and KICK or current V/I
      if(firstGuess)
      {   
		int deltaI = (_lastVolts - setPoint)/rDUT;       // I = E(diff)/R          
        DACval = _lastDAC + ampsToDAC(setI);          
        if(printMe) 
        {
           Serial.printf(" . rDUT %3.2f I %3.2f [%i], ", rDUT, setI, DACval);
           Serial.printf(" openV %3.2f, readV %3.2f, readI %3.2f \n", openV, readV, readI);
        }
        firstGuess = false;
      }  
      else // iterate:  find proportional gap between Voltage and desired V      
      {		
		delta = readV/setPoint;  
		// may need to modify when near solution
		// assume V : I
		//int d4 = C_DAMPING3 * (delta - 1)/rDUT; 
		float iNew = readI * delta;
		int delta4 = (ampsToDAC(iNew) - ampsToDAC(readI));
		int delta3 = 1.0 * (float)delta4;
		DACval = _lastDAC + delta3;  // voltage too high: more amps: DAC++
		DACval = constrain(DACval, 0, ampsToDAC(dynSet.current)); // ensure we don't go above the current limit.
		firstGuess = false;
		if(printMe) Serial.printf(" CV~  iNew %2.3f, delta %2.2f[%i], [%i + %i = %i] ", iNew, delta, delta4, _lastDAC, delta3, DACval);
      }            
      break; // End CV
	  default: // this is an error condition
		DACval = 0;	 
  }     
  _lastVolts = readV; // save the readings for next time (CV mode)
  _lastAmps  = readI;
  if(DACval > CDAC_OPMAX) Serial.printf("DAC_O: %3.2f %i\n", setPoint, DACval);

  DACval = constrain(DACval, 0, CDAC_OPMAX); // ensure we don't go above the maximum allowed setting.
  return DACval;
}

uint8_t _DACaddr;
void control_setup(void)
{	
  outOn = false;  // ensure output is off 
  pinMode(SW_ON, OUTPUT);
  digitalWrite(SW_ON, LOW); // OFF to start, when outOn is true: in Output mode to drive LED. false (Off): Input mode to read switch.
  // Brief pulse to reset output relay (OFF)
  pinMode(SW_OFF, OUTPUT);
  digitalWrite(SW_OFF, HIGH);
  delay(50);
  digitalWrite(SW_OFF, LOW);
  pinMode(SW_OFF, INPUT_PULLDOWN); // always in Input mode after this 

  //float volts, realWorld;
  
  // **********  DAC***********
 // MCP45begin(WSTART, DIGI_V); 
  _DACaddr = probeDAC(DAC_BASE, DAC_SCAN);	// maybe anywhere in 0x60-0x67
  dac.begin(_DACaddr);
  dac.setVoltage(0, true);	// set DAC startup (EEPROM) and current value to 0
  delay(100); 				// EEPROM write takes some (undefined) time [could poll the status register]
 
  // watchdog?
  // wTimerBegin(100); // mS

  
  // ****************** ADC ***********
  // start ADS1115 ADC
  pinMode(ADSRDY, INPUT_PULLUP);
  ads.setGain(ADS_GAIN);        // 1x gain   +/- 4.096V  1 bit = 2mV      0.125mV
  ads.begin(I2C_SPEED);  // I2C speed, 1000000 (1M = FMPlus) is probably OK, 400000 is safe. Defaults to 128SPS.
  ads.setRate(7);	// 860SPS
 // Serial.println("ADS BEGIN done");
  //  delay(100);
  
   // zero the output current reading
  delay(10);
  int iZero = readADS_I();
 // myADC[IOUT].minVolts =  convertHAL(iZero, &halCal); 


  // start the regular conversion process
  attachInterrupt(digitalPinToInterrupt(ADSRDY), ads_ISR, FALLING);
 // Serial.println("ATTACH int done");
  //  delay(100);  
  ads.adsStartSingleDiffRdy(adsConfig, ADSMUX_START);
  
  // esp ADC
  analogReadResolution(12);
  //analogSetSamples(3);
  pinMode(ADC_BV, INPUT); // ext pullups
  pinMode(ADC_BI, INPUT);
  pinMode(ADC_BT, INPUT);
  
/********** ENCODER and Tac Switches *************/
  enc.setChangedHandler(rotate);
  butL.setPressedHandler(clickedL);
  butR.setPressedHandler(clickedR);
  
  // Fan: attach the channel to the GPIO to be controlled
  ledcAttachPin(PWM_PIN, PWMChannel);  
  // configure  PWM functionalitites
  ledcSetup(PWMChannel, PWM_FREQ, PWM_RESOLUTION);
  setFan(FAN_MIN_PWM);	// start the fan on idle
  
  // Daughter board detect
   pinMode(DPRESENT, INPUT_PULLUP); // no ext pullup, pull down for daughter present
   delay(1);
   daughter_mul = (digitalRead(DPRESENT))? 1 : 2; // double the output current if a daughter board is present
  
  // set the mode variables  
  setMode(pSet.mode); // set initial mode indicator (from EE)
  setNORb(true);	// normal mode B
  //dynSet = pSet; // done in setMode and setModeB
}

#define DEBOUNCE_OO  5
uint8_t onSwitch, offSwitch;
int oo_samestate = 0;
// more complex conditions, so can't use Button2 library
void processOnOffSw(void)
{
	uint8_t onVal, offVal;
	pinMode(SW_ON, INPUT_PULLDOWN); // pin drives LED, so only Input while reading
	// SW_OFF is always in Input mode, except when being pulsed by an Off command
	onVal = digitalRead(SW_ON);
	offVal = digitalRead(SW_OFF);	
	pinMode(SW_ON, OUTPUT);
	digitalWrite(SW_ON, (outOn)? HIGH : LOW); // re-assert LED status using first channel - will be the same for all channels
	
	//debounce: must have same state for DEBOUNCE_OO cycles
	if(onVal == onSwitch && offVal == offSwitch)
		oo_samestate++;
	else
	{
		onSwitch = onVal;
		offSwitch = offVal; 
		oo_samestate = 0;
		return;
	}	
	
	if (oo_samestate >= DEBOUNCE_OO)
	{
		if (onVal == HIGH && !outOn) // switch on, if not already on
		{
			onOff(-1, true);
		}
		if (offVal == HIGH && outOn) // switch off
		{
			onOff(-1, false);
		}	
	}	
}

//used by SCPI as well as panel buttons
//sets Onoff and kills any running mode
// status false = OFF
void onOffKill(int8_t channel, bool status)
{
	onOff(channel, status); // hardware on/off
			
	if(!status) // stop any  running BAT or STEP mode operations
	{
		setModeB(MODE_NOR); 
		_curState = BMODE_OFF;  
		dacSet(0);
		dynSet = pSet;	// go back to standard settings
	}	
}

void onOff(int8_t channel, bool status)
{
	outOn  = status;
	//Serial.printf("Changing On/Off [ch %i] to %i [pin %i]\n", i, status, chanPins[i].onPin);
	// physical control 
	if(status) 		// ON
	{
		digitalWrite(SW_ON, HIGH);  // always in OUTPUT mode, except when being explicitly read
	}
	else 			// OFF
	{ 
		dacSet(0); // set control voltage to OFF	
		// pulse Off pin - may be a SCPI command		
		digitalWrite(SW_ON, LOW); // on pin LOW		
		pinMode(SW_OFF, OUTPUT); 			
		digitalWrite(SW_OFF, HIGH); 
		delay(1);
		digitalWrite(SW_OFF, LOW);
		pinMode(SW_OFF, INPUT_PULLDOWN);  // reset pin for panel control
	}	
}
// display counts and converted values
void dispConv(void)
{
	/*
	Serial.printf("Vin  [%i] = %5.2f [%i counts = %5.2fV]\n", VIN,    myADC[VIN].curVal,  adsReadings[VIN],  myADC[VIN].curVolts );
	Serial.printf("Vout [%i] = %5.2f [%i counts = %5.2fV]\n", VOUT,   myADC[VOUT].curVal, adsReadings[VOUT], myADC[VOUT].curVolts );
	Serial.printf("Iout [%i] = %5.2f [%i counts = %5.2fV]\n", IOUT,   myADC[IOUT].curVal, adsReadings[IOUT], myADC[IOUT].curVolts );
	Serial.printf("Temp [%i] = %5.2f [%i counts = %5.2fV]\n\n", TEMP, myADC[TEMP].curVal, adsReadings[TEMP], myADC[TEMP].curVolts );
	*/
}
void currentZeroCal(void)
{
	  if(outOn == false)
	  {
		//myADC[IOUT].minVolts = myADC[IOUT].curVolts - IOUT_OFFSET;
		//Serial.printf("Current zero = %5.3f\n",  myADC[IOUT].minVolts);
	  }
}

void errorI(bool shutOff, bool DACzero, float reading, char errMsg[])
{
	char msgBuf[128];
	if (shutOff){		
		onOffKill(-1, false);    // shut off output		
	}
	if(DACzero){
		//shutDown = true;  //change setpoints to safe values		
		dacSet(0);
	}
	// display screen error message, indicator, and store for SCPI 
	sprintf(msgBuf,"*** Error: \n%s \n[%5.2f]\n", errMsg, reading);
	screenError(msgBuf, ERR_BG_B, 10, false);
}


int fanControl(void)
{
	float ff;
	if(meas.ESPtemp < TEMP_FANSTART)
		return setFan(FAN_MIN_PWM);
	else
	{
		ff = fmap(meas.ESPtemp, TEMP_FANSTART, TEMP_FANFULL, FAN_MIN_PWM, 100);	
		return setFan(ff);
	}
}

#endif