/* 
 * File:   SPECTRAL_MIDI.c
 * Author: Jeremy Leach 2020
 */
#include "mcc_generated_files/mcc.h"
#include "SPECTRAL_MIDI.h"
#include "SPECTRAL_messaging.h" 

/******************************************************************************/
/***** Variables **************************************************************/
/******************************************************************************/
volatile uint8_t adc_change_interval_count = 0;
volatile uint8_t midi_rx_LED_hundreths = 0;
static volatile uint8_t flag_midi_rx_timed_out = 0;
static volatile uint8_t flag_midi_rx_buffer_overflow = 0;
static volatile tone_instance_allocation tone_instance_allocation_table[max_tone_processors * instances_per_tone_processor];
static volatile uint8_t last_allocation_processor_index;
static volatile fifo_midi_buffer midi_input_buffer ; //__at(160);      //35 + 4 bytes. BANK 1 
static volatile channel_info channel[max_midi_channels]; 
static volatile uint8_t last_allocation_id;
volatile performance_t performance[performances_per_patchset];
extern volatile uint8_t performance_id;
extern volatile uint8_t flag_pending_send_performance_to_mixer;
extern volatile uint8_t flag_pending_select_performance;
extern volatile uint8_t flag_volume_check_pending;
extern volatile uint8_t flag_cc_check_pending;
volatile uint8_t bank_lsb;
volatile uint8_t bank_msb;
volatile uint16_t bank;
volatile uint8_t program_change;
volatile uint16_t default_performance_bank;
volatile uint16_t default_patch_bank;

    
/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/
uint8_t GetPatchForChannel(uint8_t midi_channel)
{
return channel[midi_channel].midi_channel_patch_setting;
}

uint8_t get_channel_max_instances(uint8_t c,uint8_t for_performance_id)
{
    uint8_t processor_id;
    uint8_t result = 0;
    for (processor_id = 0 ; processor_id < max_tone_processors;processor_id++)
    {
        if (performance[for_performance_id].processor_channel_allocation[processor_id] == c)
        {
            result += instances_per_tone_processor;
        }
    }
    
    return result;
}

void SPECTRAL_MIDI_Initialise(uint8_t for_performance_id)
{
    flag_pending_send_performance_to_mixer = 0;
    flag_pending_select_performance = 0;
    
    uint8_t processor_id;
    uint8_t ipp;
    uint8_t i;
    uint8_t flag_eeprom_configured;
    
    //Clear the MIDI buffer
    for (i = 0; i< midi_buffer_size;i++){midi_input_buffer.buf[i] = 0;}
    
    //Check if the EEPROM is configured
    if (eeprom18_read(0) == eeprom_loaded_indicator){flag_eeprom_configured =1;}else{flag_eeprom_configured=0;}
    
    //Clear the instance allocation table and set the MIDI channels
    i = 0;
    for (processor_id = 0 ; processor_id < max_tone_processors;processor_id++)
    {
        for(ipp = 0;ipp <instances_per_tone_processor;ipp++)
            {
            if (flag_eeprom_configured == 1)
                {tone_instance_allocation_table[i].midi_channel = performance[for_performance_id].processor_channel_allocation[processor_id];}
            else
                {tone_instance_allocation_table[i].midi_channel = 0;}
            
            tone_instance_allocation_table[i].note_number = 0;
            tone_instance_allocation_table[i].flag_held_on_by_sustain = 0;
            tone_instance_allocation_table[i].allocation_id = 0;
            i++;
            }
    }

    last_allocation_processor_index = 0;

    uint8_t c;
    for(c=0;c<max_midi_channels;c++)
        {
        channel[c].flag_sustain_on = 0;
        channel[c].midi_channel_patch_setting = 0; 
        channel[c].last_midi_note_number = 0;
        channel[c].total_load = 0;
        channel[c].max_instances = get_channel_max_instances(c,for_performance_id);
        }
    
    last_allocation_id = 0;
    bank_lsb = 0;
    bank_msb = 0;
    bank = 0;
}

tone_instance FindAllocatedToneInstance(uint8_t midi_channel,uint8_t note_number)
{
    tone_instance ti;
    uint8_t p;
    uint8_t i;
    uint8_t iat_index; 
    
    ti.processor_id = 0;
    ti.instance_id = 0;
    ti.flag_null = 1;
    iat_index = 0;
    
    for(p = 0;p < max_tone_processors;p++)
       {
           for(i = 0;i < instances_per_tone_processor;i++)
           {
               if ((tone_instance_allocation_table[iat_index].midi_channel == midi_channel) && (tone_instance_allocation_table[iat_index].note_number == note_number))
               {
                   ti.processor_id = p;
                   ti.instance_id = i;
                   ti.iat_index = iat_index;
                   ti.flag_null = 0;
                   return ti;
               }
               iat_index ++;
           }  
       }
    return ti;
}

tone_instance AllocateToneInstance(uint8_t midi_channel,uint8_t note_number)
{ 
    tone_instance ti;
    uint8_t iat_index;
    uint8_t p;
    uint8_t i;  
    
     /*Don't allocate if already allocated. This can happen when sustain is pressed because the sustain is keeping the
     current instance of this note number active even though a note off was received. When the key is pressed again
     the new note on would generate another instance of the same note unless we do this check.
     */        
    ti = FindAllocatedToneInstance(midi_channel,note_number);
    
    if (ti.flag_null == 0) 
        {
        //Re-start this note
        tone_instance_allocation_table[ti.iat_index].midi_channel = midi_channel;
        tone_instance_allocation_table[ti.iat_index].note_number = note_number;
        tone_instance_allocation_table[ti.iat_index].flag_held_on_by_sustain = 0;
        tone_instance_allocation_table[ti.iat_index].allocation_id = last_allocation_id;last_allocation_id++;         
        last_allocation_processor_index = ti.processor_id;
        return ti;
        }
    else if (channel[midi_channel].total_load == channel[midi_channel].max_instances)
        {
        //=============== Steal the oldest note ==========================
        uint16_t min_allocation_id; min_allocation_id = 256;
        iat_index = 0;
        ti.flag_null = 0;
        
        //Find the instance with the lowest allocation id.
        for (p=0;p<max_tone_processors;p++)
            {
            for(i = 0;i<instances_per_tone_processor;i++)
                {
                if ((tone_instance_allocation_table[iat_index].note_number > 0) && ((uint16_t)tone_instance_allocation_table[iat_index].allocation_id <min_allocation_id) && (tone_instance_allocation_table[iat_index].midi_channel == midi_channel))
                    {
                    min_allocation_id = tone_instance_allocation_table[iat_index].allocation_id;
                    ti.instance_id = i;
                    ti.processor_id = p;
                    ti.iat_index = iat_index;
                    }
                iat_index++;
                }
            }
        //Set this instance to the new note. Don't end the last note on this instance because it will end anyway !
        tone_instance_allocation_table[ti.iat_index].midi_channel = midi_channel;
        tone_instance_allocation_table[ti.iat_index].note_number = note_number;
        tone_instance_allocation_table[ti.iat_index].flag_held_on_by_sustain = 0;
        tone_instance_allocation_table[ti.iat_index].allocation_id = last_allocation_id;last_allocation_id++;         
        last_allocation_processor_index = ti.processor_id;
        return ti;
        }
    else
        {
        //Method used is to try to jump between processors every time an allocation is made, thereby spreading the load.
        //This is easier to implement and much faster than keeping a detailed 'loading' for each processor and picking the 'least loaded'.      
        p = (uint8_t)(last_allocation_processor_index + 1);//The next processor
        do
            {
            if(p >= max_tone_processors){p=0;}
            iat_index = (uint8_t)((p << 2) - p); // NOTE: p << 2 - p is the same as p * 3.
            for (i = 0; i < instances_per_tone_processor;i++)
                {
                    if ((tone_instance_allocation_table[iat_index].note_number == 0) && (tone_instance_allocation_table[iat_index].midi_channel == midi_channel))
                        {
                        //Unnecessary : tone_instance_allocation_table[iat_index].midi_channel = midi_channel;
                        tone_instance_allocation_table[iat_index].note_number = note_number;
                        tone_instance_allocation_table[iat_index].flag_held_on_by_sustain = 0;
                        tone_instance_allocation_table[iat_index].allocation_id = last_allocation_id;
                        last_allocation_id++;
                        ti.flag_null = 0;
                        ti.iat_index = iat_index;
                        ti.processor_id = p;
                        ti.instance_id = i;
                        
                        last_allocation_processor_index = p;
                        channel[midi_channel].total_load ++;
                        return ti;
                        }
                iat_index++;
                }
            p++;
            } while(1);
        }
    
    return ti;
}

uint8_t fMidiBufferHasData(volatile fifo_midi_buffer *f)
{
    if (f->head != f->tail)
        {return 1;}
    else
        {return 0;}
}

uint8_t ReadByteFromMidiBuffer(volatile fifo_midi_buffer *f)
{   //Assumes there is data in the buffer (use fMidiBufferHasData)
    uint8_t b = f->buf[f->tail];
    f->count--;
    f->tail++;
    if (f->tail == midi_buffer_size)
        { f->tail = 0;}
    
    return b;       
}

uint8_t WaitForMidiByte()
{
    //NOTE: If the loop below was simply one NOP then it would timeout in 5000 cycles or 1ms (at 20MHz clock).
    //So this is only a very rough timeout and it will be considerably more.
    flag_midi_rx_timed_out = false;
    unsigned int i;
    for (i=1;i <= 5000; i++)
        {
        if (fMidiBufferHasData(&midi_input_buffer) == 1)
            { return ReadByteFromMidiBuffer(&midi_input_buffer);
            }
        else
            NOP();
        }
    
    //timed out
    flag_midi_rx_timed_out = true;
    return 0; 
}

void Get2EventDataBytes(midi_event *e,uint8_t first_byte)
    {
        if ((first_byte & 0b10000000) > 0)
             { 
             //Get two bytes
             e->data1 = WaitForMidiByte();
             if (!flag_midi_rx_timed_out)
                {e->data2 = WaitForMidiByte();}       
             }
        else
            { 
            //Count the first byte that has already been received as a data byte
            e->data1 = first_byte;
            e->data2 = WaitForMidiByte();
            }
    }

void Get1EventDataBytes(midi_event *e,uint8_t first_byte)
    {
    if ((first_byte & 0b10000000) > 0)
        {e->data1 = WaitForMidiByte();}
    else
        {e->data1 = first_byte;}
    }

void ProcessNote(midi_event e,uint8_t status_without_channel,uint8_t midi_channel)
{
    tone_instance ti;
    
    if (midi_channel >=max_midi_channels){return;}
    
    //Change any note on with zero velocity to Note Off. Some manufacturers don't use note off at all but best to stick to official MIDI spec.
    if ( (status_without_channel == STATUS_note_on) && (e.data2 == 0) )
        {
        e.status = (uint8_t)(STATUS_note_off | (e.status & 0b00001111));
        status_without_channel = STATUS_note_off;
        }    

    if (status_without_channel == STATUS_note_on)
        {
        ti = AllocateToneInstance(midi_channel,e.data1);
        if (ti.flag_null == 0)
            {
            SPECTRAL_ForwardMIDIMessage(ti.processor_id,ti.instance_id,e,channel[midi_channel].last_midi_note_number);
            channel[midi_channel].last_midi_note_number = e.data1;
            }
        }   
    else if (status_without_channel == STATUS_note_off)
        {
        ti = FindAllocatedToneInstance(midi_channel,e.data1);
        //Note that if note stealing is happening then it is possible that the note number is longer found. 
        if (ti.flag_null == 0)
            {
            if (channel[midi_channel].flag_sustain_on == 1)
                {tone_instance_allocation_table[ti.iat_index].flag_held_on_by_sustain = 1;}
            else
                {
                SPECTRAL_ForwardMIDIMessage(ti.processor_id,ti.instance_id,e,channel[midi_channel].last_midi_note_number);
                //tone_instance_allocation_table[ti.iat_index].midi_channel = 0;
                tone_instance_allocation_table[ti.iat_index].note_number = 0;
                tone_instance_allocation_table[ti.iat_index].flag_held_on_by_sustain = 0;
                if (channel[midi_channel].total_load > 0){channel[midi_channel].total_load--;}if (channel[midi_channel].total_load == 0){last_allocation_id = 0;}
                }        
            }
        }
}

void ProcessReceivedMidi()
{   uint8_t b;
    uint8_t s;
    uint8_t midi_channel;
    midi_event e;
    
    b = ReadByteFromMidiBuffer(&midi_input_buffer);
        
    //Check if a status byte (bit 7 set) 
    if ((b & 0b10000000) == 0b10000000)
        {
        s = b;
        if ( (s >= 0x80) && (s <= 0xEF) ) 
            {
            //store status when a Voice Category Status (0x80 to 0xEF) is received
            midi_input_buffer.midi_running_status = b;
            }
        else if ( (s >= STATUS_sysex_start) && (s <= STATUS_sysex_end) ) 
            {
            //Clear running status when a System Common Category Status (ie, 0xF0 to 0xF7) is received. 
            //Things like Sysex, section position pointer. The message after this message MUST have a status byte.
            midi_input_buffer.midi_running_status = 0;
            if (s == STATUS_sysex_start) //SysEx start
                {
                    uint8_t SysexHeader[4];
                    uint8_t i;
                    for(i = 0;i <=3;i++)
                    {SysexHeader[i] = WaitForMidiByte();}

                    //Wait until SysEX end is received. Doing NOTHING else except this loop ensures no buffer over-flow.
                    uint8_t sb;
                    sb = 0;
                    while(sb != STATUS_sysex_end){sb = WaitForMidiByte();};   
                }
            return;
            }
        else if (s >= 0xF8) 
            {
            return;
            }
        }
    else
        {
        //RX bytes isn't status byte so get the running status.
        s = midi_input_buffer.midi_running_status;
        } 

    if (s == 0)
    {
        //If running status is zero it is a System Common Category Status message (like Sysex) and ignore the RX byte.
        return;
    }

    e.status = s;
    midi_channel = (uint8_t)(s & 0b00001111);//Channel starts at 0.
    uint8_t status_without_channel = (uint8_t)(s & 0b11110000);

    if ( (status_without_channel == STATUS_note_on) || (status_without_channel == STATUS_note_off) )
        {
        Get2EventDataBytes(&e,b); //The note number and velocity.

        if (!flag_midi_rx_timed_out)
            {        
            ProcessNote(e,status_without_channel,midi_channel);
            }
        }
    else if (status_without_channel == STATUS_control_change)
        {
        Get2EventDataBytes(&e,b); //The Controller byte and the controller value byte
        if (!flag_midi_rx_timed_out)
            {           
            if (e.data1 == cc_sustain_pedal)
                {
                if ((e.data2 <=63) && (channel[midi_channel].flag_sustain_on == 1)) //Filter out multiple messages. We just see sustain as ON and OFF without a level of sustain.
                    {
                    //=========== Sustain off ==================
                    channel[midi_channel].flag_sustain_on = 0;
                    uint8_t p;
                    uint8_t i;
                    uint8_t iat_index; 
                    iat_index = 0;
                    midi_event eno;
                    eno.status = (uint8_t)(((uint8_t)STATUS_note_off | midi_channel));
                    eno.data2 = 0;
                    e.data2 = 0;//Force the sustain value to be completely off

                    for(p = 0;p < max_tone_processors;p++)
                       {
                        for(i = 0;i < instances_per_tone_processor;i++)
                            {
                                if ((tone_instance_allocation_table[iat_index].midi_channel == midi_channel) && (tone_instance_allocation_table[iat_index].flag_held_on_by_sustain == 1))
                                {
                                 eno.data1 = tone_instance_allocation_table[iat_index].note_number;
                                 SPECTRAL_ForwardMIDIMessage(p,i,eno,channel[midi_channel].last_midi_note_number);
                                 //tone_instance_allocation_table[iat_index].midi_channel = 0;
                                 tone_instance_allocation_table[iat_index].note_number = 0;
                                 tone_instance_allocation_table[iat_index].flag_held_on_by_sustain = 0;
                                 if (channel[midi_channel].total_load > 0){channel[midi_channel].total_load--;}; if (channel[midi_channel].total_load == 0){last_allocation_id = 0;}
                                }
                                iat_index ++;
                            }  
                       }
                    SPECTRAL_ForwardMIDIMessage(0,0,e,channel[midi_channel].last_midi_note_number); 
                    
                    }
                else if ((e.data2 >63) && (channel[midi_channel].flag_sustain_on == 0))
                    {
                    //============= Sustain on ================
                    channel[midi_channel].flag_sustain_on = 1;
                    e.data2 = 127;//Force the sustain value to be completely on
                    SPECTRAL_ForwardMIDIMessage(0,0,e,channel[midi_channel].last_midi_note_number); 
                    }
 
                }
            else
                {
                //Forward any other control change messages
                SPECTRAL_ForwardMIDIMessage(0,0,e,channel[midi_channel].last_midi_note_number); 
                }
            }
        }
    else if (status_without_channel == STATUS_program_change)
        {
        Get1EventDataBytes(&e,b); 
        program_change = e.data1;
        if (bank == default_performance_bank)
            {
            if(program_change < performances_per_patchset)
                {
                performance_id = program_change;
                flag_pending_select_performance = 1;
                }
            }
        else if (bank == default_patch_bank)
            {
            if(program_change < patches_per_patchset)
                {
//                //***TEST
//                while (1)
//                {OUT_MIDI_SENSE_SetHigh();}
                           
                SPECTRAL_SendSelectPatch(program_change);
                }        
            }
        }
    else
        { 
        switch(status_without_channel)
            {
                case STATUS_channel_aftertouch:
                    Get1EventDataBytes(&e,b); //One data byte
                    break;
                case STATUS_aftertouch:
                    Get2EventDataBytes(&e,b); //Two data bytes
                    break;
                case STATUS_pitch_bend:
                    Get2EventDataBytes(&e,b); //Two data bytes
                    SPECTRAL_ForwardMIDIMessage(0,0,e,channel[midi_channel].last_midi_note_number);
                    break;
            }
        }
}

void Rx_MIDI_Event_InterruptHandler() 
{   
        if (midi_input_buffer.count >= max_midi_buffer_count)
            {flag_midi_rx_buffer_overflow = 1;
            //INTCONbits.GIE = 0;OUT_MIDI_SENSE_SetHigh();while(1){}
            }
        else
            {
            midi_input_buffer.buf[midi_input_buffer.head] = RCREG; //Resets PIR1bits.RCIF 
            midi_input_buffer.head++;
            midi_input_buffer.count++;
            if (midi_input_buffer.head == midi_buffer_size){midi_input_buffer.head = 0;}
            }
}

void ProcessMIDI() 
{
if (flag_pending_select_performance == 1)
    {SelectPerformance();}    
    
if (flag_pending_send_performance_to_mixer == 1)
    {
    RelayPerformancesToMixer();
    SPECTRAL_SavePerformancesToEEPROM();
    SPECTRAL_SaveBankNumbersToEEPROM();
    }

while (fMidiBufferHasData(&midi_input_buffer) == 1){ProcessReceivedMidi();} 
}

void MIDI_TMR1_InterruptHandler()
{
    flag_cc_check_pending  = 1;
    
    adc_change_interval_count++; 
    if (adc_change_interval_count == volume_update_10ms)
        { 
        adc_change_interval_count = 0; 
        flag_volume_check_pending = 1;
        }
    
    if (midi_rx_LED_hundreths > 0)
    {
        midi_rx_LED_hundreths --;
        if (midi_rx_LED_hundreths == 0)
        {
           OUT_MIDI_SENSE_SetLow();
        }   
    }
}