﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using Spectral1.DATA_ACCESS;
using static Spectral1.PARSER.c_dsl_functions;
using static Spectral1.PARSER.c_lexer;
using System.IO;
using static Spectral1.DATA_ACCESS.DA_Spectral;
using static Spectral1.PARSER.c_parser;
using System.Reflection;

namespace Spectral1.PARSER
{
    public partial class PL_UC_SpectralParser : UserControl
    {
        //Ideas from : https://jack-vanlightly.com/blog/2016/2/3/creating-a-simple-tokenizer-lexer-in-c
        //A domain-specific language (DSL) is a computer language specialized to a particular application domain.
        //https://medium.com/factory-mind/regex-tutorial-a-simple-cheatsheet-by-examples-649dc1c3f285
        //https://www.codeproject.com/Articles/5839/DIY-Intellisense

        //#############################################
        //#### SDL : SPECTRAL DEFINITION LANGUAGE  ####
        //#############################################

        #region "================= DECLARATIONS ===================="
        private c_parser parser = new c_parser();
        private bool _macro_recording = false;
        private parser_modes _parser_mode = parser_modes.all;
        private recording_modes current_recording_mode = recording_modes.none;
        private PL_Main _plm;
        #endregion

        #region "================== PROPERTIES ====================="
        public  bool macro_recording
        {
            get{ return _macro_recording; }
            set{ _macro_recording = value; }
        }

        public parser_modes parser_mode
        {
            get { return _parser_mode; }
            set { _parser_mode = value; }
        }

        public bool got_code
        {
            get { return richTextBox1.TextLength > 0; }
        }

        #endregion

        #region "================== MENU BUTTON EVENTS ========================="
        private void ts_clear_code_Click(object sender, EventArgs e)
        {
            richTextBox1.Clear();
        }

        private void toolStripButton_save_Click(object sender, EventArgs e)
        {
            SaveFileDialog saveFile1 = new SaveFileDialog();

            string last_code_directory = Properties.Settings.Default.last_code_directory;
            if (last_code_directory != null)
            {
                if (Directory.Exists(last_code_directory))
                {
                    saveFile1.InitialDirectory = last_code_directory;
                }
            }

            saveFile1.Title = "Save code to file";
            saveFile1.DefaultExt = "*.txt";
            saveFile1.Filter = "TXT Files|*.txt";

            if (saveFile1.ShowDialog() == System.Windows.Forms.DialogResult.OK &&
               saveFile1.FileName.Length > 0)
            {
                Properties.Settings.Default.last_code_directory = Path.GetDirectoryName(saveFile1.FileName);
                try
                {
                    richTextBox1.SaveFile(saveFile1.FileName, RichTextBoxStreamType.PlainText);
                }
                catch (Exception Ex)
                {
                    _plm.PLInfo.DisplayMinorError("Error saving file : " + Ex.Message);
                }
            }
        }

        private void toolStripButton_load_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFD = new OpenFileDialog();
            string Chosen_File = "";

            string last_code_directory = Properties.Settings.Default.last_code_directory;
            if (last_code_directory != null)
            {
                if (Directory.Exists(last_code_directory))
                {
                    openFD.InitialDirectory = last_code_directory;
                }
            }

            openFD.Title = "Load code from file";
            openFD.FileName = "";
            openFD.Filter = "TXT Files|*.txt";
            if (openFD.ShowDialog() != DialogResult.Cancel)
            {
                Chosen_File = openFD.FileName;
                richTextBox1.LoadFile(Chosen_File, RichTextBoxStreamType.PlainText);
                parse();
            }
            RefreshUI();
        }

        private void toolStripButton_parse_Click(object sender, EventArgs e)
        {
            parse();
            RefreshUI();
        }

        private void toolStripButton_execute_Click(object sender, EventArgs e)
        {
            if (parse() == false) return;

            this.Cursor = Cursors.WaitCursor;

            c_dsl_execute dsle = new c_dsl_execute(_plm);
            dsle.execute_sdl(parser.parsed_list_of_functions,richTextBox1,true);

            _plm.SyncVisible();
            this.Cursor = Cursors.Default;
        }

        private void ts_macro_generate_from_current_config_Click(object sender, EventArgs e)
        {
            macro_recording = true;

            clear_code();
            add_code_define_patch();
            parser.generate_code_for_current_patch(ref _plm, this);
            add_code_end_definition();
            add_code_define_waveset();
            parser.generate_code_from_current_waveset(ref _plm, this);
            add_code_end_definition();

            macro_recording = false;
        }

        private void ts_copy_to_clipboard_Click(object sender, EventArgs e)
        {
            richTextBox1.SelectAll();
            richTextBox1.Copy();
        }

        #endregion

        #region "================== SELECT FUNCTION EVENTS - WAVESET > WAVESET =================
        private void ts_ws_wsf_morphAcrossNoteSectors_Click(object sender, EventArgs e)
        {
            add_code_ws_wsf_morphAcrossNoteSectors(0, 0);
        }

        private void ts_ws_wsf_morphAcrossIntensityLayers_Click(object sender, EventArgs e)
        {
            add_code_ws_wsf_morphAcrossIntensityLayers(0, 0);
        }

        private void ts_ws_wsf_setCurrentIntensityLayer_Click(object sender, EventArgs e)
        {
            add_code_ws_wsf_setCurrentIntensityLayer(0);
        }

        private void ts_ws_wsf_setCurrentNoteSector_Click(object sender, EventArgs e)
        {
            add_code_ws_wsf_setCurrentNoteSector(0);
        }

        #endregion

        #region "================== SELECT FUNCTION EVENTS - WAVESET > NOTE-SECTOR =================
        private void ts_ws_nsf_copyTo_Click(object sender, EventArgs e)
        {
            add_code_ws_nsf_copyTo(0);
        }

        private void ts_ws_nsf_copyToRange_Click(object sender, EventArgs e)
        {
            add_code_ws_nsf_copyToRange(0, 0);
        }

        private void ts_ws_nsf_morphWaveformBlocks_Click(object sender, EventArgs e)
        {
            add_code_ws_nsf_morphWaveformBlocks(0, 0);
        }

        private void ts_ws_nsf_morphWaveforms_Click(object sender, EventArgs e)
        {
            add_code_ws_nsf_morphWaveforms(0, 0, 0);
        }

        #endregion

        #region "================== SELECT FUNCTION EVENTS - WAVESET > INTENSITY-LAYER =================
        private void ts_ws_ilf_copyTo_Click(object sender, EventArgs e)
        {
            add_code_ws_ilf_copyTo(0);
        }

        private void ts_ws_ilf_copyToRange_Click(object sender, EventArgs e)
        {
            add_code_ws_ilf_copyToRange(0, 0);
        }

        private void ts_ws_ilf_shapeTheHarmonics_Click(object sender, EventArgs e)
        {
            add_code_ws_ilf_shapeTheHarmonics(directions.up, 0, dbchanges.minus_3db);
        }

        private void ts_ws_ilf_morphWaveformBlocks_Click(object sender, EventArgs e)
        {
            add_code_ws_ilf_morphWaveformBlocks(0, 0);
        }

        private void ts_ws_ilf_morphWaveforms_Click(object sender, EventArgs e)
        {
            add_code_ws_ilf_morphWaveforms(0, 0, 0);
        }


        #endregion

        #region "================== SELECT FUNCTION EVENTS - WAVESET > WAVEFORM-BLOCK =================
        private void ts_ws_wbf_setCurrentWaveform_Click(object sender, EventArgs e)
        {
            add_code_ws_wbf_setCurrentWaveform(0);
        }

        private void ts_ws_wbf_morphWaveforms_Click(object sender, EventArgs e)
        {
            add_code_ws_wbf_morphWaveforms(0, 0, morph_modes.linear);
        }

        private void ts_ws_wbf_copyTo_Click(object sender, EventArgs e)
        {
            add_code_ws_wbf_copyTo(0, 0);
        }

        private void ts_ws_wbf_copyAcrossNoteSectors_Click(object sender, EventArgs e)
        {
            add_code_ws_wbf_copyAcrossNoteSectors(0, 0);
        }

        private void ts_ws_wbf_copyAcrossIntensityLayers_Click(object sender, EventArgs e)
        {
            add_code_ws_wbf_copyAcrossIntensityLayers(0, 0);
        }

        private void ts_ws_wbf_copyAcrossAllDimensions_Click(object sender, EventArgs e)
        {
            add_code_wbf_copyAcrossAllDimensions();
        }


        #endregion

        #region "================== SELECT FUNCTION EVENTS - WAVESET > WAVEFORMS =================

        private void ts_ws_wf_copyTo_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_copyTo(0);
        }

        private void ts_ws_wf_copyAcrossNoteSectors_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_copyAcrossNoteSectors(0, 0);
        }

        private void ts_ws_wf_copyAcrossIntensityLayers_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_copyAcrossIntensityLayers(0, 0);
        }

        private void ts_ws_wf_copyAcrossAllDimensions_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_copyAcrossAllDimensions();
        }

        private void ts_ws_wf_setName_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setName("Enter name");
        }

        private void ts_ws_wf_setHarmonicLevels_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicLevels(0);
        }

        private void ts_ws_wf_setHarmonicLevelsFromCSV_Click(object sender, EventArgs e)
        {
            List<int> LevelCSV = new List<int> ();
            for (int i = 0; i < max_harmonics; i++)
            { LevelCSV.Add(0); }
            add_code_ws_wf_setHarmonicLevelsFromCSV(LevelCSV);
        }

        private void ts_ws_wf_setHarmonicLevelsFromLevelShapeType_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicLevelsFromLevelShapeType(harmonic_level_shape_types.all_harmonics_zero);
        }

        private void ts_ws_wf_setHarmonicLevelsFromInstrumentType_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicLevelsFromInstrumentType(instrument_types.violin);
        }

        private void ts_ws_wf_setHarmonicLevelsFromWaveformShape_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicLevelsFromWaveformShape(waveform_shapes.square);
        }

        private void ts_ws_wf_setHarmonicLevel_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicLevel(0, 0);
        }

        private void ts_ws_wf_morphHarmonicLevels_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_morphHarmonicLevels(0, 0);
        }

        private void ts_ws_wf_setHarmonicPhases_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicPhases(phases.Normal);
        }

        private void ts_ws_wf_setHarmonicPhasesFromCSV_Click(object sender, EventArgs e)
        {
            List<phases> PhaseCSV = new List<phases>();
            for (int i = 0; i < max_harmonics; i++)
            { PhaseCSV.Add(phases.Normal); }
            add_code_ws_wf_setHarmonicPhasesFromCSV(PhaseCSV);
        }

        private void ts_ws_wf_setHarmonicPhasesFromPhaseShapeType_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicPhasesFromPhaseShapeType(harmonic_phase_shape_types.all_in_phase);
        }

        private void ts_ws_wf_setHarmonicPhasesFromInstrumentType_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicPhasesFromInstrumentType(instrument_types.violin);
        }

        private void ts_ws_wf_setHarmonicPhasesFromWaveformShape_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicPhasesFromWaveformShape(waveform_shapes.square);
        }

        private void ts_ws_wf_setHarmonicPhase_Click(object sender, EventArgs e)
        {
            add_code_ws_wf_setHarmonicPhase(0, phases.Normal);
        }
        #endregion

        #region "================== SELECT FUNCTION EVENTS - PATCH > GENERAL ==================="
        private void ts_p_g_set_portamento_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetPortamento("True","100");
        }

        private void ts_p_g_setScalingSplit_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetScalingSplit("c4");
        }

        private void ts_p_g_setPatchGain_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetPatchGain("100");
        }

        private void ts_p_g_setDetuningTool_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetDetuning("Regular","0");
        }

        private void ts_p_g_setActiveOscillators_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetActiveOscillators("1");
        }

        private void ts_p_g_setDetuningMode_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetDetuningMode("Cents");
        }

        private void ts_p_g_setOscillatorDetuning_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetOscillatorDetuning("1","0.0");
        }

        private void ts_p_g_setLFO_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetLFO("Tremolo","True","Sine","0.0","None","None");
        }

        private void ts_p_g_setEnvelopeGainController_Click(object sender, EventArgs e)
        {
            add_code_p_g_SetEnvelopeGainControllero("Amplitude","0","None");
        }
        #endregion

        #region "================== SELECT FUNCTION EVENTS - PATCH > ADSR ==================="
        private void ts_p_a_configureSection_Click(object sender, EventArgs e)
        {
            add_code_p_a_ConfigureSection("Attack","True","0","0","0","None","OneShot");
        }

        private void ts_p_aconfigureEnvelope_Click(object sender, EventArgs e)
        {
            add_code_p_a_ConfigureEnvelope("Attack","Amplitude","Linear","500","0","0","200","0","0","0","0","0");
        }
        #endregion

        #region "==================== METHODS - GENERAL ======================="
        public PL_UC_SpectralParser()
        {
            InitializeComponent();
            RefreshUI();
        }

        public void initialise(PL_Main plm, parser_modes parser_mode)
        {
            _plm = plm;
            _parser_mode = parser_mode;

            clear_code();
            RefreshUI();
        }

        public void clear_code()
        {
            richTextBox1.Clear();
        }

        private bool parse()
        {
            this.Cursor = Cursors.WaitCursor;
      
            bool parsed_ok;
            tableLayoutPanel1.RowStyles[1].Height = 0;
            RichTextBox rtb = new RichTextBox();

            try
            {
                //Remove any existing formatting (e.g from pasting)
                string existing_text = richTextBox1.Text;
                richTextBox1.Clear();
                richTextBox1.ForeColor = System.Drawing.Color.Gray;
                richTextBox1.Text = existing_text;

                //Load temporary RichTextBox to do the work on (to avoid user seeing the edits)
                rtb.Rtf = richTextBox1.Rtf;

                //Tokenize
                List<c_lexer_token> list_of_tokens = parser.lexer.Tokenize(ref rtb);

                //Parse
                parser.Parse(ref rtb, list_of_tokens);

                //Load results
                richTextBox1.Rtf = rtb.Rtf;

                //Enable execution if got this far
                toolStripButton_execute.Enabled = true;
                parsed_ok = true;

                tableLayoutPanel1.RowStyles[1].Height = 0;

            }
            catch (Exception Ex)
            {
                parsed_ok = false;
                richTextBox1.Rtf = rtb.Rtf;
                tableLayoutPanel1.RowStyles[1].Height = 50;
                toolStripButton_execute.Enabled = false;
                richTextBox1.Select(parser.rtb_found_start_index, 2);
                richTextBox1.SelectionBackColor = System.Drawing.Color.Red;
                richTextBox1.Select(parser.rtb_found_start_index, 0);
                textBox_error.Text = Ex.Message;
            }
            finally
            {
                rtb.Dispose();
                richTextBox1.DeselectAll();
                this.Cursor = Cursors.Default;
            }

            return parsed_ok;
        }

        public void add_code_comment(string comment)
        {
            add_code(1, @"//" + comment + "\r\n");
        }

        public void add_blank_line()
        {
            richTextBox1.AppendText("\r\n");
        }

        private void add_code(int indent, string code_string)
        {
            if (macro_recording == false) return;

            const string tab = "    ";
            string text_to_append = "";
            for (int i = 0; i < indent; i++)
            { text_to_append += tab; }
            text_to_append +=code_string;
            richTextBox1.AppendText(text_to_append);
        }

        public void RefreshUI()
        {
            if (_plm == null)
            {
                ts_macro_generate_from_current_config.Enabled = false;
                ts_macro_start.Enabled = false;
                ts_macro_stop.Enabled = false;
                ts_macro.Enabled = false;
                toolStripButton_execute.Enabled = false;
                ts_ws.Visible = false;
                ts_p.Visible = false;
                ts_f.Visible = false;
            }
            else
            {
                bool got_current_patch_and_waveset = ((_plm.BL.BLPatch.current_patch != null) && (_plm.BL.BLWaveSet.current_waveform_set != null));
                ts_macro.Enabled = got_current_patch_and_waveset;
                ts_macro_start.Enabled = ((_macro_recording == false) && got_current_patch_and_waveset);
                ts_macro_stop.Enabled = _macro_recording;
                ts_macro_generate_from_current_config.Enabled = got_current_patch_and_waveset;
                ts_ws.Visible = ((_plm.BL.BLWaveSet.current_waveform_set != null)&&(_macro_recording));
                ts_p.Visible = ((_plm.BL.BLPatch.current_patch != null) && (_macro_recording));
                ts_f.Visible = ((_plm.BL.BLFilter.current_body_resonance_filter != null)&& (_macro_recording));

                toolStripButton_execute.Enabled = got_current_patch_and_waveset && (parser.parsed_list_of_functions != null) && (parser.parsed_list_of_functions.list_of_functions.Count > 0);
            }

            if (_macro_recording)
            { ts_macro.ForeColor = System.Drawing.Color.Red; }
            else
            { ts_macro.ForeColor = System.Drawing.Color.Black; }
        }

        #endregion

        #region "====================== METHODS - CONVERSION TO STRING =============="
        private string convert_direction(directions d)
        {
            if (d == directions.up)
                { return "Up"; }
            else
            { return "Down"; }
        }

        private string convert_dbchange(dbchanges c)
        {
            switch (c)
            {
                case dbchanges.minus_12db:
                    return "-12dB";
                case dbchanges.minus_24db:
                    return "-23dB";
                case dbchanges.minus_3db:
                    return "-3dB";
                case dbchanges.minus_48db:
                    return "-48dB";
                case dbchanges.minus_6db:
                    return "-6dB";
                case dbchanges.nochange:
                    return "0dB";
                case dbchanges.plus_12db:
                    return "+12dB";
                case dbchanges.plus_24db:
                    return "+24dB";
                case dbchanges.plus_3db:
                    return "+3dB";
                case dbchanges.plus_48db:
                    return "+48dB";
                case dbchanges.plus_6db:
                    return "+6dB";
                default:
                    return "0dB";
            }
        }

        private string convert_harmonic_levelshapetype(harmonic_level_shape_types s)
        {
            switch (s)
            {
                case harmonic_level_shape_types.all_harmonics_max:
                    return "AllMax";
                case harmonic_level_shape_types.all_harmonics_zero:
                    return "AllZero";
                case harmonic_level_shape_types.exponential_all:
                    return "ExponentialAll";
                case harmonic_level_shape_types.exponential_odd:
                    return "ExponentialOdd";
                case harmonic_level_shape_types.linearly_descreasing:
                    return "LinearlyDescreasing";
                default:
                    return "AllMax";
            }
        }

        private string convert_harmonic_phaseshapetype(harmonic_phase_shape_types s)
        {
            switch (s)
            {
                case harmonic_phase_shape_types.all_in_phase:
                    return "AllInPhase";
                case harmonic_phase_shape_types.invert_3711:
                    return "Invert3711";
                case harmonic_phase_shape_types.invert_even:
                    return "InvertEven";
                case harmonic_phase_shape_types.invert_odd:
                    return "InvertOdd";
                default:
                    return "InvertOdd";
            }
        }

        private string convert_instrumenttype(instrument_types i)
        {
            switch (i)
            {
                case instrument_types.cello:
                    return "Cello";
                case instrument_types.clarinet:
                    return "Clarinet";
                case instrument_types.flute:
                    return "Flute";
                case instrument_types.guitar:
                    return "Guitar";
                case instrument_types.muted_guitar:
                    return "MutedGuitar";
                case instrument_types.trumpet:
                    return "Trumpet";
                case instrument_types.violin:
                    return "Violin";
                default:
                    return "Violin";
            }
        }

        private string convert_waveformshape(waveform_shapes s)
        {
            switch (s)
            {
                case waveform_shapes.saw:
                    return "Saw";
                case waveform_shapes.square:
                    return "Square";
                case waveform_shapes.triangle:
                    return "Triangle";
                default:
                    return "Triangle";
            }
        }

        private string convert_phase(phases p)
        {
            switch (p)
            {
                case phases.Normal:
                    return "Normal";
                case phases.Inverted:
                    return "Inverted";
                default:
                    return "Inverted";
            }
        }

        private string convert_morph_mode(morph_modes m)
        {
            if (m == morph_modes.exponential)
            { return "Exponential"; }
            else
            { return "Linear"; }
        }
        #endregion

        #region "==================== METHODS - ADD CODE: WAVESET > WAVESET ================="
        public void add_code_ws_wsf_morphAcrossNoteSectors(int NoteSectorFrom,int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"WaveSet.MorphAcrossNoteSectors(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ");\r\n");
        }
        public void add_code_ws_wsf_morphAcrossIntensityLayers(int IntensityLayerFrom, int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"WaveSet.MorphAcrossIntensityLayers(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
        }
        public void add_code_ws_wsf_setCurrentIntensityLayer(int IntensityLayer)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"WaveSet.SetCurrentIntensityLayer(IntensityLayer:" + IntensityLayer.ToString() +  ");\r\n");
        }
        public void add_code_ws_wsf_setCurrentNoteSector(int NoteSector)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"WaveSet.SetCurrentNoteSector(NoteSector:" + NoteSector.ToString() + ");\r\n");
        }
        #endregion

        #region "==================== METHODS - ADD CODE: WAVESET > NOTE-SECTOR ================="
        public void add_code_ws_nsf_copyTo(int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            string nst = NoteSectorTo.ToString();
            if (nst == "-1")
            {
                add_code(1, "CurrentNoteSector.CopyToRange(NoteSectorFrom:0,NoteSectorTo:" + (max_note_sectors - 1).ToString() + ");\r\n");
            }
            else
            {
                add_code(1, "CurrentNoteSector.CopyTo(NoteSectorTo:" + nst + ");\r\n");
            }
            
        }
        public void add_code_ws_nsf_copyToRange(int NoteSectorFrom, int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentNoteSector.CopyToRange(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ");\r\n");
        }
        public void add_code_ws_nsf_morphWaveformBlocks(int IntensityLayerFrom,int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentNoteSector.MorphWaveformBlocks(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
        }
        public void add_code_ws_nsf_morphWaveforms(int IntensityLayerFrom,int IntensityLayerTo,int Waveform)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentNoteSector.MorphWaveforms(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ",Waveform:" + Waveform.ToString() + ");\r\n");
        }
        #endregion

        #region "==================== METHODS - ADD CODE: WAVESET > INTENSITY-LAYER ================="
        public void add_code_ws_ilf_copyTo(int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            if (IntensityLayerTo == -1)
            {
                add_code(1, "CurrentIntensityLayer.CopyToRange(IntensityLayerFrom:0,IntensityLayerTo:" + (max_intensity_layers - 1).ToString() + ");\r\n");
            }
            else
            {
                add_code(1, "CurrentIntensityLayer.CopyTo(IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
            }
        }
        public void add_code_ws_ilf_copyToRange(int IntensityLayerFrom, int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentIntensityLayer.CopyToRange(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
        }
        public void add_code_ws_ilf_shapeTheHarmonics(directions Direction, int HarmonicIDFrom, dbchanges Slope)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentIntensityLayer.ShapeTheHarmonics(Direction:" + convert_direction(Direction) + ",HarmonicIDFrom:" + HarmonicIDFrom.ToString() + ",Slope:" + convert_dbchange(Slope) + ");\r\n");
        }
        public void add_code_ws_ilf_morphWaveformBlocks(int NoteSectorFrom, int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentIntensityLayer.MorphWaveformBlocks(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ");\r\n");
        }
        public void add_code_ws_ilf_morphWaveforms(int NoteSectorFrom, int NoteSectorTo,int Waveform)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentIntensityLayer.MorphWaveforms(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ",Waveform:" + Waveform.ToString() + ");\r\n");
        }
        #endregion

        #region "==================== METHODS - ADD CODE: WAVESET > WAVEFORM-BLOCK ================="
        public void add_code_ws_wbf_setCurrentWaveform(int Waveform)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.SetCurrentWaveform(Waveform:" + Waveform.ToString()  + ");\r\n");
        }
        public void add_code_ws_wbf_morphWaveforms(int WaveformFrom, int WaveformTo, morph_modes m)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.MorphWaveforms(WaveformFrom:" + WaveformFrom.ToString() + ",WaveformTo:" + WaveformTo.ToString()  + ",MorphType:" + convert_morph_mode(m) + ");\r\n");
        }
        public void add_code_ws_wbf_copyTo(int NoteSector,int IntensityLayer)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.CopyTo(NoteSector:" + NoteSector.ToString() + ",IntensityLayer:" + IntensityLayer.ToString() + ");\r\n");
        }
        public void add_code_ws_wbf_copyAcrossNoteSectors(int NoteSectorFrom, int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.CopyAcrossNoteSectors(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ");\r\n");
        }
        public void add_code_ws_wbf_copyAcrossIntensityLayers(int IntensityLayerFrom, int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.CopyAcrossIntensityLayers(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
        }
        public void add_code_wbf_copyAcrossAllDimensions()
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveformBlock.CopyAcrossAllDimensions();\r\n");
        }
        #endregion

        #region "==================== METHODS - ADD CODE: WAVESET > WAVEFORMS ================="
        public void add_code_ws_wf_copyTo(int WaveformTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.CopyTo(WaveformTo:" + WaveformTo.ToString() +  ");\r\n");
        }

        private void ts_ws_wf_copyToRange_Click(object sender, EventArgs e)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1, "CurrentWaveform.CopyToRange(WaveformFrom:0,WaveformTo:5);\r\n");
        }

        public void add_code_ws_wf_copyToRange(int WaveformFrom, int WaveformTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1, "CurrentWaveform.CopyToRange(WaveformFrom:" + WaveformFrom.ToString() + ",WaveformTo:" + WaveformTo.ToString() + ");\r\n");
        }

        public void add_code_ws_wf_copyAcrossNoteSectors(int NoteSectorFrom, int NoteSectorTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.CopyAcrossNoteSectors(NoteSectorFrom:" + NoteSectorFrom.ToString() + ",NoteSectorTo:" + NoteSectorTo.ToString() + ");\r\n");
        }
        public void add_code_ws_wf_copyAcrossIntensityLayers(int IntensityLayerFrom, int IntensityLayerTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.CopyAcrossIntensityLayers(IntensityLayerFrom:" + IntensityLayerFrom.ToString() + ",IntensityLayerTo:" + IntensityLayerTo.ToString() + ");\r\n");
        }

        public void add_code_ws_wf_copyAcrossAllDimensions()
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.CopyAcrossAllDimensions();\r\n");
        }
        public void add_code_ws_wf_setName(string WaveformName)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetName(WaveformName:'" + WaveformName  + "');\r\n");
        }
        public void add_code_ws_wf_setHarmonicLevels(int Level)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicLevels(Level:" + Level.ToString() + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicLevelsFromCSV(List<int> LevelCSV)
        {
            switch_recording_mode(recording_modes.waveset);
            string csv = "";
            for (int i = 0; i < LevelCSV.Count; i++)
            {
                csv += LevelCSV[i].ToString();
                if (i < (LevelCSV.Count - 1)) { csv += ","; }
            }
            add_code(1,"CurrentWaveform.SetHarmonicLevelsFromCSV(LevelCSV:" + csv + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicLevelsFromLevelShapeType(harmonic_level_shape_types HarmonicLevelShapeType)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicLevelsFromLevelShapeType(HarmonicLevelShapeType:" + convert_harmonic_levelshapetype(HarmonicLevelShapeType) + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicLevelsFromInstrumentType(instrument_types InstrumentType)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicLevelsFromInstrumentType(InstrumentType:" + convert_instrumenttype(InstrumentType) + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicLevelsFromWaveformShape(waveform_shapes WaveformShape)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicLevelsFromWaveformShape(WaveformShape:" + convert_waveformshape(WaveformShape) + ");\r\n");

        }
        public void add_code_ws_wf_setHarmonicLevel(int HarmonicID,int Level)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicLevel(HarmonicID:" + HarmonicID.ToString() + ",Level:" + Level.ToString() + ");\r\n");
        }
        public void add_code_ws_wf_morphHarmonicLevels(int HarmonicIDFrom, int HarmonicIDTo)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.MorphHarmonicLevels(HarmonicIDFrom:" + HarmonicIDFrom.ToString() + ",HarmonicIDTo:"  + HarmonicIDTo.ToString() + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicPhases(phases phase)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicPhases(Phase:" + convert_phase(phase) + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicPhasesFromCSV(List<phases> PhaseCSV)
        {
            switch_recording_mode(recording_modes.waveset);
            string csv = "";
            for (int i = 0; i < PhaseCSV.Count; i++)
            {
                csv += convert_phase(PhaseCSV[i]);
                if (i < (PhaseCSV.Count - 1)) { csv += ","; }
            }
            add_code(1,"CurrentWaveform.SetHarmonicPhasesFromCSV(PhasesCSV:" + csv + ");\r\n");

        }
        public void add_code_ws_wf_setHarmonicPhasesFromPhaseShapeType(harmonic_phase_shape_types HarmonicPhaseShapeType)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicPhasesFromPhaseShapeType(HarmonicPhaseShapeType:" + convert_harmonic_phaseshapetype(HarmonicPhaseShapeType) + ");\r\n");
        }
        public void add_code_ws_wf_setHarmonicPhasesFromInstrumentType(instrument_types InstrumentType)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicPhasesFromInstrumentType(InstrumentType:" + convert_instrumenttype(InstrumentType) + ");\r\n");

        }
        public void add_code_ws_wf_setHarmonicPhasesFromWaveformShape(waveform_shapes WaveformShape)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicPhasesFromWaveformShape(WaveformShape:" + convert_waveformshape(WaveformShape) + ");\r\n");

        }
        public void add_code_ws_wf_setHarmonicPhase(int HarmonicID,phases Phase)
        {
            switch_recording_mode(recording_modes.waveset);
            add_code(1,"CurrentWaveform.SetHarmonicPhase(HarmonicID:" + HarmonicID.ToString() + ",Phase:" + convert_phase(Phase) + ");\r\n");
        }
        #endregion

        #region "=================== METHODS = ADD CODE: PATCH > GENERAL ================"
        public void add_code_p_g_SetPortamento(string enabled,string perc)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetPortamento(Enabled:" + enabled + ",PortamentoAmount:" + perc +"%);\r\n");
        }
        public void add_code_p_g_SetScalingSplit(string note_name)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetScalingSplit(Note:" + note_name + ");\r\n");
        }
        public void add_code_p_g_SetPatchGain(string PatchGain)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetPatchGain(Gain:" + PatchGain + "%);\r\n");
        }
        public void add_code_p_g_SetDetuning(string type,string detuning)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetDetuning(DetuningType:" + type + ",Detuning:"+ detuning +"%);\r\n");
        }
        public void add_code_p_g_SetActiveOscillators(string osc_count)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetActiveOscillators(OscillatorCount:" + osc_count + ");\r\n");
        }
        public void add_code_p_g_SetDetuningMode(string detuning_mode_name)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetDetuningMode(DetuningMode:" + detuning_mode_name + ");\r\n");
        }
        public void add_code_p_g_SetOscillatorDetuning(string oscillator,string detuning)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetOscillatorDetuning(Oscillator:" + oscillator+ ",Detuning:"+ detuning+");\r\n");
        }
        public void add_code_p_g_SetLFO(string LFOType,string Enabled,string WaveType,string Frequency,string FrequencyCC, string DepthCC)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetLFO(LFOType:" + LFOType + ",Enabled:" + Enabled +  ",WaveType:" + WaveType + ",Frequency:" + Frequency + ",FrequencyCC:" + FrequencyCC + ",DepthCC:" + DepthCC + ");\r\n");
        }
        public void add_code_p_g_SetEnvelopeGainControllero(string EnvType,string InitialLevel,String GainCC)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "General.SetEnvelopeGainController(Envelope:" + EnvType+ ",InitialLevel:" + InitialLevel + ",GainCC:" + GainCC+ ");\r\n");
        }

        #endregion

        #region "=================== METHODS = ADD CODE: PATCH > ADSR ================"
        public void add_code_p_a_ConfigureSection(string Section,string Enabled,string Duration,string EndKSU, string EndKSL,string Sample,string SampleMode)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "ADSR.ConfigureSection(Section:"+ Section + ",Enabled:"+ Enabled + ",Duration:"+ Duration + ",EndKSU:"+ EndKSU + "%,EndKSL:"+ EndKSL + "%,Sample:"+ Sample + ",SampleMode:"+ SampleMode +");\r\n");
        }
        public void add_code_p_a_ConfigureEnvelope(string Section, string Envelope, string EnvelopeType, string Target, string TargetKSU, string TargetKSL, string LinearDelta, string LinearDeltaKSU, string LinearDeltaKSL, string ExpMult, string ExpMultKSU, string ExpMultKSL)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "ADSR.ConfigureEnvelope(Section:"+ Section + ",Envelope:"+ Envelope + ",EnvelopeType:"+ EnvelopeType + ",Target:"+ Target + "%,TargetKSU:"+ TargetKSU + "%,TargetKSL:"+ TargetKSL + "%,LinearDelta:"+ LinearDelta + ",LinearDeltaKSU:"+ LinearDeltaKSU + "%,LinearDeltaKSL:"+ LinearDeltaKSL + "%,ExpMult:"+ ExpMult + "%,ExpMultKSU:"+ ExpMultKSU + "%,ExpMultKSL:"+ ExpMultKSL +"%);\r\n");
        }

        public void add_code_p_a_load_preset(string adsr_preset_name,string bulk_value)
        {
            switch_recording_mode(recording_modes.patch);
            add_code(1, "ADSR.LoadPreset(PresetName:" + adsr_preset_name + ",BulkValue:" + bulk_value + ");\r\n");
        }

        public void add_code_p_a_ConfigureEnvelope_DERIVED(int patch_id, int adsr_section_id, int adsr_section_envelope_config_id)
        {
            switch_recording_mode(recording_modes.patch);
            string Section = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_section_name(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string Envelope = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_envelope_name(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string EnvelopeType = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_envelope_type(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string Target = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_target(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string TargetKSU = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_target_KSU(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string TargetKSL = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_target_KSL(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string LinearDelta = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_linear_delta(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string LinearDeltaKSU = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_linear_delta_KSU(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string LinearDeltaKSL = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_linear_delta_KSL(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string ExpMult = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_exp_mult(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string ExpMultKSU = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_exp_mult_KSU(patch_id, adsr_section_id, adsr_section_envelope_config_id);
            string ExpMultKSL = _plm.BL.BLPatch.current_patch.get_string_adsr_envelope_config_exp_mult_KSL(patch_id, adsr_section_id, adsr_section_envelope_config_id);

            add_code_p_a_ConfigureEnvelope(Section, Envelope, EnvelopeType, Target, TargetKSU, TargetKSL, LinearDelta, LinearDeltaKSU, LinearDeltaKSL, ExpMult, ExpMultKSU, ExpMultKSL);

        }
        #endregion

        #region "==================== METHODS - MACRO CONTROL ==========================="
        private void add_code_define_patch()
        {
            add_code(0, "Define.Patch\r\n");
            add_code(0, "{\r\n");
        }

        private void add_code_define_waveset()
        {
            add_code(0, "Define.WaveSet\r\n");
            add_code(0, "{\r\n");
            add_code(1, "WaveSet.SetCurrentNoteSector(NoteSector:" + _plm.BL.BLWaveSet.current_waveform_set.current_note_sector_id.ToString() + ");\r\n");
            add_code(1, "WaveSet.SetCurrentIntensityLayer(IntensityLayer:" + _plm.BL.BLWaveSet.current_waveform_set.current_intensity_layer_id.ToString() + ");\r\n");
            add_code(1, "CurrentWaveformBlock.SetCurrentWaveform(Waveform:" + _plm.BL.BLWaveSet.current_waveform_set.current_block.current_waveform_id + ");\r\n");
        }

        private void add_code_define_filter()
        {
            add_code(0, "Define.BodyResonanceFilter\r\n");
            add_code(0, "{\r\n");
        }

        private void add_code_end_definition()
        {
            add_code(0, "}\r\n");
        }

        private void ts_macro_start_Click(object sender, EventArgs e)
        {
            _macro_recording = true;
            current_recording_mode = recording_modes.none;
            RefreshUI();
        }

        private void switch_recording_mode(recording_modes new_recording_mode)
        {
            if (new_recording_mode != current_recording_mode)
            {
                if (current_recording_mode != recording_modes.none){add_code_end_definition(); }
                switch (new_recording_mode)
                {
                    case recording_modes.none:
                        //Not sure this mode will ever be used
                        break;
                    case recording_modes.patch:
                        add_code_define_patch();
                        break;
                    case recording_modes.filter:
                        add_code_define_filter();
                        break;
                    case recording_modes.waveset:
                        add_code_define_waveset();
                        break;
                }
                current_recording_mode = new_recording_mode;
            }
        }

        private void ts_macro_stop_Click(object sender, EventArgs e)
        {
            add_code_end_definition();
            _macro_recording = false;
            current_recording_mode = recording_modes.none;
            RefreshUI();
        }




        #endregion

        private void toolStrip1_ItemClicked(object sender, ToolStripItemClickedEventArgs e)
        {

        }


    }
}
