/* 
 * File:   Spectral_Interrupts.c
 * Author: Jeremy Leach
 */

#include "Spectral_Interrupts.h"
#include "Spectral_Pins.h"
#include "Spectral_SPI_1.h"
#include "Spectral_Sounds.h"
#include "Spectral_Messaging.h"

/******************************************************************************/
/***** Declarations ****************************************************************/
/******************************************************************************/
static volatile uint16_t trigger_count_10ms;
volatile uint16_t flag_in_play_mode;
extern volatile system_t system;
extern volatile instance_t instance[max_instances]; 

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/

void  Interrupts_Initialize (void)
{
    // ==========INTCON1=========================================================
    INTCON1bits.COVTE = 0; //Catastrophic overflow trap disabled
    INTCON1bits.NSTDIS = 0; //Interrupt nesting NOT disabled

    //============INTCON2=========================================================
    INTCON2bits.DISI = 0; //0 = DISI (Disable Interrupts) instruction is not active

    //============Clear flags before enabling=====================================      
    IFS0bits.SPI1IF = 0; // SPI1 Event Interrupt Flag Status bit
    IFS0bits.SPI1EIF = 0; //SPI1 Error Interrupt Flag Status bit
    IFS0bits.T1IF = 0; //Timer1 
    
    //============Interrupt priority  1 = lowest priority, 7 = highest priority =======
    IPC2bits.SPI1EIP = spi1_int_priority;
    IPC2bits.SPI1IP = spi1_int_priority; //SPI1 reception is highest priority
    IPC1bits.T2IP = timer2_int_priority; //Timer2 used for sample calculation, is mid-priority
    IPC0bits.T1IP = timer1_int_priority; //Timer1 used for envelope calcs and not as important as SPI and won't make sampling over-run
    
    //============ IEC0: INTERRUPT ENABLE CONTROL REGISTER 0 =====================
    IEC0bits.SPI1IE = 1; //SPI1 Event Interrupt Enable bit
    IEC0bits.SPI1EIE = 1;//SPI1 Error Interrupt Enable bit
    IEC0bits.T1IE = 1;   //Timer1 Event Interrupt Enable bit
    T1CONbits.TON = 1; // Start Timer1
    IEC0bits.T2IE = 1;   //Timer2 Event Interrupt Enable bit
    //REMOVED !!! Timer2 interrupt is used as a 'software interrupt' and the flag set manually in the higher-priority ISR of the SPI .T2CONbits.TON = 1; // Start Timer2
 
    //============= Other ========================================================
    trigger_count_10ms = 0;
    //SPI1RxState.status = SPI1_status_idle;
    INTCON2bits.GIE = 1; //Enable interrupts
    flag_in_play_mode = 1; //Start in play mode
}

//NOTE:See detail here: file:///C:/Program%20Files%20(x86)/Microchip/xc16/v1.36/docs/vector_docs/PIC33EP512MC502.html

void __attribute__((__interrupt__, no_auto_psv)) _StackError(void)
{
    //INTCON1bits.STKERR = 0; //Clear the trap flag
    INTCON2bits.GIE = 0;OUT_TEST_PORT_SetHigh(); while(1){}//***TEST
}


void __attribute__((__interrupt__, auto_psv)) _SPI1Interrupt(void)
{    
    /*
     The SPIx Event Interrupt Flag Status bit (SPIxIF) is set when the SPIx Receive Buffer Full Status
    bit (SPIRBF) in the SPIxSTAT register is set. The interrupt flag cannot be cleared by hardware.
    It must be reset in software. 
     */
    uint16_t w;
    if(SPI1STATbits.SPIRBF == 1)
        {    
            w = SPI1BUF; //NOTE: SPIRBF is automatically cleared in hardware when core reads SPIxBUF location, reading SPIxRXB
            while (SPI1STATbits.SPITBF == 1){}; //Wait until Tx buffer empty (should be immediately after Rx buffer full and shift reg is loaded)
            
            if (w == StartPlayModeTP)
                {
                SPI1BUF = 0;
                flag_in_play_mode = 1;
                }
            else if (w == StopPlayModeTP)
                {
                SPI1BUF = 0;
                flag_in_play_mode = 0;
                }
            else if (((w & 0x8000) == 0x8000) && (flag_in_play_mode == 1))
                {
                uint16_t ucc_index = (w & 0b0111000000000000)>>12;
                uint16_t v = w & 0b0000111111111111;
                channel.ucc_current_values[ucc_index] = v << 4;//Make the value 16 bit
                
                if(ucc_index == channel.patch.waveform_set.timbre_controller1_source)
                {   
                    if(instance[0].flag_active == 1)instance[0].flag_timbre_recalc_pending =1;
                    if(instance[1].flag_active == 1)instance[1].flag_timbre_recalc_pending =1;
                    if(instance[2].flag_active == 1)instance[2].flag_timbre_recalc_pending =1;
                }
                
                SPI1BUF = system.sample_delta16;
                system.sample_delta16 = 0;//So that if this interrupt fires again before a new value is calculated then it doesn't create offset.
                system.last_sample = system.current_sample;
                IFS0bits.T2IF = 1;//This fires Timer2(used as software interrupt) immediately after this ISR returns.
                }
            else
                {                
                SPI1BUF = 0;
                WriteWordToSPI1RxBuffer(w);
                }    
        }
   
    IFS0bits.SPI1IF = 0;
}

void __attribute__((__interrupt__, auto_psv)) _SPI1ErrInterrupt(void)
{
    IFS0bits.SPI1EIF = 0;
    INTCON2bits.GIE = 0;OUT_TEST_PORT_SetHigh();while(1){}//***TEST
}

void __attribute__((__interrupt__, no_auto_psv)) _T1Interrupt(void)
{
    IFS0bits.T1IF = 0;
    process_1ms_event();
    
    trigger_count_10ms++;
    if (trigger_count_10ms == 10)
        {
        trigger_count_10ms = 0;
         process_10ms_event();
        }
}

void __attribute__((__interrupt__, no_auto_psv)) _T2Interrupt(void)
{
    IFS0bits.T2IF = 0;
    UpdateSampleValue();
}
