/* 
 * File:   Spectral_I2C.c
 * Author: Jeremy Leach
 */
#include <xc.h>
#include "Spectral_I2C.h"
#include "Spectral_Sound.h"
#include "Spectral_SPI_2.h"
#include "Spectral_Timer1.h"

/******************************************************************************/
/***** Declarations ***********************************************************/
/******************************************************************************/
#define FCY 40000000UL
extern volatile patch_row_t patch;
volatile unsigned char epb[PAGESIZE];
volatile uint16_t epb_index;
volatile uint16_t e_page_index;
volatile uint16_t flag_messaging_EEPROM;

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/

/*********************************************************************
* Overview:		Initialises the I2C(1) peripheral
* Note:			Sets up Master mode, No slew rate control, 100Khz
********************************************************************/
unsigned int I2CInitialize(void)
{	
//ODCBbits.ODCB8 = 1;
//ODCBbits.ODCB9 = 1;
    
    
    /*
     Max clock freq for 24LC256 is 400 kHz
     * 
     * I2CBRG = (Fcy/FSCL) - (FCY/1,111,111)
     * FSCL = 100KHz
     * FCY = 40MHz
     * So I2CBRG = (40,000,000/100,000) - (40,000,000 / 1,111,111) = 400 - 36 = 364
     */
    I2C1BRG = 364;//3640;//364;//100KHz

	I2C1CON = 0x1200; //initialise the I2C peripheral for Master Mode, No Slew Rate Control, and leave the peripheral switched off.
    I2C1CONbits.IPMIEN = 0; //The IPMIEN bit (I2CxCON<11>) should not be set when operating as a master
    I2C1RCV = 0x0000;
	I2C1TRN = 0x0000;
	I2C1CONbits.I2CEN = 1; //0x9200; //enable the peripheral
    
    flag_messaging_EEPROM = 0;
    
    return(0);
}

/*********************************************************************
* Function:        StartI2C()
* Overview:		Generates an I2C Start Condition and returns status
********************************************************************/
unsigned int StartI2C(void)
{
	I2C1CONbits.SEN = 1;		//Generate Start COndition
	while (I2C1CONbits.SEN);	//Wait for Start COndition
	return(I2C1STATbits.S);	    //Optionally return status
}


/*********************************************************************
* Function:        RestartI2C()
* Overview:		Generates a restart condition and optionally returns status
********************************************************************/
unsigned int RestartI2C(void)
{
	I2C1CONbits.RSEN = 1;		//Generate Restart		
	while (I2C1CONbits.RSEN);	//Wait for restart	
	return(I2C1STATbits.S);	    //Optional - return status
}


/*********************************************************************
* Function:        StopI2C()
* Overview:		Generates a bus stop condition and returns status 
********************************************************************/
unsigned int StopI2C(void)
{
	I2C1CONbits.PEN = 1;		//Generate Stop Condition
	while (I2C1CONbits.PEN);	//Wait for Stop
	return(I2C1STATbits.P);	//Optional - return status
}


/*********************************************************************
* Function:        WriteI2C()
* Overview:		Writes a byte out to the bus
********************************************************************/
unsigned int WriteI2C(unsigned char byte)
{
	while (I2C1STATbits.TRSTAT);	//Wait for bus to be idle . NOTE: I un-remmed this
	I2C1TRN = byte;					//Load byte to I2C1 Transmit buffer
	while (I2C1STATbits.TBF);		//wait for data transmission
    return(0);
}

/*********************************************************************
* Function:        IdleI2C()
* Overview:		Waits for bus to become Idle
********************************************************************/
unsigned int IdleI2C(void)
{
	while (I2C1STATbits.TRSTAT);		//Wait for bus Idle
    return(0);
}


/*********************************************************************
* Function:        HDSequentialReadI2C()
* Overview:		Performs a sequential read of length bytes starting at address
*				and places data in array pointed to by *rdptr
********************************************************************/
unsigned int HDSequentialReadI2C(unsigned char ControlByte, unsigned char HighAdd, unsigned char LowAdd, volatile unsigned char *rdptr, unsigned char length)
{
	IdleI2C();						//Ensure Module is Idle
	StartI2C();						//Initiate start condition
	WriteI2C(ControlByte & 0xFE);   //write 1 byte - R/W bit should be 0 for write
	IdleI2C();						//Ensure module is Idle
	WriteI2C(HighAdd);				//Write High word address
	IdleI2C();						//Ensure module is idle
	WriteI2C(LowAdd);				//Write Low word address
	IdleI2C();						//Ensure module is idle
	RestartI2C();					//Generate I2C Restart Condition
	WriteI2C(ControlByte | 0x01);	//Write 1 byte - R/W bit should be 1 for read
	IdleI2C();						//Ensure bus is idle
	getsI2C(rdptr, length);			//Read in multiple bytes
	NotAckI2C();					//Send Not Ack
	StopI2C();						//Send stop condition
	return(0);
}


/*********************************************************************
* Function:        ACKStatus()
* Overview:		Return the Acknowledge status on the bus
********************************************************************/
unsigned int ACKStatus(void)
{
	return (!I2C1STATbits.ACKSTAT);		//Return Ack Status
}


/*********************************************************************
* Function:        NotAckI2C()
* Overview:		Generates a NO Acknowledge on the Bus
********************************************************************/
unsigned int NotAckI2C(void)
{
	I2C1CONbits.ACKDT = 1;			//Set for NotACk
	I2C1CONbits.ACKEN = 1;
	while(I2C1CONbits.ACKEN);		//wait for ACK to complete
	I2C1CONbits.ACKDT = 0;			//Set for NotACk
    return(0);
}


/*********************************************************************
* Function:        AckI2C()
* Overview:		Generates an Acknowledge.
********************************************************************/
unsigned int AckI2C(void)
{
	I2C1CONbits.ACKDT = 0;			//Set for ACk
	I2C1CONbits.ACKEN = 1;
	while(I2C1CONbits.ACKEN);		//wait for ACK to complete
    return(0);
}


/*********************************************************************
* Function:       getsI2C()
* Overview:		read Length number of Bytes into array
********************************************************************/
unsigned int getsI2C(volatile unsigned char *rdptr, unsigned char Length)
{
	while (Length --)
	{
		*rdptr++ = getI2C();		//get a single byte
		
		if(I2C1STATbits.BCL)		//Test for Bus collision
		{
			return(-1);
		}

		if(Length)
		{
			AckI2C();				//Acknowledge until all read
		}
	}
	return(0);
}


/*********************************************************************
* Function:        getI2C()
* Overview:		Read a single byte from Bus
********************************************************************/
unsigned int getI2C(void)
{
	I2C1CONbits.RCEN = 1;			//Enable Master receive
	Nop();
	while(!I2C1STATbits.RBF);		//Wait for receive bufer to be full
	return(I2C1RCV);				//Return data in buffer
}


/*********************************************************************
* Function:        EEAckPolling()
* Overview:		polls the bus for an Acknowledge from device
********************************************************************/
unsigned int EEAckPolling(unsigned char control)
{
	IdleI2C();				//wait for bus Idle
	StartI2C();				//Generate Start condition
	
	if(I2C1STATbits.BCL)
	{
		return(-1);			//Bus collision, return
	}

	else
	{
		if(WriteI2C(control))
		{
			return(-3);		//error return
		}

		IdleI2C();			//wait for bus idle
		if(I2C1STATbits.BCL)
		{
			return(-1);		//error return
		}

		while(ACKStatus())
		{
			RestartI2C();	//generate restart
			if(I2C1STATbits.BCL)
			{
				return(-1);	//error return
			}

			if(WriteI2C(control))
			{
				return(-3);
			}

			IdleI2C();
		}
	}
	StopI2C();				//send stop condition
	if(I2C1STATbits.BCL)
	{
		return(-1);
	}
	return(0);
}


/*********************************************************************
* Function:        putstringI2C()
* Overview:		writes a string of data upto PAGESIZE from array
********************************************************************/
unsigned int putstringI2C(volatile unsigned char *wrptr)
{
	unsigned char x;

	for(x = 0; x < PAGESIZE; x++)		//Transmit Data Until Pagesize
	{	
		if(WriteI2C(*wrptr))			//Write 1 byte
		{
			return(-3);				//Return with Write Collision
		}
		IdleI2C();					//Wait for Idle bus

        /*
         NOTE: The The 24XX256 does not generate any Acknowledge bits if an internal
         programming cycle is in progress
         */
		if(I2C1STATbits.ACKSTAT)  
		{
			return(-2);				//Bus responded with Not ACK
		}
		wrptr++;
	}
	return(0);
}


/*********************************************************************
* Function:        HDPageWriteI2C()
* Overview:		Write a page of data from array pointed to be wrptr
*				starting at address from HighAdd and LowAdd
* Note:			Address must start on a page boundary
********************************************************************/
unsigned int HDPageWriteI2C(unsigned char ControlByte, unsigned char HighAdd, unsigned char LowAdd, volatile unsigned char *wrptr)
{
	IdleI2C();				//wait for bus Idle
	StartI2C();				//Generate Start condition
	WriteI2C(ControlByte & 0xFE); //send controlbyte  - R/W bit should be 0 for write
	IdleI2C();				//wait for bus Idle
	WriteI2C(HighAdd);		//send High Address
	IdleI2C();				//wait for bus Idle
	WriteI2C(LowAdd);		//send Low Address
	IdleI2C();				//wait for bus Idle
	putstringI2C(wrptr);	//Send data
	IdleI2C();				//wait for bus Idle
	StopI2C();				//Generate a stop
    
    //At the end of writing 64 bytes and the stop, the 24LC256 goes into an internal write cycle and DOES NOT acknowledge.
    //Wait for this write cycle to finish by going through a repeated polling routine.
//    while(ACKStatus())
//        {
//        StartI2C();		
//        WriteI2C(EEPROM_ControlByteWrite);
//        }
    delay_for_ms(5);
    
	return(0);
}

//=======SAVE METHODS ==========================================================

void SavePageToEEPROM(void)
{
    uint16_t e_address = e_page_index << PAGE_SIZE_AS_SHIFT;
    unsigned char HighAdd = (unsigned char)(e_address >>8);
    unsigned char LowAdd = (unsigned char)(e_address - (HighAdd <<8));
    HDPageWriteI2C(EEPROM_ControlByteWrite, HighAdd, LowAdd, &epb[0]);
}

void SaveByteToEPB(unsigned char b)
{
epb[epb_index] = b;
epb_index ++;
if (epb_index == PAGESIZE)
    {
    SavePageToEEPROM();
    e_page_index ++;
    epb_index = 0;
    }
}

void SaveWordToEPB(uint16_t w)
{
    unsigned char MSB = (unsigned char)(w >>8);
    unsigned char LSB = (unsigned char)(w - (MSB<<8));
    SaveByteToEPB(LSB);
    SaveByteToEPB(MSB);
}

void save_adsr_section(uint16_t adsr_section_id)
{
    SaveWordToEPB(patch.adsr_section[adsr_section_id].flag_active);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].end_time_ms) ;
    SaveWordToEPB(patch.adsr_section[adsr_section_id].end_time_ms_KU);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].end_time_ms_KL);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].inharmonic_sample_playback_mode_id);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].inharmonic_sample_id);
}

void save_adsr_section_envelope_config(uint16_t adsr_section_id,uint16_t adsr_section_envelope_config_id)
{
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type) ;
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KU);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KL);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_lin_delta);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_exp_multiplier);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KU);
    SaveWordToEPB(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KL);
}

void save_lfo_envelope_config(uint16_t lfo_envelope_config_id)
{
   SaveWordToEPB(patch.lfo_envelope_config[lfo_envelope_config_id].default_wt_inc_q11_5);
   SaveWordToEPB(patch.lfo_envelope_config[lfo_envelope_config_id].depth_cc_source);
   SaveWordToEPB(patch.lfo_envelope_config[lfo_envelope_config_id].enabled);
   SaveWordToEPB(patch.lfo_envelope_config[lfo_envelope_config_id].lfo_wave_type_id);
   SaveWordToEPB(patch.lfo_envelope_config[lfo_envelope_config_id].freq_cc_source);
}

void save_envelope_control(uint16_t envelope_control_id)
{
    SaveWordToEPB(patch.env_gain_CC[envelope_control_id]);
}

void save_initial_depth_value(uint16_t envelope_control_id)
{
    SaveWordToEPB(patch.env_initial_depth_value[envelope_control_id]);
}

void save_body_resonance_filter_band(void)
{
    uint16_t j;
    for (j = 0; j < max_body_resonance_filter_bands; j++)
    {
        SaveWordToEPB(patch.body_resonance_bands[j].level);
        SaveWordToEPB(patch.body_resonance_bands[j].slope);
    }
}

void save_waveform_harmonic(uint16_t n,uint16_t i,uint16_t waveform_id)
{
    uint16_t h;
    for(h = 0;h < max_harmonics;h++)
        {
          SaveByteToEPB((unsigned char)(patch.waveform_set.waveform_harmonic[n][i][waveform_id][h].level & 0xFF)); //Note Byte !!!
        }
}

void save_waveform_set()
{
    SaveWordToEPB(patch.waveform_set.timbre_controller1_source);
    SaveWordToEPB(patch.waveform_set.timbre_controller2_source);
    SaveWordToEPB(patch.waveform_set.timbre_mode);
}

void SavePatchToEEPROM(uint16_t module_patch_id)
{
    flag_messaging_EEPROM = 1;
    OUT_MIXALERT_PORT_SetHigh();
    uint16_t e_address;
    uint16_t adsr_section_id;
    uint16_t adsr_section_envelope_config_id;
    uint16_t lfo_envelope_config_id;
    uint16_t envelope_control_id;
    uint16_t env_initial_depth_value;
    uint16_t n;
    uint16_t i;
    uint16_t waveform_id;
 
    epb_index = 0;
    e_address = (uint16_t)(__builtin_muluu(module_patch_id,(uint16_t)(PAGES_PER_PATCH << PAGE_SIZE_AS_SHIFT)));
    e_page_index = e_address >> PAGE_SIZE_AS_SHIFT;

    SaveWordToEPB(patch.sustain_enabled);
    SaveWordToEPB(patch.active_layers);
    SaveWordToEPB(patch.degree_of_random_detuning);
    SaveWordToEPB(patch.degree_of_random_phase);
    SaveWordToEPB(patch.degree_of_regular_detuning);
    SaveWordToEPB(patch.pitch_bend_enabled);
    SaveWordToEPB(patch.portamento_enabled);
    SaveWordToEPB(patch.portamento_rate);
    SaveWordToEPB(patch.detuning_mode_id);
    SaveWordToEPB(patch.layer_config[0].st_inc_q11_5_detune);
    SaveWordToEPB(patch.layer_config[1].st_inc_q11_5_detune);
    SaveWordToEPB(patch.layer_config[2].st_inc_q11_5_detune);
    SaveWordToEPB(patch.key_scale_split_note_id);
     
    for (adsr_section_id = 0; adsr_section_id < max_adsr_sections;adsr_section_id++)
        {
        save_adsr_section(adsr_section_id);
        for (adsr_section_envelope_config_id = 0; adsr_section_envelope_config_id < max_envelopes;adsr_section_envelope_config_id++)
            {
            save_adsr_section_envelope_config(adsr_section_id,adsr_section_envelope_config_id);
            }
        }

    for (lfo_envelope_config_id = 0; lfo_envelope_config_id < max_lfos;lfo_envelope_config_id ++)
        {
        save_lfo_envelope_config(lfo_envelope_config_id);
        }
    
    for (envelope_control_id = 0; envelope_control_id < max_envelopes ; envelope_control_id++)
        {
        save_envelope_control(envelope_control_id);
        }
    
    for (env_initial_depth_value = 0; env_initial_depth_value < max_envelopes;env_initial_depth_value++ )
        {
        save_initial_depth_value(env_initial_depth_value);
        }

    save_body_resonance_filter_band();

    for (n = 0;n < max_note_sectors;n++)
        {
            for (i = 0;i < max_intensity_layers;i++)
                {
                for (waveform_id = 0;waveform_id < max_waveforms;waveform_id++)
                    {
                    save_waveform_harmonic(n,i,waveform_id);
                    }
                }
        }
    
    save_waveform_set();
    
    SavePageToEEPROM();//Force current page to be saved (which is likely to be incomplete)
    OUT_MIXALERT_PORT_SetLow();
    flag_messaging_EEPROM = 0;
}

//========LOAD METHODS =========================================================

void LoadPageFromEEPROM(void)
{
uint16_t e_address = e_page_index << PAGE_SIZE_AS_SHIFT;
unsigned char HighAdd = (unsigned char)(e_address >>8);
unsigned char LowAdd = (unsigned char)(e_address - (HighAdd <<8)); 
HDSequentialReadI2C(EEPROM_ControlByteRead, HighAdd, LowAdd, &epb[0], PAGESIZE);
}

unsigned char LoadByteFromEPB(void)
{
    unsigned char b = epb[epb_index];
    epb_index ++;
    if (epb_index == PAGESIZE)
        {
        e_page_index ++;
        LoadPageFromEEPROM();
        epb_index = 0;
        }
    return b;
}

uint16_t LoadWordFromEPB(void)
{
    unsigned char LSB = LoadByteFromEPB();
    unsigned char MSB = LoadByteFromEPB();
    return (uint16_t)((MSB << 8) + LSB);
}

void load_adsr_section(uint16_t adsr_section_id)
{
    patch.adsr_section[adsr_section_id].flag_active = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].end_time_ms = LoadWordFromEPB() ;
    patch.adsr_section[adsr_section_id].end_time_ms_KU = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].end_time_ms_KL = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].inharmonic_sample_playback_mode_id = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].inharmonic_sample_id = LoadWordFromEPB();
}

void load_adsr_section_envelope_config(uint16_t adsr_section_id,uint16_t adsr_section_envelope_config_id)
{
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type = LoadWordFromEPB() ;
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KU = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KL = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_lin_delta = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_exp_multiplier = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KU = LoadWordFromEPB();
    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KL = LoadWordFromEPB();
}

void load_lfo_envelope_config(uint16_t lfo_envelope_config_id)
{
   patch.lfo_envelope_config[lfo_envelope_config_id].default_wt_inc_q11_5 = LoadWordFromEPB();
   patch.lfo_envelope_config[lfo_envelope_config_id].depth_cc_source = LoadWordFromEPB();
   patch.lfo_envelope_config[lfo_envelope_config_id].enabled = LoadWordFromEPB();
   patch.lfo_envelope_config[lfo_envelope_config_id].lfo_wave_type_id = LoadWordFromEPB();
   patch.lfo_envelope_config[lfo_envelope_config_id].freq_cc_source = LoadWordFromEPB();
}

void load_envelope_control(uint16_t envelope_control_id)
{
    patch.env_gain_CC[envelope_control_id] = LoadWordFromEPB();
}

void load_initial_depth_value(uint16_t initial_depth_value)
{
    patch.env_initial_depth_value[initial_depth_value] = LoadWordFromEPB();
}

void load_body_resonance_filter_band(void)
{
    uint16_t j;
    for (j = 0; j < max_body_resonance_filter_bands; j++)
    {
        patch.body_resonance_bands[j].level = LoadWordFromEPB();
        patch.body_resonance_bands[j].slope = LoadWordFromEPB();
    }
}

void load_waveform_harmonic(uint16_t n,uint16_t i,uint16_t waveform_id)
{
    uint16_t h;
    for(h = 0;h < max_harmonics;h++)
        {
          patch.waveform_set.waveform_harmonic[n][i][waveform_id][h].level  = LoadByteFromEPB(); //NOTE: Byte !!
        }
}

void load_waveform_set()
{
    patch.waveform_set.timbre_controller1_source = LoadWordFromEPB();
    patch.waveform_set.timbre_controller2_source = LoadWordFromEPB();
    patch.waveform_set.timbre_mode = LoadWordFromEPB();
}


void LoadPatchFromEEPROM(uint16_t module_patch_id)
{
    flag_messaging_EEPROM = 1;
    OUT_MIXALERT_PORT_SetHigh();
    uint16_t e_address;
    uint16_t adsr_section_id;
    uint16_t adsr_section_envelope_config_id;
    uint16_t lfo_envelope_config_id;
    uint16_t envelope_control_id;
    uint16_t env_initial_depth_value;
    uint16_t n;
    uint16_t i;
    uint16_t waveform_id;
 
    epb_index = 0;
    e_address = (uint16_t)(__builtin_muluu(module_patch_id,(uint16_t)(PAGES_PER_PATCH << PAGE_SIZE_AS_SHIFT)));
    e_page_index = e_address >> PAGE_SIZE_AS_SHIFT;
    
    LoadPageFromEEPROM();//Force load of initial page
    patch.sustain_enabled = LoadWordFromEPB();
    patch.active_layers = LoadWordFromEPB();
    patch.degree_of_random_detuning = LoadWordFromEPB();
    patch.degree_of_random_phase = LoadWordFromEPB();
    patch.degree_of_regular_detuning = LoadWordFromEPB();
    patch.pitch_bend_enabled = LoadWordFromEPB();
    patch.portamento_enabled = LoadWordFromEPB();
    patch.portamento_rate = LoadWordFromEPB();
    patch.detuning_mode_id = LoadWordFromEPB();
    patch.layer_config[0].st_inc_q11_5_detune = LoadWordFromEPB();
    patch.layer_config[1].st_inc_q11_5_detune = LoadWordFromEPB();
    patch.layer_config[2].st_inc_q11_5_detune = LoadWordFromEPB();
    patch.key_scale_split_note_id = LoadWordFromEPB();
     
    for (adsr_section_id = 0; adsr_section_id < max_adsr_sections;adsr_section_id++)
        {
        load_adsr_section(adsr_section_id);
        for (adsr_section_envelope_config_id = 0; adsr_section_envelope_config_id < max_envelopes;adsr_section_envelope_config_id++)
            {
            load_adsr_section_envelope_config(adsr_section_id,adsr_section_envelope_config_id);
            }
        }

    for (lfo_envelope_config_id = 0; lfo_envelope_config_id < max_lfos;lfo_envelope_config_id ++)
        {
        load_lfo_envelope_config(lfo_envelope_config_id);
        }
    
    for (envelope_control_id = 0; envelope_control_id < max_envelopes ; envelope_control_id++)
        {
        load_envelope_control(envelope_control_id);
        }
    
    for (env_initial_depth_value = 0; env_initial_depth_value < max_envelopes ; env_initial_depth_value++)
        {
        load_initial_depth_value(env_initial_depth_value);
        }

    load_body_resonance_filter_band();

    for (n = 0;n < max_note_sectors;n++)
        {
            for (i = 0;i < max_intensity_layers;i++)
                {
                for (waveform_id = 0;waveform_id < max_waveforms;waveform_id++)
                    {
                    load_waveform_harmonic(n,i,waveform_id);
                    }
                }
        }
    
    load_waveform_set();
    
    OUT_MIXALERT_PORT_SetLow();
    flag_messaging_EEPROM = 0;
}
