/* 
 * File:   Spectral_Sound.c
 * Author: Jeremy Leach
 */

#include "Spectral_Sound.h"

/******************************************************************************/
/***** Declarations ***********************************************************/
/******************************************************************************/
volatile agc_t agc;
volatile uint16_t global_gain;
static volatile haas_fifo_t hb;
static volatile uint16_t haas_delay_buffer_to_use;
/*
 Sine Table increment lookup for MIDI notes 21	(A0	freq 27.5Hz) and upwards. 88 note keyboard.
 The word values are Q11.5 format. So the value in the table =  increment x 32.
 */ 
static const uint16_t NoteSTIncLookup[88]={
41,	43,	46,	49,	52,	55,	58,	61,
65,	69,	73,	77,	82,	87,	92,	97,
103,	109,	116,	123,	130,	138,	146,	155,
164,	174,	184,	195,	206,	219,	232,	245,
260,	276,	292,	309,	328,	347,	368,	390,
413,	437,	463,	491,	520,	551,	584,	619,
655,	694,	736,	779,	826,	875,	927,	982,
1040,	1102,	1168,	1237,	1311,	1389,	1471,	1559,
1651,	1749,	1853,	1964,	2080,	2204,	2335,	2474,
2621,	2777,	2942,	3117,	3302,	3499,	3707,	3927,
4161,	4408,	4670,	4948,	5242,	5554,	5884,	6234
};

static const uint16_t NoteHzLookup[88]={
28,	29,	31,	33,	35,	37,	39,	41,
44,	46,	49,	52,	55,	58,	62,	65,
69,	73,	78,	82,	87,	92,	98,	104,
110,	117,	123,	131,	139,	147,	156,	165,
175,	185,	196,	208,	220,	233,	247,	262,
277,	294,	311,	330,	349,	370,	392,	415,
440,	466,	494,	523,	554,	587,	622,	659,
698,	740,	784,	831,	880,	932,	988,	1047,
1109,	1175,	1245,	1319,	1397,	1480,	1568,	1661,
1760,	1865,	1976,	2093,	2217,	2349,	2489,	2637,
2794,	2960,	3136,	3322,	3520,	3729,	3951,	4186
};

static const uint16_t AGCNumeratorLookup[max_agc_lookup_values]={61673,58240,55170,52407,49907,47635,45561,43660,41911,40297,38803,37415,36124,34918,33791,32733,31741,30806,29925,29093,28306,27560,26853,26181,25542,24933,24353,23799,23270,22764,22279,21815,21369,20941,20531,20135,19755,19389,19036,18696,18368,18051,17745,17449,17163,16886,16618,16358,16106,15862,15625,15395,15172,14955,14744,14540,14340,14146,13958,13774,13595,13421,13251,13085,12923,12766,12612,12462,12315,12172,12032,11895,11761,11630,11503,11378,11255,11135,11018,10903,10791,10681,10573,10467,10363,10262,10162,10064,9968,9874,9782,9691,9602,9515,9429,9345,9262,9181,9101,9023,8946,8870,8795,8722,8650,8579,8509,8440,8373,8306,8241,8177,8113,8051,7989,7929,7869,7810,7753,7695,7639,7584,7529,7476,7423,7370,7319,7268,7218,7168,7119,7071,7024,6977,6931,6885,6840,6796,6752,6709,6666,6624,6582,6541,6500,6460,6420,6381,6343,6304,6267,6229,6192,6156,6120,6084,6049,6014,5980,5946};
extern volatile int32_t tone_proc_sample_sum;
/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/
void set_haas_delay_buffer_to_use(uint16_t value)
{
    haas_delay_buffer_to_use = value;
    hb.index = 0;
}

void SoundInitialize(void)
{  
    global_gain = 58650;//230 * 255.  Limit so we don't overload op amp
    uint16_t i;
    
    haas_delay_buffer_to_use = HAASFifoBuffer_size;
    
    //============== Clear HAAS buffer ========================
    for (i = 0;i < HAASFifoBuffer_size; i++){hb.buf[i] = 0;}
    hb.index = 0;
    
    //============== Clear AGC buffer =========================
    for(i = 0;i < AgcRingBuffer_size;i++){agc.buf[i] = 0;}
    agc.window_index = 0;
    agc.buffer_index = 0;
    agc.env_target = 65535;
    agc.env_value = 65535;
    agc.env_lin_delta = 0;
    agc.window_peak = 0;
    agc.old_simplistic_env_target = 0;
    agc.outputL = 0;
    agc.outputR = 0;
    
    //============== Initialize DAC1 ==========================

    //N1 = 2 (set in Main through CLKCIVbits.PLLPRE, see explanation of weird numbering in main)
    //PLLDIV = 20 (Again see weird numbering in main)
    //PLLPOST = 2 (again see main)
    //Giving FOSC = 80MHz
    
    ACLKCONbits.SELACLK = 0 ;   // 0 = PLL output (here is Fvco = 160MHz) provides the source clock for Auxiliary Clock Divider. 
    ACLKCONbits.APSTSCLR = 0b111; //Auxiliary clock output divider. 111= Divide by 1,   011 = Divide by 16. 
    
    DAC1STATbits.ROEN = 1;      // Right Channel DAC Output Enabled 
    DAC1STATbits.LOEN = 1;      // Left Channel DAC Output Enabled 
    DAC1STATbits.RITYPE = 0;    // Right Channel Interrupt if FIFO is not Full 
    DAC1STATbits.LITYPE = 0;    // Left Channel Interrupt if FIFO is not Full 
    DAC1CONbits.AMPON = 0;      // Amplifier Disabled During Sleep and Idle Modes 
    DAC1CONbits.DACSIDL = 0;    // Continue operation in idle mode
    
    /*
    DAC clock must be equal to sampling rate x 256 
    160MHz/14/256 =44,642.857 Hz sampling rate. 160MHz/15/256 =41,666.66 Hz sampling rate. 
     */ 
    DAC1CONbits.DACFDIV = 14;//13;//MEANS Divide value + 1 (0 means 1 .... 127 means 128))
    
    DAC1CONbits.FORM = 1;       // Data Format is signed 
    DAC1DFLT = 0;               // Default value set to Midpoint 
    IFS4bits.DAC1RIF = 0;       // Clear Right Channel Interrupt Flag 
    IFS4bits.DAC1LIF = 0;       // Clear Left Channel Interrupt Flag 
    IEC4bits.DAC1RIE = 1;       // Right Channel Interrupt Enabled 
    IEC4bits.DAC1LIE = 1;       // Left Channel Interrupt Enabled 
    DAC1CONbits.DACEN = 1;      // DAC1 Module Enabled 
    //Need this to kick the DAC into life
    DAC1RDAT = 0;
    DAC1LDAT = 0;   
}

void UpdateAGC(void)
{
    const uint32_t threshold = 32700;

    int32_t new_sample;
    uint32_t absSample;
    uint32_t temp32;
    
    uint16_t MSW;
    uint16_t LSW;
    int16_t result;
    uint16_t simplistic_env_target;
    int16_t temp_result;
   
    //======== Calculate output, by applying gain to buffered value. 
    //The gain has been previously calculated by the agc to ensure the result fits in int16.      
    if(agc.buf[agc.buffer_index] > 0)
        {temp32 = agc.buf[agc.buffer_index];}
    else
        {temp32 = (uint32_t)((int32_t)(0 - agc.buf[agc.buffer_index]));}
    MSW = (uint16_t)((temp32 >> 16) & 0xFFFFUL);
    LSW = (uint16_t)((temp32 & 0xFFFFUL));
    result = (int16_t)
                    (   (uint16_t)__builtin_muluu(MSW,agc.env_value)   //This is zero for signals under 65535
                        + 
                        (uint16_t)(__builtin_muluu(LSW,agc.env_value)>>16)
                    );
    
    if(agc.buf[agc.buffer_index] < 0){result = (int16_t)(0 - result);}
    
    //Set outputL
    temp_result = (int16_t)(__builtin_mulsu(result,global_gain)>>16); 
    
    IPC19bits.DAC1LIP = 0;
    IPC19bits.DAC1RIP = 0;
    agc.outputL = temp_result; 
    agc.outputR = hb.buf[hb.index]; //OutputR is a delayed exact copy of OutputL, and relies on the HAAS effect to create stereo-widening.
    IPC19bits.DAC1LIP = 7;
    IPC19bits.DAC1RIP = 7;
    
    hb.buf[hb.index] = agc.outputL;
    hb.index ++; if (hb.index == haas_delay_buffer_to_use){hb.index = 0;}

    //======== Store the new sample in the buffer
    new_sample = (int32_t)(tone_proc_sample_sum <<2); //NOTE multiply by 4 to increase volume lost through the LPF noise reduction pre-emphasis !
    agc.buf[agc.buffer_index] = new_sample; 
    
    //======== Get absolute sample values
    if (new_sample > 0){absSample = (uint32_t)(new_sample);} else {absSample = (uint32_t)((int32_t)(0 - new_sample));}
    
    //======== Update window peak based on the absolute values of the new sample.
    if (absSample > agc.window_peak){ agc.window_peak = absSample;}
    
    //======== Increment buffer index
    agc.buffer_index++; if (agc.buffer_index == AgcRingBuffer_size){agc.buffer_index = 0;}

    //======== Increment window index
    agc.window_index++;
    if (agc.window_index == AgcWindow_size)
        {
        //==== Set new target ===================
        if (agc.window_peak > threshold)  
            {
            simplistic_env_target =  AGCNumeratorLookup[(uint16_t)((uint32_t)(agc.window_peak - threshold)>>11)];
            }
        else
            {
            simplistic_env_target = 65535;
            }
    
        if (simplistic_env_target > agc.old_simplistic_env_target)
            {agc.env_target = agc.old_simplistic_env_target;}
        else
            {agc.env_target = simplistic_env_target;}
        
        if (agc.env_target > agc.env_value) //RELEASE
            {
            //Want a relatively slow release so shift 
            agc.env_lin_delta = (agc.env_target - agc.env_value)>>8;//14
            if (agc.env_lin_delta == 0){agc.env_lin_delta = 1;}
            }
        else if (agc.env_target < agc.env_value) //ATTACK
            {
            //Must have attack that guarantees overload is avoided
            agc.env_lin_delta = ((agc.env_value - agc.env_target)>>AgcWindow_size_as_shift);
            //Initially subtract the remainder
            agc.env_value -=((agc.env_value - agc.env_target) - (agc.env_lin_delta << AgcWindow_size_as_shift));
            }
        else
            {agc.env_lin_delta = 0;}        
       
        agc.old_simplistic_env_target = simplistic_env_target;
        agc.window_index = 0;
        
        agc.window_peak = 0;
        }

    //======== Update gain envelope
    if (agc.env_value < agc.env_target)
        {
        if (agc.env_value > (agc.env_target - agc.env_lin_delta ))
            {agc.env_value = agc.env_target;}
        else
            {agc.env_value += agc.env_lin_delta;}
        }
    else if (agc.env_value > agc.env_target)
        {
        if (agc.env_value < (agc.env_target + agc.env_lin_delta))
            {agc.env_value = agc.env_target;}
        else
            {agc.env_value -= agc.env_lin_delta;}
        }         
}







