/* 
 * File:   Spectral_SPI_2.c
 * Author: Jeremy Leach
 */

#include "Spectral_SPI_2.h"  
#include "Spectral_Sound.h"
#include "Spectral_I2C.h"
#include "Spectral_Interrupts.h"

/******************************************************************************/
/***** Definitions ************************************************************/
/******************************************************************************/ 
volatile SPI2TxState_t SPI2TxState;
volatile int32_t tone_proc_sample_sum;
volatile uint16_t zero_delta_count;
volatile patch_row_t patch;
static volatile uint16_t payload[40];
volatile uint16_t alert_delay_ms;
volatile int16_t patch_id_to_save;
volatile int16_t patch_id_to_load;
volatile int16_t performance_id_to_select;
static volatile performance_t performance[performances_per_patchset];
extern volatile uint16_t global_gain;
extern volatile fifo_buffer8_t SPI1RxBuffer;
extern volatile uint16_t flag_do_non_sample_stuff;
volatile uint16_t ucc_index;
volatile uint16_t ucc_value_LSB;
volatile uint16_t ucc_value_MSB;
volatile uint16_t tp_control_value;

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/ 

void SPI2Initialize(void)
{
    alert_delay_ms = 0;
    tone_proc_sample_sum = 0;
    patch_id_to_save = -1;
    patch_id_to_load = -1;
    performance_id_to_select = -1;
    
    //Initialise SPI2 slave selects to high
    OUT_SPI2_SS1_PORT_SetHigh();
    OUT_SPI2_SS2_PORT_SetHigh();
    OUT_SPI2_SS3_PORT_SetHigh();
    OUT_SPI2_SS4_PORT_SetHigh();
    OUT_SPI2_SS5_PORT_SetHigh();
    OUT_SPI2_SS6_PORT_SetHigh();
      
    //Re-map SPI2 to the pins being used (as a MASTER)
    RPINR22bits.SDI2R   = 0;        //SPI2 Data input. Tied to RB0 (RP0).
    RPOR0bits.RP1R      = 0b01011;  //SPI2 Clock output. Tied to RB1 (RP1)
    RPOR1bits.RP2R      = 0b01010;  //SDO2. Tied to RB2
    
    //Initially disable interrupt
    IFS2bits.SPI2IF     = 0;        //Clear the interrupt flag
    IFS2bits.SPI2EIF    = 0;        //Clear the interrupt flag
    IEC2bits.SPI2IE     = 0;        //Disable SPI2 Event Interrupt
    IEC2bits.SPI2EIE    = 0;        //Disable SPI2 Error event interrupt
    
    //====SPI2CON1. Configure for Master, 16 bit. 
    SPI2CON1bits.DISSCK = 0;    //Disable SCKx pin bit (SPI Master mode only). 0 = Internal SPI clock is enabled    
    SPI2CON1bits.DISSDO = 0;    //Disable SDOx pin bit. 0 = SDOx pin is controlled by the module    
    SPI2CON1bits.MODE16 = 1;    //Word/Byte Communication Select bit. 1 = Communication is word-wide (16 bits).
    SPI2CON1bits.SMP    = 0;    //SPIx Data Input Sample Phase bit. Master mode:0 = Input data sampled at middle of data output time. 
    SPI2CON1bits.CKE    = 0;    //SPIx Clock Edge Select bit. 0 = Serial output data changes on transition from Idle clock state to active clock state.
    SPI2CON1bits.CKP    = 0;    //Clock Polarity Select bit. 0 = Idle state for clock is a low level; active state is a high level
    SPI2CON1bits.MSTEN  = 1;    //Master Mode Enable bit. 1 = Master mode. 0 = Slave mode    
    
    //Select Primary and secondary prescalers to give freq of 10Mhz 
    SPI2CON1bits.PPRE   = 0b11; //Primary Pre-scale bits (00 = 64:1,01 = 16:1,10=4:1,11 = 1:1)
    SPI2CON1bits.SPRE   = 0b011;//0b010;//0b001;//0b010;//0b011;//0b100;//Secondary Pre-scale bits (Master mode) (111 = 1:1, 110 = 2:1,101 = 3:1, 100 = 4:1,011 = 5:1, 010 = 6:1,001 = 7:1,000 = 8:1)
    
    SPI2CON1bits.SSEN   = 0;    //Slave Select Enable bit (Slave mode).0 = SSx pin is not used by the module.
    SPI2CON2bits.FRMEN  = 0;    //Framed SPIx Support bit 1= Framed SPIx support enabled
    
    //Set status     
    SPI2TxState.status = SPI2_status_recipient;
    SPI2STATbits.SPIEN  = 1;    //SPIx Enable bit.  1 = Enables module and configures SCKx, SDOx, SDIx and SSx as serial port pins

    tp_control_value = 0x8000;
}

void SPI2TxSetSS(uint8_t processor_id)
{
       switch (processor_id)
    {
        case no_recipient:
            break;
        case processor0:
            OUT_SPI2_SS1_PORT_SetHigh();
            break;
        case processor1:
            OUT_SPI2_SS2_PORT_SetHigh();
            break;
        case processor2:
            OUT_SPI2_SS3_PORT_SetHigh();
            break;
        case processor3:
            OUT_SPI2_SS4_PORT_SetHigh();
            break;
        case processor4:
            OUT_SPI2_SS5_PORT_SetHigh();
            break;
        case processor5:
            OUT_SPI2_SS6_PORT_SetHigh();
            break;
    }
}

void SPI2TxClearSS(uint8_t processor_id)
{
       switch (processor_id)
    {
        case no_recipient:
            break;
        case processor0:
            OUT_SPI2_SS1_PORT_SetLow();
            break;
        case processor1:
            OUT_SPI2_SS2_PORT_SetLow();
            break;
        case processor2:
            OUT_SPI2_SS3_PORT_SetLow();
            break;
        case processor3:
            OUT_SPI2_SS4_PORT_SetLow();
            break;
        case processor4:
            OUT_SPI2_SS5_PORT_SetLow();
            break;
        case processor5:
            OUT_SPI2_SS6_PORT_SetLow();
            break;
    }
}

void ChangePlayMode(uint16_t play_mode)
{
    uint16_t p;
    int16_t w;
    for (p=0;p < max_processors; p++)
        {
        SPI2TxClearSS(p);
        SPI2BUF = play_mode; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
        SPI2TxSetSS(p);
        } 
}

void SendMessageToToneProcs(uint16_t message_type_id,uint16_t payload_length,uint16_t recipient_id)
{
    uint16_t p;
    int16_t w;
    uint16_t i;
    
    if (recipient_id == all_processors)
        {
        for (p=0;p < max_processors; p++)
            {
            SPI2TxClearSS(p);

            SPI2BUF = message_type_id; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
            for (i=0;i < payload_length;i++)
            {
            SPI2BUF = payload[i]; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
            }

            SPI2TxSetSS(p);
            }    
        }
    else
        {
        SPI2TxClearSS(recipient_id);

        SPI2BUF = message_type_id; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
        for (i=0;i < payload_length;i++)
        {
        SPI2BUF = payload[i]; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
        }

        SPI2TxSetSS(recipient_id);
        }
}

void send_adsr_section(uint16_t adsr_section_id,uint16_t recipient_id)
{
    payload[0] = adsr_section_id;
    payload[1] = patch.adsr_section[adsr_section_id].flag_active;
    payload[2] = patch.adsr_section[adsr_section_id].end_time_ms ;
    payload[3] = patch.adsr_section[adsr_section_id].end_time_ms_KU ;
    payload[4] = patch.adsr_section[adsr_section_id].end_time_ms_KL ;
    payload[5] = patch.adsr_section[adsr_section_id].inharmonic_sample_playback_mode_id;
    payload[6] = patch.adsr_section[adsr_section_id].inharmonic_sample_id;

    SendMessageToToneProcs( msg_type_adsr_section,7,recipient_id); 
}

void send_adsr_section_envelope_config(uint16_t adsr_section_id,uint16_t adsr_section_envelope_config_id,uint16_t recipient_id)
{
    payload[0] = adsr_section_id;
    payload[1] = adsr_section_envelope_config_id;
    payload[2] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type ;
    payload[3] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target;
    payload[4] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KU;
    payload[5] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KL;
    payload[6] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_lin_delta;
    payload[7] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_exp_multiplier;
    payload[8] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KU;
    payload[9] = patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KL;

    SendMessageToToneProcs( msg_type_adsr_section_envelope_config,10,recipient_id);
}

void send_lfo_envelope_config(uint16_t lfo_envelope_config_id,uint16_t recipient_id)
{
   payload[0] = lfo_envelope_config_id;
   payload[1] = patch.lfo_envelope_config[lfo_envelope_config_id].default_wt_inc_q11_5;
   payload[2] = patch.lfo_envelope_config[lfo_envelope_config_id].depth_cc_source;
   payload[3] = patch.lfo_envelope_config[lfo_envelope_config_id].enabled;
   payload[4] = patch.lfo_envelope_config[lfo_envelope_config_id].lfo_wave_type_id;
   payload[5] = patch.lfo_envelope_config[lfo_envelope_config_id].freq_cc_source;
   
   SendMessageToToneProcs( msg_type_lfo_envelope_config,6,recipient_id);
}

void send_envelope_control(uint16_t envelope_control_id,uint16_t recipient_id)
{
    payload[0] = envelope_control_id;
    payload[1] = patch.env_initial_depth_value[envelope_control_id];
    payload[2] = patch.env_gain_CC[envelope_control_id];
    
    SendMessageToToneProcs(msg_type_envelope_control,3,recipient_id);
}

void send_body_resonance_filter_band(uint16_t b,uint16_t recipient_id)
{
    payload[0] = b;
    payload[1] = patch.body_resonance_bands[b].level;
    payload[2] = patch.body_resonance_bands[b].slope;
    SendMessageToToneProcs( msg_type_body_resonance_filter_band,3,recipient_id);
}

void send_patch(uint16_t recipient_id)
{
    payload[0] = patch.active_layers;
    payload[1] = patch.degree_of_random_detuning;
    payload[2] = patch.degree_of_random_phase;
    payload[3] = patch.degree_of_regular_detuning;
    payload[4] = patch.pitch_bend_enabled;
    payload[5] = patch.portamento_enabled;
    payload[6] = patch.portamento_rate;
    payload[7] = patch.sustain_enabled;
    payload[8] = patch.detuning_mode_id;
    payload[9] = patch.layer_config[0].st_inc_q11_5_detune;
    payload[10] = patch.layer_config[1].st_inc_q11_5_detune;
    payload[11]= patch.layer_config[2].st_inc_q11_5_detune;
    payload[12]= patch.key_scale_split_note_id;

    SendMessageToToneProcs( msg_type_patch,13,recipient_id);
}

void send_waveform_harmonic(uint16_t n,uint16_t i,uint16_t waveform_id,uint16_t recipient_id)
{
    payload[0] = n;
    payload[1] = i;
    payload[2] = waveform_id;
    uint16_t h;
    for(h = 0;h < max_harmonics;h++)
        {
          payload[h + 3] = patch.waveform_set.waveform_harmonic[n][i][waveform_id][h].level;
        }
    SendMessageToToneProcs( msg_type_waveform_harmonic,35,recipient_id);
}

void send_waveform_set(uint16_t recipient_id)
{
    payload[0] = patch.waveform_set.timbre_controller1_source;
    payload[1] = patch.waveform_set.timbre_controller2_source;
    payload[2] = patch.waveform_set.timbre_mode;
    SendMessageToToneProcs(msg_type_waveform_set,3,recipient_id);
}

void send_channel_volume(uint16_t recipient_id,uint16_t volume)
{
    payload[0] = volume;
    SendMessageToToneProcs(ChannelVolume,1,recipient_id);
}

void send_all_patch_data(uint16_t recipient_id)
{
    OUT_MIXALERT_PORT_SetHigh();
    uint16_t adsr_section_id;
    uint16_t adsr_section_envelope_config_id;
    uint16_t lfo_envelope_config_id;
    uint16_t envelope_control_id;
    uint16_t n;
    uint16_t i;
    uint16_t waveform_id;
    uint16_t b;
    
    send_patch(recipient_id);
    
    for (adsr_section_id = 0; adsr_section_id < max_adsr_sections;adsr_section_id++)
        {
        send_adsr_section(adsr_section_id,recipient_id);
        for (adsr_section_envelope_config_id = 0; adsr_section_envelope_config_id < max_envelopes;adsr_section_envelope_config_id++)
            {
            send_adsr_section_envelope_config(adsr_section_id,adsr_section_envelope_config_id,recipient_id);
            }
        }

    for (lfo_envelope_config_id = 0; lfo_envelope_config_id < max_lfos;lfo_envelope_config_id ++)
        {
        send_lfo_envelope_config(lfo_envelope_config_id,recipient_id);
        }
    
    for (envelope_control_id = 0; envelope_control_id < max_envelopes ; envelope_control_id++)
    {
    send_envelope_control(envelope_control_id,recipient_id);
    }
    
    for (b = 0;b < max_body_resonance_filter_bands;b++)
    {
        send_body_resonance_filter_band(b,recipient_id);
    }

    for (n = 0;n < max_note_sectors;n++)
        {
            for (i = 0;i < max_intensity_layers;i++)
                {
                for (waveform_id = 0;waveform_id < max_waveforms;waveform_id++)
                    {
                    send_waveform_harmonic(n,i,waveform_id,recipient_id);
                    }
                }
        }
    
    send_waveform_set(recipient_id);
    OUT_MIXALERT_PORT_SetLow();
}

uint16_t get_patch_for_processor(uint16_t performance_id,uint16_t processor_id)
{
    uint16_t channel_id = performance[performance_id].processor_channel_allocation[processor_id];
    return performance[performance_id].pc[channel_id].module_patch_id;
}

void SelectPerformance(uint16_t p_id)
{
ChangePlayMode(StopPlayModeTP);
set_haas_delay_buffer_to_use ((uint16_t) __builtin_muluu(44,performance[p_id].haas_delay_ms)); //44 * 25 = 1100 which is the max buffer size and represents 24.6 ms delay

uint16_t module_patch_id;
uint16_t processor_id;
uint16_t flag_loaded_from_EEPROM;
uint16_t channel;
for(module_patch_id = 0;module_patch_id <patches_per_patchset;module_patch_id++)
    {
        flag_loaded_from_EEPROM = 0;
        for(processor_id =0;processor_id < max_processors;processor_id++)
        {
            if (get_patch_for_processor(p_id,processor_id) == module_patch_id)
            {
                if (flag_loaded_from_EEPROM == 0 )
                    {
                    LoadPatchFromEEPROM(module_patch_id);
                    flag_loaded_from_EEPROM = 1;
                    }
                send_all_patch_data(processor_id);
                
                channel = performance[p_id].processor_channel_allocation[processor_id];
                send_channel_volume(processor_id,performance[p_id].pc[channel].volume);
            }
        }
    }
ChangePlayMode(StartPlayModeTP);
}

inline void ProcessSPI1Message(void)
{
    uint8_t b;
    int16_t w;
    b = ReadByteFromSPI1RxBuffer();
    
    tp_control_value = 0x8000;

    if (SPI2TxState.status == SPI2_status_recipient)
        {
        SPI2TxState.recipient_id = b;
        SPI2TxState.status = SPI2_status_message_type;     
        }
    else if (SPI2TxState.recipient_id == all_processors)
        {
        switch (SPI2TxState.status)
            {
            case SPI2_status_message_type://If recipient is all_processor then the type can ONLY be CC
                SPI2TxState.message_type_id = b;  
                SPI2TxState.status = SPI2_status_ucc_index; 
                break;
            case SPI2_status_ucc_index:
                ucc_index = (uint16_t)b;
                SPI2TxState.status = SPI2_status_ucc_value_LSB; 
                break;
            case SPI2_status_ucc_value_LSB:
                ucc_value_LSB = (uint16_t)b;
                SPI2TxState.status = SPI2_status_ucc_value_MSB; 
                break;  
            case SPI2_status_ucc_value_MSB:
                ucc_value_MSB = (uint16_t)b;
                tp_control_value = ((uint16_t)0x8000 | (uint16_t)(ucc_index<<12)) | ((ucc_value_MSB<<8) + ucc_value_LSB);
                SPI2TxState.status = SPI2_status_recipient; 
                break;  
            }
        }
    else if (SPI2TxState.recipient_id <= processor5)
        {
        switch (SPI2TxState.status)
            {
            case SPI2_status_message_type:
                SPI2TxClearSS(SPI2TxState.recipient_id); //Select Tone processor
                while (SPI2STATbits.SPITBF == 1){};SPI2BUF = b;while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
                SPI2TxSetSS(SPI2TxState.recipient_id); //De-select Tone processor
                
                SPI2TxState.message_type_id = b;  
                SPI2TxState.status = SPI2_status_payload_length_LSB;   
                break;
            case SPI2_status_payload_length_LSB:
                SPI2TxState.payload_length_LSB = b;
                SPI2TxState.status = SPI2_status_payload_length_MSB;
                break;                   
            case SPI2_status_payload_length_MSB:
                SPI2TxState.payload_length_MSB = b;
                SPI2TxState.payload_length = (uint16_t)((SPI2TxState.payload_length_MSB<<8) + SPI2TxState.payload_length_LSB);
                SPI2TxState.payload_words_pending =  SPI2TxState.payload_length;
                SPI2TxState.status = SPI2_status_payload;
                break;                  
            case SPI2_status_payload: 
                SPI2TxClearSS(SPI2TxState.recipient_id); //Select Tone processor
                while (SPI2STATbits.SPITBF == 1){};SPI2BUF = b; while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; tone_proc_sample_sum +=(int32_t)w;
                SPI2TxSetSS(SPI2TxState.recipient_id); //De-select Tone processor
                
                SPI2TxState.payload_words_pending--;
                if (SPI2TxState.payload_words_pending == 0)
                    {
                    SPI2TxState.status = SPI2_status_recipient;
                    } 
                break;           
            }
        }
    else 
    {
    //MIXER MESSAGE
    OUT_MIXALERT_PORT_SetHigh(); 
    alert_delay_ms =  mix_alert_delay_ms;
    SPI2TxState.message_type_id = b;

    switch(SPI2TxState.message_type_id)
        {
            case  msg_type_adsr_section :
                {
                    uint8_t adsr_section_id = WaitForByteFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].flag_active = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].end_time_ms = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].end_time_ms_KU = (int16_t)WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].end_time_ms_KL = (int16_t)WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].inharmonic_sample_playback_mode_id = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].inharmonic_sample_id = WaitForWordFromSPI1RxBuffer();

                    send_adsr_section(adsr_section_id,all_processors); 
                }
                break;

            case msg_type_adsr_section_envelope_config :
                {
                    uint8_t adsr_section_id = WaitForByteFromSPI1RxBuffer();
                    uint8_t adsr_section_envelope_config_id = WaitForByteFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KU = (int16_t)WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KL = (int16_t)WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_lin_delta = WaitForWordFromSPI1RxBuffer();
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_exp_multiplier = WaitForWordFromSPI1RxBuffer();

                    uint16_t depth_delta_KU_linear;
                    uint16_t depth_delta_KL_linear;
                    uint16_t depth_delta_KU_exp;
                    uint16_t depth_delta_KL_exp;
                    uint16_t depth_delta_KU;
                    uint16_t depth_delta_KL;

                    depth_delta_KU_linear = (int16_t)WaitForWordFromSPI1RxBuffer();
                    depth_delta_KL_linear  = (int16_t)WaitForWordFromSPI1RxBuffer();
                    depth_delta_KU_exp = (int16_t)WaitForWordFromSPI1RxBuffer();
                    depth_delta_KL_exp  = (int16_t)WaitForWordFromSPI1RxBuffer();

                    if(patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type == env_type_linear)
                        {
                        depth_delta_KU = depth_delta_KU_linear;
                        depth_delta_KL = depth_delta_KL_linear;
                        }
                    else
                        {
                        depth_delta_KU = depth_delta_KU_exp;
                        depth_delta_KL = depth_delta_KL_exp;
                        }
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KU = depth_delta_KU;
                    patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KL = depth_delta_KL;

                    send_adsr_section_envelope_config(adsr_section_id,adsr_section_envelope_config_id,all_processors);  
                }
                break;

            case msg_type_lfo_envelope_config :
                {
                    uint8_t lfo_envelope_config_id = WaitForByteFromSPI1RxBuffer();
                    patch.lfo_envelope_config[lfo_envelope_config_id].default_wt_inc_q11_5= WaitForWordFromSPI1RxBuffer();
                    patch.lfo_envelope_config[lfo_envelope_config_id].depth_cc_source= WaitForWordFromSPI1RxBuffer();
                    patch.lfo_envelope_config[lfo_envelope_config_id].enabled= WaitForWordFromSPI1RxBuffer();
                    patch.lfo_envelope_config[lfo_envelope_config_id].lfo_wave_type_id = WaitForWordFromSPI1RxBuffer();
                    patch.lfo_envelope_config[lfo_envelope_config_id].freq_cc_source= WaitForWordFromSPI1RxBuffer();

                    send_lfo_envelope_config(lfo_envelope_config_id,all_processors);
                }
                break;                

            case msg_type_envelope_control:
                {
                    uint8_t envelope_control_id = WaitForByteFromSPI1RxBuffer();
                    patch.env_initial_depth_value[envelope_control_id] = WaitForWordFromSPI1RxBuffer();
                    patch.env_gain_CC[envelope_control_id] = WaitForWordFromSPI1RxBuffer();

                    send_envelope_control(envelope_control_id,all_processors); 
                }
                break;

            case msg_type_body_resonance_filter_band:
                {
                    uint16_t b = WaitForByteFromSPI1RxBuffer();
                    patch.body_resonance_bands[b].level = WaitForWordFromSPI1RxBuffer();
                    patch.body_resonance_bands[b].slope = WaitForWordFromSPI1RxBuffer();

                    send_body_resonance_filter_band(b,all_processors);
                }
                break;
            case StartPlayMode :
                {
                WaitForByteFromSPI1RxBuffer(); //Empty payload
                ChangePlayMode(StartPlayModeTP);
                break;
                }
            case StopPlayMode :
                {
                WaitForByteFromSPI1RxBuffer(); //Empty payload
                ChangePlayMode(StopPlayModeTP);
                break;
                }
            case msg_type_patch :
                patch.active_layers = WaitForWordFromSPI1RxBuffer();
                patch.degree_of_random_detuning = WaitForWordFromSPI1RxBuffer();
                patch.degree_of_random_phase = WaitForWordFromSPI1RxBuffer();
                patch.degree_of_regular_detuning = WaitForWordFromSPI1RxBuffer();
                patch.pitch_bend_enabled = WaitForWordFromSPI1RxBuffer();
                patch.portamento_enabled = WaitForWordFromSPI1RxBuffer();
                patch.portamento_rate = WaitForWordFromSPI1RxBuffer();
                patch.sustain_enabled = WaitForWordFromSPI1RxBuffer();
                patch.detuning_mode_id = WaitForWordFromSPI1RxBuffer();
                patch.layer_config[0].st_inc_q11_5_detune = WaitForWordFromSPI1RxBuffer();
                patch.layer_config[1].st_inc_q11_5_detune = WaitForWordFromSPI1RxBuffer();
                patch.layer_config[2].st_inc_q11_5_detune = WaitForWordFromSPI1RxBuffer();
                patch.key_scale_split_note_id = WaitForWordFromSPI1RxBuffer();
                send_patch(all_processors);
                break;

            case msg_type_waveform_harmonic :
                {
                    uint8_t n = WaitForByteFromSPI1RxBuffer();
                    uint8_t i = WaitForByteFromSPI1RxBuffer();
                    uint8_t waveform_id = WaitForByteFromSPI1RxBuffer();
                    uint8_t h;
                    for(h = 0;h < max_harmonics;h++)
                        {
                        patch.waveform_set.waveform_harmonic[n][i][waveform_id][h].level = WaitForByteFromSPI1RxBuffer();
                        }

                    send_waveform_harmonic(n,i,waveform_id,all_processors);
                }
                break;

            case msg_type_waveform_set :
                patch.waveform_set.timbre_controller1_source = WaitForWordFromSPI1RxBuffer();
                patch.waveform_set.timbre_controller2_source = WaitForWordFromSPI1RxBuffer();
                patch.waveform_set.timbre_mode = WaitForWordFromSPI1RxBuffer();
                send_waveform_set(all_processors);
                break;

            case msg_type_save_patch_to_EEPROM :
                patch_id_to_save = (int16_t)WaitForByteFromSPI1RxBuffer();
                flag_do_non_sample_stuff = 1;
                break;

            case msg_type_load_patch_from_EEPROM :
//                //***TEST
//                while (1)
//                {OUT_MIXALERT_PORT_SetHigh();}
                
                WaitForWordFromSPI1RxBuffer(); //Payload length, not used
                patch_id_to_load = (int16_t)WaitForByteFromSPI1RxBuffer();
                flag_do_non_sample_stuff = 1;
                break;

            case ChangeVolume :
                {
                    WaitForWordFromSPI1RxBuffer(); //Payload length, not used
                    uint8_t new_volume_value = WaitForByteFromSPI1RxBuffer();
                    //NOTE: Limit the value because need to reduce max to stop op-amp getting overloaded.
                    if (new_volume_value >230){new_volume_value = 230;}
                    global_gain =(uint16_t)(new_volume_value <<8);
                    break;
                }
            case msg_type_performance :
                { 
                    uint16_t performance_id = (uint16_t)WaitForByteFromSPI1RxBuffer();
                    uint16_t i;
                    for(i = 0;i < performance_size;i++)
                        {performance[performance_id].data[i] = (uint16_t)WaitForByteFromSPI1RxBuffer();}                        
                    break;
                } 
            case msg_type_select_performance :
                {
                    performance_id_to_select = (uint16_t)WaitForByteFromSPI1RxBuffer();
                    flag_do_non_sample_stuff = 1;
                    break;
                }

        }
        SPI2TxState.status = SPI2_status_recipient; 
    }   
}


