/* 
 * File:   Spectral_SPI_1.c
 * Author: Jeremy Leach
 */
#include "Spectral_SPI_1.h"
#include "Spectral_Timer1.h"

/******************************************************************************/
/***** Declarations ***********************************************************/
/******************************************************************************/ 
volatile fifo_buffer8_t SPI1RxBuffer; //Must be declared volatile because changed by ISR.

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/ 

void SPI1Initialize(void) 
{   
   //Re-map SPI1 to the pins being used (SPI from the master)
    RPINR20bits.SDI1R   = 7;  //SPI1 Data input. Tied to RP7.
    RPINR20bits.SCK1R   = 10; //SPI1 Clock input. Tied to RP10.
    RPINR21bits.SS1R    = 11; //SPI1 Slave select input. Tied to RP11.
    RPOR3bits.RP6R      = 0b00111;//SDO1     

    SPI1BUF = 0;
    
    //Initialise the Rx buffer
    SPI1RxBuffer.head   = 0;
    SPI1RxBuffer.tail   = 0;
    SPI1RxBuffer.count  = 0;
    SPI1RxBuffer.flag_overflow = 0;
        
    //Disable interrupts and clear flags
    IFS0bits.SPI1IF     = 0; //Clear the interrupt flag
    IFS0bits.SPI1EIF    = 0; //Clear the interrupt flag
    IEC0bits.SPI1IE     = 0; //Disable SPI1 Event Interrupt
    IEC0bits.SPI1EIE    = 0; //Disable SPI1 Error event interrupt
    
    //SPI1CON: 
    SPI1CON1bits.DISSCK =0; //Internal serial clock is enabled
    SPI1CON1bits.DISSDO =0; //0 = SDOx pin is controlled by the module    
    SPI1CON1bits.MODE16 =0; //0 = Communication is byte-wide (8 bits)
    SPI1CON1bits.SMP    =0; //0 = Input data sampled at middle of data output time. Must be cleared in Slave mode.
    SPI1CON1bits.CKE    =0; //0 = Serial output data changes on transition from Idle clock state to active clock state (refer to bit 6)
    SPI1CON1bits.CKP    =0; //0 = Idle state for clock is a low level; active state is a high level
    SPI1CON1bits.MSTEN  =0; //0 = Slave mode    
    SPI1STATbits.SPIROV =0; //Receive Overflow Flag bit. 1 = A new byte/word is completely received and discarded. The user software has not read the previous data in the SPIxBUF register. 0 = No overflow has occurred         
    SPI1CON1bits.SSEN   =0;//1; //Removed SS because not necessary (but still leave on master).1 = SSx pin is used for Slave mode.
 
    SPI1CON2bits.FRMEN  =0; //0= Framed SPIx support disabled  
    SPI1STATbits.SPIEN = 1; //SPIx Enable bit.  1 = Enables module and configures SCKx, SDOx, SDIx and SSx as serial port pins
}

uint8_t ReadByteFromSPI1RxBuffer(void)
{   //Assumes there is data in the buffer 

    IEC0bits.SPI1IE = 0;//Disable SPI1 interrupt that manipulates buffer
    uint8_t b = SPI1RxBuffer.buf[SPI1RxBuffer.tail];
    SPI1RxBuffer.count--;
    SPI1RxBuffer.tail++;

    if (SPI1RxBuffer.tail == FifoBuffer_size){ SPI1RxBuffer.tail = 0;}
    IEC0bits.SPI1IE = 1;//Re-enable SPI1 interrupt
    return b;       
}

uint8_t WaitForByteFromSPI1RxBuffer(void)
{
    while(SPI1RxBuffer.count==0){}
    return ReadByteFromSPI1RxBuffer();
}

uint16_t WaitForWordFromSPI1RxBuffer(void)
{
    uint8_t LSB =  WaitForByteFromSPI1RxBuffer();
    uint8_t MSB =  WaitForByteFromSPI1RxBuffer();
    return (uint16_t)((MSB<<8) + LSB);
}

void WriteByteToSPI1RxBuffer(uint8_t data)
{
if (SPI1RxBuffer.count >= max_FifoBuffer_count)
    {
    SPI1RxBuffer.flag_overflow = 1;
    OUT_SPI2_SS1_PORT_SetLow(); OUT_MIXALERT_PORT_SetHigh(); while(1){};//FLAG ERROR THROUGH SS LINE
    }
else 
    {
    SPI1RxBuffer.buf[SPI1RxBuffer.head] = data; 
    SPI1RxBuffer.head++; 
    SPI1RxBuffer.count++;
    
    if (SPI1RxBuffer.head == FifoBuffer_size){SPI1RxBuffer.head = 0;}
    }
}
