/* 
 * File:   Spectral_Sounds.h
 * Author: Jeremy Leach
 */

#ifndef SPECTRAL_SOUNDS_H
#define	SPECTRAL_SOUNDS_H

#ifdef	__cplusplus
extern "C" {
#endif

#include <xc.h>
#include "Spectral_Messaging.h"
    
/******************************************************************************/
/***** Definitions ************************************************************/
/******************************************************************************/
#define max_instances 3
#define max_waveforms 5
#define max_lfos 3
#define max_harmonics 32
#define max_adsr_sections 4
#define max_layers 2
#define max_envelopes 10
#define max_non_lfo_envelopes 7
#define max_audio_freq 15000
#define sampling_freq 41666
#define max_intensity_layers 3
#define max_note_sectors 5
//NOTE: Had to change max controllers from 95 to 25, the actual number of controllers used. So need mapping between controller is and index.
#define max_controllers 8
#define max_body_resonance_filter_bands 32
#define eq_band_lookup_table_length 1024
#define midi_note_range 88
    
#define wavetable_samples 2048
#define sample_length 4096
#define midi_note_number_a0 21
#define midi_note_number_c3 60
    
#define KS_buffer_size 682
#define ks_lowest_midi_note_number 36
    
#define spi1_int_priority 7
#define timer2_int_priority 6
#define timer1_int_priority 5
    
#define max_to_hold_on_by_sustain 2
    
/******************************************************************************/
/***** Enums ******************************************************************/
/******************************************************************************/

    enum detune_modes
    {
        dm_cents = 0,
        dm_hz = 1
    };
    
    enum env_types
    {
        env_type_none = 0,
        env_type_linear = 1,
        env_type_exponential = 2
    };
    
    enum pv_indices {
        p0v0 = 0,
        p0v1 = 1,
        p1v0 = 2,
        p1v1 = 3
    };
    
    enum adsr_sections {
        adsr_attack = 0,
        adsr_decay = 1,
        adsr_sustain = 2,
        adsr_release = 3
    };
    
    enum inharmonic_sample_playback_modes{
        sample_playback_mode_none = 0,
        sample_playback_mode_one_shot = 1,
        sample_playback_mode_looped = 2
    };    
    
    enum inharmonic_samples
    {
        inharmonic_sample_none = 0,
        inharmonic_sample_hammer1 = 1,
        inharmonic_sample_hammer2 = 2,
        inharmonic_sample_hammer3 = 3,
        inharmonic_sample_hammer4 = 4,
        inharmonic_sample_bowscrape1 = 5,
        inharmonic_sample_bowscrape2 = 6,
        inharmonic_sample_keyup = 7,
        inharmonic_sample_keyclick = 8,
        inharmonic_sample_KS_very_long_not_filtered = 9,
        inharmonic_sample_KS_long_not_filtered = 10,
        inharmonic_sample_KS_normal_not_filtered = 11,
        inharmonic_sample_KS_short_not_filtered = 12,
        inharmonic_sample_KS_very_short_not_filtered = 13,
        inharmonic_sample_KS_very_long_filtered = 14,
        inharmonic_sample_KS_long_filtered = 15,
        inharmonic_sample_KS_normal_filtered = 16,
        inharmonic_sample_KS_short_filtered = 17,
        inharmonic_sample_KS_very_short_filtered = 18
    };
    
    enum timbre_modes{
        timbre_mode_2d = 2,
        timbre_envelope = 3
    };
    
    enum lfos
    {
        lfo_tremolo = 0,
        lfo_vibrato = 1,
        lfo_timbre = 2
    };
    
    enum patch_row_types
    {
        pt_general_midi = 0,
        pt_user = 1
    };
    
    enum envelope
    {
        env_amplitude = 0,
        env_noise = 1,
        env_noise_cutoff_level = 2,
        env_timbre = 3,
        env_sample = 4,
        env_pitch = 5,
        env_portamento = 6,
        env_tremolo = 7,
        env_vibrato = 8,
        env_timbre_lfo = 9
    };
    
    enum lfo_wave_types
    {
        sine = 0,
        square = 1,
        triangle = 2,
        saw = 3
    };
    
/******************************************************************************/
/***** Types - Database related ***********************************************/
/******************************************************************************/

typedef struct  {
        int16_t buf[KS_buffer_size];
        uint16_t index;
        uint16_t size;
        uint16_t multiplier;
        } fifo_KSBuffer;
    
typedef struct {
    uint16_t level;
    //uint16_t phase;  
}waveform_harmonic_row_t;

typedef struct  {   
    waveform_harmonic_row_t waveform_harmonic[max_note_sectors][max_intensity_layers][max_waveforms][max_harmonics];
    uint16_t timbre_mode;
    uint16_t timbre_controller1_source;
    uint16_t timbre_controller2_source;
} waveform_set_row_t;

typedef struct  {
    uint16_t enabled;
    uint16_t lfo_wave_type_id;
    uint16_t default_wt_inc_q11_5;
    uint16_t freq_cc_source;    
    uint16_t depth_cc_source;  
} lfo_envelope_config_row_t;

typedef struct  {
    uint16_t depth_env_type;
    uint16_t depth_env_target;
    int16_t depth_env_target_KU;
    int16_t depth_env_target_KL;
    uint16_t depth_env_lin_delta;
    uint16_t depth_env_exp_multiplier;
    int16_t depth_delta_KU;
    int16_t depth_delta_KL;
    
} adsr_section_envelope_config_row_t;

typedef struct {
    uint16_t level;
    int16_t slope;
} body_resonance_band_row_t;

typedef struct  {   
    uint16_t flag_active;
    uint16_t end_time_ms;
    int16_t end_time_ms_KU;
    int16_t end_time_ms_KL;
    uint16_t inharmonic_sample_playback_mode_id;
    uint16_t inharmonic_sample_id;
    //-------------------------------------
    adsr_section_envelope_config_row_t adsr_section_envelope_config[max_envelopes];
} adsr_section_row_t;

typedef struct {
   uint16_t st_inc_q11_5_detune;
} layer_config_t;

typedef struct  {   
    uint16_t sustain_enabled;
    uint16_t pitch_bend_enabled;   
    uint16_t portamento_enabled; 
    uint16_t portamento_rate;
    uint16_t degree_of_regular_detuning;
    uint16_t degree_of_random_detuning;
    uint16_t degree_of_random_phase;
    uint16_t active_layers;
    uint16_t detuning_mode_id;
  //----------------------------------
    lfo_envelope_config_row_t lfo_envelope_config[max_lfos];
    waveform_set_row_t waveform_set;
    adsr_section_row_t adsr_section[max_adsr_sections];
    layer_config_t layer_config[max_layers];
    uint16_t patch_gain;
    uint16_t key_scale_split_note_id;
  //----------------------------------
    uint16_t env_gain_CC[max_envelopes];
    uint16_t env_initial_depth_value[max_envelopes];
  //---------------------------------
    body_resonance_band_row_t body_resonance_bands[max_body_resonance_filter_bands];
} patch_row_t;

/******************************************************************************/
/***** Types - local **********************************************************/
/******************************************************************************/    
typedef struct  {
    uint16_t current_wt_inc_q11_5;
    uint16_t wt_index_q11_5;
    int16_t gain; //Signed
    uint16_t depth_env_value;
    uint16_t flag_update_pending;
    //-------------------------------------
    uint16_t KS_depth_env_target[max_adsr_sections];
    uint16_t KS_depth_delta[max_adsr_sections];
} lfo_envelope_current_t;

typedef struct  {
    uint16_t depth_env_value;
    uint16_t depth_env_value_scaled_by_CC;
    //-------------------------------------
    uint16_t KS_depth_env_target[max_adsr_sections];
    uint16_t KS_depth_delta[max_adsr_sections];
} envelope_current_t;

typedef struct  {
    uint16_t event_32Sample_trigger_count;
    int32_t current_sample;
    int32_t last_sample;
    int16_t sample_delta16;
    uint16_t rnd_num;
    uint16_t current_rough_wavetable_id;
} system_t;

typedef struct {
   uint16_t wt_inc_q11_5;
   uint16_t wt_index_q11_5;
   uint16_t calculated_detune;
} layer_current_t;
    
typedef struct  {
        uint16_t flag_active;
        uint16_t flag_amplitude_env_started;
        uint16_t wavetable_index;
        uint16_t flag_held_on_by_sustain;
        uint16_t flag_timbre_recalc_pending;
        uint16_t midi_note_id;
        uint16_t note_sector_id_low;
        uint16_t note_sector_note_b;
        uint16_t intensity_layer_id_low;
        uint16_t intensity_layer_intensity_b;
        uint16_t velocity16;
        uint16_t adsr_section_id;
        uint16_t last_timbre_controller1_value;
        uint16_t last_timbre_controller2_value;
        int16_t noise_lpf_output;
        //-------------------------------------
        uint16_t fund_freq_hz;
        uint16_t fund_wt_inc_q11_5;
        //-------------------------------------
        uint16_t adsr_section_elapsed_time_ms;
        uint16_t inharmonic_elapsed_samples;
        __eds__ int16_t *inharmonic_sample_ptr;
        __eds__ int16_t *inharmonic_sample_ptr_start;
        uint16_t flag_inharmonic_sample_active;
        //-------------------------------------
        uint16_t trem_offset_gain;
        uint16_t lfo_timbre_offset_gain;
        uint16_t overall_gain;
        uint16_t cutoff_gain;
        //-------------------------------------
        adsr_section_envelope_config_row_t portamento_env_config;
        //-------------------------------------      
        layer_current_t layer[max_layers];
        envelope_current_t env_current[max_non_lfo_envelopes];  
        uint16_t last_timbre_depth_env_value_scaled_by_CC;
        lfo_envelope_current_t lfo_env_current[max_lfos]; 
        //--------------------------------------
        int16_t amplitude_intermediate_sum;
        int16_t amplitude_intermediate_delta;
        int16_t flag_amplitude_interp_mult32;
        //--------------------------------------
        int16_t key_scale_split_diff;
        uint16_t KS_end_time[max_adsr_sections];
        //--------------------------------------
        uint16_t max_delta_inc_q11_5_UP;
        } instance_t; 

typedef struct  {
    uint16_t flag_sustain_on;
    uint16_t ucc_current_values[max_controllers];
    uint16_t midi_channel;
    patch_row_t patch;
    int32_t sample_sum;
    uint16_t volume;
} channel_info;

extern volatile channel_info channel;

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/
int16_t interpolated_sine_lookup(uint16_t st_index_q11_5);
int16_t interpolated_wt_lookup(uint16_t i,uint16_t y);
uint16_t BoundedUSWFromSL(int32_t L);
int16_t SquareLookup(uint16_t wt_index);
int16_t SawLookup(uint16_t wt_index);
int16_t TriangleLookup(uint16_t wt_index);
void ProcessControlChange(msg_instance_cc cc);
void CheckForInharmonicSamplePlayback(uint16_t i,uint16_t section_id);
void ProcessInstanceNoteOff(uint16_t i);
void Sounds_Initialize(void);
void StepEnvelopeNotKeyScaled(volatile adsr_section_envelope_config_row_t *s,volatile envelope_current_t *c);
void UpdateAmplitudeDepthValue(uint16_t i);
void StepAmplitudeEnvelope(uint16_t i,uint16_t a);
void StepEnvelope(volatile adsr_section_envelope_config_row_t *s,volatile envelope_current_t *c,uint16_t a);
void StepLFOEnvelope(volatile lfo_envelope_config_row_t *p,volatile adsr_section_envelope_config_row_t *s,volatile lfo_envelope_current_t *c,uint16_t a,uint16_t i);
uint16_t GetEnvCCValue16(uint16_t i,uint16_t envelope_id);
uint16_t GetLFOFreqEnvCCValue16(uint16_t i,uint16_t lfo_envelope_id);
void process_10ms_event(void);
void process_1ms_event(void);
uint16_t get_filter_gain(uint16_t freq_hz);
void CalculateInstanceWavetable(uint16_t instance_id);
void TimbreUpdate(uint16_t i);
void InitialiseInstance(uint16_t i,uint16_t midi_note_number,uint16_t midi_velocity,uint16_t midi_channel,uint16_t last_midi_note_number);
void CheckForEndOfSection(uint16_t i);
void UpdateSampleValue(void);

#ifdef	__cplusplus
}
#endif

#endif	/* SPECTRAL_SOUNDS_H */

