/* 
 * File:   Spectral_SPI_1.c
 * Author: Jeremy Leach
 */
#define  FCY  70000000UL
#include <libpic30.h>        // __delayXXX() functions macros defined here
#include "Spectral_Messaging.h"
#include "Spectral_SPI_1.h"
#include "Spectral_Pins.h" 
#include "Spectral_Sounds.h"

/******************************************************************************/
/***** Declarations ***********************************************************/
/******************************************************************************/ 
static volatile fifo_SPI1RxBuffer16 SPI1RxBuffer; //Must be declared volatile because changed by ISR.
static volatile uint16_t flag_SPI1_rx_buffer_overflow;
static const uint16_t Velocity16Lookup[128]={4607,4607,4607,4607,4607,4607,4607,4607,4607,4607,4607,4607,5119,5119,5119,5119,5119,5631,5631,5631,5631,6143,6143,6143,6655,6655,6655,7167,7167,7679,7679,8191,8191,8703,8703,9215,9215,9727,9727,10239,10239,10751,11263,11263,11775,11775,12287,12799,12799,13311,13823,14335,14335,14847,15359,15871,16383,16383,16895,17407,17919,18431,18943,19455,19967,20479,20991,21503,22015,22527,23039,23551,24063,24575,25087,25599,26111,26623,27135,28159,28671,29183,29695,30207,31231,31743,32255,32767,33791,34303,34815,35839,36351,36863,37887,38399,38911,39935,40447,41471,41983,43007,43519,44543,45055,46079,46591,47615,48639,49151,50175,50687,51711,52735,53247,54271,55295,55807,56831,57855,58879,59903,60415,61439,62463,63487,64511,65535};
extern volatile channel_info channel;

/******************************************************************************/
/***** Methods ****************************************************************/
/******************************************************************************/ 

void SPI1_Initialize(void) 
{      
    SPI1BUF = 0;
    
    //Initialise the Rx buffer
    SPI1RxBuffer.head   = 0;
    SPI1RxBuffer.tail   = 0;
    SPI1RxBuffer.count  = 0;
    flag_SPI1_rx_buffer_overflow = 0;
        
    //Disable interrupts and clear flags
    IFS0bits.SPI1IF     = 0; //Clear the interrupt flag
    IFS0bits.SPI1EIF    = 0; //Clear the interrupt flag
    IEC0bits.SPI1IE     = 0; //Disable SPI1 Event Interrupt
    IEC0bits.SPI1EIE    = 0; //Disable SPI1 Error event interrupt
    
    //SPI1CON: 
    SPI1CON1bits.DISSCK = 0; //Internal serial clock is enabled
    SPI1CON1bits.DISSDO = 0; //0 = SDOx pin is controlled by the module    
    SPI1CON1bits.MODE16 = 1; //0 = Communication is Word (16 bits))
    SPI1CON1bits.SMP    = 0; //0 = Input data sampled at middle of data output time. Must be cleared in Slave mode.
    SPI1CON1bits.CKE    = 0; //0 = Serial output data changes on transition from Idle clock state to active clock state
    SPI1CON1bits.CKP    = 0; //0 = Idle state for clock is a low level; active state is a high level
    SPI1CON1bits.MSTEN  = 0; //0 = Slave mode    
    SPI1STATbits.SPIROV = 0; // Receive Overflow Flag bit. 1 = A new byte/word is completely received and discarded. The user software has not read the previous data in the SPIxBUF register. 0 = No overflow has occurred         
    SPI1CON1bits.SSEN   = 1; //1 = SSx pin is used for Slave mode.
 
    SPI1CON2bits.FRMEN  = 0; //0 = Framed SPIx support disabled
    SPI1CON2bits.SPIBEN = 0; //0 = Enhanced Buffer disabled.
    
    SPI1STATbits.SPIEN  = 1;   // SPIx Enable bit.  1 = Enables module and configures SCKx, SDOx, SDIx and SSx as serial port pins
}

void ClearSPI1RxBuffer(void)
{
    SPI1RxBuffer.head = 0;
    SPI1RxBuffer.tail = 0;
    SPI1RxBuffer.count = 0;
    flag_SPI1_rx_buffer_overflow = 0;
}

uint16_t ReadWordFromSPI1RxBuffer(void)
{   //Assumes there is data in the buffer
    IEC0bits.SPI1IE = 0;//Disable SPI1 interrupt that manipulates buffer
    uint16_t w = SPI1RxBuffer.buf[SPI1RxBuffer.tail];
    SPI1RxBuffer.count--;
    SPI1RxBuffer.tail++;

    if (SPI1RxBuffer.tail == SPI1RxBuffer_size){ SPI1RxBuffer.tail = 0;}
    IEC0bits.SPI1IE = 1;//Re-enable SPI1 interrupt
    return w;       
}

uint16_t WaitForWordFromRxBuffer(void)
{
    while(SPI1RxBuffer.count==0){}
    return ReadWordFromSPI1RxBuffer();
}

void WriteWordToSPI1RxBuffer(uint16_t data)
{
if (SPI1RxBuffer.count >= max_SPI1RxBuffer_count)
    {
    INTCON2bits.GIE = 0;
    while(1);
    flag_SPI1_rx_buffer_overflow = 1;
    }
else 
    {
    SPI1RxBuffer.buf[SPI1RxBuffer.head] = data; 
    SPI1RxBuffer.head++; 
    SPI1RxBuffer.count++;
    
    if (SPI1RxBuffer.head == SPI1RxBuffer_size){SPI1RxBuffer.head = 0;}
    }
}

int16_t ProcessSPIRxBuffer(void)
{
uint16_t note_on_instance_id = -1;
if (SPI1RxBuffer.count>0)
{ 
    uint16_t message_type_id;
    uint16_t adsr_section_id;
    message_type_id = (uint16_t)(ReadWordFromSPI1RxBuffer());
                
    switch (message_type_id)
    {
        case InstanceNoteOn:
        {       
            msg_instance_noteon inon;
            inon.instance_id = WaitForWordFromRxBuffer();    
            inon.midi_channel = WaitForWordFromRxBuffer();
            inon.patch_id = WaitForWordFromRxBuffer();          
            inon.note_id = WaitForWordFromRxBuffer();
            inon.last_midi_note_number = WaitForWordFromRxBuffer();
            inon.velocity_id = WaitForWordFromRxBuffer();
            
            /*
             * NOTE VELOCITY:
             * The lookup translates the MIDI velocity into 'Loudness'. The Loudness curve that synths use (and used here) 
             * is more a squared-law rather than log (there is research online).
             * We also limit the minimum to 8 (MIDI velocity 0 to 127) because this helps with SNR. 
             * Halving perceived loudness is approximately reducing amplitude by a third, so this gives slightly more dynamic range than that. 
             */
            uint16_t velocity16 = Velocity16Lookup[inon.velocity_id]; 
            
            InitialiseInstance(inon.instance_id,inon.note_id,velocity16,inon.midi_channel,inon.last_midi_note_number);             
            break;
        }
        case InstanceNoteOff: 
        {
            msg_instance_noteoff inoff;
            inoff.instance_id = WaitForWordFromRxBuffer();
             
            ProcessInstanceNoteOff(inoff.instance_id);  
            break; 
        }
        case msg_type_adsr_section:
        {
            adsr_section_id = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].flag_active = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].end_time_ms = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].end_time_ms_KU = (int16_t)WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].end_time_ms_KL = (int16_t)WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].inharmonic_sample_playback_mode_id = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].inharmonic_sample_id = WaitForWordFromRxBuffer();
            break;                  
        }
        case msg_type_adsr_section_envelope_config:
        {
            adsr_section_id = WaitForWordFromRxBuffer();
            uint16_t adsr_section_envelope_config_id = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_type = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target = WaitForWordFromRxBuffer();                          
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KU = (int16_t)WaitForWordFromRxBuffer();                       
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_target_KL = (int16_t)WaitForWordFromRxBuffer();                       
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_lin_delta = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_env_exp_multiplier = WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KU = (int16_t)WaitForWordFromRxBuffer();
            channel.patch.adsr_section[adsr_section_id].adsr_section_envelope_config[adsr_section_envelope_config_id].depth_delta_KL = (int16_t)WaitForWordFromRxBuffer();

            break;
        }
        case msg_type_biquad_eq_calculated:
        {
            WaitForWordFromRxBuffer();//uint16_t biquad_eq_calculated_id = WaitForWordFromRxBuffer();
            WaitForWordFromRxBuffer();//channel.patch.biquad_eq_calculated[biquad_eq_calculated_id].a0= WaitForWordFromRxBuffer();
            WaitForWordFromRxBuffer();//channel.patch.biquad_eq_calculated[biquad_eq_calculated_id].a1= WaitForWordFromRxBuffer();
            WaitForWordFromRxBuffer();//channel.patch.biquad_eq_calculated[biquad_eq_calculated_id].a2= WaitForWordFromRxBuffer();
            WaitForWordFromRxBuffer();//channel.patch.biquad_eq_calculated[biquad_eq_calculated_id].b1= WaitForWordFromRxBuffer();
            WaitForWordFromRxBuffer();//channel.patch.biquad_eq_calculated[biquad_eq_calculated_id].b2= WaitForWordFromRxBuffer();            
            break;
        }
        case msg_type_lfo_envelope_config:
        {
            uint16_t lfo_envelope_config_id = WaitForWordFromRxBuffer();
            channel.patch.lfo_envelope_config[lfo_envelope_config_id].default_wt_inc_q11_5= WaitForWordFromRxBuffer();
            channel.patch.lfo_envelope_config[lfo_envelope_config_id].depth_cc_source= WaitForWordFromRxBuffer();
            channel.patch.lfo_envelope_config[lfo_envelope_config_id].enabled= WaitForWordFromRxBuffer();
            channel.patch.lfo_envelope_config[lfo_envelope_config_id].lfo_wave_type_id= WaitForWordFromRxBuffer();
            channel.patch.lfo_envelope_config[lfo_envelope_config_id].freq_cc_source= WaitForWordFromRxBuffer();
            break;
        }
        case msg_type_envelope_control:
        {
            uint16_t envelope_control_id = WaitForWordFromRxBuffer();
            channel.patch.env_initial_depth_value[envelope_control_id] = WaitForWordFromRxBuffer();
            channel.patch.env_gain_CC[envelope_control_id] = WaitForWordFromRxBuffer();
            break;
        }
        case msg_type_body_resonance_filter_band:
        {
            uint16_t b;
            b = WaitForWordFromRxBuffer();
            channel.patch.body_resonance_bands[b].level = WaitForWordFromRxBuffer();
            channel.patch.body_resonance_bands[b].slope = WaitForWordFromRxBuffer();
            break;
        }
        case InstanceControlChange:
        {
            msg_instance_cc cc;
            cc.midi_channel = WaitForWordFromRxBuffer();
            cc.midi_control = WaitForWordFromRxBuffer();
            cc.control_value = WaitForWordFromRxBuffer();
            ProcessControlChange(cc);
            break; 
        }
        case msg_type_patch:
        {
            channel.patch.active_layers = WaitForWordFromRxBuffer();
            channel.patch.degree_of_random_detuning = WaitForWordFromRxBuffer();
            channel.patch.degree_of_random_phase = WaitForWordFromRxBuffer();
            channel.patch.degree_of_regular_detuning = WaitForWordFromRxBuffer();
            channel.patch.pitch_bend_enabled = WaitForWordFromRxBuffer();
            channel.patch.portamento_enabled = WaitForWordFromRxBuffer();
            channel.patch.portamento_rate = WaitForWordFromRxBuffer();
            channel.patch.sustain_enabled = WaitForWordFromRxBuffer();
            channel.patch.detuning_mode_id = WaitForWordFromRxBuffer();
            channel.patch.layer_config[0].st_inc_q11_5_detune = WaitForWordFromRxBuffer();
            channel.patch.layer_config[1].st_inc_q11_5_detune = WaitForWordFromRxBuffer();        
            channel.patch.patch_gain = WaitForWordFromRxBuffer();
            channel.patch.key_scale_split_note_id = WaitForWordFromRxBuffer();
            break;
        }
        case msg_type_waveform_harmonic:
        {
            uint16_t n = WaitForWordFromRxBuffer();
            uint16_t i = WaitForWordFromRxBuffer();
            uint16_t waveform_id = WaitForWordFromRxBuffer();
            uint16_t h;

            for (h = 0;h < max_harmonics;h++)
                {
                 channel.patch.waveform_set.waveform_harmonic[n][i][waveform_id][h].level = WaitForWordFromRxBuffer();
                } 
            break;
        }
        case msg_type_waveform_set:
        {
            channel.patch.waveform_set.timbre_controller1_source = WaitForWordFromRxBuffer();
            channel.patch.waveform_set.timbre_controller2_source = WaitForWordFromRxBuffer();
            channel.patch.waveform_set.timbre_mode = WaitForWordFromRxBuffer();  
            break;
        } 
        case ChannelVolume :
        {
            uint16_t v = WaitForWordFromRxBuffer();
            if (v > 0){v = (v<<8) + 255;}
            channel.volume = v;
            break;
        }  
    }
}
return note_on_instance_id;
}

