/*
 * File:   main.c
 * Author: Jeremy Leach
 */


// DSPIC33FJ128GP802 Configuration Bit Settings

// 'C' source line config statements

// FBS
#pragma config BWRP = WRPROTECT_OFF     // Boot Segment Write Protect (Boot Segment may be written)
#pragma config BSS = NO_FLASH           // Boot Segment Program Flash Code Protection (No Boot program Flash segment)
#pragma config RBS = NO_RAM             // Boot Segment RAM Protection (No Boot RAM)

// FSS
#pragma config SWRP = WRPROTECT_OFF     // Secure Segment Program Write Protect (Secure segment may be written)
#pragma config SSS = NO_FLASH           // Secure Segment Program Flash Code Protection (No Secure Segment)
#pragma config RSS = NO_RAM             // Secure Segment Data RAM Protection (No Secure RAM)

// FGS
#pragma config GWRP = OFF               // General Code Segment Write Protect (User program memory is not write-protected)
#pragma config GSS = OFF                // General Segment Code Protection (User program memory is not code-protected)

// FOSCSEL
#pragma config FNOSC = PRIPLL           // Oscillator Mode (Primary Oscillator (XT, HS, EC) w/ PLL)
#pragma config IESO = ON                // Internal External Switch Over Mode (Start-up device with FRC, then automatically switch to user-selected oscillator source when ready)

// FOSC
#pragma config POSCMD = EC              // Primary Oscillator Source (EC Oscillator Mode)
#pragma config OSCIOFNC = ON            // OSC2 Pin Function (OSC2 pin has digital I/O function)
#pragma config IOL1WAY = OFF            // Peripheral Pin Select Configuration (Allow Multiple Re-configurations)
#pragma config FCKSM = CSDCMD           // Clock Switching and Monitor (Both Clock Switching and Fail-Safe Clock Monitor are disabled)

// FWDT
#pragma config WDTPOST = PS32768        // Watchdog Timer Postscaler (1:32,768)
#pragma config WDTPRE = PR128           // WDT Prescaler (1:128)
#pragma config WINDIS = OFF             // Watchdog Timer Window (Watchdog Timer in Non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (Watchdog timer enabled/disabled by user software)

// FPOR
#pragma config FPWRT = PWR128           // POR Timer Value (128ms)
#pragma config ALTI2C = OFF             // Alternate I2C  pins (I2C mapped to SDA1/SCL1 pins)

// FICD
#pragma config ICS = PGD1               // Comm Channel Select (Communicate on PGC1/EMUC1 and PGD1/EMUD1)
#pragma config JTAGEN = OFF             // JTAG Port Enable (JTAG is Disabled)

// #pragma config statements should precede project file includes.
// Use project enums instead of #define for ON and OFF.

#include <xc.h>
#include "Spectral_Pins.h"
#include "Spectral_SPI_1.h"
#include "Spectral_SPI_2.h"
#include "Spectral_Sound.h"
#include "Spectral_Timer1.h"
#include "Spectral_Interrupts.h"
#include "Spectral_I2C.h"

//=============== DECLARATIONS =================
extern volatile int32_t tone_proc_sample_sum;
extern volatile uint16_t flag_sample_refresh_pending;
extern volatile int16_t patch_id_to_save;
extern volatile int16_t patch_id_to_load;
extern volatile int16_t performance_id_to_select;
extern volatile int16_t processor_being_sampled;
extern volatile fifo_buffer8_t SPI1RxBuffer;
extern volatile uint16_t tp_control_value;
volatile uint16_t flag_do_non_sample_stuff;

//=============== MAIN =========================
int main(void) {
    
    //Set the instruction frequency to 40MHz (MIPS).
    CLKDIVbits.PLLPRE   = 0; //(0 + offset 2 = 2). Options are 2,3 ..33)16MHz external clock / 2 = 8 MHz. which is within the acceptable range of 0.8-8 MHz.
    PLLFBDbits.PLLDIV   = 18;// (18 + offset 2 = 20)8MHz * 20 = 160MHz. which is within the 100-200 MHz ranged needed. 
    CLKDIVbits.PLLPOST  = 0; //(0+ offset 2 = 2). 160MHz / 2 = 80 MHz which is FOSC
    while(!OSCCONbits.LOCK); //Wait for PLL ready
    //FCY = FOSC/2 = 40MIPS
    
    //Disable selected peripherals to reduce power consumption. 1 = Disable.
    PMD1bits.AD1MD  = 1; //ADC1 Module Disable bit
    PMD1bits.C1MD   = 1; //ECAN1 Module Disable bit
    PMD1bits.DCIMD  = 1; //DCI Module Disable bit
    PMD1bits.I2C1MD = 0; //I2C1 Module Disable bit
    PMD1bits.SPI1MD = 0; //SPI1 Module Disable bit
    PMD1bits.SPI2MD = 0; //SPI2 Module Disable bit
    PMD1bits.T1MD   = 1; //Timer1 Module Disable bit
    PMD1bits.T2MD   = 0; //Timer2 Module Disable bit
    PMD1bits.T3MD   = 1; //Timer3 Module Disable bit
    PMD1bits.T4MD   = 1; //Timer4 Module Disable bit
    PMD1bits.T5MD   = 1; //Timer5 Module Disable bit
    PMD1bits.U1MD   = 1; //UART1 Module Disable bit
    PMD1bits.U2MD   = 1; //UART2 Module Disable bit

    PMD2bits.IC1MD  = 1;  //Input Capture 1 Module Disable bit
    PMD2bits.IC2MD  = 1;  //Input Capture 2 Module Disable bit
    PMD2bits.IC7MD  = 1;  //Input Capture 7 Module Disable bit
    PMD2bits.IC8MD  = 1;  //Input Capture 8 Module Disable bit
    PMD2bits.OC1MD  = 1;  //Output Compare 1 Module Disable bit
    PMD2bits.OC2MD  = 1;  //Output Compare 1 Module Disable bit   
    PMD2bits.OC3MD  = 1;  //Output Compare 1 Module Disable bit
    PMD2bits.OC4MD  = 1;  //Output Compare 1 Module Disable bit
        
    PMD3bits.CMPMD  = 1;  //Comparator Module Disable bit
    PMD3bits.CRCMD  = 1;  //CRC Module Disable bit
    PMD3bits.DAC1MD = 0;  //DAC1 Module Disable bit
    PMD3bits.PMPMD  = 1;  //Parallel Master Port Module Disable bit
    PMD3bits.RTCCMD = 1;  //Real-Time Clock and Calendar Module Disable bit
    
    Spectral_Pins_Initialize();      
    SPI1Initialize();
    SPI2Initialize();
    
    SoundInitialize();
    Timer_Initialize();
    I2CInitialize();
    Interrupts_Initialize();
    
    flag_do_non_sample_stuff = 0;
    //Make sure tone processors are initialised. NOTE !! POR Timer Value (128ms). So it waits 128ms before it even starts code execution!
    //Also note default BOR threshold is 2.5V
    delay_for_ms(100); 
    
    int16_t w;
    while (1)
    {
        //========== Wait for sample flag
        while(flag_sample_refresh_pending == 0){}  
        
        //Process a single byte of SPI1 message (or more if it's a message for the Mixer))
        if (SPI1RxBuffer.count > 0){ProcessSPI1Message();}

        //========== Sample Tone Processors 
        //------------ Tone proc 0 ---------
        OUT_SPI2_SS1_PORT_SetLow();SPI2BUF =  tp_control_value;
        while(SPI2STATbits.SPIRBF == 0){}; w = (int16_t)SPI2BUF; 
        //------------ Tone proc 1 ---------
        OUT_SPI2_SS1_PORT_SetHigh();OUT_SPI2_SS2_PORT_SetLow();SPI2BUF =  tp_control_value;
        tone_proc_sample_sum +=(int32_t)w;//Add proc0
        while(SPI2STATbits.SPIRBF == 0){}; w = (int16_t)SPI2BUF; 
        //------------ Tone proc 2 ---------
        OUT_SPI2_SS2_PORT_SetHigh();OUT_SPI2_SS3_PORT_SetLow();SPI2BUF =  tp_control_value;
        tone_proc_sample_sum +=(int32_t)w;//Add proc1
        while(SPI2STATbits.SPIRBF == 0){}; w = (int16_t)SPI2BUF;  
        //------------ Tone proc 3 ---------
        OUT_SPI2_SS3_PORT_SetHigh();OUT_SPI2_SS4_PORT_SetLow();SPI2BUF =  tp_control_value;
        tone_proc_sample_sum +=(int32_t)w;//Add proc2
        while(SPI2STATbits.SPIRBF == 0){}; w = (int16_t)SPI2BUF;
        //------------ Tone proc 4 ---------
        OUT_SPI2_SS4_PORT_SetHigh();OUT_SPI2_SS5_PORT_SetLow();SPI2BUF =  tp_control_value;
        tone_proc_sample_sum +=(int32_t)w;//Add proc3
        while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF;
        //------------ Tone proc 5 ---------
        OUT_SPI2_SS5_PORT_SetHigh();OUT_SPI2_SS6_PORT_SetLow();SPI2BUF =  tp_control_value;
        tone_proc_sample_sum +=(int32_t)w;//Add proc4
        while(SPI2STATbits.SPIRBF == 0){};w = (int16_t)SPI2BUF; 
        OUT_SPI2_SS6_PORT_SetHigh();
        tone_proc_sample_sum +=(int32_t)w;//Add proc5
        
        //========== Update AGC
        UpdateAGC();
        
        flag_sample_refresh_pending = 0;

        //========== Handle any non-sample tasks 
        if (flag_do_non_sample_stuff == 1)
            {
            if (patch_id_to_save != -1)
                {
                SavePatchToEEPROM(patch_id_to_save);
                patch_id_to_save = -1;
                }
            else if (patch_id_to_load != -1)
                {               
                ChangePlayMode(StopPlayModeTP);
                LoadPatchFromEEPROM(patch_id_to_load);
                send_all_patch_data(all_processors);
                patch_id_to_load =-1;
                ChangePlayMode(StartPlayModeTP);
                }
            else if (performance_id_to_select != -1)
                {
                SelectPerformance(performance_id_to_select);
                performance_id_to_select = -1;
                }
            flag_do_non_sample_stuff = 0;
            }
    }
    
    return 1;
}
