/* 
 * File:   SPECTRAL_messaging.c
 * Author: Jeremy Leach 2019
 */
#include "mcc_generated_files/mcc.h"
#include "SPECTRAL_MIDI.h"
#include "SPECTRAL_messaging.h"
#include "mcc_generated_files/device_config.h"


/******************************************************************************/
/***** Definitions ************************************************************/
/******************************************************************************/ 
static volatile uint8_t read_buffer[wr_buffer_size];
static volatile uint8_t write_buffer[wr_buffer_size];
static volatile bool flag_midi_message_in_progress = false;
static volatile uint8_t usb_message_state = waiting_for_start;
static volatile uint8_t usb_payload_LSB;
static volatile uint8_t usb_payload_MSB;
static volatile uint16_t usb_payload_bytes_left;
static volatile uint16_t usb_payload_body_index;
static volatile uint8_t current_recipient_id;
static volatile uint8_t current_message_type_id;
extern volatile performance_t performance[performances_per_patchset];
extern volatile uint8_t midi_rx_LED_hundreths;
volatile uint8_t performance_id;
volatile uint8_t flag_pending_send_performance_to_mixer;
volatile uint8_t flag_pending_select_performance;
volatile ucc_item ucc_items[max_ucc_items];
static const uint8_t cc_index_mapping[max_midi_controllers]={0,2,3,0,5,0,0,0,0,0,0,4,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
extern volatile uint8_t bank_lsb;
extern volatile uint8_t bank_msb;
extern volatile uint16_t bank;
extern volatile uint16_t default_performance_bank;
extern volatile uint16_t default_patch_bank;

/******************************************************************************/
/***** Public Methods *********************************************************/
/******************************************************************************/

void SPECTRAL_messaging_initialise()
{
    uint8_t i;
    for (i=0;i<max_ucc_items;i++)
    {
        ucc_items[i].ucc_value = 0;
        ucc_items[i].ucc_status = not_used;
    }
    performance_id = performance_id_not_set;
    flag_pending_send_performance_to_mixer = 0;
    flag_pending_select_performance = 0;
    flag_midi_message_in_progress = false;    
}

void delay_ms(uint16_t delaytime)
{
    //The delay value(the number of instruction cycles that will be executed by the delay) 
    //should be less than 179,200 for PIC18 devices. At 48MHz clock this is only 3.7ms max !
    //So better to have function like this for milliseconds.
    do
    {
        OUT_MIDI_SENSE_SetHigh();
        __delay_ms(1); 
    } while (--delaytime);
    OUT_MIDI_SENSE_SetLow();
}

void SPECTRAL_ProcessUSBData ()
{
    if( USBGetDeviceState() < CONFIGURED_STATE ){return;}
    if( USBIsDeviceSuspended()== true ){return;}
    if(( USBUSARTIsTxTrfReady() == true) && (flag_midi_message_in_progress == false))
    {
        uint8_t i = 0;
        uint8_t numBytesRead;

        numBytesRead = getsUSBUSART(read_buffer, sizeof(read_buffer));
        if (numBytesRead > 0 )  
        {
            OUT_SS_PORT = 0; //Select slave
            while (i<numBytesRead)
            {
                uint8_t b = read_buffer[i];
                
                switch (usb_message_state)
                {
                    case waiting_for_start:
                        if (b == message_start_char){usb_message_state = waiting_for_length_LSB;}
                        break;
                    case waiting_for_length_LSB:
                        usb_payload_LSB = b;
                        usb_message_state = waiting_for_length_MSB;
                        break;
                    case waiting_for_length_MSB:
                        usb_payload_MSB = b;
                        usb_payload_bytes_left = (uint16_t)((usb_payload_MSB<<8) + usb_payload_LSB);
                        usb_message_state = waiting_for_payload_recipient_id;
                        break;
                    case waiting_for_payload_recipient_id:
                        current_recipient_id = b;
                        if (current_recipient_id != master){SPI1_Exchange8bit(b);}  
                        usb_payload_bytes_left --;
                        usb_message_state = waiting_for_payload_message_type_id;
                        break;
                    case waiting_for_payload_message_type_id:
                        current_message_type_id = b;
                        if (current_recipient_id != master){SPI1_Exchange8bit(b);}
                        usb_payload_bytes_left --;
                        usb_message_state = waiting_for_other_payload_bytes;
                        usb_payload_body_index = 0;
                        break;
                    case waiting_for_other_payload_bytes:
                        if (current_recipient_id != master)
                            {SPI1_Exchange8bit(b);}
                        else
                            {
                            switch (current_message_type_id)
                                {
                                case msg_type_performance:
                                    if (usb_payload_body_index == 0)
                                        {performance_id = b;}
                                    else
                                        {performance[performance_id].data[usb_payload_body_index - 1] = b;}
                                    usb_payload_body_index ++;
                                    break;
                                case msg_type_relay_performances_to_mixer:
                                    flag_pending_send_performance_to_mixer = 1;
                                    break;
                                case msg_default_banks:
                                    if (usb_payload_body_index == 0)
                                        {bank_lsb = b;}
                                    else if (usb_payload_body_index == 1)
                                        {bank_msb = b;
                                        default_performance_bank = (uint16_t)((uint16_t)(bank_msb<<7) + (uint16_t)bank_lsb);//NOTE<<7 and not 8
                                        }
                                    else if (usb_payload_body_index == 2)
                                        {bank_lsb = b;}
                                    else if (usb_payload_body_index == 3)
                                        {bank_msb = b;
                                        default_patch_bank = (uint16_t)((uint16_t)(bank_msb<<7) + (uint16_t)bank_lsb);//NOTE<<7 and not 8
                                        }
                                    usb_payload_body_index ++;
                                    break;
                                case msg_type_select_performance:
                                    performance_id = b;
                                    flag_pending_select_performance = 1;
                                    break;   
                                }
                            }
                        usb_payload_bytes_left --;
                        if (usb_payload_bytes_left == 0){usb_message_state = waiting_for_start;}
                        break;                        
                }
                
                i++;
            }
            OUT_SS_PORT = 1; //De-select Slave
        }
    }
    CDCTxService(); 
}

unsigned uint8_t eeprom18_read(uint8_t offset)
{
    EECON1bits.EEPGD = 0; //accesses data EEPROM memory
    EECON1bits.CFGS = 0; //accesses data EEPROM memory

    //This doesn't exist o this chip : EEADRH = offset >> 8;
    EEADR = offset;

    EECON1bits.RD = 1; //initiates an EEPROM read
    Nop(); //it can be read after one NOP instruction

    return EEDATA;
}

void eeprom18_write(unsigned int offset, unsigned uint8_t value)
{
    //Fromm : https://www.medo64.com/2018/08/eeprom-access-routines-for-pic18/
    EECON1bits.EEPGD = 0; //accesses data EEPROM memory
    EECON1bits.CFGS = 0; //accesses data EEPROM memory

    //This doesn't exist o this chip : EEADRH = offset >> 8;
    EEADR = offset;

    EEDATA = value;

    EECON1bits.WREN = 1; //allows write cycles
    unsigned char oldGIE = (unsigned char)GIE; //interrupts be disabled during this code segment

    EECON2 = 0x55; //write sequence unlock
    EECON2 = 0xAA; //write sequence unlock

    EECON1bits.WR = 1; //initiates a data EEPROM erase/write cycle
    while(EECON1bits.WR); //waits for write cycle to complete

    GIE = oldGIE; //restore interrupts
    EECON1bits.WREN = 0; //disable write
}

void SelectPerformance(void)
{
    OUT_SS_PORT = 0; //Select slave 
    msg_select_performance sp;
    sp.message_recipient = mixer;
    sp.message_type = msg_type_select_performance;
    sp.performance_id = performance_id;
    SPI1_Exchange8bitBuffer(sp.data,msg_select_performance_size,NULL);
    OUT_SS_PORT = 1; //De-select Slave
    
    SPECTRAL_MIDI_Initialise(performance_id);
    
    flag_pending_select_performance = 0;
    __delay_us(10); 
}

void RelayPerformancesToMixer(void)
{
    while(usb_message_state != waiting_for_start){};
    
    flag_midi_message_in_progress = true;

    msg_performance p;
    uint8_t perf_id;
    uint8_t i;
    
    p.message_recipient = mixer;
    p.message_type = msg_type_performance;    
    
    for (perf_id = 0;perf_id < performances_per_patchset;perf_id++)
        {
        OUT_SS_PORT = 0; //Select slave 
        
        p.performance_id = perf_id;
        for(i = 0;i < performance_size;i++){p.performance.data[i] = performance[perf_id].data[i];}
        SPI1_Exchange8bitBuffer(p.data,msg_performance_size,NULL);

        OUT_SS_PORT = 1; //De-select Slave
        __delay_us(10); //Give time for Mixer to process message
        }

    //Select the first performance
    performance_id = 0;
    SelectPerformance();
    
    //====Clear flags
    flag_pending_send_performance_to_mixer = 0;
    flag_midi_message_in_progress = false;    
}

void SPECTRAL_SavePerformancesToEEPROM(void)
{
    //31 * 8 = 248 bytes. Only 256 available.
    uint8_t ee_addr = 1;//Address 0 is used as a flag to indicate if the EEPROM is configured
    uint8_t perf_id;
    uint8_t i;
    for (perf_id = 0;perf_id < performances_per_patchset;perf_id++)
        {
        for(i = 0;i < performance_size;i++){eeprom18_write(ee_addr++,performance[perf_id].data[i]);}
        }
    eeprom18_write(0,eeprom_loaded_indicator); //Use location 0 as special indicator to flag if EEPOM is loaded.
}

void SPECTRAL_LoadPerformancesFromEEPROM(void)
{
    //1 + (31 * 8) = 249 bytes. Only 256 available.
    uint8_t ee_addr = 1;//Address 0 is used as a flag to indicate if the EEPROM is configured
    uint8_t perf_id;
    uint8_t i;
    for (perf_id = 0;perf_id < performances_per_patchset;perf_id++)
        {
        for(i = 0;i < performance_size;i++){performance[perf_id].data[i] = eeprom18_read(ee_addr++);}
        }
}

void SPECTRAL_LoadBankNumbersFromEEPROM(void)
{
    uint8_t LSB;
    uint8_t MSB;
    MSB = eeprom18_read(EEPROM_perfbank_msb);
    LSB = eeprom18_read(EEPROM_perfbank_lsb);
    default_performance_bank = (uint16_t)((uint16_t)((uint16_t)MSB <<7) + (uint16_t)LSB); //NOTE <<7 and not 8
    MSB = eeprom18_read(EEPROM_patchbank_msb);
    LSB = eeprom18_read(EEPROM_patchbank_lsb);
    default_patch_bank = (uint16_t)((uint16_t)((uint16_t)MSB <<7) + (uint16_t)LSB); //NOTE <<7 and not 8
}

void SPECTRAL_SaveBankNumbersToEEPROM(void)
{
    uint8_t b;
    b = (uint8_t)((uint16_t)default_performance_bank>>7);//NOTE >>7 and not 8
    eeprom18_write(EEPROM_perfbank_msb,b);
    b = (uint8_t)((uint16_t)default_performance_bank - (uint16_t)((uint16_t)b<<7));
    eeprom18_write(EEPROM_perfbank_lsb,b);
    b = (uint8_t)((uint16_t)default_patch_bank>>7);//NOTE >>7 and not 8
    eeprom18_write(EEPROM_patchbank_msb,b);
    b = (uint8_t)((uint16_t)default_patch_bank - ((uint16_t)b<<7));
    eeprom18_write(EEPROM_patchbank_lsb,b);   
}

void SPECTRAL_ForwardMIDIMessage(uint8_t processor_id,uint8_t instance_id,midi_event e,uint8_t last_midi_note_number)
{
    midi_rx_LED_hundreths = midi_sense_delay_hundreths;OUT_MIDI_SENSE_SetHigh();
    
    while(usb_message_state != waiting_for_start){};
    
    flag_midi_message_in_progress = true;
    OUT_SS_PORT = 0; //Select slave
    
    uint8_t midi_channel;
    uint8_t status_without_channel;
    
    midi_channel= (uint8_t)(e.status & 0b00001111);//Channel starts at 0.
    status_without_channel = (uint8_t)(e.status & 0b11110000);
    switch (status_without_channel) 
    {
        case STATUS_note_on:  
            { //NOTE: Need bracket because declaring variable inside CASE
            msg_instance_noteon m;
            m.message_recipient = processor_id;
            m.message_type = InstanceNoteOn;
            m.payload_length = msg_instance_noteon_len - msg_header_length;
            m.instance_id = instance_id;
            m.midi_channel = midi_channel;
            m.patch_id = GetPatchForChannel(midi_channel); 
            m.note_id = e.data1;
            m.last_midi_note_number = last_midi_note_number;
            if (performance_id != performance_id_not_set)
                {
                 m.velocity_id = (uint8_t)((e.data2 * performance[performance_id].pc[midi_channel].volume)>>8); //Utilises 8 x 8, 1 cycle hardware multiply
                }
            else
                {
                 m.velocity_id = e.data2;
                }
           
            if ((m.note_id >=Standard88Keyboard_bottomnote) && (m.note_id <= Standard88Keyboard_topnote)) //Limit to 88 note keyboard range (unpredictable results outside of this))
                {SPI1_Exchange8bitBuffer(m.data,msg_instance_noteon_len,NULL);}
            break;
            }
        case STATUS_note_off:
            { 
            //Still sent if note is outside of standard 88 note keyboard range but 'does nothing' at the Tone Processors.
            msg_instance_noteoff m;
            m.message_recipient = processor_id;
            m.message_type = InstanceNoteOff;
            m.payload_length = msg_instance_noteoff_len - msg_header_length;
            m.instance_id = instance_id;
            SPI1_Exchange8bitBuffer(m.data,msg_instance_noteoff_len,NULL);
            break;
            }
        case STATUS_control_change:
            {
            if (e.data1 == cc_bank_msb) 
                {
                bank_msb = e.data2;
                bank = (uint16_t)((uint16_t)((uint16_t)bank_msb << 7) + (uint16_t)(bank_lsb));
                }
            else if (e.data1 == cc_bank_lsb) 
                {
                bank_lsb = e.data2;
                bank = (uint16_t)((uint16_t)((uint16_t)bank_msb << 7) + (uint16_t)(bank_lsb));          
                }
            else
                {
                //Detect change. Don't send immediately. 
                uint8_t mapped_cc_index = cc_index_mapping[e.data1];//Convert midi control ids to this systems ids.
                uint16_t new_value = (uint16_t)e.data2 << 5; //Make into a 12 bit value
                if ((ucc_items[mapped_cc_index].ucc_status != pending_sending) && (ucc_items[mapped_cc_index].ucc_value != new_value))
                    {
                    ucc_items[mapped_cc_index].ucc_status = value_changed;
                    ucc_items[mapped_cc_index].ucc_value = new_value;
                    }                         
                }
            break;
            }
        case STATUS_pitch_bend:
            {
            //Don't send immediately. Status byte : 1110 CCCC, Data byte 1 : 0LLL LLLL, Data byte 2 : 0MMM MMMM
            uint16_t new_value = (((uint16_t)(e.data2)<<7) | (uint16_t)(e.data1))>>2; //Our pitch-bend is 12 bit not 14 bit
            if ((ucc_items[ucc_pitch_bend].ucc_status != pending_sending) && (ucc_items[ucc_pitch_bend].ucc_value != new_value))
                {
                ucc_items[ucc_pitch_bend].ucc_status = value_changed;
                ucc_items[ucc_pitch_bend].ucc_value = new_value;
                } 
            break;
            }
    }
    
    OUT_SS_PORT = 1; //De-select Slave
    flag_midi_message_in_progress = false;
}

void SPECTRAL_SendUCC(uint8_t ucc_id,uint16_t ucc_value)
{
midi_rx_LED_hundreths = midi_sense_delay_hundreths;OUT_MIDI_SENSE_SetHigh();
while(usb_message_state != waiting_for_start){};
    
    flag_midi_message_in_progress = true;
    OUT_SS_PORT = 0; //Select slave
 
    msg_ucc m;
    m.message_recipient = all_processors;
    m.message_type = UsedControlChange;
    m.ucc_id = ucc_id;
    m.ucc_value = ucc_value;

    SPI1_Exchange8bitBuffer(m.data,msg_ucc_len,NULL);
    
    OUT_SS_PORT = 1; //De-select Slave
    flag_midi_message_in_progress = false;
}

void SPECTRAL_SendVolumeChange(uint8_t new_volume_value)
{
    OUT_SS_PORT = 0; //Select slave
    
    msg_volume_change m;
    m.message_recipient = mixer;
    m.message_type = ChangeVolume;
    m.payload_length = msg_volume_change_len - msg_header_length;
    m.new_volume_value = new_volume_value;        

    SPI1_Exchange8bitBuffer(m.data,msg_volume_change_len,NULL);
    
    OUT_SS_PORT = 1; //De-select Slave
}

void SPECTRAL_SendSelectPatch(uint8_t module_patch_id)
{
    OUT_SS_PORT = 0; //Select slave
    
    midi_rx_LED_hundreths = midi_sense_delay_hundreths;OUT_MIDI_SENSE_SetHigh();
    
    msg_select_patch m;
    m.message_recipient = mixer;
    m.message_type = msg_type_load_patch_from_EEPROM;
    m.payload_length = msg_select_patch_len - msg_header_length;
    m.module_patch_id = module_patch_id;        

    SPI1_Exchange8bitBuffer(m.data,msg_select_patch_len,NULL);
    
    OUT_SS_PORT = 1; //De-select Slave
}


